//==============================================
//       rpgsetup.pas
//
//         Delphi.
//          .
//
//      Copyright 1998-2000 Polaris Software
//      http://members.xoom.com/PolarisSoft
//      mailto: PolarisLib@mail.ru
//==============================================
{$M+}

unit rPgSetup;

{$I POLARIS.INC}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  CommDlg, Printers;

type
  TrPageSetupOption    = (poDefaultMinMargins, poDisableMargins, poDisableOrientation,
                          poDisablePagePainting, poDisablePaper, poDisablePrinter, poNoWarning,
                          poShowHelp);
  TrPageSetupOptions   = set of TrPageSetupOption;
  TrPSPaperType        = (ptPaper, ptEnvelope);
  TrPSPrinterType      = (ptDotMatrix, ptHPPCL);
  TrPSPaperSize        = (rpsDEFAULT, rpsLETTER, rpsLETTERSMALL, rpsTABLOID, rpsLEDGER,
                          rpsLEGAL, rpsSTATEMENT, rpsEXECUTIVE, rpsA3, rpsA4, rpsA4SMALL,
                          rpsA5, rpsB4, rpsB5, rpsFOLIO, rpsQUARTO, rps10X14, rps11X17,
                          rpsNOTE, rpsENV_9, rpsENV_10, rpsENV_11, rpsENV_12, rpsENV_14,
                          rpsCSHEET, rpsDSHEET, rpsESHEET, rpsENV_DL, rpsENV_C5, rpsENV_C3,
                          rpsENV_C4, rpsENV_C6, rpsENV_C65, rpsENV_B4, rpsENV_B5, rpsENV_B6,
                          rpsENV_ITALY, rpsENV_MONARCH, rpsENV_PERSONAL, rpsFANFOLD_US,
                          rpsFANFOLD_STD_GERMAN, rpsFANFOLD_LGL_GERMAN);

  TrPSPaintWhat        = (pwFullPage, pwMinimumMargins, pwMargins,
                          pwGreekText, pwEnvStamp, pwYAFullPage);

  TrPSMeasurements     = (pmDefault, pmMillimeters, pmInches);
  TrPSPrinterEvent     = procedure(Sender: TObject; Wnd: HWND) of object;

  PrPSDlgData          = ^TPageSetupDlg;

  TrPSInitPaintPageEvent = function(Sender: TObject; PaperSize: TrPSPaperSize;
     PaperType: TrPSPaperType; PaperOrientation: TPrinterOrientation;
     PrinterType: TrPSPrinterType; pSetupData: PrPSDlgData): Boolean of object;
  TrPSPaintPageEvent     = function(Sender: TObject; PaintWhat: TrPSPaintWhat;
     Canvas: TCanvas; Rect: TRect): Boolean of object;

  TrPageSetupDialog = class;

  TrPSRect = class(TPersistent)
  private
    FRect: TRect;
    FOwnerDlg: TrPageSetupDialog;
    {$IFDEF VER125}
    function GetLeft: Integer;
    function GetRight: Integer;
    function GetTop: Integer;
    function GetBottom: Integer;
    procedure SetLeft(Value: Integer);
    procedure SetRight(Value: Integer);
    procedure SetTop(Value: Integer);
    procedure SetBottom(Value: Integer);
    {$ENDIF}
    function GetRealRect(Index: Integer): Double;
    procedure SetRealRect(Index: Integer; Value: Double);
  public
    constructor Create(Owner: TrPageSetupDialog); virtual;
    destructor Destroy; override;

    function Compare(Other: TrPSRect): Boolean;
    property Rect: TRect read FRect write FRect;
    property absLeft: Integer   read {$IFDEF VER125} GetLeft {$ELSE} FRect.Left {$ENDIF}
       write {$IFDEF VER125} SetLeft {$ELSE} FRect.Left {$ENDIF};
    property absRight: Integer  read {$IFDEF VER125} GetRight {$ELSE} FRect.Right {$ENDIF}
       write {$IFDEF VER125} SetRight {$ELSE} FRect.Right {$ENDIF};
    property absTop: Integer    read {$IFDEF VER125} GetTop {$ELSE} FRect.Top {$ENDIF}
       write {$IFDEF VER125} SetTop {$ELSE} FRect.Top {$ENDIF};
    property absBottom: Integer read {$IFDEF VER125} GetBottom {$ELSE} FRect.Bottom {$ENDIF}
       write {$IFDEF VER125} SetBottom {$ELSE} FRect.Bottom {$ENDIF};
  published
    property Left: Double index 0   read GetRealRect write SetRealRect;
    property Right: Double index 1  read GetRealRect write SetRealRect;
    property Top: Double index 2    read GetRealRect write SetRealRect;
    property Bottom: Double index 3 read GetRealRect write SetRealRect;
  end;

  TrPSPoint = class(TPersistent)
  private
    FPoint: TPoint;
    FOwnerDlg: TrPageSetupDialog;
  protected
    function GetX: longint;
    function GetY: longint;
    function GetRealCoord(Index: Integer): Double;
    procedure SetRealCoord(Index: Integer; Value: Double);
    procedure SetX(Val: longint);
    procedure SetY(Val: longint);
  public
    constructor Create(Owner: TrPageSetupDialog); virtual;
    destructor Destroy; override;
    function Compare(Other: TrPSPoint): Boolean;

    property Point: TPoint read FPoint write FPoint;
    property X: longint read GetX write SetX;
    property Y: longint read GetY write SetY;
  published
    property Width: Double index 0 read GetRealCoord write SetRealCoord;
    property Height: Double index 1 read GetRealCoord write SetRealCoord;
  end;

  TrPageSetupDialog = class(TCommonDialog)
  private
    FPrinter:         TPrinter;
    FGettingDefaults: Boolean;
    FCentered:        Boolean;
    FOptions:         TrPageSetupOptions;
    FOrientation:     TPrinterOrientation;
    FCustomData:      LPARAM;
    FPaperSize:       TrPSPoint;
    FMinimumMargins:  TrPSRect;
    FMargins:         TrPSRect;
    FMeasurements:    TrPSMeasurements;
    FOnPrinter:       TrPSPrinterEvent;
    FOnInitPaintPage: TrPSInitPaintPageEvent;
    FOnPaintPage:     TrPSPaintPageEvent;

    function DoPrinter(Wnd: HWND): Boolean;
    function DoExecute(Func: Pointer): Boolean;
    procedure GetPrinter(var DeviceMode, DeviceNames: THandle);
    procedure SetPrinter(DeviceMode, DeviceNames: THandle);

    procedure _SetPrinter(Value: TPrinter);
  protected
    function _Printer(Wnd: HWND): Boolean; virtual;
    function StorePaperSize: Boolean;
    function StoreMinimumMargins: Boolean;
    function StoreMargins: Boolean;
    function GetDefaultMeasurements: TrPSMeasurements;
    function GetCurrentMeasurements: TrPSMeasurements;
    function GetPaperSizeType: TrPSPaperSize;
    function GetPaperSizeName: String;

    procedure SetName(const NewName: TComponentName); override;
    procedure SetPaperSize(const Val: TrPSPoint);
    procedure SetMinimumMargins(const Val: TrPSRect);
    procedure SetMargins(const Val: TrPSRect);
    procedure SetMeasurements(Val: TrPSMeasurements);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function Execute: Boolean; override;
    function ReadCurrentValues: Boolean; virtual;
    function FromMeasurementVal(Val: Integer): Double;
    function ToMeasurementVal(Val: Double): Integer;

    property PaperSizeType: TrPSPaperSize          read GetPaperSizeType;
    property PaperSizeName: String                 read GetPaperSizeName;
    property DefaultMeasurements: TrPSMeasurements read GetDefaultMeasurements;
    property CurrentMeasurements: TrPSMeasurements read GetCurrentMeasurements;
    property CustomData: LPARAM                    read FCustomData   write FCustomData;
    property Printer: TPrinter                     read FPrinter      write _SetPrinter;
  published
    property Measurements: TrPSMeasurements   read FMeasurements   write SetMeasurements nodefault;
    property PaperSize: TrPSPoint             read FPaperSize      write SetPaperSize stored StorePaperSize;
    property MinMargins: TrPSRect             read FMinimumMargins write SetMinimumMargins stored StoreMinimumMargins;
    property Margins: TrPSRect                read FMargins        write SetMargins stored StoreMargins;
    property Centered: Boolean                read FCentered       write FCentered default TRUE;
    property Options: TrPageSetupOptions      read FOptions        write FOptions default [poDefaultMinMargins];
    property Orientation: TPrinterOrientation read FOrientation    write FOrientation;

    { Events }
    property OnPrinter: TrPSPrinterEvent             read FOnPrinter write FOnPrinter;
    property OnInitPaintPage: TrPSInitPaintPageEvent read FOnInitPaintPage write FOnInitPaintPage;
    property OnPaintPage: TrPSPaintPageEvent         read FOnPaintPage write FOnPaintPage;
  end;

implementation

{$IFDEF VER110}
uses
  Dlgs;
{$ENDIF}

const
  IDPRINTERBTN = {$IFDEF VER110} Dlgs.psh3 {$ELSE} $0402 {$ENDIF};
  MeasurementsDiv : array [pmMillimeters..pmInches] of Double = (100.0,1000.0);

var
  HelpMsg: Integer;
  DefOrientation: TPrinterOrientation;
  DefMeasurements: TrPSMeasurements;
  DefPaperSizeI: TrPSPoint;
  DefMinimumMarginsI: TrPSRect;
  DefMarginsI: TrPSRect;
  DefPaperSizeM: TrPSPoint;
  DefMinimumMarginsM: TrPSRect;
  DefMarginsM: TrPSRect;
  HookCtl3D: Boolean;
  PageSetupDialog: TrPageSetupDialog;

procedure CenterWindow(Wnd: HWnd);
var
  Rect: TRect;
begin
  GetWindowRect(Wnd, Rect);
  SetWindowPos(Wnd, 0,
     (GetSystemMetrics(SM_CXSCREEN) - Rect.Right + Rect.Left) div 2,
     (GetSystemMetrics(SM_CYSCREEN) - Rect.Bottom + Rect.Top) div 3,
     0, 0, SWP_NOACTIVATE or SWP_NOSIZE or SWP_NOZOrdER);
end;

function DialogHook(Wnd: HWnd; Msg: UINT; WParam: WPARAM; LParam: LPARAM): UINT; stdcall;
begin
  Result := 0;
  case Msg of
  WM_INITDIALOG: begin
                   if HookCtl3D then begin
                     Subclass3DDlg(Wnd, CTL3D_ALL);
                     SetAutoSubClass(True);
                   end;
                   if PageSetupDialog.Centered
                   then CenterWindow(Wnd);
                   Result := 1;
                 end;
  WM_DESTROY:    if HookCtl3D then SetAutoSubClass(False);
  end;
end;

function PageSetupDialogHook(Wnd: HWnd; Msg: UINT; WParam: WPARAM;
                             LParam: LPARAM): UINT; stdcall;
const
  PagePaintWhat: array[WM_PSD_FULLPAGERECT..
                       WM_PSD_YAFULLPAGERECT] of TrPSPaintWhat = (
    pwFullPage, pwMinimumMargins, pwMargins,
    pwGreekText, pwEnvStamp, pwYAFullPage
  );
  PRINTER_MASK = $00000002;
  ORIENT_MASK  = $00000004;
  PAPER_MASK   = $00000008;
var
  PaperData:   Word;
  Paper:       TrPSPaperType;
  Orient:      TPrinterOrientation;
  Printer:     TrPSPrinterType;
  PaintRect:   TRect;
  PaintCanvas: TCanvas;
begin
  if (Msg = WM_COMMAND) and (LongRec(WParam).Lo = IDPRINTERBTN) and
     (LongRec(WParam).Hi = BN_CLICKED) then
    Result := Ord(PageSetupDialog.DoPrinter(Wnd))
  else begin
    if Assigned(PageSetupDialog.FOnInitPaintPage) and
       Assigned(PageSetupDialog.FOnPaintPage) then begin
      case Msg of
      WM_PSD_PAGESETUPDLG: begin
            PaperData := HiWord(WParam);
            if (PaperData AND PAPER_MASK > 0) then Paper := ptEnvelope
            else Paper := ptPaper;
            if (PaperData AND ORIENT_MASK > 0) then Orient := poPortrait
            else Orient := poLandscape;
            if (PaperData AND PAPER_MASK > 0) then Printer := ptHPPCL
            else Printer := ptDotMatrix;
            Result := Ord(PageSetupDialog.FOnInitPaintPage(PageSetupDialog,
               TrPSPaperSize(LoWord(WParam)),
               Paper, Orient, Printer, PrPSDlgData(LParam)));
      end;
      WM_PSD_FULLPAGERECT,
      WM_PSD_MINMARGINRECT,
      WM_PSD_MARGINRECT,
      WM_PSD_GREEKTEXTRECT,
      WM_PSD_ENVSTAMPRECT,
      WM_PSD_YAFULLPAGERECT: begin
        if LParam <> 0 then PaintRect := PRect(LParam)^
        else PaintRect := Rect(0,0,0,0);
        PaintCanvas := TCanvas.Create;
        PaintCanvas.Handle := HDC(WParam);
        try
          Result := Ord(PageSetupDialog.FOnPaintPage(PageSetupDialog,
                        PagePaintWhat[Msg], PaintCanvas, PaintRect));
        finally
          PaintCanvas.Free;   { This better not be deleting the DC! }
        end;
      end;
      else Result := DialogHook(Wnd, Msg, wParam, lParam);
      end;
    end
    else Result := DialogHook(Wnd, Msg, wParam, lParam);
  end;
end;

constructor TrPSRect.Create(Owner: TrPageSetupDialog);
begin
  FOwnerDlg := Owner;
end;

destructor TrPSRect.Destroy;
begin
  FOwnerDlg := nil;
  inherited Destroy;
end;

{$IFDEF VER125}
function TrPSRect.GetLeft: Integer;
begin Result := FRect.Left; end;

procedure TrPSRect.SetLeft(Value: Integer);
begin FRect.Left := Value; end;

function TrPSRect.GetRight: Integer;
begin Result := FRect.Right; end;

procedure TrPSRect.SetRight(Value: Integer);
begin FRect.Right := Value; end;

function TrPSRect.GetTop: Integer;
begin Result := FRect.Top; end;

procedure TrPSRect.SetTop(Value: Integer);
begin FRect.Top := Value; end;

function TrPSRect.GetBottom: Integer;
begin Result := FRect.Bottom; end;

procedure TrPSRect.SetBottom(Value: Integer);
begin FRect.Bottom := Value; end;
{$ENDIF}

function TrPSRect.GetRealRect(Index: Integer): Double;
var
  Tmp: TrPSMeasurements;
begin
  if Assigned(FOwnerDlg)
  then Tmp := FOwnerDlg.CurrentMeasurements
  else Tmp := DefMeasurements;
  case Index of
    0: Result := FRect.Left / MeasurementsDiv[Tmp];
    1: Result := FRect.Right / MeasurementsDiv[Tmp];
    2: Result := FRect.Top / MeasurementsDiv[Tmp];
    3: Result := FRect.Bottom / MeasurementsDiv[Tmp];
  else Result := 0;
  end;
end;

procedure TrPSRect.SetRealRect(Index: Integer; Value: Double);
var
  Tmp: TrPSMeasurements;
begin
  if Assigned(FOwnerDlg)
  then Tmp := FOwnerDlg.CurrentMeasurements
  else Tmp := DefMeasurements;
  case Index of
    0: FRect.Left   := Round(Value * MeasurementsDiv[Tmp]);
    1: FRect.Right  := Round(Value * MeasurementsDiv[Tmp]);
    2: FRect.Top    := Round(Value * MeasurementsDiv[Tmp]);
    3: FRect.Bottom := Round(Value * MeasurementsDiv[Tmp]);
  end;
end;

function TrPSRect.Compare(Other: TrPSRect): Boolean;
begin Result := EqualRect(FRect, Other.Rect); end;

constructor TrPSPoint.Create(Owner: TrPageSetupDialog);
begin
  FOwnerDlg := Owner;
end;

destructor TrPSPoint.Destroy;
begin
  FOwnerDlg := nil;
  inherited Destroy;
end;

function TrPSPoint.Compare(Other: TrPSPoint): Boolean;
begin
  Result := (FPoint.X = Other.Point.X) and (FPoint.Y = Other.Point.Y);
end;

function TrPSPoint.GetX: longint;
begin Result := FPoint.X; end;

procedure TrPSPoint.SetX(Val: longint);
begin FPoint.X := Val; end;

function TrPSPoint.GetY: longint;
begin Result := FPoint.Y; end;

procedure TrPSPoint.SetY(Val: longint);
begin FPoint.Y := Val; end;

function TrPSPoint.GetRealCoord(Index: Integer): Double;
var
  Tmp: TrPSMeasurements;
begin
  if Assigned(FOwnerDlg)
  then Tmp := FOwnerDlg.CurrentMeasurements
  else Tmp := DefMeasurements;
  case Index of
    0: Result := FPoint.X / MeasurementsDiv[Tmp];
    1: Result := FPoint.Y / MeasurementsDiv[Tmp];
  else Result := 0;
  end;
end;

procedure TrPSPoint.SetRealCoord(Index: Integer; Value: Double);
var
  Tmp: TrPSMeasurements;
begin
  if FOwnerDlg<>nil
  then Tmp := FOwnerDlg.CurrentMeasurements
  else Tmp := DefMeasurements;
  case Index of
    0: FPoint.X := Round(Value * MeasurementsDiv[Tmp]);
    1: FPoint.Y := Round(Value * MeasurementsDiv[Tmp]);
  end;
end;

constructor TrPageSetupDialog.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  FPrinter := Printers.Printer;

  if Assigned(FPrinter) and (pcOrientation in FPrinter.Capabilities)
  then FOrientation := FPrinter.Orientation
  else FOrientation := poPortrait;

  FCentered        := TRUE;
  FOptions         := [poDefaultMinMargins];
  FOnPrinter       := nil;
  FOnInitPaintPage := nil;
  FOnPaintPage     := nil;
  FCustomData      := 0;
  FMeasurements    := pmDefault;
  FPaperSize       := TrPSPoint.Create(Self);
  FMinimumMargins  := TrPSRect.Create(Self);
  FMargins         := TrPSRect.Create(Self);
  if CurrentMeasurements = pmInches
  then begin
    FPaperSize.Point     := DefPaperSizeI.Point;
    FMinimumMargins.Rect := DefMinimumMarginsI.Rect;
    FMargins.Rect        := DefMarginsI.Rect;
  end
  else begin
    FPaperSize.Point     := DefPaperSizeM.Point;
    FMinimumMargins.Rect := DefMinimumMarginsM.Rect;
    FMargins.Rect        := DefMarginsM.Rect;
  end;
end;

destructor TrPageSetupDialog.Destroy;
begin
  FPaperSize.Free;
  FMinimumMargins.Free;
  FMargins.Free;

  inherited Destroy;
end;

procedure TrPageSetupDialog.SetName(const NewName: TComponentName);
begin
  inherited Setname(NewName);
  if not (csLoading in ComponentState)
  then ReadCurrentValues;
end;

procedure TrPageSetupDialog.SetPaperSize(const Val: TrPSPoint);
begin FPaperSize.Point := Val.Point; end;

function TrPageSetupDialog.StorePaperSize: Boolean;
begin
  if CurrentMeasurements = pmInches
  then Result := not PaperSize.Compare(DefPaperSizeI)
  else Result := not PaperSize.Compare(DefPaperSizeM);
end;

procedure TrPageSetupDialog.SetMinimumMargins(const Val: TrPSRect);
begin
  FMinimumMargins.Rect := Val.Rect;
  ReadCurrentValues;
end;

function TrPageSetupDialog.StoreMinimumMargins: Boolean;
begin
  if CurrentMeasurements = pmInches
  then Result := not MinMargins.Compare(DefMinimumMarginsI)
  else Result := not MinMargins.Compare(DefMinimumMarginsM);
end;

procedure TrPageSetupDialog.SetMargins(const Val: TrPSRect);
begin FMargins.Rect := Val.Rect; end;

function TrPageSetupDialog.StoreMargins: Boolean;
begin
  if CurrentMeasurements = pmInches
  then Result := not Margins.Compare(DefMarginsI)
  else Result := not Margins.Compare(DefMarginsM);
end;

procedure TrPageSetupDialog.SetMeasurements(Val: TrPSMeasurements);
var
  TempVal: TrPSMeasurements;
  Coef: Double;
begin
  if Val = pmDefault
  then TempVal := DefaultMeasurements
  else TempVal := Val;
  if CurrentMeasurements <> TempVal
  then begin
    if TempVal = pmInches then
      //   
      Coef := 1 / 2.54
    else
      //  
      Coef := 2.54;
    with PaperSize do begin
      X := Round(X * Coef);
      Y := Round(Y * Coef);
    end;
    with MinMargins do begin
      absTop    := Round(absTop * Coef);
      absLeft   := Round(absLeft * Coef);
      absRight  := Round(absRight * Coef);
      absBottom := Round(absBottom * Coef);
    end;
    with Margins do begin
      absTop    := Round(absTop * Coef);
      absLeft   := Round(absLeft * Coef);
      absRight  := Round(absRight * Coef);
      absBottom := Round(absBottom * Coef);
    end;
  end;
  FMeasurements := Val;
  ReadCurrentValues;
end;

function TrPageSetupDialog.GetDefaultMeasurements: TrPSMeasurements;
begin
  if GetLocaleChar(LOCALE_USER_DEFAULT,LOCALE_IMEASURE,'0') = '0'
  then Result:= pmMillimeters
  else Result:= pmInches;
end;

function TrPageSetupDialog.GetCurrentMeasurements: TrPSMeasurements;
begin
  if FMeasurements = pmDefault
  then Result := DefaultMeasurements
  else Result := FMeasurements;
end;

procedure TrPageSetupDialog.GetPrinter(var DeviceMode, DeviceNames: THandle);
var
  Device, Driver, Port: array[0..79] of char;
  DevNames: PDevNames;
  Offset: PChar;
begin
  try
    FPrinter.GetPrinter(Device, Driver, Port, DeviceMode);
  except
    DeviceMode := 0;
  end;
  if DeviceMode <> 0
  then begin
    DeviceNames := GlobalAlloc(GHND, SizeOf(TDevNames) + StrLen(Device) +
       StrLen(Driver) + StrLen(Port) + 3);
    DevNames := PDevNames(GlobalLock(DeviceNames));
    try
      Offset := PChar(DevNames) + SizeOf(TDevnames);
      with DevNames^ do begin
        wDriverOffset := Longint(Offset) - Longint(DevNames);
        Offset := StrECopy(Offset, Driver) + 1;
        wDeviceOffset := Longint(Offset) - Longint(DevNames);
        Offset := StrECopy(Offset, Device) + 1;
        wOutputOffset := Longint(Offset) - Longint(DevNames);;
        StrCopy(Offset, Port);
      end;
    finally
      GlobalUnlock(DeviceNames);
    end;
  end;
end;

procedure TrPageSetupDialog.SetPrinter(DeviceMode, DeviceNames: THandle);
var
  DevNames: PDevNames;
begin
  DevNames := PDevNames(GlobalLock(DeviceNames));
  try
    with DevNames^ do
      FPrinter.SetPrinter(PChar(DevNames) + wDeviceOffset, PChar(DevNames) +
         wDriverOffset, PChar(DevNames) + wOutputOffset, DeviceMode);
  finally
    GlobalUnlock(DeviceNames);
    GlobalFree(DeviceNames);
  end;
end;

function CopyData(Handle: THandle): THandle;
var
  Src, Dest: PChar;
  Size: Integer;
begin
  Result := 0;
  if Handle <> 0
  then begin
    Size := GlobalSize(Handle);
    Result := GlobalAlloc(GHND, Size);
    if Result <> 0
    then try
      Src := GlobalLock(Handle);
      Dest := GlobalLock(Result);
      if (Src <> nil) and (Dest <> nil)
      then Move(Src^, Dest^, Size);
    finally
      GlobalUnlock(Handle);
      GlobalUnlock(Result);
    end
  end
end;

function TrPageSetupDialog.DoExecute(Func: Pointer): Boolean;
const
  PageSetupOptions: array [TrPageSetupOption] of DWord = (
     PSD_DEFAULTMINMARGINS, PSD_DISABLEMARGINS, PSD_DISABLEORIENTATION,
     PSD_DISABLEPAGEPAINTING, PSD_DISABLEPAPER, PSD_DISABLEPRINTER,
     PSD_NOWARNING, PSD_SHOWHELP
    );
  PageSetupMeasurements: array [TrPSMeasurements] of DWord = (
     0, PSD_INHUNDREDTHSOFMILLIMETERS, PSD_INTHOUSANDTHSOFINCHES
    );
var
  Option: TrPageSetupOption;
  PageSetup: TPageSetupDlg;
  SavePageSetupDialog: TrPageSetupDialog;
  DevHandle: THandle;
  PDM: PDeviceMode;
  Temp: Short;
begin
  FillChar(PageSetup, SizeOf(PageSetup), 0);
  with PageSetup do try
    if FGettingDefaults and (FPrinter.Printers.Count < 1)
    then begin
      //   
      ptPaperSize := Point(8500, 11000); // 8 1/2" X 11"
      rtMinMargin := Rect(250, 250, 250, 250); // 1/4"
      rtMargin := rtMinMargin; // 1/4"
      Result := TRUE;
    end
    else begin
      PageSetup.hInstance := HInstance;
      lStructSize := SizeOf(TPageSetupDlg);

      if FGettingDefaults
      then
        Flags := PSD_MARGINS or PSD_DEFAULTMINMARGINS or PSD_RETURNDEFAULT or
           PSD_INTHOUSANDTHSOFINCHES
      else begin
        Flags := PSD_MARGINS;
        Flags := Flags OR PageSetupMeasurements[CurrentMeasurements];
        if not (poDefaultMinMargins in FOptions)
        then Flags := Flags or PSD_MINMARGINS;

        if Assigned(FOnPrinter) or Assigned(FOnInitPaintPage) or
           Assigned(FOnPaintPage) or FCentered
        then begin
          Flags := Flags or PSD_ENABLEPAGESETUPHOOK;
          lpfnPageSetupHook := PageSetupDialogHook;
        end;

        for Option := Low(Option) to High(Option) do
          if Option in FOptions
          then Flags := Flags OR PageSetupOptions[Option];
        if Assigned(FOnInitPaintPage) and Assigned(FOnPaintPage)
        then begin
          Flags := Flags OR PSD_ENABLEPAGEPAINTHOOK;
          lpfnPagePaintHook := PageSetupDialogHook;
        end;
        HookCtl3D := Ctl3D;
        lCustData := FCustomData;

        GetPrinter(DevHandle, hDevNames);
        hDevMode := CopyData(DevHandle);
        if pcOrientation in FPrinter.Capabilities then
        try
          PDM := GlobalLock(HDevMode);
          PDM.dmOrientation := Ord(FOrientation);
          Inc(PDM.dmOrientation);
        finally
          GlobalUnlock(HDevMode);
        end;

        ptPaperSize := FPaperSize.Point;
        rtMinMargin := FMinimumMargins.Rect;
        rtMargin := FMargins.Rect;
        if (Flags and PSD_MINMARGINS) <> 0 then begin
          if rtMargin.Left < rtMinMargin.Left then rtMargin.Left := rtMinMargin.Left;
          if rtMargin.Right < rtMinMargin.Right then rtMargin.Right := rtMinMargin.Right;
          if rtMargin.Top < rtMinMargin.Top then rtMargin.Top := rtMinMargin.Top;
          if rtMargin.Bottom < rtMinMargin.Bottom then rtMargin.Bottom := rtMinMargin.Bottom;
        end;
      end;

      hWndOwner := Application.Handle;

      SavePageSetupDialog := PageSetupDialog;
      PageSetupDialog := Self;
      if FGettingDefaults
      then Result := PageSetupDlg(PageSetup)
      else Result := TaskModalDialog(Func, PageSetup);
      PageSetupDialog := SavePageSetupDialog;
    end;

    if Result
    then begin
      if FGettingDefaults// and (CurrentMeasurements = pmMillimeters)
      then begin
        if (CurrentMeasurements = pmMillimeters)
        then begin
          if (csLoading in ComponentState) or
             (DefPaperSizeM.Compare(FPaperSize))
          then begin
            FPaperSize.X := Round(ptPaperSize.X * 2.54);
            FPaperSize.Y := Round(ptPaperSize.Y * 2.54);
         end;
          if (csLoading in ComponentState) or
             (DefMinimumMarginsM.Compare(FMinimumMargins))
          then begin
            FMinimumMargins.absLeft := Round(rtMinMargin.Left * 2.54);
            FMinimumMargins.absTop := Round(rtMinMargin.Top * 2.54);
            FMinimumMargins.absRight := Round(rtMinMargin.Right * 2.54);
            FMinimumMargins.absBottom := Round(rtMinMargin.Bottom * 2.54);
          end;
          if (csLoading in ComponentState) or
             (DefMarginsM.Compare(FMargins))
          then begin
            FMargins.absLeft := Round(rtMargin.Left * 2.54);
            FMargins.absTop := Round(rtMargin.Top * 2.54);
            FMargins.absRight := Round(rtMargin.Right * 2.54);
            FMargins.absBottom := Round(rtMargin.Bottom * 2.54);
          end;
        end;
        if pcOrientation in FPrinter.Capabilities then
        try
          PDM := GlobalLock(HDevMode);
          Temp := PDM.dmOrientation;
          Dec(Temp);
          Orientation := TPrinterOrientation(Temp);
        finally
          GlobalUnlock(HDevMode);
        end;
      end
      else begin
        FPaperSize.Point := ptPaperSize;
        FMinimumMargins.Rect := rtMinMargin;
        FMargins.Rect := rtMargin;
        if pcOrientation in FPrinter.Capabilities then
        try
          PDM := GlobalLock(HDevMode);
          Temp := PDM.dmOrientation;
          Dec(Temp);
          Orientation := TPrinterOrientation(Temp);
        finally
          GlobalUnlock(HDevMode);
        end;
      end;

      if not FGettingDefaults
      then SetPrinter(hDevMode, hDevNames);
    end
    else begin
      if hDevMode <> 0 then GlobalFree(hDevMode);
      if hDevNames <> 0 then GlobalFree(hDevNames);
    end;
  finally
  end;
end;

function TrPageSetupDialog.ReadCurrentValues: Boolean;
begin
  FGettingDefaults := TRUE;
  try
    Result := DoExecute(@PageSetupDlg)
  finally
    FGettingDefaults := FALSE;
  end;
end;

function TrPageSetupDialog.FromMeasurementVal(Val: Integer): Double;
begin Result := Val / MeasurementsDiv[CurrentMeasurements]; end;

function TrPageSetupDialog.ToMeasurementVal(Val: Double): Integer;
begin Result := Round(Val * MeasurementsDiv[CurrentMeasurements]); end;

function TrPageSetupDialog.Execute: Boolean;
begin
  FGettingDefaults := FALSE;
  Result := DoExecute(@PageSetupDlg);
end;

function TrPageSetupDialog._Printer(Wnd: HWND): Boolean;
begin
  Result :=  Assigned(FOnPrinter);
  if Result
  then FOnPrinter(Self, Wnd);
end;

function TrPageSetupDialog.DoPrinter(Wnd: HWND): Boolean;
begin
  try
    Result := _Printer(Wnd);
  except
    Result := FALSE;
    Application.HandleException(Self);
  end;
end;

function TrPageSetupDialog.GetPaperSizeType: TrPSPaperSize;
var
  Device, Driver, Port: array[0..79] of char;
  HDevMode: THandle;
  PDevMode: PDeviceMode;
begin
  Result := rpsDEFAULT;
  try
    FPrinter.GetPrinter(Device, Driver, Port, HDevMode);
  except
    HDevMode := 0;
  end;
  if HDevMode <> 0
  then try
    PDevMode := GlobalLock(HDevMode);
    Result := TrPSPaperSize(PDevMode.dmPaperSize);
  finally
    GlobalUnlock(HDevMode);
  end;
end;

function TrPageSetupDialog.GetPaperSizeName: String;
var
  Device, Driver, Port: array[0..79] of char;
  HDevMode: THandle;
  PDevMode: PDeviceMode;
begin
  Result := '';
  try
    FPrinter.GetPrinter(Device, Driver, Port, HDevMode);
  except
    HDevMode := 0;
  end;
  if HDevMode <> 0
  then try
    PDevMode := GlobalLock(HDevMode);
    Result := PChar(@PDevMode.dmFormName);
  finally
    GlobalUnlock(HDevMode);
  end;
end;

procedure TrPageSetupDialog._SetPrinter(Value: TPrinter);
begin
  if Value <> FPrinter
  then begin
    FPrinter := Value;
    if FPrinter = nil then FPrinter := Printers.Printer;
//    FOrientation := FPrinter.Orientation;
    ReadCurrentValues;
  end;
end;

procedure InitGlobals;
var
  PageSetup: TPageSetupDlg;
  PDM: PDeviceMode;
begin
  HelpMsg := RegisterWindowMessage(HelpMsgString);

  DefPaperSizeI := TrPSPoint.Create(nil);
  DefMinimumMarginsI := TrPSRect.Create(nil);
  DefMarginsI := TrPSRect.Create(nil);
  DefOrientation := poPortrait;

  if Printers.Printer.Printers.Count > 0 then begin
    FillChar(PageSetup, SizeOf(PageSetup), 0);
    PageSetup.hInstance := HInstance;
    with PageSetup do begin
      lStructSize := SizeOf(TPageSetupDlg);
      hWndOwner := Application.Handle;
      Flags := PSD_MARGINS or PSD_DEFAULTMINMARGINS or PSD_INTHOUSANDTHSOFINCHES
         or PSD_RETURNDEFAULT;
      if PageSetupDlg(PageSetup)
      then begin
        DefPaperSizeI.Point := ptPaperSize;
        DefMinimumMarginsI.Rect := rtMinMargin;
        DefMarginsI.Rect := rtMargin;
        try
          PDM := GlobalLock(HDevMode);
          DefOrientation := TPrinterOrientation(PDM.dmOrientation-1);
        finally
          GlobalUnlock(HDevMode);
        end;
      end;
      if hDevMode <> 0
      then GlobalFree(hDevMode);
      if hDevNames <> 0
      then GlobalFree(hDevNames);
    end;
  end else begin
    DefPaperSizeI.Point := Point(8500, 11000); // 8 1/2" X 11" 
    DefMinimumMarginsI.Rect := Rect(250, 250, 250, 250); // 1/4"
    DefMarginsI.Rect := DefMinimumMarginsI.Rect; // 1/4"
  end;

  if GetLocaleChar(LOCALE_USER_DEFAULT,LOCALE_IMEASURE,'0') = '0'
  then DefMeasurements:= pmMillimeters
  else DefMeasurements:= pmInches;

  DefPaperSizeM := TrPSPoint.Create(nil);
  DefMinimumMarginsM := TrPSRect.Create(nil);
  DefMarginsM := TrPSRect.Create(nil);

  // 1/1000  -> 1/100 
  DefPaperSizeM.X := Round(DefPaperSizeI.X * 2.54);
  DefPaperSizeM.Y := Round(DefPaperSizeI.Y * 2.54);
  DefMinimumMarginsM.absTop    := Round(DefMinimumMarginsI.absTop * 2.54);
  DefMinimumMarginsM.absLeft   := Round(DefMinimumMarginsI.absLeft * 2.54);
  DefMinimumMarginsM.absRight  := Round(DefMinimumMarginsI.absRight * 2.54);
  DefMinimumMarginsM.absBottom := Round(DefMinimumMarginsI.absBottom * 2.54);
  DefMarginsM.absTop    := Round(DefMarginsI.absTop * 2.54);
  DefMarginsM.absLeft   := Round(DefMarginsI.absLeft * 2.54);
  DefMarginsM.absRight  := Round(DefMarginsI.absRight * 2.54);
  DefMarginsM.absBottom := Round(DefMarginsI.absBottom * 2.54);
end;

procedure DoneGlobals;
begin
  DefPaperSizeI.Free;
  DefMinimumMarginsI.Free;
  DefMarginsI.Free;
  DefPaperSizeM.Free;
  DefMinimumMarginsM.Free;
  DefMarginsM.Free;
end;

initialization
  InitGlobals;
finalization
  DoneGlobals;
end.
