//==============================================
//       rperiod.pas
//
//         Delphi.
//        .
//
//      Copyright 1998-2000 Polaris Software
//      http://members.xoom.com/PolarisSoft
//      mailto: PolarisLib@mail.ru
//==============================================
unit rPeriod;

{$I POLARIS.INC}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, Mask, DateUtil, rUtils, rButtons, Buttons, Spin, ExtCtrls,
  ToolEdit, rSpeedButton, rConst;

const
  HalfYearItems: array[1..2] of string[2] = ('I','II');
  QuartItems: array[1..4] of string[3] = ('I','II','III','IV');

type
  TrDates = array[1..2] of TrDate;

  TTypePeriod = (bdYear,bdHalfYear,bdQuart,bdMonth,bdWeek,bdDay,bdAny);

  TrPeriod = class
  public
    D1, D2: TDateTime;
    procedure AssignDate(dDate: TDateTime; dTypePeriod: TTypePeriod; Offset: Integer);
    procedure AssignDates(Date1,Date2: TrDate);
    procedure AssignNum(NumPeriod,Year: Integer; dTypePeriod: TTypePeriod);
    function GetName(NeedPrefix: Boolean): string;
    function GetType: TTypePeriod;
    function GetDates: TrDates;
  end;

  TrPeriodComboEdit = class(TrCustomComboEdit)
  private
    FPeriod: TrPeriod;
    FNeedPrefix: Boolean;
    function GetDate1: TDate;
    function GetDate2: TDate;
    procedure SetDate1(Value: TDate);
    procedure SetDate2(Value: TDate);
  protected
    procedure ButtonClick; override;
    function IsNotDefaultGR: Boolean;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property Period: TrPeriod read FPeriod;
    property Date1: TDate read GetDate1 write SetDate1;
    property Date2: TDate read GetDate2 write SetDate2;
    procedure AssignDate(dDate: TDateTime; dTypePeriod: TTypePeriod; Offset: Integer);
    procedure AssignNum(NumPeriod,Year: Integer; dTypePeriod: TTypePeriod);
  published
    property NumGlyphs default 2;
    property GlyphResKind;
    property GlyphResource stored IsNotDefaultGR;
    property OnCloseUp;
    property NeedPrefix: Boolean read FNeedPrefix write FNeedPrefix default True;
  end;

  TFormBandDate = class(TForm)
    RadioGroup1: TRadioGroup;
    SpinYear: TSpinEdit;
    ComboHalfYear: TComboBox;
    ComboQuart: TComboBox;
    BitBtn1: TrBitBtn;
    BitBtn2: TBitBtn;
    ComboMonth: TComboBox;
    SpinWeek: TSpinEdit;
    GroupBox1: TGroupBox;
    Label1: TLabel;
    Label2: TLabel;
    Bulb: TrSpeedButton;
    DateEdit1: TDateEdit;
    DateEdit2: TDateEdit;
    Label4: TLabel;
    Label5: TLabel;
    BitBtn3: TBitBtn;
    procedure DateEditExit(Sender: TObject);
    procedure ComboHalfYearChange(Sender: TObject);
    procedure SpinYearChange(Sender: TObject);
    procedure ComboQuartChange(Sender: TObject);
    procedure ComboMonthChange(Sender: TObject);
    procedure SpinWeekChange(Sender: TObject);
    procedure RadioGroup1Click(Sender: TObject);
    procedure BulbClick(Sender: TObject);
    procedure DateEdit1AcceptDate(Sender: TObject; var Date: TDateTime;
      var Action: Boolean);
    procedure BitBtn1Click(Sender: TObject);
    procedure FormKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure FormDestroy(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    function FormHelp(Command: Word; Data: Integer;
      var CallHelp: Boolean): Boolean;
  private
    FPeriod: TrPeriod; //  
    procedure SetCtrlEnabled;
    procedure SetCtrlValues;
    procedure RefreshDateEdits;
  public
    D1,D2: TDate;      //  
    constructor CreatePeriod(AOwner: TComponent; AD1,AD2: TDate);
  end;

function GetNumPeriod(dDate: TDateTime; dTypePeriod: TTypePeriod): Integer;
function EditPeriod(dPeriod: TrPeriod): TModalResult;

var
  FormBandDate: TFormBandDate;
  DMY: TrDates;

implementation

{$R *.DFM}

function GetNumPeriod(dDate: TDateTime; dTypePeriod: TTypePeriod): Integer;
begin
  with DMY[1] do begin
    DecodeDate(dDate,Y,M,D);
    case dTypePeriod of
      bdYear: Result := Y;
      bdHalfYear: Result := (M div 7) + 1;
      bdQuart: Result := ((M-1) div 3) + 1;
      bdMonth: Result := M;
      bdWeek: Result :=
        Trunc((dDate-(EncodeDate(Y,1,1)-DayOfWeek(EncodeDate(Y,1,1))+3))/7);
      bdDay,bdAny: Result := 0;
    else Result := 0;
    end;
  end;
end;

function EditPeriod(dPeriod: TrPeriod): TModalResult;
begin
  with TFormBandDate.CreatePeriod(Application,dPeriod.D1,dPeriod.D2) do
    try
      Result := ShowModal;
      if Result = mrOK then begin
        dPeriod.D1 := D1;
        dPeriod.D2 := D2;
      end;
    finally
      Free;
    end;
end;

{ TrPeriod }

procedure TrPeriod.AssignDate(dDate: TDateTime; dTypePeriod: TTypePeriod; Offset: Integer);
var
  mon: Integer;
begin
  D1 := dDate;
  with DMY[1] do
    if Offset <> 0 then begin
      DecodeDate(D1,Y,M,D);
      mon := M;
      //    Offset
      case dTypePeriod of
        bdYear: Y := Y+Offset;
        bdHalfYear: Y := Y+IncCycle(mon,6*Offset,1,12);
        bdQuart: Y := Y+IncCycle(mon,3*Offset,1,12);
        bdMonth: Y := Y+IncCycle(mon,Offset,1,12);
        bdWeek: begin
                  D1 := D1+7*Offset;
                  DecodeDate(D1,Y,M,D);
                  mon := M;
                end;
        bdDay,bdAny: begin
                       D1 := D1+Offset;
                       D2 := D1;
                       exit;
                     end;
      end;
      D1 := EncodeDate(Y,mon,D);
    end;
  AssignNum(GetNumPeriod(D1,dTypePeriod),DMY[1].Y,dTypePeriod);
end;

procedure TrPeriod.AssignDates(Date1,Date2: TrDate);
begin
  with Date1 do D1 := EncodeDate(Y,M,D);
  with Date2 do D2 := EncodeDate(Y,M,D);
end;

procedure TrPeriod.AssignNum(NumPeriod,Year: Integer; dTypePeriod: TTypePeriod);
begin
  with DMY[1] do begin
    D := 1;
    Y := Year;
    DMY[2].Y := Year;
    case dTypePeriod of
      bdYear: begin
                DMY[2].D := 31;
                M := 1;
                DMY[2].M := 12;
              end;
      bdHalfYear: begin
                    DMY[2].M := NumPeriod*6;
                    M := DMY[2].M-5;
                    DMY[2].D := DaysPerMonth(Y,DMY[2].M);
                  end;
      bdQuart: begin
                 DMY[2].M := NumPeriod*3;
                 M := DMY[2].M-2;
                 DMY[2].D := DaysPerMonth(Y,DMY[2].M);
               end;
      bdMonth: begin
                 DMY[2].M := NumPeriod;
                 M := DMY[2].M;
                 DMY[2].D := DaysPerMonth(Y,DMY[2].M);
               end;
      bdWeek: begin
                D1 := EncodeDate(Y,1,1)+NumPeriod*7 -
                             DayOfweek(EncodeDate(Y,1,1))+2;
                D2 := D1+6;
                exit;
              end;
      else exit;
    end;
  end;
  AssignDates(DMY[1],DMY[2]);
end;

function TrPeriod.GetName(NeedPrefix: Boolean): string;
var
  Prefix: string;
begin
  if NeedPrefix then Prefix := srPeriodPrefix+' ' else Prefix := '';
  with DMY[1] do begin
    DecodeDate(D2,Y,M,D);
    case GetType of
      bdYear:     Result := Format(srPeriodYear,[Prefix,Y]);
      bdHalfYear: Result := Format(srPeriodHalfYear,[Prefix,HalfYearItems[M div 7 +1],Y]);
      bdQuart:    Result := Format(srPeriodQrt,[Prefix,QuartItems[(M-1) div 3 +1],Y]);
      bdMonth:    Result := Format(srPeriodMon,[Prefix,LongMonthNames[M],Y]);
      bdDay:      Result := Format(srPeriodDay,[Prefix,AdaptedDateStr(D1)]);
    else
      if NeedPrefix then
        Result := Format(srPeriodCustomPrfx,[DateToStr(D1),DateToStr(D2)])
      else
        Result := Format(srPeriodCustom,[DateToStr(D1),DateToStr(D2)]);
    end;
  end;
end;

function TrPeriod.GetType: TTypePeriod;
begin
  with DMY[1] do DecodeDate(D1,Y,M,D);
  with DMY[2] do begin
    DecodeDate(D2,Y,M,D);
    if (DMY[1].Y=Y) then
      {  }
      if (DMY[1].D=D) and (DMY[1].M=M) then
        Result := bdDay
      else
        if (DMY[1].D=1) then
          {  }
          if (DMY[1].M=1) and (M=12) and (D=31) then
            Result := bdYear
          {  }
          else if ((DMY[1].M=1) and (D=30) and (M=6)) or
                  ((DMY[1].M=7) and (D=31) and (M=12)) then
            Result := bdHalfYear
          {  }
          else if (DMY[1].M mod 3=1) and (D=DaysPerMonth(Y,M)) and
                  (M-DMY[1].M=2) and (M mod 3=0) then
            Result := bdQuart
          {  }
          else if (DMY[1].M=M) and (D=DaysPerMonth(Y,M)) then
            Result := bdMonth
          else Result := bdAny
        {  }
        else if (DayOfWeek(D1)=2) and (DayOfWeek(D2)=1) and
                (Trunc(D2-D1)=6) then
               Result := bdWeek
        else Result := bdAny
    else Result := bdAny;
  end;
end;

function TrPeriod.GetDates: TrDates;
begin
  with Result[1] do DecodeDate(D1,Y,M,D);
  with Result[2] do DecodeDate(D2,Y,M,D);
end;

{ TrPeriodComboEdit }

constructor TrPeriodComboEdit.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FPeriod := TrPeriod.Create;
  GlyphResource := srPeriodGlyph;
  NumGlyphs := 2;
  FNeedPrefix := True;
  Date1 := 0;
  Date2 := 0;
end;

destructor TrPeriodComboEdit.Destroy;
begin
  FPeriod.Free;
  inherited Destroy;
end;

procedure TrPeriodComboEdit.ButtonClick;
begin
  inherited;
  if EditPeriod(Period) = mrOK then begin
    Text := Period.GetName(FNeedPrefix);
    if Assigned(OnCloseUp) then OnCloseUp(Self);
  end;
end;

function TrPeriodComboEdit.IsNotDefaultGR: Boolean;
begin
  Result := UpperCase(GlyphResource) <> srPeriodGlyph;
end;

function TrPeriodComboEdit.GetDate1: TDate;
begin
  Result := FPeriod.D1;
end;

function TrPeriodComboEdit.GetDate2: TDate;
begin
  Result := FPeriod.D2;
end;

procedure TrPeriodComboEdit.SetDate1;
begin
  if FPeriod.D1 <> Value then begin
    FPeriod.D1 := Value;
    Text := FPeriod.GetName(FNeedPrefix);
  end;
end;

procedure TrPeriodComboEdit.SetDate2;
begin
  if FPeriod.D2 <> Value then begin
    FPeriod.D2 := Value;
    Text := FPeriod.GetName(FNeedPrefix);
  end;
end;

procedure TrPeriodComboEdit.AssignDate(dDate: TDateTime; dTypePeriod: TTypePeriod; Offset: Integer);
begin
  FPeriod.AssignDate(dDate, dTypePeriod, Offset);
  Text := FPeriod.GetName(FNeedPrefix);
end;

procedure TrPeriodComboEdit.AssignNum(NumPeriod,Year: Integer; dTypePeriod: TTypePeriod);
begin
  FPeriod.AssignNum(NumPeriod, Year, dTypePeriod);
  Text := FPeriod.GetName(FNeedPrefix);
end;

{ TFormBandDate }

constructor TFormBandDate.CreatePeriod(AOwner: TComponent; AD1,AD2: TDate);
var
  i: Integer;
begin
  FPeriod := TrPeriod.Create;
  FPeriod.D1 := AD1;
  FPeriod.D2 := AD2;
  D1 := AD1;
  D2 := AD2;
  inherited Create(AOwner);
  for i:=1 to 2 do ComboHalfYear.Items.Add(HalfYearItems[i]);
  for i:=1 to 4 do ComboQuart.Items.Add(QuartItems[i]);
  for i:=1 to 12 do ComboMonth.Items.Add(LongMonthNames[i]);
  SetCtrlValues;
  RefreshDateEdits;
end;

procedure TFormBandDate.SetCtrlEnabled;
begin
  with RadioGroup1  do begin
    SpinYear.Enabled := (ItemIndex<>Ord(bdAny))and(ItemIndex<>Ord(bdWeek));
    ComboHalfYear.Enabled := (ItemIndex=Ord(bdHalfYear));
    ComboQuart.Enabled := (ItemIndex=Ord(bdQuart));
    ComboMonth.Enabled := (ItemIndex in [Ord(bdMonth),Ord(bdDay)]);
    SpinWeek.Enabled := (ItemIndex=Ord(bdWeek));
  end;
end;

procedure TFormBandDate.SetCtrlValues;
begin
  DMY := FPeriod.GetDates;
  with DMY[2] do begin
    SpinYear.Value := Y;
    ComboHalfYear.ItemIndex := M div 7;
    ComboQuart.ItemIndex := (M-1) div 3;
    ComboMonth.ItemIndex := M-1;
    SpinWeek.Value := Trunc((FPeriod.D2-
            (EncodeDate(Y,1,1) - DayOfWeek(EncodeDate(Y,1,1))+3))/7);
  end;
  RadioGroup1.ItemIndex := Ord(FPeriod.GetType);
  Bulb.Down := (DMY[1].D=1) and (DMY[1].M=1);
end;

procedure TFormBandDate.RefreshDateEdits;
begin
  DateEdit1.Date := FPeriod.D1;
  DateEdit2.Date := FPeriod.D2;
end;

procedure TFormBandDate.DateEditExit(Sender: TObject);
begin
  if Sender = DateEdit1 then
    FPeriod.D1 := TDateEdit(Sender).Date
  else begin
    FPeriod.D2 := TDateEdit(Sender).Date;
    if RadioGroup1.ItemIndex=5 then begin
      FPeriod.D1 := FPeriod.D2;
      DateEdit1.Date := FPeriod.D2;
    end;
  end;
  SetCtrlValues;
  SetCtrlEnabled;
end;

procedure TFormBandDate.SpinYearChange(Sender: TObject);
begin
  if RadioGroup1.ItemIndex = Ord(bdYear) then
    FPeriod.AssignNum(SpinYear.Value,SpinYear.Value,bdYear)
  else if SpinYear.Enabled then begin
    with DMY[1] do begin
      DecodeDate(FPeriod.D1,Y,M,D);
      Y := SpinYear.Value;
      FPeriod.D1 := EncodeDate(Y,M,D);
    end;
    with DMY[2] do begin
      DecodeDate(FPeriod.D2,Y,M,D);
      FPeriod.D2 := EncodeDate(DMY[1].Y,M,D);
    end;
  end;
  RefreshDateEdits;
end;

procedure TFormBandDate.ComboHalfYearChange(Sender: TObject);
begin
  with ComboHalfYear do begin
    Bulb.Down := (ItemIndex=0);
    ComboQuart.ItemIndex := ItemIndex*2+1;
    ComboMonth.ItemIndex := ItemIndex*6+5;
    FPeriod.AssignNum(ItemIndex+1,SpinYear.Value,bdHalfYear);
  end;
  RefreshDateEdits;
end;

procedure TFormBandDate.ComboQuartChange(Sender: TObject);
begin
  with ComboQuart do begin
    Bulb.Down := (ItemIndex=0);
    ComboHalfYear.ItemIndex := ItemIndex div 2;
    ComboMonth.ItemIndex := ItemIndex*3+2;
    FPeriod.AssignNum(ItemIndex+1,SpinYear.Value,bdQuart);
  end;
  RefreshDateEdits;
end;

procedure TFormBandDate.ComboMonthChange(Sender: TObject);
begin
  if RadioGroup1.ItemIndex = Ord(bdMonth) then
    with ComboMonth do begin
      Bulb.Down := (ItemIndex=0);
      ComboHalfYear.ItemIndex := ItemIndex div 6;
      ComboQuart.ItemIndex := ItemIndex div 3;
      FPeriod.AssignNum(ItemIndex+1,SpinYear.Value,bdMonth);
    end
  else begin
    with DMY[1] do begin
      DecodeDate(FPeriod.D1,Y,M,D);
      M := ComboMonth.ItemIndex+1;
      FPeriod.D1 := EncodeDate(Y,M,D);
    end;
    with DMY[2] do begin
      DecodeDate(FPeriod.D2,Y,M,D);
      FPeriod.D2 := EncodeDate(Y,DMY[1].M,D);
    end;
  end;
  RefreshDateEdits;
end;

procedure TFormBandDate.SpinWeekChange(Sender: TObject);
begin
  if RadioGroup1.ItemIndex = Ord(bdWeek) then begin
    FPeriod.AssignNum(SpinWeek.Value,SpinYear.Value,bdWeek);
    with DMY[2] do begin
      DecodeDate(FPeriod.D2,Y,M,D);
      ComboHalfYear.ItemIndex := M div 6;
      ComboQuart.ItemIndex := M div 3;
    end;
    SetCtrlValues;
    RefreshDateEdits;
  end;
end;

procedure TFormBandDate.RadioGroup1Click(Sender: TObject);
begin
  SetCtrlEnabled;
  DateEdit1.Enabled := RadioGroup1.ItemIndex <> 5;
  case RadioGroup1.ItemIndex of
    0: begin
         SpinYearChange(Sender);
         Bulb.Down := True;
       end;
    1: ComboHalfYearChange(Sender);
    2: ComboQuartChange(Sender);
    3: ComboMonthChange(Sender);
    4: SpinWeekChange(Sender);
    5: begin
         FPeriod.D1 := FPeriod.D2;
         with DMY[1] do begin
           DecodeDate(FPeriod.D1,Y,M,D);
           Bulb.Down := (D=1) and (M=1);
         end;
         RefreshDateEdits;
       end;
  end;
end;

procedure TFormBandDate.BulbClick(Sender: TObject);
begin
  if Bulb.Down then begin
    with DMY[1] do begin
      D := 1;
      M := 1;
      FPeriod.D1 := EncodeDate(SpinYear.Value,M,D);
    end;
    SetCtrlValues;
    SetCtrlEnabled;
    RefreshDateEdits;
  end else Bulb.Down := True;
end;

procedure TFormBandDate.DateEdit1AcceptDate(Sender: TObject;
  var Date: TDateTime; var Action: Boolean);
begin
  if Action then begin
    TDateEdit(Sender).Date := Date;
    DateEditExit(Sender);
  end;
end;

procedure TFormBandDate.BitBtn1Click(Sender: TObject);
begin
  if FPeriod.D1 > Fperiod.D2 then begin
    rMsgBox(srPeriodErr, MB_OK+MB_ICONHAND);
    exit;
  end;
  D1 := FPeriod.D1;
  D2 := FPeriod.D2;
  ModalResult := mrOK;
end;

procedure TFormBandDate.FormKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  if (Key = VK_RETURN) and (ssCtrl in Shift) then BitBtn1.Click;
end;

procedure TFormBandDate.FormDestroy(Sender: TObject);
begin
  FPeriod.Free;
end;

procedure TFormBandDate.FormCreate(Sender: TObject);
begin
  HelpFile := srHelpFile;
end;

function TFormBandDate.FormHelp(Command: Word; Data: Integer;
  var CallHelp: Boolean): Boolean;
begin
  CallHelp := not ((Command = HELP_CONTEXTPOPUP) and (Data = HelpContext));
  Result := True;
end;

end.
