//==============================================
//       rlocatedlg.pas
//
//         Delphi.
//         DataSet.
//
//      Copyright 1998-2000 Polaris Software
//      http://members.xoom.com/PolarisSoft
//      mailto: PolarisLib@mail.ru
//==============================================
unit rLocateDlg;

{$I POLARIS.INC}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  DB, {DBTables, }StdCtrls, ComCtrls, Buttons, rButtons, dbGrids,
  rUtils, rDBUtils, rConst, rDBConst;

type
  TOnCompareEvent = function (OnField: TField;
                              Text: String;
                              CaseSensitive: Boolean): Boolean of object;

  TrLocDataLink = class;

  TrLocateDlg = class(TComponent)
  private
    { Private declarations }
    FDBGrid : TDBGrid;
    FDataLink  : TrLocDataLink;
    FNextEnable: Boolean;
    FAccessed  : Boolean;
    FValueList : TStrings;
    LastField  : String;
    LastValue  : String;
    FLastDS    : TDataSet;
    FVisible   : Boolean;
    FCaseSens  : Boolean;
    FAtEntry   : Boolean;
    FFindCtrl  : TControl;
    FNextCtrl  : TControl;
    FOnCompare : TOnCompareEvent;
    FMaxItems  : Integer;

    function GetDataSource: TDataSource;
    procedure SetDataSource(Value: TDataSource);
    function GetAccessed: Boolean;
    procedure SetFindCtrl(Value: TControl);
    procedure SetNextCtrl(Value: TControl);
    procedure SetMaxItems(Value: Integer);
    procedure SetDBGrid(Value: TDBGrid);
  protected
    { Protected declarations }
    FLastPos: TBookMark;
    procedure AccessedChanged;
    procedure Notification(AComponent: TComponent;
                           Operation: TOperation); override;
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

    procedure Execute(GoFirst: Boolean);
    procedure ClearBuffer;
    property Accessed: Boolean read GetAccessed;
    property NextEnable: Boolean read FNextEnable;
  published
    { Published declarations }
    property DataSource: TDataSource read GetDataSource write SetDataSource;
    property DBGrid: TDBGrid read FDBGrid write SetDBGrid;
    property OnlyVisible: Boolean read FVisible write FVisible default True;
    property FindControl: TControl read FFindCtrl write SetFindCtrl;
    property NextControl: TControl read FNextCtrl write SetNextCtrl;
    property OnCompare: TOnCompareEvent read FOnCompare write FOnCompare;
    property AtEntry: Boolean read FAtEntry write FAtEntry default False;
    property MaxBufferItems: Integer read FMaxItems write SetMaxItems default 15;
  end;

  TrLocDataLink = class(TDataLink)
  private
    FLocateDlg: TrLocateDlg;
  protected
    procedure EditingChanged; override;
    procedure DataSetChanged; override;
    procedure ActiveChanged; override;
  public
    constructor Create(ALocateDlg: TrLocateDlg);
    destructor Destroy; override;
  end;

  TrCommonFindDlg = class(TForm)
    OKBtn: TrBitBtn;
    CancelBtn: TrBitBtn;
    Animate1: TAnimate;
    GroupBox1: TGroupBox;
    Label1: TLabel;
    FieldBox: TComboBox;
    Label2: TLabel;
    CaseSens: TCheckBox;
    ValueBox: TComboBox;
    FindAtEntry: TCheckBox;
    BitBtn1: TBitBtn;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure CancelBtnClick(Sender: TObject);
    procedure OKBtnClick(Sender: TObject);
    procedure FieldBoxChange(Sender: TObject);
    procedure ValueBoxExit(Sender: TObject);
    procedure FormDeactivate(Sender: TObject);
    procedure ValueBoxChange(Sender: TObject);
    procedure FormPaint(Sender: TObject);
    function FormHelp(Command: Word; Data: Integer;
      var CallHelp: Boolean): Boolean;
  private
    { Private declarations }
    FCanceled,
    FProcessed: Boolean;
    FLastField: String;

    function DoCompare(OnField: TField; Text: String): Boolean;
    procedure DoFind;
    function GetField: TField;
    function GetAtEntry: Boolean;
    procedure SetAtEntry(Value: Boolean);
  public
    { Public declarations }
    FindDataSet: TDataSet;
    VisibleOnly: Boolean;
    GoFirst    : Boolean;
    FMaxItems  : Integer;
    OnCompare  : TOnCompareEvent;

    procedure Initialized(AGrid          : TDBGrid;
                          ADataSet       : TDataSet;
                          ACaseSens      : Boolean;
                          AVisibleOnly   : Boolean;
                          AGoFirst       : Boolean;
                          AAtEntry       : Boolean;
                          AValueList     : TStrings;
                          ALastValue     : String;
                          ALastField     : String;
                          AMaxItems      : Integer);
    property AtEntry: Boolean read GetAtEntry write SetAtEntry;
  end;

var
  rCommonFindDlg: TrCommonFindDlg;

implementation

{$R *.DFM}

type
  TCrackDataSet=class(TDataSet);

function CompareBM(DataSet:  TDataSet;
                   BM1, BM2: TBookMark): Integer;
var
  DS: TCrackDataSet;
begin
//  with DataSet as TBDEDataSet do
//    Result := CompareBookmarks( BM1, BM2);
  Result := 1;
  DS := TCrackDataSet(DataSet);
  with DS do
    if CompareMem(BM1, BM2, BookmarkSize) then Result := 0;
end;

{ TrLocDataLink }

constructor TrLocDataLink.Create(ALocateDlg: TrLocateDlg);
begin
  inherited Create;
  FLocateDlg := ALocateDlg;
end;

destructor TrLocDataLink.Destroy;
begin
  FLocateDlg := nil;
  inherited Destroy
end;

procedure TrLocDataLink.EditingChanged;
begin
  if FLocateDlg <> nil
  then FLocateDlg.AccessedChanged;
end;

procedure TrLocDataLink.DataSetChanged;
begin
  if FLocateDlg <> nil
  then FLocateDlg.AccessedChanged;
end;

procedure TrLocDataLink.ActiveChanged;
begin
  if FLocateDlg <> nil
  then FLocateDlg.AccessedChanged;
end;

{ TrLocateDlg }

constructor TrLocateDlg.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FDataLink  := TrLocDataLink.Create(Self);
  FDBGrid    := nil;
  FLastDS    := nil;
  FNextEnable:= False;
  FAccessed  := False;
  FVisible   := True;
  FCaseSens  := True;
  FAtEntry   := False;
  FValueList := TStringList.Create;
  LastField  := '';
  LastValue  := '';
  FOnCompare := nil;
  FLastPos   := nil;
  FFindCtrl  := nil;
  FNextCtrl  := nil;
  FMaxItems  := 15;
end;

destructor TrLocateDlg.Destroy;
begin
  {$IFDEF POLARIS_D4}
  if (FLastPos <> nil) and
     Assigned(FDataLink) and
     Assigned(FDataLink.DataSet) and
     FDataLink.DataSet.BookmarkValid(FLastPos)
  then FDataLink.DataSet.FreeBookmark(FLastPos);
  {$ELSE}
  if FLastPos <> nil
  then StrDispose(FLastPos);
  {$ENDIF}
  FDataLink.Free;
  FValueList.Free;
  FDBGrid    := nil;
  FDataLink  := nil;
  FValueList := nil;
  FLastDS    := nil;
  FFindCtrl  := nil;
  FNextCtrl  := nil;
  inherited Destroy;
end;

procedure TrLocateDlg.Notification(AComponent: TComponent;
                                    Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove)
  then begin
    if (FDBGrid <> nil) and
       (AComponent = DBGrid) then DBGrid := nil;
    if (FDataLink <> nil) and
//       (AComponent = DataSource) then DataSource := nil;
       (AComponent = FDataLink.DataSource) then FDataLink.DataSource := nil;
    if (FFindCtrl <> nil) and
       (AComponent = FindControl) then FindControl := nil;
    if (FNextCtrl <> nil) and
       (AComponent = NextControl) then FindControl := nil;
  end;
end;

procedure TrLocateDlg.SetDataSource(Value: TDataSource);
begin
//  if (Value <> nil) and ((FDBGrid <> nil) and (FDBGrid.DataSource<>Value))
//  then Value := FDBGrid.DataSource;
  if (Value <> nil)
  then FDBGrid := nil;
  FDataLink.DataSource := Value;
  if not (csLoading in ComponentState)
  then AccessedChanged;
  if Value <> nil then Value.FreeNotification(Self)
end;

procedure TrLocateDlg.SetDBGrid(Value: TDBGrid);
var
  FDS: TDataSource;
begin
  if FDBGrid <> nil then FDS := nil else FDS := DataSource;
  DataSource := nil;
  if Value <> FDBGrid then begin
    FDBGrid := Value;
    if Value <> nil then Value.FreeNotification(Self)
  end;
  if FDBGrid <> nil then FDS := FDBGrid.DataSource;
  FDataLink.DataSource := FDS;
  if FDS <> nil then FDS.FreeNotification(Self)
end;

procedure TrLocateDlg.SetFindCtrl(Value: TControl);
begin
  if FFindCtrl <> Value
  then begin
    FFindCtrl := Value;
    if not (csLoading in ComponentState)
    then AccessedChanged;
    if Value <> nil then Value.FreeNotification(Self)
  end
end;

procedure TrLocateDlg.SetNextCtrl(Value: TControl);
begin
  if FNextCtrl <> Value
  then begin
    FNextCtrl := Value;
    if not (csLoading in ComponentState)
    then AccessedChanged;
    if Value <> nil then Value.FreeNotification(Self)
  end
end;

function TrLocateDlg.GetDataSource: TDataSource;
begin
//  Result := FDataLink.DataSource;
  Result := nil;
  if csDesigning in ComponentState then begin
    if FDBGrid = nil then Result := FDataLink.DataSource;
  end
  else
    if FDBGrid = nil then Result := FDataLink.DataSource
    else Result := FDBGrid.DataSource
end;

procedure TrLocateDlg.AccessedChanged;
begin
  if FLastDS <> FDataLink.DataSet
  then begin
//    FValueList.Clear;
    LastValue   := '';
    LastField   := '';
    FNextEnable := False;
    FLastDS     := FDataLink.DataSet;
  end;

//  FAccessed := (DataSource <> nil) and (DataSource.DataSet <> nil) and FDataLink.Active and not FDataLink.Editing;
  FAccessed := (FDataLink.DataSource <> nil) and (FDataLink.DataSet <> nil) and FDataLink.Active and not FDataLink.Editing;
  if FFindCtrl <> nil
  then FFindCtrl.Enabled := FAccessed;
  if FNextCtrl <> nil
  then FNextCtrl.Enabled := FAccessed and FNextEnable;
end;

function TrLocateDlg.GetAccessed: Boolean;
begin
  AccessedChanged;
  Result := FAccessed;
end;

procedure TrLocateDlg.Execute(GoFirst: Boolean);
var
  Dlg: TrCommonFindDlg;
  IsOK: Boolean;
  BM: TBookMark;
begin
  if Accessed
  then begin
    if not (NextEnable or GoFirst)
    then begin
      rMsgBox(srFirstLocateNeed, MB_OK+MB_ICONERROR);
      exit;
    end;
    IsOk := False;
    Dlg := TrCommonFindDlg.Create(Application);
    try
      FDataLink.DataSet.DisableControls;
      Dlg.Initialized( FDBGrid, FDataLink.DataSet, FCaseSens, FVisible, GoFirst,
                       FAtEntry, FValueList, LastValue, LastField, FMaxItems);
      BM := FDataLink.DataSet.GetBookMark;
      if (not GoFirst) and (CompareBM(FDataLink.DataSet, BM, FLastPos)=0)
      then FDataLink.DataSet.Next;
      if Dlg.ShowModal <> mrOK
      then FDataLink.DataSet.GotoBookMark(BM);
      {$IFDEF POLARIS_D4}
      if (BM<>nil) and FDataLink.DataSet.BookmarkValid(BM)
      then FDataLink.DataSet.FreeBookmark(BM);
      {$ELSE}
      if (BM<>nil) then StrDispose(BM);
      {$ENDIF}
      IsOk := True;
      if Dlg.ModalResult < 0
      then
        rMsgBox(Format(srNotFound,[Dlg.ValueBox.Text,Dlg.FieldBox.Text]),
                MB_OK+MB_ICONINFORMATION)
    finally
      FDataLink.DataSet.EnableControls;
      if IsOK
      then begin
      {$IFDEF POLARIS_D4}
        if (FLastPos <> nil) and FDataLink.DataSet.BookmarkValid(FLastPos)
        then FDataLink.DataSet.FreeBookmark(FLastPos);
      {$ELSE}
        if FLastPos <> nil
        then StrDispose(FLastPos);
      {$ENDIF}
        FLastPos := FDataLink.DataSet.GetBookMark;
        FCaseSens  := Dlg.CaseSens.Checked;
        LastField  := Trim(Dlg.FieldBox.Text);
        LastValue  := Trim(Dlg.ValueBox.Text);
        FNextEnable:= (Dlg.ModalResult <> mrCancel) and (LastValue <> '');
        FValueList.Assign(Dlg.ValueBox.Items);
        FAtEntry   := Dlg.AtEntry;
      end;
      Dlg.Free;
    end;
  end
  else raise Exception.Create(srLocateDenied);
  AccessedChanged;
end;

procedure TrLocateDlg.ClearBuffer;
begin
  FValueList.Clear;
end;

procedure TrLocateDlg.SetMaxItems(Value: Integer);
begin
  if (Value >= 0) and (Value <> FMaxItems)
  then FMaxItems := Value;
end;

{ TrCommonFindDlg }

procedure TrCommonFindDlg.FormCreate(Sender: TObject);
begin
  HelpFile   := srHelpFile;
  FMaxItems  := 15;
  FCanceled  := False;
  FProcessed := False;
  FLastField := '';
  FindDataSet:= nil;
  VisibleOnly:= True;
  GoFirst    := True;
  OnCompare  := nil;
end;

procedure TrCommonFindDlg.FormDestroy(Sender: TObject);
begin
  FindDataSet := nil
end;

function TrCommonFindDlg.GetAtEntry: Boolean;
begin
  Result := FindAtEntry.Checked;
end;

procedure TrCommonFindDlg.SetAtEntry(Value: Boolean);
begin
  FindAtEntry.Checked := Value;
end;


procedure TrCommonFindDlg.CancelBtnClick(Sender: TObject);
begin
  if FProcessed
  then FCanceled := True
  else ModalResult := mrCancel
end;

procedure TrCommonFindDlg.OKBtnClick(Sender: TObject);
begin
  DoFind;
end;

procedure TrCommonFindDlg.FieldBoxChange(Sender: TObject);
begin
{
  if FLastField <> Trim(FieldBox.Text)
  then begin
    ValueBox.Items.Clear;
    ValueBox.Text := '';
  end;
}
  ValueBoxChange(Sender);
end;

procedure TrCommonFindDlg.ValueBoxExit(Sender: TObject);
var
  I: Integer;
begin
  ValueBox.Text := Trim(ValueBox.Text);
  if ValueBox.Text <> ''
  then begin
    I := ValueBox.Items.IndexOf(ValueBox.Text);
    if I < 0
    then begin
      if ValueBox.Items.Count > FMaxItems
      then ValueBox.Items.Delete(ValueBox.Items.Count-1);
      ValueBox.Items.Insert(0, ValueBox.Text);
    end
  end;
  OKBtn.Enabled := ValueBox.Text <> '';
end;

function TrCommonFindDlg.DoCompare(OnField: TField; Text: String): Boolean;
var
  From: String;
begin
  if Assigned(OnCompare)
  then Result := OnCompare(OnField, Text, CaseSens.Checked)
  else begin
    From := OnField.DisplayText;
    if CaseSens.Checked
    then begin
      Text := ANSIUpperCase(Text);
      From := ANSIUpperCase(From);
    end;
    if AtEntry
    then Result := Pos(Text, From)<>0
    else Result := Pos(Text, From)=1;
  end;
end;

function TrCommonFindDlg.GetField: TField;
var
  I: Integer;
begin
  Result := nil;
  I := FieldBox.Items.IndexOf(FieldBox.Text);
  if I > -1
  then begin
    I := Integer(FieldBox.Items.Objects[I]);
    Result := FindDataSet.Fields[I];
  end
end;

procedure TrCommonFindDlg.DoFind;
const
  Res: Array[Boolean] of TModalResult = (mrOK, mrCancel);
var
  Text : String;
  Field: TField;
begin
  Text := Trim(ValueBox.Text);
  if Text <> ''
  then begin
    Field := GetField;
    if Field <> nil
    then begin
      Animate1.Visible := True;
      Animate1.Active  := True;
      if GoFirst
      then FindDataSet.First;
      if not FindDataSet.EOF then begin
        while (not FindDataSet.EOF) and (not DoCompare(Field, Text))
        and (not FCanceled) do begin
          FProcessed := True;
          Application.ProcessMessages;
          FindDataSet.Next;
        end;
        FProcessed := False;
        ModalResult := Res[FCanceled];
        if not FCanceled and not DoCompare( Field, Text)
        then ModalResult := -1;
      end
      else ModalResult := -1;
    end
    else raise Exception.CreateFmt(srFieldNotFound,[FieldBox.Text]);
  end
  else raise Exception.Create(srNoLocatingString)
end;

procedure TrCommonFindDlg.FormDeactivate(Sender: TObject);
begin
  if Animate1.Visible then Animate1.Visible := False;
  if Animate1.Active then Animate1.Active := False;
end;

procedure TrCommonFindDlg.Initialized(AGrid        : TDBGrid;
                                      ADataSet     : TDataSet;
                                      ACaseSens    : Boolean;
                                      AVisibleOnly : Boolean;
                                      AGoFirst     : Boolean;
                                      AAtEntry     : Boolean;
                                      AValueList   : TStrings;
                                      ALastValue   : String;
                                      ALastField   : String;
                                      AMaxItems    : Integer);
const
  NotFindSet: Set of TFieldType = [ftUnknown, ftBytes, ftVarBytes, ftAutoInc,
                                   ftBlob, ftMemo, ftGraphic, ftFmtMemo,
                                   ftParadoxOle, ftDBaseOle, ftTypedBinary,
                                   ftCursor];
var
  I: Integer;
  FTitle: String;
begin
{  FMaxItems   := AMaxItems;
  FindDataSet := ADataSet;
  FieldBox.Items.Clear;
  for I := 0 to FindDataSet.FieldCount-1 do
  begin
    if VisibleOnly and (not FindDataSet.Fields[I].Visible)
    then Continue;
    if FindDataSet.Fields[I].DataType in NotFindSet
    then Continue;
    if Trim(FindDataSet.Fields[I].DisplayName)<>''
    then FieldBox.Items.AddObject(Trim(FindDataSet.Fields[I].DisplayName), TObject(I));
    if FieldBox.Text = ''
    then FieldBox.ItemIndex := 0;
  end;
}
  if (AGrid <> nil) and (AGrid.DataSource <> nil) and (AGrid.DataSource.DataSet <> nil)
  and (AGrid.DataSource.DataSet <> ADataSet) then ADataSet := AGrid.DataSource.DataSet;
  FindDataSet := ADataSet;
  FieldBox.Items.Clear;
  if (AGrid <> nil) and (AGrid.Columns.Count <> 0) and
     (AGrid.DataSource <> nil) and (AGrid.DataSource.DataSet <> nil)
  then begin
    for I := 0 to AGrid.Columns.Count-1 do
    if AGrid.Columns[I].Field <> nil then begin
      if VisibleOnly and (not AGrid.Columns[I].{$IFDEF POLARIS_D3}Field.{$ENDIF}Visible)
      then Continue;
      if AGrid.Columns[I].Field.DataType in NotFindSet
      then Continue;
      FTitle := Trim(AGrid.Columns[I].Title.Caption);
      if FTitle=EmptyStr then FTitle := Trim(AGrid.Columns[I].Field.DisplayName);
      if FTitle <> EmptyStr
      then FieldBox.Items.AddObject(Trim(FTitle), TObject(AGrid.Columns[I].Field.Index));
      if FieldBox.Text = EmptyStr
      then FieldBox.ItemIndex := 0;
    end;
    if FieldBox.Text = EmptyStr
    then AGrid := nil;
  end
  else AGrid := nil;
  if AGrid = nil then
    for I := 0 to FindDataSet.FieldCount-1 do
    begin
      if VisibleOnly and (not FindDataSet.Fields[I].Visible)
      then Continue;
      if FindDataSet.Fields[I].DataType in NotFindSet
      then Continue;
      if Trim(FindDataSet.Fields[I].DisplayName)<>''
      then FieldBox.Items.AddObject(Trim(FindDataSet.Fields[I].DisplayName), TObject(I));
      if FieldBox.Text = ''
      then FieldBox.ItemIndex := 0;
    end;
  FMaxItems   := AMaxItems;

  I := FieldBox.Items.IndexOf(ALastField);
  if I < 0
  then I := 0;
  FieldBox.ItemIndex := I;
  FLastField := FieldBox.Text;
  VisibleOnly      := AVisibleOnly;
  GoFirst          := AGoFirst;
  AtEntry          := AAtEntry;
  ValueBox.Items.Assign(AValueList);
  ValueBox.Text    := ALastValue;
  CaseSens.Checked := ACaseSens;
  FieldBoxChange(Self);
  GroupBox1.Enabled := GoFirst;
end;

procedure TrCommonFindDlg.ValueBoxChange(Sender: TObject);
begin
  OkBtn.Enabled := (Trim(FieldBox.Text) <> '') and (Trim(ValueBox.Text) <> '');
end;

procedure TrCommonFindDlg.FormPaint(Sender: TObject);
begin
  if not (FProcessed or GoFirst)
  then DoFind;
end;

function TrCommonFindDlg.FormHelp(Command: Word; Data: Integer;
  var CallHelp: Boolean): Boolean;
begin
  CallHelp := not ((Command = HELP_CONTEXTPOPUP) and (Data = HelpContext));
  Result := True;
end;

end.
