//==============================================
//       rlabel.pas
//
//         Delphi.
//        TLabel.
//
//      Copyright 1998-2000 Polaris Software
//      http://members.xoom.com/PolarisSoft
//      mailto: PolarisLib@mail.ru
//==============================================
unit rLabel;

{$I POLARIS.INC}

interface

uses
  Windows, Registry, Messages, Classes, Controls, Graphics,
  StdCtrls, ExtCtrls, Forms;

type
  TrCustomLabel = class(TGraphicControl)
  private
    FTimer : Ttimer;
    FTimerInterval: Word;
    BlinkState : boolean;
    FBlinkTransparent : boolean;
    FBlinkColor : TColor;
    FBlinkTopColor : TColor;
    FBlinkBottomColor : TColor;

    FFocusControl: TWinControl;
    FAlignment: TAlignment;
    FAutoSize: Boolean;
    FLayout: TTextLayout;
    FWordWrap: Boolean;
    FShowAccelChar: Boolean;
    FShowFocus: Boolean;
    FFocused: Boolean;
    FMouseInControl: Boolean;
    FDragging: Boolean;
    FOnMouseEnter: TNotifyEvent;
    FOnMouseLeave: TNotifyEvent;

    FRaisedTop : boolean;
    FRaisedBottom : boolean;
    FRaisedTopColor : TColor;
    FRaisedBottomColor : TColor;
    FRaisedTopDistance : TBorderWidth;
    FRaisedBottomDistance : TBorderWidth;
    FBorderWidth: TBorderWidth;
    FBevelWidth: TBevelWidth;
    FBevels: Array[0..1] of TPanelBevel;
    FText3D,
    FBlink,
    FBTransparent: Boolean;

    procedure SetRaisedTop(Value: boolean);
    procedure SetRaisedBottom(Value: boolean);
    procedure SetRaisedTopColor(Value: TColor);
    procedure SetRaisedBottomColor(Value: TColor);
    procedure SetRaisedTopDistance(Value: TBorderWidth);
    procedure SetRaisedBottomDistance(Value: TBorderWidth);

    procedure SetBorderWidth(Value: TBorderWidth);

    procedure SetBevelWidth(Value: TBevelWidth);
    function GetBevels(Index: Integer): TPanelBevel;
    procedure SetBevels(Index: Integer; Value: TPanelBevel);

    procedure SetText3D(Value: Boolean);

    procedure SetBlink(value: boolean);
    procedure SetBlinkSpeed(Value: Word);
    function GetBlink : boolean;
    procedure TimerTimer(Sender: TObject);

    procedure SetTransparentBorder(Value: Boolean);

    procedure DoDrawText(var Rect: TRect; Flags: Word);
    function GetTransparent: Boolean;
    procedure UpdateTracking;
    procedure SetAlignment(Value: TAlignment);
    procedure SetAutoSize(Value: Boolean);
    procedure SetFocusControl(Value: TWinControl);
    procedure SetLayout(Value: TTextLayout);
    procedure SetShowAccelChar(Value: Boolean);
    procedure SetTransparent(Value: Boolean);
    procedure SetWordWrap(Value: Boolean);
    procedure SetShowFocus(Value: Boolean);
    procedure CMTextChanged(var Message: TMessage); message CM_TEXTCHANGED;
    procedure CMFocusChanged(var Message: TCMFocusChanged); message CM_FOCUSCHANGED;
    procedure CMFontChanged(var Message: TMessage); message CM_FONTCHANGED;
    procedure CMDialogChar(var Message: TCMDialogChar); message CM_DIALOGCHAR;
    procedure CMEnabledChanged(var Message: TMessage); message CM_ENABLEDCHANGED;
    procedure CMMouseEnter(var Message: TMessage); message CM_MOUSEENTER;
    procedure CMMouseLeave(var Message: TMessage); message CM_MOUSELEAVE;
    procedure CMVisibleChanged(var Message: TMessage); message CM_VISIBLECHANGED;
    procedure WMRButtonDown(var Message: TWMRButtonDown); message WM_RBUTTONDOWN;
    procedure WMRButtonUp(var Message: TWMRButtonUp); message WM_RBUTTONUP;
  protected
    procedure AdjustBounds;
    function GetDefaultFontColor: TColor; virtual;
    function GetLabelCaption: string; virtual;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
    procedure Paint; override;
    procedure MouseEnter; dynamic;
    procedure MouseLeave; dynamic;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

    property Alignment: TAlignment read FAlignment write SetAlignment default taLeftJustify;
    property AutoSize: Boolean read FAutoSize write SetAutoSize default True;
    property FocusControl: TWinControl read FFocusControl write SetFocusControl;
    property Layout: TTextLayout read FLayout write SetLayout default tlTop;
    property ShowAccelChar: Boolean read FShowAccelChar write SetShowAccelChar default True;
    property ShowFocus: Boolean read FShowFocus write SetShowFocus default False;
    property Transparent: Boolean read GetTransparent write SetTransparent default False;
    property WordWrap: Boolean read FWordWrap write SetWordWrap default False;
    property OnMouseEnter: TNotifyEvent read FOnMouseEnter write FOnMouseEnter;
    property OnMouseLeave: TNotifyEvent read FOnMouseLeave write FOnMouseLeave;

    property RaisedTop : boolean read FRaisedTop write SetRaisedTop default True;
    property RaisedBottom : boolean read FRaisedBottom write SetRaisedBottom default True;
    property RaisedTopColor : TColor read FRaisedTopColor write SetRaisedTopColor default clBtnHighlight;
    property RaisedBottomColor : TColor read FRaisedBottomColor write SetRaisedBottomColor default clBtnShadow;
    property RaisedTopDistance : TBorderWidth read FRaisedTopDistance write SetRaisedTopDistance default 1;
    property RaisedBottomDistance : TBorderWidth read FRaisedBottomDistance write SetRaisedBottomDistance default 1;

    property BorderWidth: TBorderWidth read FBorderWidth write SetBorderWidth default 0;

    property BevelWidth: TBevelWidth read FBevelWidth write SetBevelWidth Default 1;
    property BevelOuter: TPanelBevel index 0 read GetBevels write SetBevels default bvNone;
    property BevelInner: TPanelBevel index 1 read GetBevels write SetBevels default bvNone;
    property Text3D: Boolean read FText3D write SetText3D default True;

    property Blinking : boolean read GetBlink;
    property BlinkColor : TColor read FBlinkColor write FBlinkColor default clGrayText;
    property BlinkTransparent : boolean read FBlinkTransparent write FBlinkTransparent default True;
    property Blink: Boolean read FBlink write SetBlink default False;
    property BlinkSpeed : Word read FTimerInterval write SetBlinkSpeed default 500;
    property BlinkTopColor : TColor read FBlinkTopColor write FBlinkTopColor default clBtnShadow;
    property BlinkBottomColor : TColor read FBlinkBottomColor write FBlinkBottomColor default clBtnHighlight;

    property BorderTransparent: Boolean read FBTransparent write SetTransparentBorder default False;

    property Align;
    property Color;
    property Caption;
    property Font;

    property Canvas;
    property MouseInControl: Boolean read FMouseInControl;
  end;

  TrLabel = class(TrCustomLabel)
  published
    property Blink;
    property BlinkSpeed;
    property BlinkColor;
    property BlinkTopColor;
    property BlinkBottomColor;
    property BlinkTransparent;
    property RaisedTop;
    property RaisedBottom;
    property RaisedTopColor;
    property RaisedBottomColor;
    property RaisedTopDistance;
    property RaisedBottomDistance;

    property BorderWidth;
    property BevelWidth;
    property BorderTransparent;
    property BevelOuter;
    property BevelInner;
    property Text3D;
    property AutoSize;

    property Enabled;
    property Align;
    property Alignment;
    property Caption;
    property Color;
    property DragCursor;
    property DragMode;
    property FocusControl;
    property Font;
{$IFDEF POLARIS_D4}
    property Anchors;
    property BiDiMode;
    property Constraints;
    property DragKind;
    property ParentBiDiMode;
{$ENDIF}
    property Layout;
    property ParentColor;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowAccelChar;
    property ShowHint;
    property Transparent;
    property Visible;
    property WordWrap;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDrag;
{$IFDEF POLARIS_D4}
    property OnEndDock;
    property OnStartDock;
{$ENDIF}
  end;

implementation

uses VCLUtils;

const
  Alignments: array[TAlignment] of Word = (DT_LEFT, DT_RIGHT, DT_CENTER);
  WordWraps: array[Boolean] of Word = (0, DT_WORDBREAK);

{ TrCustomLabel }

procedure Draw3DText(DC: HDC; Str: PChar; Count: Integer; var Rect: TRect;
  Format: Word;
  DrawTopDistance, DrawBottomDistance: Boolean;
  ShadowTopSize: Byte; ShadowTopColor: TColorRef;
  ShadowBottomSize: Byte; ShadowBottomColor: TColorRef);
var
  RShadow,
  RText, RTopShadow, RBottomShadow: TRect;
  Color: TColorRef;
begin
  RText := Rect;
  RShadow := Rect;
  RTopShadow := Rect;
  RBottomShadow := Rect;

  if DrawTopDistance then begin
    Color := SetTextColor(DC, ShadowTopColor);
    OffsetRect(RTopShadow, -ShadowTopSize, -ShadowTopSize);
    DrawText(DC, Str, Count, RTopShadow, Format);
    OffsetRect(RTopShadow, ShadowTopSize, ShadowTopSize);
    RShadow := RTopShadow;
    SetTextColor(DC, Color);
  end;
  if DrawBottomDistance then begin
    Color := SetTextColor(DC, ShadowBottomColor);
    OffsetRect(RBottomShadow, ShadowBottomSize, ShadowBottomSize);
    DrawText(DC, Str, Count, RBottomShadow, Format);
    OffsetRect(RBottomShadow, -ShadowBottomSize, -ShadowBottomSize);
    if DrawTopDistance then
      UnionRect(RShadow, RTopShadow, RBottomShadow)
    else RShadow := RBottomShadow;
    SetTextColor(DC, Color);
  end;

  DrawText(DC, Str, Count, RText, Format);
  if not (DrawTopDistance or DrawBottomDistance)
  then RShadow := RText;
  UnionRect(Rect, RText, RShadow);
end;

constructor TrCustomLabel.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ControlStyle := ControlStyle + [csOpaque,csReplicatable];
  Width := 65;
  Height := 17;
  FAutoSize := True;
  FShowAccelChar := True;
  FRaisedTop := True;
  FRaisedBottom := True;
  FRaisedTopColor := clBtnHighlight;
  FRaisedBottomColor := clBtnShadow;
  FRaisedTopDistance := 1;
  FRaisedBottomDistance := 1;
  FBevelWidth           := 1;
  FillChar(FBevels, SizeOf(FBevels), bvNone);
  FBorderWidth := 0;
  FText3D := True;
  FBlinkTransparent     := true;
  FBlinkColor           := clGrayText;
  FBlinkTopColor        := clBtnShadow;
  FBlinkBottomColor     := clBtnHighlight;
  FTimer                := nil;
  FTimerInterval        := 500;
  BlinkState            := false;
  FBTransparent         := False;
end;

destructor TrCustomLabel.Destroy;
begin
  if FTimer <> nil then FTimer.Free;
  inherited Destroy;
end;

function TrCustomLabel.GetLabelCaption: string;
begin
  Result := Caption;
end;

function TrCustomLabel.GetDefaultFontColor: TColor;
begin
  Result := Font.Color;
end;

procedure TrCustomLabel.DoDrawText(var Rect: TRect; Flags: Word);
var
  Text: string;
  dRB, dRT: Integer;
  TextColor,
  TopColor,
  BColor: TColor;
  FRT, FRB: Boolean;
begin
  Text := GetLabelCaption;
  if (Flags and DT_CALCRECT <> 0) and ((Text = '') or FShowAccelChar and
    (Text[1] = '&') and (Text[2] = #0)) then Text := Text + ' ';
  if not FShowAccelChar then Flags := Flags or DT_NOPREFIX;
{$IFDEF POLARIS_D4}
  Flags := DrawTextBiDiModeFlags(Flags);
{$ENDIF}
  dRT := FRaisedTopDistance;
  dRB := FRaisedBottomDistance;
  if BlinkState then begin
    TextColor := FBlinkColor;
    TopColor  := FBlinkTopColor;
    BColor    := FBlinkBottomColor;
  end
  else begin
    TextColor := GetDefaultFontColor;
    TopColor  := FRaisedTopColor;
    BColor    := FRaisedBottomColor;
  end;
  Canvas.Font := Font;
  Canvas.Font.Color := TextColor;
  FRT := FText3D and FRaisedTop;
  FRB := FText3D and FRaisedBottom;
  if not Enabled then begin
    FRT := False;
    FRB := True;
    dRT := 1;
    dRB := 1;
    TopColor := clBtnHighlight;
    BColor   := clBtnHighlight;
    Canvas.Font.Color := clGrayText;
  end;
  Draw3DText(Canvas.Handle, PChar(Text), Length(Text), Rect, Flags,
    FRT, FRB,
    dRT, ColorToRGB(TopColor),
    dRB, ColorToRGB(BColor));
end;

procedure TrCustomLabel.Paint;
var
  cRect,
  Rect: TRect;
  dX, I, dY,
  dT, dB, dD,
  DrawStyle: Integer;
begin
  cRect := Canvas.ClipRect;
try
  if not Enabled and not (csDesigning in ComponentState) then
    FDragging := False;
  with Canvas do begin
    if not Transparent and not FBTransparent then begin
      Brush.Color := Self.Color;
      Brush.Style := bsSolid;
      FillRect(ClientRect);
    end;
    Brush.Style := bsClear;
    Rect := ClientRect;

    case FBevels[0] of
    bvLowered: Frame3D(Canvas, Rect, clBtnShadow, clBtnHighlight, FBevelWidth);
    bvRaised: Frame3D(Canvas, Rect, clBtnHighlight, clBtnShadow, FBevelWidth);
    end;

    InflateRect(Rect, -FBorderWidth, -FBorderWidth);

    case FBevels[1] of
    bvLowered: Frame3D(Canvas, Rect, clBtnShadow, clBtnHighlight, FBevelWidth);
    bvRaised: Frame3D(Canvas, Rect, clBtnHighlight, clBtnShadow, FBevelWidth);
    end;

    IntersectClipRect(Canvas.Handle, Rect.Left, Rect.Top, Rect.Right, Rect.Bottom);

    if FBTransparent and not Transparent then begin
      Brush.Color := Self.Color;
      Brush.Style := bsSolid;
      FillRect(Rect);
    end;
    Brush.Style := bsClear;
    dD := FBorderWidth;
    for I := 0 to 1 do
      if FBevels[I] <> bvNone then Inc(dD, FBevelWidth);
    InflateRect(Rect, -1, 0);
    DrawStyle := DT_EXPANDTABS or WordWraps[FWordWrap] or Alignments[FAlignment];
    dT := Integer(FRaisedTop and FText3D)*FRaisedTopDistance;
    dB := Integer(FRaisedBottom and FText3D)*FRaisedBottomDistance;
    case FLayout of
    tlTop   : dY := dT;
    tlBottom: dY := -dB;
    else dY := 0;
    end;
    case FAlignment of
    taLeftJustify : dX := dT;
    taRightJustify: dX := -dB;
    else dX := 0;
    end;
    { Calculate vertical layout }
    if FLayout <> tlTop then begin
      DoDrawText(Rect, DrawStyle or DT_CALCRECT);
      case FAlignment of
      taLeftJustify : I := dD;
      taRightJustify: I := -dD;
      else I := 0;
      end;
      Rect.Left := ClientRect.Left;
      Rect.Right := ClientRect.Right;
      if FLayout = tlBottom then OffsetRect(Rect, 0, Height - dD - Rect.Bottom)
      else OffsetRect(Rect, 0, (Height - dD - Rect.Bottom) div 2);
      OffsetRect(Rect, I, 0);
    end;

    OffsetRect(Rect, dX, dY);

    DoDrawText(Rect, DrawStyle);
    if FShowFocus and Assigned(FFocusControl) and FFocused and
      not (csDesigning in ComponentState) then
    begin
      InflateRect(Rect, 1, 0);
      Brush.Color := Self.Color;
      DrawFocusRect(Rect);
    end;
  end;
finally
  IntersectClipRect(Canvas.Handle, cRect.Left, cRect.Top, cRect.Right, cRect.Bottom);
end;
end;

procedure TrCustomLabel.AdjustBounds;
var
  DC: HDC;
  dX,
  X: Integer;
  Rect: TRect;
  AAlignment: TAlignment;
begin
  if not (csReading in ComponentState) and FAutoSize then begin
    Rect := ClientRect;
    dX := FBorderWidth;
    for X:= 0 to 1 do
      if FBevels[X] <> bvNone then
        Inc(dX, FBevelWidth);
    X := 0;
    if FText3D then begin
      if FRaisedTop then Inc(X, FRaisedTopDistance);
      if FRaisedBottom then Inc(X, FRaisedBottomDistance);
    end;
    X := X div 2;
    Inc(dX, X);
    InflateRect(Rect, -dX, -dX);
    InflateRect(Rect, -1, 0);
    DC := GetDC(0);
    Canvas.Handle := DC;
    DoDrawText(Rect, DT_EXPANDTABS or DT_CALCRECT or WordWraps[FWordWrap]);
    InflateRect(Rect, dX, dX);
    Canvas.Handle := 0;
    ReleaseDC(0, DC);
    InflateRect(Rect, 1, 0);
    X := Left;
    AAlignment := FAlignment;
{$IFDEF POLARIS_D4}
    if UseRightToLeftAlignment then ChangeBiDiModeAlignment(AAlignment);
{$ENDIF}
    if AAlignment = taRightJustify then Inc(X, Width - Rect.Right);
    SetBounds(X, Top, Rect.Right, Rect.Bottom);
  end;
end;

procedure TrCustomLabel.SetAlignment(Value: TAlignment);
begin
  if FAlignment <> Value then begin
    FAlignment := Value;
    Invalidate;
  end;
end;

procedure TrCustomLabel.SetAutoSize(Value: Boolean);
begin
  if FAutoSize <> Value then begin
    FAutoSize := Value;
    AdjustBounds;
  end;
end;

procedure TrCustomLabel.SetLayout(Value: TTextLayout);
begin
  if FLayout <> Value then begin
    FLayout := Value;
    Invalidate;
  end;
end;

function TrCustomLabel.GetTransparent: Boolean;
begin
  Result := not (csOpaque in ControlStyle);
end;

procedure TrCustomLabel.SetFocusControl(Value: TWinControl);
begin
  FFocusControl := Value;
  if Value <> nil then Value.FreeNotification(Self);
  if FShowFocus then Invalidate;
end;

procedure TrCustomLabel.SetShowAccelChar(Value: Boolean);
begin
  if FShowAccelChar <> Value then begin
    FShowAccelChar := Value;
    Invalidate;
  end;
end;

procedure TrCustomLabel.SetTransparent(Value: Boolean);
begin
  if Transparent <> Value then begin
    if Value then ControlStyle := ControlStyle - [csOpaque]
    else ControlStyle := ControlStyle + [csOpaque];
    Invalidate;
  end;
end;

procedure TrCustomLabel.SetShowFocus(Value: Boolean);
begin
  if FShowFocus <> Value then begin
    FShowFocus := Value;
    Invalidate;
  end;
end;

procedure TrCustomLabel.SetWordWrap(Value: Boolean);
begin
  if FWordWrap <> Value then begin
    FWordWrap := Value;
    AdjustBounds;
  end;
end;

procedure TrCustomLabel.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (AComponent = FFocusControl) then
    FocusControl := nil;
end;

procedure TrCustomLabel.MouseDown(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
begin
  inherited MouseDown(Button, Shift, X, Y);
  if (Button = mbLeft) and Enabled then begin
    FDragging := True;
  end;
end;

procedure TrCustomLabel.MouseUp(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
begin
  inherited MouseUp(Button, Shift, X, Y);
  if FDragging and (Button = mbLeft) then FDragging := False;
  UpdateTracking;
end;

procedure TrCustomLabel.MouseEnter;
begin
  if Assigned(FOnMouseEnter) then FOnMouseEnter(Self);
end;

procedure TrCustomLabel.MouseLeave;
begin
  if Assigned(FOnMouseLeave) then FOnMouseLeave(Self);
end;

procedure TrCustomLabel.UpdateTracking;
var
  P: TPoint;
  OldValue: Boolean;
begin
  OldValue := FMouseInControl;
  GetCursorPos(P);
  FMouseInControl := Enabled and (FindDragTarget(P, True) = Self);// and
    IsForegroundTask;
  if (FMouseInControl <> OldValue) then
    if FMouseInControl then MouseEnter else MouseLeave;
end;

procedure TrCustomLabel.CMFocusChanged(var Message: TCMFocusChanged);
var
  Active: Boolean;
begin
  Active := Assigned(FFocusControl) and (Message.Sender = FFocusControl);
  if FFocused <> Active then begin
    FFocused := Active;
    if FShowFocus then Invalidate;
  end;
  inherited;
end;

procedure TrCustomLabel.CMTextChanged(var Message: TMessage);
begin
  Invalidate;
  AdjustBounds;
end;

procedure TrCustomLabel.CMFontChanged(var Message: TMessage);
begin
  inherited;
  AdjustBounds;
end;

procedure TrCustomLabel.CMDialogChar(var Message: TCMDialogChar);
begin
  if (FFocusControl <> nil) and Enabled and ShowAccelChar and
    IsAccel(Message.CharCode, GetLabelCaption) then
    with FFocusControl do
      if CanFocus then begin
        SetFocus;
        Message.Result := 1;
      end;
end;

procedure TrCustomLabel.WMRButtonDown(var Message: TWMRButtonDown);
begin
  inherited;
  UpdateTracking;
end;

procedure TrCustomLabel.WMRButtonUp(var Message: TWMRButtonUp);
begin
  inherited;
  UpdateTracking;
end;

procedure TrCustomLabel.CMEnabledChanged(var Message: TMessage);
begin
  inherited;
  UpdateTracking;
end;

procedure TrCustomLabel.CMVisibleChanged(var Message: TMessage);
begin
  inherited;
  if Visible then UpdateTracking;
end;

procedure TrCustomLabel.CMMouseEnter(var Message: TMessage);
begin
  inherited;
  if not FMouseInControl and Enabled and IsForegroundTask then begin
    FMouseInControl := True;
    MouseEnter;
  end;
end;

procedure TrCustomLabel.CMMouseLeave(var Message: TMessage);
begin
  inherited;
  if FMouseInControl and Enabled and not FDragging then begin
    FMouseInControl := False;
    MouseLeave;
  end;
end;

procedure TrCustomLabel.SetRaisedTop(Value: boolean);
begin
  if FRaisedTop <> Value then begin
    FRaisedTop := Value;
    AdjustBounds;
    Invalidate;
  end;
end;

procedure TrCustomLabel.SetRaisedBottom(Value: boolean);
begin
  if FRaisedBottom <> Value
  then begin
    FRaisedBottom := Value;
    AdjustBounds;
    Invalidate;
  end;
end;

procedure TrCustomLabel.SetRaisedTopColor(Value: TColor);
begin
  if FRaisedTopColor <> Value
  then begin
    FRaisedTopColor := Value;
    AdjustBounds;
    Invalidate;
  end;
end;

procedure TrCustomLabel.SetRaisedBottomColor(Value: TColor);
begin
  if FRaisedBottomColor <> Value
  then begin
    FRaisedBottomColor := Value;
    AdjustBounds;
    Invalidate;
  end;
end;

procedure TrCustomLabel.SetRaisedTopDistance(Value: TBorderWidth);
begin
  if FRaisedTopDistance <> Value
  then begin
    FRaisedTopDistance := Value;
    AdjustBounds;
    Invalidate;
  end;
end;

procedure TrCustomLabel.SetRaisedBottomDistance(Value: TBorderWidth);
begin
  if FRaisedBottomDistance <> Value
  then begin
    FRaisedBottomDistance := Value;
    AdjustBounds;
    Invalidate;
  end;
end;

function TrCustomLabel.GetBevels(Index: Integer): TPanelBevel;
begin
  Result := FBevels[Index];
end;

procedure TrCustomLabel.SetBevels(Index: Integer; Value: TPanelBevel);
begin
  if FBevels[Index] <> Value
  then begin
    FBevels[Index] := Value;
    AdjustBounds;
    Invalidate;
  end;
end;

procedure TrCustomLabel.SetBevelWidth(Value: TBevelWidth);
begin
  if Value <> FBevelWidth then begin
    FBevelWidth := Value;
    AdjustBounds;
    Invalidate;
  end;
end;

procedure TrCustomLabel.SetBorderWidth(Value: TBorderWidth);
begin
  if Value <> FBorderWidth then begin
    FBorderWidth := Value;
    AdjustBounds;
    Invalidate;
  end;
end;

procedure TrCustomLabel.SetText3D(Value: Boolean);
begin
  if FText3D <> Value
  then begin
    FText3D := Value;
    AdjustBounds;
    Invalidate;
  end;
end;

procedure TrCustomLabel.SetBlink(value: boolean);
begin
  if FBlink <> Value
  then begin
    FBlink := Value;
    if FTimer = nil
    then begin
      FTimer := TTimer.Create(Self);
      FTimer.Interval := FTimerInterval;
      FTimer.OnTimer  := TimerTimer;
    end
    else begin
      FTimer.Free;
      FTimer := nil;
    end;
    if FTimer <> nil then FTimer.Enabled := Value;
    BlinkState := false;
    Invalidate;
  end;
end;

procedure TrCustomLabel.SetBlinkSpeed(Value: Word);
begin
  if FTimerInterval <> Value
  then begin
    FTimerInterval := Value;
    if FTimer <> nil then FTimer.Interval := FTimerInterval;
  end;
end;

function TrCustomLabel.GetBlink : boolean;
begin
  Result := False;
  if FTimer <> nil then Result := FTimer.enabled;
  BlinkState := false;
end;

procedure TrCustomLabel.TimerTimer(Sender: TObject);
begin
  if Enabled then BlinkState := not BlinkState;
  Invalidate;
end;

procedure TrCustomLabel.SetTransparentBorder(Value: Boolean);
begin
  if Value <> FBTransparent
  then begin
    FBTransparent := Value;
    Invalidate;
  end;
end;

end.
