//==============================================
//       rDLLloader.pas
//
//         Delphi.
//         DLL.
//
//      Copyright 2000 Polaris Software
//      http://members.xoom.com/PolarisSoft
//      mailto: PolarisLib@mail.ru
//==============================================
unit rDLLloader;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls{, Forms, Dialogs};

type
  TrDLLloader = class(TComponent)
  private
    FActive: Boolean;
    FHInstance: HMODULE;
    FLibraryName: TFileName;
    FRequiredProcs: TStrings;
    FVersionInfo: TStrings;
    FAfterLoadLibrary: TNotifyEvent;
    FBeforeFreeLibrary: TNotifyEvent;
    FRequiredVersion: String;
    FProductVersionMS: DWORD;
    FProductVersionLS: DWORD;
    procedure ClearVersionInfo;
    function GetEntryPoint(const Name: String): Pointer;
    procedure InternalGetVersionInfo;
    function InternalExtractVersion(var Version: String; var MS, LS: DWORD): Boolean;
    procedure SetVersionInfo(const Value: TStrings);
    procedure SetLibraryName(const Value: TFileName);
    procedure SetRequiredProcs(const Value: TStrings);
    procedure SetRequiredVersion(const Value: String);
  protected
    function CheckRequired: Boolean;
    procedure DoAfterLoadLibrary; virtual;
    procedure DoBeforeFreeLibrary; virtual;
    procedure FreeDLL;
    property HInstance: HMODULE read FHInstance write FHInstance;
    procedure LoadDLL;
    procedure Loaded; override;
    procedure SetActive(const Value: Boolean); virtual;
  public
    constructor Create(AOwner: TComponent); override;
    procedure Close;
    destructor Destroy; override;
    property EntryPoint[const Name: String]: Pointer read GetEntryPoint; default;
    procedure Open;
    property ProductVersionLS: DWORD read FProductVersionLS;
    property ProductVersionMS: DWORD read FProductVersionMS;
  published
    property Active: Boolean read FActive write SetActive;
    property LibraryName: TFileName read FLibraryName write SetLibraryName;
    property RequiredProcs: TStrings read FRequiredProcs write SetRequiredProcs;
    property RequiredVersion: String read FRequiredVersion write SetRequiredVersion;
    property VersionInfo: TStrings read FVersionInfo write SetVersionInfo stored False;
    property AfterLoadLibrary: TNotifyEvent read FAfterLoadLibrary write FAfterLoadLibrary;
    property BeforeFreeLibrary: TNotifyEvent read FBeforeFreeLibrary write FBeforeFreeLibrary;
  end;

function FormatVersion(var Version: String; var MS, LS: DWORD): Boolean;

resourcestring
  srDLLLoadError = '  '^M'%s.';
  srDLLChangeNameError = '  '^M'  .';
  srDLLCheckRequiredError = '     .';
  srDLLCheckVersionError = '   .';
  srDLLVersionFormatError = '   .';

implementation

uses
  Procs, StrUtils;

function FormatVersion(var Version: String; var MS, LS: DWORD): Boolean;
var
  i, WC: Integer;
  S: String;
  V: array [1..4] of DWORD;
begin
  Result := False;
  S := '';
  MS := 0;
  LS := 0;
  try
    WC := WordCount(Version, ['.']);
    if WC > 4 then exit;
    for i := 1 to 4 do
    begin
      if i <= WC then
      begin
        V[i] := StrToInt(ExtractDelimited(i, Version, ['.']));
        S := S + IntToStr(V[i]);
        if i < WC then S := S + '.';
      end else
        V[i] := 0;
    end;
    Version := S;
    MS := V[1] shl 16 + V[2];
    LS := V[3] shl 16 + V[4];
    Result := True;
  except
  end;
end;

{ TrDLLloader }

constructor TrDLLloader.Create(AOwner: TComponent);
begin
  inherited;
  FActive := False;
  FHInstance := 0;
  FLibraryName := '';
  FProductVersionLS := 0;
  FProductVersionMS := 0;
  FRequiredProcs := TStringList.Create;
  FRequiredVersion := '';
  FVersionInfo := TStringList.Create;
  FAfterLoadLibrary := nil;
  FBeforeFreeLibrary := nil;
end;

destructor TrDLLloader.Destroy;
begin
  FreeDLL;
  FVersionInfo.Free;
  FRequiredProcs.Free;
  inherited;
end;

function TrDLLloader.CheckRequired: Boolean;
var
  i: Integer;
  hProc: Pointer;
begin
  if FHInstance = 0 then
    Result := False
  else
  begin
    Result := True;
    for i := 0 to FRequiredProcs.Count-1 do
      try
        hProc := EntryPoint[FRequiredProcs[i]];
        if not Assigned(hProc) then begin
           Result := False;
           break;
        end;
      except
        Result := False;
      end;
  end;
end;

procedure TrDLLloader.DoAfterLoadLibrary;
begin
  if Assigned(FAfterLoadLibrary) then
    FAfterLoadLibrary(Self);
end;

procedure TrDLLloader.DoBeforeFreeLibrary;
begin
  if Assigned(FBeforeFreeLibrary) then
    FBeforeFreeLibrary(Self);
end;

procedure TrDLLloader.FreeDLL;
begin
  if (FHInstance <> 0) and not (csDesigning in ComponentState) then
  begin
    DoBeforeFreeLibrary;
    ClearVersionInfo;
    FreeLibrary(FHInstance);
    FHInstance := 0;
  end;
end;

procedure TrDLLloader.ClearVersionInfo;
begin
  FProductVersionLS := 0;
  FProductVersionMS := 0;
  FRequiredProcs.Clear;
end;

function TrDLLloader.GetEntryPoint(const Name: String): Pointer;
begin
  if FHInstance = 0 then
    Result := nil
  else
    Result := GetProcAddress(FHInstance, PChar(Name));
end;

procedure TrDLLloader.InternalGetVersionInfo;
var
  VersionBuffer: Pointer;
  VersionSize, Dummy: DWord;
  PSize: Integer;
  VSFixedFileInfo: PVSFixedFileInfo;
begin
  VersionSize := GetFileVersionInfoSize(PChar(FLibraryName), Dummy);
  if VersionSize <> 0 then begin
    PSize := VersionSize;
    GetMem(VersionBuffer, PSize);
    try
      if GetFileVersionInfo(PChar(FLibraryName), Dummy, VersionSize, VersionBuffer) and
         VerQueryValue(VersionBuffer, '\', Pointer(VSFixedFileInfo), VersionSize)
      then begin
        FProductVersionMS := VSFixedFileInfo^.dwProductVersionMS;
        FProductVersionLS := VSFixedFileInfo^.dwProductVersionLS;
      end;
    finally
      FreeMem(VersionBuffer,PSize);
    end;
  end;
  GetVersionInfo(FVersionInfo, FLibraryName);
end;

function TrDLLloader.InternalExtractVersion(var Version: String; var MS, LS: DWORD): Boolean;
begin
  Result := FormatVersion(Version, MS, LS);
end;

procedure TrDLLloader.SetVersionInfo(const Value: TStrings);
begin
end;

procedure TrDLLloader.LoadDLL;
var
  MS, LS: DWORD;
  PV: String;
begin
  if not (csDesigning in ComponentState) and (FLibraryName <> '') then
  begin
    PV := VersionInfo.Values['FileVersion'];
    InternalExtractVersion(PV, FProductVersionMS, FProductVersionLS);
    InternalExtractVersion(FRequiredVersion, MS, LS);
    if (FProductVersionMS < MS)
      or ((FProductVersionMS = MS) and (FProductVersionLS < LS))
    then begin
      ClearVersionInfo;
      raise Exception.Create(srDLLCheckVersionError);
    end;
    FHInstance := LoadLibrary(PChar(FLibraryName));
    if FHInstance = 0 then
    begin
      ClearVersionInfo;
      raise Exception.CreateFmt(srDLLLoadError, [FLibraryName]);
    end;
    if not CheckRequired then begin
      try
        FreeLibrary(FHInstance);
      finally
        FHInstance := 0;
      end;
      ClearVersionInfo;
      raise Exception.Create(srDLLCheckRequiredError);
    end;
    DoAfterLoadLibrary;
  end;
end;

procedure TrDLLloader.Loaded;
begin
  inherited;
  if FLibraryName <> '' then
    InternalGetVersionInfo;
  if FActive then LoadDLL;
end;

procedure TrDLLloader.SetActive(const Value: Boolean);
begin
  if csReading in ComponentState then
    FActive := Value
  else if FActive <> Value then
  begin
    FActive := Value;
    FreeDLL;
    if FActive then
    try
      LoadDLL;
    except
      FActive := False;
      raise;
    end;
  end;
end;

procedure TrDLLloader.SetLibraryName(const Value: TFileName);
begin
  if csReading in ComponentState then
    FLibraryName := Value
  else if FLibraryName <> Value then
  begin
    FVersionInfo.Clear;
    if Active and not (csDesigning in ComponentState) then
      raise Exception.Create(srDLLChangeNameError);
    FLibraryName := Value;
    if FLibraryName <> '' then
      GetVersionInfo(FVersionInfo, FLibraryName);
  end;
end;

procedure TrDLLloader.SetRequiredProcs(const Value: TStrings);
begin
  FRequiredProcs.Assign(Value);
end;

procedure TrDLLloader.SetRequiredVersion(const Value: String);
var
  S: String;
  MS, LS: DWORD;
begin
  S := Value;
  if InternalExtractVersion(S, MS, LS) then
    FRequiredVersion := S
  else
    raise Exception.Create(srDLLVersionFormatError);
end;

procedure TrDLLloader.Close;
begin
  Active := False;
end;

procedure TrDLLloader.Open;
begin
  Active := True;
end;

end.
