//==============================================
//       rdbtreeview.pas
//
//         Delphi.
//      DBAware   TDataSet.
//
//      Copyright 1998-2000 Polaris Software
//      http://members.xoom.com/PolarisSoft
//      mailto: PolarisLib@mail.ru
//==============================================
unit rDBTreeView;

{$I POLARIS.INC}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ComCtrls, DB, DBCtrls, ToolEdit, Stdctrls, Buttons, rDBConst, Consts;

type

  ErDBTreeView = class(Exception);
  TrCustomDBTreeView = class;

  { TrTreeViewDataLink }
  TrTreeViewDataLink = class(TDataLink)
  private
    FControl: TrCustomDBTreeView;
  protected
    procedure ActiveChanged;  override;
    procedure DataSetChanged; override;
    procedure EditingChanged; override;
    procedure RecordChanged(Field: TField); override;
//**
    procedure DataSetScrolled(Distance: Integer); override;
    procedure LayoutChanged; override;
  end;

  { TTreeDataList }

  PTreeDataEntry = ^TTreeDataEntry;
  TTreeDataEntry = record
    KeyValue,
    ChildValue,
    ParentValue: Variant;
//    Text: String;
    Node:        TTreeNode;
  end;

  TTreeDataList = class(TObject)
  private
    FList:    TList;
    FIndex:   Integer;
    FEmpty,
    FCurrent: TTreeDataEntry;

    procedure SetIndex(Item: Integer);
    function  GetTreeEntry(Index: Integer): TTreeDataEntry;
    procedure SetTreeEntry(Index: Integer; AValue: TTreeDataEntry);
    function  GetTreeNode(Index: Integer): TTreeNode;
    procedure SetTreeNode(Index: Integer; Node: TTReeNode);
    function  GetItems(Index: Integer): PTreeDataEntry;
    procedure SetItems(Index: Integer; Item: PTreeDataEntry);
    function  GetCount: Integer;
  protected
    function Compare(Index1, Index2: Integer): Boolean;
    function CompareEntryData(Index1, Index2: Integer): Boolean;
    function CompareEntry(Value1, Value2: TTreeDataEntry): Boolean;
    function FullCompareEntry(Value1, Value2: TTreeDataEntry): Boolean;
    procedure ChangePosition(OldPosition, NewPosition: Integer);
  public
    constructor Create;
    destructor Destroy; override;

    procedure Add(const KeyValue, ChildValue, ParentValue: Variant);
    procedure AddEntry(const Entry: TTreeDataEntry);
    procedure Clear;
    function IndexOf      (Item: Pointer):        Integer;
    function IndexOfEntry (Item: TTreeDataEntry): Integer;
    function IndexOfKey   (KeyValue: Variant):   Integer;
    function IndexOfParent(ParentValue: Variant): Integer;
    function ChildrenPos  (ChildValue: Variant; var Pos: Integer): Integer;

    function IsEmptyEntry(Index: Integer): Boolean;

    property List: TList read FList;
    property Index: Integer read FIndex write SetIndex;
    property Count: Integer read GetCount;
    property CurrentEntry: TTreeDataEntry read FCurrent;
    property Items[Index: Integer]: PTreeDataEntry read GetItems write SetItems;
    property EmptyEntry: TTreeDataEntry read FEmpty;
    property TreeEntry[Index: Integer]: TTreeDataEntry read GetTreeEntry write SetTreeEntry;
    property TreeNode[Index: Integer]: TTreeNode read GetTreeNode write SetTreeNode;
  end;

  { TrCustomDBTreeView }

  TrTreeGetTextEvent   = procedure(Sender: TObject; DataSet: TDataSet; var Text: String) of object;
  TrTreeKeyValueChanged = procedure(Sender: TObject; Node: TTreeNode) of object;

  TrCustomDBTreeView = class(TCustomTreeView)
  private
    { Private declarations }
    FOnKeyChanged:    TrTreeKeyValueChanged;
    FOnGetText:       TrTreeGetTextEvent;
    FValue,
    FKeyFieldName,
    FDisplayFields,
    FChildrenFieldName,
    FParentFieldName: String;
    FIsResync,
    FSelectedEnabled,
    FAutoSelect:      Boolean;
    FLinkData:        TrTreeViewDataLink;
    FKeyValue:        Variant;
    FListFields:      TList;
    FStringList:      TStringList;
    FTreeData:        TTreeDataList;
                           {Selected, Parent}
    FImageItedexes:   Array[Boolean,  Boolean] of Integer;
    FPrevState:       TDataSetState;
    FCurrentRecord,
    FRecordIndex,
    FRecordCount:     Integer;
    FSearchString:    String;

    function GetChildIndex: Integer;
    function GetChildSelected: Integer;
    function GetParentIndex: Integer;
    function GetParentSelected: Integer;
    function GetDisplayValue: String;
    procedure SetChildIndex(AValue: Integer);
    procedure SetChildSelected(AValue: Integer);
    procedure SetParentIndex(AValue: Integer);
    procedure SetParentSelected(AValue: Integer);

    function GetLinkDataSource: TDataSource;
    procedure SetChildrenFieldName(AValue: String);
    procedure SetKeyFieldName     (AValue: String);
    procedure SetParentFieldName  (AValue: String);
    procedure SetDisplayFields    (AValue: String);
    procedure SetLinkDataSource   (AValue: TDataSource);
    procedure SetValue            (AValue: String); virtual;
    procedure SetKeyValue         (AValue: Variant); virtual;

    procedure ChangeImages;

    procedure GetImageIndexes(Node: TTreeNode);
  protected
    { Protected declarations }
    FLookupControl,
    FListActive,
    FHasMoved,
    FRefreshEnabled,
    FHasInsert,
    FHasCreated:   Boolean;
    FKeyField,
    FChildrenField,
    FParentField:  TField;
    FDataSet:      TDataSet;
    FEditNode:     TTreeNode;

    procedure InheritedReadOnly(AValue: Boolean);
    procedure KeyValueChanged; virtual; //dynamic;
//    procedure Click; override;
    procedure KeyUp(var Key: Word; Shift: TShiftState); override;
    function  CanEdit(Node: TTreeNode): Boolean; override;
    procedure Change(Node: TTreeNode); override;

    function GetListField(Index: Integer): TField;
    function GetListFieldCount:            Integer;

    procedure CreateNodes; virtual;
    procedure ClearNodes;

    function  CheckActive:  Boolean;
    function  CheckDataSet: Boolean;
    function  CheckSource:  Boolean;
    function  CheckBrowse:  Boolean;
    procedure CheckNotCircular; virtual; //dynamic;
    procedure SyncData(Node: TTreeNode);

//    procedure ChangeDisplayNames;
    procedure NewLayout; virtual; //dynamic;
    procedure InitFields(ShowError: Boolean);

    function  DisplayString: String;
    procedure ActiveChanged;                virtual; //dynamic;
    procedure DataChanged;                  virtual; //dynamic;
    procedure EditingChanged;               virtual; //dynamic;
    procedure RecordChanged(Field: TField); virtual; //dynamic;
    procedure ReSyncRecords;                virtual; //dynamic;
    procedure ChangeChildOf (Node: TTreeNode; ChildValue: Variant);  virtual; //dynamic;
    procedure ChangeParentOf(Node: TTreeNode; ParentValue: Variant); virtual; //dynamic;
    procedure ChangeDataOf  (Node: TTreeNode; KeyValue: Variant);   virtual; //dynamic;

    procedure Notification(AComponent: TComponent;
      Operation: TOperation); override;

    procedure SetValueKey(const AValue: Variant; const AString: String); virtual; //dynamic;
    procedure SelectKeyValue(const AValue: Variant); virtual; //dynamic;
    procedure ChangeNodeText(Node: TTreeNode; const NewText: String);
    procedure SetSearchString(AValue: String);

    procedure DoEnter; override;
    procedure DoExit; override;

    { Properties }
    { Internal }
    {R}
    property DisplayValue: String read GetDisplayValue;
    property SearchString: String read FSearchString write SetSearchString;
    {RW}
    property AutoSelectItem:Boolean read FAutoSelect write FAutoSelect default False;
    property ChildIndex:    Integer read GetChildIndex write SetChildIndex default -1;
    property ChildSelected: Integer read GetChildSelected write SetChildSelected default -1;
    property ParentIndex:   Integer read GetParentIndex write SetParentIndex default -1;
    property ParentSelected:Integer read GetParentSelected write SetParentSelected default -1;
    property LinkDataSource:TDataSource read  GetLinkDataSource
                                        write SetLinkDataSource;
    property ChildrenField: String read  FChildrenFieldName
                                   write SetChildrenFieldName;
    property KeyField:      String read  FKeyFieldName
                                   write SetKeyFieldName;
    property ParentField:   String read  FParentFieldName
                                   write SetParentFieldName;
    property DisplayFields: String read  FDisplayFields
                                   write SetDisplayFields;
    property Items stored False;

    {Events}
    property OnGetText: TrTreeGetTextEvent read FOnGetText write FOnGetText;
    property OnKeyChanged: TrTreeKeyValueChanged read FOnKeyChanged write FOnKeyChanged;

  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor  Destroy;                    override;
    procedure RefreshNodes;
    function Locate(const S: String; var Index: Integer; var Node: TTreeNode): Boolean;

    {R}
    property ListFieldCount:Integer  read  GetListFieldCount;
    property ChildFld:      TField   read  FChildrenField;
    property ParentFld:     TField   read  FParentField;
    property ListField[Index: Integer]: TField read  GetListField;
    {RW}
    property Value: String read FValue write SetValue;
    property KeyValue: Variant read FKeyValue write SetKeyValue;
  published
    { Published declarations }
  end;

  TrDBTreeView = class(TrCustomDBTreeView)
  public
    property Items;

  published
    property Align;
    property BorderStyle;
    property Color;
    property Ctl3D;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Font;
    property HideSelection;
    property Images;
    property Indent;
    property ParentColor default False;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ReadOnly;
    property RightClickSelect;
    property ShowButtons;
    property ShowHint;
    property ShowLines;
    property ShowRoot;
    property SortType;
    property StateImages;
    property TabOrder;
    property TabStop default True;
    property Visible;
    property OnChange;
    property OnChanging;
    property OnClick;
    property OnCollapsing;
    property OnCollapsed;
    property OnCompare;
    property OnDblClick;
    property OnDeletion;
    property OnDragDrop;
    property OnDragOver;
    property OnEdited;
    property OnEditing;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnExpanding;
    property OnExpanded;
    property OnGetImageIndex;
    property OnGetSelectedIndex;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDrag;
    {$IFDEF POLARIS_D4}
    property Anchors;
    property AutoExpand;
    property BiDiMode;
    property BorderWidth;
    property ChangeDelay;
    property Constraints;
    property DragKind;
    property HotTrack;
    property ParentBiDiMode;
    property RowSelect;
    property ToolTips;
    property OnCustomDraw;
    property OnCustomDrawItem;
    property OnEndDock;
    property OnStartDock;
    {$ENDIF}
    { New Property }
    property AutoSelectItem;
    property ChildIndex;
    property ChildSelected;
    property ParentIndex;
    property ParentSelected;
    property ChildrenField;
    property KeyField;
    property ParentField;
    property DisplayFields;
    property DataSource:TDataSource read  GetLinkDataSource
                                    write SetLinkDataSource;
    property OnGetText;
    property OnKeyChanged;
  end;

  { TrDBLookupTree }
  TrDBLookupTree = class;

  TrDataSourceLink = class(TDataLink)
  private
    FControl: TrDBLookupTree;
  protected
    procedure ActiveChanged; override;
    procedure LayoutChanged; override;
    procedure FocusControl(Field: TFieldRef); override;
    procedure RecordChanged(Field: TField); override;
{???}
    procedure EditingChanged; override;
  end;

  TrDBLookupTree = class(TrCustomDBTreeView)
  private
    FReadOnly: Boolean;
    FDataLink: TrDataSourceLink;
    FDataFieldName: String;
    FDataField: TField;
    FMasterField: TField;

    procedure SetReadOnly(AValue: Boolean);
    procedure SetDataFieldName(AValue: String);
    function GetDataSource: TDataSource;
    procedure SetDataSource(AValue: TDataSource);

    procedure DataLinkActiveChanged; virtual;
    procedure CheckDataLinkActiveChanged; virtual;
    procedure DataLinkRecordChanged(Field: TField); virtual;
    {???}
    procedure DataLinkEditChanged; virtual;

  protected
    FHasEdit: Boolean;
    FPrevValue: Variant;

    procedure Notification(AComponent: TComponent;
                           Operation: TOperation); override;
    procedure CheckNotCircular; override;
//    procedure SetKeyValue         (AValue: Variant); override;
    procedure SelectKeyValue(const AValue: Variant); override;
    function CanModify: Boolean;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property Items;
  published
    property Align;
    property BorderStyle;
    property Color;
    property Ctl3D;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Font;
    property HideSelection;
    property Images;
    property Indent;
    property ParentColor default False;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ReadOnly: Boolean read FReadOnly write SetReadOnly;
    property RightClickSelect;
    property ShowButtons;
    property ShowHint;
    property ShowLines;
    property ShowRoot;
    property SortType;
    property StateImages;
    property TabOrder;
    property TabStop default True;
    property Visible;
    property OnChange;
    property OnChanging;
    property OnClick;
    property OnCollapsing;
    property OnCollapsed;
    property OnCompare;
    property OnDblClick;
    property OnDeletion;
    property OnDragDrop;
    property OnDragOver;
    property OnEdited;
    property OnEditing;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnExpanding;
    property OnExpanded;
    property OnGetImageIndex;
    property OnGetSelectedIndex;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDrag;
    {$IFDEF POLARIS_D4}
    property Anchors;
    property AutoExpand;
    property BiDiMode;
    property BorderWidth;
    property ChangeDelay;
    property Constraints;
    property DragKind;
    property HotTrack;
    property ParentBiDiMode;
    property RowSelect;
    property ToolTips;
    property OnCustomDraw;
    property OnCustomDrawItem;
    property OnEndDock;
    property OnStartDock;
    {$ENDIF}
    { New property }
    property DataField: String read FDataFieldName write SetDataFieldName;
    property DataSource: TDataSource read GetDataSource write SetDataSource;
    property LookupSource: TDataSource read GetLinkDataSource write SetLinkDataSource;
    property LookupField: String read FKeyFieldName write SetKeyFieldName;
    property LookupDisplay: String read FDisplayFields write SetDisplayFields;
    property ChildrenField;
    property ParentField;
    property ChildIndex;
    property ChildSelected;
    property ParentIndex;
    property ParentSelected;
    property OnGetText;
    property OnKeyChanged;
  end;

  TrDBLookupComboTree = class;

  TrPopupTree = class(TPopupWindow)
  private
//    FTreeList: TrDBLookupTree;
    FIsCreateTree: Boolean;
    FTreeList: TrCustomDBTreeView;
    FNoneLabel: TStaticText;
    FDropDownRows: Integer;
    SNoneLabel: String;

    procedure SetNoneLabel(AValue: String);

    procedure WMMouseActivate(var Message: TMessage); message WM_MOUSEACTIVATE;
    procedure WMKeyDown(var Message: TWMKeyDown); message WM_KEYDOWN;

    procedure SetDropDownRows(AValue: Integer);
    procedure TreeMouseUp(Sender: TObject; Button: TMouseButton;
                          Shift: TShiftState; X, Y: Integer);
    procedure LabelClick(Sender: TObject);
    procedure TreeChange(Sender: TObject; Node: TTreeNode);
  protected
//    procedure Click; override;

    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure KeyPress(var Key: Char); override;
    function GetValue: Variant; override;
    procedure SetValue(const Value: Variant); override;

    function GetDisplayValue: String;
    function GetSearchString: String;
    procedure SetSearchString(AValue: String);

    procedure EndCreated;

    property DisplayValue: String read GetDisplayValue;
    property SearcString: String read GetSearchString write SetSearchString;

    property NoneLabel: String read SNoneLabel write SetNoneLabel;
  public
    constructor Create(AOwner: TComponent); override;
    function GetPopupText: String; override;

//    procedure Hide; override;
    procedure Show(Origin: TPoint); override;

    property DropDownRows: Integer read FDropDownRows write SetDropDownRows;
  end;

  { TrDBLookupComboTree }
  TrDBLookupComboTree = class(TCustomComboEdit)
  private
// Design time only
    FDTDataSource:     TDataSource;
    FDTLookupField:    String ;
    FDTLookupDisplay:  String;
    FDTChildrenField:  String;
    FDTParentField:    String;
    FDTChildIndex:     Integer;
    FDTChildSelected:  Integer;
    FDTParentIndex:    Integer;
    FDTParentSelected: Integer;
    FDTImages:         TImageList;
    FDTDropDownCount:  Integer;
// ****************
    FValue:         String;
    FKeyValue:      Variant;
    FDisplayString: String;
    FDataLink:      TFieldDataLink;
    FChanging,
    FListVisible:   Boolean;
//    FDropDownCount: Integer;
    FDropDownWidth: Integer;
    FOnDropDown:    TNotifyEvent;
    FOnCloseUp:     TNotifyEvent;
//    FInListDown:    Boolean;
    FOnGetImageIndex: TTVExpandedEvent;
    FOnGetSelectedIndex: TTVExpandedEvent;
    SNoneLabel: String;

    procedure SetNoneLabel(AValue: String);
    procedure SetOnGetImageIndex(AValue: TTVExpandedEvent);
    procedure SetOnGetSelectedIndex(AValue: TTVExpandedEvent);

    procedure ActiveChange(Sender: TObject);
    procedure DataChange(Sender: TObject);
    procedure EditingChange(Sender: TObject);
    procedure UpdateData(Sender: TObject);
    procedure KeyChanged(Sender: TObject; Node: TTreeNode);
//    procedure PopupCloseUp(Sender: TObject; Accept: Boolean);

    procedure SetValue(AValue: String);
    procedure SetKeyValue(AValue: Variant);

    procedure SetDropDownWidth(AValue: Integer);
    function GetDropDownCount: Integer;
    procedure SetDropDownCount(AValue: Integer);
    function GetDataField: String;
    procedure SetDataField(AValue: String);
    function GetDataSource: TDataSource;
    procedure SetDataSource(AValue: TDataSource);
    function GetLinkDataSource: TDataSource;
    procedure SetLinkDataSource(AValue: TDataSource);
    function GetKeyField: String;
    procedure SetKeyField(AValue: String);
    function GetDisplayFields: String;
    procedure SetDisplayFields(AValue: String);
    function GetChildrenField: String;
    procedure SetChildrenField(AValue: String);
    function GetParentField: String;
    procedure SetParentField(AValue: String);
    procedure SetReadOnly(AValue: Boolean);

    function GetChildIndex: Integer;
    procedure SetChildIndex(AValue: Integer);
    function GetChildSelected: Integer;
    procedure SetChildSelected(AValue: Integer);
    function GetParentIndex: Integer;
    procedure SetParentIndex(AValue: Integer);
    function GetParentSelected: Integer;
    procedure SetParentSelected(AValue: Integer);

    function GetImageList: TImageList;
    procedure SetImageList(AValue: TImageList);

    {    procedure InvalidateText;}
    procedure WMKillFocus(var Message: TWMKillFocus); message WM_KILLFOCUS;
    procedure WKKeyDown(var Message: TWMKeyDown); message WM_KEYDOWN;
//    procedure WMChar(var Message: TWMChar); message WM_CHAR;

    procedure CMGetDataLink(var Message: TMessage); message CM_GETDATALINK;
    procedure CMEnabledChanged(var Message: TMessage); message CM_ENABLEDCHANGED;
  protected

    procedure CheckNotCircular; virtual;

    procedure DoEnter; override;
    procedure DoExit; override;

    procedure ResetField;

    function GetReadOnly: Boolean; override;
    function EditCanModify: Boolean; override;
    function CanModify: Boolean;

    procedure Notification(AComponent: TComponent;
              Operation: TOperation); override;

    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure KeyPress(var Key: Char); override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState;
              X, Y: Integer); override;

    procedure ButtonClick; override;
    procedure Change; override;
    procedure DoChange; override;
    procedure PopupChange; override;
    procedure PopupDropDown(DisableEdit: Boolean); override;
    procedure PopupCloseUp(Sender: TObject; Accept: Boolean); override;
    procedure AcceptValue(const Value: Variant); override;
    function AcceptPopup(var Value: Variant): Boolean; override;
    procedure SetPopupValue(const Value: Variant); override;
    function GetPopupValue: Variant; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

    property KeyValue: Variant read FKeyValue write SetKeyValue;
    property Value: String read FValue write SetValue;
    property Text;
    property GlyphKind;
    { Ensure GlyphKind is published before Glyph and ButtonWidth }
    property Glyph;
    property ButtonWidth;
  published
    property DropDownWidth: Integer read FDropDownWidth write SetDropDownWidth default 0;
    property DropDownCount: Integer read GetDropDownCount write SetDropDownCount default 8;
    property OnDropDown: TNotifyEvent read FOnDropDown write FOnDropDown;
    property OnCloseUp:  TNotifyEvent read FOnCloseUp write FOnCloseUp;

    property DataField: String read GetDataField write SetDataField;
    property DataSource: TDataSource read GetDataSource write SetDataSource;
    property LookupSource: TDataSource read GetLinkDataSource write SetLinkDataSource;
    property LookupField: String read GetKeyField write SetKeyField;
    property LookupDisplay: String read GetDisplayFields write SetDisplayFields;
    property ChildrenField: String read GetChildrenField write SetChildrenField;
    property ParentField: String read GetParentField write SetParentField;
    property ReadOnly: Boolean read GetReadOnly write SetReadOnly default False;

    property ChildIndex: Integer read GetChildIndex write SetChildIndex default -1;
    property ChildSelected: Integer read GetChildSelected write SetChildSelected default -1;
    property ParentIndex: Integer read GetParentIndex write SetParentIndex default -1;
    property ParentSelected: Integer read GetParentSelected write SetParentSelected default -1;
    property Images: TImageList read GetImageList write SetImageList;

    property DisplayEmpty: String read SNoneLabel write SetNoneLabel;

    property Alignment;
    property BorderStyle;
    property ButtonHint;
    property CharCase;
    property Color;
    property Ctl3D;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Font;
    property HideSelection;
{$IFDEF POLARIS_D4}
    property Anchors;
    property BiDiMode;
    property Constraints;
    property DragKind;
    property ParentBiDiMode;
{$ENDIF}
{$IFNDEF VER90}
    property ImeMode;
    property ImeName;
{$ENDIF}
//    property NumGlyphs;
    property OEMConvert;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnButtonClick;
    property OnChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;

    property OnGetImageIndex: TTVExpandedEvent read FOnGetImageIndex
                                               write SetOnGetImageIndex;
    property OnGetSelectedIndex: TTVExpandedEvent read FOnGetSelectedIndex
                                                  write SetOnGetSelectedIndex;

    property OnStartDrag;
{$IFDEF POLARIS_D4}
    property OnEndDock;
    property OnStartDock;
{$ENDIF}
  end;

implementation

uses DBConsts, DBUtils, MaxMin, extctrls, commctrl;

function TestKeyShiftState(Shift, TestShift: TShiftState): Boolean;
var
  B: Boolean;
begin
  B := False;
  if ssShift in TestShift then B := B or (ssShift in Shift);
  if ssAlt in TestShift then B := B or (ssAlt in Shift);
  if ssCtrl in TestShift then B := B or (ssCtrl in Shift);
  if ssLeft in TestShift then B := B or (ssLeft in Shift);
  if ssRight in TestShift then B := B or (ssRight in Shift);
  if ssMiddle in TestShift then B := B or (ssMiddle in Shift);
  if ssDouble in TestShift then B := B or (ssDOuble in Shift);
  Result := B;
end;

function VarEquals(const V1, V2: Variant): Boolean;
begin
  Result := False;
  if VarIsEmpty(V1) then Result := VarIsEmpty(V2)
  else if VarIsNull(V1) then Result := VarIsNull(V2)
    else if not VarIsEmpty(V2) and not VarIsNull(V2)
      then Result := V1=V2;
{
  Result := False;
  try
    Result := V1 = V2;
  except
  end;
}
end;

procedure FontSetDefault(Font: TFont);
begin
  Font.Name := 'MS Sans Serif';
  Font.Size := 8;
  Font.Style := [];
  Font.Color := clWindowText;
end;

{ TTreeDataList }

constructor TTreeDataList.Create;
begin
  inherited Create;
  FList   := TList.Create;
  FIndex := -1;
  with FEmpty do begin
    VarClear(KeyValue);
    VarClear(ChildValue);
    VarClear(ParentValue);
    Node := nil;
  end;
end;

destructor TTreeDataList.Destroy;
begin
  FList.Free;
  inherited Destroy;
end;

function TTreeDataList.Compare(Index1, Index2: Integer): Boolean;
begin
  Result := FullCompareEntry(TreeEntry[Index1], TreeEntry[Index2])
end;

function TTreeDataList.CompareEntryData(Index1, Index2: Integer): Boolean;
begin
  Result := CompareEntry(TreeEntry[Index1], TreeEntry[Index2])
end;

function TTreeDataList.CompareEntry(Value1, Value2: TTreeDataEntry): Boolean;
begin
  Result := VarEquals(Value1.KeyValue, Value2.KeyValue) and
            VarEquals(Value1.ChildValue, Value2.ChildValue) and
            VarEquals(Value1.ParentValue, Value2.ParentValue);
end;

function TTreeDataList.FullCompareEntry(Value1, Value2: TTreeDataEntry): Boolean;
begin
  Result := CompareEntry(Value1, Value2) and
            (Value1.Node <> Value2.Node);
end;

procedure TTreeDataList.ChangePosition(OldPosition, NewPosition: Integer);
begin
  FList.Move(OldPosition, NewPosition);
end;

procedure TTreeDataList.SetIndex(Item: Integer);
begin
  if Item <> FIndex then
    if (Item >= 0) and (Item < FList.Count)
    then begin
      FIndex := Item;
      FCurrent := TreeEntry[Item];
    end
    else begin
      FIndex := -1;
      FCurrent := FEmpty;
    end
end;

function TTreeDataList.GetTreeEntry(Index: Integer): TTreeDataEntry;
begin
  Result := TTreeDataEntry(FList.Items[Index]^);
end;

procedure TTreeDataList.SetTreeEntry(Index: Integer; AValue: TTreeDataEntry);
begin
  if not FullCompareEntry(AValue, TTreeDataEntry(FList.Items[Index]^))
  then with PTreeDataEntry(FList.Items[Index])^ do begin
    KeyValue  := AValue.KeyValue;
    ChildValue := AValue.ChildValue;
    ParentValue:= AValue.ParentValue;
    Node       := AValue.Node;
  end;
end;

function TTreeDataList.GetTreeNode(Index: Integer): TTreeNode;
begin
  Result := TreeEntry[Index].Node;
end;

procedure TTreeDataList.SetTreeNode(Index: Integer; Node: TTReeNode);
begin
  if PTreeDataEntry(FList.Items[Index])^.Node <> Node
  then PTreeDataEntry(FList.Items[Index])^.Node := Node;
end;

function TTreeDataList.GetItems(Index: Integer): PTreeDataEntry;
begin
  Result := PTreeDataEntry(FList.Items[Index]);
end;

procedure TTreeDataList.SetItems(Index: Integer; Item: PTreeDataEntry);
begin
  FList.Items[Index] := Item;
end;

function TTreeDataList.GetCount: Integer;
begin
  Result := FList.Count;
end;

procedure TTreeDataList.Add(const KeyValue, ChildValue, ParentValue: Variant);
var
  P: PTreeDataEntry;
begin
  New(P);
  P^.KeyValue   := KeyValue;
  P^.ChildValue  := ChildValue;
  P^.ParentValue := ParentValue;
  FList.Add(P);
end;

procedure TTreeDataList.AddEntry(const Entry: TTreeDataEntry);
var
  P: PTreeDataEntry;
begin
  New(P);
  with P^ do begin
    KeyValue   := Entry.KeyValue;
    ChildValue  := Entry.ChildValue;
    ParentValue := Entry.ParentValue;
    Node        := Entry.Node;
    Node.Data   := P;
  end;
  FList.Add(P);
end;

procedure TTreeDataList.Clear;
var
  I: Integer;
  P: PTreeDataEntry;
begin
  for I := 0 to FList.Count-1 do begin
    P := FList.Items[I];
    Dispose(P);
  end;
  FList.Clear;
end;

function TTreeDataList.IndexOf(Item: Pointer): Integer;
begin
  Result := FList.IndexOf(Item);
end;

function TTreeDataList.IndexOfEntry(Item: TTreeDataEntry): Integer;
var
  I: Integer;
begin
  Result := -1;
  for I := 0 to FList.Count-1 do
    if CompareEntry(Item, TTreeDataEntry(FList.Items[I]^))
    then begin
      Result := I;
      Break;
    end;
end;

function TTreeDataList.IndexOfKey(KeyValue: Variant): Integer;
var
  I: Integer;
begin
  Result := -1;
  for I := 0 to FList.Count-1 do
    if VarEquals(KeyValue, TTreeDataEntry(FList.Items[I]^).KeyValue)
    then begin
      Result := I;
      Break;
    end;
end;

function TTreeDataList.IndexOfParent(ParentValue: Variant): Integer;
var
  I: Integer;
begin
  Result := -1;
  for I := 0 to FList.Count-1 do
    if VarEquals(ParentValue, TTreeDataEntry(FList.Items[I]^).ParentValue)
    then begin
      Result := I;
      Break;
    end;
end;

function TTreeDataList.ChildrenPos(ChildValue: Variant; var Pos: Integer): Integer;
var
  I: Integer;
begin
  Result := -1;
  if (Pos >= 0) and (Pos < FList.Count) then
    for I := Pos to FList.Count-1 do
      if VarEquals(ChildValue, TTreeDataEntry(FList.Items[I]^).ChildValue)
      then begin
        Result := I;
        Break
      end;
end;

function TTreeDataList.IsEmptyEntry(Index: Integer): Boolean;
begin
  Result := FullCompareEntry(FEmpty, TreeEntry[Index]);
end;

{ TrTreeViewDataLink }

procedure TrTreeViewDataLink.ActiveChanged;
begin
  if FControl <> nil then FControl.ActiveChanged;
end;

//**
procedure TrTreeViewDataLink.DataSetScrolled(Distance: Integer);
begin
end;

procedure TrTreeViewDataLink.LayoutChanged;
begin
end;
//**

procedure TrTreeViewDataLink.DataSetChanged;
begin
  if FControl <> nil then FControl.DataChanged;
end;

procedure TrTreeViewDataLink.EditingChanged;
begin
  if FControl <> nil then FControl.EditingChanged;
end;

procedure TrTreeViewDataLink.RecordChanged(Field: TField);
begin
  if FControl <> nil then FControl.RecordChanged(Field);
end;

{TrCustomDBTreeView}

constructor TrCustomDBTreeView.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FLookupControl   := False;
  FHasMoved        := True;
  FIsResync        := False;
  FAutoSelect      := False;
  FListActive      := False;
  FEditNode        := nil;
  FillChar(FImageItedexes, SizeOf(FImageItedexes), -1);
  VarClear(FKeyValue);
  FValue            := '';
  FPrevState        := dsInactive;
  FCurrentRecord    := -1;
  FRecordCount      := 0;
  FRecordIndex      := 0;
  FRefreshEnabled   := True;
  FHasInsert        := False;
  FHasCreated       := False;
  FDataSet          := nil;
  FOnGetText        := nil;
  FStringList       := TStringList.Create;
  FStringList.Sorted:= True;
  FStringList.Duplicates := dupAccept;
  FListFields       := TList.Create;
  FTreeData         := TTreeDataList.Create;
  ReadOnly          := True;
  FSelectedEnabled  := False;
  FKeyFieldName     := '';
  FDisplayFields    := '';
  FChildrenFieldName:= '';
  FParentFieldName  := '';
  FKeyField         := nil;
  FChildrenField    := nil;
  FParentField      := nil;
  FLinkData         := TrTreeViewDataLink.Create;
  FLinkData.FControl:= Self;
  FSearchString     := '';
end;

destructor TrCustomDBTreeView.Destroy;
begin
  FEditNode := nil;
//  ClearNodes;
  FDataSet := nil;
  FLinkData.FControl := nil;
  FStringList.Free;
  FLinkData.Free;
  FListFields.Free;
  FTreeData.Free;
  inherited Destroy;
end;

function TrCustomDBTreeView.Locate(const S: String; var Index: Integer;
                                   var Node: TTreeNode): Boolean;
var
  L, H, I, C, J: Integer;
begin
  Result := False;
  Index := -1;
  Node  := nil;
  L := 0;
  H := FStringList.Count - 1;
  J := Length(S);
  if J = 0 then Exit;
  while L <= H do
  begin
    I := (L + H) shr 1;
    C := AnsiCompareText(Copy(FStringList.Strings[I],1,J), S);
    if C < 0 then L := I + 1 else
    begin
      H := I - 1;
      if C = 0 then
      begin
        Result := True;
        if FStringList.Duplicates <> dupAccept then L := I;
      end;
    end;
  end;
  if Result then begin
    Index := L;
    Node := TTreeNode(FStringList.Objects[L]);
  end;
end;

procedure TrCustomDBTreeView.DoEnter;
begin
  FHasMoved := False;
  Change(Selected);
  inherited DoEnter;
end;

procedure TrCustomDBTreeView.DoExit;
begin
  FHasMoved := True;
  inherited DoExit;
end;

function TrCustomDBTreeView.GetChildIndex: Integer;
begin
  Result := FImageItedexes[False, False]
end;

function TrCustomDBTreeView.GetChildSelected: Integer;
begin
  Result := FImageItedexes[False, True]
end;

function TrCustomDBTreeView.GetParentIndex: Integer;
begin
  Result := FImageItedexes[True, False]
end;

function TrCustomDBTreeView.GetParentSelected: Integer;
begin
  Result := FImageItedexes[True, True]
end;

function TrCustomDBTreeView.GetDisplayValue: String;
begin
  Result := '';
  if Selected <> nil
  then Result := Selected.Text;
end;

procedure TrCustomDBTreeView.SetChildIndex(AValue: Integer);
begin
  if FImageItedexes[False, False] <> AValue
  then begin
    if AValue < -1 then AValue := -1;
    FImageItedexes[False, False] := AValue;
    ChangeImages;
  end;
end;

procedure TrCustomDBTreeView.SetChildSelected(AValue: Integer);
begin
  if FImageItedexes[False, True] <> AValue
  then begin
    if AValue < -1 then AValue := -1;
    FImageItedexes[False, True] := AValue;
    ChangeImages;
  end;
end;

procedure TrCustomDBTreeView.SetParentIndex(AValue: Integer);
begin
  if FImageItedexes[True, False] <> AValue
  then begin
    if AValue < -1 then AValue := -1;
    FImageItedexes[True, False] := AValue;
    ChangeImages;
  end;
end;

procedure TrCustomDBTreeView.SetParentSelected(AValue: Integer);
begin
  if FImageItedexes[True, True] <> AValue
  then begin
    if AValue < -1 then AValue := -1;
    FImageItedexes[True, True] := AValue;
    ChangeImages;
  end;
end;

procedure TrCustomDBTreeView.SetSearchString(AValue: String);
var
  I: Integer;
  Node: TTreeNode;
begin
  I := -1;
  Node := nil;
  if (ANSICompareText(AValue, FSearchString) <> 0)
  then if Locate(AValue, I, Node)
    then FSearchString := AValue
    else FSearchString := '';
  Selected := Node;
end;

procedure TrCustomDBTreeView.ChangeImages;
var
  I: Integer;
begin
  for I := 0 to Items.Count-1 do
    GetImageIndexes(Items[I]);
end;

procedure TrCustomDBTreeView.GetImageIndexes(Node: TTreeNode);
begin
  if Node <> nil then begin
    Node.ImageIndex := FImageItedexes[Node.HasChildren, False];
    Node.SelectedIndex := FImageItedexes[Node.HasChildren, True]
  end;
end;

procedure TrCustomDBTreeView.KeyUp(var Key: Word; Shift: TShiftState);
begin
  inherited KeyUp(Key, Shift);
end;

function TrCustomDBTreeView.CanEdit(Node: TTreeNode): Boolean;
begin
  inherited CanEdit(Node);
  Result := False;
end;

procedure TrCustomDBTreeView.Change(Node: TTreeNode);
begin
  if not (csDestroying in ComponentState)
  then begin
    if not FHasCreated
    then SyncData(Node);
//    FSelectedEnabled := True;
  end;
  inherited Change(Node);
  KeyValueChanged;
end;

procedure TrCustomDBTreeView.InheritedReadOnly(AValue: Boolean);
begin
  inherited ReadOnly := AValue;
end;

procedure TrCustomDBTreeView.KeyValueChanged;
begin
  if Assigned(FOnKeyChanged) then FOnKeyChanged(Self, Selected);
end;

procedure TrCustomDBTreeView.ActiveChanged;
begin
  FListActive := False;
  if FDataSet <> FLinkData.DataSet
  then FDataSet := FLinkData.DataSet;
  if CheckActive then begin
    CheckNotCircular;
    InitFields(False);
    FLinkData.BufferCount := FDataSet.RecordCount;
    ReSyncRecords;
  end;
//  DataChanged;
end;

procedure TrCustomDBTreeView.ChangeChildOf(Node: TTreeNode; ChildValue: Variant);
var
  I:  Integer;
  NN,
  TN: TTreeNode;
  B:  Boolean;
begin
  if (FTreeData <> nil) and (FTreeData.Count > 0) and (Node <> nil) and (Node.Data <> nil)
  then begin
    I := FTreeData.IndexOfParent(ChildValue);
    B := False;
    if I >= 0
    then begin
      TN := FTreeData.TreeNode[I];
      B := not TN.HasChildren;
      if B then NN := Items.AddChild(TN, '')
      else NN := TN.GetLastChild;
    end
    else NN := Items.GetFirstNode;
    Node.MoveTo(NN, naAdd);
    if B then NN.Free;
    PTreeDataEntry(Node.Data)^.ChildValue := ChildValue;
    GetImageIndexes(Node);
  end;
end;

procedure TrCustomDBTreeView.ChangeParentOf(Node: TTreeNode; ParentValue: Variant);
var
  I, P: Integer;
  TN:   TTreeNode;
  B:    Boolean;
begin
  if (FTreeData <> nil) and (FTreeData.Count > 0) and (Node <> nil) and (Node.Data <> nil)
  then begin
    PTreeDataEntry(Node.Data)^.ParentValue := ParentValue;
    P := 0;
    while P >= 0 do begin
      I := FTreeData.ChildrenPos(ParentValue, P);
      if I < 0 then Break;
      if (FTreeData.IndexOf(Node.Data)<>I)
      and (Node <> FTreeData.TreeNode[I].Parent)
      then begin
        B := not Node.HasChildren;
        if B then TN := Items.AddChild(Node,'')
        else TN := Node.GetLastChild;
        FTreeData.TreeNode[I].MoveTo(TN, naAdd);
        if B then TN.Free;
      end;
      P := I+1;
    end;
    GetImageIndexes(Node);
  end;
end;

procedure TrCustomDBTreeView.ChangeDataOf(Node: TTreeNode; KeyValue: Variant);
begin
  if (FTreeData <> nil) and (FTreeData.Count > 0) and (Node <> nil) and (Node.Data <> nil)
  then begin
    PTreeDataEntry(Node.Data)^.KeyValue := KeyValue;
  end;
end;

procedure TrCustomDBTreeView.ReSyncRecords;
var
  L: Integer;
begin
  FIsResync := True;
  try
    FRecordCount := FLinkData.RecordCount;
    if FRecordCount <> FLinkData.BufferCount
    then begin
      L := FLinkData.ActiveRecord;
      FLinkData.BufferCount := FRecordCount;
      FLinkData.ActiveRecord := L;
    end;
//    if FLinkData.BufferCount < 5
//    then FLinkData.BufferCount := 5;
    FLinkData.DataSet.UpdateCursorPos;
    if FDataSet.State <> dsInsert then CreateNodes
  finally
    FIsResync := False;
  end;
end;

procedure TrCustomDBTreeView.DataChanged;
var
  TN: TTreeNode;
begin
  if CheckActive
  then begin
  {
    if (FEditNode <> nil) and (FCurrentRecord <> FLinkData.ActiveRecord)
    and (FPrevState = dsInsert)
    then begin
      Inc(FRecordCount, Byte(FPrevState = dsInsert));
      FTreeData.ChangePosition(FCurrentRecord, FLinkData.ActiveRecord);
    end;
  }
    if (FTreeData.Count > 0) and FHasMoved and not FHasCreated and (FCurrentRecord<>FLinkData.ActiveRecord)
    then begin
      TN := FTreeData.TreeNode[FLinkData.ActiveRecord];
      while (TN <> nil) and not TN.IsVisible do TN := TN.Parent;
      if TN <> nil then TopItem := TN;
    end;
//    if not FIsResync then begin
//          (?)
      if not (FPrevState in [dsInsert, dsEdit]) and
         ((FRefreshEnabled and (FCurrentRecord=FLinkData.ActiveRecord))
          or (FLinkData.RecordCount <> FLinkData.BufferCount))
      then ReSyncRecords;
//    end;
    FCurrentRecord := FLinkData.ActiveRecord;
  end
  else begin
    FCurrentRecord := -1;
    FRecordCount   := 0;
    FRecordIndex   := 0;
    FValue         := '';
    FKeyValue      := null;
    ClearNodes;
  end;
end;

procedure TrCustomDBTreeView.ChangeNodeText(Node: TTreeNode; const NewText: String);
var
  I: Integer;
begin
  if Node <> nil then begin
    I := FStringList.IndexOfObject(Node);
    if I >= 0 then begin
      if ANSICompareText(FStringList.Strings[I], NewText) <> 0
      then begin
        FStringList.Delete(I);
        FStringList.AddObject(NewText,Node);
      end;
    end
    else FStringList.AddObject(NewText,Node);
    Node.Text := NewText;
  end;
end;

procedure TrCustomDBTreeView.EditingChanged;
begin
  if CheckActive then begin
    FHasCreated := True;
    Case FLinkData.DataSet.State of
    dsEdit  : begin
                FCurrentRecord := FLinkData.ActiveRecord;
                FEditNode := FTreeData.TreeNode[FLinkData.ActiveRecord];
              end;
    dsInsert: begin
                with FTreeData.EmptyEntry do
                  FTreeData.Add(KeyValue, ChildValue, ParentValue);
                FCurrentRecord := FTreeData.Count-1;
                FEditNode := Items.AddChild(nil,'');
                FTreeData.TreeNode[FCurrentRecord] := FEditNode;
                ChangeNodeText(FEditNode, '');
                FLinkData.BufferCount := FLinkData.BufferCount+2;
              end;
    else begin
      FEditNode := nil;
      FHasCreated := False;
    end
    end;
    FPrevState := FLinkData.DataSet.State;
  end;
end;

procedure TrCustomDBTreeView.RecordChanged(Field: TField);
begin
  if (Field = nil) or (Field=FKeyField)
  or (Field=FChildrenField) or (Field=FParentField)
  or ((Field<>nil) and
      (Pos(AnsiUpperCase(Field.FieldName),
           AnsiUpperCase(FDisplayFields))>0))
  then begin
    if FEditNode <> nil then
      if Field <> nil then
        if Field = FChildrenField then ChangeChildOf(FEditNode, Field.Value)
        else if Field = FParentField then ChangeParentOf(FEditNode, Field.Value)
          else if Field = FKeyField then ChangeDataOf(FEditNode, Field.Value)
            else ChangeNodeText( FEditNode, DisplayString)
//              FEditNode.Text := DisplayString
      else begin
        ChangeChildOf (FEditNode, FChildrenField.Value);
        ChangeParentOf(FEditNode, FParentField.Value);
        CHangeDataOf  (FEditNode, FKeyField.Value);
//        FEditNode.Text := DisplayString;
        ChangeNodeText( FEditNode, DisplayString);
      end;
  end;
end;

function TrCustomDBTreeView.CheckSource: Boolean;
begin
  Result := FLinkData <> nil
end;

function TrCustomDBTreeView.CheckDataSet: Boolean;
begin
  Result := CheckSource and (FDataSet<>nil)
end;

function TrCustomDBTreeView.CheckActive: Boolean;
begin
  Result := CheckDataSet and FDataSet.Active
end;

function TrCustomDBTreeView.CheckBrowse: Boolean;
begin
  Result := CheckDataSet and (FDataSet.State in [dsBrowse]);
end;

procedure TrCustomDBTreeView.CheckNotCircular;
begin

end;

procedure TrCustomDBTreeView.SyncData(Node: TTreeNode);
var
  I: Integer;
begin
  if CheckActive
  then begin
    I := -1;
    if (Node <> nil)
    then I := FTreeData.IndexOf(Node.Data);
    if not (I < 0)
    then begin
      I := I - FLinkData.ActiveRecord;
      FLinkData.DataSet.MoveBy(I);
      I := FLinkData.ActiveRecord;
    end;
    FTreeData.Index := I;
    if I >= 0
      then begin
        SetValueKey(FKeyField.Value, FKeyField.AsString);
//        SelectKeyValue(FKeyField.Value);
      end
      else SetValueKey(null{FTreeData.EmptyEntry.KeyValue},'');
  end;
end;

procedure TrCustomDBTreeView.ClearNodes;
begin
  if not (csDestroying in ComponentState)
  then Items.BeginUpdate;
  try
    FTreeData.Clear;
    Items.Clear;
  finally
    if not (csDestroying in ComponentState)
    then Items.EndUpdate;
  end;
end;

procedure TrCustomDBTreeView.CreateNodes;
var
  L, I,
  J, P:    Integer;
  NN, TN:     TTreeNode;
  B:          Boolean;
  T:          TTreeDataEntry;
  List:       TTreeDataList;
  LST:        TSortType;

begin
  if (FKeyField = nil) or (FChildrenField = nil) or
     (FParentField = nil) or (FListFields.Count = 0)
     {(FDisplayFields = '')} then exit;
  LST := SortType;
  Items.BeginUpdate;
  try
    SortType := stNone;
    FDataSet.DisableControls;
    FHasCreated     := True;
    FRefreshEnabled := False;
    try
      L    := FLinkData.ActiveRecord;
      List := TTreeDataList.Create;
      for I := 0 to FRecordCount-1 do begin
        FLinkData.ActiveRecord := I;
        with T do begin
          KeyValue   := FKeyField.Value;
          ChildValue  := FChildrenField.Value;
          ParentValue := FParentField.Value;
//          Text        := DisplayString;
          Node        := nil;
        end;
        J := FTreeData.IndexOfEntry(T);
        if J >= 0 then T.Node := FTreeData.TreeNode[J]
        else begin
          J := List.IndexOfParent(T.ChildValue);
          TN := nil;
          if J >= 0 then TN := List.TreeNode[J];
          T.Node := Items.AddChild(TN, '');
        end;
        List.AddEntry(T);
//        if T.Node.Text <> DisplayString then T.Node.Text := DisplayString;
        ChangeNodeText(T.Node, DisplayString);
      end;
      for I := 0 to List.Count-1 do begin
        P := 0;
        while P >= 0 do begin
          J := List.ChildrenPos(List.TreeEntry[I].ParentValue, P);
          if J < 0 then Break;
          if (List.IndexOf(List.TreeNode[I].Data)<>J)
          and (List.TreeNode[I] <> List.TreeNode[J].Parent)
          then begin
            B := not List.TreeNode[I].HasChildren;
            if B then TN := Items.AddChild(List.TreeNode[I],'')
            else TN := List.TreeNode[I].GetLastChild;
            List.TreeNode[J].MoveTo(TN, naAdd);
            if B then TN.Free;
          end;
          P := J+1;
        end;
      end;
      TN := Items.GetFirstNode;
      GetImageIndexes(TN);
      B  := False;
      while TN <> nil do begin
        NN := TN.GetNext;
        I  := -1;
        if TN.Data <> nil
        then I := List.IndexOf(TN.Data);
        if not B and (Selected=TN) then B := True;
        if I < 0 then begin
          J := FStringList.IndexOfObject(TN);
          if J >= 0 then FStringList.Delete(J);
          TN.Free;
        end;
        TN := NN;
        GetImageIndexes(TN);
      end;
      if FTreeData <> nil then FTreeData.Free;
      FTreeData := List;
      if not(Items.GetFirstNode=nil) then begin //<--------------
        I := FTreeData.IndexOf(Items.GetFirstNode.Data);
        J := FTreeData.IndexOfKey(FKeyValue);
        if J < 0
        then if FAutoSelect
          then KeyValue := FTreeData.TreeEntry[I].KeyValue
          else KeyValue := FTreeData.EmptyEntry.KeyValue
        else I := J;
        if FLookupControl and (J < 0)
        then Selected := nil
        else Selected := FTreeData.TreeNode[I];
      end; //<--------------
      FTreeData.Index := L;
      FLinkData.ActiveRecord := L;
    finally
      FDataSet.EnableControls;
    end;
  finally
    FHasCreated     := False;
    FRefreshEnabled := True;
    SortType := LST;
    Items.EndUpdate;
    TopItem         := Selected;
  end;
end;

procedure TrCustomDBTreeView.RefreshNodes;
begin
  if CheckActive then
  CreateNodes;
end;

{
procedure TrCustomDBTreeView.ChangeDisplayNames;
var
  I, J, K: Integer;
begin
  FHasCreated := True;
  try
    K := FDataLink.ActiveRecord;
    for I := 0 to Items.Count-1 do begin
      J := FTreeData.IndexOf(Items[I].Data);
      if J >= 0 then begin
        FDataLink.ActiveRecord := J;
        Items[I].Text := DisplayString;
      end;
    end;
    FDataLink.ActiveRecord := K;
  finally
    FHasCreated := False;
  end;
end;
}

procedure TrCustomDBTreeView.InitFields(ShowError: Boolean);
var
  Pos: Integer;
  Dfn,
  Fld: String;
begin
  FListActive := False;
  FKeyField    := nil;
  FChildrenField   := nil;
  FParentField := nil;
  FListFields.Clear;
  if not CheckActive
  or (FChildrenFieldName = EmptyStr)
  or (FParentFieldName = EmptyStr) then Exit;
  with FDataSet do begin
    FChildrenField := FindField(FChildrenFieldName);
    if (FChildrenField = nil) and ShowError then
      raise EInvalidOperation.Create(Format(SFieldNotFound, [Self.Name, FChildrenFieldName]))
    else if FChildrenField <> nil
      then begin
        FParentField := FindField(FParentFieldName);
        if (FParentField = nil) and ShowError then
          raise EInvalidOperation.Create(Format(SFieldNotFound, [Self.Name, FParentFieldName]))
        else if FParentField <> nil then begin
          if FKeyFieldName <> EmptyStr
          then begin
            FKeyField := FindField(FKeyFieldName);
            if (FKeyField = nil) and ShowError then
              raise EInvalidOperation.Create(Format(SFieldNotFound, [Self.Name, FKeyFieldName]))
          end;
          if FKeyField = nil then FKeyField := FParentField;
          Pos := 1;
          if FDisplayFields = EmptyStr then Dfn := FKeyFieldName
          else Dfn := FDisplayFields;
          while Pos < Length(Dfn) do begin
            Fld := ExtractFieldName(Dfn, Pos);
            if (FindField(Fld)=nil)
            and ShowError then
              raise EInvalidOperation.Create(Format(SFieldNotFound, [Self.Name, Fld]))
          end;
          GetFieldList(FListFields, Dfn);
          FListActive := True;
        end
      end
  end;
end;

procedure TrCustomDBTreeView.NewLayout;
begin
  InitFields(True);
//  LayoutChange;
  RefreshNodes;
//  ChangeDisplayNames;
//  DataChange;
end;

function TrCustomDBTreeView.GetLinkDataSource: TDataSource;
begin
  Result := FLinkData.DataSource;
end;

procedure TrCustomDBTreeView.SetLinkDataSource(AValue: TDataSource);
begin
  ClearNodes;
  FLinkData.DataSource := AValue;
  if AValue <> nil then AValue.FreeNotification(Self);
end;

procedure TrCustomDBTreeView.SetKeyFieldName(AValue: String);
begin
  if ANSICompareText(AValue, FKeyFieldName) <> 0
  then begin
    FKeyFieldName := AValue;
    NewLayout;
  end;
end;

procedure TrCustomDBTreeView.SetChildrenFieldName(AValue: String);
begin
  if ANSICompareText(AValue, FParentFieldName) = 0
  then raise Exception.Create(trdbtreeEqualsField);
  if ANSICompareText(AValue, FChildrenFieldName) <> 0
  then begin
    FChildrenFieldName := AValue;
    NewLayout;
  end;
end;

procedure TrCustomDBTreeView.SetParentFieldName(AValue: String);
begin
  if ANSICompareText(AValue, FChildrenFieldName) = 0
  then raise Exception.Create(trdbtreeEqualsField);
  if ANSICompareText(AValue, FParentFieldName) <> 0
  then begin
    FParentFieldName := AValue;
    NewLayout;
  end;
end;

procedure TrCustomDBTreeView.SetDisplayFields(AValue: String);
begin
  if ANSICompareText(AValue, FDisplayFields) <> 0
  then begin
    FDisplayFields := AValue;
    NewLayout;
  end;
end;

function TrCustomDBTreeView.GetListField(Index: Integer): TField;
begin
  Result := nil;
  if (ListFieldCount > 0) and (Index <= ListFieldCount)
  then Result := TField(FListFields.Items[Index])
end;

function TrCustomDBTreeView.GetListFieldCount: Integer;
begin
  Result := FListFields.Count;
end;

function TrCustomDBTreeView.DisplayString: String;
var
  I: Integer;
  S: String;
begin
  Result := '';
  if FListFields.Count > 0
  then for I := 0 to FListFields.Count-1 do begin
    S := TField(FListFields.Items[I]).DisplayText;
    if Result = EmptyStr
    then Result := S
    else Result := Result+' '+S;
  end
  else if FKeyField <> nil
    then Result := FKeyField.DisplayText;
  if Assigned(FOnGetText)
  then FOnGetText(Self, FDataSet, Result);
end;

procedure TrCustomDBTreeView.Notification(AComponent: TComponent; Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (FLinkData <> nil) and (AComponent = LinkDataSource)
  then begin
    ClearNodes;
    LinkDataSource := nil;
  end
end;

procedure TrCustomDBTreeView.SetValueKey(const AValue: Variant; const AString: String);
var
  S: String;
begin
  if not VarEquals(FKeyValue, AValue)
  then begin
    FKeyValue := AValue;
    S := Trim(AString);
    if VarIsEmpty(AValue)
    then S := 'EMPTY'
    else if VarIsNull(AValue)
      then S := 'NULL';
    FValue    := AString;
    SelectKeyValue(FKeyValue);
  end;
end;

procedure TrCustomDBTreeView.SelectKeyValue(const AValue: Variant);
begin
end;

procedure TrCustomDBTreeView.SetValue(AValue: String);
var
  V: Variant;
begin
  if (AValue <> FValue) or (AValue='')
  then begin
    AValue  := Trim(AValue);
    FValue  := AValue;
    if (AnsiCompareText(AValue,'NULL')=0) or (FValue=EmptyStr)
    then V := Null
    else if AnsiCompareText(AValue,'EMPTY')=0
      then VarClear(V)
      else V := AValue;
    if ANSICompareText(AValue, 'TRUE') = 0
    then KeyValue := True
    else if ANSICompareText(AValue,'FALSE') = 0
      then KeyValue := False
      else KeyValue := V;
  end;
end;

procedure TrCustomDBTreeView.SetKeyValue(AValue: Variant);
var
  I: Integer;
begin
  if not VarEquals(AValue, FKeyValue)
  then begin
    if FLinkData.Active
    then begin
      if FHasCreated and FAutoSelect then SetValueKey(AValue, VarToStr(AValue));
      I := FTreeData.IndexOfKey(AValue);
      if I < 0 then Selected := nil
      else if Selected <> FTreeData.TreeNode[I]
        then Selected := FTreeData.TreeNode[I];
    end
    else SetValueKey(AValue, VarToStr(AValue));
  end;
end;

{ TrDBTreeView }

{
function TrDBTreeView.GetDataSource: TDataSource;
begin
  Result := LinkDataSource;
end;

procedure TrDBTreeView.SetDataSource(Value: TDataSource);
begin
  LinkDataSource := Value;
end;
}
{ TrDataSourceLink }

procedure TrDataSourceLink.ActiveChanged;
begin
  if FControl <> nil then FControl.DataLinkActiveChanged;
end;

procedure TrDataSourceLink.LayoutChanged;
begin
  if FControl <> nil then FControl.CheckDataLinkActiveChanged;
end;

procedure TrDataSourceLink.RecordChanged(Field: TField);
begin
  if FControl <> nil then FControl.DataLinkRecordChanged(Field);
end;

procedure TrDataSourceLink.FocusControl(Field: TFieldRef);
begin
  if (Field^ <> nil) and (FControl <> nil) and
    (Field^ = FControl.FDataField) and FControl.CanFocus then
  begin
    Field^ := nil;
    FControl.SetFocus;
  end;
end;

{???}
procedure TrDataSourceLink.EditingChanged;
begin
  if FControl <> nil then FControl.DataLinkEditChanged;
end;

{ TrDBLookupTree }

constructor TrDBLookupTree.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FReadOnly      := False;
  FLookupControl := True;
  FDataField   := nil;
  FMasterField := nil;
  FHasEdit     := False;
  FDataLink    := TrDataSourceLink.Create;
  FDataLink.FControl := Self;
end;

destructor TrDBLookupTree.Destroy;
begin
  FDataLink.FControl := nil;
  FDataLink.Free;
  inherited Destroy;
end;

procedure TrDBLookupTree.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if Operation = opRemove then begin
    if (FDataLink <> nil) and (AComponent = DataSource) then DataSource := nil;
  end;
end;

procedure TrDBLookupTree.CheckNotCircular;
begin
  if FDataLink.Active and ((DataSource = LookupSource)
     or (FDataLink.DataSet = FLinkData.DataSet))
  then _DBError(SCircularDataLink);
end;

procedure TrDBLookupTree.DataLinkActiveChanged;
begin
  FDataField   := nil;
  FMasterField := nil;
  if FDataLink.Active and (FDataFieldName <> '') then begin
    CheckNotCircular;
    FDataField := FDataLink.DataSet.FieldByName(FDataFieldName);
    FMasterField := FDataField;
  end;
  DataLinkRecordChanged(nil);
end;

procedure TrDBLookupTree.CheckDataLinkActiveChanged;
var
  TestField: TField;
begin
  if FDataLink.Active and (FDataFieldName <> '') then begin
    TestField := FDataLink.DataSet.FieldByName(FDataFieldName);
    if Pointer(FDataField) <> Pointer(TestField) then begin
      FDataField := nil;
      FMasterField := nil;
      CheckNotCircular;
      FDataField := TestField;
      FMasterField := FDataField;
      DataLinkRecordChanged(nil);
    end;
  end;
end;

{???}
procedure TrDBLookupTree.DataLinkEditChanged;
begin
  FHasEdit := FDataLink.DataSet.State in [dsInsert, dsEdit];
  if FHasEdit and (FMasterField <> nil)
  then FPrevValue := FMasterField.Value
  else VarClear(FPrevValue);
end;

{***}
procedure TrDBLookupTree.DataLinkRecordChanged(Field: TField);
begin
  if (Field = nil) or (Field = FMasterField) then begin
    if (FMasterField <> nil) then begin
      if FHasEdit and VarEquals(FMasterField.Value, FPrevValue) then Exit;
      Value := FMasterField.AsString;
      if FHasEdit and not VarEquals(FMasterField.Value, FPrevValue)
      then FPrevValue := FMasterField.Value;
    end// else Value := '';
  end;
end;

procedure TrDBLookupTree.SetReadOnly(AValue: Boolean);
begin
  FReadOnly := AValue;
end;

procedure TrDBLookupTree.SetDataFieldName(AValue: String);
begin
  if FDataFieldName <> AValue then begin
    FDataFieldName := AValue;
    DataLinkActiveChanged;
  end;
end;

function TrDBLookupTree.GetDataSource: TDataSource;
begin
  Result := FDataLink.DataSource;
end;

procedure TrDBLookupTree.SetDataSource(AValue: TDataSource);
begin
  FDataLink.DataSource := AValue;
  if AValue <> nil then AValue.FreeNotification(Self);
end;

function TrDBLookupTree.CanModify: Boolean;
begin
  Result := FListActive and not FMasterField.ReadOnly and ((FDataLink.DataSource = nil) or
    (FMasterField <> nil) and FMasterField.CanModify);
end;

//procedure SetKeyValue(AValue: Variant);

procedure TrDBLookupTree.SelectKeyValue(const AValue: Variant);
begin
  if FMasterField <> nil then begin
    if (VarType(AValue) = varUnknown) or VarEquals(FMasterField.Value, AValue) then Exit;
    if CanModify then begin
      if FDataField = FMasterField
      then FDataField.DataSet.Edit;
      FHasEdit := True;
      FMasterField.Value := AValue;
      FHasEdit := False;
    end
//    else Exit;
  end;
//  else SetValueKey(AValue);
end;

{ TrPopupTree }

type
  TLocDBLookupTree = class(TrCustomDBTreeView)
  private
    FRowCount: Integer;

    procedure CreateNodes; override;
//    function GetDisplayText: String;
    procedure SetRowCount(AValue: Integer);
    procedure CMEnableChanged(var Message: TMessage); message CM_ENABLEDCHANGED;
    procedure WMMouseActivate(var Message: TMessage); message WM_MOUSEACTIVATE;

  protected
//    procedure DataChanged; override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure CreateParams(var Params: TCreateParams); override;
    function CalcHeight: Integer;

//    procedure SetValueKey(const AValue: Variant; const AString: String); override; //dynamic;

  public
    constructor Create(AOwner: TComponent); override;
//    property DisplayText: String read GetDisplayText;
    property RowCount:Integer read FRowCount write SetRowCount;
    property SelectEnabled: Boolean read FSelectedEnabled;
  end;

constructor TLocDBLookupTree.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
//  ControlStyle := [csCaptureMouse, csClickEvents, csDoubleClicks];
  ControlStyle := [csClickEvents, csDoubleClicks];
  FLookupControl := True;
  HideSelection := False;
  Ctl3D        := False;
  Enabled      := False;
  BorderStyle  := bsNone;
  TabStop      := False;
  FRowCount    := 7;
  Height       := CalcHeight;
end;

procedure TLocDBLookupTree.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
 with Params do
   Style := Style and not (WS_BORDER or WS_TABSTOP or WS_DISABLED or WS_POPUP);
end;

procedure TLocDBLookupTree.WMMouseActivate(var Message: TMessage);
begin
  Message.Result := MA_NOACTIVATE;
end;

procedure TLocDBLookupTree.CreateNodes;
begin
  TrPopupTree(Parent).FIsCreateTree := False;
  inherited CreateNodes;
  TrPopupTree(Parent).FIsCreateTree := True;
  TrPopupTree(Parent).EndCreated;
end;

procedure TLocDBLookupTree.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  FSelectedEnabled := htOnItem in GetHitTestInfoAt(X,Y);
  inherited MouseDown(Button, Shift, X, Y);
  if htOnItem in GetHitTestInfoAt(X,Y)
  then TrDBLookupComboTree(Parent.Parent).PopupCloseUp(Parent.Parent, True);
end;

procedure TLocDBLookupTree.MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  inherited MouseUp(Button, Shift, X, Y);
  FSelectedEnabled := False;
end;

procedure TLocDBLookupTree.SetRowCount(AValue: Integer);
begin
  if FRowCount <> AValue
  then begin
    FRowCount := AValue;
    if FRowCount < 1 then FRowCount := 1;
    if FRowCount > 50 then FRowCount := 50;
    Height := CalcHeight;
  end;
end;

function TLocDBLookupTree.CalcHeight: Integer;
var
  DC: HDC;
  SaveFont: HFont;
  SysMetrics, Metrics: TTextMetric;
  CH: Integer;
begin
  DC := GetDC(0);
  GetTextMetrics(DC, SysMetrics);
  SaveFont := SelectObject(DC, Font.Handle);
  GetTextMetrics(DC, Metrics);
  SelectObject(DC, SaveFont);
  ReleaseDC(0, DC);
  CH := SysMetrics.tmHeight;
  Result := 0;   // ??????
  if (Images <> nil) and (Result < Images.Height) then CH := Images.Height;
  Result := CH*FRowCount+GetSystemMetrics(SM_CXHSCROLL)+2;
end;

procedure TLocDBLookupTree.CMEnableChanged(var Message: TMessage);
begin
  if HandleAllocated and not (csDesigning in ComponentState) then
    EnableWindow(Handle, True);
end;
{
function TLocDBLookupTree.GetDisplayText: String;
begin
  if Selected = nil
  then Result := TrPopupTree(Parent).SNoneLabel
  else Result := Selected.Text;
end;

procedure TLocDBLookupTree.SetValue(AValue: String);
begin
  inherited SetValue(AValue);
end;

procedure TLocDBLookupTree.SetKeyValue(AValue: Variant);
begin
  inherited SetKeyValue(AValue);
end;
}
type
  TLocLabel = class(TStaticText)
  private
    procedure WMMouseActivate(var Message: TMessage); message WM_MOUSEACTIVATE;
    procedure CMEnableChanged(var Message: TMessage); message CM_ENABLEDCHANGED;
  protected
    procedure CreateParams(var Params: TCreateParams); override;
  public
    constructor Create(AOwner: TComponent); override;
  end;

constructor TLocLabel.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
//  ControlStyle := [csCaptureMouse, csClickEvents, csDoubleClicks];
  ControlStyle := [csClickEvents, csDoubleClicks];
  Ctl3D        := False;
  Enabled      := False;
end;


procedure TLocLabel.WMMouseActivate(var Message: TMessage);
begin
  Message.Result := MA_NOACTIVATE;
end;

procedure TLocLabel.CMEnableChanged(var Message: TMessage);
begin
  if HandleAllocated and not (csDesigning in ComponentState) then
    EnableWindow(Handle, True);
end;

procedure TLocLabel.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  with Params do
    Style := Style and not (WS_BORDER or WS_TABSTOP or WS_DISABLED or WS_POPUP);
end;

{ TrPopupTree }

constructor TrPopupTree.Create(AOwner: TComponent);
var
  Control: TPanel;
begin
  inherited Create(AOwner);
  FIsCreateTree := False;
  ControlStyle := [csOpaque];
  Parent := TWinControl(AOwner);
  SNoneLabel := srNone;
  FDropDownRows := 8;
  Height := 121;
//  Width  := 115;
  Width  := 300;
//  Ctl3D  := False;
  Color  := clBtnFace;
  FTreeList := nil;
  FontSetDefault(Font);
  OnMouseUp := TreeMouseUp;
  if AOwner is TControl then ShowHint := TControl(AOwner).ShowHint
  else ShowHint := True;
  if (csDesigning in ComponentState) then Exit;
  Control := TPanel.Create(Self);
  with Control as TPanel do begin
    Parent := Self;
    Align := alTop;
    ParentColor := True;
    Width := Self.Width;
    Height := 18;
    BevelInner := bvNone;
    BevelOuter := bvNone;
    Ctl3D  := False;
    ParentColor := True;
    ControlStyle := ControlStyle + [csReplicatable];
  end;
  FTreeList := TLocDBLookupTree.Create(Self);
  with FTreeList as TLocDBLookupTree do begin
    Parent := Self;
    Align  := alClient;
    RowCount := FDropDownRows-1;
//    Self.Height := 18+Height;
    Self.Height := 18+Height;
    Ctl3D  := False;
    AutoSelectItem := False;
    OnChange := TreeChange;
    OnMouseUp := TreeMouseUp;
  end;
  FNoneLabel := TLocLabel.Create(Self);
  with FNoneLabel do begin
    Parent := Control;
    Color := FTreeList.Color;
    Align := alClient;
    Caption := SNoneLabel;
    OnClick := LabelClick;
    ControlStyle := ControlStyle + [csReplicatable];
  end;
end;

procedure TrPopupTree.SetNoneLabel(AValue: String);
begin
  if SNoneLabel <> Trim(AValue)
  then begin
    SNoneLabel := Trim(AValue);
    FNoneLabel.Caption := SNoneLabel;
  end;
end;

procedure TrPopupTree.EndCreated;
begin
  TrDBLookupComboTree(Parent).DataChange(Parent);
end;

procedure TrPopupTree.WMKeyDown(var Message: TWMKeyDown);
var
  M: TMessage;
  Key: Word;
  Shift: TShiftState;
  B: Boolean;
begin
  M := TMessage(Message);
  Shift := KeyDataToShiftState(Message.KeyData);
  Key   := Message.CharCode;
  B := TestKeyShiftState(Shift, [ssShift, ssAlt, ssCtrl]);
  if (Key in [VK_HOME, VK_END, VK_LEFT, VK_RIGHT])
  or (not Visible and B and (Key in [VK_UP, VK_DOWN, VK_PRIOR, VK_NEXT])) then Exit;
  inherited;
//  if not Visible and B then Exit;
  if not (Visible or (Key in [VK_UP, VK_DOWN])) then Message.Result := 1;
  if not B
  then case Key of
    VK_PRIOR,
    VK_UP: with TLocDBLookupTree(FTreeList) do
      if (Selected = nil)
      or ((Selected <> nil) and (Selected = Items[0]))
      then begin
        Self.SetValue(null);
        Message.Result := 1;
    end;
{
    VK_RETURN: if Visible then begin
      TrDBLookupComboTree(Parent).PopupCloseUp(Parent, True);
      Message.Result := 1;
    end;
    VK_ESCAPE: begin
      if Visible
      then TrDBLookupComboTree(Parent).PopupCloseUp(Parent, False)
      else Self.SetValue(null);
      Message.Result := 1;
    end;
}
  end;
  if (Message.Result = 0)
  then
    FTreeList.Perform(M.Msg, M.WParam, M.LParam);
end;

{
procedure TrPopupTree.WMChar(var Message: TWMChar);
var
  M: TMessage;
begin
//  M := TMessage(Message);
  inherited;
//  if Visible
//  then FTreeList.Perform(M.Msg, M.WParam, M.LParam);
end;
}

procedure TrPopupTree.WMMouseActivate(var Message: TMessage);
begin
  Message.Result := MA_NOACTIVATE;
end;

function TrPopupTree.GetPopupText: String;
begin
  Result := DisplayValue;
end;

procedure TrPopupTree.Show(Origin: TPoint);
begin
  inherited Show(Origin);
  with TLocDBLookupTree(FTreeList) do begin
    if Items.Count = 0 then TopItem := nil
    else if Selected <> nil then TopItem := Selected
      else TopItem := Items[0];
  end;
end;

procedure TrPopupTree.SetDropDownRows(AValue: Integer);
begin
  if AValue <> FDropDownRows then begin
    FDropDownRows := AValue;
    if FDropDownRows < 2 then FDropDownRows := 2;
    if FDropDownRows > 50 then FDropDownRows := 50;
    if not (csDesigning in ComponentState) and (FTreeList <> nil)
    then with FTreeList as TLocDBLookupTree do begin
      RowCount := FDropDownRows-1;
      Self.Height := 18+Height
    end;
  end;
end;

procedure TrPopupTree.TreeMouseUp(Sender: TObject; Button: TMouseButton;
                          Shift: TShiftState; X, Y: Integer);
begin
  if (Button = mbLeft) and (Shift = []) then begin
    if htOnItem in TLocDBLookupTree(FTreeList).GetHitTestInfoAt(X,Y) then
      CloseUp(True);
  end;
end;

procedure TrPopupTree.LabelClick(Sender: TObject);
begin
  SetValue(null);
  TrDBLookupComboTree(Parent).PopupCloseUp(Parent, True);
end;

procedure TrPopupTree.TreeChange(Sender: TObject; Node: TTreeNode);
begin
  if Node = nil
  then begin
    FNoneLabel.Color := clBtnFace;
  end
  else begin
    FNoneLabel.Color := FTreeList.Color;
  end;
//  if Assign(OnCha
//  TrDBLookupComboTree(Parent).Text := DisplayValue;
end;

procedure TrPopupTree.KeyDown(var Key: Word; Shift: TShiftState);
begin
  inherited KeyDown(Key, Shift);
  if FTreeList <> nil
  then FTreeList.KeyDown( Key, Shift);
end;

procedure TrPopupTree.KeyPress(var Key: Char);
begin
  if Key in ['+','-']
  then begin
    if (FTreeList <> nil) and (FTreeList.Selected <> nil) and
       (FTreeList.Selected.HasChildren) and Visible
    then
      case Key of
      '-': if FTreeList.Selected.Expanded then FTreeList.Selected.Collapse(True);
      '+': if not FTreeList.Selected.Expanded then FTreeList.Selected.Expand(True);
      end;
    Key := #0;
  end;
  inherited KeyPress(Key);
end;
function TrPopupTree.GetValue: Variant;
begin
  if (csDesigning in ComponentState)
  then Result := null
  else begin
    if (FTreeList <> nil) and (TLocDBLookupTree(FTreeList).Selected <> nil)
    then Result := TLocDBLookupTree(FTreeList).KeyValue
    else Result := null
  end;
end;

procedure TrPopupTree.SetValue(const Value: Variant);
begin
  if VarIsEmpty(Value) or VarIsNull(Value)
  then begin
    TLocDBLookupTree(FTreeList).Value := '';
    TreeChange(Self, nil)
  end
  else TLocDBLookupTree(FTreeList).KeyValue := Value;
end;

function TrPopupTree.GetDisplayValue: String;
begin
  Result := TLocDBLookupTree(FTreeList).DisplayValue;
  if (Result = '') and
    (VarIsNull(TLocDBLookupTree(FTreeList).KeyValue)
  or VarIsEmpty(TLocDBLookupTree(FTreeList).KeyValue))
  then Result := SNoneLabel;
end;

function TrPopupTree.GetSearchString: String;
begin
  Result := TLocDBLookupTree(FTreeList).FSearchString;
end;

procedure TrPopupTree.SetSearchString(AValue: String);
begin
  TLocDBLookupTree(FTreeList).SetSearchString(AValue);
end;

{ TrDBLookupComboTree }

type
  ComboCrack = class(TCustomComboEdit);

constructor TrDBLookupComboTree.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
//  ControlStyle := ControlStyle - [csCaptureMouse];
//  ReadOnly := False;
//  FInListDown := False;
//  AlwaysEnable := True;     
//  DirectInput    := True;
  PopupAlign     := epaLeft;
  Parent         := TWinControl(AOwner);
  AlwaysEnable   := False;
  FChanging      := False;
  FListVisible   := False;
  SNoneLabel     := srNone;
//  FDropDownWidth := 197;
  FDropDownWidth := 0;

  FDataLink                 := TFieldDataLink.Create;
  FDataLink.Control         := Self;
  FDataLink.OnActiveChange  := ActiveChange;
  FDataLink.OnDataChange    := DataChange;
  FDataLink.OnEditingChange := EditingChange;
  FDataLink.OnUpdateData    := UpdateData;

  EditMask  := '';
  Alignment := taLeftJustify;

  FOnGetImageIndex    := nil;
  FOnGetSelectedIndex := nil;

  FValue         := '';
  FKeyValue      := null;
//  FDisplayString := SDefNoneLabel;
  FDisplayString := SNoneLabel;

//
//    GlyphKind := gkDropDown; { force update }
//    ComboCrack(Self).FButton.FStandart := False;
//    ControlState := ControlState - [csCreating];
//

  try
    if not (csDesigning in ComponentState)
    then begin
      FPopup := TrPopupTree.Create(Self);
      TrPopupTree(FPopup).FTreeList.OnKeyChanged := KeyChanged;
      TrPopupTree(FPopup).SearcString := '';
    end;
    GlyphKind := gkDropDown; { force update }
    ComboCrack(Self).FButton.FStandart := False;
  finally
    ControlState := ControlState - [csCreating];
  end;
  if (csDesigning in ComponentState)
  then begin
    FDTDataSource    := nil;
    FDTLookupField   := '';
    FDTLookupDisplay := '';
    FDTChildrenField := '';
    FDTParentField   := '';
    FDTChildIndex    := -1;
    FDTChildSelected := -1;
    FDTParentIndex   := -1;
    FDTParentSelected:= -1;
    FDTImages        := nil;
    FDTDropDownCount := 8;
  end
  else begin

    ChildIndex     := -1;
    ChildSelected  := -1;
    ParentIndex    := -1;
    ParentSelected := -1;
    DropDownCount  := 8;
//    FDropDownWidth := Width;
    Images         := nil;
  end;
  FOnDropDown   := nil;
  FOnCloseUp    := nil;
end;

destructor TrDBLookupComboTree.Destroy;
begin
//  FDataList.Free;
  FDTDataSource := nil;
  FDTImages     := nil;
  FDataLink.Free;
  FDataLink := nil;
  inherited Destroy;
end;

procedure TrDBLookupComboTree.ActiveChange(Sender: TObject);
begin
  if FDataLink.Active then begin
    CheckNotCircular;
  end;
  DataChange(Sender);
end;

procedure TrDBLookupComboTree.DataChange(Sender: TObject);
//var I : Integer;
begin
  if (FDataLink.Field <> nil) and not (csDesigning in ComponentState) then
  begin

    if VarIsEmpty(FDataLink.Field.Value) or VarIsNull(FDataLink.Field.Value)
    then begin
       FValue := '';
       TrPopupTree(FPopup).SetValue(null);
       FDisplayString := SNoneLabel;
    end
    else begin
//      if not TrPopupTree(FPopup).FIsCreateTree then Exit;
      FValue := FDataLink.Field.Value;
      TrPopupTree(FPopup).SetValue(FDataLink.Field.Value);
      FDisplayString := TrPopupTree(FPopup).DisplayValue;
    end

//    SetKeyValue(FDataLink.Field.Value);
{
    FValue := FDataLink.Field.Value;
    TrPopupTree(FPopup).SetValue(FDataLink.Field.Value);
    FDisplayString := TrPopupTree(FPopup).DisplayValue;
}
  end
  else begin
    if csDesigning in ComponentState then EditText := Name
    else if (FDataLink.Field <> nil) then begin
      FValue := '';
      TrPopupTree(FPopup).SetValue(null);
      FDisplayString := SNoneLabel;
    end
//      KeyValue := null;
  end;
//  Text := FDisplayString;
  Text := FDisplayString;
//  SelectAll;
end;

procedure TrDBLookupComboTree.EditingChange(Sender: TObject);
begin
//  inherited ReadOnly := not FDataLink.Editing;
end;

procedure TrDBLookupComboTree.UpdateData(Sender: TObject);
begin
  ValidateEdit;

//  FDataLink.Field.Text := Text;
end;

procedure TrDBLookupComboTree.KeyChanged(Sender: TObject; Node: TTreeNode);
begin
//  Text := TrPopupTree(FPopup).GetDisplayValue;
  if not (csDesigning in ComponentState)
  then KeyValue := TrPopupTree(FPopup).FTreeList.FKeyValue;
//  SetCursor(Length(Text));
//  AcceptValue(KeyValue);
  if not FPopupVisible then begin
    AcceptValue(KeyValue);
  end;
  Invalidate;
//  if not CanModify then Text := FDisplayString;
end;

procedure TrDBLookupComboTree.CMGetDataLink(var Message: TMessage);
begin
  Message.Result := Integer(FDataLink);
end;

procedure TrDBLookupComboTree.CMEnabledChanged(var Message: TMessage);
begin
  inherited;
  Invalidate;
end;

procedure TrDBLookupComboTree.WMKillFocus(var Message: TWMKillFocus);
var
  H: {$IFDEF POLARIS_D4} HWND {$ELSE} Integer {$ENDIF};
begin
  H := Message.FocusedWnd;
  if (H = FPopup.Handle) or (H = Handle)
  then begin
//    if FInListDown and TLocDBLookupTree(TrPopupTree(FPopup).FTreeList).SelectEnabled
//    then inherited;
//    FInListDown := H = FPopup.Handle;
    if H <> FPopup.Handle
    then begin
      Message.FocusedWnd := Handle;
      Message.Result := 1;
    end
    else Windows.SetFocus(Handle);
//    if (Message.FocusedWnd = FPopup.Handle)
//    then Windows.SetFocus(Handle);
  end
  else inherited;
end;

procedure TrDBLookupComboTree.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) then begin
    if (FDataLink <> nil) and (AComponent = DataSource) then DataSource := nil;
    if AComponent = Images then Images := nil;
  end
end;

procedure TrDBLookupComboTree.WKKeyDown(var Message: TWMKeyDown);
var
  M : TMessage;
begin
  M := TMessage(Message);
  if not ((Message.CharCode in [VK_UP, VK_PRIOR, VK_NEXT]) or
     (not TestKeyShiftState(KeyDataToShiftState(Message.KeyData), [ssAlt])
     and (Message.CharCode = VK_DOWN)))
  then inherited;
  if (FPopup <> nil) and (Message.Result = 0) and not ReadOnly
  then
    FPopup.Perform(M.Msg, M.WParam, M.LParam);
end;

procedure TrDBLookupComboTree.SetValue(AValue: String);
var
  I: Integer;
begin
  if (csDesigning in ComponentState) then exit;
  if FValue <> AValue
  then begin
    I := TrPopupTree(FPopup).FTreeList.FTreeData.IndexOfKey(AValue);
    if I >= 0 then begin
      FValue := AValue;
      FKeyValue := TrPopupTree(FPopup).FTreeList.FTreeData.TreeEntry[I].KeyValue;
      FDisplayString := TrPopupTree(FPopup).FTreeList.FTreeData.TreeNode[I].Text;
    end
    else begin
      FValue := '';
      FKeyValue := null;
      FDisplayString := SNoneLabel;
    end;
{
    if (DataSource = nil)
    then Text := FDisplayString
    else if CanModify and FDataLink.Edit and not FPopupVisible then begin
        FDataLink.Field.AsString := FValue;
      end;
}
  end;
end;

procedure TrDBLookupComboTree.SetKeyValue(AValue: Variant);
var
  I: Integer;
begin
  if (csDesigning in ComponentState) then exit;
  if not VarEquals(FKeyValue, AValue)
  then begin
    I := TrPopupTree(FPopup).FTreeList.FTreeData.IndexOfKey(AValue);
    if I >= 0 then begin
      FValue := VarToStr(AValue);
      FKeyValue := TrPopupTree(FPopup).FTreeList.FTreeData.TreeEntry[I].KeyValue;
      FDisplayString := TrPopupTree(FPopup).FTreeList.FTreeData.TreeNode[I].Text;
    end
    else begin
      FValue := '';
      FKeyValue := null;
      FDisplayString := SNoneLabel;
    end;
  if not (Assigned(FDataLink) and Assigned(FDataLink.Field))
  then Text := FDisplayString;

{
    if (DataSource = nil)
    then Text := FDisplayString
    else if CanModify and FDataLink.Edit and not FPopupVisible then begin
        FDataLink.Field.Value := FKeyValue;
      end;
}
  end;
end;

procedure TrDBLookupComboTree.SetDropDownWidth(AValue: Integer);
begin
  FDropDownWidth := AValue;
end;

procedure TrDBLookupComboTree.SetNoneLabel(AValue: String);
var
  B: Boolean;
begin
  if SNoneLabel <> Trim(AValue)
  then begin
    B := Text = SNoneLabel;
    SNoneLabel := Trim(AValue);
    if not (csDesigning in ComponentState)
    then// begin
      TrPopupTree(FPopup).NoneLabel := SNoneLabel;
//    end;
    if B then begin
      Text := SNoneLabel;
      FDisplayString := SNoneLabel;
    end;
  end;
end;

function TrDBLookupComboTree.GetDropDownCount: Integer;
begin
  if (csDesigning in ComponentState)
  then Result := FDTDropDownCount
  else Result := TrPopupTree(FPopup).FDropDownRows;
end;

procedure TrDBLookupComboTree.SetDropDownCount(AValue: Integer);
begin
  if (csDesigning in ComponentState)
  then begin
    if AValue <> FDTDropDownCount then begin
      FDTDropDownCount := AValue;
      if FDTDropDownCount < 2 then FDTDropDownCount := 2;
      if FDTDropDownCount > 50 then FDTDropDownCount := 50;
    end;
  end
  else TrPopupTree(FPopup).SetDropDownRows(AValue);
end;

function TrDBLookupComboTree.GetDataField: String;
begin
  Result := FDataLink.FieldName;
end;

procedure TrDBLookupComboTree.SetDataField(AValue: String);
begin
  FDataLink.FieldName := AValue;
  DataChange(Self);
end;

function TrDBLookupComboTree.GetDataSource: TDataSource;
begin
  Result := FDataLink.DataSource;
end;

procedure TrDBLookupComboTree.SetDataSource(AValue: TDataSource);
begin
{$IFDEF POLARIS_D4}
  if not (FDataLink.DataSourceFixed and (csLoading in ComponentState)) then
{$ENDIF}
    FDataLink.DataSource := AValue;
  if AValue <> nil then AValue.FreeNotification(Self);
end;

function TrDBLookupComboTree.GetLinkDataSource: TDataSource;
begin
  if (csDesigning in ComponentState)
  then Result := FDTDataSource
  else Result := TLocDBLookupTree(TrPopupTree(FPopup).FTreeList).GetLinkDataSource;
end;

procedure TrDBLookupComboTree.SetLinkDataSource(AValue: TDataSource);
begin
//
  if (csDesigning in ComponentState)
  then begin
    FDTDataSource := AValue;
    if AValue <> nil then AValue.FreeNotification(Self);
  end
  else begin
    TLocDBLookupTree(TrPopupTree(FPopup).FTreeList).SetLinkDataSource(AValue);
    DataChange(Self);
  end;
end;

function TrDBLookupComboTree.GetKeyField: String;
begin
  if (csDesigning in ComponentState)
  then Result := FDTLookupField
  else Result := TLocDBLookupTree(TrPopupTree(FPopup).FTreeList).KeyField;
  //LookupField;
end;

procedure TrDBLookupComboTree.SetKeyField(AValue: String);
begin
  if (csDesigning in ComponentState)
  then FDTLookupField := AValue
  else begin
    TLocDBLookupTree(TrPopupTree(FPopup).FTreeList).KeyField := AValue;
    DataChange(Self);
  end;
  //LookupField := AValue;
end;

function TrDBLookupComboTree.GetDisplayFields: String;
begin
  if (csDesigning in ComponentState)
  then Result := FDTLookupDisplay
  else Result := TLocDBLookupTree(TrPopupTree(FPopup).FTreeList).DisplayFields;
end;

procedure TrDBLookupComboTree.SetDisplayFields(AValue: String);
begin
  if (csDesigning in ComponentState)
  then FDTLookupDisplay := AValue
  else begin
    TLocDBLookupTree(TrPopupTree(FPopup).FTreeList).DisplayFields := AValue;
    DataChange(Self);
  end
end;

function TrDBLookupComboTree.GetChildrenField: String;
begin
  if (csDesigning in ComponentState)
  then Result := FDTChildrenField
  else Result := TLocDBLookupTree(TrPopupTree(FPopup).FTreeList).ChildrenField;
end;

procedure TrDBLookupComboTree.SetChildrenField(AValue: String);
begin
  if (csDesigning in ComponentState)
  then FDTChildrenField := AValue
  else begin
    TLocDBLookupTree(TrPopupTree(FPopup).FTreeList).ChildrenField := AValue;
    DataChange(Self);
  end
end;

function TrDBLookupComboTree.GetParentField: String;
begin
  if (csDesigning in ComponentState)
  then Result := FDTParentField
  else Result := TLocDBLookupTree(TrPopupTree(FPopup).FTreeList).ParentField;
end;

procedure TrDBLookupComboTree.SetParentField(AValue: String);
begin
  if (csDesigning in ComponentState)
  then FDTParentField := AValue
  else begin
    TLocDBLookupTree(TrPopupTree(FPopup).FTreeList).ParentField := AValue;
    DataChange(Self);
  end
end;

function TrDBLookupComboTree.GetReadOnly: Boolean;
begin
  Result := FDataLink.ReadOnly;
end;

procedure TrDBLookupComboTree.SetReadOnly(AValue: Boolean);
begin
  FDataLink.ReadOnly := AValue;
  inherited ReadOnly := FDataLink.ReadOnly;
end;

function TrDBLookupComboTree.GetChildIndex: Integer;
begin
  if (csDesigning in ComponentState)
  then Result := FDTChildIndex
  else Result := TrPopupTree(FPopup).FTreeList.ChildIndex;
end;

procedure TrDBLookupComboTree.SetChildIndex(AValue: Integer);
begin
  if (csDesigning in ComponentState)
  then FDTChildIndex := AValue
  else TrPopupTree(FPopup).FTreeList.ChildIndex := AValue;
end;

function TrDBLookupComboTree.GetChildSelected: Integer;
begin
  if (csDesigning in ComponentState)
  then Result := FDTChildSelected
  else Result := TrPopupTree(FPopup).FTreeList.ChildSelected;
end;

procedure TrDBLookupComboTree.SetChildSelected(AValue: Integer);
begin
  if (csDesigning in ComponentState)
  then FDTChildSelected := AValue
  else TrPopupTree(FPopup).FTreeList.ChildSelected := AValue;
end;

function TrDBLookupComboTree.GetParentIndex: Integer;
begin
  if (csDesigning in ComponentState)
  then Result := FDTParentIndex
  else Result := TrPopupTree(FPopup).FTreeList.ParentIndex;
end;

procedure TrDBLookupComboTree.SetParentIndex(AValue: Integer);
begin
  if (csDesigning in ComponentState)
  then FDTParentIndex := AValue
  else TrPopupTree(FPopup).FTreeList.ParentIndex := AValue;
end;

function TrDBLookupComboTree.GetParentSelected: Integer;
begin
  if (csDesigning in ComponentState)
  then Result := FDTParentSelected
  else Result := TrPopupTree(FPopup).FTreeList.ParentSelected;
end;

procedure TrDBLookupComboTree.SetParentSelected(AValue: Integer);
begin
  if (csDesigning in ComponentState)
  then FDTParentSelected := AValue
  else TrPopupTree(FPopup).FTreeList.ParentSelected := AValue;
end;

function TrDBLookupComboTree.CanModify: Boolean;
begin
  Result := FDataLink.CanModify;
end;

function TrDBLookupComboTree.GetImageList: TImageList;
begin
  if (csDesigning in ComponentState)
  then Result := FDTImages
  else
  {$IFDEF POLARIS_D4}
  Result := TImageList(TrPopupTree(FPopup).FTreeList.Images);
  {$ELSE}
  Result := TrPopupTree(FPopup).FTreeList.Images;
  {$ENDIF}
end;

procedure TrDBLookupComboTree.SetImageList(AValue: TImageList);
begin
  if (csDesigning in ComponentState)
  then FDTImages := AValue
  else TrPopupTree(FPopup).FTreeList.Images := AValue;
end;

function TrDBLookupComboTree.EditCanModify: Boolean;
begin
  Result := not ReadOnly;
//  Result := FDataLink.Edit;
end;

procedure TrDBLookupComboTree.KeyDown(var Key: Word; Shift: TShiftState);
begin
  inherited KeyDown(Key, Shift);
  Invalidate;
//  if TrPopupTree(FPopup).Visible then TrPopupTree(FPopup).KeyDown(Key, Shift);
end;

procedure TrDBLookupComboTree.KeyPress(var Key: Char);
begin
  if (csDesigning in ComponentState) then begin
    inherited KeyPress(Key);
    exit;
  end;
  if FPopupVisible then begin
    if Key in [#13, #27]
    then begin
      PopupCloseUp(Self, Key=#13);
      Key := #0;
    end
    else TrPopupTree(FPopup).KeyPress(Key);
  end
  else begin
    if Key in [#32..#255] then begin
      PopupDropDown(False);
      if FPopupVisible then TrPopupTree(FPopup).KeyPress(Key);
    end
    else if (Key = #27) {and FEscapeClear and (not ValueIsEmpty(FValue))}
      and ((FValue <> '') and (FValue<>SNoneLabel))
      and CanModify then
    begin
      ResetField;
      inherited SelectAll;
      if (FValue = '') or (FValue=SNoneLabel) then Key := #0;   {ALEX}
    end;
  end;
  inherited KeyPress(Key);
  if (Key in [#13, #27]) then
    GetParentForm(Self).Perform(CM_DIALOGKEY, Byte(Key), 0);

{
  inherited KeyPress(Key);
  if FPopupVisible
  then begin
    TrPopupTree(FPopup).KeyPress(Key);
  end;
}
  Invalidate;
end;

procedure TrDBLookupComboTree.DoChange;
begin
  inherited DoChange;
end;

procedure TrDBLookupComboTree.Change;
begin
  inherited Change;
  if not (csDesigning in ComponentState)
  then begin
//    TrPopupTree(FPopup).SetValue(FKeyValue);
    TrPopupTree(FPopup).SearcString := Text;
  end;
{
  if not FPopupVisible and not VarEquals(FKeyValue, FDataLink.Field.Value)
  then begin
    if CanModify and FDataLink.Edit then FDataLink.Field.Value := FKeyValue;
  end;
}
end;

procedure TrDBLookupComboTree.PopupChange;
begin
  inherited PopupChange;
end;

procedure TrDBLookupComboTree.PopupDropDown(DisableEdit: Boolean);
var
  P: TPoint;
  Y: Integer;
begin
//  Inherited PopupDropDown(False);
  if (csDesigning in ComponentState) then Exit;
  if (FPopup <> nil) and not (ReadOnly or FPopupVisible) then begin
    if FDropDownWidth <> 0
    then TrPopupTree(FPopup).Width := FDropDownWidth
    else TrPopupTree(FPopup).Width := Width;
    P := Parent.ClientToScreen(Point(Left, Top));
    Y := P.Y + Height;
    if Y + FPopup.Height > Screen.Height then
      Y := P.Y - FPopup.Height;
    case PopupAlign of
      epaRight:
        begin
          Dec(P.X, FPopup.Width - Width);
          if P.X < 0 then Inc(P.X, FPopup.Width - Width);
        end;
      epaLeft:
        begin
          if P.X + FPopup.Width > Screen.Width then
            Dec(P.X, FPopup.Width - Width);
        end;
    end;
    if P.X < 0 then P.X := 0
    else if P.X + FPopup.Width > Screen.Width then
      P.X := Screen.Width - FPopup.Width;
    if (Text <> '') and (Text <> SNoneLabel) then SetPopupValue(Text)
    else SetPopupValue(Null);
    if CanFocus then SetFocus;
    ShowPopup(Point(P.X, Y));
    FPopupVisible := True;
    if DisableEdit then begin
      inherited ReadOnly := True;
      HideCaret(Handle);
    end;
  end;
end;

function TrDBLookupComboTree.AcceptPopup(var Value: Variant): Boolean;
begin
  if Assigned(FDataLink) and Assigned(FDataLink.Field)
  then Result := not VarEquals(KeyValue, FDataLink.Field.Value)
  else Result := True;
end;

procedure TrDBLookupComboTree.SetPopupValue(const Value: Variant);
begin
  if not (csDesigning in ComponentState)
  then
//  TLocDBLookupTree(TrPopupTree(FPopup).FTreeList).SetKeyValue(KeyValue);
  TrPopupTree(FPopup).SetValue(KeyValue);
end;

function TrDBLookupComboTree.GetPopupValue: Variant;
begin
  if not (csDesigning in ComponentState)
  then Result := TrPopupTree(FPopup).DisplayValue;
end;

procedure TrDBLookupComboTree.AcceptValue(const Value: Variant);
var
  AValue: Variant;
begin
  AValue := Value;
  if Assigned(FDataLink) and Assigned(FDataLink.Field) then begin
    if not VarEquals(AValue, FDataLink.Field.Value) then begin
//    if FDataLink.Field.Value
//    Text := Value;
      Modified := True;
      if CanModify
      then begin
        if FDataLink.Edit then
          FDataLink.Field.Value := AValue;
        SelectAll;
      end;

//      UpdatePopupVisible;
//      DoChange;
    end;
//  end
//  else begin
//     TrPopupTree(FPopup).SetValue(AValue);
//     KeyValue := AValue;
//     TrPopupTree(FPopup).FTreeList.KeyValue := AValue;
//     FDisplayString := TrPopupTree(FPopup).DisplayValue;
//     Text := FDisplayString;
  end;
  UpdatePopupVisible;
  DoChange;
end;

procedure TrDBLookupComboTree.PopupCloseUp(Sender: TObject; Accept: Boolean);
var
  AValue: Variant;
begin
//  if Sender = FPopup then Exit;
  if (FPopup <> nil) and FPopupVisible then begin
    if GetCapture <> 0 then SendMessage(GetCapture, WM_CANCELMODE, 0, 0);
//    AValue := GetPopupValue;
    if not (csDesigning in ComponentState)
    then
    AValue := TrPopupTree(FPopup).FTreeList.FKeyValue;
    HidePopup;
    try
      try
        if CanFocus then begin
          SetFocus;
          if GetFocus = Handle then ComboCrack(Self).SetShowCaret;
        end;
      except
        { ignore exceptions }
      end;
      SetDirectInput(DirectInput);
      Invalidate;
      if Accept and AcceptPopup(AValue) and EditCanModify then begin
        AcceptValue(AValue);
//        if FFocused then inherited SelectAll;
      end;
      DataChange(Self);
      if FFocused then inherited SelectAll;
    finally
      FPopupVisible := False;
    end;
  end;
end;

procedure TrDBLookupComboTree.ButtonClick;
begin
  if Assigned(OnButtonClick) then OnButtonClick(Self);
  if FPopup <> nil then begin
    if FPopupVisible then PopupCloseUp(Self, False) else PopupDropDown(False);
  end;
end;

procedure TrDBLookupComboTree.MouseDown(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
begin
  if (FPopup <> nil) and (Button = mbLeft) then begin
    if CanFocus then SetFocus;
    if not FFocused then Exit;
    if FPopupVisible then PopupCloseUp(Self, False);
    {else if (not ReadOnly or AlwaysEnable) and (not DirectInput) then
      PopupDropDown(True);}
  end;
  inherited MouseDown(Button, Shift, X, Y);
end;

procedure TrDBLookupComboTree.ResetField;
begin
  if (FDataLink.DataSource <> nil) and (FDataLink.Field <> nil)
  and CanModify then begin
    if FDataLink.Edit then FDataLink.Field.Value := null;
  end;
end;

procedure TrDBLookupComboTree.DoEnter;
begin
  inherited DoEnter;
  inherited SelectAll;
end;

procedure TrDBLookupComboTree.DoExit;
begin
  inherited DoExit;
  inherited SelectAll;
end;

procedure TrDBLookupComboTree.CheckNotCircular;
begin
  if (LookupSource <> nil) and FDataLink.Active and ((DataSource = LookupSource)
     or (FDataLink.DataSet = LookupSource.DataSet))
  then _DBError(SCircularDataLink);
end;

procedure TrDBLookupComboTree.SetOnGetImageIndex(AValue: TTVExpandedEvent);
begin
  if (csDesigning in ComponentState)
  then FOnGetImageIndex := AValue
  else TrPopupTree(FPopup).FTreeList.OnGetImageIndex := AValue
end;

procedure TrDBLookupComboTree.SetOnGetSelectedIndex(AValue: TTVExpandedEvent);
begin
  if (csDesigning in ComponentState)
  then FOnGetSelectedIndex := AValue
  else TrPopupTree(FPopup).FTreeList.OnGetSelectedIndex := AValue
end;

end.
