//==============================================
//       rdbgrid.pas
//
//         Delphi.
//       TRxDBGrid.
//
//      Copyright 1998-2000 Polaris Software
//      http://members.xoom.com/PolarisSoft
//      mailto: PolarisLib@mail.ru
//==============================================

unit rDBGrid;

{$I POLARIS.INC}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  DB, rSetupFields, RxDBCtrl, DBGrids, Grids, rDBConst;

type
  TCellAttribute = procedure(Sender: TObject; Field: TField; Canvas: TCanvas; State: TGridDrawState) of object;

  TrDBGrid = class(TRxDBGrid)
  private
     FSetupField : TrSetupFields;
     FRegKey : String;
     //
     FNewDefaultRowHeight: Integer;
     FOnApplyCellAttribute: TCellAttribute;
     FGridAutoSize: Boolean;
     FGridAutoWidth: Boolean;
     FDisplayMemo: Boolean;
     FMultiLineTitles: Boolean;
     FDisplayImages: Boolean;
     FDisplayBool: Boolean;
     FRowSizingAllowed: Boolean;
     FCol : Integer;
     FRow :Integer;
     FDefaultDrawing : Boolean;
     //
     function GetDataSource: TDataSource;
     procedure SetDataSource(Value: TDataSource);
     //
     procedure UpdateTitlesHeight;
     procedure SetGridAutoSize(Value: Boolean);
     procedure SetGridAutoWidth(Value: Boolean);
     function  GetDefaultRowHeight: Integer;
     procedure SetDisplayMemo(Value: Boolean);
     procedure SetMultiLineTitles(Value: Boolean);
     procedure SetRowSizingAllowed(Value: Boolean);
     procedure SetDisplayImages(Value: Boolean);
     procedure SetDisplayBool(Value: Boolean);
     procedure TextRectDefault(Rect : TRect; Column : TColumn);
     procedure SetCheckBoxSize;
     procedure SetDefaultDrawing(Value: Boolean);
     //
  protected
    FOnColResize: TNotifyEvent;
    procedure Loaded; override;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
    procedure DrawColumnCell(const Rect: TRect; DataCol: Integer; Column: TColumn; State: TGridDrawState); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure DblClick; override;
    procedure DrawCell(ACol, ARow: Longint; ARect: TRect; AState: TGridDrawState); override;
    procedure ColWidthsChanged; override;
    procedure LayoutChanged; override;
    procedure SetDefaultRowHeight(Value: Integer);
    procedure RowHeightsChanged; override;
    procedure ApplyCellAttribute(Field: TField; Canvas: TCanvas; State: TGridDrawState);
    procedure CellAttribute(Field: TField; Canvas: TCanvas; State: TGridDrawState); dynamic;
    procedure WMSize(var Message: TWMSize); message WM_SIZE;
    procedure CMFontChanged(var Message: TMessage); message CM_FONTCHANGED;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property DataLink;
    property ColCount;
    procedure RunSetup;
    procedure SetRegKey(Value : String);
    property SetupFields: TrSetupFields read FSetupField ;
    property OnColumnResize: TNotifyEvent read FOnColResize write FOnColResize;
  published
    { Published declarations }
    property DataSource: TDataSource read GetDataSource write SetDataSource;
    property RegKey: String read FRegKey write SetRegKey;
    //
    property DefaultDrawing: Boolean read FDefaultDrawing write SetDefaultDrawing default True;
    property GridAutoSize: Boolean read FGridAutoSize write SetGridAutoSize default False;
    property GridAutoWidth: Boolean read FGridAutoWidth write SetGridAutoWidth default False;
    property DisplayMemo: Boolean read FDisplayMemo write SetDisplayMemo default False;
    property MultiLineTitles: Boolean read FMultiLineTitles write SetMultiLineTitles default False;
    property DefaultRowHeight: Integer read GetDefaultRowHeight write SetDefaultRowHeight;
    property RowSizingAllowed: Boolean read FRowSizingAllowed write SetRowSizingAllowed default True;
    property OnApplyCellAttribute: TCellAttribute read FOnApplyCellAttribute write FOnApplyCellAttribute;
    property DisplayImages: Boolean read FDisplayImages write SetDisplayImages default False;
    property DisplayBoolean: Boolean read FDisplayBool write SetDisplayBool default False;
    //
  end;

  TrDBGridMod = class(TCustomGrid)
    public
      property Options;
    end;

const
  MinRowHeight = 17;

implementation

uses rDBGridCtls;

var
  GCheckWidth, GCheckHeight: Integer;

{Const
  GridBmpNames: array[TGridPicture] of PChar =
    ('DBG_BLOB', 'DBG_MEMO', 'DBG_PICT', 'DBG_OLE', 'DBG_OBJECT', 'DBG_DATA',
     'DBG_NOTEMPTY', 'DBG_SMDOWN', 'DBG_SMUP');

  GridBitmaps: array[TGridPicture] of TBitmap =
    (nil, nil, nil, nil, nil, nil, nil, nil, nil);}

{function GetGridBitmap(BmpType: TGridPicture): TBitmap;
begin
  if GridBitmaps[BmpType] = nil then begin
    GridBitmaps[BmpType] := TBitmap.Create;
    GridBitmaps[BmpType].Handle := LoadBitmap(HInstance, GridBmpNames[BmpType]);
  end;
  Result := GridBitmaps[BmpType];
end;}

{ TrDBGrid }

constructor TrDBGrid.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
//  inherited DefaultDrawing := True;
  SetCheckBoxSize;
  FDefaultDrawing := True;
  GridAutoSize := False;
  GridAutoWidth := False;
  MultiLineTitles := False;
  FRowSizingAllowed := True;
  FDisplayMemo := False;
  FDisplayBool := False;
  FDisplayImages := False;
  if not (csDesigning in ComponentState) then
    FSetupField := TrSetupFields.Create(AOwner);
end;

destructor TrDBGrid.Destroy;
begin
  if Assigned(FSetupField) then FSetupField.Free;
  inherited Destroy;
end;

procedure TrDBGrid.Loaded;
begin
  inherited Loaded;
  RowSizingAllowed := FRowSizingAllowed;
end;

procedure TrDBGrid.Notification(AComponent: TComponent;Operation: TOperation);
begin
  inherited;
  if (Operation = opRemove) then
    if (AComponent = FSetupField) and (FSetupField <> nil) then
           FSetupField := nil;
end;

procedure TrDBGrid.RunSetup;
begin
  if FSetupField <> nil then SetupFields.Execute;
end;

procedure TrDBGrid.SetRegKey;
begin
  if (Value <> FRegKey) then begin
    FRegKey := Value;
    if not (csDesigning in ComponentState) then
      FSetupField.RegKey := Value;
  end;
end;

function TrDBGrid.GetDataSource;
begin
  Result := inherited DataSource;
end;

procedure TrDBGrid.SetDataSource (Value : TDataSource);
begin
  inherited DataSource := Value;
  if not (csDesigning in ComponentState) then begin
    if Value <> nil then begin
      FSetupField.DBGrid := Self;
      FSetupField.RegKey := FRegKey;
      FSetupField.SetupColumns;
    end else FSetupField.DBGrid := nil;
  end;
end;

//
procedure TrDBGrid.SetGridAutoSize(Value: Boolean);
var
  RowNo: Integer;
  ColNo: Integer;
  WidthNew: Integer;
  WidthMax: Integer;
begin
  FGridAutoSize := Value;
  if FGridAutoSize = True then FGridAutoWidth := False;
  if not (csDesigning in ComponentState) then begin
    if (DataLink <> nil) and (DataLink.Active) then begin
      DataLink.DataSet.DisableControls;
      try
        if FGridAutoSize = True then begin
          for ColNo := 0 to (ColCount - 2) do begin
            WidthMax := 0;
            WidthNew := 0;
            DataLink.DataSet.First;
            for RowNo := 0 to (RowCount - 1) do
              if Fields[ColNo].Value <> Null then
                WidthNew := Canvas.TextWidth(Fields[ColNo].DisplayText);
            if WidthNew > WidthMax then WidthMax := WidthNew;
            DataLink.DataSet.Next;
            try
              Columns[ColNo].Width := (WidthMax + (GridLineWidth + 5))
            except
            end;
            DataLink.DataSet.First;
          end;
          Invalidate;
        end
        else Columns.RestoreDefaults;
      finally
        DataLink.DataSet.EnableControls;
      end;
    end;
  end;
end;

procedure TrDBGrid.SetGridAutoWidth(Value: Boolean);
var
  Available, Remaining, i, w: Integer;
begin
  FGridAutoWidth := Value;
  if FGridAutoWidth = True then FGridAutoSize := False;
  if not (csDesigning in ComponentState) then begin
    if (DataLink <> nil) and (DataLink.Active) then begin
      DataLink.DataSet.DisableControls;
      try
        if FGridAutoWidth = True then begin
          with TrDBGrid(Self) do begin
            Available := ClientWidth;
            for i := 0 to (FixedCols) do
              Dec(Available, ColWidths[i] + GridlineWidth);
            Remaining := ColCount - (FixedCols+1);
            w := Available div Remaining;
            for i := (FixedCols + 1) to ColCount - 2 do begin
              ColWidths[i] := w - GridLineWidth;
              Dec(Available, w);
            end;
            ColWidths[ColCount-1] := Available;
          end;
          Invalidate;
        end
        else SetGridAutoSize(FGridAutoSize);
      finally
        DataLink.DataSet.EnableControls;
      end;
    end;
  end;
end;

procedure TrDBGrid.SetDisplayMemo(Value: Boolean);
begin
  if (FDisplayMemo <> Value) and not DefaultDrawing then begin
    FDisplayMemo := Value;
    Invalidate;
  end;
end;

procedure TrDBGrid.SetMultiLineTitles(Value: Boolean);
begin
  if FMultiLineTitles <> Value then begin
    FMultiLineTitles := Value;
    Invalidate;
  end;
  UpdateTitlesHeight;
end;

procedure TrDBGrid.UpdateTitlesHeight;
var
  i, j, MaxHeight: Integer;
  RRect: TRect;
begin
  if FMultiLineTitles then begin
    if (dgTitles in Options) then begin
      MaxHeight := 0;
      for i := 0 to Columns.Count - 1 do begin
        RRect := CellRect(0, 0);
        RRect.Right := Columns[i].Width - 1;
        RRect.Left := 0;
        Canvas.Font := Columns[i].Title.Font;
        j := DrawText(Canvas.Handle,
                      PChar(Columns[i].Title.Caption),
                      Length(Columns[i].Title.Caption),
                      RRect,
                      DT_EXPANDTABS or DT_CALCRECT or DT_WORDBREAK);
        if j > MaxHeight then MaxHeight := j;
      end;
      if (MaxHeight <> 0) then begin
        if (dgRowLines in Options) then Inc(MaxHeight, 3)
        else Inc(MaxHeight, 2);
        RowHeights[0] := MaxHeight;
        if RowHeights[0] < MinRowHeight then RowHeights[0] := MinRowHeight;
      end;
    end;
  end
  else if RowHeights[0] <> MinRowHeight then RowHeights[0] := MinRowHeight;
end;

procedure TrDBGrid.DrawCell(ACol, ARow: Longint; ARect: TRect; AState: TGridDrawState);
const
  AlignFlags: array[TAlignment] of Integer =
      (DT_LEFT or DT_WORDBREAK or DT_EXPANDTABS or DT_NOPREFIX,
      DT_RIGHT or DT_WORDBREAK or DT_EXPANDTABS or DT_NOPREFIX,
      DT_CENTER or DT_WORDBREAK or DT_EXPANDTABS or DT_NOPREFIX);
var
  TitleText: string;
  TmpCol: LongInt;
  TmpRect: TRect;
  AField : TField;
  BackColor : TColor;
  SortMarker : TSortMarker;
begin
  inherited DrawCell(ACol, ARow, ARect, AState);

  if (dgIndicator in Options) then TmpCol := ACol - 1
  else TmpCol := ACol;

  with ARect do begin
    if (FMultiLineTitles) and (ARow = 0) and (TmpCol >= 0)
       and (dgTitles in Options) and (Assigned(Columns[TmpCol])) then begin
      TitleText := Columns[TmpCol].Title.Caption;
      if TitleText <> '' then begin
        Canvas.Font := Columns[TmpCol].Title.Font;
        Canvas.Brush.Color := Columns[TmpCol].Title.Color;
        TmpRect := ARect;
        Inc(TmpRect.Top, 2);
        Inc(TmpRect.Left, 3);
        Dec(TmpRect.Right, 1);
        Dec(TmpRect.Bottom, 1);
        Canvas.FillRect(TmpRect);
        DrawText(Canvas.Handle, PChar(TitleText), Length(TitleText), TmpRect, AlignFlags[Columns[TmpCol].Title.Alignment]);

        SortMarker := smNone;
        AField := Columns[TmpCol].Field;
        if (AField <> nil) and (Assigned(OnGetBtnParams)) then begin
          BackColor := Canvas.Brush.Color;
          OnGetBtnParams(Self, AField, Canvas.Font, BackColor, SortMarker, False);
          Canvas.Brush.Color := BackColor;
        end;

        if (AField <> nil) then with Canvas, ARect do
        {case SortMarker of
          smDown: Bmp := GetGridBitmap(gpMarkDown);
          smUp: Bmp := GetGridBitmap(gpMarkUp);
          else Bmp := nil;
        end;}
          case SortMarker of
          smDown: begin
              Pen.Color := clBtnShadow;
              MoveTo(Right - 13, Bottom - 12);
              LineTo(Right - 6,  Bottom - 12);
              Pen.Color := clBtnHighlight;
              LineTo(Right - 9,  Bottom - 5);
              Pen.Color := clBtnShadow;
              LineTo(Right - 13, Bottom - 12);
              Pixels[Right - 5,  Bottom - 12] := clBtnShadow;
            end;
          smUp: begin
              Pen.Color := clBtnHighlight;
              MoveTo(Right - 13, Bottom - 5);
              LineTo(Right - 5,  Bottom - 6);
              LineTo(Right - 9,  Bottom - 12);
              Pen.Color := clBtnShadow;
              LineTo(Right - 13, Bottom - 5);
              Pixels[Right - 13, Bottom - 5] := clBtnShadow;
            end;
          end;
      end;
    end;
  end;
end;

procedure TrDBGrid.ColWidthsChanged;
begin
  inherited;
  try
    if Assigned(FOnColResize) then FOnColResize(TrDBGrid(Self));
  except
  end;
end;

procedure TrDBGrid.LayoutChanged;
begin
  inherited LayoutChanged;
  SetDefaultRowHeight(FNewDefaultRowHeight);
  if FMultiLineTitles then UpdateTitlesHeight;
end;

function TrDBGrid.GetDefaultRowHeight: Integer;
begin
  Result := inherited DefaultRowHeight;
end;

procedure TrDBGrid.SetDefaultRowHeight(Value: Integer);
begin
  if Value <= 0 then Value := inherited DefaultRowHeight;
  inherited DefaultRowHeight := Value;
  FNewDefaultRowHeight := Value;
  if dgTitles in Options then begin
    Canvas.Font := TitleFont;
    RowHeights[0] := Canvas.TextHeight('Wg') + 4;
  end;
end;

procedure TrDBGrid.RowHeightsChanged;
var
  i, ThisHasChanged, Def: Integer;
begin
  ThisHasChanged := -1;
  Def := DefaultRowHeight;
  for i := Ord(dgTitles in Options) to RowCount do
    if RowHeights[i] <> Def then begin
      ThisHasChanged := i;
      Break;
    end;
  if ThisHasChanged <> -1 then begin
    SetDefaultRowHeight(RowHeights[ThisHasChanged]);
    RecreateWnd;
  end;
  inherited;
end;

procedure TrDBGrid.CMFontChanged(var Message: TMessage);
var
  h: integer;
begin
  inherited;
  Canvas.Font.Assign(Font);
  h := Canvas.TextHeight('Wg');
  if MinRowHeight < h then SetDefaultRowHeight(h);
  if (csDesigning in ComponentState) then Invalidate;
end;

procedure TrDBGrid.SetRowSizingAllowed(Value: Boolean);
begin
  FRowSizingAllowed := Value;
  with TrDBGridMod(Self) do begin
    Options := Options + [goColSizing, goThumbTracking, goTabs, goColMoving, goRowMoving, goDrawFocusSelected];
    if Value then
      Options := Options + [goRowSizing]
    else
      Options := Options - [goRowSizing];
  end;
end;

procedure TrDBGrid.DrawColumnCell(const Rect: TRect; DataCol: Integer; Column: TColumn; State: TGridDrawState);
var
  Bmp: TBitmap;
  Style: UINT;
  TmpRect: TRect;
  FactorX, FactorY: Double;
  TmpWidth, TmpHeight: Integer;
  CWidth, CHeight, BWidth, BHeight: Integer;
  vAlignment: TAlignment;
  i: Integer;
begin
  BeginUpdate;
  i := 0;
  with Canvas do begin
    Lock;
    TmpRect := Rect;
    FillRect(Rect);

    if (Column <> nil) and (Column.Field <> nil) then begin
      if not (gdSelected in State) and not (gdFixed in State) then
          ApplyCellAttribute(Column.Field, Canvas, State);

      if (Column.Field.DataType = ftGraphic) or
         (Column.Field.DataType = ftTypedBinary) then begin
        if (FDisplayImages) then begin
          Bmp := nil;
          try
            Bmp := TBitmap.Create;
            Bmp.Assign(Column.Field);
            begin
              CWidth := Rect.Right - Rect.Left + 1 - 1;
              CHeight := Rect.Bottom - Rect.Top + 1 - 1;
              BWidth := Bmp.Width;
              BHeight := Bmp.Height;
              try
                if BWidth = 0 then FactorX := 0
                else FactorX := CWidth / BWidth;
                if BHeight = 0 then FactorY := 0
                else FactorY := CHeight / BHeight;
              except
                FactorX := 1;
                FactorY := 1;
              end;
              if FactorY < FactorX then begin
                TmpWidth := Round(BWidth * FactorY);
                TmpHeight := CHeight;
              end
              else begin
                TmpHeight := Round(BHeight * FactorX);
                TmpWidth := CWidth;
              end;
              SetRect(TmpRect, 0, 0, TmpWidth, TmpHeight);
              OffSetRect(TmpRect, Rect.Left + ((CWidth - TmpWidth) div 2), Rect.Top + ((CHeight - TmpHeight) div 2));
              StretchDraw(TmpRect, Bmp);
            end
          finally
            Bmp.Free;
          end;
        end
        else TextRectDefault(Rect, Column);
      end

      else if (Column.Field is TMemoField) then begin
        if (FDisplayMemo) then begin
          TmpRect := Rect;
          Inc(TmpRect.Top, 2);
          Inc(TmpRect.Left, 2);
          DrawText(Canvas.Handle, PChar(Column.Field.AsString), Length(Column.Field.AsString), TmpRect, DT_WORDBREAK or DT_NOPREFIX);
        end
        else TextRectDefault(Rect, Column);
      end

      else if (Column.Field is TBooleanField) then begin
        if (FDisplayBool) then begin
          TmpRect := Rect;
          TmpRect.Left := Rect.Left + (Rect.Right - Rect.Left - GCheckWidth) div 2;
          TmpRect.Right := TmpRect.Left + GCheckWidth;
          TmpRect.Top := Rect.Top + (Rect.Bottom - Rect.Top - GCheckWidth) div 2;
          TmpRect.Bottom := TmpRect.Top + GCheckHeight;
          Style := DFCS_CHECKED;
          if Column.Field.AsBoolean = True then style := DFCS_CHECKED
          else if Column.Field.AsBoolean = False then style := DFCS_BUTTONCHECK;
          DrawFrameControl(Canvas.Handle, TmpRect, DFC_BUTTON, Style);
        end
        else begin
          vAlignment := Column.Field.Alignment;
          case vAlignment of
            taRightJustify: begin
                 SetTextAlign(Canvas.Handle, TA_RIGHT);
                 I := Rect.Right - 2;
               end;
            taLeftJustify : begin
                 SetTextAlign(Canvas.Handle, TA_LEFT);
                 I := Rect.Left + 2;
               end;
            taCenter      : begin
                 SetTextAlign(Canvas.Handle, TA_CENTER);
                 I := (Rect.Right + Rect.Left) div 2;
               end;
          end;
          TextRect(Rect, I, Rect.Top + 2, Column.Field.DisplayText);
          SetTextAlign(Canvas.Handle, TA_LEFT);
        end;
      end

      else begin
        vAlignment := Column.Field.Alignment;
        case vAlignment of
          taRightJustify : begin
              SetTextAlign(Canvas.Handle, TA_RIGHT);
              i := Rect.Right - 3;
            end;
          taLeftJustify  : begin
              SetTextAlign(Canvas.Handle, TA_LEFT);
              i := Rect.Left + 2;
            end;
          taCenter       : begin
             SetTextAlign(Canvas.Handle, TA_CENTER);
             i := (Rect.Right + Rect.Left) div 2;
           end;
        end;
        TextRect(Rect, I, Rect.Top + 2, Column.Field.DisplayText);
        SetTextAlign(Canvas.Handle, TA_LEFT);
      end;
    end;
    Unlock;
  end;
  inherited DrawColumnCell(Rect, DataCol, Column, State);
  EndUpdate;
end;

procedure TrDBGrid.ApplyCellAttribute(Field: TField; Canvas: TCanvas; State: TGridDrawState);
begin
  if Assigned(FOnApplyCellAttribute) then
      FOnApplyCellAttribute(Self, Field, Canvas, State);
end;

procedure TrDBGrid.CellAttribute(Field: TField; Canvas: TCanvas; State: TGridDrawState);
begin
  ApplyCellAttribute(Field, Canvas, State);
end;

procedure TrDBGrid.SetDisplayImages(Value: Boolean);
begin
  if (FDisplayImages <> Value) and not DefaultDrawing then begin
    FDisplayImages := Value;
    Invalidate;
  end;
end;

procedure TrDBGrid.WMSize(var Message: TWMSize);
begin
  try
    inherited;
    if (DataLink <> nil) and (Datalink.Active) then begin
      if FGridAutoSize then SetGridAutoSize(FGridAutoSize);
      if FGridAutoWidth then SetGridAutoWidth(FGridAutoWidth);
      Invalidate;
    end;
  except
  end;
end;

procedure TrDBGrid.MouseUp(Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  inherited MouseUp(Button, Shift, X, Y);
  TrDBGrid(Self).MouseToCell(X, Y, FCol, FRow);
  if (dgIndicator in Options) then FCol := FCol - Byte(FCol <> 0);
end;

procedure TrDBGrid.DblClick;
var
  AField : TField;
begin
  if Assigned(OnDblClick) then inherited DblClick
  else begin
    if (FCol >= 0) and Assigned(Columns[FCol].Field) then AField := Columns[FCol].Field
    else AField := nil;
    if not (csDesigning in ComponentState) then begin
      BeginUpdate;
      if ((Datalink <> nil) and (Datalink.Active) and (AField <> nil)) then begin
        if (FDisplayBool) and (AField.DataType = ftBoolean)
           and (CanEditModify = True) then begin
          if not (ReadOnly) and not (DataLink.Readonly) then begin
            DataSource.Dataset.Edit;
            try
              AField.AsBoolean := not AField.AsBoolean;
            except
              raise;
            end;
          end;
        end;
        if (AField.DataType = ftGraphic) or (AField.DataType = ftTypedBinary) then begin
          Application.CreateForm(TrDBGridCtlsForm, rDBGridCtlsForm);
          rDBGridCtlsForm.BlobCaption := srGridImage;
          rDBGridCtlsForm.BlobField := AField;
          rDBGridCtlsForm.DataSource := DataSource;
          if not (ReadOnly) and not (DataLink.Readonly) then begin
            DataSource.DataSet.Edit;
            try
              rDBGridCtlsForm.ShowModal;
            except
              raise;
            end;
          end else rDBGridCtlsForm.ShowModal;
          rDBGridCtlsForm.Free;
        end;
        if (AField.DataType = ftMemo) or (AField.DataType = ftFmtMemo) then begin
          Application.CreateForm(TrDBGridCtlsForm, rDBGridCtlsForm);
          rDBGridCtlsForm.BlobCaption := srGridMemo;
          rDBGridCtlsForm.BlobField := AField;
          rDBGridCtlsForm.DataSource := DataSource;
          if not (ReadOnly) and not (DataLink.Readonly) then begin
            DataSource.DataSet.Edit;
            try
              rDBGridCtlsForm.ShowModal;
            except
              raise;
            end
          end else rDBGridCtlsForm.ShowModal;
          rDBGridCtlsForm.Free;
        end;
      end;
      EndUpdate;
    end;
  end;
end;

procedure TrDBGrid.TextRectDefault(Rect : TRect; Column : TColumn);
var
  i : Integer;
begin
  SetTextAlign(Canvas.Handle, TA_CENTER);
  i := (Rect.Right + Rect.Left) div 2;
  Canvas.TextRect(Rect, i, Rect.Top + 2, Column.Field.DisplayText);
  SetTextAlign(Canvas.Handle, TA_LEFT);
end;

procedure TrDBGrid.SetDisplayBool(value: Boolean);
begin
  if (FDisplayBool <> Value) and not DefaultDrawing then begin
    FDisplayBool := Value;
    Invalidate;
  end;
end;

procedure TrDBGrid.SetCheckBoxSize;
begin
  with TBitmap.Create do
    try
      Handle := LoadBitmap(0, PChar(32759));
      GCheckWidth := Width div 4;
      GCheckHeight := Height div 3;
    finally
      Free;
    end;
end;

procedure TrDBGrid.SetDefaultDrawing;
begin
  if FDefaultDrawing <> Value then begin
    inherited DefaultDrawing := Value;
    FDefaultDrawing := Value;
    if FDefaultDrawing then begin
      FDisplayBool   := False;
      FDisplayMemo   := False;
      FDisplayImages := False;
    end;
    Invalidate;
  end;
end;

end.
