//==============================================
//       rdbconnect.pas
//
//         Delphi.
//         .
//
//      Copyright 1998-2000 Polaris Software
//      http://members.xoom.com/PolarisSoft
//      mailto: PolarisLib@mail.ru
//==============================================
unit rDBConnect;

{$I POLARIS.INC}

interface

uses Windows, Classes, Graphics, Forms, Controls, StdCtrls,
     Buttons, ExtCtrls, SysUtils, DB, DBTables, BDE, ComCtrls, Registry,
     rConst, rDBConst, rButtons;

type
  TrDBConnect = class(TComponent)
  private
    FDatabase: TDatabase;
    FRegKey: string;
    FDrivers: string;
    FAlias: string;
    FAliasEnabled: Boolean;
    FUser: string;
    FUserEnabled: Boolean;
    FPassword: string;
    FAutoLogon: Boolean;
    FMaxAttempts: Integer;
    procedure SetDatabase(Value: TDatabase);
    procedure SetRegKey(const Value: string);
  protected
    CountAttempts: Integer;
    procedure Notification(AComponent: TComponent;
      Operation: TOperation); override;
    procedure DoConnect(AAlias,AUser,APassword: string); virtual;
  public
    constructor Create(AOwner: TComponent); override;
    function Connect: Boolean; virtual;
  published
    property Database: TDatabase read FDatabase write SetDatabase;
    property RegKey: string read FRegKey write SetRegKey;
    property Drivers: string read FDrivers write FDrivers;
    property Alias: string read FAlias write FAlias;
    property AliasEnabled: Boolean read FAliasEnabled write FAliasEnabled default True;
    property User: string read FUser write FUser;
    property UserEnabled: Boolean read FUserEnabled write FUserEnabled default True;
    property Password: string read FPassword write FPassword;
    property AutoLogon: Boolean read FAutoLogon write FAutoLogon default False;
    property MaxAttempts: Integer read FMaxAttempts write FMaxAttempts default 3;
  end;

  TrPasswordDlg = class(TForm)
    OKBtn: TrBitBtn;
    CancelBtn: TBitBtn;
    Bevel1: TBevel;
    Label2: TLabel;
    Image1: TImage;
    CBAlias: TComboBox;
    Bevel2: TBevel;
    EditUser: TEdit;
    Label3: TLabel;
    EditPassword: TEdit;
    OpenPanel: TPanel;
    Animate1: TAnimate;
    BitBtn1: TBitBtn;
    procedure EditUserExit(Sender: TObject);
    procedure CBAliasChange(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure EditPasswordChange(Sender: TObject);
    procedure OKBtnClick(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormActivate(Sender: TObject);
    function FormHelp(Command: Word; Data: Integer;
      var CallHelp: Boolean): Boolean;
  private
    paramList: TStringList;
  public
  end;

var
  PasswordDlg: TrPasswordDlg;

implementation

uses StrUtils, rUtils;

{$R *.DFM}

{ TrDBConnect }

constructor TrDBConnect.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FMaxAttempts := 3;
  FAliasEnabled := True;
  FUserEnabled := True;
  FAutoLogon := False;
end;

procedure TrDBConnect.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (AComponent = FDatabase) then
    Database := nil;
end;

procedure TrDBConnect.SetDatabase(Value: TDatabase);
begin
  if FDatabase <> Value then begin
    FDatabase := Value;
    if Value <> nil then Value.FreeNotification(Self);
  end;
end;

procedure TrDBConnect.SetRegKey;
begin
  if FRegKey <> Value then begin
    FRegKey := Value;
    with TRegistry.Create do begin
      if KeyExists(FRegKey) then begin
        OpenKey(FRegKey,False);
        if ValueExists(srDatabaseValue) then
          Alias := ReadString(srDatabaseValue)
        else Alias := '';
        if ValueExists(srUserValue) then
          User := ReadString(srUserValue)
        else User := '';
        CloseKey;
      end;
      Free;
    end;
  end;
end;

procedure TrDBConnect.DoConnect(AAlias,AUser,APassword: string);
var
  TH: HDBIDB;
  Options: FLDDesc;
begin
  FillChar(Options, sizeof(Options), #0);
  Options.iOffset := 0;
  Options.iLen := Length(AUser) + 1;
  StrPCopy(Options.szName, szUSERNAME);
  try
    if FDatabase = nil then SysUtils.Abort;
    Check(DbiOpenDataBase(PChar(StrToOem(AAlias)), nil, dbiReadWrite, dbiOpenShared,
                          PChar(StrToOem(APassword)), 1, @Options,
                          PChar(StrToOem(AUser)), TH));
    Check(DbiCloseDatabase(TH));
    with FDatabase do begin
      Close;
      AliasName := AAlias;
      LoginPrompt := False;
      Params.Values[szUSERNAME] := AUser;
      Params.Values[szPASSWORD] := APassword;
      Open;
    end;
    FAlias := AAlias;
    FUser := AUser;
    FPassword := APassword;
    if FRegKey <> '' then
      with TRegistry.Create do begin
        OpenKey(FRegKey,True);
        WriteString(srDatabaseValue,AAlias);
        WriteString(srUserValue,AUser);
        CloseKey;
        Free;
      end;
  except
    Dec(CountAttempts);
    raise;
  end;
end;

function TrDBConnect.Connect: Boolean;
begin
  CountAttempts := FMaxAttempts;
  if FAutoLogon then
    try
      DoConnect(FAlias,FUser,FPassword);
      Result := True;
      exit;
    finally
    end;
  with TrPasswordDlg.Create(Self) do begin
    CBAlias.Enabled := FAliasEnabled;
    with CBAlias do begin
      ItemIndex := Items.IndexOf(FAlias);
      OnChange(CBAlias);
    end;
    EditUser.Enabled := FUserEnabled;
    if FUser <> '' then EditUser.Text := FUser;
    Result := (ShowModal = mrOK);
    Free;
  end;
end;

{ TrPasswordDlg }

procedure TrPasswordDlg.EditUserExit(Sender: TObject);
begin
  EditUser.Text := Trim(EditUser.Text);
end;

procedure TrPasswordDlg.CBAliasChange(Sender: TObject);
begin
  if CBAlias.ItemIndex <> -1 then begin
    if EditUser.Text = '' then begin
      Session.GetAliasParams(CBAlias.Items[CBAlias.ItemIndex],paramList);
      EditUser.Text := AnsiUpperCase(paramList.Values[szUSERNAME]);
      EditPassword.Text := '';
    end;
    EditPasswordChange(Sender);
    if (EditUser.Text <> '') or (not EditUser.Enabled) then
      ActiveControl := EditPassword
    else
      ActiveControl := EditUser;
  end;
end;

procedure TrPasswordDlg.FormCreate(Sender: TObject);
var
  i: Integer;
  s: string;
begin
  HelpFile := srHelpFile;
  LoadBitmapFromResource(Image1.Picture.Bitmap,hInstance,srConnectRes,rkGIF);
  paramList := TStringList.Create;
  Session.GetAliasNames(CBAlias.Items);
  if (Owner is TrDBConnect) and (TrDBConnect(Owner).FDrivers <> '') then begin
    s := ';'+AnsiUpperCase(TrDBConnect(Owner).FDrivers)+';';
    for i:=CBAlias.Items.Count-1 downto 0 do
      if AnsiPos(';'+AnsiUpperCase(Session.GetAliasDriverName(CBAlias.Items[i]))+';',s) = 0 then
        CBAlias.Items.Delete(i);
  end;
  CBAlias.ItemIndex := -1;
  if CBAlias.Items.Count = 0 then
    rMsgBox(PChar(ConvertCodes(srConnectNoDB)), MB_OK+MB_ICONHAND)
  else
    CBAliasChange(Self);
end;

procedure TrPasswordDlg.EditPasswordChange(Sender: TObject);
begin
  OKBtn.Enabled := (CBAlias.Text <> '') and
       ((Session.GetAliasDriverName(CBAlias.Items[CBAlias.ItemIndex])=szCFGDBSTANDARD) or
       ((EditUser.Text <> '') and (EditPassword.Text <> '')));
end;

procedure TrPasswordDlg.OKBtnClick(Sender: TObject);
begin
  if Owner is TrDBConnect then begin
    OpenPanel.Visible := True;
    Animate1.Active := True;
    try
      TrDBConnect(Owner).DoConnect(CBAlias.Text,EditUser.Text,EditPassword.Text);
      ModalResult := mrOK;
    except
      if TrDBConnect(Owner).CountAttempts = 0 then
        ModalResult := mrCancel
      else begin
        OpenPanel.Visible := False;
        Animate1.Active := False;
        if EditUser.Text = '' then EditUser.SetFocus
        else begin
          EditPassword.SelectAll;
          EditPassword.SetFocus;
        end;
      end;
      raise;
    end;
  end;
end;

procedure TrPasswordDlg.FormDestroy(Sender: TObject);
begin
  paramList.Free;
end;

procedure TrPasswordDlg.FormActivate(Sender: TObject);
begin
  if CBAlias.Text = '' then
    ActiveControl := CBAlias
  else if (EditUser.Text <> '') or (not EditUser.Enabled) then
    ActiveControl := EditPassword
  else
    ActiveControl := EditUser;
end;

function TrPasswordDlg.FormHelp(Command: Word; Data: Integer;
  var CallHelp: Boolean): Boolean;
begin
  CallHelp := not ((Command = HELP_CONTEXTPOPUP) and (Data = HelpContext));
  Result := True;
end;

end.

