//==============================================
//       rBkgnd.pas
//
//         Delphi.
//        .
//
//      Copyright 1998-2000 Polaris Software
//      http://members.xoom.com/PolarisSoft
//      mailto: PolarisLib@mail.ru
//==============================================
unit rBkgnd;

interface

uses Windows, SysUtils, Classes, Graphics, Forms, Controls, StdCtrls,
  Buttons, ExtCtrls, Dialogs, FileCtrl, ColorGrd, ComCtrls,
  RXCombos, Spin, rButtons, Registry, Messages, StrUtils, Consts;

type

//rHookWindow
  PClass = ^TClass;
  THookMessageEvent = procedure (Sender: TObject; var Msg: TMessage;
    var Handled: Boolean) of object;

  TrWindowHook = class(TComponent)
  private
    FActive: Boolean;
    FControl: TWinControl;
    FControlHook: TObject;
    FBeforeMessage: THookMessageEvent;
    FAfterMessage: THookMessageEvent;
    FClientBeforeMessage: THookMessageEvent;
    FClientAfterMessage: THookMessageEvent;
    function GetWinControl: TWinControl;
    function GetHookHandle: HWnd;
    procedure SetActive(Value: Boolean);
    procedure SetWinControl(Value: TWinControl);
    function IsForm: Boolean;
    function NotIsForm: Boolean;
    function DoUnhookControl: Pointer;
    procedure ReadForm(Reader: TReader);
    procedure WriteForm(Writer: TWriter);
  protected
    procedure DefineProperties(Filer: TFiler); override;
    procedure DoAfterMessage(var Msg: TMessage; var Handled: Boolean); dynamic;
    procedure DoBeforeMessage(var Msg: TMessage; var Handled: Boolean); dynamic;
    procedure DoClientAfterMessage(var Msg: TMessage; var Handled: Boolean); dynamic;
    procedure DoClientBeforeMessage(var Msg: TMessage; var Handled: Boolean); dynamic;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure HookControl;
    procedure UnhookControl;
    property HookWindow: HWnd read GetHookHandle;
  published
    property Active: Boolean read FActive write SetActive default True;
    property WinControl: TWinControl read GetWinControl write SetWinControl
      stored NotIsForm;
    property BeforeMessage: THookMessageEvent read FBeforeMessage write FBeforeMessage;
    property AfterMessage: THookMessageEvent read FAfterMessage write FAfterMessage;
    property ClientBeforeMessage: THookMessageEvent read FClientBeforeMessage write FClientBeforeMessage;
    property ClientAfterMessage: THookMessageEvent read FClientAfterMessage write FClientAfterMessage;
  end;

// RBackground
  TrViewMode = (rvmNone, rvmTile, rvmCenter, rvmStretch);
  TrViewWindowMode = (rvwmNone, rvwmMaximized, rvwmCentered);
  PrBackgroundStruct = ^TrBackgroundStruct;
  TrBackgroundStruct = record
    FileName: TFileName;
    Color: TColor;
    ViewMode: TrViewMode;
    XPos, YPos,
    Width, Height: Integer;
    State: TrViewWindowMode;
  end;
  TrPosSizeEnabled = (rpsPosEnabled, rpsSizeEnabled);
  TrPosSizeSet = Set of TrPosSizeEnabled;
{$IFNDEF VER100}
  TCustomForm = TForm;
{$ENDIF}
  THackForm = class(TCustomForm);

  TrBackground = class(TComponent)
  private
    { Private declarations }
    FBindControl: TCustomForm;
    FPrevRect: TRect;
    FPosSizeEnabled : TrPosSizeSet;
    FSaveSplitterPos,
    FSaveToolBar,
    FRedrawBkg,
    FAutoSave,
    FUpdate: Boolean;
    FLastFileName,
    FProgramKey: String;
    FCurrentBkgnd: TrBackgroundStruct;
    FPicture: TBitmap;
    FReg: TRegistry;
    FFileHistory: TStringList;
    FMaxValue: Array[0..3] of Integer;
    FHook: TrWindowHook;
    FCloseEvent: TCloseEvent;
    FOnHideEvent,
    FCreateEvent: TNotifyEvent;
    procedure SetProgramKey(Value: String);
    procedure SetBindControl(Value: TCustomForm);
    function  GetBindControl: TCustomform;
    procedure SetCurrentBkgnd(Value: TrBackgroundStruct);
    procedure SetBkgndColor(Value: TColor);
    procedure SetBkgndFileName(Value: TFileName);
    procedure SetBkgndViewMode(Value: TrViewMode);
    procedure SetRedrawBkg(Value: Boolean);
  protected
    { Protected declarations }
    procedure BeforeHook(Sender: TObject; var Msg: TMessage; var Handled: Boolean);
    procedure ClientBeforeHook(Sender: TObject; var Msg: TMessage; var Handled: Boolean);
    procedure ClientAfterHook(Sender: TObject; var Msg: TMessage; var Handled: Boolean);

    function GetCurrentRect: TRect;
    procedure Loaded; override;
    procedure DrawBackGround(DC: hDC);
    procedure MakeMax(Sender: TObject);
    procedure ChangeSizePos;
    procedure Notification(AComponent: TComponent;
      Operation: TOperation); override;

    procedure LoadOtherControl;

    function LoadBackground(var Value: TrBackgroundStruct): Boolean;
    function TestBackground(Value1, Value2: TrBackgroundStruct): Boolean;
    property BindControl: TCustomForm read GetBindControl write SetBindControl;
    function IsNotDefaultRegKey: Boolean;
    function IsNotDefaultFileName: Boolean;
    function CurrentForm: THackForm;
    procedure RefreshForm;
    procedure DestroyMessage(Sender: TObject);

    procedure OnFormClose(Sender: TObject; var Action: TCloseAction);
    procedure OnFormCreate(Sender: TObject);
    procedure OnFormHide(Sender: TObject);
    procedure InternalSave(Sender: TObject);
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function Execute: Boolean;
    procedure SaveBackground;
    property CurrentBkgnd: TrBackgroundStruct read FCurrentBkgnd write SetCurrentBkgnd;
  published
    { Published declarations }
    property RegKey: String read FProgramKey write SetProgramKey stored IsNotDefaultRegKey;
    property Color: TColor read FCurrentBkgnd.Color write SetBkgndColor default clBtnFace;
    property FileName: TFileName read FCurrentBkgnd.FileName write SetBkgndFileName stored IsNotDefaultFileName;
    property ViewMode: TrViewMode read FCurrentBkgnd.ViewMode write SetBkgndViewMode default rvmNone;
    property PosSizeEnabled: TrPosSizeSet read FPosSizeEnabled write FPosSizeEnabled default [rpsPosEnabled, rpsSizeEnabled];
    property AutoSave: Boolean read FAutoSave write FAutoSave default False;
    property SaveSplitterPos: Boolean read FSaveSplitterPos write FSaveSplitterPos default True;
    property SaveToolBar: Boolean read FSaveToolBar write FSaveToolBar default True;
    property RedrawBkg: Boolean read FRedrawBkg write SetRedrawBkg default True;
  end;

  TrBackgroundDlg = class(TForm)
    TestPanel: TPanel;
    GroupBox1: TGroupBox;
    GroupBox2: TGroupBox;
    FileComboBox: TComboBox;
    ColorPanel: TImage;
    TestImage: TImage;
    ColorBtn: TButton;
    FileBtn: TButton;
    ComboBox2: TComboBox;
    Label1: TLabel;
    ColorComboBox1: TColorComboBox;
    PosBox: TGroupBox;
    RLeft: TSpinEdit;
    Label2: TLabel;
    Label3: TLabel;
    RTop: TSpinEdit;
    OKBtn: TrBitBtn;
    CancelBtn: TrBitBtn;
    RestoreBtn: TrBitBtn;
    Bevel1: TBevel;
    SizeBox: TGroupBox;
    Label4: TLabel;
    Label5: TLabel;
    RHeight: TSpinEdit;
    RWidth: TSpinEdit;
    Panel1: TPanel;
    rBitBtn1: TrBitBtn;
    RPosition: TComboBox;
    Label6: TLabel;
    procedure ColorBtnClick(Sender: TObject);
    procedure FileComboBoxChange(Sender: TObject);
    procedure FileBtnClick(Sender: TObject);
    procedure FormPaint(Sender: TObject);
    procedure ComboBox1Change(Sender: TObject);
    procedure RestoreBtnClick(Sender: TObject);
    procedure ComboBox2Change(Sender: TObject);
    function FormHelp(Command: Word; Data: Integer;
      var CallHelp: Boolean): Boolean;
    procedure RHeightChange(Sender: TObject);
    procedure RPositionChange(Sender: TObject);
  private
    { Private declarations }
    FDefBkgnd,
    FBkgnd: TrBackgroundStruct;
    FullName: String;
    procedure SetDefBkgnd(Value: TrBackgroundStruct);
    procedure SetBkgnd(Value: TrBackgroundStruct);
    procedure SetBkgColor(Value: TColor);
    procedure SetBkgFileName(Value: TFileName);
    procedure SetBkgView(Value: TrViewMode);
    function TestBkgValue(Value1, Value2: TrBackgroundStruct): Boolean;
    function GetEnableSize: Boolean;
    procedure SetEnableSize(Value: Boolean);
    function GetPosEnable: Boolean;
    procedure SetPosEnable(Value: Boolean);
    function GetSizeEnable: Boolean;
    procedure SetSizeEnable(Value: Boolean);
    function GetPosSize(Index: Integer): Integer;
    procedure SetPosSize(Index: Integer; Value: Integer);
    function GetFormPosition: TrViewWindowMode;
    procedure SetFormPosition(Value : TrViewWindowMode);
  protected
    procedure DoChangeSizePos;
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    procedure SetmaxValue(Index: Integer; Value: Integer);
    property DefBkgnd: TrBackgroundStruct read FDefBkgnd write SetDefBkgnd;
    property EnableSize: Boolean read GetEnableSize write SetEnableSize default True;
    property BackgroundColor: TColor read FBkgnd.Color write SetBkgColor;
    property BackgroundBitmap: TFileName read FBkgnd.FileName write SetBkgFileName;
    property BackgoundView: TrViewMode read FBkgnd.ViewMode write SetBkgView;
    property ChangePos: Boolean read GetPosEnable write SetPosEnable;
    property ChangeSize: Boolean read GetSizeEnable write SetSizeEnable;
    property XPos: Integer index 0 read GetPosSize write SetPosSize;
    property YPos: Integer index 1 read GetPosSize write SetPosSize;
    property FormWidth: Integer index 2 read GetPosSize write SetPosSize;
    property FormHeight: Integer index 3 read GetPosSize write SetPosSize;
    property FormState: TrViewWindowMode read GetFormPosition write SetFormPosition;
  end;

const
  CM_RECREATEWINDOW  = CM_BASE + 102;
  CM_DESTROYHOOK     = CM_BASE + 103;

var
  rBackgroundDlg: TrBackgroundDlg;

implementation

uses extdlgs, rConst, Procs, rUtils, ErrorMes;


type
  THack = class(TWinControl);
  THookOrder = (hoBeforeMsg, hoAfterMsg);

{ TControlHook }

  TControlHook = class(TObject)
  private
    FControl: TWinControl;
    FNewWndProc: Pointer;
    FPrevWndProc: Pointer;
    FNewClientWndProc: Pointer;
    FPrevClientWndProc: Pointer;
    FList: TList;
    FDestroying: Boolean;
    procedure SetWinControl(Value: TWinControl);
    procedure HookWndProc(var AMsg: TMessage);
    procedure HookClientWndProc(var AMsg: TMessage);
    procedure NotifyHooks(Order: THookOrder; var Msg: TMessage;
      var Handled: Boolean);
    procedure NotifyClientHooks(Order: THookOrder; var Msg: TMessage;
      var Handled: Boolean);
  public
    constructor Create;
    destructor Destroy; override;
    procedure HookControl;
    procedure UnhookControl;
    procedure AddHook(AHook: TrWindowHook);
    procedure RemoveHook(AHook: TrWindowHook);
    property WinControl: TWinControl read FControl write SetWinControl;
  end;

{ THookList }

  THookList = class(TList)
  private
    FHandle: HWnd;
    procedure WndProc(var Msg: TMessage);
  public
    constructor Create;
    destructor Destroy; override;
    function FindControlHook(AControl: TWinControl): TControlHook;
    function GetControlHook(AControl: TWinControl): TControlHook;
    property Handle: HWnd read FHandle;
  end;

var
  HookList: THookList;

//====================================
function GetHookList: THookList;
begin
  if HookList = nil then HookList := THookList.Create;
  Result := HookList;
end;

procedure DropHookList; far;
begin
  HookList.Free;
  HookList := nil;
end;

{ TControlHook }

constructor TControlHook.Create;
begin
  inherited Create;
  FList := TList.Create;
  FNewWndProc := MakeObjectInstance(HookWndProc);
  FPrevWndProc := nil;
  FNewClientWndProc := MakeObjectInstance(HookClientWndProc);
  FPrevClientWndProc := nil;
  FControl := nil;
end;

destructor TControlHook.Destroy;
begin
  FDestroying := True;
  if Assigned(HookList) then
    if HookList.IndexOf(Self) >= 0 then HookList.Remove(Self);
  while FList.Count > 0 do RemoveHook(TrWindowHook(FList.Last));
  FControl := nil;
  FList.Free;
  FreeObjectInstance(FNewWndProc);
  FNewWndProc := nil;
  FreeObjectInstance(FNewClientWndProc);
  FNewClientWndProc := nil;
  inherited Destroy;
end;

procedure TControlHook.AddHook(AHook: TrWindowHook);
begin
  if FList.IndexOf(AHook) < 0 then begin
    FList.Add(AHook);
    AHook.FControlHook := Self;
    WinControl := AHook.FControl;
  end;
  HookControl;
end;

procedure TControlHook.RemoveHook(AHook: TrWindowHook);
begin
  AHook.FControlHook := nil;
  FList.Remove(AHook);
  if FList.Count = 0 then UnhookControl;
end;

procedure TControlHook.NotifyHooks(Order: THookOrder; var Msg: TMessage;
  var Handled: Boolean);
var
  I: Integer;
begin
  if (FList.Count > 0) and Assigned(FControl) and
    not FDestroying then begin
    for I := FList.Count - 1 downto 0 do begin
      if not (csDestroying in FControl.ComponentState)
      then try
        if Order = hoBeforeMsg then
          TrWindowHook(FList[I]).DoBeforeMessage(Msg, Handled)
        else if Order = hoAfterMsg then
          TrWindowHook(FList[I]).DoAfterMessage(Msg, Handled);
      except
        Application.HandleException(Self);
      end;
      if Handled then Break;
    end;
  end;
end;

procedure TControlHook.NotifyClientHooks(Order: THookOrder; var Msg: TMessage;
  var Handled: Boolean);
var
  I: Integer;
begin
  if (FList.Count > 0) and Assigned(FControl) and
    not (FDestroying or (csDestroying in FControl.ComponentState)) then
    for I := FList.Count - 1 downto 0 do begin
      try
        if Order = hoBeforeMsg then
          TrWindowHook(FList[I]).DoClientBeforeMessage(Msg, Handled)
        else if Order = hoAfterMsg then
          TrWindowHook(FList[I]).DoClientAfterMessage(Msg, Handled);
      except
        Application.HandleException(Self);
      end;
      if Handled then Break;
    end;
end;

procedure TControlHook.HookControl;
var
  P: Pointer;
  Frm: THackForm;
begin
  if Assigned(FControl) and not ((csDesigning in FControl.ComponentState) or
    (csDestroying in FControl.ComponentState) or FDestroying) then
  begin
    FControl.HandleNeeded;
    P := Pointer(GetWindowLong(FControl.Handle, GWL_WNDPROC));
    if (P <> FNewWndProc) then begin
      FPrevWndProc := P;
      SetWindowLong(FControl.Handle, GWL_WNDPROC, LongInt(FNewWndProc));
    end;
    if (FControl is TCustomForm) and (THackForm(FControl).FormStyle = fsMDIForm) then begin
      Frm := THackForm(FControl);
      P := Pointer(GetWindowLong(Frm.ClientHandle, GWL_WNDPROC));
      if (P <> FNewClientWndProc) then begin
        FPrevClientWndProc := P;
        SetWindowLong(Frm.ClientHandle, GWL_WNDPROC, LongInt(FNewClientWndProc));
      end;
    end;
  end;
end;

procedure TControlHook.UnhookControl;
var
  Frm: THackForm;
begin
  if Assigned(FControl) then begin
    if Assigned(FPrevWndProc) and FControl.HandleAllocated
    then begin
      if (Pointer(GetWindowLong(FControl.Handle, GWL_WNDPROC)) = FNewWndProc)
      then SetWindowLong(FControl.Handle, GWL_WNDPROC, LongInt(FPrevWndProc));
      if Assigned(FPrevClientWndProc) and (FControl is TCustomForm) and (THackForm(FControl).FormStyle = fsMDIForm)
      then begin
        Frm := THackForm(FControl);
        if (Pointer(GetWindowLong(Frm.ClientHandle, GWL_WNDPROC)) = FNewClientWndProc)
        then SetWindowLong(Frm.ClientHandle, GWL_WNDPROC, LongInt(FPrevClientWndProc));
      end;
    end;
  end;
  FPrevWndProc := nil;
  FPrevClientWndProc := nil;
end;

procedure TControlHook.HookWndProc(var AMsg: TMessage);
var
  Handled: Boolean;
begin
  Handled := False;
  if Assigned(FControl) then begin
    if (AMsg.Msg <> WM_QUIT) then NotifyHooks(hoBeforeMsg, AMsg, Handled);
    with AMsg do begin
      if (not Handled) or (Msg = WM_DESTROY) then
        try
          if AMsg.Result = 0 then
            if Assigned(FPrevWndProc) then
              Result := CallWindowProc(FPrevWndProc, FControl.Handle, Msg,
                WParam, LParam)
            else
              Result := CallWindowProc(THack(FControl).DefWndProc,
                FControl.Handle, Msg, WParam, LParam);
        finally
          NotifyHooks(hoAfterMsg, AMsg, Handled);
        end;
      if Msg = WM_DESTROY then begin
        UnhookControl;
        if Assigned(HookList) and not (FDestroying or
          (csDestroying in FControl.ComponentState)) then
          PostMessage(HookList.FHandle, CM_RECREATEWINDOW, 0, Longint(Self));
      end;
    end;
  end;
end;

procedure TControlHook.HookClientWndProc(var AMsg: TMessage);
var
  Handled: Boolean;
begin
  Handled := False;
  if Assigned(FControl) then begin
    if (AMsg.Msg <> WM_QUIT) then NotifyClientHooks(hoBeforeMsg, AMsg, Handled);
    with AMsg do begin
      if (not Handled) or (Msg = WM_DESTROY) then
        try
          if AMsg.Result = 0 then
            if Assigned(FPrevClientWndProc) then
              Result := CallWindowProc(FPrevClientWndProc, FControl.Handle, Msg,
                WParam, LParam)
            else
              Result := CallWindowProc(THack(FControl).DefWndProc,
                FControl.Handle, Msg, WParam, LParam);
        finally
          NotifyClientHooks(hoAfterMsg, AMsg, Handled);
        end;
    end;
  end;
end;

procedure TControlHook.SetWinControl(Value: TWinControl);
begin
  if Value <> FControl then begin
    UnhookControl;
    FControl := Value;
    if FList.Count > 0 then HookControl;
  end;
end;

{ THookList }

constructor THookList.Create;
begin
  inherited Create;
  FHandle := AllocateHWnd(WndProc);
end;

destructor THookList.Destroy;
begin
  while Count > 0 do TControlHook(Last).Free;
  DeallocateHWnd(FHandle);
  inherited Destroy;
end;

procedure THookList.WndProc(var Msg: TMessage);
var
  Hook: TControlHook;
begin
  try
    with Msg do begin
      if Msg = CM_RECREATEWINDOW then begin
        Hook := TControlHook(LParam);
        if (Hook <> nil) and (IndexOf(Hook) >= 0) then
          Hook.HookControl;
      end
      else if Msg = CM_DESTROYHOOK then begin
        Hook := TControlHook(LParam);
        if Assigned(Hook) and (IndexOf(Hook) >= 0) and
          (Hook.FList.Count = 0) then Hook.Free;
      end
      else Result := DefWindowProc(FHandle, Msg, wParam, lParam);
    end;
  except
    Application.HandleException(Self);
  end;
end;

function THookList.FindControlHook(AControl: TWinControl): TControlHook;
var
  I: Integer;
begin
  if Assigned(AControl) then
    for I := 0 to Count - 1 do
      if (TControlHook(Items[I]).WinControl = AControl) then begin
        Result := TControlHook(Items[I]);
        Exit;
      end;
  Result := nil;
end;

function THookList.GetControlHook(AControl: TWinControl): TControlHook;
begin
  Result := FindControlHook(AControl);
  if Result = nil then begin
    Result := TControlHook.Create;
    try
      Add(Result);
      Result.WinControl := AControl;
    except
      Result.Free;
      raise;
    end;
  end;
end;

{ TrWindowHook }

constructor TrWindowHook.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FActive := True;
end;

destructor TrWindowHook.Destroy;
begin
  Active := False;
  WinControl := nil;
  inherited Destroy;
end;

procedure TrWindowHook.SetActive(Value: Boolean);
begin
  if FActive <> Value then
    if Value then HookControl else UnhookControl;
end;

function TrWindowHook.GetHookHandle: HWnd;
begin
  if Assigned(HookList) then Result := HookList.Handle
  else
{$IFDEF WIN32}
    Result := INVALID_HANDLE_VALUE;
{$ELSE}
    Result := 0;
{$ENDIF}
end;

procedure TrWindowHook.HookControl;
begin
  if Assigned(FControl) and not (csDestroying in ComponentState) then
    GetHookList.GetControlHook(FControl).AddHook(Self);
  FActive := True;
end;

function TrWindowHook.DoUnhookControl: Pointer;
begin
  Result := FControlHook;
  if Result <> nil then TControlHook(Result).RemoveHook(Self);
  FActive := False;
end;

procedure TrWindowHook.UnhookControl;
begin
  DoUnhookControl;
  FActive := False;
end;

function TrWindowHook.NotIsForm: Boolean;
begin
  Result := (WinControl <> nil) and not (WinControl is TCustomForm);
end;

function TrWindowHook.IsForm: Boolean;
begin
  Result := (WinControl <> nil) and ((WinControl = Owner) and
    (Owner is TCustomForm));
end;

procedure TrWindowHook.ReadForm(Reader: TReader);
begin
  if Reader.ReadBoolean then
    if Owner is TCustomForm then WinControl := TWinControl(Owner);
end;

procedure TrWindowHook.WriteForm(Writer: TWriter);
begin
  Writer.WriteBoolean(IsForm);
end;

procedure TrWindowHook.DefineProperties(Filer: TFiler);
{$IFDEF WIN32}
  function DoWrite: Boolean;
  begin
    if Assigned(Filer.Ancestor) then
      Result := IsForm <> TrWindowHook(Filer.Ancestor).IsForm
    else Result := IsForm;
  end;
{$ENDIF}
begin
  inherited DefineProperties(Filer);
  Filer.DefineProperty('IsForm', ReadForm, WriteForm,
    {$IFDEF WIN32} DoWrite {$ELSE} IsForm {$ENDIF});
end;

function TrWindowHook.GetWinControl: TWinControl;
begin
  if Assigned(FControlHook) then Result := TControlHook(FControlHook).WinControl
  else Result := FControl;
end;

procedure TrWindowHook.DoAfterMessage(var Msg: TMessage; var Handled: Boolean);
begin
  if Assigned(FAfterMessage) then FAfterMessage(Self, Msg, Handled);
end;

procedure TrWindowHook.DoBeforeMessage(var Msg: TMessage; var Handled: Boolean);
begin
  if Assigned(FBeforeMessage) then FBeforeMessage(Self, Msg, Handled);
end;

procedure TrWindowHook.DoClientAfterMessage(var Msg: TMessage; var Handled: Boolean);
begin
  if Assigned(FClientAfterMessage) then FClientAfterMessage(Self, Msg, Handled);
end;

procedure TrWindowHook.DoClientBeforeMessage(var Msg: TMessage; var Handled: Boolean);
begin
  if Assigned(FClientBeforeMessage) then FClientBeforeMessage(Self, Msg, Handled);
end;

procedure TrWindowHook.Notification(AComponent: TComponent; Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (AComponent = WinControl) and (Operation = opRemove) then
    WinControl := nil
  else if (Operation = opRemove) and ((Owner = AComponent) or
    (Owner = nil)) then WinControl := nil;
end;

procedure TrWindowHook.SetWinControl(Value: TWinControl);
var
  SaveActive: Boolean;
  Hook: TControlHook;
begin
  if Value <> WinControl then begin
    SaveActive := FActive;
    Hook := TControlHook(DoUnhookControl);
    FControl := Value;
{$IFDEF WIN32}
    if Value <> nil then Value.FreeNotification(Self);
{$ENDIF}
    if Assigned(Hook) and (Hook.FList.Count = 0) and Assigned(HookList) then
      PostMessage(HookList.Handle, CM_DESTROYHOOK, 0, Longint(Hook));
    if SaveActive then HookControl;
  end;
end;
//====================================

{$R *.DFM}

constructor TrBackgroundDlg.Create(AOwner: TComponent);
var
  SearchRec: TSearchRec;
  Result: Integer;
  fExt, S: String;
  I, N: Integer;
begin
  inherited Create(AOwner);
  S := ExtractWord(2,GraphicFilter(TGraphic),['|']);
  HelpFile := srHelpFile;
  SizeBox.Visible := True;
  FullName := ExtractFileDir(ExpandFileName(Application.ExeName))+'\';
  with FileComboBox do begin
    Items.Clear;
    Items.AddObject(srNone, nil);
    ItemIndex := 0;
    if Trim(S) = '' then S := '*.bmp';
    N := WordCount(S,[';']);
    for I := 1 to N do begin
      fExt := ExtractWord(I,S,[';']);
      Result := FindFirst(ExtractFilePath(Application.ExeName)+fExt, faArchive, SearchRec);
      while Result = 0 do begin
        Items.AddObject(ExtractFileName(SearchRec.Name),
                        TObject(StrNew(PChar(FullName+ExtractFileName(SearchRec.Name)))));
        Result := FindNext(SearchRec);
      end;
      FindClose(SearchRec);
    end;
  end;
  ComboBox2.ItemIndex := 0;
  FBkgnd.Color    := 0;
  SetBkgColor(clBtnFace);
  FBkgnd.FileName := '';
  FBkgnd.ViewMode := rvmNone;
  FDefBkgnd.Color    := clBtnFace;
  FDefBkgnd.FileName := '';
  FDefBkgnd.ViewMode := rvmNone;
  FDefBkgnd.State    := rvwmNone;
end;

procedure TrBackgroundDlg.SetDefBkgnd(Value: TrBackgroundStruct);
begin
  if not TestBkgValue(FDefBkgnd, Value) then begin
    if TestBkgValue(FDefBkgnd, FBkgnd) then SetBkgnd(FDefBkgnd);
    FDefBkgnd := Value;
  end
end;

procedure TrBackgroundDlg.SetBkgnd(Value: TrBackgroundStruct);
begin
  if not TestBkgValue(FBkgnd, Value) then begin
    SetBkgColor(Value.Color);
    SetBkgFileName(Value.FileName);
    SetBkgView(Value.ViewMode);
    SetPosSize(0, Value.XPos);
    SetPosSize(1, Value.YPos);
    SetPosSize(2, Value.Width);
    SetPosSize(3, Value.Height);
    SetFormPosition( Value.State);
  end;
end;

function TrBackgroundDlg.GetFormPosition: TrViewWindowMode;
begin
  Result := TrViewWindowMode(RPosition.ItemIndex);
end;

procedure TrBackgroundDlg.SetFormPosition(Value : TrViewWindowMode);
begin
  if RPosition.ItemIndex <> ord(Value) then RPosition.ItemIndex := ord(Value);
end;

procedure TrBackgroundDlg.RPositionChange(Sender: TObject);
begin
  SetFormPosition(TrViewWindowMode(RPosition.ItemIndex));
end;

function TrBackgroundDlg.TestBkgValue(Value1, Value2: TrBackgroundStruct): Boolean;
begin
  Result := (Value1.FileName = Value2.FileName) and
            (Value1.Color = Value2.Color) and
            (Value1.ViewMode = Value2.ViewMode) and
            (Value1.XPos = Value2.XPos) and
            (Value1.YPos = Value2.YPos) and
            (Value1.Width = Value2.Width) and
            (Value1.Height = Value2.Height);
end;

procedure TrBackgroundDlg.SetBkgView(Value: TrViewMode);
begin
  if (FBkgnd.FileName > '') and (Value in [rvmTile..rvmStretch]) then FBkgnd.ViewMode := Value
  else FBkgnd.ViewMode := rvmNone;
  if FBkgnd.ViewMode in [rvmTile..rvmStretch]
  then ComboBox2.ItemIndex := ord(FBkgnd.ViewMode)-1
  else if FBkgnd.FileName <> '' then SetBkgFileName('');
  ComboBox2.Enabled := FBkgnd.FileName <> '';
  Label1.Enabled := ComboBox2.Enabled;
  TestImage.Visible := (FBkgnd.ViewMode in [rvmCenter, rvmStretch]) or
                       ((FBkgnd.ViewMode = rvmTile) and (
                       (TestImage.Picture.Bitmap.Height > TestImage.Height) or
                       (TestImage.Picture.Bitmap.Width > TestImage.Width)));
  TestImage.Stretch := (FBkgnd.ViewMode = rvmStretch) or
                       ((FBkgnd.ViewMode in [rvmTile..rvmStretch]) and (
                       (TestImage.Picture.Bitmap.Height > TestImage.Height) or
                       (TestImage.Picture.Bitmap.Width > TestImage.Width)));
  Repaint;
  FormPaint(Self);
end;

procedure TrBackgroundDlg.SetBkgFileName(Value: TFileName);
var
  I  : Integer;
  C,S: String;
  P: TPicture;
begin
  S := ExpandFileName(Value);
  if Length(S) > 0 then begin
    FBkgnd.FileName := S;
    C := ExtractFileName(Value);
    I := FileComboBox.Items.IndexOf(C);
    if I < 1 then begin
      I := FileComboBox.Items.IndexOf(S);
      if I < 1 then I := FileComboBox.Items.AddObject(S,TObject(StrNew(PChar(S))));
    end;
    P := TPicture.Create;
    try
      P.LoadFromFile(S);
      if TestImage.Picture.Bitmap <> nil
      then TestImage.Picture.Bitmap := TBitmap.Create;
      TestImage.Picture.Bitmap.Width  := P.Width;
      TestImage.Picture.Bitmap.Height := P.Height;
      TestImage.Picture.Bitmap.Canvas.Brush.Color := FBkgnd.Color;
      TestImage.Picture.Bitmap.Canvas.FillRect(TestImage.Picture.Bitmap.Canvas.ClipRect);
      TestImage.Picture.Bitmap.Canvas.Draw(0,0,P.Graphic);
    finally
      P.Free;
    end;
  end
  else begin
    I := 0;
    FBkgnd.FileName := '';
    TestImage.Picture := nil;
  end;
  ComboBox2.Enabled := FBkgnd.FileName <> '';
  Label1.Enabled := ComboBox2.Enabled;
  FileComboBox.ItemIndex := I;
  SetBkgView(TrViewMode(ComboBox2.ItemIndex+1));
end;

procedure TrBackgroundDlg.SetBkgColor(Value: TColor);

begin
  FBkgnd.Color := Value;
  if ColorComboBox1.ColorValue <> FBkgnd.Color
  then ColorComboBox1.ColorValue := FBkgnd.Color;
  FileComboBoxChange(Self);
  Repaint;
end;


procedure TrBackgroundDlg.ColorBtnClick(Sender: TObject);
var
  ColorDialog1: TColorDialog;
begin
  ColorDialog1 := TColorDialog.Create(Self);
  try
    ColorDialog1.Color := FBkgnd.Color;
    if ColorDialog1.Execute then begin
      SetBkgColor(ColorDialog1.Color);
      Repaint;
    end;
  finally
    ColorDialog1.Free;
  end;
end;

procedure TrBackgroundDlg.FileComboBoxChange(Sender: TObject);
begin
  with FileComboBox do
   if Items.Objects[ItemIndex] <> nil then
     SetBkgFileName(StrPas(PChar(Items.Objects[ItemIndex])))
   else SetBkgFileName('');
end;

procedure TrBackgroundDlg.FileBtnClick(Sender: TObject);
var
  OpenDialog1: TOpenPictureDialog;
begin
  OpenDialog1 := TOpenPictureDialog.Create(Self);
  try
    OpenDialog1.Title := srBkgndPic;
    OpenDialog1.InitialDir := ExtractFilePath(FBkgnd.FileName);
    OpenDialog1.FileName   := FBkgnd.FileName;
    if OpenDialog1.Execute then
      SetBkgFileName(OpenDialog1.FileName);
  finally
    OpenDialog1.Free;
  end;
end;

procedure TrBackgroundDlg.FormPaint(Sender: TObject);
var
  R, C: Integer;
begin
  with ColorPanel.Canvas do begin
    Brush.Color := FBkgnd.Color;
    Brush.Style := bsSolid;
    FillRect(Rect(0,0, Width, Height));
    if not TestImage.Visible then
    with FileComboBox do
     if (FBkgnd.FileName <> '') and (FBkgnd.ViewMode=rvmTile) then
       for R := 0 to TestPanel.Height div TestImage.Picture.Height do
         for C := 0 to TestPanel.Width div TestImage.Picture.Width do
       ColorPanel.Canvas.Draw( C*TestImage.Picture.Width, R*TestImage.Picture.Height, TestImage.Picture.Graphic);
  end;
end;

procedure TrBackgroundDlg.ComboBox1Change(Sender: TObject);
begin
  SetBkgColor(ColorComboBox1.ColorValue);
end;

procedure TrBackgroundDlg.RestoreBtnClick(Sender: TObject);
begin
  SetBkgnd(FDefBkgnd);
end;

procedure TrBackgroundDlg.ComboBox2Change(Sender: TObject);
begin
  if FBkgnd.ViewMode <> TrViewMode(ComboBox2.ItemIndex+1)
  then SetBkgView(TrViewMode(ComboBox2.ItemIndex+1));
end;

function TrBackgroundDlg.GetEnableSize: Boolean;
begin
  Result := SizeBox.Visible
end;

procedure TrBackgroundDlg.SetEnableSize(Value: Boolean);
begin
  SizeBox.Visible := Value;
end;

function TrBackgroundDlg.GetPosEnable: Boolean;
begin
  Result := PosBox.Visible;
end;

procedure TrBackgroundDlg.SetPosEnable(Value: Boolean);
begin
  if PosBox.Visible <> Value then begin
    PosBox.Visible := Value;
    DoChangeSizePos;
  end;
end;

function TrBackgroundDlg.GetSizeEnable: Boolean;
begin
  Result := SizeBox.Visible;
end;

procedure TrBackgroundDlg.SetSizeEnable(Value: Boolean);
begin
  if SizeBox.Visible <> Value then begin
    SizeBox.Visible := Value;
    DoChangeSizePos;
  end;
end;

procedure TrBackgroundDlg.DoChangeSizePos;
begin
  if PosBox.Visible or SizeBox.Visible
  then ClientHeight := PosBox.Top+PosBox.Height+8+Bevel1.Height+Panel1.Height
  else ClientHeight := PosBox.Top+Bevel1.Height+Panel1.Height
end;

function TrBackgroundDlg.GetPosSize(Index: Integer): Integer;
begin
  Result := 0;
  case Index of
  0: Result := RLeft.Value;
  1: Result := RTop.Value;
  2: Result := RWidth.Value;
  3: Result := RHeight.Value;
  end;
end;

procedure TrBackgroundDlg.SetPosSize(Index: Integer; Value: Integer);
begin
  case Index of
  0: RLeft.Value  := Value;
  1: RTop.Value   := Value;
  2: RWidth.Value := Value;
  3: RHeight.Value:= Value;
  end;
end;

procedure TrBackgroundDlg.SetmaxValue(Index: Integer; Value: Integer);
begin
  if Value > 0 then
  case Index of
  0: begin
       RLeft.MaxValue  := Value;
       RLeft.MaxLength := Length(IntToStr(Value));
     end;
  1: begin
       RTop.MaxValue   := Value;
       RTop.MaxLength := Length(IntToStr(Value));
     end;
  2: begin
       RWidth.MaxValue := Value;
       RWidth.MaxLength := Length(IntToStr(Value));
     end;
  3: begin
       RHeight.MaxValue:= Value;
       RHeight.MaxLength := Length(IntToStr(Value));
     end;
  end;
end;

function TrBackgroundDlg.FormHelp(Command: Word; Data: Integer;
  var CallHelp: Boolean): Boolean;
begin
  CallHelp := not ((Command = HELP_CONTEXTPOPUP) and (Data = HelpContext));
  Result := True;
end;

procedure TrBackgroundDlg.RHeightChange(Sender: TObject);
var
  X1, X0, M: Integer;
begin
  if TSpinEdit(Sender).Text = '' then Exit;
  X1 := TSpinEdit(Sender).Value;
  M  := TSpinEdit(Sender).MaxValue;
  if Sender = RHeight then X0 := YPos
  else if Sender = RWidth then X0 := XPos
  else if Sender = RLeft then X0 := FormWidth
  else if Sender = RTop then X0 := FormHeight
  else X0 := 0;
  if (X1+X0) > M
  then begin
    X0 := M-X0;
    if (X0 < 0) or (Sender = RLeft) or (Sender = RTop) then X0 := 0;
    TSpinEdit(Sender).Value := X0;
    TSpinEdit(Sender).SelectAll;
    raise EAnyError.CreateFmt(srBkgndHeit,[X1]);
  end;
  case RPosition.ItemIndex of
  1: if ((RWidth.Value - RLeft.Value) <> RLeft.MaxValue) and
        ((RHeight.Value - RTop.Value) <> RTop.MaxValue)
     then SetFormPosition(rvwmNone);
  2: begin
        X0 := RLeft.MaxValue - (RLeft.MaxValue shr 1);
        X1 := RTop.MaxValue - (RTop.MaxValue shr 1);
        if (RLeft.Value <> X0) and (RTop.Value <> X1) then  SetFormPosition(rvwmNone);
     end;
  end;
end;

const
  DefaultBackground: TrBackgroundStruct = (
    FileName: '';
    Color: clBtnFace;
    ViewMode: rvmNone;
    XPos: 0;
    YPos: 0;
    Width: 0;
    Height: 0);

{TrBackground}
type
  TCrackFrom = class(TCustomForm);

constructor TrBackground.Create(AOwner: TComponent);
begin
  FUpdate := False;
  inherited Create(AOwner);
  FHook := TrWindowHook.Create(Self);
  FHook.BeforeMessage := BeforeHook;
  FHook.ClientBeforeMessage := ClientBeforeHook;
  FHook.ClientAfterMessage := ClientAfterHook;
  FCloseEvent  := nil;
  FCreateEvent := nil;
  FOnHideEvent := nil;
  FPrevRect := Rect(0,0,0,0);
  FRedrawBkg := True;
  FAutoSave := False;
  FSaveSplitterPos := True;
  FSaveToolBar := True;
  FPosSizeEnabled := [rpsPosEnabled, rpsSizeEnabled];
  FillChar(FMaxValue, SizeOf(FMaxValue), 0);
  FReg := TRegistry.Create;
  FFileHistory := TStringList.Create;
  FCurrentBkgnd := DefaultBackground;
  FCurrentBkgnd.Color := TForm(AOwner).Color;
  FCurrentBkgnd.XPos := TForm(AOwner).Left;
  FCurrentBkgnd.YPos := TForm(AOwner).Top;
  FCurrentBkgnd.Width := TForm(AOwner).Width;
  FCurrentBkgnd.Height := TForm(AOwner).Height;
  FProgramKey := '';
  FLastFileName := '';
  FPicture := TBitmap.Create;
  BindControl := TForm(AOwner);
end;

destructor TrBackground.Destroy;
begin
  if FBindControl <> nil then begin
    DestroyMessage(FBindControl);
  end;
  FFileHistory.Free;
  FPicture.Free;
  FReg.Free;
  FHook.Free;
  inherited Destroy;
end;

procedure TrBackground.DestroyMessage(Sender: TObject);
begin
  if FAutoSave and (Sender <> nil) then begin
    if TCrackFrom(Sender).WindowState = wsMaximized
    then FCurrentBkgnd.State := rvwmMaximized
    else begin
      MakeMax(Sender);
      if (TCrackFrom(Sender).Left = (FMaxValue[0] - TCrackFrom(Sender).Width) shr 1)
      and (TCrackFrom(Sender).Top = (FMaxValue[1] - TCrackFrom(Sender).Height) shr 1)
      then FCurrentBkgnd.State := rvwmCentered
      else FCurrentBkgnd.State := rvwmNone;
    end;
    if FCurrentBkgnd.State <> rvwmMaximized then begin
      FCurrentBkgnd.XPos   := TCrackFrom(Sender).Left;
      FCurrentBkgnd.YPos   := TCrackFrom(Sender).Top;
      FCurrentBkgnd.Width  := TCrackFrom(Sender).Width;
      FCurrentBkgnd.Height := TCrackFrom(Sender).Height;
    end;
//    SaveBackground;
    InternalSave(Sender);
  end;
end;

procedure TrBackground.BeforeHook(Sender: TObject; var Msg: TMessage; var Handled: Boolean);
var
  MyDC: hDC;
begin
  case Msg.Msg of
  WM_SHOWWINDOW: if TWMSHOWWINDOW(Msg).Show and not FUpdate then
      FUpdate := TWMSHOWWINDOW(Msg).Show and not FUpdate;
  WM_ERASEBKGND: if (THackForm(FHook.WinControl).FormStyle <> fsMDIForm) and FRedrawBkg
  then begin
    MyDC := TWMEraseBkGnd(Msg).DC;
    DrawBackGround(MyDC);
    Msg.Result := 1;
  end;
  WM_PAINT: if IsWindowVisible(THackForm(FHook.WinControl).Handle)
            then THackForm(FHook.WinControl).Invalidate;
  end;
end;

procedure TrBackground.ClientAfterHook(Sender: TObject; var Msg: TMessage; var Handled: Boolean);
var
  R: TRect;
begin
  case Msg.Msg of
  WM_VSCROLL,
  WM_HSCROLL:
    if FUpdate and Assigned(FHook.WinControl)
    then begin
      GetWindowRect(THackForm(FHook.WinControl).ClientHandle, R);
      FPrevRect := R;
      InvalidateRect(THackForm(FHook.WinControl).ClientHandle, @R, TRUE);
      RedrawWindow(THackForm(FHook.WinControl).ClientHandle, nil, 0,
            RDW_ERASENOW or RDW_UPDATENOW or RDW_FRAME or RDW_NOCHILDREN or RDW_INVALIDATE);
    end;
  $3F:
    if FUpdate and Assigned(FHook.WinControl)
    then begin
      GetWindowRect(THackForm(FHook.WinControl).ClientHandle, R);
      if not CompareMem(@R,@FPrevRect, SizeOf(TRect)) then begin
        FPrevRect := R;
        InvalidateRect(THackForm(FHook.WinControl).ClientHandle, @R, TRUE);
        RedrawWindow(THackForm(FHook.WinControl).ClientHandle, nil, 0,
              RDW_ERASENOW or RDW_UPDATENOW or RDW_FRAME or RDW_NOCHILDREN or RDW_INVALIDATE);
      end;
    end;
  end;
end;

procedure TrBackground.ClientBeforeHook(Sender: TObject; var Msg: TMessage; var Handled: Boolean);
var
  MyDC: hDC;
begin
  case Msg.Msg of
  WM_ERASEBKGND: if (THackForm(FHook.WinControl).FormStyle = fsMDIForm) and FRedrawBkg
  then begin
    MyDC := TWMEraseBkGnd(Msg).DC;
    DrawBackGround(MyDC);
    Msg.Result := 1;
  end;
  end;
end;

procedure TrBackground.SetRedrawBkg(Value: Boolean);
var
  TF: TCustomForm;
begin
  TF := BindControl;
  if FRedrawBkg <> Value then begin
    FRedrawBkg := Value;
    if (FHook.WinControl = TF) and not Value
    then begin
      FBindControl := TF;
      FHook.WinControl := nil;
    end;
    if (FBindControl = TF) and Value
    then begin
      FBindControl := nil;
      FHook.WinControl := TF;
    end;
    if FRedrawBkg and
     FUpdate
    then RefreshForm;
  end;
end;

procedure TrBackground.DrawBackGround(DC: hDC);

  procedure ClearClient(DC: hDC; Style: TrViewMode);
  var
    B: TBrush;
  begin
    B := TBrush.Create;
    try
      B.Color := FCurrentBkgnd.Color;
      FillRect(DC, CurrentForm.ClientRect, B.Handle);
    finally
      B.Free;
    end;
  end;

  procedure DrawTiled(DC: hDC);
  var
    W, H,
    Ro, Co : Integer;
    R: TRect;
  begin
    if CurrentForm.FormStyle=fsMDIForm
    then GetWindowRect(CurrentForm.ClientHandle, R)
    else GetWindowRect(CurrentForm.Handle, R);
    W := R.Right-R.Left;
    H := R.Bottom-R.Top;
    for Ro := 0 to H div FPicture.Height do
      for Co := 0 to W div FPicture.Width do
        BitBlt(DC, Co*FPicture.Width, Ro*FPicture.Height,
               FPicture.Width, FPicture.Height,
               FPicture.Canvas.Handle, 0, 0, SRCCOPY);
  end;

  procedure DrawCenter(DC: hDC; Style: TrViewMode);
  var
    W, H, Ro, Co : Integer;
    R: TRect;
  begin
    if Style in [rvmNone,rvmCenter] then ClearClient(DC,Style);
    if Style in [rvmTile, rvmNone] then begin
      if Style = rvmTile
      then DrawTiled(DC);
      Exit;
    end;
    if CurrentForm.FormStyle=fsMDIForm
    then GetWindowRect(CurrentForm.ClientHandle, R)
    else GetWindowRect(CurrentForm.Handle, R);
    W := R.Right-R.Left;
    H := R.Bottom-R.Top;
    Co := 0; Ro := 0;

    if Style<>rvmStretch
    then begin
      if (W - FPicture.Width) < 0 then Co := 0
      else Co := (W - FPicture.Width) shr 1;
      if (H  - FPicture.Height) < 0 then Ro := 0
      else Ro := (H  - FPicture.Height) shr 1 ;
      W := FPicture.Width;
      H := FPicture.Height;
    end;

     StretchBlt(DC, Co, Ro, W, H, FPicture.Canvas.Handle,
               0, 0, FPicture.Width, FPicture.Height, SRCCOPY);
   end;
var
  Style: TrViewMode;
  TG: TPicture;

begin
  if (CurrentForm <> nil)
    and CurrentForm.Visible then begin
    Style := FCurrentBkgnd.ViewMode;
    if (FCurrentBkgnd.FileName <> '') and FileExists(FCurrentBkgnd.FileName) then begin
      if AnsiCompareText(FLastFileName, FCurrentBkgnd.FileName) <> 0
      then begin
        TG := TPicture.Create;
        try
          TG.LoadFromFile(FCurrentBkgnd.FileName);
          FPicture.Height := TG.Height;
          FPicture.Width  := TG.Width;
          FPicture.Canvas.Brush.Color := FCurrentBkgnd.Color;
          FPicture.Canvas.FillRect(FPicture.Canvas.ClipRect);
          FPicture.Canvas.Draw(0,0,TG.Graphic);
        finally
          TG.Free;
        end;

      end;
      FLastFileName := FCurrentBkgnd.FileName;
    end
    else begin
      FPicture.FreeImage;
      Style := rvmNone;
    end;
    DrawCenter(DC,Style);
  end;
end;

procedure TrBackground.Notification(AComponent: TComponent; Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (AComponent = CurrentForm)
  then BindControl := nil;
end;

function TrBackground.GetBindControl: TCustomform;
begin
  if FRedrawBkg
  then Result := TCustomForm(FHook.WinControl)
  else Result := FBindControl;
end;

procedure TrBackground.SetBindControl(Value: TCustomForm);
var
  TF: TCustomForm;
begin
  TF := GetBindControl;
  if {TCustomForm(FHook.WinControl)}TF <> Value then begin
    if (Value <> nil) and not (Value is TCustomForm)
    then raise Exception.Create(srBkgndBind);
    if FRedrawBkg then begin
      if (TCustomForm(FHook.WinControl) <> nil) and not (csDesigning in ComponentState) then begin
        TCrackFrom(FHook.WinControl).OnCreate := FCreateEvent;
        TCrackFrom(FHook.WinControl).OnClose := FCloseEvent;
        TCrackFrom(FHook.WinControl).OnHide := FOnHideEvent;
      end;
      FHook.WinControl := Value;
      FBindControl := nil;
    end
    else begin
      if (FBindControl <> nil) and not (csDesigning in ComponentState) then begin
        TCrackFrom(FBindControl).OnCreate := FCreateEvent;
        TCrackFrom(FBindControl).OnClose := FCloseEvent;
        TCrackFrom(FBindControl).OnHide := FOnHideEvent;
      end;
      FBindControl := Value;
      FHook.WinControl := nil;
    end;
    if (Value <> nil) then begin
      if ({TCustomForm(FHook.WinControl)}Value <> nil) and not (csDesigning in ComponentState) then begin
{
        FCreateEvent := TCrackFrom(FHook.WinControl).OnCreate;
        FCloseEvent := TCrackFrom(FHook.WinControl).OnClose;
        FOnHideEvent := TCrackFrom(FHook.WinControl).OnHide;
        TCrackFrom(FHook.WinControl).OnCreate := OnFormCreate;
        TCrackFrom(FHook.WinControl).OnClose := OnFormClose;
        TCrackFrom(FHook.WinControl).OnHide := OnFormHide;
}
        FCreateEvent := TCrackFrom(Value).OnCreate;
        FCloseEvent := TCrackFrom(Value).OnClose;
        FOnHideEvent := TCrackFrom(Value).OnHide;
        TCrackFrom(Value).OnCreate := OnFormCreate;
        TCrackFrom(Value).OnClose := OnFormClose;
        TCrackFrom(Value).OnHide := OnFormHide;
      end;
      MakeMax(Value);
      if LoadBackground(FCurrentBkgnd)
      then begin
        ChangeSizePos;
        LoadOtherControl;
      end;
    end;
  end;
end;

procedure TrBackground.OnFormHide(Sender: TObject);
begin
  DestroyMessage(Sender);
  if Assigned(FOnHideEvent) then FOnHideEvent(Sender);
end;

procedure TrBackground.OnFormClose(Sender: TObject; var Action: TCloseAction);
begin
  DestroyMessage(Sender);
  if Assigned(FCloseEvent) then FCloseEvent(Sender, Action);
end;

procedure TrBackground.OnFormCreate(Sender: TObject);
begin
  if Assigned(FCreateEvent) then FCreateEvent(Sender);
  if LoadBackground(FCurrentBkgnd)
    and (BindControl <> nil) then begin
    TCrackFrom(BindControl).Left   := FCurrentBkgnd.XPos;
    TCrackFrom(BindControl).Top    := FCurrentBkgnd.YPos;
    TCrackFrom(BindControl).Width  := FCurrentBkgnd.Width;
    TCrackFrom(BindControl).Height := FCurrentBkgnd.Height;
    MakeMax(BindControl);
    case FCurrentBkgnd.State of
    rvwmMaximized: TCrackFrom(BindControl).WindowState := wsMaximized;
    rvwmNone,
    rvwmCentered : begin
        if FCurrentBkgnd.State = rvwmCentered then begin
          TCrackFrom(BindControl).Left := (FMaxValue[0] - FCurrentBkgnd.Width) div 2;
          TCrackFrom(BindControl).Top  := (FMaxValue[1] - FCurrentBkgnd.Height) div 2;
          if TCrackFrom(BindControl).Left < 0 then TCrackFrom(BindControl).Left := 0;
          if TCrackFrom(BindControl).Top < 0 then TCrackFrom(BindControl).Top  := 0;
        end;
        if (TCrackFrom(BindControl).Left+TCrackFrom(BindControl).Width) > FMaxValue[0]
        then TCrackFrom(BindControl).Width := FMaxValue[0]-TCrackFrom(BindControl).Left;
        if (TCrackFrom(BindControl).Top+TCrackFrom(BindControl).Height) > FMaxValue[1]
        then TCrackFrom(BindControl).Height := FMaxValue[1]-TCrackFrom(BindControl).Top;
      end;
    end;
    LoadOtherControl;
  end;
end;

procedure TrBackground.SetProgramKey(Value: String);
begin
  Value := Trim(Value);
  if Value <> FProgramKey then begin
    if (FProgramKey <> EmptyStr) and FAutoSave then SaveBackground;
    FProgramKey := Value;
    if LoadBackground(FCurrentBkgnd)
    then begin
      ChangeSizePos;
      LoadOtherControl;
    end;
  end;
end;

function TrBackground.CurrentForm: THackForm;
begin
  Result := THackForm(BindControl)
end;

procedure TrBackground.RefreshForm;
begin
  if IsWindowVisible(CurrentForm.Handle) then RefreshWindow(TForm(BindControl));
end;

procedure TrBackground.MakeMax(Sender: TObject);
var
  R: TRect;
begin
  if TCrackFrom(Sender) <> nil then begin
    if not (csDesigning in ComponentState) then
      if TCrackFrom(Sender).FormStyle <> fsMDIChild
      then SystemParametersInfo(SPI_GETWORKAREA, 0, @R, 0)
      else GetClientRect(Application.MainForm.ClientHandle, R)
    else R := Rect(0,0, Screen.Width, Screen.Height);
    FMaxValue[0] := R.Right-R.Left;
    FMaxValue[2] := R.Right-R.Left;
    FMaxValue[1] := R.Bottom-R.Top;
    FMaxValue[3] := R.Bottom-R.Top;
  end;
end;

function TrBackground.Execute: Boolean;
var
  Dlg: TrBackgroundDlg;
  I: Integer;
  Df: TrBackgroundStruct;
begin
  Result := False;
  Dlg := TrBackgroundDlg.Create(Application);
  try
    LoadBackground(Df);
    Dlg.DefBkgnd := Df;
    for I := 0 to FFileHistory.Count-1 do
      Dlg.BackgroundBitmap := FFileHistory.Strings[I];
    Dlg.BackgroundColor  := Color;
    Dlg.BackgroundBitmap := FileName;
    Dlg.BackgoundView    := ViewMode;
    Dlg.ChangeSize := rpsSizeEnabled in FPosSizeEnabled;
    Dlg.ChangePos  := rpsPosEnabled in FPosSizeEnabled;

    MakeMax(CurrentForm);
    if CurrentForm.WindowState <> wsMaximized then begin
      Dlg.XPos       := CurrentForm.Left;
      Dlg.YPos       := CurrentForm.Top;
      Dlg.FormWidth  := CurrentForm.Width;
      Dlg.FormHeight := CurrentForm.Height;
      if (CurrentForm.Left = (FMaxValue[0] - CurrentForm.Width) shr 1)
      and (CurrentForm.Top = (FMaxValue[1] - CurrentForm.Height) shr 1)
      then Dlg.FormState := rvwmCentered
      else Dlg.FormState := rvwmNone;
    end
    else begin
      Dlg.XPos       := FCurrentBkgnd.XPos;
      Dlg.YPos       := FCurrentBkgnd.YPos;
      Dlg.FormWidth  := FCurrentBkgnd.Width;
      Dlg.FormHeight := FCurrentBkgnd.Height;
      Dlg.FormState  := rvwmMaximized;
    end;
    for I := 0 to 3 do
      Dlg.SetmaxValue(I, FMaxValue[I]);
    for I := 0 to Dlg.GroupBox1.ControlCount-1 do
      Dlg.GroupBox1.Controls[I].Enabled := FRedrawBkg;
    Dlg.GroupBox1.Enabled := FRedrawBkg;
    for I := 0 to Dlg.GroupBox2.ControlCount-1 do
      Dlg.GroupBox2.Controls[I].Enabled := FRedrawBkg;
    Dlg.GroupBox2.Enabled := FRedrawBkg;
    Dlg.SizeBox.Enabled := CurrentForm.BorderStyle <> bsDialog;
    for I := 0 to Dlg.SizeBox.ControlCount-1 do
      Dlg.SizeBox.Controls[I].Enabled := Dlg.SizeBox.Enabled;
    if Dlg.ShowModal = mrOK then begin
      FCurrentBkgnd.FileName := Dlg.BackgroundBitmap;
      FCurrentBkgnd.Color    := Dlg.BackgroundColor;
      FCurrentBkgnd.ViewMode := Dlg.BackgoundView;
      FCurrentBkgnd.State := Dlg.FormState;
      if FCurrentBkgnd.State <> rvwmMaximized
      then begin
        FCurrentBkgnd.XPos   := Dlg.XPos;
        FCurrentBkgnd.YPos   := Dlg.YPos;
        FCurrentBkgnd.Width  := Dlg.FormWidth;
        FCurrentBkgnd.Height := Dlg.FormHeight;
      end;
      SaveBackground;
      ChangeSizePos;
      RefreshForm;
      Result := True;
    end;
  finally
    Dlg.Free;
  end;
end;

procedure TrBackground.InternalSave(Sender: TObject);

var
  S, SP, TBN: String;
  FC: TControl;
  I, J, N: Integer;
  OwnList: TStringList;

  function FindControl(SPL: TSplitter): TControl;
  var
    P: TPoint;
    I: Integer;
  begin
    Result := nil;
    P := Point(SPL.Left, SPL.Top);
    case SPL.Align of
      alLeft: Dec(P.X);
      alRight: Inc(P.X, SPL.Width);
      alTop: Dec(P.Y);
      alBottom: Inc(P.Y, SPL.Height);
    else
      Exit;
    end;
    for I := 0 to SPL.Parent.ControlCount - 1 do
    begin
      Result := SPL.Parent.Controls[I];
      if PtInRect(Result.BoundsRect, P) then Exit;
    end;
    Result := nil;
  end;

begin
  if (Sender <> nil) and not (csDesigning in ComponentState) then begin
    if FProgramKey = '' then Exit;
    try
      if FReg.OpenKey(FProgramKey, True) and (Sender is TCustomForm) and (TCrackFrom(Sender).Name <> EmptyStr)
      then begin
        if (rpsPosEnabled in FPosSizeEnabled)
        then S := Format(';%d;%d',[FCurrentBkgnd.XPos, FCurrentBkgnd.YPos])
        else S := ';;';
        if (rpsSizeEnabled in FPosSizeEnabled)
        then S := Format('%s;%d;%d',[S,FCurrentBkgnd.Width,FCurrentBkgnd.Height])
        else S := S+';;';
        S := S+Format(';%d',[Ord(FCurrentBkgnd.State)]);
        if FRedrawBkg
        then  S := Format('%s;%d;%d%s',[FCurrentBkgnd.FileName,
                                        Integer(FCurrentBkgnd.Color),
                                        Integer(FCurrentBkgnd.ViewMode), S])
        else S := Format(';%d;%d%s',[Integer(FCurrentBkgnd.Color),
                                     Integer(FCurrentBkgnd.ViewMode), S]);
        FReg.WriteString(TCrackFrom(Sender).Name, S);
        if not FSaveSplitterPos
        then FReg.DeleteValue(TCrackFrom(Sender).Name+srBkgndSlit)
        else begin
          SP := '';
          N := 0;
          for I := 0 to TCrackFrom(Sender).ComponentCount-1 do
          if TCrackFrom(Sender).Components[I] is TSplitter then begin
            FC := FindControl(TSplitter(TCrackFrom(Sender).Components[I]));
            if FC <> nil then begin
              Inc(N);
              SP := Format('%s;%s;%d;%d',[SP, FC.Name, FC.Height, FC.Width]);
            end
          end;
          if N > 0 then begin
            S := Format('%d%s',[N,SP]);
            FReg.WriteString(TCrackFrom(Sender).Name+srBkgndSlit, S);
          end
          else FReg.DeleteValue(TCrackFrom(Sender).Name+srBkgndSlit);
        end;
        if FSaveToolBar then begin
          OwnList := TStringList.Create;
          try
            OwnList.Duplicates := dupIgnore;
            for I := 0 to TCrackFrom(Sender).ComponentCount-1 do
            if (TCrackFrom(Sender).Components[I] is TToolBar) and
               (TToolBar(TCrackFrom(Sender).Components[I]).Parent <> nil)
            then OwnList.Add(TToolBar(TCrackFrom(Sender).Components[I]).Parent.Name);
            if OwnList.Count > 0 then begin
              for I := 0 to OwnList.Count-1 do begin
                SP := '';
                N := 0;
                for J := 0 to TCrackFrom(Sender).ComponentCount-1 do
                  if (TCrackFrom(Sender).Components[J] is TToolBar) and
                     (TToolBar(TCrackFrom(Sender).Components[J]).Parent <> nil) and
                     (TToolBar(TCrackFrom(Sender).Components[J]).Parent.Name = OwnList.Strings[I])
                  then begin
                    Inc(N);
                    SP := Format('%s;%s;%d;%d;%d;%d',[SP,
                          TToolBar(TCrackFrom(Sender).Components[J]).Name,
                          TToolBar(TCrackFrom(Sender).Components[J]).Left,
                          TToolBar(TCrackFrom(Sender).Components[J]).Top,
                          TToolBar(TCrackFrom(Sender).Components[J]).Height,
                          TToolBar(TCrackFrom(Sender).Components[J]).Width]);
                  end;
                TBN := OwnList.Strings[I];
                if TBN = '' then begin
                  TBN := '~ToolBars';
                  OwnList.Strings[I] := TBN;
                end;
                if N > 0 then begin
                   S := Format('%d%s',[N,SP]);
                   FReg.WriteString(TCrackFrom(Sender).Name+TBN, S);
                end
                else FReg.DeleteValue(TCrackFrom(Sender).Name+TBN);
              end;
              FReg.WriteString(TCrackFrom(Sender).Name+srBkgndTool, OwnList.CommaText);
            end;
          finally
            OwnList.Free;
          end;
        end;
      end;
    finally
      FReg.CloseKey;
    end;
  end;
end;

procedure TrBackground.SaveBackground;
begin
  InternalSave(CurrentForm);
end;

type
  TSPL = class(TSplitter);

procedure TrBackground.LoadOtherControl;
var
  S, SS: String;
  I, J, K, N: Integer;
  SPL : TControl;
  OwnList: TStringList;
begin
  if (csLoading in ComponentState) then Exit;
  if CurrentForm <> nil then begin
    if FProgramKey = '' then Exit;
    try
      if FReg.KeyExists(FProgramKey) and FReg.OpenKey(FProgramKey, False)
      and (CurrentForm is TCustomForm) and (CurrentForm.Name <> EmptyStr)
      and FReg.ValueExists(CurrentForm.Name)
      then begin
//
        if not FUpdate then begin
          SS := '';
          S := FReg.ReadString(CurrentForm.Name+srBkgndSlit);
          N := GetDelimCount(S, [';']);
          if N > 0 then begin
            N := N+1;
            SPL := nil;
            for I := 1 to N do begin
              SS := ExtractDelimited(I,S,[';']);
              if I = 1 then begin
                J := StrToIntDef(SS,0);
                if J = 0 then Break;
              end
              else begin
                J := 0;
                if (I-2) mod 3 <> 0 then begin
                  J := StrToIntDef(SS,0);
                  if J <= 0 then Continue;
                end;
                case (I-2) mod 3 of
                0: begin
                     SPL := nil;
                     if SS <> EmptyStr
                     then SPL := TControl(CurrentForm.FindComponent(SS));
                   end;
                1: if SPL <> nil then SPL.Height := J;
                2: if SPL <> nil then SPL.Width := J;
                end;
              end;
            end;
          end;
          OwnList := TStringList.Create;
          try
            OwnList.CommaText := FReg.ReadString(CurrentForm.Name+srBkgndTool);
            if OwnList.Count > 0 then
              for I := 0 to OwnList.Count-1 do begin
                SS := '';
                S := FReg.ReadString(CurrentForm.Name+OwnList.Strings[I]);
                N := GetDelimCount(S, [';']);
                if N > 0 then begin
                  N := N+1;
                  SPL := nil;
                  for J := 1 to N do begin
                    SS := ExtractDelimited(J,S,[';']);
                    if J = 1 then begin
                      K := StrToIntDef(SS,0);
                      if K = 0 then Break;
                    end
                    else begin
                      K := 0;
                      if (J-2) mod 5 <> 0 then begin
                        K := StrToIntDef(SS,0);
                        if K <= 0 then Continue;
                      end;
                      case (J-2) mod 5 of
                      0: begin
                           SPL := nil;
                           if SS <> EmptyStr
                           then SPL := TControl(CurrentForm.FindComponent(SS));
                         end;
                      1: if SPL <> nil then SPL.Left   := K;
                      2: if SPL <> nil then SPL.Top    := K;
                      3: if SPL <> nil then SPL.Height := K;
                      4: if SPL <> nil then SPL.Width  := K;
                      end;
                    end;
                  end;
                end;
              end;
          finally
            OwnLIst.Free;
          end;
        end;
//
      end
    finally
      FReg.CloseKey;
    end;
  end;

end;

function TrBackground.LoadBackground(var Value: TrBackgroundStruct): Boolean;
var
  S, SS: String;
  I, J, N: Integer;
begin
  Result := False;
  if (csLoading in ComponentState) then Exit;
  if CurrentForm <> nil then begin
    if FProgramKey = '' then Exit;
    try
      if FReg.KeyExists(FProgramKey) and FReg.OpenKey(FProgramKey, False)
      and (CurrentForm is TCustomForm) and (CurrentForm.Name <> EmptyStr)
      and FReg.ValueExists(CurrentForm.Name)
      then begin
        S := FReg.ReadString(CurrentForm.Name);
        N := GetDelimCount(S, [';']);
        if N > 0 then begin
          N := N+1;
          for I := 1 to N do begin
            SS := ExtractDelimited(I,S,[';']);
            case I of
            1: begin
                 Value.FileName := SS;
                 if (SS <> '') and not FileExists(SS)
                 then Value.FileName := '';
                 if (SS <> '') and FileExists(SS) and (FFileHistory.IndexOf(SS)< 0)
                 then FFileHistory.Add(SS);
               end;
            2: begin
                 J := StrToIntDef(SS,-1);
                 if J <> -1 then Value.Color := TColor(J);
               end;
            3: begin
                 J := StrToIntDef(SS,-1);
                 if J in [0..3] then Value.ViewMode := TrViewMode(J);
               end;
            4..7: begin
                 J := StrToIntDef(SS,-1);
                 if ((I in [4..5]) and (J < 0))
                 or ((I in [6..7]) and (J <= 0))
                 and (CurrentForm <> nil)
                 then case I of
                      4: J := CurrentForm.Left;
                      5: J := CurrentForm.Top;
                      6: J := CurrentForm.Width;
                      7: J := CurrentForm.Height;
                      end;
                 case I of
                 4: Value.XPos   := J;
                 5: Value.YPos   := J;
                 6: Value.Width  := J;
                 7: Value.Height := J;
                 end;
               end;
            8 : begin
                  J := StrToIntDef(SS,0);
                  if J in [0..2] then begin
                     if (CurrentForm.BorderStyle = bsDialog) and (J = 1)
                     then J := 0;
                     Value.State := TrViewWindowMode(J);
                  end
                  else Value.State := rvwmNone;
               end;
            end;
          end;
        end
        else begin
          Value := DefaultBackground;
          Value.Color  := CurrentForm.Color;
          Value.XPos   := CurrentForm.Left;
          Value.YPos   := CurrentForm.Top;
          Value.Width  := CurrentForm.Width;
          Value.Height := CurrentForm.Height;
          if CurrentForm.WindowState = wsMaximized
          then FCurrentBkgnd.State := rvwmMaximized
          else begin
            MakeMax(CurrentForm);
            if (CurrentForm.Left = (FMaxValue[0] - Value.Width) shr 1)
            and (CurrentForm.Top = (FMaxValue[1] - Value.Height) shr 1)
            then FCurrentBkgnd.State := rvwmCentered
            else FCurrentBkgnd.State := rvwmNone;
          end
        end;

        Result := True;
      end
      else DestroyMessage(Self);
    finally
      FReg.CloseKey;
    end;
  end;
end;

procedure TrBackground.SetCurrentBkgnd(Value: TrBackgroundStruct);
begin
  if not Testbackground(FCurrentBkgnd, Value)
  then ChangeSizePos;
end;

procedure TrBackground.SetBkgndColor(Value: TColor);
begin
  if (Value <> FCurrentBkgnd.Color) then begin
    FCurrentBkgnd.Color := Value;
    if FUpdate then RefreshForm;
  end;
end;

procedure TrBackground.SetBkgndFileName(Value: TFileName);
begin
  Value := Trim(Value);
  if (AnsiCompareText(Value, FCurrentBkgnd.FileName) <> 0)
  then begin
    FCurrentBkgnd.FileName := Value;
    FLastFileName := '';
    if FUpdate then RefreshForm;
    if (Value <> '') and FileExists(Value) and (FFileHistory.IndexOf(Value)< 0)
    then FFileHistory.Add(Value);
  end;
end;

procedure TrBackground.SetBkgndViewMode(Value: TrViewMode);
begin
  if (Value <> FCurrentBkgnd.ViewMode) then begin
    FCurrentBkgnd.ViewMode := Value;
    if FUpdate then RefreshForm;
  end;
end;

function TrBackground.TestBackground(Value1, Value2: TrBackgroundStruct): Boolean;
begin
  Result := (Value1.FileName=Value2.FileName)
        and (Value1.Color   =Value2.Color)
        and (Value1.ViewMode=Value2.ViewMode)
        and (Value1.XPos    =Value2.XPos)
        and (Value1.YPos    =Value2.YPos)
        and (Value1.Width   =Value2.Width)
        and (Value1.Height  =Value2.Height)
        and (Value1.State   =Value2.State) ;
end;

function TrBackground.GetCurrentRect: TRect;
var
  X, Y, W, H: Integer;
begin
  Result := Rect(0,0,0,0);
  if (CurrentForm <> nil) then Result := CurrentForm.BoundsRect;
  if (CurrentForm <> nil) and ((rpsPosEnabled in FPosSizeEnabled)
  or (rpsSizeEnabled in FPosSizeEnabled))then begin
    if rpsSizeEnabled in FPosSizeEnabled
    then begin
      W := FCurrentBkgnd.Width; H := FCurrentBkgnd.Height;
    end
    else begin
      W := CurrentForm.Width; H := CurrentForm.Height;
    end;
    if rpsPosEnabled in FPosSizeEnabled
    then begin
      X := FCurrentBkgnd.XPos; Y := FCurrentBkgnd.YPos;
    end
    else begin
      X := CurrentForm.Left;  Y := CurrentForm.Top;
    end;
    MakeMax(CurrentForm);
    if FCurrentBkgnd.State = rvwmCentered then begin
      X := (FMaxValue[0] - W) shr 1; Y := (FMaxValue[1] - H) shr 1;
    end;
    Result := Rect(X, Y, X+W, Y+H);
  end;
end;

procedure TrBackground.ChangeSizePos;
var
  X, Y,
  W, H: Integer;
  R: TRect;
  P: TWindowPlacement;
begin
  if not (csDesigning in ComponentState) and not (csLoading  in ComponentState) then begin
    R := GetCurrentRect;
    W := R.Right-R.Left;
    H := R.Bottom-R.Top;
    p.length := SizeOf(p);
    GetWindowPlacement(CurrentForm.Handle, @p);
    p.rcNormalPosition := R;
    P.flags := WPF_RESTORETOMAXIMIZED;
    P.showCmd := SW_RESTORE;
    case FCurrentBkgnd.State of
    rvwmNone,
    rvwmCentered : begin
      if FCurrentBkgnd.State = rvwmCentered then begin
        X := (FMaxValue[0] - W) div 2;
        Y := (FMaxValue[1] - H) div 2;
        p.rcNormalPosition := Rect(X,Y, X+W,Y+H);
      end;
      if FUpdate then SetWindowPlacement(CurrentForm.Handle, @p);
    end;
    rvwmMaximized: if CurrentForm.WindowState <> wsMaximized then begin
        P.showCmd := SW_MAXIMIZE;
        if FUpdate then SetWindowPlacement(CurrentForm.Handle, @p);
      end;
    end;
    if FUpdate then RefreshForm;
  end;
end;

procedure TrBackground.Loaded;
begin
  inherited Loaded;
  if LoadBackground(FCurrentBkgnd)
  then begin
    ChangeSizePos;
    LoadOtherControl;
  end;
end;

function TrBackground.IsNotDefaultRegKey: Boolean;
begin
  Result := Trim(FProgramKey) <> '';
end;

function TrBackground.IsNotDefaultFileName: Boolean;
begin
  Result := Trim(FCurrentBkgnd.FileName) <> '';
end;

initialization
  HookList := nil;
{$IFDEF WIN32}
finalization
  DropHookList;
{$ELSE}
  AddExitProc(DropHookList);
{$ENDIF}
end.
