unit rAction;

interface

uses Classes, ActnList, Db, DBCtrls, rDBUtils, rDBConst;

type

  TrCustomDBActions = class(TAction)
  private
    FDataLink,
    FControlDataLink: TFieldDataLink;
    FPositive,
    FNegative,
    FControlPositive,
    FControlNegative: TrDBOptionSet;
    FKindAction     : TrDBAction;
    FOnActiveChanged:  TOnActiveChanged;
    FOnBeforeAction,
    FOnAfterAction:    TNotifyEvent;
    function GetDataSource: TDataSource;
    procedure SetDataSource(Value: TDataSource);
    function  GetDataField: String;
    procedure SetDataField(Value: String);
    function  GetField: TField;
    function  GetControlField: TField;
    function  GetControlDataSource: TDataSource;
    procedure SetControlDataSource(Value: TDataSource);
    function  GetControlDataField: String;
    procedure SetControlDataField(Value: String);
    procedure SetPositive(Value: TrDBOptionSet);
    procedure SetNegative(Value: TrDBOptionSet);
    function  PositiveStore: Boolean;
    function  NegativeStore: Boolean;
    procedure SetControlPositive(Value: TrDBOptionSet);
    procedure SetControlNegative(Value: TrDBOptionSet);
    procedure SetKindAction(const Value: TrDBAction);
  protected
    procedure Notification(AComponent: TComponent;
                           Operation: TOperation); override;
    function IsNotDefaultHint: Boolean;
    function IsNotDefaultCaption: Boolean;
    function IsNotDefaultShortCut: Boolean;

    property Kind: TrDBAction read FKindAction write SetKindAction default tbaCustom;
    property DataSource: TDataSource read GetDataSource write SetDataSource;
    property DataField: String read GetDataField write SetDataField;
    property Positive: TrDBOptionSet read FPositive write SetPositive
      stored PositiveStore;
    property Negative: TrDBOptionSet read FNegative write SetNegative
      stored NegativeStore;
    property ControlDataSource: TDataSource read GetControlDataSource write SetControlDataSource;
    property ControlDataField: String read GetControlDataField write SetControlDataField;
    property ControlPositive: TrDBOptionSet read FControlPositive write SetControlPositive
      default [];
    property ControlNegative: TrDBOptionSet read FControlNegative write SetControlNegative
      default [];
    property ActiveChanged: TOnActiveChanged read FOnActiveChanged write FOnActiveChanged;
    property BeforeAction: TNotifyEvent read FOnBeforeAction write FOnBeforeAction;
    property AfterAction: TNotifyEvent read FOnAfterAction write FOnAfterAction;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure ExecuteTarget(Target: TObject); override;
    procedure UpdateTarget(Target: TObject); override;
    function HandlesTarget(Target: TObject): Boolean; override;
    property Field: TField read GetField;
    property ControlField: TField read GetControlField;
  published
    property ShortCut stored IsNotDefaultShortCut;
    property Caption stored IsNotDefaultCaption;
    property Enabled default False;
    property Hint stored IsNotDefaultHint;
  end;

  TrDBActions = class(TrCustomDBActions)
  published
    property Kind;
    property DataSource;
    property DataField;
    property Positive;
    property Negative;
    property ControlDataSource;
    property ControlDataField;
    property ControlPositive;
    property ControlNegative;
    property ActiveChanged;
    property BeforeAction;
    property AfterAction;
  end;

implementation

uses SysUtils;

{ TrCustomDBActions }
constructor TrCustomDBActions.Create(AOwner: TComponent);
begin
  inherited;
  Enabled           := False;
  FDataLink         := TFieldDataLink.Create;
  FDataLink.Control := Self;
  FControlDataLink  := TFieldDataLink.Create;
  FControlDataLink.Control := Self;
  FPositive       := [];
  FNegative       := [];
  FControlPositive:= [];
  FControlNegative:= [];
  FKindAction     := tbaCustom;
end;

destructor TrCustomDBActions.Destroy;
begin
  if Assigned(FDataLink) then FDataLink.Free;
  if Assigned(FControlDataLink) then FControlDataLink.Free;
  inherited;
end;
procedure TrCustomDBActions.Notification(AComponent: TComponent;
                                     Operation: TOperation);
begin
  inherited;
  if (Operation = opRemove) then
    if (FDataLink <> nil) and (AComponent = DataSource) then DataSource := nil
    else
      if (FControlDataLink <> nil) and (AComponent = ControlDataSource) then
        ControlDataSource := nil
end;

function TrCustomDBActions.HandlesTarget(Target: TObject): Boolean;
begin
  Result :=
    ((DataSource <> nil) and (Target = DataSource) and
     (DataSource.DataSet <> nil) or (DataSource = nil))
    or
    ((ControlDataSource <> nil) and (Target = ControlDataSource) and
     (ControlDataSource.DataSet <> nil) or (ControlDataSource = nil))
    and
    (Target is TDataSource) and (TDataSource(Target).DataSet <> nil);
end;

procedure TrCustomDBActions.UpdateTarget(Target: TObject);
var
  En: Boolean;
begin
  En := Assigned(FDataLink) and
        TestOptions(FDataLink.DataSet,FDataLink.Field,FPositive,FNegative);
  if En and Assigned(FControlDataLink) and Assigned(ControlDataSource) then
    En := TestOptions(FControlDataLink.DataSet, FControlDataLink.Field,
                      FControlPositive, FControlNegative);
  if En and Assigned(FOnActiveChanged) then FOnActiveChanged(Self,FDataLink,En);
  Enabled := En;
end;

procedure TrCustomDBActions.ExecuteTarget(Target: TObject);
begin
  if {Assigned(DataSource) and
     (FKindAction <> tbaCustom) and }Assigned(FOnBeforeAction) then
    FOnBeforeAction(Self);
  if Assigned(DataSource) and
     DoAction(DataSource.DataSet,Field,FKindAction) and
     (FKindAction <> tbaCustom) and Assigned(FOnAfterAction) then
    FOnAfterAction(Self);
end;

function TrCustomDBActions.GetDataSource: TDataSource;
begin
  if Assigned(FDataLink) then Result := FDataLink.DataSource
  else Result := nil;
end;

procedure TrCustomDBActions.SetDataSource(Value: TDataSource);
begin
  if (DataSource <> Value) and
     not (FDataLink.DataSourceFixed and (csLoading in ComponentState)) then begin
    FDataLink.DataSource := Value;
    if Value <> nil then Value.FreeNotification(Self);
  end;
end;

function TrCustomDBActions.GetDataField: String;
begin
  if Assigned(FDataLink) then Result := FDataLink.FieldName
  else Result := '';
end;

procedure TrCustomDBActions.SetDataField(Value: String);
begin
  if Assigned(FDataLink) then FDataLink.FieldName := Value;
end;

function TrCustomDBActions.GetField: TField;
begin
  if Assigned(FDataLink) then Result := FDataLink.Field
  else Result := nil;
end;

function TrCustomDBActions.GetControlField: TField;
begin
  if Assigned(FControlDataLink) then Result := FControlDataLink.Field
  else Result := nil;
end;

function TrCustomDBActions.GetControlDataSource: TDataSource;
begin
  if Assigned(FControlDataLink) then Result := FControlDataLink.DataSource
  else Result := nil;
end;

procedure TrCustomDBActions.SetControlDataSource(Value: TDataSource);
begin
  if (ControlDataSource <> Value) and
     not (FControlDataLink.DataSourceFixed and (csLoading in ComponentState)) then begin
    FControlDataLink.DataSource := Value;
    if Value <> nil then Value.FreeNotification(Self);
  end;
end;

function TrCustomDBActions.GetControlDataField: String;
begin
  if Assigned(FControlDataLink) then Result := FControlDataLink.FieldName
  else Result := '';
end;

procedure TrCustomDBActions.SetControlDataField(Value: String);
begin
  if Assigned(FControlDataLink) then FControlDataLink.FieldName := Value;
end;

procedure TrCustomDBActions.SetPositive(Value: TrDBOptionSet);
begin
  FPositive := Value;
  UpdateTarget(Self)
end;

procedure TrCustomDBActions.SetNegative(Value: TrDBOptionSet);
begin
  FNegative := Value;
  UpdateTarget(Self)
end;

function TrCustomDBActions.PositiveStore;
begin
  Result := not (FPositive = Positives[FKindAction]);
end;

function TrCustomDBActions.NegativeStore;
begin
  Result := not (FNegative = Negatives[FKindAction]);
end;

function TrCustomDBActions.IsNotDefaultHint: Boolean;
var
  S: String;
begin
  S := LoadResString(DBActionHints[FKindAction]);
  Result := ANSICompareText(Hint,S)<>0;
end;

function TrCustomDBActions.IsNotDefaultCaption: Boolean;
var
  S: String;
begin
  S := LoadResString(RDBA_Caption[FKindAction]);
  Result := (ANSICompareText(Caption, Name)<>0) and (ANSICompareText(Caption,S)<>0);
end;

function TrCustomDBActions.IsNotDefaultShortCut: Boolean;
begin
  Result := ShortCut <> DBActionShortCuts[FKindAction];
end;

procedure TrCustomDBActions.SetControlPositive(Value: TrDBOptionSet);
begin
  FControlPositive := Value;
  UpdateTarget(Self)
end;

procedure TrCustomDBActions.SetControlNegative(Value: TrDBOptionSet);
begin
  FControlNegative := Value;
  UpdateTarget(Self)
end;

procedure TrCustomDBActions.SetKindAction(const Value: TrDBAction);
begin
  if FKindAction <> Value then begin
    if (LoadResString(DBActionHints[FKindAction])=Hint) or (Hint='')
    then Hint := LoadResString(DBActionHints[Value]);
    if (LoadResString(RDBA_Caption[FKindAction])=Caption) or (Caption=Name) or (Caption='')
    then Caption := LoadResString(RDBA_Caption[Value]);
    if ShortCut = DBActionShortCuts[FKindAction]
    then ShortCut := DBActionShortCuts[Value];
    ImageIndex := DBImageIndexes[Value];
    FNegative := Negatives[Value];
    FPositive := Positives[Value];
    UpdateTarget(Self);
    FKindAction := Value;
  end;
end;

end.
