//==============================================
//       Launch.pas
//
//         Delphi.
//         .
//
//      Copyright 1998-2000 Polaris Software
//      http://members.xoom.com/PolarisSoft
//      mailto: PolarisLib@mail.ru
//==============================================
unit Launch;

{$I POLARIS.INC}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls, ShellApi, StrUtils, rConst;

type
  TrLaunchMethod = (lmCreateProcess, lmShellExecute);

  TrCreationFlag = (cfDebugProcess,cfDebugOnlyThisProcess,cfSuspended,cfDetachedProcess,
                    cfNewConsole,cfNewProcessGroup,cfUnicodeEnvironment,
                    cfSeparateWOW_VDM,cfSharedWOW_VDM,cfForceDOS,cfDefaultErrorMode,
                    cfNoWindow,cfProfileUser,cfProfileKernel,cfProfileServer);
  TrCreationFlags = set of TrCreationFlag;
  TrPriorityClass = (pcIdle, pcNormal, pcHigh, pcRealTime);

  TrStartupFlag = (sfUseShowWindow,sfUsePosition,sfUseSize,sfUseConsoleChars,sfUseFillAttributes,
                   sfForceOnFeedback,sfForceOffFeedback,sfRunFullScreen,sfStdHandles,sfUseHotKey);
  TrStartupFlags = set of TrStartupFlag;
  TShowMode = (smHide, smNormal, smMaximized, smMinimized, smMinNoActive, smNoActivate);

  TrFillAttribute = (faForeBlue,faForeGreen,faForeRed,faForeIntensity,
                     faBackBlue,faBackGreen,faBackRed,faBackIntensity);
  TrFillAttributes = set of TrFillAttribute;

  TrBeforeLaunch = procedure(Sender: TObject; var StartupInfo: TStartupInfo;
                                              var ShellExecuteInfo: TShellExecuteInfo) of object;
  TrOnTimeoutElapsed = procedure(Sender: TObject; WaitInit: Boolean) of object;

const
  rCreationFlags: array[TrCreationFlag] of DWORD =
     (DEBUG_PROCESS,DEBUG_ONLY_THIS_PROCESS,CREATE_SUSPENDED,DETACHED_PROCESS,
      CREATE_NEW_CONSOLE,CREATE_NEW_PROCESS_GROUP,CREATE_UNICODE_ENVIRONMENT,
      CREATE_SEPARATE_WOW_VDM,CREATE_SHARED_WOW_VDM,CREATE_FORCEDOS,CREATE_DEFAULT_ERROR_MODE,
      CREATE_NO_WINDOW,PROFILE_USER,PROFILE_KERNEL,PROFILE_SERVER);

  rPriorityClasses: array[TrPriorityClass] of DWORD =
     (IDLE_PRIORITY_CLASS, NORMAL_PRIORITY_CLASS, HIGH_PRIORITY_CLASS, REALTIME_PRIORITY_CLASS);

  rStartupFlags: array[TrStartupFlag] of DWORD =
     (STARTF_USESHOWWINDOW,STARTF_USEPOSITION,STARTF_USESIZE,STARTF_USECOUNTCHARS,
      STARTF_USEFILLATTRIBUTE,STARTF_FORCEONFEEDBACK,STARTF_FORCEOFFFEEDBACK,
      STARTF_RUNFULLSCREEN,STARTF_USESTDHANDLES,STARTF_USEHOTKEY);

  ShowWindowModes: array[TShowMode] of Word =
     (SW_HIDE, SW_SHOWNORMAL, SW_SHOWMAXIMIZED, SW_SHOWMINIMIZED, SW_SHOWMINNOACTIVE, SW_SHOWNOACTIVATE);

  rFillAttributes: array[TrFillAttribute] of DWORD =
     (FOREGROUND_BLUE,FOREGROUND_GREEN,FOREGROUND_RED,FOREGROUND_INTENSITY,
      BACKGROUND_BLUE,BACKGROUND_GREEN,BACKGROUND_RED,BACKGROUND_INTENSITY);

type
  TLauncher = class(TComponent)
  private
    FFirstTick, FLastTick: DWORD;
    FProcessInfo: TProcessInformation;
    FFullProgramName: string;
    FParameters: string;
    FWorkDir: string;
    FLaunchMethod: TrLaunchMethod;
    FCreationFlags: TrCreationFlags;
    FPriority: TrPriorityClass;
    FShowMode: TShowMode;
    FDesktop: string;
    FConsoleTitle: string;
    FWindowLeft, FWindowTop: Integer;
    FWindowWidth, FWindowHeight: Cardinal;
    FConsoleCharsX, FConsoleCharsY: Cardinal;
    FConsoleFillAttributes: TrFillAttributes;
    FEnvironment: TStrings;
    FStartupFlags: TrStartupFlags;
    FTimer: TTimer;
    FTimerInterval: Cardinal;
    FShellOperation: string;
    FTimeoutWaitInit: Cardinal;
    FTimeoutWaitTerminate: Cardinal;
    FOnFinished: TNotifyEvent;
    FOnTimeoutElapsed: TrOnTimeoutElapsed;
    FBeforeLaunch: TrBeforeLaunch;
    procedure SetEnvironment(Value: TStrings);
  protected
    procedure RaiseError(ErrCode: Integer);
    procedure Finished; dynamic;
    procedure TimerExpired(Sender: TObject);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Launch;
    property ProcessInfo: TProcessInformation read FProcessInfo;
  published
    property ProgramName: string read FFullProgramName write FFullProgramName;
    property Parameters: string read FParameters write FParameters;
    property WorkDir: string read FWorkDir write FWorkDir;
    property LaunchMethod: TrLaunchMethod read FLaunchMethod write FLaunchMethod default lmCreateProcess;
    property CreationFlags: TrCreationFlags read FCreationFlags write FCreationFlags
       default [];
    property Priority: TrPriorityClass read FPriority write FPriority default pcNormal;
    property ShowMode: TShowMode read FShowMode write FShowMode default smNormal;
    property Desktop: string read FDesktop write FDesktop;
    property ConsoleTitle: string read FConsoleTitle write FConsoleTitle;
    property WindowLeft: Integer read FWindowLeft write FWindowLeft default 0;
    property WindowTop: Integer read FWindowTop write FWindowTop default 0;
    property WindowWidth: Cardinal read FWindowWidth write FWindowWidth default 0;
    property WindowHeight: Cardinal read FWindowHeight write FWindowHeight default 0;
    property ConsoleCharsX: Cardinal read FConsoleCharsX write FConsoleCharsX default 0;
    property ConsoleCharsY: Cardinal read FConsoleCharsY write FConsoleCharsY default 0;
    property ConsoleFillAttributes: TrFillAttributes read FConsoleFillAttributes write FConsoleFillAttributes
       default [];
    property Environment: TStrings read FEnvironment write SetEnvironment;
    property StartupFlags: TrStartupFlags read FStartupFlags write FStartupFlags default [sfUseShowWindow];
    property TimerInterval: Cardinal read FTimerInterval write FTimerInterval default 1000;
    property ShellOperation: string read FShellOperation write FShellOperation;
    property TimeoutWaitInit: Cardinal read FTimeoutWaitInit write FTimeoutWaitInit default 0;
    property TimeoutWaitTerminate: Cardinal read FTimeoutWaitTerminate write FTimeoutWaitTerminate default 0;
    property OnFinished: TNotifyEvent read FOnFinished write FOnFinished;
    property OnTimeoutElapsed: TrOnTimeoutElapsed read FOnTimeoutElapsed write FOnTimeoutElapsed;
    property BeforeLaunch: TrBeforeLaunch read FBeforeLaunch write FBeforeLaunch;
  end;

implementation

procedure TLauncher.RaiseError(ErrCode: Integer);
var
  ErrMsg: string;
begin
  case ErrCode of
    ERROR_FILE_NOT_FOUND: ErrMsg := Format(srLaunchNotFound,[AnsiUpperCase(FFullProgramName)]);
    ERROR_BAD_FORMAT, ERROR_BAD_EXE_FORMAT: ErrMsg := Format(srLaunchNotExe,[AnsiUpperCase(FFullProgramName)]);
    ERROR_DIRECTORY: ErrMsg := Format(srLaunchNoWorkDir,[AnsiUpperCase(FWorkDir)]);
  else
    RaiseLastWin32Error;
  end;
  raise EWin32Error.Create(ErrMsg);
end;

constructor TLauncher.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FLaunchMethod := lmCreateProcess;
  FPriority := pcNormal;
  FShowMode := smNormal;
  FStartupFlags := [sfUseShowWindow];
  FEnvironment := TStringList.Create;
  FTimerInterval := 1000;
  FillChar(FProcessInfo,SizeOf(FProcessInfo),0);
end;

destructor TLauncher.Destroy;
begin
  if Assigned(FTimer) then
    with FTimer do begin
      Enabled := False;
      Free;
    end;
  FEnvironment.Free;
  inherited Destroy;
end;

procedure TLauncher.SetEnvironment(Value: TStrings);
begin
  FEnvironment.Assign(Value);
end;

procedure TLauncher.Finished;
begin
  if (FTimeoutWaitTerminate > 0) and (FFirstTick+FTimeoutWaitTerminate <= FLastTick) then
    if Assigned(FOnTimeoutElapsed) then FOnTimeoutElapsed(Self,False)
    else
  else
    if Assigned(FOnFinished) then FOnFinished(Self);
end;

procedure TLauncher.TimerExpired(Sender: TObject);
var
  eCode: DWORD;
begin
  FLastTick := GetTickCount;
  if not GetExitCodeProcess(FProcessInfo.hProcess,eCode) or
     (eCode <> STILL_ACTIVE) or
     ((FTimeoutWaitTerminate > 0) and (FFirstTick+FTimeoutWaitTerminate <= FLastTick)) then begin
    FillChar(FProcessInfo,SizeOf(FProcessInfo),0);
    FTimer.Enabled := False;
    Finished;
  end;
end;

procedure TLauncher.Launch;
var
  StartupInfo: TStartupInfo;
  ShellExecuteInfo: TShellExecuteInfo;
  dwCreationFlags: DWORD;
  Success: BOOL;
  PWorkDir, lpEnv: PChar;
  I: TrStartupFlag;
  J: TrFillAttribute;
  K: TrCreationFlag;
begin
  if FWorkDir = '' then PWorkDir := NIL
                   else PWorkDir := PChar(FWorkDir);

  if FLaunchMethod = lmCreateProcess then begin
    dwCreationFlags := rPriorityClasses[FPriority];
    for K := Low(TrCreationFlag) to High(TrCreationFlag) do
      if K in FCreationFlags then dwCreationFlags := dwCreationFlags or rCreationFlags[K];
    if FEnvironment.Text = '' then lpEnv := NIL
                              else lpEnv := PChar(ReplaceStr(FEnvironment.Text,#13#10,#0)+#0);

    FillChar(StartupInfo,SizeOf(StartupInfo),0);
    with StartupInfo do begin
      cb := SizeOf(StartupInfo);
      if FDesktop = '' then lpDesktop := NIL
                       else lpDesktop := PChar(FDesktop);
      if FConsoleTitle = '' then lpTitle := NIL
                            else lpTitle := PChar(FConsoleTitle);
      dwX := FWindowLeft;
      dwY := FWindowTop;
      dwXSize := FWindowWidth;
      dwYSize := FWindowHeight;
      dwXCountChars := FConsoleCharsX;
      dwYCountChars := FConsoleCharsY;
      for J := Low(TrFillAttribute) to High(TrFillAttribute) do
        if J in FConsoleFillAttributes then
          dwFillAttribute := dwFillAttribute or rFillAttributes[J];
      for I := Low(TrStartupFlag) to High(TrStartupFlag) do
        if I in FStartupFlags then dwFlags := dwFlags or rStartupFlags[I];
      wShowWindow := ShowWindowModes[FShowMode];
    end;

    if Assigned(FBeforeLaunch) then FBeforeLaunch(Self,StartupInfo,ShellExecuteInfo);
    Success := CreateProcess(NIL, PChar(FFullProgramName+' '+FParameters),
                             NIL, NIL, False, dwCreationFlags, lpEnv,
                             PWorkDir, StartupInfo, FProcessInfo);
  end else begin  //--- ShellExecute
    FillChar(StartupInfo,SizeOf(StartupInfo),0);
    with ShellExecuteInfo do begin
      cbSize := SizeOf(ShellExecuteInfo);
      fMask := SEE_MASK_NOCLOSEPROCESS;
      Wnd := Application.Handle;
      if FShellOperation = '' then lpVerb := NIL
                              else lpVerb := PChar(FShellOperation);
      lpFile := PChar(FFullProgramName);
      lpParameters := PChar(FParameters);
      lpDirectory := PWorkDir;
      nShow := ShowWindowModes[FShowMode];
    end;

    if Assigned(FBeforeLaunch) then FBeforeLaunch(Self,StartupInfo,ShellExecuteInfo);
    Success := ShellExecuteEx(@ShellExecuteInfo);
    FProcessInfo.hProcess := ShellExecuteInfo.hProcess;
  end;

  if not Success then RaiseError(GetLastError)
  else begin
    if FTimeoutWaitInit > 0 then
      case WaitForInputIdle(FProcessInfo.hProcess,FTimeoutWaitInit) of
        0: ;
        WAIT_TIMEOUT: if Assigned(FOnTimeoutElapsed) then FOnTimeoutElapsed(Self,True);
      else
        RaiseError(GetLastError);
      end;
    if (FTimeoutWaitTerminate > 0) or Assigned(FOnFinished) then begin
      FFirstTick := GetTickCount;
      FLastTick := 0;
      if not Assigned(FTimer) then
        FTimer := TTimer.Create(Self);
      with FTimer do begin
        Interval := FTimerInterval;
        OnTimer := TimerExpired;
      end;
    end;
  end;
end;

end.
