//==============================================
//       errormes.pas
//
//         Delphi.
//       .
//
//      Copyright 1998-2000 Polaris Software
//      http://members.xoom.com/PolarisSoft
//      mailto: PolarisLib@mail.ru
//==============================================
unit ErrorMes;

{$I POLARIS.INC}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs, rUtils,
  ExtCtrls, StdCtrls, Buttons, IniFiles, StrUtils, rLabel, rConst;

type
  TErrorHandler = class;
  TrEHProc = procedure(E: Exception; Handler: TErrorHandler);
  TrEH = record
    ErrorName: string;
    Proc: TrEHProc;
  end;
  TrpEH = ^TrEH;

  TrFormErrorMes = class(TForm)
    MError: TMemo;
    BitBtnOk: TBitBtn;
    Image1: TImage;
    LError: TLabel;
    BitBtnAll: TBitBtn;
    LMessage: TrLabel;
    procedure FormCreate(Sender: TObject);
    procedure BitBtnAllClick(Sender: TObject);
    procedure FormActivate(Sender: TObject);
  private
    { Private declarations }
    OriginalHeight: Integer;
  public
    { Public declarations }
  end;

  EAnyError = class(Exception)
    FPreMessage: string;
    FDescription: string;
  public
    constructor CreateAny(PreMsg, Msg, Desc: string);
    property PreMessage: string read FPreMessage;
    property Description: string read FDescription;
  end;

  TrFileLocation = (flDefault, flProgram);

  TErrorHandler = class(TComponent)
  private
    FErrorBegins: TStrings;
    FConsts:      TStrings;
    FIniLocation: TrFileLocation;
    FLogFile:     string;
    FLogLocation: TrFileLocation;
    FUserName:    string;
    FCaption:     TCaption;
    FTitleMessage: TCaption;
    FMessage:     TCaption;
    FDescription: TStrings;
    FSaveLog:     Boolean;
    function GetIniFile: string;
    procedure SetIniFile(Value: string);
    procedure SetIniLocation(Value: TrFileLocation);
    procedure SetMessage(Value: TCaption);
    procedure SetDescription(Value: TStrings);
  protected
    { Protected declarations }
  public
    FIniFile:     TIniFile;
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Execute;
    procedure ShowMessage(aCaption,aTitleMes,aMes,aDesc: TCaption);
    function  ConvertMessage(mes: string): string;
    procedure AnalyzeException(E: Exception);
    procedure AppException(Sender: TObject; E: Exception);
    procedure SavingLog;
  published
    property Caption: TCaption     read FCaption     write FCaption;
    property IniFile: string       read GetIniFile   write SetIniFile;
    property IniLocation: TrFileLocation read FIniLocation write SetIniLocation;
    property LogFile: string       read FLogFile     write FLogFile;
    property LogLocation: TrFileLocation read FLogLocation write FLogLocation;
    property UserName: string      read FUserName    write FUserName;
    property TitleMessage: TCaption read FTitleMessage  write FTitleMessage;
    property Message: TCaption     read FMessage     write SetMessage;
    property Description: TStrings read FDescription write SetDescription;
    property SaveLog: Boolean      read FSaveLog     write FSaveLog;
  end;

procedure AddError(Name: string; Proc: TrEHProc);
procedure DeleteError(Name: string);

procedure AnyErrorEHProc(E: Exception; Handler: TErrorHandler);
procedure Win32ErrorEHProc(E: Exception; Handler: TErrorHandler);

var
  rFormErrorMes: TrFormErrorMes;
  RErrors: TList;
  i: Integer;

implementation

{$R *.DFM}

procedure AddError(Name: string; Proc: TrEHProc);
var
  ARecord: TrpEH;
begin
  New(ARecord);
  ARecord^.ErrorName := Name;
  ARecord^.Proc := Proc;
  RErrors.Add(ARecord);
end;

procedure DeleteError(Name: string);
var
  i: Integer;
begin
  for i := 0 to RErrors.Count - 1 do
    if CompareText(TrpEH(RErrors.Items[i])^.ErrorName, Name)=0 then begin
      Dispose(RErrors.Items[i]);
      RErrors.Delete(i);
      exit;
    end;
end;

procedure Win32ErrorEHProc(E: Exception; Handler: TErrorHandler);
begin
  with Handler do begin
    TitleMessage := srWinErrorTitle;
    Message := E.Message;
    Description.Text := Format(srWinErrorDesc,[GetLastError,SysErrorMessage(GetLastError)]);
    SaveLog := Description.Text <> '';
  end;
end;

{ EAnyError }

procedure AnyErrorEHProc(E: Exception; Handler: TErrorHandler);
begin
  with Handler do begin
    TitleMessage := ConvertCodes(EAnyError(E).PreMessage);
    Message := ConvertCodes(E.Message);
    Description.Text := ConvertCodes(EAnyError(E).Description);
    SaveLog := Description.Text <> '';
  end;
end;

constructor EAnyError.CreateAny;
begin
  inherited Create(Msg);
  FPreMessage := PreMsg;
  FDescription := Desc;
end;

{ TErrorHandler }

constructor TErrorHandler.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FDescription := TStringList.Create;
  FErrorBegins := TStringList.Create;
  FConsts := TStringList.Create;
end;

destructor TErrorHandler.Destroy;
begin
  inherited Destroy;
  FDescription.Free;
  if Assigned(FIniFile) then FIniFile.Free;
  FErrorBegins.Free;
  FConsts.Free;
end;

function TErrorHandler.GetIniFile;
begin
  if Assigned(FIniFile) then Result := FIniFile.FileName
  else Result := '';
end;

procedure TErrorHandler.SetIniFile(Value: string);
begin
  if Assigned(FIniFile) then begin
    FIniFile.Free;
    FIniFile := nil;
  end;
  if (Trim(Value) <> '') then begin
    if (ExtractFilePath(Value) = '') and (FIniLocation = flProgram) then
      Value := ExtractFilePath(Application.ExeName)+Value;
    FIniFile := TIniFile.Create(Value);
  end;
end;

procedure TErrorHandler.SetIniLocation(Value: TrFileLocation);
begin
  if FIniLocation <> Value then begin
    FIniLocation := Value;
    SetIniFile(FIniFile.FileName);
  end;
end;

procedure TErrorHandler.SetMessage(Value: TCaption);
begin
  FMessage := Value;
end;

procedure TErrorHandler.SetDescription(Value: TStrings);
begin
  FDescription.Assign(Value);
end;

procedure TErrorHandler.ShowMessage;
begin
  FCaption := aCaption;
  FTitleMessage := aTitleMes;
  FMessage := aMes;
  FDescription.Clear;
  FDescription.Add(aDesc);
  Execute;
end;

procedure TErrorHandler.Execute;
begin
  MessageBeep(MB_ICONHAND);
  with TrFormErrorMes.Create(Application) do
  try
    Caption := FCaption;
    LError.Caption := FTitleMessage;
    LMessage.Caption := FMessage;
    MError.Lines.Assign(FDescription);
    ShowModal;
  finally
    Free;
  end;
end;

function TErrorHandler.ConvertMessage(mes: string): string;
var
  j,k,l: Integer;
  b: Boolean;
  Params: TStrings;

  function ConvertMes(Source: string): string;
  var
    i1,i2,j1,j2: Integer;
    s1,s2: string;
  begin
    Result := ConvertCodes(Source);
    // processing message including SMESSAGE
    i1 := Pos(srEHMessage,AnsiUpperCase(Result));
    if i1 <> 0 then begin
      s1 := Copy(Result,1,i1-1);
      if s1 <> '' then j1 := Pos(s1,mes) else j1 := 0;
      if (j1 <> 0) or (s1 = '') then begin
        s2 := Copy(Result,i1+Length(srEHMessage),Length(Result));
        if s2 <> '' then j2 := Pos(s2,Copy(mes,j1+i1-1,Length(mes)))
        else j2 := Length(mes)+1;
        if j2 <> 0 then Result := Copy(mes,j1+Length(s1),j2-1)
        else Result := mes;
      end else Result := mes;
    end;
    // insert constants
    i1 := 1;
    j2 := 0;
    repeat
      j1 := Pos('<',AnsiUpperCase(Copy(Result,i1,Length(Result))));
      if j1 <> 0 then begin
        j2 := Pos('>',AnsiUpperCase(Copy(Result,i1,Length(Result))));
        if j2 <> 0 then begin
          i2 := FConsts.IndexOf(Copy(Result,i1+j1,j2-j1-1));
          if i2 <> -1 then begin
            Result := Stuff(Result,i1+j1-1,j2-j1+1,
              ConvertMes(FIniFile.ReadString(srEHConsts,FConsts.Strings[i2],'<'+FConsts.Strings[i2]+'>')));
            i1 := 1;
          end else i1 := j1+1;
        end;
      end;
    until (j1 = 0) or (j2 = 0);
  end;

begin
  Result := '';
  with FErrorBegins do begin
    for j:=0 to Count-1 do         { loop by ini-sections }
      if AnsiCompareText(ConvertCodes(Strings[j]),
            Copy(mes,1,Length(ConvertCodes(Strings[j]))))=0 then begin
        Params := TStringList.Create;
        try
          FIniFile.ReadSection(Strings[j],Params);
          for k:=0 to Params.Count-1 do begin  { loop by keys in section }
            b := False;
            for l:=1 to WordCount(Params.Strings[k],[';']) do begin { loop by words in key }
              b := Pos(AnsiUpperCase(ConvertCodes(ExtractWord(l,Params.Strings[k],[';']))),
                       AnsiUpperCase(mes))<>0;
              if not b then break;
            end;
            if b then begin
              Result := ConvertMes(FIniFile.ReadString(Strings[j],Params.Strings[k],mes));
              FSaveLog := FIniFile.ReadBool(Strings[j],srEHLogging,FSaveLog);
              break;
            end;
          end;
          if Result = '' then begin
            Result := ConvertMes(FIniFile.ReadString(Strings[j],srEHDefault,mes));
            FSaveLog := FIniFile.ReadBool(Strings[j],srEHLogging,FSaveLog);
          end;
        finally
          Params.Free;
        end;
      end;
  end;
end;

procedure TErrorHandler.AnalyzeException(E: Exception);
var
  i: Integer;
begin
  FMessage := '';
  FDescription.Clear;
  FSaveLog := False;

  if Assigned(FIniFile) then begin
    FIniFile.ReadSections(FErrorBegins);
    i := FErrorBegins.IndexOf(srEHConsts);
    if i <> -1 then begin
      FErrorBegins.Delete(i);
      FIniFile.ReadSection(srEHConsts,FConsts);
    end else FConsts.Clear;
  end;

  for i:=0 to RErrors.Count-1 do
    if AnsiCompareText(TrpEH(RErrors.Items[i])^.ErrorName, E.ClassName) = 0 then begin
      TrpEH(RErrors.Items[i])^.Proc(E, Self);
      exit;
    end;

  // unknown error
  FTitleMessage := srUnkErrorTitle;
  FMessage := ConvertCodes(Format(srUnkErrorMes,[E.Message]));
  FDescription.Text := Format(srUnkErrorDesc,[E.ClassName,E.Message]);
  FSaveLog := True;
end;

procedure TErrorHandler.AppException(Sender: TObject; E: Exception);
begin
  FCaption := Application.Title;
  AnalyzeException(E);
  if FSaveLog then SavingLog;
  Execute;
end;

procedure TErrorHandler.SavingLog;
var
  F: Text;
  LogName: TFileName;
begin
  if Trim(FLogFile) = '' then exit;
  if (ExtractFilePath(FLogFile) = '') and (FLogLocation = flProgram) then
    LogName := ExtractFilePath(Application.ExeName)+FLogFile
  else
    LogName := FLogFile;
  AssignFile(F,LogName);
  if FileExists(LogName) then Append(F) else Rewrite(F);
  try
    WriteLn(F,DateTimeToStr(Now), srEHUser, FUserName, srEHProgram, Application.ExeName);
    if Assigned(Screen.ActiveForm) then
      with Screen.ActiveForm do begin
        WriteLn(F,srEHForm,Name,', ',Caption);
        if Assigned(ActiveControl) then
          WriteLn(F,srEHControl,ActiveControl.Name);
      end;
    if FDescription.Text <> '' then WriteLn(F,FDescription.Text)
    else WriteLn(F,FMessage);
    Flush(F);
  finally
    CloseFile(F);
  end;
end;

{ TrFormErrorMes }

procedure TrFormErrorMes.FormCreate(Sender: TObject);
begin
  HelpFile   := srHelpFile;
  Image1.Picture.Icon.Handle := LoadIcon(0,IDI_HAND);
  OriginalHeight := Height;
end;

procedure TrFormErrorMes.BitBtnAllClick(Sender: TObject);
begin
  with MError do begin
    Visible := not Visible;
    if Visible then begin
      Self.Height := OriginalHeight;
      BitBtnAll.Caption := srEHSummary;
      BitBtnAll.HelpContext := 10303;
    end else begin
      Self.Height := Self.Height-(Self.ClientHeight-Top);
      BitBtnAll.Caption := srEHDetail;
      BitBtnAll.HelpContext := 10302;
    end;
  end;
end;

procedure TrFormErrorMes.FormActivate(Sender: TObject);
begin
  BitBtnAll.Enabled := (MError.Text <> '');
  if not MError.Visible then
    Height := Height-(ClientHeight-MError.Top);
end;

procedure DisposeErrors;
var
  i: Integer;
begin
  for i := 0 to RErrors.Count - 1 do Dispose(RErrors.Items[i]);
end;

initialization
  RErrors := TList.Create;
  AddError('EAnyError',AnyErrorEHProc);
  AddError('EWin32Error',Win32ErrorEHProc);
finalization
  DisposeErrors;
  RErrors.Free;
end.
