//==============================================
//       dictitem.pas
//
//         Delphi.
//      ,   TDictionaryMgr.
//
//      Copyright 1998-2000 Polaris Software
//      http://members.xoom.com/PolarisSoft
//      mailto: PolarisLib@mail.ru
//==============================================
unit DictItem;

{$I POLARIS.INC}

interface

uses
  Windows,
  Messages,
  SysUtils,
  Classes,
  Graphics,
  Controls,
  Dialogs,
  Menus,
  StdCtrls,
  ToolWin,
  ComCtrls,
  rDBConst,
  BDE,
  DB,
  DBTables;

type
  TImageIndex = Integer;
  TDictionaryOwner = TTreeView;
  {$IFNDEF POLARIS_D3}
  TDBDataSet = TDataSet;
  {$ENDIF}
  TDictionaryItem   = class;

  ErDictionaryError = class(Exception);

{ TDictionaryMgr }

  TDictItemChangeEvent = procedure(Item: TDictionaryItem) of object;
{
  TActiveItemChangeEvent = procedure(Item: TDictionaryItem;
                                     Node: TTreeNode) of object;
  TDictChangedEvent = procedure(Sender: TObject;
                                Item: TDictionaryItem;
                                Node: TTreeNode) of object;
  TDictExpandedEvent = procedure(Sender: TObject;
                                 Item: TDictionaryItem;
                                 Node: TTreeNode) of object;
}

  TDictionaryMgr = class(TComponent)
  private
    FOldDataSet       : TDataSet;
    FMgrOwner         : TDictionaryOwner;
    FActiveItem       : TDictionaryItem;
    FNodeProxies      : TList;
    FBeforeItemChange : TDictItemChangeEvent;
    FAfterItemChange  : TDictItemChangeEvent;
//    FActiveItemChange : TActiveItemChangeEvent;
    FSOnChanged       : TTVChangedEvent;
    FSOnDeletion      : TTVExpandedEvent;
    FSOnExpanding     : TTVExpandingEvent;

    FSGetImageIndex   : TTVExpandedEvent;
    FSGetSelectedIndex: TTVExpandedEvent;

    FDataSource       : TDataSource;
    FActiveNode       : TTreeNode;

    function GetNodeCount: Integer;
    function GetCount: Integer;
    function GetIndex: Integer;
    function GetDictionaryItem(Index: Integer): TDictionaryItem;
    function FindFreeNode: TTreeNode;
    procedure SetNodeProxies(Value: TList);
    procedure SetIndex(Value: Integer);
    procedure SetActiveItem(Value: TDictionaryItem);
    procedure SetMgrOwner(Value: TDictionaryOwner);
    procedure SetDataSource(Value: TDataSource);

    procedure RemoveProxy(Value: TDictionaryItem);
    procedure AddProxy(Value: TDictionaryItem);
    procedure DestroyProxies;
    procedure SyncTreeView;
  protected
    procedure Loaded; override;
    procedure Notification(AComponent: TComponent;
                           AOperation: TOperation); override;
    procedure GetChildren(Proc: TGetChildProc {$IFDEF POLARIS_D3};
                          Root: TComponent {$ENDIF}); override;

    procedure Changed(Sender: TObject; Node: TTreeNode);
    procedure Deleted(Sender: TObject; Node: TTreeNode);
    procedure Expanding(Sender: TObject; Node: TTreeNode; var AllowExpansion: Boolean);

    procedure GetImageIndex(Sender: TObject; Node: TTreeNode);
    procedure GetSelectedIndex(Sender: TObject; Node: TTreeNode);

    function IsRootNode(Node: TTreeNode): Boolean;

    function IndexByNode(Node: TTreeNode): Integer;
    procedure ActiveChanged;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure SetItemDataSet(Value: TDataSet);
    //(Value: TDictionaryItem);
    function IndexByItem(Item: TDictionaryItem): Integer;
    function ItemByNode(Node: TTreeNode): TDictionaryItem;
    procedure Resync;
    function GetUniqueName(Component: TComponent): String;
    function NodeByName(Value: String): TTreeNode;
    procedure RefreshItem(Item: TDictionaryItem);
    procedure Refresh;

    property Count: Integer read GetCount;
    property NodeCount: Integer read GetNodeCount;
    property Item[Index: Integer]: TDictionaryItem read GetDictionaryItem;
    property Index: Integer read GetIndex write SetIndex;
    property ActiveItem: TDictionaryItem read FActiveItem
             write SetActiveItem;
  published
  {PROPERTIES}
    property MgrOwner: TDictionaryOwner read FMgrOwner
             write SetMgrOwner;
    property NodeProxies: TList read FNodeProxies
             write SetNodeProxies;
    property DataSource: TDataSource read FDataSource
             write SetDataSource;
  {EVENTS}
    property BeforeItemChange: TDictItemChangeEvent read FBeforeItemChange
             write FBeforeItemChange;
    property AfterItemChange: TDictItemChangeEvent read FAfterItemChange
             write FAfterItemChange;
{
    property ActiveItemChange: TActiveItemChangeEvent read FActiveItemChange
             write FActiveItemChange;
}

  end;

{ TCloneDBDataSet }
  TCloneDBDataSet = class(TDBDataSet)
  protected
    function CreateHandle: HDBICur; override;
    procedure OpenCursor {$IFDEF POLARIS_D3}(InfoQuery: Boolean){$ENDIF}; override;
  end;

{ TDictionaryItem }
  TSetDictFilterEvent = procedure ( DataSet:     TDataSet;
                                    ParentField: TField;
                                    OwnerField:  TField;
                                    Node:        TTreeNode;
                                    var Filter:  String) of object;
  TGetDisplayStringEvent = procedure( DataSet:      TDataSet;
                                      DisplayField: TField;
                                      var Text:     String) of object;

  TSetConnectedFilterEvent = procedure ( DataSet:        TDataSet;
                                         ConnectedField: TField;
                                         SourceField:    TField;
                                         var Filter:     String) of object;

  TSetImageIndexEvent = procedure( DataSet: TDataSet;
                                   Node: TTreeNode) of object;

  TDictionaryItem = class(TComponent)
  private
    { Private declarations }
    FItemOwner        : TDictionaryMgr;
    FOriginalFilter   : String;
    FOriginalFiltered : Boolean;
    FOriginalOnFilter : TFilterRecordEvent;
    FOriginalAfterOpen: TDataSetNotifyEvent;
    FOriginalBeforeClose: TDataSetNotifyEvent;

    FNodeName         : String;
    FDictDataSet      : TDBDataSet;
    FClon             : TCloneDBDataSet;
    FParentFieldName  : String;
    FOwnerFieldName   : String;
    FOwnerFilter       : String;
    FDisplayFieldName : String;
    FSourceFieldName  : String;
    FMaxLevel         : Integer;
    //*
    FRootDataSet      : TDataSet;

    FConnectedDataSet   : TDataSet;
    FConnectedFieldName : String;

    FOnSetDictFilter     : TSetDictFilterEvent;
    FOnGetDisplayField   : TGetDisplayStringEvent;
    FOnSetConnectedFilter: TSetConnectedFilterEvent;
    FOnSetNodeImageIndex :TSetImageIndexEvent;

    FNodeImageIndex   : TImageIndex;
    FNodeSelectedIndex: TImageIndex;
    FNodeStateIndex   : TImageIndex;

    procedure SetNodeImageIndex(Value: TImageIndex);
    procedure SetNodeSelectedIndex(Value: TImageIndex);
    procedure SetNodeStateIndex(Value: TImageIndex);

    procedure SetNodeName(Value: String);
    function GetDataBase: TDataBase;
    function GetParentField: TField;
    function GetOwnerField: TField;
    function GetSourceField: TField;
    function GetDisplayField: TField;
    function GetConnectedField: TField;

    procedure SetDictionaryMgr(Value: TDictionaryMgr);
    procedure SetDictDataSet(Value: TDBDataSet);
    procedure SetConnectedDataSet(Value: TDataSet);
    procedure SetParentFieldName(Value: String);
    procedure SetOwnerFieldName(Value: String);
    procedure SetDisplayFieldName(Value: String);
    procedure SetSourceFieldName(Value: String);
    procedure SetMaxLevel(Value: Integer);
    procedure SetConnectedFieldName(Value: String);
    //*
    procedure SetRootDataSet(Value: TDataSet);

  protected
    { Protected declarations }

    procedure Notification(AComponent: TComponent;
                           AOperation: TOperation); override;
    procedure DictAfterOpen(DataSet: TDataSet);
    procedure DictBeforeClose(DataSet: TDataSet);
    function GetDictFilter(Node: TTreeNode) :String;
    function GetDisplayText(BookMark: TBookMark):String;
    function GetConnectedFilter(ConnectedFieldName: String;
                                Node: TTreeNode): String;
    function GetTreeView: TTreeView;

    function DictFilter(Node: TTreeNode): Boolean;

    procedure DoClearNode(Node: TTreeNode);

    procedure ExpandNode(Node: TTreeNode);

    procedure FreeDictionary;
    function FindNode(Node: TTreeNode; BookMark: TBookmark): TTreeNode;
    function FindNodeByName(Node: TTreeNode; Text: String): TTreeNode;
    function CheckField(DataSet: TDataSet; FieldName: String): TField;
    function CheckOpen: Boolean;
    procedure GotoBookMark(Node: TTreeNode);
    procedure FreeBookMark(Node: TTreeNode);
    procedure ReCreateClone;

    procedure ReLookNode(Node: TTreeNode);

    function AddNewNode(Node: TTreeNode; BookMark: TBookMark; Text: String): TTreeNode;
    procedure ReadState(Reader: TReader); override;
    function GetRootNode: TTreeNode;
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function HasParent: Boolean; override;
    function IsRootNode(Node: TTreeNode): Boolean;
    procedure RefreshNode;

    function IsLinkedTo(DataSet: TDataSet): Boolean;
    function IsConnectedTo(DataSet: TDataSet): Boolean;
    //*
    function IsRootTo(DataSet: TDataSet): Boolean;

    property TreeView: TTreeView read GetTreeView;
    property ParentField: TField read GetParentField;
    property OwnerField: TField read GetOwnerField;
    property DisplayField: TField read GetDisplayField;
    property ConnectedField: TField read GetConnectedField;
    property SourceField: TField read GetSourceField;
    property DataBase: TDataBase read GetDataBase;
    property ItemOwner: TDictionaryMgr read FItemOwner write SetDictionaryMgr;
    property RootNode: TTreeNode read GetRootNode;
  published
    { Published declarations }
    {Property}
    property NodeName: String read FNodeName
             write SetNodeName;
    property DataSet: TDBDataSet read FDictDataSet write SetDictDataSet;
//*
    property RootDataSet: TDataSet read FRootDataSet write SetRootDataSet;
//*
    property ParentFieldName: String read FParentFieldName
             write SetParentFieldName;
    property OwnerFieldName: String read FOwnerFieldName
             write SetOwnerFieldName;
    property DisplayFieldName: String read FDisplayFieldName
             write SetDisplayFieldName;
    property OwnerFilter: String read FOwnerFilter
             write FOwnerFilter;
    property SourceFieldName: String read FSourceFieldName
             write SetSourceFieldName;
    property MaxLevel: Integer read FMaxLevel
             write SetMaxLevel default 0;
    property ConnectedDataSet: TDataSet read FConnectedDataSet
             write SetConnectedDataSet;
    property ConnectedFieldName: String read FConnectedFieldName
             write SetConnectedFieldName;

    property NodeImageIndex: TImageIndex read FNodeImageIndex
             write SetNodeImageIndex default 0;
    property NodeSelectedIndex: TImageIndex read FNodeSelectedIndex
             write SetNodeSelectedIndex default 0;
    property NodeStateIndex: TImageIndex read FNodeStateIndex
             write SetNodeStateIndex default -1;

    {Events}
    property OnSetDictFilter: TSetDictFilterEvent read FOnSetDictFilter
                                                  write FOnSetDictFilter;
    property OnGetDisplayString: TGetDisplayStringEvent read FOnGetDisplayField
                                                       write FOnGetDisplayField;
    property OnSetConnectedFilter: TSetConnectedFilterEvent read FOnSetConnectedFilter
                                                              write FOnSetConnectedFilter;

    property OnSetNodeImageIndex: TSetImageIndexEvent read FOnSetNodeImageIndex
                                                      write FOnSetNodeImageIndex;
  end;

implementation
uses
  Forms;

function CheckNode(Node: TTreeNode): Boolean;
var
  TempNode: TTReeNode;
begin
  Result := False;
  if Node <> nil
  then begin
    TempNode := Node.GetFirstChild;
    Result := (TempNode <> nil) and (TempNode.Text = #0) and (TempNode.Data = nil)
  end
end;

function TCloneDBDataSet.CreateHandle: HDBICur;
begin
  if (Owner <> nil) and (Owner is TDictionaryItem) and (TDictionaryItem(Owner).DataSet <> nil)
//  (Owner is TDBDataSet)
//  then Check(DBICloneCursor(TDBDataSet(Owner).Handle, True, False, Result))
  then Check(DBICloneCursor(TDictionaryItem(Owner).DataSet.Handle, True, False, Result))
  else Result := nil;
end;

procedure TCloneDBDataSet.OpenCursor;
begin
  if (Owner <> nil) and (Owner is TDictionaryItem) and (TDictionaryItem(Owner).DataSet <> nil)
// (Owner is TDBDataSet)
  then begin
//    DataBaseName := TDBDataSet(Owner).DataBaseName;
//    SessionName  := TDBDataSet(Owner).SessionName;
    DataBaseName := TDictionaryItem(Owner).DataSet.DataBaseName;
    SessionName  := TDictionaryItem(Owner).DataSet.SessionName;
  end;
  inherited OpenCursor( InfoQuery);
end;

{ TDictionaryItem }
const
  HasDelimetedField: set of TFieldType = [ftString, ftDate, ftTime, ftDateTime];
  DelimeterStr: array[Boolean] of String[1] = ('',#39);

constructor TDictionaryItem.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FItemOwner := nil;
  FOriginalFilter   := '';
  FOriginalFiltered := False;
  FOriginalOnFilter := nil;

  FOwnerFilter      := '';
  FParentFieldName  := '';
  FOwnerFieldName   := '';
  FDisplayFieldName := '';
  FMaxLevel         := 0;

  FNodeImageIndex   := 0;
  FNodeSelectedIndex:= 0;
  FNodeStateIndex   := -1;

  FNodeName         := '';
  FDictDataSet      := nil;
  FRootDataSet      := nil;
  FClon             := nil;
  FConnectedDataSet   := nil;
  FConnectedFieldName := '';

  FOnSetDictFilter     := nil;
  FOnGetDisplayField   := nil;
  FOnSetNodeImageIndex := nil;
  FOnSetConnectedFilter:= nil;
  FOriginalAfterOpen   := nil;
end;

destructor TDictionaryItem.Destroy;
begin
  FreeDictionary;
  if Assigned(FItemOwner)
  then FItemOwner.RemoveProxy(Self);
  FConnectedDataSet := nil;
  FRootDataSet := nil;
  if FDictDataSet <> nil
  then begin
    FDictDataSet.AfterOpen := FOriginalAfterOpen;
    FDictDataSet.BeforeOpen := FOriginalBeforeClose;
    FDictDataSet := nil;
  end;
  inherited Destroy;
end;

procedure TDictionaryItem.Notification(AComponent: TComponent;
                                       AOperation: TOperation);
begin
  inherited Notification(AComponent,AOperation);
  if AOperation = opRemove
  then begin
    if AComponent = FItemOwner then ItemOwner := nil;
    if AComponent = FDictDataSet then DataSet := nil;
    if AComponent = FRootDataSet then RootDataSet := nil;
    if AComponent = FConnectedDataSet then ConnectedDataSet := nil;
  end
end;

procedure TDictionaryItem.DictAfterOpen(DataSet: TDataSet);
begin
  if not ((csLoading in ComponentState) or (csDesigning in ComponentState))
  then begin
    if Assigned(FOriginalAfterOpen)
    then FOriginalAfterOpen(DataSet);
    ReCreateClone;
  end
end;

procedure TDictionaryItem.DictBeforeClose(DataSet: TDataSet);
begin
  if not ((csLoading in ComponentState) or (csDesigning in ComponentState))
  then begin
    if Assigned(FOriginalBeforeClose)
    then FOriginalBeforeClose(DataSet);
    if FClon <> nil
    then begin
      FClon.Free;
      FClon := nil;
    end;
  end;
end;


procedure TDictionaryItem.SetMaxLevel(Value: Integer);
begin
  if (Value > -2) and (Value <=32767)
  then begin
    if Value <> FMaxLevel then
      FMaxLevel := Value;
  end
  else raise ErDictionaryError.Create(trdMaxLevelError);
end;

function TDictionaryItem.CheckOpen: Boolean;
begin
  Result := FClon.Active;
end;

function TDictionaryItem.GetDataBase: TDataBase;
begin
  Result := FClon.DataBase;
end;

procedure TDictionaryItem.SetDictDataSet(Value: TDBDataSet);
begin
  if FDictDataSet <> Value then
  begin
    if (FClon <> nil)
    then begin
      FClon.Free;
      FClon := nil;
    end;
    FDictDataSet := nil;
    DoClearNode(RootNode);
    FDictDataSet := Value;
    if FDictDataSet <> nil
    then begin
      FOriginalAfterOpen := FDictDataSet.AfterOpen;
      FOriginalBeforeClose := FDictDataSet.BeforeOpen;
      FDictDataSet.AfterOpen := DictAfterOpen;
      FDictDataSet.BeforeOpen := DictBeforeClose;
      if FDictDataSet.Active
      then ReCreateClone
      else
        if not ((csLoading in ComponentState) or (csDesigning in ComponentState))
        then FDictDataSet.Open;
    end;
    if RootNode.Expanded and CheckNode(RootNode)
    then ExpandNode(RootNode);
    if Value <> nil then Value.FreeNotification(Self);
  end;
end;

function TDictionaryItem.HasParent: Boolean;
begin
  Result := True;
end;

function TDictionaryItem.GetTreeView: TTreeView;
begin
  Result := nil;
  if FItemOwner <> nil
  then Result := TTreeView(FItemOwner.MgrOwner);
end;

procedure TDictionaryItem.SetNodeName(Value: String);
//const LevelValue: Array[Boolean] of Integer = (0,1);
begin
  if Value <> FNodeName
  then begin
    if FItemOwner <> nil
    then
     if FItemOwner.NodeByName(Value) <>  nil
     then begin
       FreeDictionary;
       FNodeName := Value;
       if not ((csLoading in ComponentState) or (csDesigning in ComponentState))
       then begin
         ExpandNode(RootNode);
         ReLookNode(RootNode);
       end
     end
     else begin
       FNodeName := '';
       FMaxLevel := 0
     end
    else
      FNodeName := Value;
  end
end;

function TDictionaryItem.GetRootNode: TTreeNode;
begin
  Result := nil;
  if FItemOwner <> nil
  then Result := FItemOwner.NodeByName(FNodeName)
end;

procedure TDictionaryItem.ReadState(Reader: TReader);
begin
  inherited ReadState(Reader);
  if Reader.Parent is TDictionaryMgr
  then ItemOwner := TDictionaryMgr(Reader.Parent);
end;


procedure TDictionaryItem.RefreshNode;
begin
  if Assigned(FDictDataSet)
  then begin
    if FDictDataSet is TTable
    then FClon.Refresh
    else ReCreateClone;
    ExpandNode(RootNode);
  end;
end;

procedure TDictionaryItem.ReCreateClone;
begin
  if csDesigning in ComponentState
  then Exit;
  if Assigned(FClon) then begin
    FClon.Free;
    FClon := nil;
  end;
  if Assigned(FDictDataSet) then begin
    FClon := TCloneDBDataSet.Create(Self);
    FClon.Filtered := False;
//    FClon.Active := True;
    FClon.Open;
    FClon.First;
    if RootNode <> nil
    then begin
      ExpandNode(RootNode);
      ReLookNode(RootNode);
    end;
  end
  else FClon := nil;
end;

function TDictionaryItem.CheckField(DataSet: TDataSet; FieldName: String): TField;
begin
  Result := nil;
  if Assigned(DataSet) and (FieldName <> '') then
    Result := DataSet.FindField(FieldName)
end;

function TDictionaryItem.GetSourceField: TField;
begin
  Result := CheckField( FClon, FSourceFieldName);
end;

function TDictionaryItem.GetParentField: TField;
begin
  Result := CheckField( FClon, FParentFieldName);
end;

function TDictionaryItem.GetDisplayField: TField;
begin
  Result := CheckField( FClon, FDisplayFieldName);
end;

function TDictionaryItem.GetOwnerField: TField;
begin
  Result := CheckField( FClon, FOwnerFieldName);
end;

function TDictionaryItem.GetConnectedField: TField;
begin
  Result := CheckField( FConnectedDataSet, FConnectedFieldName);
end;

procedure TDictionaryItem.FreeBookMark(Node: TTreeNode);
begin
  if (Node <> nil) and (Node.Data <> nil)
  then begin
  {$IFDEF POLARIS_D4}
    FClon.FreeBookMark(TBookMark(Node.Data));
  {$ELSE}
    try
      StrDispose(Node.Data);
    except
    end;
  {$ENDIF}
    Node.Data := nil;
  end;
end;

procedure TDictionaryItem.DoClearNode(Node: TTreeNode);
var
  LastNode,
  NextNode: TTreeNode;
begin
  if Node = nil then Exit;
  LastNode := Node.GetFirstChild;
  while LastNode <> nil do begin
    NextNode := LastNode.GetNextSibling;
    FreeBookMark(LastNode);
    if LastNode.HasChildren
    then DoClearNode(LastNode);
    LastNode.DeleteChildren;
    LastNode := NextNode;
  end;
end;

procedure TDictionaryItem.FreeDictionary;
begin
  if RootNode <> nil
  then DoClearNode(RootNode);
end;

procedure TDictionaryItem.GotoBookMark(Node: TTreeNode);
begin
  if (FClon <> nil) and (Node <> nil) and (not IsRootNode(Node)) and (Node.Data <> nil)
  then
    try
      FClon.GotoBookMark(Node.Data);
    except
    end;
end;

function TDictionaryItem.FindNode(Node: TTreeNode; BookMark: TBookmark): TTreeNode;
begin
  Result := nil;
  while Node <> nil do begin
    if TBookMark(Node.Data) = BookMark then begin
      Result := Node;
      Break;
    end;
    Node := Node.GetNextSibling;
  end;
end;

function TDictionaryItem.FindNodeByName(Node: TTreeNode; Text: String): TTreeNode;
begin
  Result := nil;
  while Node <> nil do begin
    if Node.Text = Text then begin
      GotoBookMark(Node);
      Result := Node;
      Break;
    end;
    Node := Node.GetNextSibling;
  end;
end;

function TDictionaryItem.IsRootNode(Node: TTreeNode): Boolean;
begin
  Result := False;
  if RootNode <> nil then
    Result := RootNode=Node;
end;

function TDictionaryItem.GetDictFilter( Node: TTreeNode) :String;
var
  IsDelim: Boolean;
  f_Parent,
  f_Owner: TField;
begin
  Result := '';
  f_Owner  := nil;
  f_Parent := nil;
  try
    if not((FClon = nil) or
       (FMaxLevel=0)) then
    try
      f_Owner  := OwnerField;
      f_Parent := ParentField;
      if not IsRootNode(Node)
      then GotoBookMark(Node);
//    if Assigned(FOnSetDictFilter) then
//      FOnSetDictFilter( FClon {FDictDataSet}, f_Parent, f_Owner, Node, Result)
//    else
      if IsRootNode(Node)
      then
        if (f_Owner <> nil) and (f_Parent <> nil)
        then Result := '['+FOwnerFieldName+']= null'
        else Result := Trim(FOwnerFilter)
      else
        if f_Owner <> nil then begin
          IsDelim := f_Owner.DataType in HasDelimetedField;
          Result := '['+FOwnerFieldName+']='+DelimeterStr[IsDelim];
          if f_Parent <> nil
          then
            Result := Result + f_Parent.Text+DelimeterStr[IsDelim];
        end;
    except
      Result := '';
      raise;
    end;
  finally
    if Assigned(FOnSetDictFilter) then
      FOnSetDictFilter( FClon {FDictDataSet}, f_Parent, f_Owner, Node, Result)
  end;
end;

function TDictionaryItem.DictFilter( Node: TTreeNode): Boolean;
begin
  if Assigned(FClon) and (FMaxLevel <> 0)
  then FClon.Filter := GetDictFilter(Node);
  Result := True;
end;

function TDictionaryItem.GetDisplayText( BookMark: TBookMark):String;
begin
  Result := '';
  if (RootNode <> nil) and (FClon <> nil) and (BookMark <> nil)
  then begin
    FClon.GotoBookMark(BookMark);
    if Assigned(FOnGetDisplayField)
    then FOnGetDisplayField(FClon, DisplayField, Result)
    else
      if DisplayField <> nil
      then Result := DisplayField.DisplayText
      else
        if SourceField <> nil
        then Result := SourceField.DisplayText
  end
end;

function TDictionaryItem.GetConnectedFilter(ConnectedFieldName: String; Node: TTreeNode): String;
var
  IsDelim: Boolean;
  f_Connected,
  f_Source: TField;
begin
  f_Connected := nil;
  f_Source    := nil;
  Result := '';
  if not ((Node = nil) or IsRootNode(Node)) then begin
//  else begin
    GotoBookMark(Node);
    f_Connected := ConnectedField;
    f_Source    := SourceField;
//    if Assigned(FOnSetConnectedFilter) then
//       FOnSetConnectedFilter( FClon, f_Connected, f_Source, Result)
//    else
    if (f_Connected<>nil) and (f_Source<>nil) then begin
      IsDelim := f_Connected.DataType in HasDelimetedField;
      Result := '['+FConnectedFieldName+']='+DelimeterStr[IsDelim]+f_Source.Text+DelimeterStr[IsDelim];
    end
  end;
  if Assigned(FOnSetConnectedFilter) then
     FOnSetConnectedFilter( FClon, f_Connected, f_Source, Result)
end;

procedure TDictionaryItem.ExpandNode(Node: TTreeNode);
var
  LastNode,
  FirstNode: TTreeNode;
  BM,
  TempBM: TBookMark;
  LastFilter: String;
  DText : String;

begin
  BM := nil;
  if (not Assigned(FClon)) or (Node = nil) or (FItemOwner = nil) or
     ((FMaxLevel > -1) and ((Node.Level+1) > FMaxLevel)) then Exit;
  TreeView.Items.BeginUpdate;
  try
    LastFilter := FClon.Filter;
    try
      DictFilter(Node);
    except
      Exit;
    end;
    if not Node.HasChildren
    then TreeView.Items.AddChildObject(Node, #0, nil);
    FirstNode := Node.GetFirstChild;
    BM := FClon.GetBookMark;
    FClon.Filtered := True;
    while FirstNode <> nil do begin
      LastNode := FirstNode.GetNextSibling;
      if FirstNode.Data = nil
      then begin
          DoClearNode(FirstNode);
          FirstNode.Delete;
      end
      else
        try
          FClon.GotoBookMark(FirstNode.Data);
          FirstNode.Text := GetDisplayText(FirstNode.Data);
        except
          DoClearNode(FirstNode);
          FirstNode.Delete;
        end;
      FirstNode := LastNode;
    end;
    FirstNode := Node.GetFirstChild;
//    FClon.Filtered := True;
    FClon.First;
    while not FClon.EOF do begin
      TempBM := FClon.GetBookMark;
      DText := GetDisplayText(TempBM);
      LastNode := FindNodeByName(FirstNode, DText);
      if LastNode = nil
      then AddNewNode(Node, TempBM, DText )
      else begin
      {$IFDEF POLARIS_D4}
        FreeBookmark(LastNode);
      {$ELSE}
        try
          StrDispose(LastNode.Data);
        except
        end;
      {$ENDIF}
        LastNode.Data := TempBM;
//        if LastNode.HasChildren and LastNode.Expanded
//        then begin
          ExpandNode(LastNode);
          if not FClon.Filtered
          then FClon.Filtered := True;
          if TempBM <> nil
          then FClon.GotoBookMark(TempBM);
//        end;
      end;
      FClon.Next;
    end;
    FClon.Filtered := False;
  finally
    TreeView.Items.EndUpdate;
    FClon.Filter := LastFilter;
    if BM <> nil
    then FClon.GotoBookMark(BM);
  end
end;

procedure TDictionaryItem.SetParentFieldName(Value: String);
begin
  if Value<>FParentFieldName then
    FParentFieldName := Value;
end;

procedure TDictionaryItem.SetOwnerFieldName(Value: String);
begin
  if Value <> FOwnerFieldName
  then FOwnerFieldName := Value;
end;

procedure TDictionaryItem.SetDisplayFieldName(Value: String);
begin
  if Value<>FDisplayFieldName
  then FDisplayFieldName := Value;
end;

procedure TDictionaryItem.SetSourceFieldName(Value: String);
begin
  if Value<>FSourceFieldName
  then FSourceFieldName := Value;
end;

procedure TDictionaryItem.SetConnectedFieldName(Value: String);
begin
  if Value<>FConnectedFieldName
  then FConnectedFieldName := Value;
end;

procedure TDictionaryItem.ReLookNode(Node: TTreeNode);
begin
  if IsRootNode(Node)
  then Node := Node.GetFirstChild;
  while Node <> nil do
    with Node do
    begin
      ImageIndex := Integer(FNodeImageIndex);
      SelectedIndex := Integer(FNodeSelectedIndex);
      StateIndex := Integer(FNodeStateIndex);
      if  Assigned(FOnSetNodeImageIndex)
      then FOnSetNodeImageIndex(FClon, Node);
      Node := Node.GetNextSibling;
    end;
end;

procedure TDictionaryItem.SetNodeImageIndex(Value: TImageIndex);
begin
  if Value <> FNodeImageIndex
  then begin
    FNodeImageIndex := Value;
//    if (FItemOwner <> nil) and (RootNode <> nil)
//    then ReLookNode(RootNode)
  end
end;

procedure TDictionaryItem.SetNodeSelectedIndex(Value: TImageIndex);
begin
  if Value <> FNodeSelectedIndex
  then begin
    FNodeSelectedIndex := Value;
//    if (FItemOwner <> nil) and (RootNode <> nil)
//    then ReLookNode(RootNode)
  end;
end;

procedure TDictionaryItem.SetNodeStateIndex(Value: TImageIndex);
begin
  if FNodeStateIndex <> Value
  then begin
    FNodeStateIndex := Value;
//    if (FItemOwner <> nil) and (RootNode <> nil)
//    then ReLookNode(RootNode)
  end;
end;

procedure TDictionaryItem.SetDictionaryMgr(Value: TDictionaryMgr);
begin
  if FItemOwner <> Value
  then begin
    if Assigned(FItemOwner)
    then FItemOwner.RemoveProxy(Self);
    if Assigned(Value)
    then Value.AddProxy(Self);
  end;
end;

function TDictionaryItem.AddNewNode(Node: TTreeNode; BookMark: TBookMark; Text: String): TTreeNode;
begin
  Result := nil;
  if (FItemOwner <> nil) and (Node <> nil) then
  begin
    Result := TreeView.Items.AddChildObject(Node, Text, BookMark);
    if Assigned(FOnSetNodeImageIndex)
    then FOnSetNodeImageIndex(FClon,Result)
    else begin
      Result.ImageIndex := Integer(FNodeImageIndex);
      Result.SelectedIndex := Integer(FNodeSelectedIndex);
      Result.StateIndex := Integer(FNodeStateIndex);
    end;
    if (FMaxLevel < 0) or ((Result.Level+1) <= FMaxLevel)
    then TreeView.Items.AddChildObject(Result, #0, nil);
  end
end;

procedure TDictionaryItem.SetRootDataSet(Value: TDataSet);
begin
  if Value <> FRootDataSet
  then begin
    FRootDataSet := Value;
    if Value <> nil then Value.FreeNotification(Self);
  end;
end;

procedure TDictionaryItem.SetConnectedDataSet(Value: TDataSet);
begin
  if Value <> FConnectedDataSet
  then begin
    if FConnectedDataSet <> nil
    then begin
      if FOriginalFilter <> ''
      then FConnectedDataSet.Filter := FOriginalFilter;
      if Assigned(FOriginalOnFilter)
      then FConnectedDataSet.OnFilterRecord := FOriginalOnFilter;
      FConnectedDataSet.Filtered := FOriginalFiltered;
    end;
    FOriginalFilter := '';
    FOriginalFiltered := False;
    FOriginalOnFilter := nil;

    if Value <> nil
    then begin
      FOriginalFilter := Value.Filter;
      FOriginalFiltered := Value.Filtered;
      FOriginalOnFilter := Value.OnFilterRecord;
      Value.FreeNotification(Self);
    end;
    FConnectedDataSet := Value
  end
end;

function TDictionaryItem.IsLinkedTo(DataSet: TDataSet): Boolean;
begin
  Result := (DataSet <> nil) and (DataSet = FDictDataSet);
end;

function TDictionaryItem.IsRootTo(DataSet: TDataSet): Boolean;
begin
  Result := (DataSet <> nil) and (DataSet = FRootDataSet);
end;

function TDictionaryItem.IsConnectedTo(DataSet: TDataSet): Boolean;
begin
  Result := (DataSet <> nil) and (DataSet = FConnectedDataSet);
end;

{ TDictionaryMgr }

constructor TDictionaryMgr.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FOldDataSet  := nil;
  FActiveNode  := nil;
  FNodeProxies := TList.Create;
//  FBeforeItemChange := nil;
//  FAfterItemChange  := nil;

  FDataSource  := nil;
  FSOnChanged  := nil;
  FSOnDeletion := nil;
  FSOnExpanding:= nil;

  FSGetImageIndex   := nil;
  FSGetSelectedIndex:= nil;

end;

destructor TDictionaryMgr.Destroy;
begin
  FDataSource := nil;
  FOldDataSet := nil;
  FActiveNode := nil;
  DestroyProxies;
  FNodeProxies.Free;
  inherited Destroy;
end;

procedure TDictionaryMgr.DestroyProxies;
var
  Item: TDictionaryItem;
begin
  while FNodeProxies.Count > 0 do begin
    Item := FNodeProxies.Last;
    RemoveProxy(Item);
    Item.Free;
  end;
end;

procedure TDictionaryMgr.AddProxy(Value: TDictionaryItem);
begin
  FNodeProxies.Add(Value);
  Value.FItemOwner := Self;
  if Value <> nil then Self.FreeNotification(Value);
end;

procedure TDictionaryMgr.RemoveProxy(Value: TDictionaryItem);
begin
  Value.FItemOwner := nil;
  FNodeProxies.Remove(Value);
end;

function TDictionaryMgr.GetDictionaryItem(Index: Integer): TDictionaryItem;
begin
  if Index < Count
  then Result := TDictionaryItem(FNodeProxies.Items[Index])
  else raise ErDictionaryError.CreateFmt(trdActiveItem, [Index, Count]);
end;

function TDictionaryMgr.GetCount: Integer;
begin
  Result := FNodeProxies.Count;
end;

procedure TDictionaryMgr.SetActiveItem(Value: TDictionaryItem);
begin
  if Assigned(FBeforeItemChange)
  then FBeforeItemChange(Value);
  if Assigned(ActiveItem) and Assigned(ActiveItem.ConnectedDataSet)
  then begin
    ActiveItem.ConnectedDataSet.Filter   := '';
    ActiveItem.ConnectedDataSet.Filtered := False;
  end;
  FActiveItem := Value;
  if Value <> nil then Value.FreeNotification(Self);
  ActiveChanged;
  if Assigned(FAfterItemChange)
  then FAfterItemChange(Value);
end;

function TDictionaryMgr.GetIndex: Integer;
begin
  Result := -1;
  if FActiveItem <> nil then
    Result := IndexByItem(FActiveItem);
end;

procedure TDictionaryMgr.SetIndex(Value: Integer);
begin
  if Value < Count
  then Changed(Self, TDictionaryItem(FNodeProxies.Items[Value]).RootNode)
  else raise ErDictionaryError.Create(trdActiveItem);
end;

function TDictionaryMgr.IndexByItem(Item: TDictionaryItem): Integer;
var
  I: Integer;
begin
  Result := -1;
  if Count > 0 then
  for I := 0 to Count-1 do
    if TDictionaryItem(FNodeProxies.Items[I]) = Item then
    begin
      Result := I;
      Break;
    end
end;

procedure TDictionaryMgr.SetMgrOwner(Value: TDictionaryOwner);
begin
  if Value <> FMgrOwner then
  begin
    FMgrOwner := Value;
    if Value <> nil
    then Value.FreeNotification(Self);
    if not (csLoading in ComponentState) then Resync;
  end
end;

function TDictionaryMgr.GetNodeCount: Integer;
var
  Node: TTreeNode;
begin
  Result := 0;
  if Assigned( FMgrOwner) then
  begin
    Node := FMgrOwner.Items[0];
    while Node <> nil do
    begin
      Inc(Result);
      Node := Node.GetNextSibling;
    end
  end
end;

function TDictionaryMgr.GetUniqueName(Component: TComponent): String;
var
  I: Integer;
  Temp: String;
  Comp: TComponent;
  OwnerForm: TForm;
begin
  Result := '';
  OwnerForm := Owner as TForm;
  if (Component <> nil)
  then Temp := Component.ClassName
  else Temp := TDictionaryItem.ClassName;
  if UpCase(Temp[1]) = 'T'
  then System.Delete(Temp,1,1);
  I := 1;
  repeat
    Result := Temp + IntToStr(I);
    Comp := OwnerForm.FindComponent(Result);
    Inc(I);
  until (Comp = nil) or (Comp = Component);
end;

function TDictionaryMgr.IndexByNode(Node: TTreeNode): Integer;
var
  I: Integer;
begin
  Result := -1;
  if (Node <> nil) and  (Count > 0) then
    for I := 0 to Count-1 do
      if Node = Item[I].RootNode then
      begin
        Result := I;
        Exit;
      end
end;

function TDictionaryMgr.FindFreeNode: TTreeNode;
var
  Temp: TTreeNode;
begin
  Result := nil;
  if FMgrOwner <> nil
  then begin
    Temp := FMgrOwner.Items[0];
    while Temp <> nil do
    begin
      if IndexByNode(Temp) = -1 then
      begin
        Result := Temp;
        Exit;
      end;
      Temp := Temp.GetNextSibling
    end
  end
end;

procedure TDictionaryMgr.SyncTreeView;
begin
  if not (csDesigning in ComponentState) then
    if FMgrOwner <> nil then
    begin
      FSOnChanged  := FMgrOwner.OnChange;
      FSOnDeletion := FMgrOwner.OnDeletion;
      FSOnExpanding := FMgrOwner.OnExpanding;

      FSGetImageIndex   := FMgrOwner.OnGetImageIndex;
      FSGetSelectedIndex:= FMgrOwner.OnGetSelectedIndex;

      FMgrOwner.OnChange    := Changed;
      FMgrOwner.OnDeletion  := Deleted;
      FMgrOwner.OnExpanding := Expanding;

      FMgrOwner.OnGetImageIndex    := GetImageIndex;
      FMgrOwner.OnGetSelectedIndex := GetSelectedIndex;

    end
    else begin
      FSOnChanged  := nil;
      FSOnDeletion := nil;
      FSOnExpanding:= nil;
    end
end;

procedure TDictionaryMgr.Resync;
var
  I: Integer;
  TempNode: TTreeNode;
  NewProxy: TDictionaryItem;
begin
  if FMgrOwner = nil then Exit;
  if not ((csLoading in ComponentState) or (csDesigning in ComponentState))
  then SyncTreeView;
  if NodeCount > Count then
  begin
    TempNode := FindFreeNode;
    while TempNode <> nil do
    begin
      NewProxy := TDictionaryItem.Create(Owner);
      if NewProxy = nil then raise Exception.Create('1111');
      NewProxy.Name := GetUniqueName(NewProxy);
      AddProxy(NewProxy);
      NewProxy.NodeName := TempNode.Text;
      TempNode := FindFreeNode;
    end
  end;

  for I := Count-1 downto 0 do
    if Count > NodeCount then begin
      if (Item[I].NodeName = '') or
         (NodeByName(Item[I].NodeName) = nil)
      then Item[I].Free;
    end
    else Break;
end;

function TDictionaryMgr.ItemByNode(Node: TTreeNode): TDictionaryItem;
var
  I: Integer;
  TempNode: TTreeNode;
begin
  Result := nil;
  if Node = nil then Exit;
  TempNode := Node;
  while TempNode.Parent <> nil do
    TempNode := TempNode.Parent;
  I := IndexByNode(TempNode);
  if I <> -1
  then Result := Item[I];
{
  TempNode := Node;
  while TempNode <> nil do
  begin
    I := IndexByNode(TempNode);
    if I <> -1 then
    begin
      Result := Item[I];
      Exit;
    end;
    TempNode := TempNode.GetPrev
  end
}
end;

procedure TDictionaryMgr.SetItemDataSet(Value: TDataSet);
{(Value: TDictionaryItem);}
begin
  if DataSource <> nil
  then begin
    DataSource.DataSet := Value;
    if Value <> nil then Value.FreeNotification(Self);
  end;
end;

procedure TDictionaryMgr.ActiveChanged;
var
  IsRoot: Boolean;
  TempDS: TDataSet;
begin
  TempDS := nil;
  if FActiveItem <> nil
  then begin
    IsRoot := FActiveItem.IsRootNode(FActiveNode);
    TempDS := FActiveItem.ConnectedDataSet;
    if IsRoot and (FActiveItem.FRootDataSet <> nil)
    then TempDS := FActiveItem.RootDataSet;
    if (TempDS <> nil)// and (not IsRoot)
      then begin
        TempDS.DisableControls;
        if TempDS <> FActiveItem.RootDataSet
        then begin
//!!! NEW
          SetItemDataSet(TempDS);
          TempDS.Filter := FActiveItem.GetConnectedFilter(FActiveItem.ConnectedFieldName, FActiveNode);
//        TempDS.Filtered := ((not IsRoot) and ((TempDS.Filter <> '')
//                           and (FActiveItem.ConnectedDataSet = TempDS));
          TempDS.Filtered := (TempDS.Filter <> '') and (FActiveItem.ConnectedDataSet = TempDS);
        end;
        TempDS.EnableControls;
      end
  end;
  if (DataSource <> nil) and (DataSource.DataSet <> TempDS)
  then SetItemDataSet(TempDS);
end;

procedure TDictionaryMgr.SetDataSource(Value: TDataSource);
begin
  if (FDataSource <> nil) and (FDataSource <> Value)
  then FDataSource.DataSet := FOldDataSet;
  FDataSource := Value;
  if not (csLoading in ComponentState)
  then ActiveChanged;
  if Value <> nil
  then begin
    Value.FreeNotification(Self);
    FOldDataSet := Value.DataSet;
  end;
end;

function TDictionaryMgr.IsRootNode(Node: TTreeNode): Boolean;
var
  I: Integer;
begin
  Result := False;
  for I := 0 to Count-1 do
    if Item[I].IsRootNode(Node)
    then begin
      Result := True;
      Break;
    end;
end;

procedure TDictionaryMgr.GetImageIndex(Sender: TObject; Node: TTreeNode);
var
  VItem: TDictionaryItem;
begin
  VItem := ItemByNode(Node);
  if (VItem <> nil) and not IsRootNode(Node)
  then Node.ImageIndex := VItem.NodeImageIndex;

  if Assigned(FSGetImageIndex) then FSGetImageIndex(Sender, Node);
end;

procedure TDictionaryMgr.GetSelectedIndex(Sender: TObject; Node: TTreeNode);
var
  VItem: TDictionaryItem;
begin
  VItem := ItemByNode(Node);
  if (VItem <> nil) and not IsRootNode(Node)
  then Node.SelectedIndex := ActiveItem.NodeSelectedIndex;

  if Assigned(FSGetSelectedIndex) then FSGetSelectedIndex(Sender, Node);
end;

procedure TDictionaryMgr.Changed(Sender: TObject; Node: TTreeNode);
var
  CurItem: TDictionaryItem;
begin
  CurItem := ItemByNode(Node);
  if CurItem <> nil
  then begin
    FActiveNode := Node;
    if Node.Expanded and CheckNode(Node)
    then CurItem.ExpandNode(Node);
    SetActiveItem(CurItem);
//    if CurItem <> FActiveItem
//    then ActiveItem := CurItem;
//    if CurItem <> FActiveItem
//    then SetActiveItem(CurItem)
//    begin
//      if Assigned(FBeforeItemChange)
//      then FBeforeItemChange(CurItem);
//      FActiveItem := CurItem;
//    end;
{
    if CurItem.ConnectedDataSet <> nil
    then begin
      CurItem.ConnectedDataSet.DisableControls;
      CurItem.ConnectedDataSet.Filter := CurItem.GetConnectedFilter(CurItem.ConnectedFieldName, Node);
      CurItem.ConnectedDataSet.Filtered := not CurItem.IsRootNode(Node);
      CurItem.ConnectedDataSet.EnableControls;

    end
}
  end
  else begin
    SetActiveItem(nil);
//    ActiveItem  := nil;
    FActiveNode := nil;
  end;
//  ActiveChanged;

  if Assigned(FSOnChanged)
  then FSOnChanged(Sender, Node)
end;

procedure TDictionaryMgr.Deleted(Sender: TObject; Node: TTreeNode);
begin
  if (Node <> nil) and (Node.Data <> nil)
  then begin
//   ItemByNode(Node).FClon.FreeBookmark(TBookMark(Node.Data));
  {$IFNDEF POLARIS_D4}
   try
     StrDispose(TBookMark(Node.Data));
   except
   end;
   {$ENDIF}
    Node.Data := nil;
  end;
  if Assigned(FSOnDeletion)
  then FSOnDeletion(Sender, Node);
end;

procedure TDictionaryMgr.Expanding(Sender: TObject; Node: TTreeNode; var AllowExpansion: Boolean);
var
  CurItem: TDictionaryItem;
begin
  CurItem := ItemByNode(Node);
  if (CurItem <> nil) and CheckNode(Node)
  then CurItem.ExpandNode(Node);
  if Assigned(FSOnExpanding)
  then FSOnExpanding(Sender, Node, AllowExpansion)
end;

procedure TDictionaryMgr.Loaded;
begin
  inherited Loaded;
  SyncTreeView;
  if not (csDesigning in ComponentState) and (FMgrOwner <> nil)
    and (Count > 0) and (Item[0] <> nil) and (Item[0].RootNode <> nil)
  then FMgrOwner.Selected := Item[0].RootNode;
end;

procedure TDictionaryMgr.Notification(AComponent: TComponent;
                                      AOperation: TOperation);
begin
  inherited Notification(AComponent,AOperation);
  if AOperation = opRemove
  then begin
    if AComponent = MgrOwner
    then MgrOwner := nil;
    if (FDataSource <> nil) and (AComponent = FDataSource)
    then DataSource := nil;
  end
end;

procedure TDictionaryMgr.GetChildren(Proc: TGetChildProc {$IFDEF POLARIS_D3};
                                     Root: TComponent {$ENDIF});
var
  I: Integer;
begin
  inherited GetChildren(Proc{$IFDEF POLARIS_D3}, Root{$ENDIF});
  for I := 0 to Count - 1 do
    Proc(Item[I]);
end;

procedure TDictionaryMgr.SetNodeProxies(Value: TList);
begin
end;

function TDictionaryMgr.NodeByName(Value: String): TTreeNode;
var
  Temp: TTreeNode;
begin
  Result := nil;
  if FMgrOwner <> nil
  then begin
    Temp := FMgrOwner.Items[0];
    while Temp <> nil do
    begin
      if CompareStr(Temp.Text,Value)=0
      then begin
        Result := Temp;
        Exit;
      end;
      Temp := Temp.GetNextSibling;
    end
  end
end;

procedure TDictionaryMgr.RefreshItem(Item: TDictionaryItem);
begin
  if Item <> nil
  then Item.RefreshNode;
end;

procedure TDictionaryMgr.Refresh;
var
  I: Integer;
begin
  if Count > 0
  then for I := 0 to Count-1 do
    RefreshItem(Item[I]);
end;

initialization
  RegisterClass(TDictionaryItem);
end.

