//==============================================
//       DVEditor.pas
//
//         Delphi.
//      DesignTime ( )   TDictionaryMgr.
//
//      Copyright 1998-2000 Polaris Software
//      http://members.xoom.com/PolarisSoft
//      mailto: PolarisLib@mail.ru
//==============================================
unit DVEditor;

{$I POLARIS.INC}

interface

uses
  Windows,
  Messages,
  SysUtils,
  Classes,
  Graphics,
  Controls,
  Forms,
  Dialogs,
  StdCtrls,
  ComCtrls,
  DsgnIntf,
  DsgnWnds,
  LibIntf,
  TypInfo,
  DB,
//  DBTables,
  DictItem,
{$IFDEF POLARIS_D5}
  Contnrs,
{$ENDIF}
  Grids, ExtCtrls;

type
{$IFDEF POLARIS_D4}
  TDesigner = IDesigner;
  TFormDesigner = IFormDesigner;
{$ENDIF}

{$IFNDEF POLARIS_D5}
  TDesignerSelectionList = TComponentList; 
{$ENDIF}

{ TDictEditor }
  TDictEditor = class({TForm}TDesignWindow)
    BtnPanel: TPanel;
    CloseButton: TButton;
    DeleteButton: TButton;
    NodeGrid: TDrawGrid;
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure CloseButtonClick(Sender: TObject);
    procedure NodeGridDrawCell(Sender: TObject; Col, Row: Integer;
      Rect: TRect; State: TGridDrawState);
    procedure NodeGridSelectCell(Sender: TObject; Col, Row: Integer;
      var CanSelect: Boolean);
    procedure FormResize(Sender: TObject);
    procedure NodeGridKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure FormShow(Sender: TObject);
    procedure DeleteButtonClick(Sender: TObject);
  private
    { Private declarations }
    FDictionaryMgr: TDictionaryMgr;
    FDeleting: Boolean;
    procedure SetDictionaryMgr(Value: TDictionaryMgr);
    function GetForm: TCustomForm;
    procedure UpdateData;
    function CheckDictionaryMgr: Boolean;
    procedure SelectProxy(Proxy: TDictionaryItem);
    function ProxyByRow(Row: Integer): TDictionaryItem;
  protected
    { Protected declarations }
    function UniqueName(Component: TComponent): String; override;
    procedure Activated; override;
  public
    { Public declarations }
    procedure FormModified; override;
  {$IFDEF POLARIS_D3}
    procedure FormClosed(Form: TCustomForm); override;
  {$ELSE}
    procedure FormClosed(Form: TForm); override;
  {$ENDIF}
    function GetEditState: TEditState; override;
  {$IFDEF POLARIS_D4}
    procedure ComponentDeleted(Component: IPersistent); override;
  {$ELSE}
    procedure ComponentDeleted(Component: TComponent); override;
  {$ENDIF}

    property DictionaryMgr: TDictionaryMgr read FDictionaryMgr
             write SetDictionaryMgr;
    property OwnerForm: TCustomForm read GetForm;
  end;

{ --TDictionaryMgr-- }
{ TNodeListProperty }
  TNodeListProperty = class(TPropertyEditor)
    function GetAttributes: TPropertyAttributes; override;
    function GetValue: String; override;
    procedure Edit; override;
  end;
{ TMgrOwnerProperty }
  TMgrOwnerProperty = class(TComponentProperty)
    procedure GetValues(Proc: TGetStrProc); override;
  end;

{ TActiveItemProperty }
  TActiveItemProperty = class(TComponentProperty)
    procedure GetValues(Proc: TGetStrProc); override;
  end;

{ **Component Editor for TDictionaryMgr** }
{ TDictionaryMgrEditor }
  TDictionaryMgrEditor = class(TComponentEditor)
    procedure ExecuteVerb(Index: Integer); override;
    function GetVerb(Index: Integer): String; override;
    function GetVerbCount: Integer; override;
  end;

{ --TDictionaryItem-- }
{ TRootNodeProperty }
  TRootNodeProperty = class(TStringProperty)
    function GetAttributes: TPropertyAttributes; override;
    procedure GetValues(Proc: TGetStrProc); override;
  end;

{ TDictDataSetProperty }
  TDictDataSetProperty = class(TComponentProperty)
  private
    FCheckProc: TGetStrProc;
    procedure CheckComponent(const Value: string);
  public
    procedure GetValues(Proc: TGetStrProc); override;
  end;

{ TItemOwnerProperty }
//!!!!!!!!!!!!!!!!!!!!!!!!
  TItemOwnerProperty = class(TComponentProperty)
  private
    FCheckProc: TGetStrProc;
    procedure CheckComponent(const Value: string);
  public
    procedure GetValues(Proc: TGetStrProc); override;
  end;
//!!!!!!!!!!!!!!!!!!!!!!!!

{ TConnectedDataSetProperty }
  TConnectedDataSetProperty = class(TComponentProperty)
  private
    FCheckProc: TGetStrProc;
    procedure CheckComponent(const Value: string);
  public
    procedure GetValues(Proc: TGetStrProc); override;
  end;

{ TRootDataSetProperty }
  TRootDataSetProperty = class(TComponentProperty)
  private
    FCheckProc: TGetStrProc;
    procedure CheckComponent(const Value: string);
  public
    procedure GetValues(Proc: TGetStrProc); override;
  end;

{ TDictFieldProperty }
  TDictFieldProperty = class(TStringProperty)
    function GetAttributes: TPropertyAttributes; override;
    procedure GetValues(Proc: TGetStrProc); override;
    function GetDictionaryPropName: string; virtual;
    procedure GetValueList(List: TStrings); virtual;
  end;

{ TConnectedFieldProperty }
  TConnectedFieldProperty = class(TDictFieldProperty)
    procedure GetValueList(List: TStrings); override;
  end;

{ TMaxLevelProperty }
  TMaxLevelProperty = class(TIntegerProperty)
    procedure SetValue(const Value: String); override;
  end;

resourcestring
  dpeItem = 'Item';
  dpeNode = 'Node text';
  dpeNodeEditor = 'Edit Nodes...';
  dpeMaxLevelError = 'Out of rannge. Range is -1..32767';
implementation

{$R *.DFM}

{ Tools }

function FindEditor(Manager: TDictionaryMgr): TDictEditor;
var
  I: Integer;
begin
  Result := nil;
  for I := 0 to Screen.FormCount - 1 do
    if (Screen.Forms[I] is TDictEditor)
    and (TDictEditor(Screen.Forms[I]).DictionaryMgr = Manager) then
    begin
      Result := TDictEditor(Screen.Forms[I]);
      Break;
    end
end;

procedure ShowNodeEditor(Designer: TDesigner;
                         Manager: TDictionaryMgr);
var
  Editor: TDictEditor;
begin
  if Manager = nil then Exit;
  Editor := FindEditor(Manager);
  if Editor <> nil then
  begin
    Editor.Show;
    if Editor.WindowState = wsMinimized
    then Editor.WindowState := wsNormal
  end
  else begin
    Editor := TDictEditor.Create(Application);
    try
      Editor.Designer := TFormDesigner(Designer);
      Editor.DictionaryMgr := Manager;
      Editor.Show;
    except
      Editor.Free;
      raise;
    end
  end
end;

const
  DrawBitmap: TBitmap = nil;
type
  TVertAlignment = (vaTopJustify, vaCenter, vaBottomJustify);

procedure UsesBitmap;
begin
  if DrawBitmap = nil then begin
    DrawBitmap := TBitmap.Create;
    DrawBitmap.Monochrome := True;
  end;
end;

procedure ReleaseBitmap; far;
begin
  if DrawBitmap <> nil then DrawBitmap.Free;
  DrawBitmap := nil;
end;

function Max(A, B: LongInt): LongInt;
begin
  if A > B
  then Result := A
  else Result := B
end;

procedure WriteText(ACanvas: TCanvas; const ARect: TRect; DX, DY: Integer;
  const Text: string; Format: Word);
var
  S: array[0..255] of Char;
  B, R: TRect;
begin
  UsesBitmap;
  if Format = DT_LEFT then
    ExtTextOut(ACanvas.Handle, ARect.Left + DX, ARect.Top + DY, ETO_OPAQUE or
      ETO_CLIPPED, @ARect, StrPCopy(S, Text), Length(Text), nil)
  else if Format = DT_RIGHT then
    with ACanvas do
      ExtTextOut(Handle, ARect.Right - TextWidth(Text) - 3, ARect.Top + DY,
        ETO_OPAQUE or ETO_CLIPPED, @ARect, StrPCopy(S, Text), Length(Text), nil)
  else
  begin
    with DrawBitmap, ARect do
    begin
      Width := Max(Width, Right - Left);
      Height := Max(Height, Bottom - Top);
      R := Rect(DX, DY, Right - Left - 1, Bottom - Top - 1);
      B := Rect(0, 0, Right - Left, Bottom - Top);
    end;
    with DrawBitmap.Canvas do begin
      Font := ACanvas.Font;
      Font.Color := clBlack;
      FillRect(B);
      DrawText(Handle, StrPCopy(S, Text), Length(Text), R, Format);
    end;
    with DrawBitmap, ARect do ACanvas.CopyRect(ARect, Canvas, B);
  end;
end;

procedure DrawCellText(Grid: TCustomGrid; ACol, ARow: Longint; const S: string;
  const ARect: TRect; Align: TAlignment; VertAlign: TVertAlignment);
const
  Formats: array[TAlignment] of Word = (DT_LEFT, DT_RIGHT,
    DT_CENTER or DT_WORDBREAK or DT_EXPANDTABS or DT_NOPREFIX);
var
  H: Integer;
begin
  case VertAlign of
    vaTopJustify: H := 2;
    vaCenter:
      with TDrawGrid(Grid) do begin
        H := (ARect.Bottom - ARect.Top - Canvas.TextHeight('W')) div 2;
      end;
    else {vaBottomJustify} begin
      with TDrawGrid(Grid) do begin
        H := ARect.Bottom - ARect.Top - Canvas.TextHeight('W');
      end;
    end;
  end;
  WriteText(TDrawGrid(Grid).Canvas, ARect, 2, H, S, Formats[Align]);
end;

{ TMgrOwnerProperty }

procedure TMgrOwnerProperty.GetValues(Proc: TGetStrProc);
var
  I: Integer;
  Component: TComponent;
begin
  for I := 0 to Designer.Form.ComponentCount - 1 do
  begin
    Component := Designer.Form.Components[I];
    if (Component is TTreeView) and (Component.Name <> '')
    then Proc(Component.Name)
  end
end;

{ TActiveItemProperty }

procedure TActiveItemProperty.GetValues(Proc: TGetStrProc);
var
  I: Integer;
  Component: TComponent;
  Mgr: TDictionaryMgr;
begin
  Mgr := TDictionaryMgr(GetComponent(0));
  for I := 0 to Designer.Form.ComponentCount - 1 do
  begin
    Component := Designer.Form.Components[I];
    if (Component is TDictionaryItem)
    and (TDictionaryItem(Component).ItemOwner = Mgr)
    and (Component.Name <> '')
    then Proc(Component.Name)
  end
end;

{ TRootNodeProperty }

function TRootNodeProperty.GetAttributes: TPropertyAttributes;
begin
  Result := [paValueList];
end;

procedure TRootNodeProperty.GetValues(Proc: TGetStrProc);
var
  DictItem: TDictionaryItem;
  Temp: TTreeNode;
begin
  DictItem := GetComponent(0) as TDictionaryItem;
  if (DictItem <> nil) and (DictItem.ItemOwner <> nil)
  and (DictItem.ItemOwner.MgrOwner <> nil) then
  begin
    Temp := DictItem.ItemOwner.MgrOwner.Items[0];
    while Temp <> nil do
    begin
      Proc(Temp.Text);
      Temp := Temp.GetNextSibling;
    end
  end;
end;

{ TDictDataSetProperty }

procedure TDictDataSetProperty.CheckComponent(const Value: string);
var
  J: Integer;
//  Dataset: TDBDataset;
  Dataset: TDataset;
begin
//  Dataset := TDBDataset(Designer.GetComponent(Value));
  Dataset := TDataset(Designer.GetComponent(Value));
  for J := 0 to PropCount - 1 do
    if TDictionaryItem(GetComponent(J)).IsLinkedTo(Dataset) then
      Exit;
  FCheckProc(Value);
end;

procedure TDictDataSetProperty.GetValues(Proc: TGetStrProc);
begin
  FCheckProc := Proc;
  inherited GetValues(CheckComponent);
end;

{ TItemOwnerProperty }
//!!!!!!!!!!!!!!!!!!!!!!!

procedure TItemOwnerProperty.CheckComponent(const Value: string);
var
  J: Integer;
  DM: TDictionaryMgr;
begin
  DM := TDictionaryMgr(Designer.GetComponent(Value));
  for J := 0 to PropCount - 1 do
    if TDictionaryMgr(GetComponent(J)) = DM then
      Exit;
  FCheckProc(Value);
end;

procedure TItemOwnerProperty.GetValues(Proc: TGetStrProc);
begin
  FCheckProc := Proc;
  inherited GetValues(CheckComponent);
end;
//!!!!!!!!!!!!!!!!

{ TConnectedDataSetProperty }

procedure TConnectedDataSetProperty.CheckComponent(const Value: string);
var
  J: Integer;
  Dataset: TDataset; //!TDBDataSet
begin
  Dataset := TDataset(Designer.GetComponent(Value));
  for J := 0 to PropCount - 1 do
    if TDictionaryItem(GetComponent(J)).IsConnectedTo(Dataset) then
      Exit;
  FCheckProc(Value);
end;

procedure TConnectedDataSetProperty.GetValues(Proc: TGetStrProc);
begin
  FCheckProc := Proc;
  inherited GetValues(CheckComponent);
end;

{ TRootDataSetProperty }

procedure TRootDataSetProperty.CheckComponent(const Value: string);
var
  J: Integer;
  Dataset: TDataset;//! TDBDataSet
begin
//  Dataset := TDBDataset(Designer.GetComponent(Value));
  Dataset := TDataset(Designer.GetComponent(Value));
  for J := 0 to PropCount - 1 do
    if TDictionaryItem(GetComponent(J)).IsRootTo(Dataset) then
      Exit;
  FCheckProc(Value);
end;

procedure TRootDataSetProperty.GetValues(Proc: TGetStrProc);
begin
  FCheckProc := Proc;
  inherited GetValues(CheckComponent);
end;

{ TDictFieldProperty }
function TDictFieldProperty.GetAttributes: TPropertyAttributes;
begin
  Result := [paValueList, paSortList];
end;

procedure TDictFieldProperty.GetValues(Proc: TGetStrProc);
var
  I: Integer;
  Values: TStringList;
begin
  Values := TStringList.Create;
  try
    GetValueList(Values);
    for I := 0 to Values.Count - 1 do Proc(Values[I]);
  finally
    Values.Free;
  end;
end;

function TDictFieldProperty.GetDictionaryPropName: string;
begin
  Result := 'DictionaryItem';
end;

procedure TDictFieldProperty.GetValueList(List: TStrings);
var
  DictItem: TDictionaryItem;
begin
  DictItem := TDictionaryItem(GetComponent(0));
  if (DictItem <> nil) and (DictItem.DataSet <> nil) then
   DictItem.DataSet.GetFieldNames(List);
end;

{ TConnectedFieldProperty }
procedure TConnectedFieldProperty.GetValueList(List: TStrings);
var
  DictItem: TDictionaryItem;
begin
  DictItem := TDictionaryItem(GetComponent(0));
  if (DictItem <> nil) and (DictItem.ConnectedDataSet <> nil) then
    DictItem.ConnectedDataSet.GetFieldNames(List);
end;

{ TMaxLevelProperty }

procedure TMaxLevelProperty.SetValue(const Value: String);
var
  L: Longint;
begin
  L := StrToInt(Value);                    { convert string to number }
  with GetTypeData(GetPropType)^ do        { this uses compiler data for type Integer }
  if (L < -1) or (L > 32767) then          { make sure it's in range... }
    raise EPropertyError.Create(dpeMaxLevelError);{ ...otherwise, raise exception }
  SetOrdValue(L);                          { if in range, go ahead and set value }
end;

{ TNodeListProperty }

function TNodeListProperty.GetAttributes: TPropertyAttributes;
begin
  Result := [paDialog, paReadOnly];
end;

function TNodeListProperty.GetValue: String;
begin
  FmtStr(Result, '(%s)', [TList.ClassName]);
end;

procedure TNodeListProperty.Edit;
begin
  ShowNodeEditor(Designer, TDictionaryMgr(GetComponent(0)))
end;

{ TDictionaryMgrEditor }

procedure TDictionaryMgrEditor.ExecuteVerb(Index: Integer);
begin
  case Index of
  0: ShowNodeEditor(Designer, TDictionaryMgr(Component));
  end
end;

function TDictionaryMgrEditor.GetVerb(Index: Integer): String;
begin
  case Index of
  0: Result := dpeNodeEditor;
  end;
end;

function TDictionaryMgrEditor.GetVerbCount: Integer;
begin
  Result := 1;
end;

{ TDictEditor }

procedure TDictEditor.FormClose(Sender: TObject; var Action: TCloseAction);
begin
  Action := caFree;
end;

procedure TDictEditor.CloseButtonClick(Sender: TObject);
begin
  Close;
end;

procedure TDictEditor.NodeGridDrawCell(Sender: TObject; Col, Row: Integer;
  Rect: TRect; State: TGridDrawState);
var
  CellText: String;
  Proxy: TDictionaryItem;
begin
  CellText := '';
  if gdFixed in State then
   case Col of
   0: CellText := dpeItem;
   1: CellText := dpeNode
   end
  else begin
    Proxy := ProxyByRow(Row-1);
//    Proxy := FDictionaryMgr.Item[Row-1];
    if Proxy <> nil then
    case Col of
    0: CellText := Proxy.Name;
    1: CellText := Proxy.NodeName;
    end
  end;
  DrawCellText(NodeGrid, Col, Row, CellText, Rect, taLeftJustify, vaCenter);
end;

procedure TDictEditor.NodeGridSelectCell(Sender: TObject; Col,
  Row: Integer; var CanSelect: Boolean);
begin
  SelectProxy(ProxyByRow(Row-1));
end;

procedure TDictEditor.FormResize(Sender: TObject);
begin
  with NodeGrid do
  begin
    DefaultColWidth := (ClientWidth - 1) div 2;
    ColWidths[1] := ClientWidth - ColWidths[0] - 1
  end
end;

procedure TDictEditor.NodeGridKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  if Shift = [] then
  case Key of
  VK_RETURN:
    if ProxyByRow(NodeGrid.Row - 1) <> nil then
      ActivateInspector(#0);
  VK_DELETE: DeleteButtonClick(nil);
  end
end;

procedure TDictEditor.FormShow(Sender: TObject);
begin
  if FDictionaryMgr.MgrOwner <> nil then
    Caption := Format('Editing %s',[FDictionaryMgr.MgrOwner.Name])
end;

function TDictEditor.CheckDictionaryMgr: Boolean;
begin
  Result := (FDictionaryMgr <> nil) and (FDictionaryMgr.Owner <> nil)
            and (Designer.Form <> nil)
end;

function TDictEditor.GetForm: TCustomForm;
begin
  Result := Designer.Form
end;

function TDictEditor.UniqueName(Component: TComponent): String;
begin
  Result := DictionaryMgr.GetUniqueName(Component);
end;

procedure TDictEditor.SetDictionaryMgr(Value: TDictionaryMgr) ;
begin
  if FDictionaryMgr <> Value then
  begin
    FDictionaryMgr := Value;
    UpdateData;
  end
end;

procedure TDictEditor.Activated;
begin
  SelectProxy(ProxyByRow(NodeGrid.Row-1));
end;

procedure TDictEditor.SelectProxy(Proxy: TDictionaryItem);
var
  FComponents: TDesignerSelectionList;
begin
  if CheckDictionaryMgr and Active then
  begin
    FComponents := TDesignerSelectionList.Create;
    if Proxy <> nil
    then FComponents.Add(Proxy)
    else FComponents.Add(FDictionaryMgr);
    SetSelection(FComponents);
  end
end;

function TDictEditor.ProxyByRow(Row: Integer): TDictionaryItem;
begin
  Result := nil;
  if CheckDictionaryMgr and
     (Row >= 0) and (Row < FDictionaryMgr.Count)
  then Result := FDictionaryMgr.Item[Row]
end;

procedure TDictEditor.UpdateData;
var
  ProxyCount: Integer;
begin
  if CheckDictionaryMgr then
  begin
    if not FDeleting then FDictionaryMgr.Resync;
    ProxyCount := FDictionaryMgr.NodeProxies.Count;
    if ProxyCount = 0 then
    begin
      NodeGrid.RowCount := 2;
      SelectProxy(nil)
    end
    else NodeGrid.RowCount := ProxyCount + 1;
    DeleteButton.Enabled := ProxyCount > 0;
    NodeGrid.Invalidate
  end
end;

procedure TDictEditor.DeleteButtonClick(Sender: TObject);
var
  Proxy: TDictionaryItem;
begin
  Proxy := ProxyByRow(NodeGrid.Row - 1);
  if Proxy <> nil then
  begin
    FDeleting := True;
    try
      Proxy.Free;
      Designer.Modified;
    finally
      FDeleting := False;
    end;
  end
end;

procedure TDictEditor.FormModified;
begin
  if not (csDestroying in ComponentState) then UpdateData;
end;

{$IFDEF POLARIS_D3}
procedure TDictEditor.FormClosed(Form: TCustomForm);
{$ELSE}
procedure TDictEditor.FormClosed(Form: TForm);
{$ENDIF}
begin
  if Form = OwnerForm then Close;
end;

function TDictEditor.GetEditState: TEditState;
begin
  Result := []
end;

{$IFDEF POLARIS_D4}
procedure TDictEditor.ComponentDeleted(Component: IPersistent);
{$ELSE}
procedure TDictEditor.ComponentDeleted(Component: TComponent);
{$ENDIF}
begin
{$IFDEF POLARIS_D4}
  if ExtractPersistent(Component) = FDictionaryMgr then
{$ELSE}
  if Component = FDictionaryMgr then
{$ENDIF}
  begin
    FDictionaryMgr := nil;
    Close;
  end
end;

initialization
finalization
  ReleaseBitmap;
end.
