//==============================================
//       dbrspeedbutton.pas
//
//         Delphi.
//      DBAware - TrSpeedButton.
//
//      Copyright 1998-2000 Polaris Software
//      http://members.xoom.com/PolarisSoft
//      mailto: PolarisLib@mail.ru
//==============================================
unit DBrSpeedButton;

{$I POLARIS.INC}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  Buttons, rSpeedButton, DB, DBCtrls, Menus, rDBConst, rDBUtils, DBrBitBtn;

type

  TrSpeedDataLink = class;

  TCustomDBrSpeedButton = class(TrSpeedButton)
  private
    { Private declarations }
    FKindAction:       TrDBAction;
    FDataLink:         TrSpeedDataLink;
    FButtonOptions:    TrDBOptionSet;
    FNotButtonOptions: TrDBOptionSet;
    FControlDataLink:  TrSpeedDataLink;
    FControlPositive:  TrDBOptionSet;
    FControlNegative:  TrDBOptionSet;
    FOnAfterAction:    TNotifyEvent;
    FOnActiveChanged:  TOnActiveChanged;
    procedure SetKindAction(const Value: TrDBAction);
    function  GetDataField: string;
    procedure SetDataField(const Value: string);
    function  GetField: TField;
    function  GetDataSource: TDataSource;
    procedure SetDataSource(Value: TDataSource);
    procedure SetButtonOptions(Value: TrDBOptionSet);
    procedure SetNotButtonOptions(Value: TrDBOptionSet);
    function   GetControlDataField: string;
    procedure  SetControlDataField(const Value: string);
    function   GetControlField: TField;
    function   GetControlDataSource: TDataSource;
    procedure  SetControlDataSource(Value: TDataSource);
    procedure  SetControlPositive(Value: TrDBOptionSet);
    procedure  SetControlNegative(Value: TrDBOptionSet);
    function PositiveStore: Boolean;
    function NegativeStore: Boolean;
    function ControlShortCutStore: Boolean;
    function MenuShortCutStore: Boolean;
  protected
    { Protected declarations }
    function IsNotDefault: Boolean;
    function IsNotDefaultHint: Boolean;
    property GlyphResource stored IsNotDefault;
    property KindAction: TrDBAction read FKindAction write SetKindAction default tbaCustom;
    property DataField: string read GetDataField write SetDataField;
    property DataSource: TDataSource read GetDataSource write SetDataSource;
    property Positive: TrDBOptionSet read FButtonOptions write SetButtonOptions
      stored PositiveStore;
    property Negative: TrDBOptionSet read FNotButtonOptions write SetNotButtonOptions
      stored NegativeStore;
    property ControlDataField: string read GetControlDataField write SetControlDataField;
    property ControlDataSource: TDataSource read GetControlDataSource write SetControlDataSource;
    property ControlPositive: TrDBOptionSet read FControlPositive write SetControlPositive
      default [];
    property ControlNegative: TrDBOptionSet read FControlNegative write SetControlNegative
      default [];
    property OnAfterAction: TNotifyEvent read FOnAfterAction write FOnAfterAction;
    property OnActiveChanged: TOnActiveChanged read FOnActiveChanged write FOnActiveChanged;
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Notification(AComponent: TComponent;
                           Operation: TOperation); override;
    property Field: TField read GetField;
    property ControlField: TField read GetControlField;
    procedure Click; override;
    procedure ActiveChanged; virtual;
  published
    property Enabled default False;
    property Hint stored IsNotDefaultHint;
    property NumGlyphs default 2;
    property ControlShortCut stored ControlShortCutStore;
    property MenuShortCut stored MenuShortCutStore;
  end;

  TDBrSpeedButton = class(TCustomDBrSpeedButton)
  public
    constructor Create(AOwner: TComponent); override;
  published
    property GlyphResKind;
    property GlyphResource;
    property KindAction;
    property DataField;
    property DataSource;
    property ControlDataField;
    property ControlDataSource;
    property Width default 21;
    property Height default 21;
    property Positive;
    property Negative;
    property ControlPositive;
    property ControlNegative;
    property OnAfterAction;
    property OnActiveChanged;
  end;

  TDBNULLSpeedButton = class(TCustomDBrSpeedButton)
  public
    constructor Create(AOwner: TComponent); override;
  published
    property Align;
    property GlyphResKind;
    property GlyphResource;
    property DataField;
    property DataSource;
    property Width default 21;
    property Height default 21;
    property OnAfterAction;
    property OnActiveChanged;
  end;

  TrSpeedDataLink = class(TFieldDataLink)
  private
    FButton: TCustomDBrSpeedButton;
  protected
    procedure EditingChanged; override;
    procedure DataSetChanged; override;
    procedure ActiveChanged; override;
    procedure RecordChanged(Field: TField); override;
  public
    constructor Create(ATool: TCustomDBrSpeedButton);
    destructor Destroy; override;
  end;

implementation

uses DBUtils;

{ TrSpeedDataLink }

procedure TrSpeedDataLink.EditingChanged;
begin
  inherited;
  if FButton <> nil then FButton.ActiveChanged;
end;

procedure TrSpeedDataLink.DataSetChanged;
begin
  inherited;
  if FButton <> nil then FButton.ActiveChanged;
end;

procedure TrSpeedDataLink.ActiveChanged;
begin
  inherited;
  if FButton <> nil then FButton.ActiveChanged;
end;

procedure TrSpeedDataLink.RecordChanged;
begin
  inherited;
  if FButton <> nil then FButton.ActiveChanged;
end;

constructor TrSpeedDataLink.Create(ATool: TCustomDBrSpeedButton);
begin
  inherited Create;
  FButton := ATool;
end;

destructor TrSpeedDataLink.Destroy;
begin
  FButton := nil;
  inherited;
end;

{ TCustomDBrSpeedButton }

constructor TCustomDBrSpeedButton.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FDataLink := TrSpeedDataLink.Create(Self);
  FDataLink.Control := Self;
  FControlDataLink := TrSpeedDataLink.Create(Self);
  FControlDataLink.Control := Self;
  FButtonOptions := [];
  FNotButtonOptions := [];
  FControlPositive := [];
  FControlNegative := [];
  NumGlyphs := 2;
  FKindAction := tbaCustom;
  ActiveChanged;
end;

destructor TCustomDBrSpeedButton.Destroy;
begin
  if Assigned(FDataLink) then FDataLink.Free;
  if Assigned(FControlDataLink) then FControlDataLink.Free;
  inherited Destroy;
end;

function TCustomDBrSpeedButton.GetDataField: string;
begin
  if Assigned(FDataLink) then Result := FDataLink.FieldName
  else Result := '';
end;

procedure TCustomDBrSpeedButton.SetDataField(const Value: string);
begin
  if Assigned(FDataLink) then FDataLink.FieldName := Value;
end;

function TCustomDBrSpeedButton.GetField: TField;
begin
  if Assigned(FDataLink) then Result := FDataLink.Field
  else Result := nil;
end;

function TCustomDBrSpeedButton.GetDataSource: TDataSource;
begin
  if Assigned(FDataLink) then Result := FDataLink.DataSource
  else Result := nil;
end;

procedure TCustomDBrSpeedButton.SetDataSource(Value: TDataSource);
begin
  if (DataSource <> Value) and
     not (FDataLink.DataSourceFixed and (csLoading in ComponentState)) then begin
    FDataLink.DataSource := Value;
    if not (csLoading in ComponentState) then ActiveChanged;
    if Value <> nil then Value.FreeNotification(Self);
  end;
end;

function TCustomDBrSpeedButton.GetControlDataField: string;
begin
  if Assigned(FControlDataLink) then Result := FControlDataLink.FieldName
  else Result := '';
end;

procedure TCustomDBrSpeedButton.SetControlDataField(const Value: string);
begin
  if Assigned(FControlDataLink) then FControlDataLink.FieldName := Value;
end;

function TCustomDBrSpeedButton.GetControlField: TField;
begin
  if Assigned(FControlDataLink) then Result := FControlDataLink.Field
  else Result := nil;
end;

function TCustomDBrSpeedButton.GetControlDataSource: TDataSource;
begin
  if Assigned(FControlDataLink) then Result := FControlDataLink.DataSource
  else Result := nil;
end;

procedure TCustomDBrSpeedButton.SetControlDataSource(Value: TDataSource);
begin
  if (ControlDataSource <> Value) then begin
    if not (FControlDataLink.DataSourceFixed and (csLoading in ComponentState)) then
      FControlDataLink.DataSource := Value;
    if not (csLoading in ComponentState) then ActiveChanged;
    if Value <> nil then Value.FreeNotification(Self);
  end;
end;

procedure TCustomDBrSpeedButton.Notification(AComponent: TComponent;
                                     Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) then
    if (FDataLink <> nil) and (AComponent = DataSource) then DataSource := nil
    else
      if (FControlDataLink <> nil) and (AComponent = ControlDataSource) then
        ControlDataSource := nil
end;

function TCustomDBrSpeedButton.PositiveStore;
begin
  Result := not (FButtonOptions = Positives[FKindAction]);
end;

function TCustomDBrSpeedButton.NegativeStore;
begin
  Result := not (FNotButtonOptions = Negatives[FKindAction]);
end;

procedure TCustomDBrSpeedButton.SetButtonOptions(Value: TrDBOptionSet);
begin
  FButtonOptions := Value;
  ActiveChanged;
end;

procedure TCustomDBrSpeedButton.SetNotButtonOptions(Value: TrDBOptionSet);
begin
  FNotButtonOptions := Value;
  ActiveChanged;
end;

procedure TCustomDBrSpeedButton.SetControlPositive(Value: TrDBOptionSet);
begin
  FControlPositive := Value;
  ActiveChanged;
end;

procedure TCustomDBrSpeedButton.SetControlNegative(Value: TrDBOptionSet);
begin
  FControlNegative := Value;
  ActiveChanged;
end;

function TCustomDBrSpeedButton.IsNotDefaultHint: Boolean;
var
  S: String;
begin
  S := LoadResString(DBActionHints[FKindAction]);
  Result := ANSICompareText(Hint,S)<>0;
end;

function TCustomDBrSpeedButton.IsNotDefault: Boolean;
begin
  Result := UpperCase(GlyphResource) <> DBActionResources[FKindAction];
end;

procedure TCustomDBrSpeedButton.SetKindAction(const Value: TrDBAction);
begin
  if FKindAction <> Value then begin
    if (DBActionResources[FKindAction]=GlyphResource) or
       (GlyphResource='') then
      GlyphResource := DBActionResources[Value];
    if (LoadResString(DBActionHints[FKindAction])=Hint) or (Hint='')
    then begin
      Hint := LoadResString(DBActionHints[Value]);
      ShowHint := True;
    end;
    FNotButtonOptions := Negatives[Value];
    FButtonOptions := Positives[Value];
    ControlShortCut := DBActionShortCuts[Value];
    MenuShortCut := DBActionShortCuts[Value];
    ActiveChanged;
    FKindAction := Value;
  end;
end;

procedure TCustomDBrSpeedButton.Click;
begin
  inherited;
  if Assigned(FDataLink.DataSource) and
     DoAction(FDataLink.DataSource.DataSet,Field,FKindAction) and
     (FKindAction <> tbaCustom) and Assigned(FOnAfterAction) then
    FOnAfterAction(Self);
end;

procedure TCustomDBrSpeedButton.ActiveChanged;
var
  En: Boolean;
begin
  En := Assigned(FDataLink) and
        TestOptions(FDataLink.DataSet,FDataLink.Field,FButtonOptions,FNotButtonOptions);
  if En and Assigned(FControlDataLink) and Assigned(ControlDataSource) then
    En := TestOptions(FControlDataLink.DataSet, FControlDataLink.Field,
                      FControlPositive, FControlNegative);
  if En and Assigned(FOnActiveChanged) then FOnActiveChanged(Self,FDataLink,En);
  Enabled := En;
end;

function TCustomDBrSpeedButton.ControlShortCutStore: Boolean;
begin
  Result := not (ControlShortCut = DBActionShortCuts[FKindAction]);
end;

function TCustomDBrSpeedButton.MenuShortCutStore: Boolean;
begin
  Result := not (MenuShortCut = DBActionShortCuts[FKindAction]);
end;

{ TDBrSpeedButton }

constructor TDBrSpeedButton.Create(AOwner: TComponent);
begin
  inherited;
  Width := 21;
  Height := 21;
end;

{ TDBNULLSpeedButton }

constructor TDBNULLSpeedButton.Create(AOwner: TComponent);
begin
  inherited;
  ControlStyle := ControlStyle-[csSetCaption];
  Width := 21;
  Height := 21;
  KindAction := tbaNULL;
end;

end.
