//==============================================
//       dbrbitbtn.pas
//
//         Delphi.
//      DBAware - TCustomRBitBtn.
//
//      Copyright 1998-2000 Polaris Software
//      http://members.xoom.com/PolarisSoft
//      mailto: PolarisLib@mail.ru
//==============================================
unit DBrBitBtn;

{$I POLARIS.INC}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  Buttons, DB, DBCtrls, Menus, rButtons, rDBConst, rDBUtils;

type

  TrBBDataLink = class;

  TCustomDBrBitBtn = class(TCustomRBitBtn)
  private
    { Private declarations }
    FDataLink:         TrBBDataLink;
    FButtonOptions:    TrDBOptionSet;
    FNotButtonOptions: TrDBOptionSet;
    FControlDataLink:  TrBBDataLink;
    FControlPositive:  TrDBOptionSet;
    FControlNegative:  TrDBOptionSet;
    FOnActiveChanged:  TOnActiveChanged;
    FKindAction:       TrDBAction;
    FOnAfterAction:    TNotifyEvent;
    function  GetDataField: string;
    procedure SetDataField(const Value: string);
    function  GetField: TField;
    function  GetDataSource: TDataSource;
    procedure SetDataSource(Value: TDataSource);
    procedure SetButtonOptions(Value: TrDBOptionSet);
    procedure SetNotButtonOptions(Value: TrDBOptionSet);
    function   GetControlDataField: string;
    procedure  SetControlDataField(const Value: string);
    function   GetControlField: TField;
    function   GetControlDataSource: TDataSource;
    procedure  SetControlDataSource(Value: TDataSource);
    procedure  SetControlPositive(Value: TrDBOptionSet);
    procedure  SetControlNegative(Value: TrDBOptionSet);
    procedure SetKindAction(const Value: TrDBAction);
    function  PositiveStore: Boolean;
    function  NegativeStore: Boolean;
  protected
    { Protected declarations }
    procedure Notification(AComponent: TComponent;
                           Operation: TOperation); override;
    function IsNotDefault: Boolean;
    function IsNotDefaultHint: Boolean;
    property GlyphResource stored IsNotDefault;
    property KindAction: TrDBAction read FKindAction write SetKindAction default tbaCustom;
    property DataField: string read GetDataField write SetDataField;
    property DataSource: TDataSource read GetDataSource write SetDataSource;
    property Positive: TrDBOptionSet read FButtonOptions write SetButtonOptions
      stored PositiveStore;
    property Negative: TrDBOptionSet read FNotButtonOptions write SetNotButtonOptions
      stored NegativeStore;
    property ControlDataField: string read GetControlDataField write SetControlDataField;
    property ControlDataSource: TDataSource read GetControlDataSource write SetControlDataSource;
    property ControlPositive: TrDBOptionSet read FControlPositive write SetControlPositive
      default [];
    property ControlNegative: TrDBOptionSet read FControlNegative write SetControlNegative
      default [];
    property OnAfterAction: TNotifyEvent read FOnAfterAction write FOnAfterAction;
    property OnActiveChanged: TOnActiveChanged read FOnActiveChanged write FOnActiveChanged;
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure ActiveChanged; virtual;
    property Field: TField read GetField;
    property ControlField: TField read GetControlField;
    procedure Click; override;
  published
    property Enabled default False;
    property Hint stored IsNotDefaultHint;
    property NumGlyphs default 2;
  end;

  TDBrBitBtn = class(TCustomDBrBitBtn)
  published
    property Align;
    property GlyphResKind;
    property GlyphResource;
    property KindAction;
    property DataField;
    property DataSource;
    property ControlDataField;
    property ControlDataSource;
    property Positive;
    property Negative;
    property ControlPositive;
    property ControlNegative;
    property OnAfterAction;
    property OnActiveChanged;
  end;

  TDBNULLButton = class(TCustomDBrBitBtn)
  public
    constructor Create(AOwner: TComponent); override;
  published
    property Align;
    property GlyphResKind;
    property GlyphResource;
    property DataField;
    property DataSource;
    property Width default 21;
    property Height default 21;
    property OnAfterAction;
    property OnActiveChanged;
  end;

  TDBSwitchButton = class(TCustomDBrBitBtn)
  private
    FGlyphs: TImageList;
    FCaptions: TStrings;
    FValues: TStrings;
    FPosition: Integer;
    FNULLisZero: Boolean;
    function GetFieldState: Integer;
    procedure DataChange(Sender: TObject);
    function GetReadOnly: Boolean;
    procedure SetReadOnly(Value: Boolean);
    procedure SetGlyphs(Value: TImageList);
    procedure UpdateGlyph;
    procedure SetCaptions(Value: TStrings);
    procedure UpdateCaption;
    procedure SetValues(Value: TStrings);
    procedure SetPosition(const Value: Integer);
    procedure UpdateData(Sender: TObject);
    function ValueMatch(const ValueList, Value: string): Boolean;
    procedure CMExit(var Message: TCMExit); message CM_EXIT;
    procedure CMGetDataLink(var Message: TMessage); message CM_GETDATALINK;
  protected
    procedure KeyPress(var Key: Char); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Click; override;
    property Position: Integer read FPosition write SetPosition default -1;
  published
    property Align;
    property ReadOnly: Boolean read GetReadOnly write SetReadOnly default False;
    property Glyphs: TImageList read FGlyphs write SetGlyphs;
    property Captions: TStrings read FCaptions write SetCaptions;
    property Values: TStrings read FValues write SetValues;
    property NULLisZero: Boolean read FNULLisZero write FNULLisZero default True;
    property DataField;
    property DataSource;
  end;

  TrBBDataLink = class(TFieldDataLink)
  private
    FButton: TCustomDBrBitBtn;
  protected
    procedure EditingChanged; override;
    procedure DataSetChanged; override;
    procedure ActiveChanged; override;
    procedure RecordChanged(Field: TField); override;
  public
    constructor Create(ATool: TCustomDBrBitBtn);
    destructor Destroy; override;
  end;

implementation

uses DBUtils;

{ TrBBDataLink }

procedure TrBBDataLink.EditingChanged;
begin
  inherited;
  if FButton <> nil then FButton.ActiveChanged;
end;

procedure TrBBDataLink.DataSetChanged;
begin
  inherited;
  if FButton <> nil then FButton.ActiveChanged;
end;

procedure TrBBDataLink.ActiveChanged;
begin
  inherited;
  if FButton <> nil then FButton.ActiveChanged;
end;

procedure TrBBDataLink.RecordChanged;
begin
  inherited;
  if FButton <> nil then FButton.ActiveChanged;
end;

constructor TrBBDataLink.Create(ATool: TCustomDBrBitBtn);
begin
  inherited Create;
  FButton := ATool;
end;

destructor TrBBDataLink.Destroy;
begin
  FButton := nil;
  inherited;
end;

{ TCustomDBrBitBtn }

constructor TCustomDBrBitBtn.Create(AOwner: TComponent);
begin
  inherited;
  FDataLink := TrBBDataLink.Create(Self);
  FDataLink.Control := Self;
  FControlDataLink := TrBBDataLink.Create(Self);
  FControlDataLink.Control := Self;
  FButtonOptions := [];
  FNotButtonOptions := [];
  FControlPositive := [];
  FControlNegative := [];
  NumGlyphs := 2;
  FKindAction := tbaCustom;
  ActiveChanged;
end;

destructor TCustomDBrBitBtn.Destroy;
begin
  if Assigned(FDataLink) then FDataLink.Free;
  if Assigned(FControlDataLink) then FControlDataLink.Free;
  inherited;
end;

function TCustomDBrBitBtn.GetDataField: string;
begin
  if Assigned(FDataLink) then Result := FDataLink.FieldName
  else Result := '';
end;

procedure TCustomDBrBitBtn.SetDataField(const Value: string);
begin
  if Assigned(FDataLink) then FDataLink.FieldName := Value;
end;

function TCustomDBrBitBtn.GetField: TField;
begin
  if Assigned(FDataLink) then Result := FDataLink.Field
  else Result := nil;
end;

function TCustomDBrBitBtn.GetDataSource: TDataSource;
begin
  if Assigned(FDataLink) then Result := FDataLink.DataSource
  else Result := nil;
end;

procedure TCustomDBrBitBtn.SetDataSource(Value: TDataSource);
begin
  if (DataSource <> Value) and
     not (FDataLink.DataSourceFixed and (csLoading in ComponentState)) then begin
    FDataLink.DataSource := Value;
    if not (csLoading in ComponentState) then ActiveChanged;
    if Value <> nil then Value.FreeNotification(Self);
  end;
end;

function TCustomDBrBitBtn.GetControlDataField: string;
begin
  if Assigned(FControlDataLink) then Result := FControlDataLink.FieldName
  else Result := '';
end;

procedure TCustomDBrBitBtn.SetControlDataField(const Value: string);
begin
  if Assigned(FControlDataLink) then FControlDataLink.FieldName := Value;
end;

function TCustomDBrBitBtn.GetControlField: TField;
begin
  if Assigned(FControlDataLink) then Result := FControlDataLink.Field
  else Result := nil;
end;

function TCustomDBrBitBtn.GetControlDataSource: TDataSource;
begin
  if Assigned(FControlDataLink) then Result := FControlDataLink.DataSource
  else Result := nil;
end;

procedure TCustomDBrBitBtn.SetControlDataSource(Value: TDataSource);
begin
  if (ControlDataSource <> Value) then begin
    if not (FControlDataLink.DataSourceFixed and (csLoading in ComponentState)) then
      FControlDataLink.DataSource := Value;
    if not (csLoading in ComponentState) then ActiveChanged;
    if Value <> nil then Value.FreeNotification(Self);
  end;
end;

procedure TCustomDBrBitBtn.Notification(AComponent: TComponent;
                                     Operation: TOperation);
begin
  inherited;
  if (Operation = opRemove) then
    if (FDataLink <> nil) and (AComponent = DataSource) then DataSource := nil
    else
      if (FControlDataLink <> nil) and (AComponent = ControlDataSource) then
        ControlDataSource := nil
end;

function TCustomDBrBitBtn.PositiveStore;
begin
  Result := not (FButtonOptions = Positives[FKindAction]);
end;

function TCustomDBrBitBtn.NegativeStore;
begin
  Result := not (FNotButtonOptions = Negatives[FKindAction]);
end;

procedure TCustomDBrBitBtn.SetButtonOptions(Value: TrDBOptionSet);
begin
  FButtonOptions := Value;
  ActiveChanged;
end;

procedure TCustomDBrBitBtn.SetNotButtonOptions(Value: TrDBOptionSet);
begin
  FNotButtonOptions := Value;
  ActiveChanged;
end;

procedure TCustomDBrBitBtn.SetControlPositive(Value: TrDBOptionSet);
begin
  FControlPositive := Value;
  ActiveChanged;
end;

procedure TCustomDBrBitBtn.SetControlNegative(Value: TrDBOptionSet);
begin
  FControlNegative := Value;
  ActiveChanged;
end;

function TCustomDBrBitBtn.IsNotDefaultHint: Boolean;
var
  S: String;
begin
  S := LoadResString(DBActionHints[FKindAction]);
  Result := ANSICompareText(Hint,S)<>0;
end;

function TCustomDBrBitBtn.IsNotDefault: Boolean;
begin
  Result := UpperCase(GlyphResource) <> DBActionResources[FKindAction];
end;

procedure TCustomDBrBitBtn.SetKindAction(const Value: TrDBAction);
begin
  if FKindAction <> Value then begin
    if (DBActionResources[FKindAction]=GlyphResource) or
       (GlyphResource='') then
      GlyphResource := DBActionResources[Value];
    if (LoadResString(DBActionHints[FKindAction])=Hint) or (Hint='')
    then begin
      Hint := LoadResString(DBActionHints[Value]);
      ShowHint := True;
    end;
    FNotButtonOptions := Negatives[Value];
    FButtonOptions := Positives[Value];
    ActiveChanged;
    FKindAction := Value;
  end;
end;

procedure TCustomDBrBitBtn.Click;
begin
  inherited;
  if Assigned(FDataLink.DataSource) and
     DoAction(FDataLink.DataSource.DataSet,Field,FKindAction) and
     (FKindAction <> tbaCustom) and Assigned(FOnAfterAction) then
    FOnAfterAction(Self);
end;

procedure TCustomDBrBitBtn.ActiveChanged;
var
  En: Boolean;
begin
  En := Assigned(FDataLink) and
        TestOptions(FDataLink.DataSet,FDataLink.Field,FButtonOptions,FNotButtonOptions);
  if En and Assigned(FControlDataLink) and Assigned(ControlDataSource) then
    En := TestOptions(FControlDataLink.DataSet, FControlDataLink.Field,
                      FControlPositive, FControlNegative);
  if En and Assigned(FOnActiveChanged) then FOnActiveChanged(Self,FDataLink,En);
  Enabled := En;
end;

{ TDBNULLButton }

constructor TDBNULLButton.Create(AOwner: TComponent);
begin
  inherited;
  ControlStyle := ControlStyle-[csSetCaption];
  Width := 21;
  Height := 21;
  KindAction := tbaNULL;
end;

{ TDBSwitchButton }

constructor TDBSwitchButton.Create(AOwner: TComponent);
begin
  inherited;
  ControlStyle := ControlStyle-[csSetCaption];
  FCaptions := TStringList.Create;
  FValues := TStringList.Create;
  FPosition := -1;
  FDataLink.OnDataChange := DataChange;
  FDataLink.OnUpdateData := UpdateData;
  FNULLisZero := True;
end;

destructor TDBSwitchButton.Destroy;
begin
  FCaptions.Free;
  FValues.Free;
  inherited;
end;

procedure TDBSwitchButton.Click;
begin
  if FDataLink.Edit then  begin
    if FPosition >= FValues.Count-1 then
      if FNULLisZero then Position := 1 else Position := 0
    else Position := FPosition+1;
    FDataLink.Modified;
    inherited;
  end;
end;

function TDBSwitchButton.GetFieldState: Integer;
var
  Text_: string;
  i: Integer;
begin
  Result := 0;
  if FDatalink.Field <> nil then
    with FDatalink.Field do
      if not IsNull then
        if DataType = ftBoolean then begin
          if FNULLisZero then Result:=1;
          if not AsBoolean then Result := Result+1;
        end else begin
          Text_ := Text;
          for i:=0 to FValues.Count-1 do
            if ValueMatch(FValues.Strings[i], Text_) then begin
              Result := i;
              Break;
            end;
        end;
end;

procedure TDBSwitchButton.DataChange(Sender: TObject);
begin
  Position := GetFieldState;
end;

procedure TDBSwitchButton.UpdateData(Sender: TObject);
var
  Pos: Integer;
begin
  Pos := 1;
  if FNULLisZero and (FPosition = 0) then
    FDataLink.Field.Clear
  else
    if FDataLink.Field.DataType = ftBoolean then
      FDataLink.Field.AsBoolean := (FPosition=1)
    else
      if FPosition < FValues.Count then
        FDataLink.Field.Text :=
          ExtractFieldName(FValues.Strings[FPosition], Pos)
      else FDataLink.Field.Text := '';
end;

function TDBSwitchButton.ValueMatch(const ValueList, Value: string): Boolean;
var
  Pos: Integer;
begin
  Result := False;
  Pos := 1;
  while Pos <= Length(ValueList) do
    if AnsiCompareText(ExtractFieldName(ValueList, Pos), Value) = 0 then begin
      Result := True;
      Break;
    end;
end;

function TDBSwitchButton.GetReadOnly: Boolean;
begin
  Result := FDataLink.ReadOnly;
end;

procedure TDBSwitchButton.SetReadOnly(Value: Boolean);
begin
  FDataLink.ReadOnly := Value;
end;

procedure TDBSwitchButton.KeyPress(var Key: Char);
begin
  inherited KeyPress(Key);
  case Key of
    #8, ' ': FDataLink.Edit;
    #27:     FDataLink.Reset;
  end;
end;

procedure TDBSwitchButton.SetGlyphs(Value: TImageList);
begin
  if FGlyphs <> Value then begin
    FGlyphs := Value;
    if not(csDesigning in ComponentState) then UpdateGlyph;
    if Value <> nil then Value.FreeNotification(Self);
  end;
end;

procedure TDBSwitchButton.UpdateGlyph;
var
  bufBitmap: TBitmap;
begin
  if FGlyphs <> nil then begin
    with FGlyphs do
      if (FPosition >= 0) and (FPosition < Count) then begin
        bufBitmap := TBitmap.Create;
        try
          GetBitmap(FPosition,bufBitmap);
          Glyph := bufBitmap;
        finally
          bufBitmap.Free;
        end;
      end else Glyph := nil;
  end;
end;

procedure TDBSwitchButton.SetCaptions(Value: TStrings);
begin
  FCaptions.Assign(Value);
  UpdateCaption;
end;

procedure TDBSwitchButton.UpdateCaption;
begin
  with FCaptions do
    if (FPosition >= 0) and (FPosition < Count) then
      Caption := Strings[FPosition]
    else Caption := '';
end;

procedure TDBSwitchButton.SetValues(Value: TStrings);
begin
  FValues.Assign(Value);
  DataChange(Self);
end;

procedure TDBSwitchButton.CMExit(var Message: TCMExit);
begin
  try
    FDataLink.UpdateRecord;
  except
    SetFocus;
    raise;
  end;
  inherited;
end;

procedure TDBSwitchButton.CMGetDataLink(var Message: TMessage);
begin
  Message.Result := Integer(FDataLink);
end;

procedure TDBSwitchButton.SetPosition(const Value: Integer);
begin
  if FPosition <> Value then begin
    FPosition := Value;
    FDataLink.Modified;
    UpdateGlyph;
    UpdateCaption;
    Invalidate;
  end;
end;

end.
