//==============================================
//       crosstable.pas
//
//         Delphi.
//         -.
//
//      Copyright 1998-2000 Polaris Software
//      http://members.xoom.com/PolarisSoft
//      mailto: PolarisLib@mail.ru
//==============================================
unit CrossTable;

{$I POLARIS.INC}

interface

uses Classes,Db, DBTables,SysUtils;

type
  TDataOperation = (doNone, doSum, doCount, doAverage, doMin, doMax);

type
  TrCrossTable = class(TComponent)
  private
    FSourceDataSet      : TDataSet;
    FRowFieldName : String;
    FColFieldName : String;
    FDataFieldName : String;

    FColLookupDataset :  TDataSet;
    FColLookupFieldName :  String;
    FColResultFieldName :  String;

    FDataOperation: TDataOperation;
    FDatabase : TDatabase;
    FTargetDataSet: TDataSet;
    FFilter : String;
    FRowAlign : TAlignment;
    FColAlign : TAlignment;
    procedure SetTargetDataSet(Value: TDataSet);
    procedure SetDatabase(Value: TDatabase);
    procedure SetColLookupDataset(Value: TDataSet);
    procedure SetSourceDataSet(Value: TDataSet);
  protected
    QuTemp1 : TQuery;
    BatchMove: TBatchMove;
    QuTemp2 : TQuery;
    DisplayLabels : TStringList;
    procedure Notification(AComponent: TComponent;
                                     Operation: TOperation); override;

  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Execute;
  published
    property SourceDataSet: TDataSet read FSourceDataSet write SetSourceDataSet;
    property RowFieldName: String read FRowFieldName
             write FRowFieldName;
    property ColFieldName: String read FColFieldName
             write FColFieldName;
    property DataFieldName: String read FDataFieldName
             write FDataFieldName;

    property ColLookupDataSet: TDataSet read FColLookupDataSet write SetColLookupDataSet;
    property ColLookupFieldName: String read FColLookupFieldName
             write FColLookupFieldName;
    property ColResultFieldName: String read FColResultFieldName
             write FColResultFieldName;

    property TargetDataSet: TDataSet read FTargetDataSet write SetTargetDataSet;
    property Database: TDatabase read FDatabase write SetDatabase;
    property DataOperation: TDataOperation read FDataOperation write FDataOperation default doSum;

    property SourceFilter: String read FFilter write FFilter;
    property RowAlign: TAlignment read FRowAlign write FRowAlign default taRightJustify;
    property ColAlign: TAlignment read FColAlign write FColAlign default taRightJustify;
  end;

implementation

procedure TrCrossTable.Notification(AComponent: TComponent;
                                     Operation: TOperation);
begin
  inherited;
  if (Operation = opRemove) then
    if (FSourceDataSet <> nil) and (AComponent = FSourceDataSet) then FSourceDataSet := nil
    else if (FTargetDataSet <> nil) and (AComponent = FTargetDataSet) then FTargetDataSet := nil
    else if (FColLookupDataSet <> nil) and (AComponent = FColLookupDataSet) then FColLookupDataSet := nil;
end;

procedure TrCrossTable.SetSourceDataSet(Value: TDataSet);
begin
  if Value <> FSourceDataSet then begin
    FSourceDataSet := Value;
    if Value <> nil then Value.FreeNotification(Self);
  end;
end;

constructor TrCrossTable.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FSourceDataSet:= nil;
  FRowFieldName:='';
  FColFieldName:='';
  FDataFieldName:='';
  FColLookupFieldName:='';
  FColResultFieldName:='';
  FFilter:='';
  FTargetDataSet:= nil;
  FDatabase:=nil;
  FColLookupDataSet := nil;
  FDataOperation := doSum;
  FRowAlign:=taRightJustify;
  FColAlign:=taRightJustify;
  QuTemp1:=TQuery.Create(Self);
  QuTemp2:=TQuery.Create(Self);
  BatchMove:=TBatchMove.Create(Self);
  DisplayLabels:=TStringList.Create;
end;

destructor TrCrossTable.Destroy;
begin
  With QuTemp1 do begin
    if Active then Close;
    Free;
  end;
    With QuTemp2 do begin
    if Active then Close;
    Free;
  end;
  BatchMove.Free;
  DisplayLabels.Free;
  FDatabase:=nil;
  FTargetDataSet:=nil;
  FColLookupDataSet:=nil;
  FSourceDataSet:=nil;
  inherited Destroy;
end;

procedure TrCrossTable.SetDatabase(Value: TDatabase);
begin
  if FDatabase <> Value then begin
    FDatabase := Value;
    if Value <> nil then Value.FreeNotification(Self);
  end;
end;

procedure TrCrossTable.SetTargetDataSet(Value: TDataSet);
begin
  if Value <> FTargetDataSet  then begin
    FTargetDataSet := Value;
    if Value <> nil then Value.FreeNotification(Self);
  end;
end;

procedure TrCrossTable.SetColLookupDataSet(Value: TDataSet);
begin
  if Value <> FColLookupDataSet  then begin
    FColLookupDataset := Value;
    if Value <> nil then Value.FreeNotification(Self);
  end;
end;

procedure TrCrossTable.Execute;
Var FieldName : String;
    n,i : Smallint;
    FDefs : TField;
begin
With FTargetDataSet do begin
  if Active then Close;
  for i:=0 to FieldCount-1 do begin
    Fields[0].Free;
  end;
  With FieldDefs do
  begin
    Clear;
    Add(FRowFieldName, FSourceDataSet.FieldByName(FRowFieldName).DataType, 0, False);
    DisplayLabels.Clear;
    DisplayLabels.Add(FRowFieldName);
  end;
end;
QuTemp2.DatabaseName:=FDatabase.DatabaseName;
With QuTemp2 do begin
  if Active then Close;
  SQL.Clear;
  if FColLookupDataSet<>nil then begin
    SQL.Add('Select Distinct A.'+FColFieldName+',B.'+FColResultFieldName+' From '+TTable(FSourceDataSet).TableName+' A,'+
            TTable(FColLookupDataSet).TableName+' B');
    if FFilter<>'' then
      SQL.Add('WHERE '+FFilter+' AND A.'+FColFieldName+'=B.'+FColLookupFieldName)
    else
      SQL.Add('WHERE A.'+FColFieldName+'=B.'+FColLookupFieldName);
  end
  else begin
    SQL.Add('Select Distinct '+FColFieldName+' From '+TTable(FSourceDataSet).TableName);
    if FFilter<>'' then
    SQL.Add('WHERE '+FFilter);
  end;
  Prepare;
  Open;
  n := 1;
  While not EOF do begin
    if FColLookupDataSet<>nil then
    DisplayLabels.Add(Fields[1].AsString);
    FieldName := 'P'+IntToStr(n);
    FTargetDataSet.FieldDefs.Add(FieldName,FSourceDataSet.FieldByName(FDataFieldName).DataType, 0, False);
    n := n+1;
    Next;
  end;
end;
with TTable(FTargetDataSet).IndexDefs do
begin
  Clear;
  Add(FRowFieldName, FRowFieldName, [ixPrimary, ixUnique]);
end;
TTable(FTargetDataSet).CreateTable;

FTargetDataSet.FieldDefs.Update;
For i:=0 to FTargetDataSet.FieldDefs.Count-1 do begin
  Case FTargetDataSet.FieldDefs[i].DataType of
    ftInteger : FDefs:=TIntegerField.Create(FTargetDataSet);
    ftFloat   : FDefs:=TFloatField.Create(FTargetDataSet);
    ftString  : FDefs:=TStringField.Create(FTargetDataSet);
    ftSmallint: FDefs:=TSmallintField.Create(FTargetDataSet);
  else
    FDefs:=TStringField.Create(FTargetDataSet);
  end;
  FDefs.FieldName:=FTargetDataSet.FieldDefs[i].Name;
  FDefs.DataSet:=FTargetDataSet;
  if FColLookupDataSet<>nil then FDefs.DisplayLabel:=DisplayLabels[i];
  if i=0 then FDefs.Alignment:=FRowAlign else
  FDefs.Alignment:=FColAlign;
  FDefs:=TField.Create(FTargetDataSet);
  FDefs.Free;
end;

BatchMove.Destination := TTable(FTargetDataSet);
QuTemp1.DatabaseName:=FDatabase.DatabaseName;
With QuTemp1 do begin
  if Active then Close;
  SQL.Clear;
  SQL.Add('Select '+FRowFieldName+' FROM '+TTable(FSourceDataSet).TableName);
  if FFilter<>'' then
    SQL.Add('WHERE '+FFilter);
end;
BatchMove.Source:=QuTemp1;
BatchMove.Mappings.Clear;
BatchMove.Mappings.Add(FRowFieldName);
BatchMove.Mode:=batAppend;
BatchMove.Execute;

      //    

With QuTemp2 do begin
   n := 1;
   BatchMove.Mode:=batUpdate;
   First;
   While not EOF do begin
     QuTemp1.Close;
     QuTemp1.SQL.Clear;
     case FDataOperation of
       doNone:
         QuTemp1.SQL.Add('Select '+FRowFieldName+','+FDataFieldName+' '+FTargetDataSet.FieldDefs.Items[n].Name);
       doSum:
         QuTemp1.SQL.Add('Select '+FRowFieldName+',Sum('+FDataFieldName+') '+FTargetDataSet.FieldDefs.Items[n].Name);
       doCount:
         QuTemp1.SQL.Add('Select '+FRowFieldName+',Count('+FDataFieldName+') '+FTargetDataSet.FieldDefs.Items[n].Name);
       doAverage:
         QuTemp1.SQL.Add('Select '+FRowFieldName+',Avg('+FDataFieldName+') '+FTargetDataSet.FieldDefs.Items[n].Name);
       doMin:
         QuTemp1.SQL.Add('Select '+FRowFieldName+',Min('+FDataFieldName+') '+FTargetDataSet.FieldDefs.Items[n].Name);
       doMax:
         QuTemp1.SQL.Add('Select '+FRowFieldName+',Max('+FDataFieldName+') '+FTargetDataSet.FieldDefs.Items[n].Name);
     end;
     QuTemp1.SQL.Add(' FROM '+TTable(FSourceDataSet).TableName);
     if FFilter<>''then
       QuTemp1.SQL.Add('WHERE '+FFilter+' AND '+ FColFieldName+'='''+Fields[0].AsString+'''')
     else
       QuTemp1.SQL.Add('WHERE '+FColFieldName+'='''+Fields[0].AsString+'''');
     if FDataOperation <> doNone then
       QuTemp1.SQL.Add('GROUP BY '+FRowFieldName);
     BatchMove.Mappings.Clear;
     BatchMove.Mappings.Add(FRowFieldName);
     BatchMove.Mappings.Add(FTargetDataSet.FieldDefs.Items[n].Name+'='+FTargetDataSet.FieldDefs.Items[n].Name);
     BatchMove.Execute;
     n:=n+1;
     Next;
  end;
end;
end;

end.
