{ ##
  @PROJECT_NAME             Windows shell folders routines and components.
  @PROJECT_DESC             A set of routines, classes and components for
                            processing Windows shell folders. Includes:
                            + Component that provides information about a
                              specified special shell folder.
                            + Class that enumerates all the special shell
                              folders.
                            + Routines to manipulate special shell folder
                              information.
                            + Component that encapsulates the Windows Browse for
                              Folder dialog box.
  @FILE                     PJShellFolders.pas
  @COMMENTS                 Run time unit that defines shell folders components,
                            classes and routines.
  @LEGAL_NOTICE             These components are distributed under the Mozilla
                            Public License - see below.
  @EMAIL                    peter.johnson@openlink.org
  @WEBSITE                  http://www.delphidabbler.com/
  @OWNER                    DelphiDabbler
  @AUTHOR                   Peter D Johnson, Llanarth, Ceredigion, Wales, UK.
  @COPYRIGHT                Copyright  2001-2004, P.D.Johnson, Llanarth,
                            Ceredigion, Wales UK.
  @CREDITS                  + Thanks to Philippe Lucarz for code that removes
                              warnings when compiling under Delphi 6 and 7.
                              Implemented in v2.1.
                            + Thanks to Ryan Fischbach for suggesting code that
                              enables adds the new dialog style to
                              TPJBrowseDialog. Implemented with changes in v2.2.
  @HISTORY(
    @REVISION(
      @VERSION              1.0
      @DATE                 01/04/2001
      @COMMENTS             Original version.
    )
    @REVISION(
      @VERSION              2.0
      @DATE                 15/06/2003
      @COMMENTS             + Removed registration procedure and TPJFolderIDPE
                              property editor to new PJShellFoldersDsgn unit and
                              removed DsgnIntf unit reference.
                            + Added boDirsOnly option to TPJBrowseDialog.Options
                              property - user can now specify if browse dialog
                              displays and accepts virtual folders or just those
                              in file system (previously the latter was only
                              option available).
                            + Added new EPJShellFolders exception and used in
                              place of Exception. Added helper routines to raise
                              exceptions and moved all error message strings to
                              resource strings.
                            + Renamed GetFolderDisplayName as
                              PIDLToFolderDisplayName, made public, and rewrote
                              code to get display names properly under WinNT and
                              Win9x.
                            + Renamed GetFolderPath function as PIDLToFolderPath
                              and made public.
                            + Added TPJBrowseDialog.OnSelChangeEx event to give
                              access to PIDL of selected folder.
                            + TPJBrowseDialog now displays help button as
                              disabled when HelpContext = 0.
                            + Moved conditionally compiled additional shell
                              folder constants missing from Delphi 3 to
                              interface section to make available to users.
    )
    @REVISION(
      @VERSION              2.1
      @DATE                 29/07/2003
      @COMMENTS             + Prevented warnings when compiling unit with Delphi
                              6 & 7 using conditional compilation. (Thanks to
                              Philippe Lucarz).
                            + Conditionally added extra CSIDL_ constants that
                              are declared in Delphi 6's ShlObj unit.
    )
    @REVISION(
      @VERSION              2.1
      @DATE                 22/08/2004
      @COMMENTS             + Added support for new dialog style - selected
                              using a new option in the Options property. Based
                              on code suggested by Ryan Fischbach.
                            + Provided default specifier of 0 for HelpContext
                              property. Thanks to Ryan Fischbach for suggesting.
                            + Added support for more new special folder (CSIDL_)
                              identifiers per MSDN documentation.
    )
  )
}


{
 * ***** BEGIN LICENSE BLOCK *****
 * 
 * Version: MPL 1.1
 * 
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with the
 * License. You may obtain a copy of the License at http://www.mozilla.org/MPL/
 * 
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
 * the specific language governing rights and limitations under the License.
 * 
 * The Original Code is Shell Folders Unit.
 * 
 * The Initial Developer of the Original Code is Peter Johnson
 * (http://www.delphidabbler.com/).
 * 
 * Portions created by the Initial Developer are Copyright (C) 2001-2004 Peter
 * Johnson. All Rights Reserved.
 * 
 * Contributor(s):
 *   Philippe Lucarz
 *   Ryan Fischbach
 * 
 * ***** END LICENSE BLOCK *****
}


unit PJShellFolders;


interface


// Determine compiler

{$IFDEF VER140}
  {$DEFINE DELPHI6ANDUP}
{$ENDIF}
{$IFDEF VER150}
  {$DEFINE DELPHI6ANDUP}
  {$DEFINE DELPHI7ANDUP}
{$ENDIF}


uses
  // Delphi
  SysUtils, Windows, Classes, Controls, Messages, ShlObj
  {$IFDEF DELPHI6ANDUP}
    // include this unit for extra shell folder identifiers
    , SHFolder
  {$ENDIF}
  ;



{$IFDEF VER100}
const
  // Constants missing from ShlObj in Delphi 3
  CSIDL_INTERNET                  = $0001;
  {$EXTERNALSYM CSIDL_INTERNET}
  CSIDL_ALTSTARTUP                = $001D;
  {$EXTERNALSYM CSIDL_ALTSTARTUP}
  CSIDL_COMMON_ALTSTARTUP         = $001E;
  {$EXTERNALSYM CSIDL_COMMON_ALTSTARTUP}
  CSIDL_COMMON_FAVORITES          = $001F;
  {$EXTERNALSYM CSIDL_COMMON_FAVORITES}
  CSIDL_INTERNET_CACHE            = $0020;
  {$EXTERNALSYM CSIDL_INTERNET_CACHE}
  CSIDL_COOKIES                   = $0021;
  {$EXTERNALSYM CSIDL_COOKIES}
  CSIDL_HISTORY                   = $0022;
  {$EXTERNALSYM CSIDL_HISTORY}
{$ENDIF}

{$IFNDEF DELPHI6ANDUP}
const
  // Extra constants provided in SHFolder unit in Delphi 6/7
  CSIDL_LOCAL_APPDATA             = $001C;
  {$EXTERNALSYM CSIDL_LOCAL_APPDATA}
  CSIDL_COMMON_APPDATA            = $0023;
  {$EXTERNALSYM CSIDL_COMMON_APPDATA}
  CSIDL_WINDOWS                   = $0024;
  {$EXTERNALSYM CSIDL_WINDOWS}
  CSIDL_SYSTEM                    = $0025;
  {$EXTERNALSYM CSIDL_SYSTEM}
  CSIDL_PROGRAM_FILES             = $0026;
  {$EXTERNALSYM CSIDL_PROGRAM_FILES}
  CSIDL_MYPICTURES                = $0027;
  {$EXTERNALSYM CSIDL_MYPICTURES}
  CSIDL_PROGRAM_FILES_COMMON      = $002B;
  {$EXTERNALSYM CSIDL_PROGRAM_FILES_COMMON}
  CSIDL_COMMON_DOCUMENTS          = $002E;
  {$EXTERNALSYM CSIDL_COMMON_DOCUMENTS}
  CSIDL_COMMON_ADMINTOOLS         = $002F;
  {$EXTERNALSYM CSIDL_COMMON_ADMINTOOLS}
  CSIDL_ADMINTOOLS                = $0030;
  {$EXTERNALSYM CSIDL_ADMINTOOLS}
{$ENDIF}

const
  // Further CSIDL constants not know to be defined elsewhere (from MSDN)
  CSIDL_CDBURN_AREA               = $003B;
  {$EXTERNALSYM CSIDL_CDBURN_AREA}
  CSIDL_COMMON_MUSIC              = $0035;
  {$EXTERNALSYM CSIDL_COMMON_MUSIC}
  CSIDL_COMMON_PICTURES           = $0036;
  {$EXTERNALSYM CSIDL_COMMON_PICTURES}
  CSIDL_COMMON_TEMPLATES          = $002D;
  {$EXTERNALSYM CSIDL_COMMON_TEMPLATES}
  CSIDL_COMMON_VIDEO              = $0037;
  {$EXTERNALSYM CSIDL_COMMON_VIDEO}
  CSIDL_MYDOCUMENTS               = $000C;
  {$EXTERNALSYM CSIDL_MYDOCUMENTS}
  CSIDL_MYMUSIC                   = $000D;
  {$EXTERNALSYM CSIDL_MYMUSIC}
  CSIDL_MYVIDEO                   = $000E;
  {$EXTERNALSYM CSIDL_MYVIDEO}
  CSIDL_PROFILE                   = $0028;
  {$EXTERNALSYM CSIDL_PROFILE}
  CSIDL_PROFILES                  = $003E;
  {$EXTERNALSYM CSIDL_PROFILES}


const
  // New dialog style flag for browse dialog box
  BIF_NEWDIALOGSTYLE = $0040;


type

  {
  IPJSpecialFolderEnum:
    Interface to enumerator of the identifiers of the Shell's special folders.
  }
  IPJSpecialFolderEnum = interface
    ['{0958B8A0-1D56-11D5-852A-EE0AA7BFE914}']
    procedure Init;
      {Intialises enumeration}
    function Next: Integer;
      {Returns next special folder identifier in enumeration, or -1 if at end
      of enumeration}
    function AtEnd: Boolean;
      {Returns true if at end of enumeration and false otherwise}
    function Count: Integer;
      {Returns number of folder ids in enumeration}
  end;


  {
  TPJSpecialFolderEnum
    Class that enumerates the indentifiers for the Shell's special folders.
  }
  TPJSpecialFolderEnum = class(TInterfacedObject,
    IPJSpecialFolderEnum)
  private
    fIndex: Integer;
      {Index of current folder in folder lookup table}
  public
    constructor Create;
      {Class constructor: intialises enumeration}
    procedure Init;
      {Intialises enumeration}
    function Next: Integer;
      {Returns next special folder identifier in enumeration, or -1 if at end
      of enumeration}
    function AtEnd: Boolean;
      {Returns true if at end of enumeration and false otherwise}
    function Count: Integer;
      {Returns number of folder ids in enumeration}
  end;


  {
  TPJSpecialFolderInfo:
    Component that provides information about the Shell's special folders.
  }
  TPJSpecialFolderInfo = class(TComponent)
  private // properties
    fFolderID: Integer;
    fPath: string;
    fDisplayName: string;
    fIsVirtual: Boolean;
    fIsSupported: Boolean;
    procedure SetFolderID(const Value: Integer);
  protected
    procedure GetCurrentFolderInfo; virtual;
      {Gets information about current special folder specified in FolderID and
      updates read only properties accordingly}
  public
    constructor Create(AOwner: TComponent); override;
      {Class constructor: sets default values and reads info about default
      special folder}
    property Path: string read fPath;
      {The path to the special folder: this is '' if folder is virtual or not
      supported}
    property DisplayName: string read fDisplayName;
      {Display name of special folder: this is '' if folder is not supported}
    property IsVirtual: Boolean read fIsVirtual;
      {Returns true if the special folder is virtual - i.e. not part of the
      physical file system. Returns false if folder is part of virtual file
      system or is not supported on this system}
    property IsSupported: Boolean read fIsSupported;
      {Not all special folders that have identifiers defined in Windows are
      supported on all operating systems. This property returns true if the
      current folder ID is supported on the underlying OS}
  published
    property FolderID: Integer
      read fFolderID write SetFolderID default CSIDL_DESKTOP;
      {The ID of the the current folder. Setting this property causes other
      properties to be updated to give information about the folder}
  end;


  {
  TPJBrowseSelChangeEvent:
    Type of event triggered by TPJBrowseDialog when selected folder changes.
  }
  TPJBrowseSelChangeEvent = procedure(Sender: TObject;
    FolderName, DisplayName: AnsiString; var StatusText: AnsiString;
    var OKEnabled: Boolean) of object;


  {
  TPJBrowseSelChangeEventEx:
    Type of event triggered by TPJBrowseDialog when selected folder changes:
    gives access to folder's PIDL.
  }
  TPJBrowseSelChangeEventEx = procedure(Sender: TObject;
    PIDL: PItemIDList; var StatusText: AnsiString;
    var OKEnabled: Boolean) of object;


  {
  TPJBrowseDlgOptions:
    Set of options available to Options property of TPJBrowseDlg.
  }
  TPJBrowseDlgOptions = set of (
    boShowHelp,           // show help button
    boContextHelp,        // show context help icon in title
    boStatusText,         // show status text in dlg box
    boDirsOnly,           // only allow selection of items in file system
    boNewDlgStyle         // use new dialog style
  );


  {
  TPJBrowseDialog:
    Displays browse dialog box.
  }
  TPJBrowseDialog = class(TComponent)
  private // properties
    fFolderName: string;
    fHeadline: string;
    fDisplayName: string;
    fRootFolderID: Integer;
    fTitle: TCaption;
    fOptions: TPJBrowseDlgOptions;
    fHelpContext: THelpContext;
    fOnInitialise: TNotifyEvent;
    fOnSelChange: TPJBrowseSelChangeEvent;
    fOnSelChangeEx: TPJBrowseSelChangeEventEx;
    fOnClose: TNotifyEvent;
    procedure SetRootFolderID(const Value: Integer);
    function GetHandle: THandle;
    procedure SetOptions(Value: TPJBrowseDlgOptions);
  private // other data
    fData: array[1..SizeOf(THandle) + SizeOf(Pointer)] of Byte;
      {Storage for information passed to and from browse dlg box callback
      function: this is cast to required types on use}
    fOldBrowseWndProc: Pointer;
      {Address of Browse dlg box's original window procedure}
    fNewBrowseWndProc: Pointer;
      {Address of new window procedure used to subclass browse dlg box}
  protected
    function GetHWND: THandle;
      {Returns the window handle of the form (if any) that owns this component}
    procedure BrowseWndProc(var Msg: TMessage); virtual;
      {Window procedure method used to subclass browse dlg box}
    procedure InitBrowseWindow;
      {Initialises the browse dlg box: called from browse dlg's callback
      function}
    procedure SelectionChanged(PIDL: PItemIDList);
      {Triggers OnSelChange event for currently selected folder and updates
      status text and OK button according to returns from the event handler:
      called from browse dlg's callback function}
    procedure IncludeHelpButton;
      {Creates a help button and displays in browse dlg box, rearranging other
      buttons as required}
  public
    constructor Create(AOwner: TComponent); override;
      {Class constructor: sets default values and window procedure used to
      subclass browse dlg box}
    destructor Destroy; override;
      {Class destructor: frees browse window procedure}
    function Execute: Boolean;
      {Displays browse dialog box and sets required properties if user OKs. Some
      of initialisation of dialog box is done the dialog box's
      BrowseCallbackProc callback function}
    property DisplayName: string read fDisplayName;
      {The display name of the selected folder}
    property Handle: THandle read GetHandle;
      {The window handle of the browse dlg box: this returns 0 if the dlg box is
      not currently displayed}
  published
    property Title: TCaption
      read fTitle write fTitle;
      {The dialog box's window title. If this property is not set (i.e. is the
      empty string) the dialog box displays 'Browse for Folder')}
    property FolderName: string
      read fFolderName write fFolderName;
      {The name of the folder chosen by the user from the dialog box. Setting
      this property before executing the dialog box causes the folder to be
      highlighted in the dialog box, if the folder exists}
    property RootFolderID: Integer
      read fRootFolderID write SetRootFolderID default CSIDL_DESKTOP;
      {The ID of the folder to be displayed as the root in the dlg box. This can
      be any of the special shell folders. If a particular folder is not
      supported on the system then an exception is raised}
    property Headline: string
      read fHeadline write fHeadline;
      {The "headline" that appears in the body of the dialog box above the tree
      view}
    property Options: TPJBrowseDlgOptions
      read fOptions write SetOptions default [boContextHelp, boDirsOnly];
      {Set of options that customise the appearance of the dialog box}
    property HelpContext: THelpContext
      read fHelpContext write fHelpContext default 0;
      {Help context used to access windows help topic. The property is when Help
      button is displayed and button is pressed, or when context help icon is
      not displayed and F1 is pressed}
    property OnInitialise: TNotifyEvent
      read fOnInitialise write fOnInitialise;
      {Event triggered when browse dlg box is initialised: event occurs after
      window title is set and initial selection is made. The dialog's window
      can be accessed via the Handle property}
    property OnSelChange: TPJBrowseSelChangeEvent
      read fOnSelChange write fOnSelChange;
      {Event triggered when the selection changes in the dlg box. The path and
      display name of the current folder are provided, as is the state of the
      OK button in the dialog box. The button state can be changed. The event
      handler can also provide any status text that is to be shown in the dialog
      box. (The text is only displayed id ShowStatusText is true). The dialog's
      window can be accessed via the Handle property}
    property OnSelChangeEx: TPJBrowseSelChangeEventEx
      read fOnSelChangeEx write fOnSelChangeEx;
      {Event triggered immediately after OnSelChanges when the selection changes
      in the dlg box. The PIDL of the selected folder is provided, as is the
      state of the OK button in the dialog box. The button state can be changed.
      The event handler can also provide any status text that is to be shown in
      the dialog box. (The text is only displayed if ShowStatusText is true).
      The dialog's window can be accessed via the Handle property}
    property OnClose: TNotifyEvent
      read fOnClose write fOnClose;
      {Event triggered when the browse dlg box closes. The dialog's window can
      be accessed via the Handle property}
  end;


  {
  EPJShellFolders:
    Class used for exceptions raised within this unit.
  }
  EPJShellFolders = class(Exception);


{ Special folder routines }

function SpecialFolderIdToStr(ID: Integer): string;
  {Returns the name of the constant used to represent the given special folder
  indentifier. Raises an exception if ID is not a valid special folder
  indentifier}

function StrToSpecialFolderID(const IDStr: string): Integer;
  {Returns the special folder identifier value associated with the given
  constant. Raises an exception if the constant is unknown}

function IsValidSpecialFolderId(ID: Integer): Boolean;
  {Returns true if the given identifier value is valid as a special folder
  identifier (i.e. defined by Windows) and false if not}

function NumSpecialFolderIds: Integer;
  {Returns number of special folder identifiers supported by Windows}


{ PIDL information routines }

function PIDLToFolderPath(PIDL: PItemIDList): string;
  {Returns the path of the folder given by PIDL}

function PIDLToFolderDisplayName(PIDL: PItemIDList): string;
  {Returns the display name for the given PIDL}


implementation


uses
  // Delphi
  ActiveX, Forms, ShellAPI;


// -----------------------------------------------------------------------------
// Error handling
// -----------------------------------------------------------------------------

resourcestring
  {Error messages}
  sBadSpecialFolderID     = 'Invalid special folder ID';
  sBadSpecialFolderIDStr  = '"%s" is not a valid special folder ID constant';
  sNoRootFolder           = 'Root folder not supported on this system';

procedure Error(const Msg: string);
  {Raises EPJShellFolders exception with given message}
begin
  raise EPJShellFolders.Create(Msg);
end;

procedure ErrorFmt(const Msg: string; Args: array of const);
  {Raises EPJShellFolder exception after formating given message per Args}
begin
  Error(Format(Msg, Args));
end;


// -----------------------------------------------------------------------------
// General PIDL management routines
// -----------------------------------------------------------------------------

procedure FreePIDL(PIDL: PItemIDList);
  {Helper procedure that frees the memory used by a given PIDL using the Shell's
  allocator}
var
  Malloc: IMalloc;  // shell's allocator
begin
  if Succeeded(SHGetMalloc(Malloc)) then
    Malloc.Free(PIDL);
end;


// -----------------------------------------------------------------------------
// PIDL information routines
// -----------------------------------------------------------------------------

function PIDLToFolderPath(PIDL: PItemIDList): string;
  {Returns the path of the folder given by PIDL}
var
  Path: PChar;  // buffer to hold folder's path
begin
  Path := StrAlloc(MAX_PATH);
  try
    SHGetPathFromIDList(PIDL, Path);
    Result := Path;
  finally
    StrDispose(Path);
  end;
end;

function PIDLToFolderDisplayName(PIDL: PItemIDList): string;
  {Returns the display name for the given PIDL}
var
  FileInfo: TSHFileInfo;  // file info passed back from SHGetFileInfo
begin
  FillChar(FileInfo, SizeOf(FileInfo), #0);
  SHGetFileInfo(
    PChar(PIDL),
    0,
    FileInfo,
    SizeOf(FileInfo),
    SHGFI_PIDL or SHGFI_DISPLAYNAME
  );
  Result := FileInfo.szDisplayName;
end;


// -----------------------------------------------------------------------------
// Special folder identifier constants and routines
// -----------------------------------------------------------------------------

const

  {Table mapping all special folder identifiers defined by Windows to the
  string representation of the constant}
  cFolders: array[1..50] of record    // table of special folder IDs
    ID:   Integer;    // special folder identifier value
    Name: string;     // constant used to represent special folder
  end =
  (
    (ID: CSIDL_ADMINTOOLS; Name: 'CSIDL_ADMINTOOLS';),
    (ID: CSIDL_ALTSTARTUP; Name: 'CSIDL_ALTSTARTUP';),
    (ID: CSIDL_APPDATA; Name: 'CSIDL_APPDATA';),
    (ID: CSIDL_BITBUCKET; Name: 'CSIDL_BITBUCKET';),
    (ID: CSIDL_CDBURN_AREA; Name: 'CSIDL_CDBURN_AREA';),
    (ID: CSIDL_COMMON_ADMINTOOLS; Name: 'CSIDL_COMMON_ADMINTOOLS';),
    (ID: CSIDL_COMMON_ALTSTARTUP; Name: 'CSIDL_COMMON_ALTSTARTUP';),
    (ID: CSIDL_COMMON_APPDATA; Name: 'CSIDL_COMMON_APPDATA';),
    (ID: CSIDL_COMMON_DESKTOPDIRECTORY; Name: 'CSIDL_COMMON_DESKTOPDIRECTORY';),
    (ID: CSIDL_COMMON_DOCUMENTS; Name: 'CSIDL_COMMON_DOCUMENTS';),
    (ID: CSIDL_COMMON_FAVORITES; Name: 'CSIDL_COMMON_FAVORITES';),
    (ID: CSIDL_COMMON_MUSIC; Name: 'CSIDL_COMMON_MUSIC';),
    (ID: CSIDL_COMMON_PICTURES; Name: 'CSIDL_COMMON_PICTURES';),
    (ID: CSIDL_COMMON_PROGRAMS; Name: 'CSIDL_COMMON_PROGRAMS';),
    (ID: CSIDL_COMMON_STARTMENU; Name: 'CSIDL_COMMON_STARTMENU';),
    (ID: CSIDL_COMMON_STARTUP; Name: 'CSIDL_COMMON_STARTUP';),
    (ID: CSIDL_COMMON_TEMPLATES; Name: 'CSIDL_COMMON_TEMPLATES';),
    (ID: CSIDL_COMMON_VIDEO; Name: 'CSIDL_COMMON_VIDEO';),
    (ID: CSIDL_CONTROLS; Name: 'CSIDL_CONTROLS';),
    (ID: CSIDL_COOKIES; Name: 'CSIDL_COOKIES';),
    (ID: CSIDL_DESKTOP; Name: 'CSIDL_DESKTOP';),
    (ID: CSIDL_DESKTOPDIRECTORY; Name: 'CSIDL_DESKTOPDIRECTORY';),
    (ID: CSIDL_DRIVES; Name: 'CSIDL_DRIVES';),
    (ID: CSIDL_FAVORITES; Name: 'CSIDL_FAVORITES';),
    (ID: CSIDL_FONTS; Name: 'CSIDL_FONTS';),
    (ID: CSIDL_HISTORY; Name: 'CSIDL_HISTORY';),
    (ID: CSIDL_INTERNET; Name: 'CSIDL_INTERNET';),
    (ID: CSIDL_INTERNET_CACHE; Name: 'CSIDL_INTERNET_CACHE';),
    (ID: CSIDL_LOCAL_APPDATA; Name: 'CSIDL_LOCAL_APPDATA';),
    (ID: CSIDL_MYDOCUMENTS; Name: 'CSIDL_MYDOCUMENTS';),
    (ID: CSIDL_MYMUSIC; Name: 'CSIDL_MYMUSIC';),
    (ID: CSIDL_MYPICTURES; Name: 'CSIDL_MYPICTURES';),
    (ID: CSIDL_MYVIDEO; Name: 'CSIDL_MYVIDEO';),
    (ID: CSIDL_NETHOOD; Name: 'CSIDL_NETHOOD';),
    (ID: CSIDL_NETWORK; Name: 'CSIDL_NETWORK';),
    (ID: CSIDL_PERSONAL; Name: 'CSIDL_PERSONAL';),
    (ID: CSIDL_PRINTERS; Name: 'CSIDL_PRINTERS';),
    (ID: CSIDL_PRINTHOOD; Name: 'CSIDL_PRINTHOOD';),
    (ID: CSIDL_PROFILE; Name: 'CSIDL_PROFILE';),
    (ID: CSIDL_PROFILES; Name: 'CSIDL_PROFILES';),
    (ID: CSIDL_PROGRAM_FILES; Name: 'CSIDL_PROGRAM_FILES';),
    (ID: CSIDL_PROGRAM_FILES_COMMON; Name: 'CSIDL_PROGRAM_FILES_COMMON';),
    (ID: CSIDL_PROGRAMS; Name: 'CSIDL_PROGRAMS';),
    (ID: CSIDL_RECENT; Name: 'CSIDL_RECENT';),
    (ID: CSIDL_SENDTO; Name: 'CSIDL_SENDTO';),
    (ID: CSIDL_STARTMENU; Name: 'CSIDL_STARTMENU';),
    (ID: CSIDL_STARTUP; Name: 'CSIDL_STARTUP';),
    (ID: CSIDL_SYSTEM; Name: 'CSIDL_SYSTEM';),
    (ID: CSIDL_TEMPLATES; Name: 'CSIDL_TEMPLATES';),
    (ID: CSIDL_WINDOWS; Name: 'CSIDL_WINDOWS';)
  );

function NumSpecialFolderIds: Integer;
  {Returns number of special folder identifiers supported by Windows}
begin
  Result := High(cFolders) - Low(cFolders) + 1;
end;

function IsValidSpecialFolderId(ID: Integer): Boolean;
  {Returns true if the given identifier value is valid as a special folder
  identifier (i.e. defined by Windows) and false if not}
var
  Idx: Integer; // loops through ID table
begin
  Result := False;
  for Idx := Low(cFolders) to High(cFolders) do
    if cFolders[Idx].ID = ID then
    begin
      Result := True;
      Break;
    end;
end;

function SpecialFolderIdToStr(ID: Integer): string;
  {Returns the name of the constant used to represent the given special folder
  indentifier. Raises an exception if ID is not a valid special folder
  indentifier}
var
  Idx: Integer; // loops thru ID table
begin
  // Assume no match
  Result := '';
  // Search for match to ID and get it's constant name
  for Idx := Low(cFolders) to High(cFolders) do
    if cFolders[Idx].ID = ID then
    begin
      Result := cFolders[Idx].Name;
      Break;
    end;
  // Raise exception if we didn't find a match
  if Result = '' then Error(sBadSpecialFolderID);
end;

function StrToSpecialFolderID(const IDStr: string): Integer;
  {Returns the special folder identifier value associated with the given
  constant. Raises an exception if the constant is unknown}
var
  Idx: Integer; // loops thru ID table
begin
  // Assume we don't find constant
  Result := -1;
  // Scan lookup table looking for constant
  for Idx := Low(cFolders) to High(cFolders) do
    if CompareText(cFolders[Idx].Name, IDStr) = 0 then
    begin
      Result := cFolders[Idx].ID;
      Break;
    end;
  // Raise exception if constant never found
  if Result = -1 then ErrorFmt(sBadSpecialFolderIDStr, [IDStr]);
end;


// -----------------------------------------------------------------------------
// TPJSpecialFolderEnum
// -----------------------------------------------------------------------------

function TPJSpecialFolderEnum.AtEnd: Boolean;
  {Returns true if at end of enumeration and false otherwise}
begin
  Result := fIndex > High(cFolders);
end;

function TPJSpecialFolderEnum.Count: Integer;
  {Returns number of folder ids in enumeration}
begin
  Result := NumSpecialFolderIds;
end;

constructor TPJSpecialFolderEnum.Create;
  {Class constructor: intialises enumeration}
begin
  inherited Create;
  Init;
end;

procedure TPJSpecialFolderEnum.Init;
  {Intialises enumeration}
begin
  fIndex := Low(cFolders);
end;

function TPJSpecialFolderEnum.Next: Integer;
  {Returns next special folder identifier in enumeration, or -1 if at end of
  enumeration}
begin
  if not AtEnd then
  begin
    Result := cFolders[fIndex].ID;
    Inc(fIndex);
  end
  else
    Result := -1;
end;


// -----------------------------------------------------------------------------
// TPJSpecialFolderInfo
// -----------------------------------------------------------------------------

constructor TPJSpecialFolderInfo.Create(AOwner: TComponent);
  {Class constructor: sets default values and reads info about default special
  folder}
begin
  inherited Create(AOwner);
  // Set default property values
  fFolderID := CSIDL_DESKTOP;
  // Now get folder info accordingly
  GetCurrentFolderInfo;
end;

procedure TPJSpecialFolderInfo.GetCurrentFolderInfo;
  {Gets information about current special folder specified in FolderID and
  updates read only properties accordingly}
var
  PIDL: PItemIDList;  // PIDL to special folder
begin
  // Get special folder's PIDL
  fIsSupported := Succeeded(SHGetSpecialFolderLocation(0, fFolderID, PIDL));
  if fIsSupported then
  begin
    try
      // Special folder is supported on this system: set required properties
      fPath := PIDLToFolderPath(PIDL);
      fDisplayName := PIDLToFolderDisplayName(PIDL);
      fIsVirtual := (fPath = '');
    finally
      FreePIDL(PIDL);
    end;
  end
  else
  begin
    // Special folder not supported on this system: set property values
    fPath := '';
    fDisplayName := '';
    fIsVirtual := False;
  end;
end;

procedure TPJSpecialFolderInfo.SetFolderID(const Value: Integer);
  {Write access method for FolderID property: reads information about given
  folder. Raises exception if Value is not a valid special folder indentifier}
begin
  if fFolderID <> Value then
  begin
    if not IsValidSpecialFolderId(Value) then Error(sBadSpecialFolderID);
    fFolderID := Value;
    GetCurrentFolderInfo;
  end;
end;


// -----------------------------------------------------------------------------
// TPJBrowseDialog
// -----------------------------------------------------------------------------

type
  {
  TCBData:
    Record that contains information passed betwenen TPJBrowseDialog component
    and the Browse dlg box's callback method. fData byte array is cast to this
    type when used. This was done to avoid declaring this type with public
    scope.
  }
  TCBData = packed record
    Handle: THandle;        // window handle of dlg box (0 if not active)
    Obj: TPJBrowseDialog;   // reference to component instance
  end;
  {
  PCBData:
    Pointer to TCBData.
  }
  PCBData = ^TCBData;

const
  {Identifiers for controls that are part of Browse dlg box. These were found
  by experiment on Windows 98. Not all these values are used in this class, but
  values left here for future information}
  cOKBtnID = $0001;
  cCancelBtnID = $0002;
  cTreeViewID = $3741;
  cHeadlineTextID = $3742;
  cStatusTextID = $3743;
  {Identifier of help button that may be added to Browse dlg box}
  cHelpBtnID = $1000;

{$IFDEF VER100}
type
  LongWord = Cardinal;    // type missing from Delphi 3
{$ENDIF}

function BrowseCallbackProc(HWnd: THandle; Msg: LongWord;
  LParam, Data: LongInt): Integer; stdcall;
  {Callback function called by browse dialog box. This function has two
  purposes: (1) to initialise the dialog box - properties not definable using
  BrowseInfo structure are set up here (2) special processing (including
  triggering event) is done when selection changes}
var
  Obj: TPJBrowseDialog;   // reference to component
  PDataRec: PCBData;      // pointer to data record passed in Data param
begin
  // This function must always return zero
  Result := 0;
  // Get reference to data structure (which is component's fData field)
  PDataRec := PCBData(Data);
  // Store dialog box's window handle in data structure
  PDataRec^.Handle := HWnd;
  // Record reference to TPJBrowseDlg component stored in Data param
  Obj := TPJBrowseDialog(PDataRec^.Obj);
  // Process event notifications
  case Msg of
    BFFM_INITIALIZED: // Perform initialisation
      // hand off to component
      Obj.InitBrowseWindow;
    BFFM_SELCHANGED:  // Perform any selection change processing
      // hand off to component: LParam contains PIDL of selected folder
      Obj.SelectionChanged(PItemIDList(LParam));
  end;
end;

procedure TPJBrowseDialog.BrowseWndProc(var Msg: TMessage);
  {Window procedure method used to subclass browse dlg box}
var
  HelpBtnHWnd: THandle; // window handle of help button
  Handled: Boolean;     // whether message was handled
begin
  // Assume we don't handle message
  Handled := False;
  // Handle supported messages
  case Msg.Msg of
    WM_COMMAND:
      // Only handle commands from Help button when it is shown
      if (boShowHelp in fOptions) and not (boNewDlgStyle in fOptions) then
      begin
        // Find help button's window
        HelpBtnHWnd := GetDlgItem(Handle, cHelpBtnID);
        // Check if message came from help button (sender's hwnd in LParam)
        if THandle(Msg.LParam) = HelpBtnHWnd then
        begin
          // Check if is button clicked notification (notification in WParamHi)
          if Msg.WParamHi = BN_CLICKED then
          begin
            // it was - display required help topic if there's a context
            if fHelpContext <> 0 then
              Application.HelpContext(fHelpContext);
            // message handled: nothing else to do
            Msg.Result := 0;
            Handled := True;
          end;
        end;
      end;
    WM_HELP:
      // Only handle this if we're not responding to context help: F1 pressed
      if not (boContextHelp in fOptions) then
      begin
        // F1 was pressed: display required help topic if there's a context
        if fHelpContext <> 0 then
          Application.HelpContext(fHelpContext);
        // message handled: nothing else to do
        Handled := True;
        Msg.Result := 1;
      end;
    WM_DESTROY:
      // Trigger OnClose event and allow message to pass to orig window proc
      if Assigned(fOnClose) then
        fOnClose(Self);
  end;
  // Pass any unhandled message to original window proc
  if not Handled then
    Msg.Result := CallWindowProc(
      fOldBrowseWndProc, Handle, Msg.Msg, Msg.WParam, Msg.LParam);
end;

constructor TPJBrowseDialog.Create(AOwner: TComponent);
  {Class constructor: sets default values and window procedure used to subclass
  browse dlg box}
begin
  inherited Create(AOwner);
  // Set default values
  fRootFolderID := CSIDL_DESKTOP;
  fOptions := [boContextHelp, boDirsOnly];
  with TCBData(fData) do
  begin
    Handle := 0;
    Obj := Self;
  end;
  // Create window procedure to be used for sub classing browse dlg box
  if not (csDesigning in ComponentState) then
    // call MakeObjectInstance from appropriate unit for compiler
    {$IFDEF DELPHI6ANDUP}
      fNewBrowseWndProc := Classes.MakeObjectInstance(BrowseWndProc);
    {$ELSE}
      fNewBrowseWndProc := Forms.MakeObjectInstance(BrowseWndProc);
    {$ENDIF}
end;

destructor TPJBrowseDialog.Destroy;
  {Class destructor: frees browse window procedure}
begin
  if Assigned(fNewBrowseWndProc) then
    // call FreeObjectInstance from appropriate unit for compiler
    {$IFDEF DELPHI6ANDUP}
      Classes.FreeObjectInstance(fNewBrowseWndProc);
    {$ELSE}
      Forms.FreeObjectInstance(fNewBrowseWndProc);
    {$ENDIF}
  inherited Destroy;
end;

function TPJBrowseDialog.Execute: Boolean;
  {Displays browse dialog box and sets required properties if user OKs. Some of
  initialisation of dialog box is done the dialog box's BrowseCallbackProc
  callback function}
var
  BI: TBrowseInfo;  // structure that controls appearance of browse dlg box
  pDisplayName: array [0..MAX_PATH + 1] of Char;  // used to return display name
  pidlRootFolder: PItemIDList;  // PIDL of root folder to be displayed
  pidlFolder: PItemIDList;      // PIDL of selected folder
  IsOleInit: Boolean;           // Whether Ole was initialised or not
begin
  // Record that OLE not initialised
  IsOleInit := False;
  // Get PIDL for required root folder (if desktop use nil)
  if fRootFolderID = CSIDL_DESKTOP then
    pidlRootFolder := nil
  else
    if not Succeeded(
      SHGetSpecialFolderLocation(0, fRootFolderID, pidlRootFolder)
    ) then
      Error(sNoRootFolder);
  try
    // Set up structure that defines properties of browse dlg box
    with BI do
    begin
      hwndOwner := GetHWND;         // window that owns dlg is component's owner
      pidlRoot := pidlRootFolder;                  // the root folder in dlg box
      pszDisplayName := pDisplayName;  // stores display name of selected folder
      lpszTitle := PChar(fHeadline);                // any body text for dlg box
      ulFlags := 0;                                          // initialise flags
      if boDirsOnly in fOptions then
        ulFlags := BIF_RETURNONLYFSDIRS;         // only accept physical folders
      if boStatusText in fOptions then
        ulFlags :=  ulFlags or BIF_STATUSTEXT // display status text if required
      else if boNewDlgStyle in fOptions then
      begin
        ulFlags := ulFlags or BIF_NEWDIALOGSTYLE;   // new dlg style if required
        OleInitialize(nil);      // new dlg style requires OLE to be initialized
        IsOleInit := True;
      end;
      lpfn := @BrowseCallbackProc;         // callback function to handle events
      lParam := Integer(@fData);         // reference to this component instance
      iImage := 0;                                                     // unused
    end;
    // Display the dlg box: returns non-nil PIDL if user OKs
    pidlFolder := SHBrowseForFolder(BI);
    TCBData(fData).Handle := 0;
    if pidlFolder <> nil then
    begin
      // User OK'd
      Result := True;
      // Store folder and display names in properties
      try
        fFolderName := PIDLToFolderPath(pidlFolder);
        fDisplayName := BI.pszDisplayName;
      finally
        // Release the selected folder PIDL's memory
        FreePIDL(pidlFolder);
      end;
    end
    else
      // User cancelled
      Result := False;
  finally
    // Un-initialize OLE if necessary
    if IsOleInit then
      OleUninitialize;
    // Release any root folder PIDL memory
    if pidlRootFolder <> nil then
      FreePIDL(pidlRootFolder);
  end;
end;

function TPJBrowseDialog.GetHandle: THandle;
  {Read access method for Handle property: returns browse dlg box handle while
  Execute method is active and 0 otherwise}
begin
  Result := TCBData(fData).Handle;
end;

function TPJBrowseDialog.GetHWND: THandle;
  {Returns the window handle of the form (if any) that owns this component}
begin
  if (Owner <> nil) and (Owner is TWinControl) then
    Result := (Owner as TWinControl).Handle
  else
    Result := 0;
end;

procedure TPJBrowseDialog.IncludeHelpButton;
  {Creates a help button and displays in browse dlg box, rearranging other
  buttons as required}
  // ---------------------------------------------------------------------------
  procedure GetBounds(HWnd, HParentWnd: THandle;
    PLeft, PTop, PWidth, PHeight: PInteger);
    {Gets left, top, width and height of the given window HWnd which has parent
    HParentWnd and returns in given parameters. If any of PLeft, PTop, PWidth
    and PHeight are nil then these values are not set}
  var
    ScreenRect: TRect;  // boundign rectangle of window in screen coordinates
    TopLeft: TPoint;    // coordinates of top left corner of widnow
  begin
    // Get bounding rectangle of window (in screen coords)
    GetWindowRect(Hwnd, ScreenRect);
    // Get top left corner of window in (parent's coordinates)
    TopLeft := ScreenRect.TopLeft;
    Windows.ScreenToClient(HParentWnd, TopLeft);
    // Calculate the required value for Left, Top, Width and Height
    if Assigned(PTop) then PTop^ := TopLeft.Y;
    if Assigned(PLeft) then PLeft^ := TopLeft.X;
    if Assigned(PWidth) then PWidth^ := ScreenRect.Right - ScreenRect.Left;
    if Assigned(PHeight) then PHeight^ := ScreenRect.Bottom - ScreenRect.Top;
  end;
  // ---------------------------------------------------------------------------
var
  HOKBtn, HCancelBtn, HHelpBtn: THandle;  // handles to buttons in dlg
  FontH: HFONT;                           // handle to font used on buttons
  BtnWidth, BtnHeight, BtnTop: Integer;   // width, height and top of dlg btns
  HelpLeft, OKLeft, CancelLeft: Integer;  // left of each of dlg's buttons
  StyleFlags: DWORD;                      // window style flags
begin
  // Get handles to OK and cancel buttons
  HOKBtn := GetDlgItem(Handle, cOKBtnID);
  HCancelBtn := GetDlgItem(Handle, cCancelBtnID);
  // Get button size and positions
  GetBounds(HOKBtn, Handle, @OKLeft, @BtnTop, @BtnWidth, @BtnHeight);
  GetBounds(HCancelBtn, Handle, @CancelLeft, nil, nil, nil);
  // Move OK & Cancel buttons left to leave room for help button on right
  HelpLeft := CancelLeft;
  CancelLeft := OKLeft;
  OKLeft := CancelLeft - (HelpLeft - CancelLeft);
  FontH := SendMessage(HOKBtn, WM_GETFONT, 0, 0);
  SetWindowPos(HOKBtn, 0, OKLeft, BtnTop, 0, 0,
    SWP_NOCOPYBITS or SWP_NOOWNERZORDER or SWP_NOSIZE	or SWP_NOZORDER);
  SetWindowPos(HCancelBtn, 0, CancelLeft, BtnTop, 0, 0,
    SWP_NOCOPYBITS or SWP_NOOWNERZORDER or SWP_NOSIZE	or SWP_NOZORDER);
  // Create the help button in the required place
  // set style flags - help button disabled if HelpContext = 0
  StyleFlags := WS_VISIBLE or WS_CHILD or BS_PUSHBUTTON or WS_TABSTOP;
  if fHelpContext = 0 then
    StyleFlags := StyleFlags or WS_DISABLED;
  // create the window
  HHelpBtn := CreateWindowEx(
    0,            // no extended style
    'Button',     // use existing button class
    '&Help',      // button's text
    StyleFlags,   // window style
    HelpLeft,     // left of button in parent's client coords
    BtnTop,       // top of button in parent's client coords
    BtnWidth,     // width of button
    BtnHeight,    // height of button
    Handle,       // handle of parent window
    cHelpBtnID,   // id of button
    GetWindowLong(Handle, GWL_HINSTANCE), // instance of owning app
    nil);         // no CREATESTRUCT data
  // Set the font of the help button to he same as other buttons
  SendMessage(HHelpBtn, WM_SETFONT, FontH, MakeLParam(1, 0));
end;

procedure TPJBrowseDialog.InitBrowseWindow;
  {Initialises the browse dlg box: called from browse dlg's callback function}
begin
  // Sub class the browse window using new wnd proc
  fOldBrowseWndProc := Pointer(
    SetWindowLong(Handle, GWL_WNDPROC, LongInt(fNewBrowseWndProc))
  );
  // Select any folder in browse dlg box as specified by user
  if FolderName <> '' then
    SendMessage(Handle, BFFM_SETSELECTION, 1, LongInt(PChar(fFolderName)));
  // If user specified title, display it in window caption
  if Title <> '' then
    SetWindowText(Handle, PChar(Title));
  // Install help button if required
  if boShowHelp in fOptions then
    IncludeHelpButton;
  // Hide context help window caption icon if required
  if not (boContextHelp in fOptions) then
    SetWindowLong(Handle, GWL_EXSTYLE,
      GetWindowLong(Handle, GWL_EXSTYLE) and not WS_EX_CONTEXTHELP);
  // Trigger to OnInitialise event
  if Assigned(fOnInitialise) then
    fOnInitialise(Self);
end;

procedure TPJBrowseDialog.SelectionChanged(PIDL: PItemIDList);
  {Triggers OnSelChange event for currently selected folder and updates status
  text and OK button according to returns from the event handler: called from
  browse dlg's callback function}
var
  SelFolder: string;    // path of the folder now selected
  SelDispName: string;  // display name of the folder now selected
  StatusText: string;   // any status text to be displayed in dlg
  HWndOKBtn: THandle;   // handle of OK button in dlg
  OKEnabled: Boolean;   // whether OK button is to be enabled or disabled
begin
  // Get folder and display name of selected item
  SelFolder := PIDLToFolderPath(PIDL);
  SelDispName := PIDLToFolderDisplayName(PIDL);
  if SelDispName = '' then
    SelDispName := SelFolder;              // no display name -> use folder name
  if SelDispName = SelFolder then
    SelDispName := ExtractFileName(SelFolder);             // don't use all path
  // Trigger and act on OnSelChange event if assigned
  // find if OK button is enabled or disabled
  HWndOKBtn := GetDlgItem(Handle, cOKBtnID);
  OKEnabled := IsWindowEnabled(HWndOKBtn);
  // set default status text value
  StatusText := '';
  // now trigger events, which may change StatusText and OKEnabled
  if Assigned(fOnSelChange) then
    fOnSelChange(Self, SelFolder, SelDispName, StatusText, OKEnabled);
  if Assigned(fOnSelChangeEx) then
    fOnSelChangeEx(Self, PIDL, StatusText, OKEnabled);
  // enable / disable OK button if user changed status of button
  if OKEnabled <> IsWindowEnabled(HWndOKBtn) then
  begin
    if OKEnabled then
      SendMessage(Handle, BFFM_ENABLEOK, 0, 1)  // format differs from API docs
    else
      SendMessage(Handle, BFFM_ENABLEOK, 0, 0); // format differs from API docs
  end;
  // Display any status text in dlg if user has specified this option
  if boStatusText in fOptions then
    SendMessage(Handle, BFFM_SETSTATUSTEXT, 0, LongInt(PChar(StatusText)));
end;

procedure TPJBrowseDialog.SetOptions(Value: TPJBrowseDlgOptions);
var
  NewItems: TPJBrowseDlgOptions;  // set of items in Value but not in Options
begin
  if (fOptions <> Value) then
  begin
    // Get any newly added items
    NewItems := Value - fOptions;
    if boNewDlgStyle in NewItems then
    begin
      // new dlg style option in new items: can't have help btn or status text
      Exclude(Value, boShowHelp);
      Exclude(Value, boStatusText);
    end
    else if [boShowHelp, boStatusText] * NewItems <> [] then
      // help btn or status text in new items: can't also have new style dlg
      Exclude(Value, boNewDlgStyle);
    fOptions := Value;
  end;
end;

procedure TPJBrowseDialog.SetRootFolderID(const Value: Integer);
  {Write access method for RootFolderID property: checks that a special folder
  with the given value is supported by the system and raises exception if not}
begin
  if fRootFolderID <> Value then
  begin
    if not IsValidSpecialFolderId(Value) then Error(sBadSpecialFolderID);
    fRootFolderID := Value;
  end;
end;

end.