{ ##
  @FILE                     PJEnvVars.pas
  @COMMENTS                 Source code for environment variable routines and
                            component.
  @PROJECT_NAME             Environment variables routines and component.
  @PROJECT_DESC             A set of routines and a component for processing and
                            managing environment variables.
  @LEGAL_NOTICE             The source code and any help files can be freely
                            distributed on a not-for-profit basis providing
                            that:
                            + the source code is not altered and
                            + these comments are not removed from the source
                              file.\
                            By not-for-profit I mean that you may recover out of
                            pocket expenses incurred in distributing the code,
                            but should not make a profit from this. If you
                            discover any bugs in this implementation, or if you
                            have any update suggestions, please contact me at
                            peter.johnson@openlink.org Please do modify the code
                            for you own use. I'd like to see any changes you
                            make - I could incorporate them into future
                            versions. Please notify me of changes on at the
                            above e-mail address. This software is provided as
                            is - no warranty is given as to its suitability for
                            any purposes to which you may wish to put it.
  @EMAIL                    peter.johnson@openlink.org
  @WEBSITE                  http://www.delphidabbler.com/
  @OWNER                    DelphiDabbler.
  @AUTHOR                   Peter D Johnson, Llanarth, Ceredigion, Wales, UK.
  @COPYRIGHT                Copyright  2001-2003, P.D.Johnson, Llanarth,
                            Ceredigion, Wales UK.
  @HISTORY(
    @REVISION(
      @VERSION              1.0
      @DATE                 02/09/2001
      @COMMENTS             Original version.
    )
    @REVISION(
      @VERSION              1.1
      @DATE                 31/07/2003
      @COMMENTS             + Changed component palette name from "PJ Stuff" to
                              "DelphiDabbler".
                            + Made EPJEnvVars exception derive from EOSError
                              rather than deprecated EWin32Error in Delphi 6 and
                              later using conditional compilation. In earlier
                              versions of Delphi EWin32Error is still used.
    )
    @REVISION(
      @VERSION              1.2
      @DATE                 10/08/2003
      @COMMENTS             Fixed bug that was causing an error when a
                            non-existant environment variable is accessed.
    )
  )
}


unit PJEnvVars;


interface


uses
  // Delphi
  SysUtils, Classes;


// Determine compiler

{$IFDEF VER140}
  {$DEFINE DELPHI6ANDUP}
{$ENDIF}
{$IFDEF VER150}
  {$DEFINE DELPHI6ANDUP}
  {$DEFINE DELPHI7ANDUP}
{$ENDIF}


// -----------------------------------------------------------------------------
// Environment variable routine declarations
// -----------------------------------------------------------------------------

function GetEnvVarValue(const VarName: string): string;
  {Returns the value for the given environment variable or '' if the variable
  does not exist}

function SetEnvVarValue(const VarName, VarValue: string): Integer;
  {Sets the given environment variable to the given value. Creates the
  environment variable if it doesn't already exist. Setting a variable to the
  empty string deletes it. Returns 0 on success or a Windows error code on
  error}

function DeleteEnvVar(const VarName: string): Integer;
  {Deletes the given environment variable. Returns 0 on success or a Windows
  error code on error}

function CreateEnvBlock(const NewEnv: TStrings; const IncludeCurrent: Boolean;
  const Buffer: Pointer; const BufSize: Integer): Integer;
  {Creates a new environment block containing the strings given in NewEnv. If
  NewEnv is nil then no new env vars are added to the block. If IncludeCurrent
  is true then the variable defined in the current process's environment block
  are included. The new block is stored in the memory pointed to by Buffer,
  which is at least BufSize bytes. The size of the block is returned. If the
  provided buffer is nil or is too small then no buffer is created. The return
  value gives the required buffer size}

function ExpandEnvVars(const Str: string): string;
  {Replaces any environment variables in the given string with their values and
  returns the string. Environment variables should be delimited by % characters
  thus: %ENVVAR%}

function GetAllEnvVars(const Vars: TStrings): Integer;
  {Copies all the environment variables available to the current process in the
  given string list, with each item in the string list representing one
  environment variable in the form NAME=VALUE. Returns the size of the
  environment block. If Vars=nil then the function simply returns the
  environment block size}


// -----------------------------------------------------------------------------
// Environment variable component declarations
// -----------------------------------------------------------------------------

type

  {
  TPJEnvVarsEnum:
    Callback method type used in TPJEnvVars.EnumNames method: called for
    each environment variable by name, passing used-supplied data.
  }
  TPJEnvVarsEnum = procedure(const VarName: string; Data: Pointer) of object;

  {
  TPJEnvVars:
    Component that encapsulates environment variables available to a program,
    permitting access to and modification of the variables.
  }
  TPJEnvVars = class(TComponent)
  private // properties
    function GetCount: Integer;
    function GetValue(Name: string): string;
    procedure SetValue(Name: string; const Value: string);
  public
    constructor Create(AOwner: TComponent); override;
      {Class contructor: ensures only one instance of the component is placed on
      a form}
    procedure EnumNames(Callback: TPJEnvVarsEnum; Data: Pointer);
      {Enumerates each environment variable: passes environment variable name
      and user supplied data pointer to given callback procedure for each
      variable in environment}
    procedure DeleteVar(const Name: string);
      {Deletes the given environment variable. Raises an exception if it is not
      possible to delete the variable}
    property Count: Integer read GetCount;
      {Count of number of environment variables}
    property Values[Name: string]: string read GetValue write SetValue;
      {Array of values of each environment variable, referenced by the
      variable's name. Referencing an unknown variable returns the empty string.
      Setting a value for an unknown variable creates it. If it is not possible
      to set a variable value then an exception is raised}
  end;

  {
  EPJEnvVars:
    Exception raised by TPJEnvVars when an environment variable error is
    encountered. Class derives either from EWin32Error in Delphi version < 6 or
    from EOSError in Delphi 6 onwards (where EWin32Error is depracated).
  }
  {$IFDEF DELPHI6ANDUP}
    EPJEnvVars = class(EOSError);
  {$ELSE}
    EPJEnvVars = class(EWin32Error);
  {$ENDIF}


// -----------------------------------------------------------------------------
// Component registration declaration
// -----------------------------------------------------------------------------

procedure Register;
  {Registers component with Delphi}


implementation


uses
  // Delphi
  Windows;


// -----------------------------------------------------------------------------
// Component registration
// -----------------------------------------------------------------------------

procedure Register;
  {Regiters component with Delphi}
begin
  RegisterComponents('DelphiDabbler', [TPJEnvVars]);
end;


// -----------------------------------------------------------------------------
// Environment variable routines
// -----------------------------------------------------------------------------

function GetEnvVarValue(const VarName: string): string;
  {Returns the value for the given environment variable or '' if the variable
  does not exist}
var
  BufSize: Integer;  // buffer size required for value (including terminal #0)
begin
  // Get required buffer size (includes space for terminal #0)
  BufSize := GetEnvironmentVariable(PChar(VarName), nil, 0);
  if BufSize > 0 then
  begin
    // Env var exists: read value into result string
    SetLength(Result, BufSize - 1); // space for #0 automatically added
    GetEnvironmentVariable(PChar(VarName), PChar(Result), BufSize);
  end
  else
    // Env var does not exist
    Result := '';
end;

function SetEnvVarValue(const VarName, VarValue: string): Integer;
  {Sets the given environment variable to the given value. Creates the
  environment variable if it doesn't already exist. Setting a variable to the
  empty string deletes it. Returns 0 on success or a Windows error code on
  error}
begin
  // Simply call SetEnvironmentVariable API function
  if SetEnvironmentVariable(PChar(VarName), PChar(VarValue)) then
    Result := 0
  else
    Result := GetLastError;
end;

function DeleteEnvVar(const VarName: string): Integer;
  {Deletes the given environment variable. Returns 0 on success or a Windows
  error code on error}
begin
  // Call SetEnvironmentVariable API function with nil value to delete var
  if SetEnvironmentVariable(PChar(VarName), nil) then
    Result := 0
  else
    Result := GetLastError;
end;

function CreateEnvBlock(const NewEnv: TStrings; const IncludeCurrent: Boolean;
  const Buffer: Pointer; const BufSize: Integer): Integer;
  {Creates a new environment block containing the strings given in NewEnv. If
  NewEnv is nil then no new env vars are added to the block. If IncludeCurrent
  is true then the variable defined in the current process's environment block
  are included. The new block is stored in the memory pointed to by Buffer,
  which is at least BufSize bytes. The size of the block is returned. If the
  provided buffer is nil or is too small then no buffer is created. The return
  value gives the required buffer size}
var
  EnvVars: TStringList; // list of env vars in new block
  Idx: Integer;         // loops through all env vars in new block
  PBuf: PChar;          // points to start of each env var entry in block
begin
  // Create string list to hold all new environment vars
  EnvVars := TStringList.Create;
  try
    // include copy of current environment block if required
    if IncludeCurrent then
      GetAllEnvVars(EnvVars);
    // store given environment vars in list
    if Assigned(NewEnv) then
      EnvVars.AddStrings(NewEnv);
    // Calculate size of new environment block
    Result := 0;
    for Idx := 0 to Pred(EnvVars.Count) do
      Inc(Result, Length(EnvVars[Idx]) + 1);
    Inc(Result);
    // Check if provided buffer is large enough and create block in it if so
    if (Buffer <> nil) and (BufSize >= Result) then
    begin
      // new environment blocks are always sorted
      EnvVars.Sorted := True;
      // do the copying
      PBuf := Buffer;
      for Idx := 0 to Pred(EnvVars.Count) do
      begin
        StrPCopy(PBuf, EnvVars[Idx]);
        Inc(PBuf, Length(EnvVars[Idx]) + 1);
      end;
      // terminate block with additional #0
      PBuf^ := #0;
    end;
  finally
    EnvVars.Free;
  end;
end;

function ExpandEnvVars(const Str: string): string;
  {Replaces any environment variables in the given string with their values and
  returns the string. Environment variables should be delimited by % characters
  thus: %ENVVAR%}
var
  BufSize: Integer; // size required for expanded string (excluding #0)
begin
  // Get required buffer size (excludes space for terminal #0)
  BufSize := ExpandEnvironmentStrings(PChar(Str), nil, 0);
  // Read expanded string into result string
  SetLength(Result, BufSize); // space for #0 automatically added
  ExpandEnvironmentStrings(PChar(Str), PChar(Result), BufSize);
end;

function GetAllEnvVars(const Vars: TStrings): Integer;
  {Copies all the environment variables available to the current process in the
  given string list, with each item in the string list representing one
  environment variable in the form NAME=VALUE. Returns the size of the
  environment block. If Vars=nil then the function simply returns the
  environment block size}
var
  PEnvVars: PChar;    // pointer to start of environment block
  PEnvEntry: PChar;   // pointer to an environment string in block
begin
  // Clear any list
  if Assigned(Vars) then
    Vars.Clear;
  // Get reference to environment block for this process
  PEnvVars := GetEnvironmentStrings;
  if PEnvVars <> nil then
  begin
    // We have a block: extract strings from it
    // Env strings are #0 separated and list ends with #0#0
    PEnvEntry := PEnvVars;
    try
      while PEnvEntry^ <> #0 do
      begin
        if Assigned(Vars) then
          Vars.Add(PEnvEntry);
        Inc(PEnvEntry, StrLen(PEnvEntry) + 1);
      end;
      // Calculate length of block
      Result := (PEnvEntry - PEnvVars) + 1;
    finally
      // Dispose of the memory block
      Windows.FreeEnvironmentStrings(PEnvEntry);
    end;
  end
  else
    // No block => zero length
    Result := 0;
end;


// -----------------------------------------------------------------------------
// Component definition
// -----------------------------------------------------------------------------

resourcestring
  // Error messages
  sSingleInstanceErr = 'Only one %s component is permitted on a form: ' +
    '%0:s is already present on %1:s';

procedure ErrorCheck(Code: Integer);
  {Private routine that checks the given code which is the return value from one
  of the functions that return Windows error codes. If zero nothing happens. If
  non-zero an exception is raised with an error message describing the error}
var
  Err: EPJEnvVars;  // reference to exception beinbg raised
begin
  // Check if we are to raise exception
  if Code <> 0 then
  begin
    // Create exception with win err code and related message
    Err := EPJEnvVars.Create(SysErrorMessage(Code));
    Err.ErrorCode := Code;
    // Raise the exception
    raise Err;
  end;
end;


{ TPJEnvVars }

constructor TPJEnvVars.Create(AOwner: TComponent);
  {Class contructor: ensures only one instance of the component is placed on a
  form}
var
  Idx: Integer; // loops thru components on Owner form
begin
  // Ensure that component is unique
  for Idx := 0 to Pred(AOwner.ComponentCount) do
    if AOwner.Components[Idx] is ClassType then
      raise Exception.CreateFmt(sSingleInstanceErr,
        [ClassName, AOwner.Components[Idx].Name, AOwner.Name]);
  // All OK: go ahead and create component
  inherited;
end;

procedure TPJEnvVars.DeleteVar(const Name: string);
  {Deletes the given environment variable. Raises an exception if it is not
  possible to delete the variable}
begin
  ErrorCheck(DeleteEnvVar(Name));
end;

procedure TPJEnvVars.EnumNames(Callback: TPJEnvVarsEnum; Data: Pointer);
  {Enumerates each environment variable: passes environment variable name and
  user supplied data pointer to given callback procedure for each variable in
  environment}
var
  Idx: Integer;         // loops thru env var list
  EnvList: TStringList; // list of env vars in form NAME=VALUE
begin
  // Create list to hold env vars
  EnvList := TStringList.Create;
  try
    // Read all env vars from environment into list
    GetAllEnvVars(EnvList);
    // Call callback proc for each name in environment, with user supplied data
    for Idx := 0 to Pred(EnvList.Count) do
      Callback(EnvList.Names[Idx], Data);
  finally
    EnvList.Free;
  end;
end;

function TPJEnvVars.GetCount: Integer;
  {Read access method for Count property}
var
  EnvList: TStringList; // list of all environment variables
begin
  // Get all env vars in string list and return number of items in list
  EnvList := TStringList.Create;
  try
    GetAllEnvVars(EnvList);
    Result := EnvList.Count;
  finally
    EnvList.Free;
  end;
end;

function TPJEnvVars.GetValue(Name: string): string;
  {Read access method for Values property}
begin
  Result := GetEnvVarValue(Name);
end;

procedure TPJEnvVars.SetValue(Name: string; const Value: string);
  {Write access method for Values property: raises exception if can't set the
  variable}
begin
  ErrorCheck(SetEnvVarValue(Name, Value));
end;

end.