{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

Author:       Alexey A. Dynnikov
EMail:        aldyn@chat.ru
WebSite:      http://www.chat.ru/~aldyn/index.html
Support:      Use the e-mail aldyn@chat.ru

Creation:     May 23, 1998
Version:      1.00

Legal issues: Copyright (C) 1998 by Alexey A. Dynnikov <aldyn@chat.ru>

              This software is provided 'as-is', without any express or
              implied warranty.  In no event will the author be held liable
              for any  damages arising from the use of this software.

              Permission is granted to anyone to use this software for any
              purpose, including commercial applications, and to alter it
              and redistribute it freely, subject to the following
              restrictions:

              1. The origin of this software must not be misrepresented,
                 you must not claim that you wrote the original software.
                 If you use this software in a product, an acknowledgment
                 in the product documentation would be appreciated but is
                 not required.

              2. Altered source versions must be plainly marked as such, and
                 must not be misrepresented as being the original software.

              3. This notice may not be removed or altered from any source
                 distribution.

 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

unit PerfInstances;

//CE_Desc_Begin(PerfInstances.pas)
{}
//CE_Desc_End

interface

uses
  Windows, SysUtils, Classes,
  PerfTitles, WinPerf, WinPerfUtils,
  PerfObjects;

type
//CE_Desc_Begin(TPerfInstances)
{\
TPerfInstances is a non-visual component designed to obtain \
information about Windows NT Performance instances.

}
//CE_Desc_End
    TPerfInstances = class(TComponent)
    private
        { Private declarations }
        _Objects : TPerfObjects;
        _ObjectName: String;
        _Titles: TPerfTitles;
        procedure SetObjectName(Value: String);
        procedure SetLocaleID(Value: String);
        function GetLocaleID: String;
        function ObjectData: PPerf_Object_Type;
    protected
        { Protected declarations }

        function GetInstanceCount: Integer;
        function GetInstanceName(Index :Integer): String;
        function GetInstanceData(Index :Integer): PPerf_Instance_Definition;
        function GetInstanceUniqueID(Index :Integer): Integer;
    public
        { Public declarations }
        constructor Create(AComponent: TComponent); override;
        destructor Destroy; override;
        property InstanceCount: Integer read GetInstanceCount;
        property InstanceName[index: Integer]: String read GetInstanceName;
        property InstanceData[index: Integer]: PPerf_Instance_Definition read GetInstanceData;
        property InstanceUniqueID[index: Integer]: Integer read GetInstanceUniqueID;

        procedure GetInstanceNames(Titles: TStrings);
        procedure Refresh;
    published
        { Published declarations }
        property ObjectName: String read _ObjectName write SetObjectName;
        property LocaleID: String read GetLocaleID write SetLocaleID;
    end;

procedure DoRegister;

implementation

//CE_Desc_Begin(TPerfInstances.Create)
{\
Create is a constructor that creates and initializes the new \
TPerfInstances object.

}
//CE_Desc_End
constructor TPerfInstances.Create(AComponent: TComponent);
begin
    inherited;
    _Objects := TPerfObjects.Create(nil);
    _Titles := TPerfTitles.Create(nil);
end;

//CE_Desc_Begin(TPerfInstances.Destroy)
{\
Destroy is a destructor that disposes the TPerfInstances \
object and frees used resources.

}
//CE_Desc_End
destructor TPerfInstances.Destroy;
begin
    _Objects.Free;
    _Titles.Free;
    inherited;
end;


//CE_Desc_Begin(TPerfInstances.ObjectName)
{\
ObjectName is a title of the performance object for which \
information about instances is to be retrieved. \
Use <%LINK TPerfObjects%> component to obtain the list of valid \
object names. If the object specified has no instances then \
an Exception will be raised.

}
//CE_Desc_End
procedure TPerfInstances.SetObjectName(Value: String);
begin
    _ObjectName:='';

    if ( csLoading in ComponentState ) or
       ( Value = '' ) or
       ( _Objects.IndexOfObject[Value] >= 0 ) then _ObjectName:=Value;
    if ( csLoading in ComponentState )  then exit;

    if ObjectData.NumInstances < 0 then
    begin
        _ObjectName:='';
        raise Exception.Create(Format('Object "%s" has no instances',[Value]));
    end;
end;


//CE_Desc_Begin(TPerfInstances.LocaleID)
{\
This property is used to control the language of <%LINK ObjectName%> \
property. Use <%LINK Locales%> function to obtain the list of Locales \
for the local computer. Changing of this property causes the change \
of ObjectName from previous language to the language corresponding \
to the LocaleID value.

}
//CE_Desc_End
procedure TPerfInstances.SetLocaleID(Value: String);
var Idx : String;
begin
    if not ( csLoading in ComponentState ) and ( _ObjectName <> '') then
    begin
        Idx := _Titles.IdxOfTitle[_ObjectName];
    end;

    _Objects.LocaleID := Value;
    _Titles.LocaleID := Value;

    if not ( csLoading in ComponentState ) and ( _ObjectName <> '') then
    begin
        ObjectName:=_Titles.TitleByIdx[Idx];
    end;
end;

function TPerfInstances.GetLocaleID: String;
begin
    result:=_Titles.LocaleID;
end;


//CE_Desc_Begin(TPerfInstances.InstanceCount)
{\
This property returns the number of instances for the \
object selected.

}
//CE_Desc_End
function TPerfInstances.GetInstanceCount: Integer;
begin
    result:=ObjectData.NumInstances;
end;


//CE_Desc_Begin(TPerfInstances.InstanceName)
{\
This property returns the name of instance for the index specified. \
Valid values of index are [0..<%LINK InstanceCount%>-1].

}
//CE_Desc_End
function TPerfInstances.GetInstanceName(Index :Integer): String;
begin
    result:=PerfInstName(InstanceData[index]);
end;


//CE_Desc_Begin(TPerfInstances.InstanceUniqueID)
{\
This property return the unique ID for the index specified. \
Valid values of index are [0..<%LINK InstanceCount%>-1]. \
Be accurate when using this property because not all performance \
objects really do have the uniqe id.

}
//CE_Desc_End
function TPerfInstances.GetInstanceUniqueID(Index :Integer): Integer;
begin
    result:=InstanceData[index].UniqueID;
end;


//CE_Desc_Begin(TPerfInstances.InstanceData)
{\
Use this property to get the pointer to the Perf_Instance_Definition \
structure for the instance of performance object with given index. \
Note that the pointer returned may become invalid as a result of \
<%LINK Refresh%> calling. Use this property each time you need a \
pointer to the Perf_Instance_Definition structure to be obtain \
the valid pointer.

}
//CE_Desc_End
function TPerfInstances.GetInstanceData(Index :Integer): PPerf_Instance_Definition;
var i,c: Integer;
    PID: PPerf_Instance_Definition;
    POT: PPerf_Object_Type;
begin
    POT:=ObjectData;
    c:=POT.NumInstances;

    if ( Index < 0 ) or ( Index >= c ) then
        raise Exception.Create('Index out of bounds');
    PID:=FirstInstance(POT);
    for i:=1 to Index do PID:=NextInstance(PID);
    result:=PID;
end;

function TPerfInstances.ObjectData: PPerf_Object_Type;
var idx: integer;
begin
    idx:=_Objects.IndexOfObject[ ObjectName ];
    result:=_Objects.ObjectData[idx];
end;

//CE_Desc_Begin(TPerfInstances.GetInstanceNames)
{\
This procedure is used to fill Titles parameter by names of \
instances of performance object specified by <%LINK ObjectName%> \
property.

}
//CE_Desc_End
procedure TPerfInstances.GetInstanceNames(Titles: TStrings);
var i,c: Integer;
    PID: PPerf_Instance_Definition;
    POT: PPerf_Object_Type;
begin
    POT:=ObjectData;
    c:=POT.NumInstances;

    Titles.Clear;

    PID:=FirstInstance(POT);
    i:=0;
    while i < c do
    begin
        Titles.Add(PerfInstName(PID));
        PID:=NextInstance(PID);
        inc(i);
    end;
end;


//CE_Desc_Begin(TPerfInstances.Refresh)
{\

}
//CE_Desc_End
procedure TPerfInstances.Refresh;
begin
    _Objects.Refresh;
end;



procedure DoRegister;
begin
    RegisterComponents('PerfUtils', [TPerfInstances]);
end;

end.
