{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

Author:       Alexey A. Dynnikov
EMail:        aldyn@chat.ru
WebSite:      http://www.chat.ru/~aldyn/index.html
Support:      Use the e-mail aldyn@chat.ru

Creation:     May 23, 1998
Version:      1.00

Legal issues: Copyright (C) 1998 by Alexey A. Dynnikov <aldyn@chat.ru>

              This software is provided 'as-is', without any express or
              implied warranty.  In no event will the author be held liable
              for any  damages arising from the use of this software.

              Permission is granted to anyone to use this software for any
              purpose, including commercial applications, and to alter it
              and redistribute it freely, subject to the following
              restrictions:

              1. The origin of this software must not be misrepresented,
                 you must not claim that you wrote the original software.
                 If you use this software in a product, an acknowledgment
                 in the product documentation would be appreciated but is
                 not required.

              2. Altered source versions must be plainly marked as such, and
                 must not be misrepresented as being the original software.

              3. This notice may not be removed or altered from any source
                 distribution.

 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}


unit PerfFilter;

//CE_Desc_Begin(PerfFilter.pas)
{}
//CE_Desc_End

interface

{$ifdef ver130}
{$L-}         // The L+ causes internal error in Delphi 5 compiler
{$O-}         // The O+ causes internal error in Delphi 5 compiler
{$Y-}         // The Y+ causes internal error in Delphi 5 compiler
{$endif}

uses
  Windows, SysUtils, Classes, Forms,
  WinPerf, WinPerfUtils, PerfData, PerfTitles;

type
    TPerfCounterItem = class;
    TPerfFilter = class;

//CE_Desc_Begin(TPerfCounterType)
{}
//CE_Desc_End
    TPerfCounterType = class
    end;

    TItemNotifyEvent = procedure (Sender: TPerfFilter; Item: TPerfCounterItem) of Object;

//CE_Desc_Begin(TPerfCounterItem)
{}
//CE_Desc_End
    TPerfCounterItem = class(TCollectionItem)
    private
        _PDB: PPerf_Data_Block;
        _POT : PPerf_OBJECT_TYPE;

        _PCD: PPerf_Counter_Definition;
        _PCD_Base: PPerf_Counter_Definition;

        _PCB: PPerf_Counter_Block;
        _PID: PPerf_Instance_Definition;
    private
        _CounterType: TPerfCounterType;

        _ObjectName: String;
        _ObjectIDX: Integer;
        function GetObjectIDX: Integer;
        procedure ObjectIdxToName;
        procedure ObjectNameToIdx;
        property ObjectIDX: Integer read GetObjectIDX;

    private
        _CounterName: String;
        _CounterIdx: Integer;
        function GetCounterIDX: Integer;
        procedure CounterIdxToName;
        procedure CounterNameToIdx;
        property CounterIDX: Integer read GetCounterIDX;

        procedure AfterChangeLocaleID;
        procedure BeforeChangeLocaleID;

        function Titles: TPerfTitles;
    private
        function GetHelp: String;
        function GetCounterDefinition: PPerf_Counter_Definition;

        function GetInstanceCount: Integer;
        function GetInstanceNames(Index :Integer): String;
        function GetInstanceUniqueID(Index :Integer): Integer;
        procedure SetObjectName(Value: String);
        procedure SetCounterName(Value: String);

        function GetCounterBlock: Boolean;
        procedure ResetCounterBlock;
        procedure CheckCtrBlock;
        procedure CheckCtrBaseBlock;
        procedure CheckIndex(Index: Integer);

        function GetHasInstances: Boolean;
    public
        Data: Pointer;

        property PerfDataBlock: PPerf_Data_Block read _PDB;
        property PerfObjectType : PPerf_OBJECT_TYPE read _POT;

        property PerfCounterDefinition: PPerf_Counter_Definition read _PCD;
        property PerfCounterDefinition_Base: PPerf_Counter_Definition read _PCD_Base;

        property PerfCounterBlock: PPerf_Counter_Block read _PCB;
        property PerfInstanceDefinition: PPerf_Instance_Definition read _PID;

        procedure Assign(Value: TPersistent); override;
        function Filter: TPerfFilter;
        property Help: String read GetHelp;
        property CounterDefinition: PPerf_Counter_Definition read GetCounterDefinition;

        property HasInstances: Boolean read GetHasInstances;
        property InstanceCount: Integer read GetInstanceCount;
        property InstanceNames[index: Integer]: String read GetInstanceNames;
        property InstanceUniqueID[index: Integer]: Integer read GetInstanceUniqueID;

        function ObjectTime: TInt64F;
        function ObjectFreq: TInt64F;

        function CtrAsInteger: TInt64F;
        function CtrBaseAsInteger: TInt64F;
        function CtrAsString: String;

        function InsCtrAsInteger(Index: Integer): TInt64F;
        function InsCtrBaseAsInteger(Index: Integer): TInt64F;
        function HasBase: Boolean;
        function InsCtrAsString(Index: Integer): String;

        constructor Create(Collection: TCollection); override;
        destructor Destroy; override;
    published
//CE_Desc_Begin(TPerfCounterItem.ObjectName)
{}
//CE_Desc_End
        property ObjectName: String read _ObjectName write SetObjectName;
//CE_Desc_Begin(TPerfCounterItem.CounterName)
{}
//CE_Desc_End
        property CounterName: String read _CounterName write SetCounterName;
//CE_Desc_Begin(TPerfCounterItem.CounterType)
{}
//CE_Desc_End
        property CounterType: TPerfCounterType read _CounterType write _CounterType stored False;
    end;

//CE_Desc_Begin(TPerfCounterItems)
{}
//CE_Desc_End
    TPerfCounterItems = class(TCollection)
    private
        _Owner: TPerfFilter;
    protected
        {$IFNDEF VER90}
        function GetOwner: TPersistent; override;
        {$ENDIF}
        function GetCounterItems(index: Integer): TPerfCounterItem;
    public
//CE_Desc_Begin(TPerfCounterItems.Owner)
{}
//CE_Desc_End
        property Owner: TPerfFilter read _Owner;
//CE_Desc_Begin(TPerfCounterItems.Items)
{}
//CE_Desc_End
        property Items[index: Integer]: TPerfCounterItem read GetCounterItems; default;
        function Add: TPerfCounterItem;
        constructor Create(Owner: TPerfFilter; ItemClass: TCollectionItemClass);
    end;

//CE_Desc_Begin(TPerfFilter)
{}
//CE_Desc_End
    TPerfFilter = class(TComponent)
    private
        { Private declarations }
        _OnItemDestroy: TItemNotifyEvent;
        _LocaleID: String;
        _Items: TPerfCounterItems;


        _Titles: TPerfTitles;
        _Helps: TPerfHelps;
        _PerfData: TPerfData;
        procedure InitPerfData;
        procedure DonePerfData;

        procedure SetHelps(Value: TPerfHelps);
        function GetHelps: TPerfHelps;

        procedure SetLocaleID(Value: String);
        procedure SetItems(Value: TPerfCounterItems);

    protected
        { Protected declarations }
        function GetPerfData: PPerf_DATA_BLOCK;
    public
        { Public declarations }
        constructor Create(AComponent: TComponent); override;
        destructor Destroy; override;
        procedure Collect;
        property PerfData: PPerf_DATA_BLOCK read GetPerfData;
    published
        { Published declarations }
//CE_Desc_Begin(TPerfFilter.LocaleID)
{}
//CE_Desc_End
        property LocaleID: String read _LocaleID write SetLocaleID;
//CE_Desc_Begin(TPerfFilter.Items)
{}
//CE_Desc_End
        property Items: TPerfCounterItems read _Items write SetItems;
//CE_Desc_Begin(TPerfFilter.Helps)
{}
//CE_Desc_End
        property Helps: TPerfHelps read GetHelps write SetHelps;
//CE_Desc_Begin(TPerfFilter.OnItemDestroy)
{}
//CE_Desc_End
        property OnItemDestroy: TItemNotifyEvent read _OnItemDestroy write _OnItemDestroy;
    end;

procedure DoRegister;


implementation


//CE_Desc_Begin(TPerfFilter.Create)
{}
//CE_Desc_End
constructor TPerfFilter.Create(AComponent: TComponent);
begin
    inherited;
    _Items := TPerfCounterItems.Create(Self,TPerfCounterItem);
    _Titles := TPerfTitles.Create(nil);
end;

//CE_Desc_Begin(TPerfFilter.Destroy)
{}
//CE_Desc_End
destructor TPerfFilter.Destroy;
begin
    _Items.Free;
    _Titles.Free;
    DonePerfData;
    inherited;
end;

procedure TPerfFilter.SetHelps(Value: TPerfHelps);
begin
    _Helps := Value;
end;

function TPerfFilter.GetHelps: TPerfHelps;
begin
    result := _Helps;
end;

procedure TPerfFilter.InitPerfData;
begin
    if _PerfData <> nil then exit;
    _PerfData := GetGlobalPerfData;
end;

procedure TPerfFilter.DonePerfData;
begin
    if _PerfData = nil then exit;
    ReleaseGlobalPerfData(_PerfData);
end;


//CE_Desc_Begin(TPerfFilter.Collect)
{}
//CE_Desc_End
procedure TPerfFilter.Collect;
var i: Integer;
    PDB: PPerf_Data_Block;
    PCI: TPerfCounterItem;
begin
    InitPerfData;
    _PerfData.Refresh;
    PDB := _PerfData.PerfData;

    for i := 0 to Items.Count-1 do
    begin
        PCI := Items[i];
        PCI.ResetCounterBlock;
        PCI._PDB := PDB;
        if not PCI.GetCounterBlock then PCI.ResetCounterBlock;
    end;
end;

procedure TPerfFilter.SetLocaleID(Value: String);
var i: Integer;
begin
    if not (csLoading in ComponentState) then
    for i := 0 to Items.Count-1 do
        Items[i].BeforeChangeLocaleID;

    _Titles.LocaleID := Value;
    if _Helps <> nil then _Helps.LocaleID := Value;
    _LocaleID := Value;

    if not (csLoading in ComponentState) then
    for i := 0 to Items.Count-1 do
        Items[i].AfterChangeLocaleID;
end;

procedure TPerfFilter.SetItems(Value: TPerfCounterItems);
begin
    Items.Assign(Value);
end;


//CE_Desc_Begin(TPerfFilter.PerfData)
{\
This property returns a pointer to the TPerf_Data_Block.

}
//CE_Desc_End
function TPerfFilter.GetPerfData: PPerf_DATA_BLOCK;
begin
    InitPerfData;
    result:=_PerfData.PerfData;
end;


//-----------------------------------------------------------------------

//CE_Desc_Begin(TPerfCounterItems.Create)
{}
//CE_Desc_End
constructor TPerfCounterItems.Create(Owner: TPerfFilter; ItemClass: TCollectionItemClass);
begin
    inherited Create(ItemClass);
    _Owner := Owner;
end;

{$IFNDEF VER90}
function TPerfCounterItems.GetOwner: TPersistent;
begin
    result := _Owner;
end;
{$ENDIF}


//CE_Desc_Begin(TPerfCounterItems.Add)
{}
//CE_Desc_End
function TPerfCounterItems.Add: TPerfCounterItem;
begin
    result := TPerfCounterItem.Create(Self);
end;

function TPerfCounterItems.GetCounterItems(index: Integer): TPerfCounterItem;
begin
    result := GetItem(index) as TPerfCounterItem;
end;

//-----------------------------------------------------------------------
//CE_Desc_Begin(TPerfCounterItem.Assign)
{}
//CE_Desc_End
procedure TPerfCounterItem.Assign(Value: TPersistent);
begin
    if not (Value is TPerfCounterItem) then
        raise Exception.Create('Incompatible types');
    inherited;
    ObjectName := (Value as TPerfCounterItem)._ObjectName;
end;

procedure TPerfCounterItem.BeforeChangeLocaleID;
begin
    ObjectNameToIdx;
    CounterNameToIdx;
end;

//CE_Desc_Begin(TPerfCounterItem.Data)
{}
//CE_Desc_End

//CE_Desc_Begin(TPerfCounterItem.Help)
{}
//CE_Desc_End

//CE_Desc_Begin(TPerfCounterItem.CounterDefinition)
{}
//CE_Desc_End

//CE_Desc_Begin(TPerfCounterItem.HasInstances)
{}
//CE_Desc_End

//CE_Desc_Begin(TPerfCounterItem.InstanceCount)
{}
//CE_Desc_End

//CE_Desc_Begin(TPerfCounterItem.InstanceNames)
{}
//CE_Desc_End

//CE_Desc_Begin(TPerfCounterItem.InstanceUniqueID)
{}
//CE_Desc_End

procedure TPerfCounterItem.AfterChangeLocaleID;
begin
    ObjectIdxToName;
    CounterIdxToName;
end;

procedure TPerfCounterItem.SetObjectName(Value: String);
begin
    if csLoading in Filter.ComponentState then _ObjectName := Value;

    if AnsiCompareText(_ObjectName,Value) = 0 then exit;

    _ObjectIDX := -2;
    CounterName := '';
    ResetCounterBlock;

    _ObjectName := Value;
    if (_ObjectName = '') or (ObjectIDX >= 0) then exit;

    _ObjectName := '';
    raise Exception.Create('Invalid performance object name');
end;


function TPerfCounterItem.Titles: TPerfTitles;
begin
    result := Filter._Titles;
end;

//CE_Desc_Begin(TPerfCounterItem.Filter)
{}
//CE_Desc_End
function TPerfCounterItem.Filter: TPerfFilter;
begin
    result := (Collection as TPerfCounterItems).Owner;
end;

procedure TPerfCounterItem.SetCounterName(Value: String);
begin
    if csLoading in Filter.ComponentState then _CounterName := Value;

    if AnsiCompareText(Value,_CounterName) = 0 then exit;

    _CounterName := Value;
    _CounterIdx := -2;
    ResetCounterBlock;
    if _CounterName = '' then exit;
    if not GetCounterBlock then
        raise Exception.CreateFmt('Invalid performance counter name : %s', [_CounterName]);
end;

procedure TPerfCounterItem.ObjectIdxToName;
begin
    if _ObjectIDX <> -1 then
        _ObjectName := Titles.TitleByIdx[IntToStr(_ObjectIDX)]
    else
        _ObjectName := '';
end;

procedure TPerfCounterItem.ObjectNameToIdx;
begin
    if ObjectName <> '' then
        _ObjectIDX := StrToInt(Titles.IdxOfTitle[ObjectName])
    else
        _ObjectIDX := -1;
end;

function TPerfCounterItem.GetObjectIDX: Integer;
begin
    if _ObjectIDX < 0 then ObjectNameToIdx;
    result := _ObjectIDX;
end;


procedure TPerfCounterItem.CounterIdxToName;
begin
    if _CounterIDX <> -1 then
        _CounterName := Titles.TitleByIdx[IntToStr(_CounterIDX)]
    else
        _CounterName := '';
end;

procedure TPerfCounterItem.CounterNameToIdx;
begin
    if _CounterName <> '' then
        _CounterIDX := StrToInt(Titles.IdxOfTitle[_CounterName])
    else
        _CounterIDX := -1;
end;

function TPerfCounterItem.GetCounterIDX: Integer;
begin
    if _CounterIDX < 0 then CounterNameToIdx;
    result := _CounterIDX;
end;

function TPerfCounterItem.GetHelp: String;
begin
    result := '';
    GetCounterBlock;
    if _PCD = nil then exit;
    if Filter.Helps = nil then exit;
    result := Filter.Helps.HelpByIdx[IntToStr(_PCD.CounterHelpTitleIndex)];
end;

function TPerfCounterItem.GetCounterDefinition: PPerf_Counter_Definition;
begin
    GetCounterBlock;
    if _PCD = nil then
        raise Exception.Create('Invalid performance counter name');
    result := _PCD;
end;

function TPerfCounterItem.GetInstanceCount: Integer;
begin
    if not HasInstances then
        raise Exception.Create('Performance object has no instances');

    result := _POT.NumInstances;
end;

function TPerfCounterItem.GetInstanceNames(Index :Integer): String;
var PID: PPerf_Instance_Definition;
    i: Integer;
begin
    CheckIndex(Index);

    PID := _PID;
    for i := 1 to Index do PID := NextInstance(PID);

    result := PerfInstName(PID);
end;

function TPerfCounterItem.GetInstanceUniqueID(Index :Integer): Integer;
var PID: PPerf_Instance_Definition;
    i: Integer;
begin
    CheckIndex(Index);

    PID := _PID;
    for i := 1 to Index do PID := NextInstance(PID);

    result := PID.UniqueID;
end;

function TPerfCounterItem.GetHasInstances: Boolean;
begin
    if _POT = nil then GetCounterBlock;
    if _POT = nil then
        raise Exception.Create('Invalid performance object name');
    result := _POT.NumInstances <> Perf_NO_Instances;
end;


function TPerfCounterItem.GetCounterBlock : Boolean;
var i: Integer;
begin
    result := False;

    if ObjectIDX < 0 then exit;
    if CounterIDX < 0 then exit;

    //-----------------------
    if _PDB = nil then
    begin
        Filter.InitPerfData;
        _PDB := Filter._PerfData.PerfData;
    end;


    //-----------------------
    if _POT = nil then
    begin
        _POT := FirstObject(_PDB);
        for i := 1 to _PDB.NumObjectTypes do
        begin
            if Integer(_POT.ObjectNameTitleIndex) = _ObjectIDX then Break;
            _POT := NextObject(_POT);
        end;
        if Integer(_POT.ObjectNameTitleIndex) <> _ObjectIDX then _POT := nil;
    end;
    if _POT = nil then exit;


    //-----------------------
    if _PCD = nil then
    begin
        _PCD := FirstCounter(_POT);
        for i := 1 to _POT.NumCounters do
        begin
            if Integer(_PCD.CounterNameTitleIndex) = _CounterIdx then break;
            _PCD := NextCounter(_PCD);
        end;
        if Integer(_PCD.CounterNameTitleIndex) <> _CounterIdx then _PCD := nil;
    end;
    if _PCD = nil then exit;


    //-----------------------
    if (_PCD.CounterType and CounterSubTypeMask) = Perf_COUNTER_FRACTION then
    begin
        _PCD_Base := NextCounter(_PCD);
        if (_PCD_Base.CounterType and CounterSubTypeMask) <> Perf_COUNTER_BASE then
            _PCD_Base := nil;
    end else
        _PCD_Base := nil;


    //-----------------------
    if _POT.NumInstances <= 0 then
    begin
        _PCB := ObjectBlock(_POT);
    end else
    begin
        _PID := FirstInstance(_POT);
        _PCB := InstanceBlock(_PID);
    end;

    result := True;
end;

procedure TPerfCounterItem.ResetCounterBlock;
begin
    _PDB := nil;
    _POT := nil;

    _PCD := nil;
    _PCD_Base := nil;

    _PID := nil;
    _PCB := nil;
end;

//CE_Desc_Begin(TPerfCounterItem.Create)
{}
//CE_Desc_End
constructor TPerfCounterItem.Create(Collection: TCollection);
begin
    inherited;
    _ObjectIDX := -2;
    _CounterIDX := -2;
end;

//CE_Desc_Begin(TPerfCounterItem.Destroy)
{}
//CE_Desc_End
destructor TPerfCounterItem.Destroy;
begin
    if Assigned(Filter.OnItemDestroy) then Filter.OnItemDestroy(Filter,Self);
    inherited;
end;

//CE_Desc_Begin(TPerfCounterItem.ObjectTime)
{}
//CE_Desc_End
function TPerfCounterItem.ObjectTime: TInt64F;
begin
    CheckCtrBlock;
    result := FInt64(_POT.PerfTime);
end;

//CE_Desc_Begin(TPerfCounterItem.ObjectFreq)
{}
//CE_Desc_End
function TPerfCounterItem.ObjectFreq: TInt64F;
begin
    CheckCtrBlock;
    result := FInt64(_POT.PerfFreq);
end;


//CE_Desc_Begin(TPerfCounterItem.CtrAsInteger)
{}
//CE_Desc_End
function TPerfCounterItem.CtrAsInteger: TInt64F;
var CS: Integer;
begin
    CheckCtrBlock;
    if HasInstances then raise Exception.Create('Instance not specified');
    CS := _PCD.CounterSize and CounterSizeMask;

    if (CS <> PERF_SIZE_DWORD) and (CS <> PERF_SIZE_LARGE) then
        raise Exception.Create('Invalid counter block size');

    if  _PCD.CounterSize = SizeOf(TInt64) then
        result := FInt64(PInt64(DWORD(_PCB) + _PCD.CounterOffset)^)
    else if  _PCD.CounterSize = SizeOf(DWord) then
    begin
        {$ifdef ver110}
        result := FInt64(Int64D(PDWord(DWORD(_PCB) + _PCD.CounterOffset)^));
        {$else}
        result := PDWord(DWORD(_PCB) + _PCD.CounterOffset)^;
        {$endif}
    end else
        raise Exception.Create('Invalid counter block size');
end;

//CE_Desc_Begin(TPerfCounterItem.CtrBaseAsInteger)
{}
//CE_Desc_End
function TPerfCounterItem.CtrBaseAsInteger: TInt64F;
var CS: Integer;
begin
    CheckCtrBaseBlock;
    if HasInstances then raise Exception.Create('Instance not specified');
    CS := _PCD_Base.CounterSize and CounterSizeMask;

    if (CS <> PERF_SIZE_DWORD) and (CS <> PERF_SIZE_LARGE) then
        raise Exception.Create('Invalid counter block size');

    if  _PCD_Base.CounterSize = SizeOf(TInt64) then
        result := FInt64(PInt64(DWORD(_PCB) + _PCD_Base.CounterOffset)^)
    else if  _PCD_Base.CounterSize = SizeOf(DWord) then
    begin
        {$ifdef ver110}
        result := FInt64(Int64D(PDWord(DWORD(_PCB) + _PCD_Base.CounterOffset)^));
        {$else}
        result := PDWord(DWORD(_PCB) + _PCD_Base.CounterOffset)^;
        {$endif}
    end else
        raise Exception.Create('Invalid counter block size');
end;


//CE_Desc_Begin(TPerfCounterItem.CtrAsString)
{}
//CE_Desc_End
function TPerfCounterItem.CtrAsString: String;
begin
    CheckCtrBlock;
    if HasInstances then raise Exception.Create('Instance not specified');
end;


//CE_Desc_Begin(TPerfCounterItem.InsCtrAsInteger)
{}
//CE_Desc_End
function TPerfCounterItem.InsCtrAsInteger(Index: Integer): TInt64F;
var CS: Integer;
    i: Integer;
    PCB_Ins: PPerf_COUNTER_BLOCK;
    PID_Ins: PPerf_Instance_Definition;

begin
    CheckCtrBlock;
    CS := _PCD.CounterSize and CounterSizeMask;

    if (CS <> PERF_SIZE_DWORD) and (CS <> PERF_SIZE_LARGE) then
        raise Exception.Create('Invalid counter block size');

    CheckIndex(Index);

    PID_Ins := _PID;
    for i := 1 to Index do PID_Ins := NextInstance( PID_Ins );
    PCB_Ins := InstanceBlock( PID_Ins );

    if  _PCD.CounterSize = SizeOf(TInt64) then
        result := FInt64(PInt64(DWORD(PCB_Ins) + _PCD.CounterOffset)^)
    else if  _PCD.CounterSize = SizeOf(DWord) then
        {$ifdef ver110}
        result := FInt64(Int64D(PDWord(DWORD(PCB_Ins) + _PCD.CounterOffset)^))
        {$else}
        result := PDWord(DWORD(PCB_Ins) + _PCD.CounterOffset)^
        {$endif}
    else
        raise Exception.Create('Invalid counter block size');
end;

//CE_Desc_Begin(TPerfCounterItem.InsCtrBaseAsInteger)
{}
//CE_Desc_End
function TPerfCounterItem.InsCtrBaseAsInteger(Index: Integer): TInt64F;
var CS: Integer;
    i: Integer;
    PCB_Ins: PPerf_COUNTER_BLOCK;
    PID_Ins: PPerf_Instance_Definition;
begin
    CheckCtrBaseBlock;
    CS := _PCD_Base.CounterSize and CounterSizeMask;

    if (CS <> PERF_SIZE_DWORD) and (CS <> PERF_SIZE_LARGE) then
        raise Exception.Create('Invalid counter block size');

    CheckIndex(Index);

    PID_Ins := _PID;
    for i := 0 to Index - 1 do PID_Ins := NextInstance( PID_Ins );
    PCB_Ins := InstanceBlock( PID_Ins );

    if  _PCD_Base.CounterSize = SizeOf(TInt64) then
        result := FInt64(PInt64(DWORD(PCB_Ins) + _PCD_Base.CounterOffset)^)
    else if  _PCD_Base.CounterSize = SizeOf(DWord) then
        {$ifdef ver110}
        result := FInt64(Int64D(PDWord(DWORD(PCB_Ins) + _PCD_Base.CounterOffset)^))
        {$else}
        result := PDWord(DWORD(PCB_Ins) + _PCD_Base.CounterOffset)^
        {$endif}
    else
        raise Exception.Create('Invalid counter block size');
end;


//CE_Desc_Begin(TPerfCounterItem.InsCtrAsString)
{}
//CE_Desc_End
function TPerfCounterItem.InsCtrAsString(Index: Integer): String;
begin
    CheckCtrBlock;
    if not HasInstances then raise Exception.Create('No instances');
end;

procedure TPerfCounterItem.CheckCtrBlock;
begin
    if ( _PCD = nil ) or ( _PCB = nil ) then
        raise Exception.Create('Invalid counter block');
end;

procedure TPerfCounterItem.CheckCtrBaseBlock;
begin
    if ( _PCD_Base = nil ) or ( _PCB = nil ) then
        raise Exception.Create('Invalid counter base block');
end;

procedure TPerfCounterItem.CheckIndex(Index: Integer);
begin
    if not HasInstances then
        raise Exception.Create(Format('Performance object "%s" has no instances',[ObjectName]));
    if (Index < 0) or (Index >= _POT.NumInstances) then
        raise Exception.Create(Format('Index "%d" out of bounds [%d..%d]',[Index,0,_POT.NumInstances]));
end;

//CE_Desc_Begin(TPerfCounterItem.HasBase)
{}
//CE_Desc_End
function TPerfCounterItem.HasBase: Boolean;
begin
    result := _PCD_Base <> nil;
end;


//-----------------------------------------------------------------------
procedure DoRegister;
begin
  RegisterComponents('PerfUtils', [TPerfFilter]);
end;


end.

