{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

Author:       Alexey A. Dynnikov
EMail:        aldyn@chat.ru
WebSite:      http://www.chat.ru/~aldyn/index.html
Support:      Use the e-mail aldyn@chat.ru

Creation:     May 23, 1998
Version:      1.00

Legal issues: Copyright (C) 1998 by Alexey A. Dynnikov <aldyn@chat.ru>

              This software is provided 'as-is', without any express or
              implied warranty.  In no event will the author be held liable
              for any  damages arising from the use of this software.

              Permission is granted to anyone to use this software for any
              purpose, including commercial applications, and to alter it
              and redistribute it freely, subject to the following
              restrictions:

              1. The origin of this software must not be misrepresented,
                 you must not claim that you wrote the original software.
                 If you use this software in a product, an acknowledgment
                 in the product documentation would be appreciated but is
                 not required.

              2. Altered source versions must be plainly marked as such, and
                 must not be misrepresented as being the original software.

              3. This notice may not be removed or altered from any source
                 distribution.

 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

unit PerfObjects;

//CE_Desc_Begin(PerfObjects.pas)
{\
The TPerfObjects non-visual component is implemented in this unit. \
This component simplifies the access to the \
Windows NT Performance objects. See the Windows API help to \
obtain information about \
Perf_Object_Type structure.

}
//CE_Desc_End

interface

uses
    Windows, SysUtils, Classes,
    WinPerf, WinPerfUtils,
    PerfTitles, PerfData;

type
//CE_Desc_Begin(TPerfObjects)
{\
TPerfObjects is a non-visual component designed to obtain information about \
Windows NT Performance objects.

}
//CE_Desc_End
    TPerfObjects = class(TComponent)
    private
        { Private declarations }
        _Titles: TPerfTitles;
        _Helps: TPerfHelps;
        _PerfData: TPerfData;
        procedure InitPerfData;
        procedure DonePerfData;
        procedure SetLocaleID(Value: String);
        function GetLocaleID: String;
    protected
        { Protected declarations }
        function GetIndexOfObject(ObjectTitle: String): Integer;
        function GetHelpOfObject(ObjectTitle: String): String;

        function GetObjectCount: Integer;
        function GetObjectTitle(index: Integer): String;
        function GetObjectHelp(index: Integer): String;
        function GetObjectData(index: Integer): PPerf_Object_Type;
        function GetPerfData: PPerf_DATA_BLOCK;
    public
        { Public declarations }
        constructor Create(AComponent : TComponent); override;
        destructor Destroy; override;

        property IndexOfObject[ObjectTitle: String]: Integer read GetIndexOfObject;
        property HelpOfObject[ObjectTitle: String]: String read GetHelpOfObject;

        property ObjectCount: Integer read GetObjectCount;
        property ObjectTitle[index: Integer]: String read GetObjectTitle;
        property ObjectHelp[index: Integer]: String read GetObjectHelp;
        property ObjectData[index: Integer]: PPerf_Object_Type read GetObjectData;

        procedure GetObjectTitles(Titles: TStrings);
        procedure GetObjectHelps(Helps: TStrings);
        procedure Refresh;

        function SystemTime : TSystemTime;
        function PerfTime : TInt64;
        function PerfFreq : TInt64;
        function PerfTime100nSec : TInt64;
        function SystemName :String;

        property PerfData: PPerf_DATA_BLOCK read GetPerfData;
    published
        { Published declarations }
        property LocaleID: String read GetLocaleID write SetLocaleID;
        property Helps: TPerfHelps read _Helps write _Helps;
    end;

procedure DoRegister;


implementation


//CE_Desc_Begin(TPerfObjects.Create)
{\
Create is a constructor that creates and initializes a \
new instance of TPerfObjects.

}
//CE_Desc_End
constructor TPerfObjects.Create(AComponent : TComponent);
begin
    inherited;
    _Titles:=TPerfTitles.Create(nil);
end;

//CE_Desc_Begin(TPerfObjects.Destroy)
{\
Destroy is a destructor that disposes an instance of TPerfObjects and \
frees its resources.

}
//CE_Desc_End
destructor TPerfObjects.Destroy;
begin
    _Titles.Free;
    DonePerfData;
    inherited;
end;


//CE_Desc_Begin(TPerfObjects.LocaleID)
{\
It contains a three-digit sting than identifies the required \
language of performance objects titles and helps.
The default value is '009' ( English ).
Use function Locales to obtain a complete list of LocaleID's \
available on the local computer.

}
//CE_Desc_End
procedure TPerfObjects.SetLocaleID(Value: String);
begin
    _Titles.LocaleID:=Value;
end;

function TPerfObjects.GetLocaleID: String;
begin
    result:=_Titles.LocaleID;
end;


//CE_Desc_Begin(TPerfObjects.IndexOfObject)
{\
Determines the index of performance object. \
If the object with title specified is not found then the result is -1.

}
//CE_Desc_End
function TPerfObjects.GetIndexOfObject(ObjectTitle: String): Integer;
var POT: PPerf_Object_Type;
    i,c,idx: DWORD;
begin
    InitPerfData;

    idx:=StrToInt(_Titles.IdxOfTitle[ObjectTitle]);

    POT:=FirstObject(_PerfData.PerfData);

    i:=0;
    c := _PerfData.PerfData.NumObjectTypes;
    while i < c do
    begin
        if POT.ObjectNameTitleIndex = idx then
        begin
            result:=i;
            exit;
        end;
        POT:=NextObject(POT);
        INC(i);
    end;
    raise Exception.Create('Unknown performance object : "'+ObjectTitle+'"');
end;

//CE_Desc_Begin(TPerfObjects.HelpOfObject)
{\
This property returns a help string for the performance object \
with the given title. If the object with title specified \
is not found then result is an empty string.

Note that the <%LINK Helps%> property should be specified to use this method.


}
//CE_Desc_End
function TPerfObjects.GetHelpOfObject(ObjectTitle: String): String;
var i: Integer;
begin
    i := IndexOfObject[ObjectTitle];
    if i < 0 then
        result := ''
    else
        result:=ObjectHelp[i];
end;


//CE_Desc_Begin(TPerfObjects.ObjectCount)
{\
This property returns the number of Performance objects on the local computer.

}
//CE_Desc_End
function TPerfObjects.GetObjectCount: Integer;
begin
    InitPerfData;
    result:=_PerfData.PerfData.NumObjectTypes;
end;

//CE_Desc_Begin(TPerfObjects.ObjectTitle)
{\
ObjectData returns a title string for specific performance object. \
The Index parameter indicates the index of the object. \
The valid values of Index are [0..ObjectCount].

Note that the <%LINK Helps%> property should be specified to use this property.

}
//CE_Desc_End
function TPerfObjects.GetObjectTitle(index: Integer): String;
begin
    result:=_Titles.TitleByIdx[IntToStr(GetObjectData(index).ObjectNameTitleIndex)];
end;

//CE_Desc_Begin(TPerfObjects.ObjectHelp)
{\
ObjectData returns a help string for specific performance object. \
The Index parameter indicates the index of the object. \
The valid values of Index are [0..ObjectCount].

Note that the <%LINK Helps%> property should be specified to use this property.

}
//CE_Desc_End
function TPerfObjects.GetObjectHelp(index: Integer): String;
begin
    if _Helps = nil then
        raise Exception.Create('No help information : object "'+Name+'"');

    result:=_Helps.HelpByIdx[IntToStr(GetObjectData(index).ObjectHelpTitleIndex)];
end;

//CE_Desc_Begin(TPerfObjects.ObjectData)
{\
ObjectData returns a pointer to the TPerf_Object_Type for \
specific performance object. \
The Index parameter indicates the index of the object. \
The valid values of Index are [0..ObjectCount].

}
//CE_Desc_End
function TPerfObjects.GetObjectData(index: Integer): PPerf_Object_Type;
var POT: PPerf_Object_Type;
    i: Integer;
begin
    if (index < 0) or (index >= ObjectCount) then
        raise Exception.Create('Index out of bounds');

    POT:=FirstObject(_PerfData.PerfData);
    i:=0;
    while i < index do
    begin
        POT:=NextObject(POT);
        INC(i);
    end;

    result:=POT;
end;


//CE_Desc_Begin(TPerfObjects.GetObjectTitles)
{\
This method fills the Titles parameter by the list of names of performance objects. \
Call this method to initialize the user-interface oriented components \
such as TMemo or TListBox.

}
//CE_Desc_End
procedure TPerfObjects.GetObjectTitles(Titles: TStrings);
var POT: PPerf_Object_Type;
    i,c: Integer;
begin
    InitPerfData;
    Titles.Clear;
    c := _PerfData.PerfData.NumObjectTypes;
    POT:=FirstObject(_PerfData.PerfData);

    i:=0;
    while i < c do
    begin
        Titles.Add(_Titles.TitleByIdx[IntToStr(POT.ObjectNameTitleIndex)]);
        POT:=NextObject(POT);
        INC(i);
    end;
end;

//CE_Desc_Begin(TPerfObjects.GetObjectHelps)
{\
This method fills the Helps parameter by the list of helps of performance objects. \
Call this method to initialize the user-interface oriented components \
such as TMemo or TListBox.

Note that the <%LINK Helps%> property should be specified to use this method.

}
//CE_Desc_End
procedure TPerfObjects.GetObjectHelps(Helps: TStrings);
var POT: PPerf_Object_Type;
    i,c: Integer;
begin
    InitPerfData;
    if _Helps = nil then
        raise Exception.Create('No help information : object "'+Name+'"');
    Helps.Clear;
    c := _PerfData.PerfData.NumObjectTypes;
    POT:=FirstObject(_PerfData.PerfData);

    i:=0;
    while i < c do
    begin
        Helps.Add(_Helps.HelpByIdx[IntToStr(POT.ObjectHelpTitleIndex)]);
        POT:=NextObject(POT);
        INC(i);
    end;
end;


procedure TPerfObjects.InitPerfData;
begin
    if _PerfData <> nil then exit;
    _PerfData:=GetGlobalPerfData;
end;

procedure TPerfObjects.DonePerfData;
begin
    if _PerfData = nil then exit;
    ReleaseGlobalPerfData(_PerfData);
end;

//CE_Desc_Begin(TPerfObjects.Refresh)
{\
Refresh re-reads the performance information. \
Call it to refresh the performance data.

}
//CE_Desc_End
procedure TPerfObjects.Refresh;
begin
    InitPerfData;
    _PerfData.Refresh;
end;

//CE_Desc_Begin(TPerfObjects.SystemTime)
{\
It returns a value of SystemTime filed of TPerf_Data_Block structure.

}
//CE_Desc_End
function TPerfObjects.SystemTime : TSystemTime;
begin
    InitPerfData;
    result:=_PerfData.PerfData.SystemTime;
end;

//CE_Desc_Begin(TPerfObjects.PerfTime)
{\
It returns the value of PerfTime of TPerf_Data_Block structure.

}
//CE_Desc_End
function TPerfObjects.PerfTime : TInt64;
begin
    InitPerfData;
    result:=_PerfData.PerfData.PerfTime;
end;

//CE_Desc_Begin(TPerfObjects.PerfFreq)
{\
It returns the value of PerfFreq of TPerf_Data_Block structure.

}
//CE_Desc_End
function TPerfObjects.PerfFreq : TInt64;
begin
    InitPerfData;
    result:=_PerfData.PerfData.PerfFreq;
end;

//CE_Desc_Begin(TPerfObjects.PerfTime100nSec)
{\
It returns the value of PerfTime100nSec of TPerf_Data_Block structure.

}
//CE_Desc_End
function TPerfObjects.PerfTime100nSec : TInt64;
begin
    InitPerfData;
    result:=_PerfData.PerfData.PerfTime100nSec;
end;


//CE_Desc_Begin(TPerfObjects.SystemName)
{\
It returns the name of the current computer.

}
//CE_Desc_End
function TPerfObjects.SystemName : String;
begin
    InitPerfData;
    result:=WideCharToString(PWideChar(DWORD(_PerfData.PerfData)+ _PerfData.PerfData.SystemNameOffset));
end;


//CE_Desc_Begin(TPerfObjects.PerfData)
{\
This property returns a pointer to the TPerf_Data_Block.

}
//CE_Desc_End
function TPerfObjects.GetPerfData: PPerf_DATA_BLOCK;
begin
    InitPerfData;
    result:=_PerfData.PerfData;
end;

//CE_Desc_Begin(TPerfObjects.Helps)
{\
It is a reference to the <%LINK TPerfHelps%> component that is used \
to obtain the help strings for the performance objects. \
If this property is not set then the <%LINK ObjectHelp%> property is not available.

}
//CE_Desc_End


//-----------------------------------------------------------------------
procedure DoRegister;
begin
    RegisterComponents('PerfUtils', [TPerfObjects]);
end;


end.
