{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

Author:       Alexey A. Dynnikov
EMail:        aldyn@chat.ru
WebSite:      http://www.chat.ru/~aldyn/index.html
Support:      Use the e-mail aldyn@chat.ru

Creation:     May 23, 1998
Version:      1.00

Legal issues: Copyright (C) 1998 by Alexey A. Dynnikov <aldyn@chat.ru>

              This software is provided 'as-is', without any express or
              implied warranty.  In no event will the author be held liable
              for any  damages arising from the use of this software.

              Permission is granted to anyone to use this software for any
              purpose, including commercial applications, and to alter it
              and redistribute it freely, subject to the following
              restrictions:

              1. The origin of this software must not be misrepresented,
                 you must not claim that you wrote the original software.
                 If you use this software in a product, an acknowledgment
                 in the product documentation would be appreciated but is
                 not required.

              2. Altered source versions must be plainly marked as such, and
                 must not be misrepresented as being the original software.

              3. This notice may not be removed or altered from any source
                 distribution.

 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

unit PerfMonitor;

//CE_Desc_Begin(PerfMonitor.pas)
{}
//CE_Desc_End

interface



uses
  Windows, SysUtils, Classes,
  PerfFilter, WinPerf, WinPerfUtils,
  PerfTitles;

type
    TPerfMonitor = class;



//CE_Desc_Begin(TPerfMonitorCounterValue)
{}
//CE_Desc_End
    TPerfMonitorCounterValue = class
    private
        _CounterType : DWORD;
        _HasBase: Boolean;
        _IsUsed: Boolean;
        _InstanceName: String;
        _ObjectKeyValue: String;
        _OnDestroy: TNotifyEvent;
        _PrevAvailable: Boolean;
        _RefreshCount: Integer;
        _FirstCounter: TInt64F;
        _FirstBase: TInt64F;
        _PrevCounter: TInt64F;
        _PrevBase: TInt64F;
        _Counter: TInt64F;
        _Base: TInt64F;
        _SysTime: TInt64F;
        _DeltaSysTime: TInt64F;
        _TotalSysTime: TInt64F;

        function GetAsFloat: Extended;
        function GetAvgAsFloat: Extended;
        function GetAsInteger: TInt64F;
        function GetAvgAsInteger: TInt64F;
        function GetAsString: String;
        function GetAvgAsString: String;
    public
        property CounterType : DWORD read _CounterType;
        property RefreshCount: Integer read _RefreshCount;
        property FirstCounter: TInt64F read _FirstCounter;
        property FirstBase: TInt64F read _FirstBase;
        property PrevCounter: TInt64F read _PrevCounter;
        property PrevBase: TInt64F read _PrevBase;
        property Counter: TInt64F read _Counter;
        property Base: TInt64F read _Base;
        property HasBase : Boolean read _HasBase;
        property ObjectKeyValue: String read _ObjectKeyValue;
        property InstanceName: String read _InstanceName;
        property OnDestroy: TNotifyEvent read _OnDestroy write _OnDestroy;
        property PrevAvailable: Boolean read _PrevAvailable;
        property AsFloat: Extended read GetAsFloat;
        property AvgAsFloat: Extended read GetAvgAsFloat;
        property AsInteger: TInt64F read GetAsInteger;
        property AvgAsInteger: TInt64F read GetAvgAsInteger;
        property AsString: String read GetAsString;
        property AvgAsString: String read GetAvgAsString;
    end;

//CE_Desc_Begin(TPerfMonitorItem)
{}
//CE_Desc_End
    TPerfMonitorItem = class(TCollectionItem)
    private
        _ObjectKey: String;
        _KeyCounters: TList;
        _InstanceValues: TList;
        _NoInstanceValue : TPerfMonitorCounterValue;
        procedure CreateKeyCounters;
        procedure FreeKeyCounters;
    private
        _CounterName: String;
        _ObjectName: String;
        _Counter: TPerfCounterItem;

        function Counter: TPerfCounterItem;

        function GetObjectName: String;
        procedure SetObjectName(Value: String);

        function GetObjectKey: String;
        procedure SetObjectKey(Value: String);

        function GetCounterName: String;
        procedure SetCounterName(Value: String);

        function GetCounterType: TPerfCounterType;
        procedure SetCounterType(Value: TPerfCounterType);
        procedure Collect;
        function GetCounterDefinition: PPerf_Counter_Definition;
        function GetInstanceCount: Integer;
        function GetInstanceValues(Index: integer): TPerfMonitorCounterValue;

        procedure CollectValue(CV: TPerfMonitorCounterValue; InstanceIdx: Integer; JustCreated: Boolean);
        procedure CollectNoInstanceValue(CV: TPerfMonitorCounterValue; JustCreated: Boolean);
        function GetSuffix: String;
    public
        property CounterDefinition: PPerf_Counter_Definition read GetCounterDefinition;
        function Monitor: TPerfMonitor;
        procedure Assign(Value: TPersistent); override;
        constructor Create(Collection: TCollection); override;
        destructor Destroy; override;

        function HasInstances: Boolean;
        property InstanceCount: Integer read GetInstanceCount;
        property InstanceValues[Index: Integer]: TPerfMonitorCounterValue read GetInstanceValues;
        property Suffix: String read GetSuffix;
    published
        property ObjectName: String read GetObjectName write SetObjectName;
        property CounterName: String read GetCounterName write SetCounterName;
        property CounterType: TPerfCounterType read GetCounterType write SetCounterType stored False;
        property ObjectKey: String read GetObjectKey write SetObjectKey;
    end;

//CE_Desc_Begin(TPerfMonitorItems)
{}
//CE_Desc_End
    TPerfMonitorItems = class(TCollection)
    private
        _Owner: TPerfMonitor;
    protected
        {$IFNDEF VER90}
        function GetOwner: TPersistent; override;
        {$ENDIF}
        function GetMonitorItems(index: Integer): TPerfMonitorItem;
    public
        property Owner: TPerfMonitor read _Owner;
        property Items[index: Integer]: TPerfMonitorItem read GetMonitorItems; default;
        function Add: TPerfMonitorItem;
        constructor Create(Owner: TPerfMonitor; ItemClass: TCollectionItemClass);
    end;

//CE_Desc_Begin(TPerfMonitor)
{}
//CE_Desc_End
    TPerfMonitor = class(TComponent)
    private
        { Private declarations }
        _JustCreated : Boolean;
        _Filter: TPerfFilter;
        _Items: TPerfMonitorItems;
        _SysTime: TInt64F;
        _FirstSysTime: TInt64F;
        _PrevSysTime: TInt64F;
        _DeltaSysTime: TInt64F;
        _TotalSysTime: TInt64F;
        procedure SetItems(Value: TPerfMonitorItems);
    protected
        { Protected declarations }
        function GetLocaleID: String;
        procedure SetLocaleID(Value: String);
        function GetHelps: TPerfHelps;
        procedure SetHelps(Value: TPerfHelps);
    public
        { Public declarations }
        constructor Create(AComponent: TComponent); override;
        destructor Destroy; override;
        procedure Collect;
    published
        { Published declarations }
//CE_Desc_Begin(TPerfMonitor.LocaleID)
{}
//CE_Desc_End
        property LocaleID: String read GetLocaleID write SetLocaleID;
//CE_Desc_Begin(TPerfMonitor.Items)
{}
//CE_Desc_End
        property Items: TPerfMonitorItems read _Items write SetItems;
//CE_Desc_Begin(TPerfMonitor.Helps)
{}
//CE_Desc_End
        property Helps: TPerfHelps read GetHelps write SetHelps;
    end;

procedure ParseKey(Key: String; Keys: TStrings);

procedure DoRegister;

implementation

//------------------------------------------------------------------------------

//CE_Desc_Begin(TPerfMonitor.Create)
{}
//CE_Desc_End
constructor TPerfMonitor.Create(AComponent: TComponent);
begin
    inherited;
    _Filter:=TPerfFilter.Create(Self);
    _Items:=TPerfMonitorItems.Create(Self,TPerfMonitorItem);
    _JustCreated:=True;
end;

//CE_Desc_Begin(TPerfMonitor.Destroy)
{}
//CE_Desc_End
destructor TPerfMonitor.Destroy;
begin
    _Items.Free;
    _Filter.Free;

    inherited;
end;

procedure TPerfMonitor.SetItems(Value: TPerfMonitorItems);
begin
    _Items.Assign(Value);
end;

function TPerfMonitor.GetLocaleID: String;
begin
    result:=_Filter.LocaleID;
end;

procedure TPerfMonitor.SetLocaleID(Value: String);
var i: Integer;
begin
    if not (csLoading in ComponentState) then
        for i:=0 to Items.Count-1 do Items[i].Counter;
    _Filter.LocaleID:=Value;
end;

function TPerfMonitor.GetHelps: TPerfHelps;
begin
    result:=_Filter.Helps;
end;

procedure TPerfMonitor.SetHelps(Value: TPerfHelps);
begin
    _Filter.Helps:=Value;
end;

//CE_Desc_Begin(TPerfMonitor.Collect)
{}
//CE_Desc_End
procedure TPerfMonitor.Collect;
var i : Integer;
    ST: TFileTime;
begin
    _Filter.Collect;
    _PrevSysTime:=_SysTime;

    SystemTimeToFileTime(_Filter.PerfData.SystemTime, ST);
    _SysTime:=FInt64(TInt64(ST));
    if _JustCreated then
    begin
        _FirstSysTime:=_SysTime;
        _JustCreated:=False;
    end;

    _DeltaSysTime := _SysTime - _PrevSysTime;
    _TotalSysTime := _SysTime - _FirstSysTime;

    for i:=0 to Items.Count-1 do
        Items[i].Collect;
end;

//------------------------------------------------------- TPerfMonitorItems ----
//CE_Desc_Begin(TPerfMonitorItems.Add)
{}
//CE_Desc_End
function TPerfMonitorItems.Add: TPerfMonitorItem;
begin
    result:=TPerfMonitorItem.Create(Self);
end;

{$IFNDEF VER90}
function TPerfMonitorItems.GetOwner :TPersistent;
begin
    result:=_Owner;
end;
{$ENDIF}

//CE_Desc_Begin(TPerfMonitorItems.Create)
{}
//CE_Desc_End
constructor TPerfMonitorItems.Create(Owner: TPerfMonitor; ItemClass: TCollectionItemClass);
begin
    inherited Create(ItemClass);
    _Owner:=Owner;
end;

//CE_Desc_Begin(TPerfMonitorItems.Items)
{}
//CE_Desc_End
function TPerfMonitorItems.GetMonitorItems(index: Integer): TPerfMonitorItem;
begin
    result:=GetItem(index) as TPerfMonitorItem;
end;



//--------------------------------------------------------- TPerfMonitorItem ---

//CE_Desc_Begin(TPerfMonitorItem.Assign)
{}
//CE_Desc_End
procedure TPerfMonitorItem.Assign(Value: TPersistent);
begin
    inherited;
    if not (Value is TPerfMonitorItem) then
        raise Exception.Create('Incompatible types');
    ObjectName:=(Value as TPerfMonitorItem).ObjectName;
    CounterName:=(Value as TPerfMonitorItem).CounterName;
end;


//CE_Desc_Begin(TPerfMonitorItem.Create)
{}
//CE_Desc_End
constructor TPerfMonitorItem.Create(Collection: TCollection);
begin
    inherited;
    _InstanceValues:=TList.Create;
end;


//CE_Desc_Begin(TPerfMonitorItem.Destroy)
{}
//CE_Desc_End
destructor TPerfMonitorItem.Destroy;
var i: Integer;
begin
    FreeKeyCounters;
    _Counter.Free;
    for i:=0 to _InstanceValues.Count-1 do TObject(_InstanceValues[i]).Free;
    _InstanceValues.Free;
    _NoInstanceValue.Free;
    inherited;
end;

//CE_Desc_Begin(TPerfMonitorItem.ObjectName)
{}
//CE_Desc_End
function TPerfMonitorItem.GetObjectName: String;
begin
    if csLoading in Monitor.ComponentState then
        result:=_ObjectName
    else
        result:=Counter.ObjectName;
end;

procedure TPerfMonitorItem.SetObjectName(Value: String);
begin
    _ObjectName:=Value;
    if not (csLoading in Monitor.ComponentState) then
        Counter.ObjectName:=Value;
end;


//CE_Desc_Begin(TPerfMonitorItem.CounterName)
{}
//CE_Desc_End
function TPerfMonitorItem.GetCounterName: String;
begin
    if csLoading in Monitor.ComponentState then
        result:=_CounterName
    else
        result:=Counter.CounterName;
end;

procedure TPerfMonitorItem.SetCounterName(Value: String);
begin
    _CounterName:=Value;
    if not (csLoading in Monitor.ComponentState) then
        Counter.CounterName:=Value;
end;

//CE_Desc_Begin(TPerfMonitorItem.Monitor)
{}
//CE_Desc_End
function TPerfMonitorItem.Monitor: TPerfMonitor;
begin
    result:=(Collection as TPerfMonitorItems)._Owner;
end;

//CE_Desc_Begin(TPerfMonitorItem.CounterType)
{}
//CE_Desc_End
function TPerfMonitorItem.GetCounterType: TPerfCounterType;
begin
    result:=Counter.CounterType;
end;

procedure TPerfMonitorItem.SetCounterType(Value: TPerfCounterType);
begin
    Counter.CounterType:=Value;
end;

//CE_Desc_Begin(TPerfMonitorItem.ObjectKey)
{}
//CE_Desc_End
function TPerfMonitorItem.GetObjectKey: String;
var i: Integer;
begin
    if csLoading in Monitor.ComponentState then
        result:=_ObjectKey
    else
    begin
        CreateKeyCounters;
        result:='';
        for i:=0 to _KeyCounters.Count - 1 do
        begin
            if i > 0 then result:=result+';';
            result:=result+TPerfCounterItem(_KeyCounters[i]).CounterName;
        end;
    end;
end;

procedure TPerfMonitorItem.SetObjectKey(Value: String);
begin
    if _ObjectKey = Value then exit;
    _ObjectKey := Value;

    if not (csLoading in Monitor.ComponentState) then
    try
        FreeKeyCounters;
        CreateKeyCounters;
    except
        ObjectKey:='';
        raise;
    end;
end;

function TPerfMonitorItem.Counter: TPerfCounterItem;
begin
    if _Counter = nil then
    begin
        _Counter:=Monitor._Filter.Items.Add as TPerfCounterItem;
        _Counter.ObjectName:=_ObjectName;
        _Counter.CounterName:=_CounterName;
//        Collect;
    end;
    result:=_Counter;
end;

procedure TPerfMonitorItem.Collect;
var i,j: Integer;
    CV: TPerfMonitorCounterValue;
    CI: TPerfCounterItem;
    S,SN: String;
begin
    if not HasInstances then
    begin
        if _NoInstanceValue = nil then
        begin
            _NoInstanceValue:=TPerfMonitorCounterValue.Create;
            CollectNoInstanceValue(_NoInstanceValue,True);
        end;
        CollectNoInstanceValue(_NoInstanceValue,False);
        exit;
    end;
    CreateKeyCounters;

    // Mark all values as Unused
    for i:=0 to _InstanceValues.Count-1 do
        TPerfMonitorCounterValue(_InstanceValues[i])._IsUsed:=False;

    for j:=0 to Counter.InstanceCount-1 do
    begin
        S:='';
        for i:=0 to _KeyCounters.Count-1 do
        begin
            CI:=TPerfCounterItem(_KeyCounters[i]);
            if i > 0 then S:=S+';';
            S:=S+FormatFloat('0',CI.InsCtrAsInteger(j));
        end;
        SN:=Counter.InstanceNames[j];

        CV:=nil;
        for i:=0 to _InstanceValues.Count-1 do
        begin
            CV:=TPerfMonitorCounterValue(_InstanceValues[i]);
            if ( AnsiCompareText(CV._InstanceName,SN) = 0 ) and
               ( AnsiCompareText(CV._ObjectKeyValue,S) = 0 ) then break;
            CV:=nil;
        end;

        if CV = nil then
        begin
            CV:=TPerfMonitorCounterValue.Create;
            CV._InstanceName:=SN;
            CV._ObjectKeyValue:=S;
            CV._PrevAvailable:=False;
            CV._HasBase:=Counter.HasBase;
            _InstanceValues.Add(CV);
            CollectValue(CV,j,True);
        end else
        CollectValue(CV,j,False);
        CV._IsUsed:=True;
    end;

    // Free unused values
    for i:=_InstanceValues.Count-1 downto 0 do
    begin
        CV:=TPerfMonitorCounterValue(_InstanceValues[i]);
        if CV._IsUsed then continue;
        CV.Free;
        _InstanceValues.Remove(CV);
    end;
end;

procedure TPerfMonitorItem.CollectValue(CV: TPerfMonitorCounterValue; InstanceIdx: Integer; JustCreated: Boolean);
var CD: PPerf_Counter_Definition;
begin
    CD:=GetCounterDefinition;
    if (CD.CounterType and Perf_DELTA_COUNTER) <> 0 then
    begin
        if not JustCreated then
        begin
            CV._PrevCounter:=CV.Counter;
            if CV.HasBase then CV._PrevBase:=CV.Base;
            CV._PrevAvailable:=True;
        end;
    end;

    CV._SysTime:=TPerfMonitorItems(Collection).Owner._SysTime;
    CV._DeltaSysTime:=TPerfMonitorItems(Collection).Owner._DeltaSysTime;
    CV._TotalSysTime:=TPerfMonitorItems(Collection).Owner._TotalSysTime;
    CV._Counter:=Counter.InsCtrAsInteger(InstanceIdx);
    CV._CounterType := CD.CounterType;
    if CV.HasBase then CV._Base:=Counter.InsCtrBaseAsInteger(InstanceIdx);
    Inc(CV._RefreshCount);


    if (CD.CounterType and Perf_DELTA_COUNTER) <> 0 then
    begin
        if JustCreated then
        begin
            CV._FirstCounter:=CV.Counter;
            if CV.HasBase then CV._FirstBase:=CV.Base;
        end;
    end;
end;

procedure TPerfMonitorItem.CollectNoInstanceValue(CV: TPerfMonitorCounterValue; JustCreated: Boolean);
var CD: PPerf_Counter_Definition;
begin
    CD:=GetCounterDefinition;
    if (CD.CounterType and Perf_DELTA_COUNTER) <> 0 then
    begin
        if not JustCreated then
        begin
            CV._PrevCounter:=CV.Counter;
            if CV.HasBase then CV._PrevBase:=CV.Base;
            CV._PrevAvailable:=True;
        end;
    end;

    CV._SysTime:=TPerfMonitorItems(Collection).Owner._SysTime;
    CV._DeltaSysTime:=TPerfMonitorItems(Collection).Owner._DeltaSysTime;
    CV._TotalSysTime:=TPerfMonitorItems(Collection).Owner._TotalSysTime;
    CV._Counter:=Counter.CtrAsInteger;
    CV._CounterType := CD.CounterType;
    if CV.HasBase then CV._Base:=Counter.CtrBaseAsInteger;
    Inc(CV._RefreshCount);

    if (CD.CounterType and Perf_DELTA_COUNTER) <> 0 then
    begin
        if JustCreated then
        begin
            CV._FirstCounter:=CV.Counter;
            if CV.HasBase then CV._FirstBase:=CV.Base;
        end;
    end;
end;


procedure TPerfMonitorItem.CreateKeyCounters;
var i: Integer;
    SL : TStringList;
    CI: TPerfCounterItem;
begin
    if _KeyCounters <> nil then exit;

    _KeyCounters:=TList.Create;

    SL:=TStringList.Create;
    ParseKey(_ObjectKey,SL);

    try
        for i:=0 to SL.Count-1 do
        begin
            CI:=Monitor._Filter.Items.Add;
            CI.ObjectName:=ObjectName;
            CI.CounterName:=SL[i];
            _KeyCounters.Add(CI);
        end;
    finally
        SL.Free;
    end;
end;

procedure TPerfMonitorItem.FreeKeyCounters;
var i: Integer;
begin
    if _KeyCounters = nil then exit;
    for i:=0 to _KeyCounters.Count-1 do TObject(_KeyCounters[i]).Free;
    _KeyCounters.Free;
    _KeyCounters:=nil;
end;

//CE_Desc_Begin(TPerfMonitorItem.CounterDefinition)
{}
//CE_Desc_End
function TPerfMonitorItem.GetCounterDefinition: PPerf_Counter_Definition;
begin
    result:=Counter.CounterDefinition;
end;

//CE_Desc_Begin(TPerfMonitorItem.HasInstances)
{}
//CE_Desc_End
function TPerfMonitorItem.HasInstances: Boolean;
begin
    result:=Counter.HasInstances;
end;


//CE_Desc_Begin(TPerfMonitorItem.InstanceCount)
{}
//CE_Desc_End
function TPerfMonitorItem.GetInstanceCount: Integer;
begin
    result:=Counter.InstanceCount;
end;

//CE_Desc_Begin(TPerfMonitorItem.InstanceValues)
{}
//CE_Desc_End
function TPerfMonitorItem.GetInstanceValues(Index: integer): TPerfMonitorCounterValue;
begin
    if not HasInstances then
        result:=_NoInstanceValue
    else
        result:=TPerfMonitorCounterValue(_InstanceValues[Index]);
end;

function TPerfMonitorItem.GetSuffix: String;
begin
    case CounterDefinition.CounterType and CounterDisplayMask of
        Perf_DISPLAY_NO_SUFFIX  : result:='';
        Perf_DISPLAY_PER_SEC    : result:='/sec';
        Perf_DISPLAY_PERCENT    : result:='%';
        Perf_DISPLAY_SECONDS    : result:='secs';
        Perf_DISPLAY_NOSHOW     : result:='';
    end;
end;



//------------------------------------------------------------------------------
//CE_Desc_Begin(TPerfMonitorCounterValue.RefreshCount)
{}
//CE_Desc_End

//CE_Desc_Begin(TPerfMonitorCounterValue.FirstCounter)
{}
//CE_Desc_End

//CE_Desc_Begin(TPerfMonitorCounterValue.FirstBase)
{}
//CE_Desc_End

//CE_Desc_Begin(TPerfMonitorCounterValue.PrevCounter)
{}
//CE_Desc_End

//CE_Desc_Begin(TPerfMonitorCounterValue.PrevBase)
{}
//CE_Desc_End

//CE_Desc_Begin(TPerfMonitorCounterValue.Counter)
{}
//CE_Desc_End

//CE_Desc_Begin(TPerfMonitorCounterValue.Base)
{}
//CE_Desc_End

//CE_Desc_Begin(TPerfMonitorCounterValue.HasBase)
{}
//CE_Desc_End

//CE_Desc_Begin(TPerfMonitorCounterValue.ObjectKeyValue)
{}
//CE_Desc_End

//CE_Desc_Begin(TPerfMonitorCounterValue.InstanceName)
{}
//CE_Desc_End

//CE_Desc_Begin(TPerfMonitorCounterValue.OnDestroy)
{}
//CE_Desc_End

//CE_Desc_Begin(TPerfMonitorCounterValue.PrevAvailable)
{}
//CE_Desc_End

function TPerfMonitorCounterValue.GetAsFloat: Extended;
var Ctr,CtrBase: TInt64F;
    CurrentTime: TFileTime;
begin
    result:=0;
    case CounterType and CounterTypeMask of
        Perf_TYPE_NUMBER : result := Counter;

        Perf_TYPE_COUNTER :
        begin
            if CounterType and Perf_DELTA_COUNTER = 0 then
                Ctr := Counter
            else
                Ctr := Counter - PrevCounter;

            CtrBase := 0;
            
            if _HasBase then
            if CounterType and Perf_DELTA_COUNTER = 0 then
                CtrBase := Base
            else
                CtrBase := Base - PrevBase;

            case CounterType and CounterSubTypeMask of
                Perf_COUNTER_VALUE      : result := Ctr;

                Perf_COUNTER_RATE       :
                    if _DeltaSysTime = 0 then
                        result := 0
                    else
                        result := Ctr / _DeltaSysTime;     // divide ctr / delta time

                Perf_COUNTER_FRACTION   :
                    if CtrBase = 0 then
                        result := 0
                    else
                        result := Ctr / CtrBase;     // divide ctr / base
                Perf_COUNTER_BASE : raise Exception.Create('PerfUtils Internal error (Perf_COUNTER_BASE)');
                Perf_COUNTER_ELAPSED    :       // subtract counter from current time
                    begin
                        GetSystemTimeAsFileTime(CurrentTime);
                        result := (FInt64(TInt64(CurrentTime)) - Ctr) / 1e7;
                    end;
                Perf_COUNTER_QUEUELEN   : result := -4;     // Use Queuelen processing func.
                Perf_COUNTER_HISTOGRAM  : result := -5;     // Counter begins or ends a histogram
            end;
        end;

        Perf_TYPE_TEXT :
            raise Exception.Create('Invalid counter type');

        Perf_TYPE_ZERO : result:=0;
    end;

    if (CounterType and Perf_INVERSE_COUNTER) <> 0 then result:=1-result;

    case CounterType and CounterDisplayMask of
        Perf_DISPLAY_PER_SEC    : result:=result*1e7;
        Perf_DISPLAY_PERCENT    : result:=result*100;
//        Perf_DISPLAY_SECONDS    : result:='secs';
    end;

end;

function TPerfMonitorCounterValue.GetAvgAsFloat: Extended;
var Ctr,CtrBase: TInt64F;
    CurrentTime: TFileTime;
begin
    CtrBase:=0;

    result:=0;
    case CounterType and CounterTypeMask of
        Perf_TYPE_NUMBER : result := Counter;

        Perf_TYPE_COUNTER :
        begin
            if CounterType and Perf_DELTA_COUNTER = 0 then
                Ctr := Counter
            else
                Ctr := Counter - FirstCounter;

            if _HasBase then
            if CounterType and Perf_DELTA_COUNTER = 0 then
                CtrBase := Base
            else
                CtrBase := Base - FirstBase;

            case CounterType and CounterSubTypeMask of
                Perf_COUNTER_VALUE      : result := Ctr;

                Perf_COUNTER_RATE       :
                    if _TotalSysTime = 0 then
                        result := 0
                    else
                        result := Ctr / _TotalSysTime;     // divide ctr / delta time

                Perf_COUNTER_FRACTION   :
                    if CtrBase = 0 then
                        result := 0
                    else
                        result := Ctr / CtrBase;     // divide ctr / base
                Perf_COUNTER_BASE : raise Exception.Create('PerfUtils Internal error (Perf_COUNTER_BASE)');
                Perf_COUNTER_ELAPSED    :       // subtract counter from current time
                    begin
                        GetSystemTimeAsFileTime(CurrentTime);
                        result := (FInt64(TInt64(CurrentTime)) - Ctr) / 1e7;
                    end;
                Perf_COUNTER_QUEUELEN   : result := -4;     // Use Queuelen processing func.
                Perf_COUNTER_HISTOGRAM  : result := -5;     // Counter begins or ends a histogram
            end;
        end;

        Perf_TYPE_TEXT :
            raise Exception.Create('Invalid counter type');

        Perf_TYPE_ZERO : result:=0;
    end;

    if (CounterType and Perf_INVERSE_COUNTER) <> 0 then result:=1-result;

    case CounterType and CounterDisplayMask of
        Perf_DISPLAY_PER_SEC    : result:=result*1e7;
        Perf_DISPLAY_PERCENT    : result:=result*100;
//        Perf_DISPLAY_SECONDS    : result:='secs';
    end;

end;

function TPerfMonitorCounterValue.GetAsInteger: TInt64F;
begin
    result:=0;
    case CounterType and CounterTypeMask of
        Perf_TYPE_NUMBER : result := Counter;

        Perf_TYPE_COUNTER :
        case CounterType and CounterSubTypeMask of
            Perf_COUNTER_VALUE      : result := Counter;
            Perf_COUNTER_RATE       : result := -1;     // divide ctr / delta time
            Perf_COUNTER_FRACTION   : result := -2;     // divide ctr / base
            Perf_COUNTER_BASE : raise Exception.Create('PerfUtils Internal error (Perf_COUNTER_BASE)');
            Perf_COUNTER_ELAPSED    : result := -3;     // subtract counter from current time
            Perf_COUNTER_QUEUELEN   : result := -4;     // Use Queuelen processing func.
            Perf_COUNTER_HISTOGRAM  : result := -5;     // Counter begins or ends a histogram
        end;

        Perf_TYPE_TEXT :
            raise Exception.Create('Invalid counter type');

        Perf_TYPE_ZERO : result:=0;
    end;
end;

function TPerfMonitorCounterValue.GetAvgAsInteger: TInt64F;
begin
    result:=0;
    case CounterType and CounterTypeMask of
        Perf_TYPE_NUMBER : result := Counter;

        Perf_TYPE_COUNTER :
        case CounterType and CounterSubTypeMask of
            Perf_COUNTER_VALUE      : result := Counter;
            Perf_COUNTER_RATE       : result := -1;     // divide ctr / delta time
            Perf_COUNTER_FRACTION   : result := -2;     // divide ctr / base
            Perf_COUNTER_BASE : raise Exception.Create('PerfUtils Internal error (Perf_COUNTER_BASE)');
            Perf_COUNTER_ELAPSED    : result := -3;     // subtract counter from current time
            Perf_COUNTER_QUEUELEN   : result := -4;     // Use Queuelen processing func.
            Perf_COUNTER_HISTOGRAM  : result := -5;     // Counter begins or ends a histogram
        end;

        Perf_TYPE_TEXT :
            raise Exception.Create('Invalid counter type');

        Perf_TYPE_ZERO : result:=0;
    end;
end;

function TPerfMonitorCounterValue.GetAsString: String;
begin

    case CounterType and CounterTypeMask of
        Perf_TYPE_NUMBER :
        case CounterType and CounterDisplayMask of
            Perf_NUMBER_HEX : result:=Int64ToHex(AsInteger,1);

            Perf_NUMBER_DECIMAL : result:=Format('%.0f',[AsFloat]);

            Perf_NUMBER_DEC_1000 : result:=Format('%.3f',[AsFloat/1000]);
        end;


        Perf_TYPE_COUNTER : result:=Format('%f',[AsFloat]);


        Perf_TYPE_TEXT :
            raise Exception.Create('Invalid counter type');

        Perf_TYPE_ZERO : result:='0';
    end;
end;

function TPerfMonitorCounterValue.GetAvgAsString: String;
begin

    case CounterType and CounterTypeMask of
        Perf_TYPE_NUMBER :
        case CounterType and CounterDisplayMask of
            Perf_NUMBER_HEX : result:=Int64ToHex(AsInteger,1);

            Perf_NUMBER_DECIMAL : result:=Format('%.0f',[AvgAsFloat]);

            Perf_NUMBER_DEC_1000 : result:=Format('%.3f',[AvgAsFloat/1000]);
        end;


        Perf_TYPE_COUNTER : result:=Format('%f',[AvgAsFloat]);


        Perf_TYPE_TEXT :
            raise Exception.Create('Invalid counter type');

        Perf_TYPE_ZERO : result:='0';
    end;
end;



//------------------------------------------------------------------------------

//CE_Desc_Begin(ParseKey)
{}
//CE_Desc_End
procedure ParseKey(Key: String; Keys: TStrings);
var i,L: Integer;
    S: String;
begin
    Keys.Clear;
    L:=Length(Key);
    i:=1;
    S:='';
    while i <= L do
    begin
        if Key[i] = ';' then
        begin
            if S <> '' then Keys.Add(S);
            S:='';
        end else
            S:=S+Key[i];
        INC(i);
    end;
    if S <> '' then Keys.Add(S);
end;


procedure DoRegister;
begin
  RegisterComponents('PerfUtils', [TPerfMonitor]);
end;

end.
