{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

Author:       Alexey A. Dynnikov
EMail:        aldyn@chat.ru
WebSite:      http://www.chat.ru/~aldyn/index.html
Support:      Use the e-mail aldyn@chat.ru

Creation:     May 23, 1998
Version:      1.00

Legal issues: Copyright (C) 1998 by Alexey A. Dynnikov <aldyn@chat.ru>

              This software is provided 'as-is', without any express or
              implied warranty.  In no event will the author be held liable
              for any  damages arising from the use of this software.

              Permission is granted to anyone to use this software for any
              purpose, including commercial applications, and to alter it
              and redistribute it freely, subject to the following
              restrictions:

              1. The origin of this software must not be misrepresented,
                 you must not claim that you wrote the original software.
                 If you use this software in a product, an acknowledgment
                 in the product documentation would be appreciated but is
                 not required.

              2. Altered source versions must be plainly marked as such, and
                 must not be misrepresented as being the original software.

              3. This notice may not be removed or altered from any source
                 distribution.

 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

unit PerfCounters;

//CE_Desc_Begin(PerfCounters.pas)
{\
The TPerfObjects non-visual component is implemented in this unit. \
This component simplifies the access to the \
Windows NT Performance counters. See the Windows API help to \
obtain information about \
Perf_Counter_Definition structure.

}
//CE_Desc_End

interface

uses
  Windows, SysUtils, Classes,
  PerfTitles, WinPerf, WinPerfUtils, PerfData,
  PerfObjects;

type
//CE_Desc_Begin(TPerfCounters)
{}
//CE_Desc_End
    TPerfCounters = class(TComponent)
    private
        { Private declarations }
        _Objects : TPerfObjects;
        _ObjectName: String;
        _Helps: TPerfHelps;
        _Titles: TPerfTitles;
        _PerfData: TPerfData;
        procedure InitPerfData;
        procedure DonePerfData;
        procedure SetObjectName(Value: String);
        procedure SetLocaleID(Value: String);
        function GetLocaleID: String;
        function ObjectData: PPerf_Object_Type;
    protected
        { Protected declarations }

        function GetCounterCount: Integer;
        function GetCounterTitle(Index :Integer): String;
        function GetCounterHelp(Index :Integer): String;
        function GetCounterData(Index :Integer): PPerf_Counter_Definition;
        function GetPerfData: PPerf_DATA_BLOCK;
    public
        { Public declarations }
        constructor Create(AComponent: TComponent); override;
        destructor Destroy; override;
        property CounterCount: Integer read GetCounterCount;
        property CounterTitle[index: Integer]: String read GetCounterTitle;
        property CounterHelp[index: Integer]: String read GetCounterHelp;
        property CounterData[index: Integer]: PPerf_Counter_Definition read GetCounterData;

        property PerfData: PPerf_DATA_BLOCK read GetPerfData;

        function IndexOfTitle(CounterTitle: String): Integer;
        procedure GetCounterTitles(Titles: TStrings);
        procedure GetCounterHelps(Helps: TStrings);
        procedure Refresh;
    published
        { Published declarations }
        property ObjectName: String read _ObjectName write SetObjectName;
        property LocaleID: String read GetLocaleID write SetLocaleID;
        property Helps: TPerfHelps read _Helps write _Helps;
    end;

procedure DoRegister;

implementation

//CE_Desc_Begin(TPerfCounters.Helps)
{}
//CE_Desc_End

//CE_Desc_Begin(TPerfCounters.Create)
{}
//CE_Desc_End
constructor TPerfCounters.Create(AComponent: TComponent);
begin
    inherited;
    _Objects := TPerfObjects.Create(nil);
    _Titles := TPerfTitles.Create(nil);
end;

//CE_Desc_Begin(TPerfCounters.Destroy)
{}
//CE_Desc_End
destructor TPerfCounters.Destroy;
begin
    _Objects.Free;
    _Titles.Free;
    DonePerfData;
    inherited;
end;

//CE_Desc_Begin(TPerfCounters.ObjectName)
{}
//CE_Desc_End
procedure TPerfCounters.SetObjectName(Value: String);
begin
    _ObjectName:='';

    if ( csLoading in ComponentState ) or
       ( Value = '' ) or
       ( _Objects.IndexOfObject[Value] >= 0 ) then _ObjectName:=Value;
end;

//CE_Desc_Begin(TPerfCounters.LocaleID)
{}
//CE_Desc_End
procedure TPerfCounters.SetLocaleID(Value: String);
var Idx : String;
begin
    if not ( csLoading in ComponentState ) and ( _ObjectName <> '') then
    begin
        Idx := _Titles.IdxOfTitle[_ObjectName];
    end;

    _Objects.LocaleID := Value;
    _Titles.LocaleID := Value;

    if not ( csLoading in ComponentState ) and ( _ObjectName <> '') then
    begin
        ObjectName:=_Titles.TitleByIdx[Idx];
    end;
end;

function TPerfCounters.GetLocaleID: String;
begin
    result:=_Titles.LocaleID;
end;


//CE_Desc_Begin(TPerfCounters.CounterCount)
{}
//CE_Desc_End
function TPerfCounters.GetCounterCount: Integer;
begin
    result:=ObjectData.NumCounters;
end;


//CE_Desc_Begin(TPerfCounters.CounterCount)
{}
//CE_Desc_End
function TPerfCounters.GetCounterTitle(Index :Integer): String;
begin
    result:=_Titles.TitleByIdx[IntToStr(CounterData[index].CounterNameTitleIndex)];
end;


//CE_Desc_Begin(TPerfCounters.CounterHelp)
{}
//CE_Desc_End
function TPerfCounters.GetCounterHelp(Index :Integer): String;
begin
    if _Helps = nil then
        raise Exception.Create('No help information : object "'+Name+'"');
    result:=_Helps.HelpByIdx[IntToStr(CounterData[index].CounterHelpTitleIndex)];
end;

//CE_Desc_Begin(TPerfCounters.CounterData)
{}
//CE_Desc_End
function TPerfCounters.GetCounterData(Index :Integer): PPerf_Counter_Definition;
var i,c: Integer;
    PCD: PPerf_Counter_Definition;
    POT: PPerf_Object_Type;
begin
    POT:=ObjectData;
    c:=POT.NumCounters;

    if ( Index < 0 ) or ( Index >= c ) then
        raise Exception.Create('Index out of bounds');
    PCD:=FirstCounter(POT);
    i:=0;
    while i < Index do
    begin
        PCD:=NextCounter(PCD);
        inc(i);
    end;
    result:=PCD;
end;

function TPerfCounters.ObjectData: PPerf_Object_Type;
var idx: integer;
begin
    idx:=_Objects.IndexOfObject[ ObjectName ];
    result:=_Objects.ObjectData[idx];
end;

//CE_Desc_Begin(TPerfCounters.IndexOfTitle)
{}
//CE_Desc_End
function TPerfCounters.IndexOfTitle(CounterTitle: String): Integer;
var i,c: Integer;
    PCD: PPerf_Counter_Definition;
    POT: PPerf_Object_Type;
begin
    POT:=ObjectData;
    c:=POT.NumCounters;

    PCD:=FirstCounter(POT);
    i:=0;
    while i < c do
    begin
        if AnsiCompareText(CounterTitle,_Titles.TitleByIdx[IntToStr(PCD.CounterNameTitleIndex)]) = 0 then
        begin
            result:=i;
            exit;
        end;
        PCD:=NextCounter(PCD);
        inc(i);
    end;
    result:=-1;
end;


//CE_Desc_Begin(TPerfCounters.GetCounterTitles)
{}
//CE_Desc_End
procedure TPerfCounters.GetCounterTitles(Titles: TStrings);
var i,c: Integer;
    PCD: PPerf_Counter_Definition;
    POT: PPerf_Object_Type;
begin
    POT:=ObjectData;
    c:=POT.NumCounters;

    Titles.Clear;

    PCD:=FirstCounter(POT);
    i:=0;
    while i < c do
    begin
        Titles.Add(_Titles.TitleByIdx[IntToStr(PCD.CounterNameTitleIndex)]);
        PCD:=NextCounter(PCD);
        inc(i);
    end;
end;

//CE_Desc_Begin(TPerfCounters.GetCounterHelps)
{}
//CE_Desc_End
procedure TPerfCounters.GetCounterHelps(Helps: TStrings);
var i,c: Integer;
    PCD: PPerf_Counter_Definition;
    POT: PPerf_Object_Type;
begin
    if _Helps = nil then
        raise Exception.Create('No help information : object "'+Name+'"');

    POT:=ObjectData;
    c:=POT.NumCounters;

    Helps.Clear;

    PCD:=FirstCounter(POT);
    i:=0;
    while i < c do
    begin
        Helps.Add(_Helps.HelpByIdx[IntToStr(PCD.CounterHelpTitleIndex)]);
        PCD:=NextCounter(PCD);
        inc(i);
    end;
end;

//CE_Desc_Begin(TPerfCounters.Refresh)
{}
//CE_Desc_End
procedure TPerfCounters.Refresh;
begin
    _Objects.Refresh;
end;

procedure TPerfCounters.InitPerfData;
begin
    if _PerfData <> nil then exit;
    _PerfData:=GetGlobalPerfData;
end;

procedure TPerfCounters.DonePerfData;
begin
    if _PerfData = nil then exit;
    ReleaseGlobalPerfData(_PerfData);
end;

//CE_Desc_Begin(TPerfCounters.PerfData)
{\
This property returns a pointer to the TPerf_Data_Block.

}
//CE_Desc_End
function TPerfCounters.GetPerfData: PPerf_DATA_BLOCK;
begin
    InitPerfData;
    result:=_PerfData.PerfData;
end;


procedure DoRegister;
begin
    RegisterComponents('PerfUtils', [TPerfCounters]);
end;

end.
