unit fmObjectInfo;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ComCtrls, PerfTitles, PerfCounters, ExtCtrls, StdCtrls, WinPerf, WinPerfUtils,
  PerfInstances;

type
  TObjectInfoForm = class(TForm)
    Panel2: TPanel;
    PageControl1: TPageControl;
    TShCounters: TTabSheet;
    TShInstances: TTabSheet;
    Panel3: TPanel;
    LBxCounters: TListBox;
    Splitter1: TSplitter;
    PerfCounters: TPerfCounters;
    PerfHelps: TPerfHelps;
    Panel1: TPanel;
    LBxCounterInfo: TListBox;
    MCounterHelp: TMemo;
    Panel4: TPanel;
    LBxInstances: TListBox;
    PerfInstances: TPerfInstances;
    Timer: TTimer;
    Splitter2: TSplitter;
    PnCounters: TPanel;
    PnCounterInfo: TPanel;
    PnCounterHelp: TPanel;
    Panel5: TPanel;
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure LBxCountersClick(Sender: TObject);
    procedure TimerTimer(Sender: TObject);
    procedure PnCountersClick(Sender: TObject);
    procedure LBxCountersEnter(Sender: TObject);
    procedure LBxCountersExit(Sender: TObject);
    procedure PnCounterInfoClick(Sender: TObject);
    procedure LBxCounterInfoEnter(Sender: TObject);
    procedure LBxCounterInfoExit(Sender: TObject);
    procedure PnCounterHelpClick(Sender: TObject);
    procedure MCounterHelpEnter(Sender: TObject);
    procedure MCounterHelpExit(Sender: TObject);
  private
    { Private declarations }
    _CounterIDX: Integer;
    function GetObjectName: String;
    procedure AddInfo(InfoDesc,Info: String);
    procedure AddCounterTypeExplanation(CT: Integer);
  public
    { Public declarations }
    property ObjectName: String read GetObjectName;
    constructor Create(ObjectName: String);
  end;

procedure SetPanelState(Panel: TPanel; Active: Boolean);

implementation

{$R *.DFM}

procedure SetPanelState(Panel: TPanel; Active: Boolean);
begin
    if Active then
    begin
        Panel.Color:=clBlue;
        Panel.Font.Color:=clHighlightText;
        Panel.Font.Style:=[fsBold];
    end else
    begin
        Panel.Color:=clBtnFace;
        Panel.Font.Color:=clBlack;
        Panel.Font.Style:=[];
    end;
end;


constructor TObjectInfoForm.Create(ObjectName: String);
var i: Integer;
    CounterTitle: String;
begin
    inherited Create(Application);
    _CounterIDX:=-1;
    PerfCounters.ObjectName:=ObjectName;
    Caption := Format('Performance object "%s" information',[ObjectName]);
    for i:=0 to PerfCounters.CounterCount - 1 do
    begin
        CounterTitle:=PerfCounters.CounterTitle[i];
        if Trim(CounterTitle) = '' then continue;
        LBxCounters.Items.Add(CounterTitle);
    end;

    try
        PerfInstances.ObjectName:=ObjectName;
        for i:=0 to PerfInstances.InstanceCount - 1 do
            LBxInstances.Items.Add(PerfInstances.InstanceName[i]);
    except
        Timer.Enabled:=False;
        TShInstances.Free;
    end;
end;

procedure TObjectInfoForm.FormClose(Sender: TObject;
  var Action: TCloseAction);
begin
    Action:=caFree;
end;

function TObjectInfoForm.GetObjectName: String;
begin
    result:=PerfCounters.ObjectName;
end;

procedure TObjectInfoForm.LBxCountersClick(Sender: TObject);
var PCD: PPerf_Counter_Definition;
begin
    if _CounterIDX = LBxCounters.ItemIndex then exit;

    _CounterIDX := LBxCounters.ItemIndex;

    LBxCounterInfo.Clear;
    MCounterHelp.Clear;

    if _CounterIDX < 0 then
    begin
        LBxCounterInfo.Items.Add('No counter selected');
        exit;
    end;

    PCD := PerfCounters.CounterData[_CounterIDX];

    AddInfo('Length',IntToStr(PCD.ByteLength));
    AddInfo('Default scale',IntToStr(PCD.DefaultScale));
    AddInfo('Detail level',IntToStr(PCD.DetailLevel));

    AddInfo('Counter type','$'+IntToHex(PCD.CounterType,8));

    AddCounterTypeExplanation(PCD.CounterType);

    AddInfo('Counter size',IntToStr(PCD.CounterSize));
    AddInfo('Counter offset',IntToStr(PCD.CounterOffset));

    MCounterHelp.Text:=PerfCounters.CounterHelp[_CounterIDX];
end;

procedure TObjectInfoForm.AddInfo(InfoDesc,Info: String);
begin
    LBxCounterInfo.Items.Add(Format('%-21s = %-12s',[InfoDesc,Info]));
end;

procedure TObjectInfoForm.TimerTimer(Sender: TObject);
var i : Integer;
    SL: TStrings;
begin
    PerfInstances.Refresh;
    SL := LBxInstances.Items;
    for i:=0 to PerfInstances.InstanceCount - 1 do
    begin
        if SL.Count <= i then
            SL.Add(PerfInstances.InstanceName[i])
        else
            SL[i]:=PerfInstances.InstanceName[i];
    end;

    while SL.Count > PerfInstances.InstanceCount do
        SL.Delete(SL.Count-1);
end;

procedure TObjectInfoForm.AddCounterTypeExplanation(CT: Integer);
var FirstPart: Boolean;

    procedure AddExplanationPart(Part: String);
    begin
        if FirstPart then
            Part:='          '+Part
        else
            Part:='       or '+Part;
        LBxCounterInfo.Items.Add(Part);
        FirstPart:=False;
    end;

begin
    FirstPart:=True;
    case CT and CounterSizeMask of
        Perf_SIZE_DWORD : AddExplanationPart('PERF_SIZE_DWORD');
        Perf_SIZE_LARGE : AddExplanationPart('PERF_SIZE_LARGE');
        Perf_SIZE_ZERO  : AddExplanationPart('Perf_SIZE_ZERO');
        Perf_SIZE_VARIABLE_LEN : AddExplanationPart('PERF_SIZE_VARIABLE_LEN');
    end;

    case CT and CounterTypeMask of
        PERF_TYPE_NUMBER :
            begin
                AddExplanationPart('PERF_TYPE_NUMBER');

                case CT and CounterSubTypeMask of
                    PERF_NUMBER_HEX : AddExplanationPart('PERF_NUMBER_HEX');
                    PERF_NUMBER_DECIMAL : AddExplanationPart('PERF_NUMBER_DECIMAL');
                    PERF_NUMBER_DEC_1000 : AddExplanationPart('PERF_NUMBER_DEC_1000');
                else
                    AddExplanationPart('PERF_NUMBER_???');
                end;
            end;
        PERF_TYPE_COUNTER :
            begin
                AddExplanationPart('PERF_TYPE_COUNTER');

                case CT and CounterSubTypeMask of
                    PERF_COUNTER_VALUE : AddExplanationPart('PERF_COUNTER_VALUE');
                    PERF_COUNTER_RATE : AddExplanationPart('PERF_COUNTER_RATE');
                    PERF_COUNTER_FRACTION : AddExplanationPart('PERF_COUNTER_FRACTION');
                    PERF_COUNTER_BASE : AddExplanationPart('PERF_COUNTER_BASE');
                    PERF_COUNTER_ELAPSED : AddExplanationPart('PERF_COUNTER_ELAPSED');
                    PERF_COUNTER_QUEUELEN : AddExplanationPart('PERF_COUNTER_QUEUELEN');
                    PERF_COUNTER_HISTOGRAM : AddExplanationPart('PERF_COUNTER_HISTOGRAM');
                else
                    AddExplanationPart('PERF_COUNTER_???');
                end;
            end;

        PERF_TYPE_TEXT :
            begin
                AddExplanationPart('PERF_TYPE_TEXT');

                case CT and CounterSubTypeMask of
                    PERF_TEXT_UNICODE : AddExplanationPart('PERF_TEXT_UNICODE');
                    PERF_TEXT_ASCII : AddExplanationPart('PERF_TEXT_ASCII');
                else
                    AddExplanationPart('PERF_TEXT_???');
                end;
            end;

        PERF_TYPE_ZERO : AddExplanationPart('PERF_TYPE_ZERO');
    else
        AddExplanationPart('PERF_TYPE_???');
    end;

    case CT and CounterTimeBaseMask of
        PERF_TIMER_TICK : AddExplanationPart('PERF_TIMER_TICK');
        PERF_TIMER_100NS : AddExplanationPart('PERF_TIMER_100NS');
        PERF_OBJECT_TIMER : AddExplanationPart('PERF_OBJECT_TIMER');
    else
        AddExplanationPart('PERF_TIME_???');
    end;

    if CT and PERF_DELTA_COUNTER <> 0 then AddExplanationPart('PERF_DELTA_COUNTER');
    if CT and PERF_DELTA_BASE <> 0 then AddExplanationPart('PERF_DELTA_BASE');
    if CT and PERF_INVERSE_COUNTER <> 0 then AddExplanationPart('PERF_INVERSE_COUNTER');
    if CT and PERF_MULTI_COUNTER <> 0 then AddExplanationPart('PERF_MULTI_COUNTER');

    case CT and CounterDisplayMask of
        PERF_DISPLAY_NO_SUFFIX : AddExplanationPart('PERF_DISPLAY_NO_SUFFIX');
        PERF_DISPLAY_PER_SEC : AddExplanationPart('PERF_DISPLAY_PER_SEC');
        PERF_DISPLAY_PERCENT : AddExplanationPart('PERF_DISPLAY_PERCENT');
        PERF_DISPLAY_SECONDS : AddExplanationPart('PERF_DISPLAY_SECONDS');
        PERF_DISPLAY_NOSHOW : AddExplanationPart('PERF_DISPLAY_NOSHOW');
    else
        AddExplanationPart('PERF_DISPLAY_???');
    end;
end;

procedure TObjectInfoForm.PnCountersClick(Sender: TObject);
begin
    ActiveControl:=LBxCounters;
end;

procedure TObjectInfoForm.LBxCountersEnter(Sender: TObject);
begin
    SetPanelState(PnCounters,True);
end;

procedure TObjectInfoForm.LBxCountersExit(Sender: TObject);
begin
    SetPanelState(PnCounters,False);
end;

procedure TObjectInfoForm.PnCounterInfoClick(Sender: TObject);
begin
    ActiveControl:=LBxCounterInfo;
end;

procedure TObjectInfoForm.LBxCounterInfoEnter(Sender: TObject);
begin
    SetPanelState(PnCounterInfo,True);
end;

procedure TObjectInfoForm.LBxCounterInfoExit(Sender: TObject);
begin
    SetPanelState(PnCounterInfo,False);
end;

procedure TObjectInfoForm.PnCounterHelpClick(Sender: TObject);
begin
    ActiveControl:=MCounterHelp;
end;

procedure TObjectInfoForm.MCounterHelpEnter(Sender: TObject);
begin
    SetPanelState(PnCounterHelp,True);
end;

procedure TObjectInfoForm.MCounterHelpExit(Sender: TObject);
begin
    SetPanelState(PnCounterHelp,False);
end;

end.
