{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

Author:       Alexey A. Dynnikov
EMail:        aldyn@chat.ru
WebSite:      http://www.chat.ru/~aldyn/index.html
Support:      Use the e-mail aldyn@chat.ru

Creation:     May 23, 1998
Version:      1.00

Legal issues: Copyright (C) 1998 by Alexey A. Dynnikov <aldyn@chat.ru>

              This software is provided 'as-is', without any express or
              implied warranty.  In no event will the author be held liable
              for any  damages arising from the use of this software.

              Permission is granted to anyone to use this software for any
              purpose, including commercial applications, and to alter it
              and redistribute it freely, subject to the following
              restrictions:

              1. The origin of this software must not be misrepresented,
                 you must not claim that you wrote the original software.
                 If you use this software in a product, an acknowledgment
                 in the product documentation would be appreciated but is
                 not required.

              2. Altered source versions must be plainly marked as such, and
                 must not be misrepresented as being the original software.

              3. This notice may not be removed or altered from any source
                 distribution.

 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

unit PerfTitles;

//CE_Desc_Begin(PerfTitles.pas)
{\
This unit implements non-visual components <%LINK TPerfTitles%> \
and <%LINK TPerfHelps%>. This components are used to obtain \
the titles and helps for the performance objects and counters.
 
}
//CE_Desc_End

interface

uses
  Windows, Classes, SysUtils;

type
//CE_Desc_Begin(TPerfInfo)
{\
TPerfInfo is a base for non-visual components <%LINK TPerfTitles%> \
and <%LINK TPerfHelps%>. \
The only available property is <%LINK LocaleID%>. }
//CE_Desc_End
    TPerfInfo = class(TComponent)
    private
        { Private declarations }
    protected
        { Protected declarations }
        _LocaleID: String;
        _Locales: TStringList;

        _Data: TObject;
        procedure GetData; virtual; abstract;

        procedure SetLocaleID(Value: String);

        function GetItemCount: Integer;
        function GetItem(Index: Integer): String;
        function GetItemIdx(Index: Integer): String;
        function GetItemByIdx(Idx: String): String;
        function GetIdxOfItem(Item: String): String;
        function GetIndexOfItem(Item: String): Integer;
        function GetIndexOfIdx(Idx: String): Integer;
    public
        { Public declarations }
        constructor Create(AComponent: TComponent); override;
        destructor Destroy; override;
        procedure Init;
    published
        { Published declarations }
        property LocaleID : String read _LocaleID write SetLocaleID;
    end;

    TPerfTitles = class(TPerfInfo)
    protected
        { Protected declarations }
        procedure GetData; override;
    public
        property TitleCount: Integer read GetItemCount;
        property Title[Index: Integer]: String read GetItem;
        property TitleIdx[Index: Integer]: String read GetItemIdx;
        property TitleByIdx[Idx: String]: String read GetItemByIdx;
        property IdxOfTitle[Title: String]: String read GetIdxOfItem;
        property IndexOfTitle[Title: String]: Integer read GetIndexOfItem;
        property IndexOfIdx[Idx: String]: Integer read GetIndexOfIdx;
    end;

//CE_Desc_Begin(TPerfHelps)
{Help not ready}
//CE_Desc_End
    TPerfHelps = class(TPerfInfo)
    protected
        { Protected declarations }
        procedure GetData; override;
    public
        property HelpCount: Integer read GetItemCount;
        property Help[Index: Integer]: String read GetItem;
        property HelpIdx[Index: Integer]: String read GetItemIdx;
        property HelpByIdx[Idx: String]: String read GetItemByIdx;
        property IdxOfHelp[Help: String]: String read GetIdxOfItem;
        property IndexOfHelp[Help: String]: Integer read GetIndexOfItem;
        property IndexOfIdx[Idx: String]: Integer read GetIndexOfIdx;
    end;

function Locales: TStrings;



procedure DoRegister;

implementation

//----------------------------------------------------------------------
procedure GetPerfInfoStrings(Locale,KeyName: String; Names : TStrings); forward;

type
    TPerfInfoData = class
        Locale: String;
        KeyName: String;
        Idx: TStringList;
        Items: TStringList;
        RefCount: Integer;
        constructor Create;
        destructor Destroy; override;
    end;

constructor TPerfInfoData.Create;
begin
    inherited;
    Idx := TStringList.Create;
    Items := TStringList.Create;
end;

destructor TPerfInfoData.Destroy;
begin
    Idx.Free;
    Items.Free;
    inherited;
end;

//----------------------------------------------------------------------

var
    PerfInfoList: TList;

function GetPerfInfo(Locale,KeyName: String): TPerfInfoData;
var i: Integer;
    Info: TPerfInfoData;
    InfoStrings: TStringList;
    S: String;
begin
    result:=nil;
    if PerfInfoList = nil then PerfInfoList := TList.Create;
    for i := 0 to PerfInfoList.Count - 1 do
    begin
        Info:= TPerfInfoData( PerfInfoList[i] );
        if AnsiCompareText(Info.Locale,Locale) <> 0 then continue;
        if AnsiCompareText(Info.KeyName,KeyName) <> 0 then continue;
        result:=Info;
        break;
    end;
    if result = nil then
    begin
        result := TPerfInfoData.Create;
        result.Locale:=Locale;
        result.KeyName:=KeyName;
        result.RefCount:=0;
        InfoStrings := TStringList.Create;
        GetPerfInfoStrings(Locale,KeyName,InfoStrings);
        for i:=0 to InfoStrings.Count - 1 do
        begin
            S:=InfoStrings.Names[i];
            result.Idx.Add(S);
            result.Items.Add(InfoStrings.Values[S]);
        end;
        PerfInfoList.Add(result);
        InfoStrings.Free;
    end;
    Inc(result.RefCount);
end;

procedure ReleasePerfInfo(var Info: TPerfInfoData);
begin
    if Info = nil then exit;
    Dec(Info.RefCount);
    if Info.RefCount <= 0 then
    begin
        PerfInfoList.Remove(Info);
        Info.Free;
        if PerfInfoList.Count = 0 then
        begin
            PerfInfoList.Free;
            PerfInfoList:=nil;
        end;
    end;
    Info := nil;
end;

const
    szPerfKey: String = 'SOFTWARE\Microsoft\Windows NT\CurrentVersion\Perflib';

procedure Initialize; forward;

//CE_Desc_Begin(TPerfInfo.Create)
{Creates a TPerfInfo object and initializes its data.}
//CE_Desc_End
constructor TPerfInfo.Create(AComponent: TComponent);
begin
    inherited;
    _Locales:=TStringList.Create;
    _Locales.Assign(Locales);
    _LocaleID:=_Locales[0];
end;

//CE_Desc_Begin(TPerfInfo.Destroy)
{\
Destroy is a destructor that disposes a TPerfInfo instance.

}
//CE_Desc_End
destructor TPerfInfo.Destroy;
begin
    _Locales.Free;
    ReleasePerfInfo(TPerfInfoData( _Data ));
    inherited;
end;


//CE_Desc_Begin(TPerfInfo.LocaleID)
{\
Contains a three-digit sting than identifies the required \
language of PerfTitles or PerfHelps.
The default value is '009' ( English ).
Use function Locales to obtain \
a complete list of LocaleID's available on local computer. }
//CE_Desc_End
procedure TPerfInfo.SetLocaleID(Value: String);
begin
    if _Locales.IndexOf(Value) < 0 then Value := _Locales[0];
    if _LocaleID <> Value then
        ReleasePerfInfo(TPerfInfoData( _Data ));
    _LocaleID:=Value;
end;

function TPerfInfo.GetItemCount: Integer;
begin
    GetData;
    result:=TPerfInfoData( _Data ).Items.Count;
end;

function TPerfInfo.GetItem(Index: Integer): String;
begin
    GetData;
    result:=TPerfInfoData( _Data ).Items[Index];
end;

function TPerfInfo.GetItemIdx(Index: Integer): String;
begin
    GetData;
    result:=TPerfInfoData( _Data ).Idx[Index];
end;

function TPerfInfo.GetItemByIdx(Idx: String): String;
var i: Integer;
begin
    GetData;
    i:=TPerfInfoData( _Data ).Idx.IndexOf(Idx);
    if i < 0 then
        result:=''
    else
        result:=TPerfInfoData( _Data ).Items[i];
end;

function TPerfInfo.GetIdxOfItem(Item: String): String;
var i: Integer;
begin
    GetData;
    i:=TPerfInfoData( _Data ).Items.IndexOf(Item);
    if i < 0 then
        result:='-1'
    else
        result:=TPerfInfoData( _Data ).Idx[i];
end;

function TPerfInfo.GetIndexOfItem(Item: String): Integer;
begin
    GetData;
    result := TPerfInfoData( _Data ).Items.IndexOf( Item );
end;

function TPerfInfo.GetIndexOfIdx(Idx: String): Integer;
begin
    GetData;
    result := TPerfInfoData( _Data ).Idx.IndexOf( Idx );
end;

procedure TPerfInfo.Init;
begin
    GetData;
end;

//----------------------------------------------------------------------

//CE_Desc_Begin(TPerfTitles)
{\
Non-visual component TPerfTitles is used to read and manipulate \
titles of Win NT Performance objects and counters. This titles are \
stored in registry at
HKEY_LOCAL_MACHINE\SOFTWARE\Microsoft\Windows NT\CurrentVersion\Perflib\<LocaleID>\Counter
as the list of strings pairs <IDX><TITLE>.
Here <IDX> is a string that corresponds to the Performance object (PERF_OBJECT_TYPE) or \
Performance counter (PPERF_COUNTER_DEFINITION) TitleIdx field.
<TITLE> is a string that describes corresponding object or counter.

}
//CE_Desc_End

//CE_Desc_Begin(TPerfTitles.TitleCount)
{\
Number of title stings loaded from registry. \
Valid values of Index in properties <%LINK Title%> and <%LINK TitleIdx%> are \
[0..TitleCount-1].

}
//CE_Desc_End

//CE_Desc_Begin(TPerfTitles.Title)
{\
Returns a Title string for the Index specified. \
Valid indexes are [0..TitleCount-1].

}
//CE_Desc_End


//CE_Desc_Begin(TPerfTitles.TitleIdx)
{\
Returns a TitleIdx string for the Index specified. \
Valid indexes are [0..TitleCount-1].

}
//CE_Desc_End

//CE_Desc_Begin(TPerfTitles.TitleByIdx)
{\
This property returns the Title string for the given Idx value. \
If the required Idx is not found then the result is empty string.}
//CE_Desc_End

//CE_Desc_Begin(TPerfTitles.IdxOfTitle)
{\
This property returns the Idx value for the given Title string. \
If the required Title is not found then the result is empty string.}
//CE_Desc_End

//CE_Desc_Begin(TPerfTitles.IndexOfTitle)
{\
This property gives the index of Title specified. \
If the Title string not found then result is -1.

}
//CE_Desc_End

//CE_Desc_Begin(TPerfTitles.IndexOfIdx)
{\
This property gives the index of Idx specified. \
If the Title string not found then result is -1.

}
//CE_Desc_End

//----------------------------------------------------------------------
//CE_Desc_Begin(TPerfHelps)
{\
Non-visual component TPerfHelps is used to read and manipulate \
helps of Win NT Performance objects and counters. This helps are \
stored in registry at
HKEY_LOCAL_MACHINE\SOFTWARE\Microsoft\Windows NT\CurrentVersion\Perflib\<LocaleID>\Help
as the list of strings pairs <IDX><HELP>.
Here <IDX> is a string that corresponds to the Performance object (PERF_OBJECT_TYPE) or \
Performance counter (PPERF_COUNTER_DEFINITION) HelpIdx field.
<HELP> is a string that describes corresponding object or counter.

}
//CE_Desc_End

//CE_Desc_Begin(TPerfHelps.HelpCount)
{\
Number of Help stings loaded from registry. \
Valid values of Index in properties <%LINK Help%> and <%LINK HelpIdx%> are \
[0..HelpCount-1].

}
//CE_Desc_End

//CE_Desc_Begin(TPerfHelps.Help)
{\
Returns a Help string for the Index specified. \
Valid indexes are [0..HelpCount-1].

}
//CE_Desc_End


//CE_Desc_Begin(TPerfHelps.HelpIdx)
{\
Returns a HelpIdx string for the Index specified. \
Valid indexes are [0..HelpCount-1].

}
//CE_Desc_End

//CE_Desc_Begin(TPerfHelps.HelpByIdx)
{\
This property returns the Help string for the given Idx value. \
If the required Idx is not found then the result is empty string.}
//CE_Desc_End

//CE_Desc_Begin(TPerfHelps.IdxOfHelp)
{\
This property returns the Idx value for the given Help string. \
If the required Help is not found then the result is empty string.}
//CE_Desc_End

//CE_Desc_Begin(TPerfHelps.IndexOfHelp)
{\
This property gives the index of Help specified. \
If the Help string not found then result is -1.

}
//CE_Desc_End

//CE_Desc_Begin(TPerfHelps.IndexOfIdx)
{\
This property gives the index of Idx specified. \
If the help string not found then result is -1.

}
//CE_Desc_End

procedure TPerfTitles.GetData;
begin
    if _Data = nil then
        _Data := GetPerfInfo(LocaleID,'Counter');
end;


//----------------------------------------------------------------------
procedure TPerfHelps.GetData;
begin
    if _Data = nil then
        _Data := GetPerfInfo(LocaleID,'Help');
end;

//----------------------------------------------------------------------

procedure DoRegister;
begin
  RegisterComponents('PerfUtils', [TPerfTitles,TPerfHelps]);
end;


//----------------------------------------------------------------------
var
    _Locales,_LocalesTmp: TStringList;
    _Initialized: Boolean;

procedure Initialize;
var
    hKeyPerflib: HKEY;      // handle to registry key
    LastWrite: TFileTime;
    Idx: Integer;
    KeyName : PChar;
    KeySize,NewKeySize: DWORD;
    R: Integer;
begin
    if _Initialized then exit;
    _Initialized:=True;
    if RegOpenKeyEx( HKEY_LOCAL_MACHINE, PChar(szPerfKey), 0, KEY_READ, hKeyPerflib) <> ERROR_SUCCESS then Exit;

    Idx:=0; R:=0;
    KeySize:=10;
    GetMem( KeyName, KeySize + 1 );

    while True do
    begin
        NewKeySize:=KeySize;
        R := RegEnumKeyEx( hKeyPerfLib, Idx, KeyName, NewKeySize, nil, nil,nil,@LastWrite);
        case R of
            ERROR_NO_MORE_ITEMS:
            begin
                R := ERROR_SUCCESS;
                break;
            end;
            ERROR_SUCCESS:
                begin
                    _Locales.Add(StrPas(KeyName));
                    INC(IDX);
                end;
            else
                break;
        end;
    end;

    FreeMem( KeyName );
    RegCloseKey( hKeyPerflib );
    if R <> ERROR_SUCCESS then raise Exception.Create(SysErrorMessage(R));
end;

//CE_Desc_Begin(Locales)
{\
It is a helper function that loads list of all available \
locales on the local computer. The values returned are \
valid values for the <%LINK TPerfInfo.LocaleID%> property.

<%SEEALSO TPerfInfo.LocaleID%>
}
//CE_Desc_End
function Locales: TStrings;
begin
    Initialize;
    _LocalesTmp.Assign(_Locales);
    result := _LocalesTmp;
end;

//------------------------------------------------------------------------
procedure GetPerfInfoStrings(Locale,KeyName: String; Names : TStrings);
var
    hKeyPerflib: HKEY;      // handle to registry key
    dwMaxValueLen: DWORD;   // maximum size of key values
    dwBuffer: DWORD;        // bytes to allocate for buffers
    dwBufferSize: DWORD;    // size of dwBuffer
    lpCurrentString: LPSTR; // pointer for enumerating data strings
    dwCounter: DWORD;       // current counter index
    lpNameStrings: PChar;
begin
    // Get the number of Counter items.
    if RegOpenKeyEx( HKEY_LOCAL_MACHINE, PChar(szPerfKey), 0, KEY_READ, hKeyPerflib) <> ERROR_SUCCESS then Exit;
    dwBufferSize := sizeof(dwBuffer);
    RegQueryValueEx( hKeyPerflib, 'Last Counter', nil, nil, @dwBuffer, @dwBufferSize );
    RegCloseKey( hKeyPerflib );


    // Open key containing counter and object names.

    if RegOpenKeyEx( HKEY_LOCAL_MACHINE, PChar(szPerfKey+'\'+Locale), 0, KEY_READ, hKeyPerflib) <> ERROR_SUCCESS then Exit;

    // Get the size of the largest value in the key (Counter or Help).

    RegQueryInfoKey( hKeyPerflib, nil, nil, nil, nil, nil, nil, nil, nil, @dwMaxValueLen, nil, nil);

    // Allocate memory for the counter and object names.

    dwBuffer := dwMaxValueLen + 1;

    GetMem(lpNameStrings, dwBuffer * sizeof(CHAR));

    // Read Counter value.
    RegQueryValueEx( hKeyPerflib, PChar(KeyName), nil, nil, PByte(lpNameStrings), @dwBuffer );

    // Load names into an array, by index.
    Names.Clear;
    lpCurrentString := lpNameStrings;
    while lpCurrentString^ <> #0 do
    begin
        dwCounter := StrToInt( lpCurrentString );
        INC( lpCurrentString, lstrlen(lpCurrentString)+1);

        if Trim(lpCurrentString) <> '' then
            Names.Add(IntToStr(dwCounter)+'='+lpCurrentString);

        INC(lpCurrentString,(lstrlen(lpCurrentString)+1) );
    end;

    FreeMem(lpNameStrings);

    RegCloseKey(hKeyPerflib);
end;




initialization
    _Initialized:=False;
    _Locales:=TStringList.Create;
    _LocalesTmp:=TStringList.Create;
end.
