{* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

Author:       Alexey A. Dynnikov
EMail:        aldyn@chat.ru
WebSite:      http://www.chat.ru/~aldyn/index.html
Support:      Use the e-mail aldyn@chat.ru

Creation:     May 23, 1998
Version:      1.00

Legal issues: Copyright (C) 1998 by Alexey A. Dynnikov <aldyn@chat.ru>

              This software is provided 'as-is', without any express or
              implied warranty.  In no event will the author be held liable
              for any  damages arising from the use of this software.

              Permission is granted to anyone to use this software for any
              purpose, including commercial applications, and to alter it
              and redistribute it freely, subject to the following
              restrictions:

              1. The origin of this software must not be misrepresented,
                 you must not claim that you wrote the original software.
                 If you use this software in a product, an acknowledgment
                 in the product documentation would be appreciated but is
                 not required.

              2. Altered source versions must be plainly marked as such, and
                 must not be misrepresented as being the original software.

              3. This notice may not be removed or altered from any source
                 distribution.

 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *}

unit PerfData;

//CE_Desc_Begin(PerfData.pas)
{}
//CE_Desc_End

interface

uses
    Windows, WinPerf, Classes, SysUtils;

type
//CE_Desc_Begin(TPerfData)
{\
TPerfData is a class that simplifies reading performance information. \
It allocates a memory required to read performance information \
specified by Key parameter of constructor and loads performance data.

}
//CE_Desc_End
    TPerfData = class
    private
        _Key: String;
        _BufferSize: DWORD;
        _PerfData : PPerf_Data_Block;
        _PrevPerfData : TPerf_Data_Block;
        function GetPrevPerfData: PPerf_Data_Block;
    public
        property Key: String read _Key;
        property PerfData : PPerf_Data_Block read _PerfData;
        property PrevPerfData : PPerf_Data_Block read GetPrevPerfData;
        procedure Refresh;
        constructor Create(Key: String);
        destructor Destroy; override;

        function SystemTime : TSystemTime;
        function PerfTime : TInt64;
        function PerfFreq : TInt64;
        function PerfTime100nSec : TInt64;
        function SystemName :String;
    end;

function GetGlobalPerfData: TPerfData;
procedure ReleaseGlobalPerfData(var PerfData: TPerfData);


implementation

//------------------------------------------------------------------------
var
    GlobalPerfData: TPerfData;
    GlobalRefCount: Integer;

//CE_Desc_Begin(GetGlobalPerfData)
{\
This function returns an instance of TPerfData object. \
Key 'Global' is used to create the instance.

Call this function to obtain a shared TPerfData object. \
When the function is called first time it creates \
a new TPerfData object and increments an internal counter. \
Next calls of this function do not create a new objects. \
Instead it increments the internal counter and returns \
the previous instance of TPerfData object. \
Call <%LINK ReleaseGlobalPerfData%> to release the \
object returned by GetGlobalPerfData.

}
//CE_Desc_End
function GetGlobalPerfData: TPerfData;
begin
    if GlobalPerfData = nil then
        GlobalPerfData := TPerfData.Create('Global');

    INC(GlobalRefCount);

    result := GlobalPerfData;
end;

//CE_Desc_Begin(ReleaseGlobalPerfData)
{\
This function releases the TPerfData object \
created by <%LINK GetGlobalPerfData%>. \
Each call of this function decrements the internal \
counter and when it becomes zero the TPerfData object \
will be destroyed.

}
//CE_Desc_End
procedure ReleaseGlobalPerfData(var PerfData: TPerfData);
begin
    if PerfData <> GlobalPerfData then exit;

    if GlobalRefCount <= 0 then exit;

    DEC(GlobalRefCount);

    if GlobalRefCount > 0 then exit;

    GlobalPerfData.Free;

    GlobalPerfData:=nil;
end;

//-----------------------------------------------------------------------
//CE_Desc_Begin(TPerfData.Refresh)
{\
Refresh re-reads performance data. \
Call it to obtain new TPerf_Data_Block data.

}
//CE_Desc_End
procedure TPerfData.Refresh;
var BS: integer;
begin
    if _PerfData <> nil then _PrevPerfData:=_PerfData^;
    
    BS:=_BufferSize;
    while RegQueryValueEx( HKEY_PerfORMANCE_DATA, PChar(Key), nil, nil,
            PByte(_PerfData), @BS ) = ERROR_MORE_DATA do
    begin
        // Get a buffer that is big enough.
        INC(_BufferSize,$1000);
        BS:=_BufferSize;
        ReallocMem( _PerfData, _BufferSize );
    end;
end;

//CE_Desc_Begin(TPerfData.Create)
{\
Create is a constructor that creates and initializes a new instance of TPerfData. \
The Key parameter of constructor specifies a filter for information retrieved. \

This parameter can be of the following form
'Global' - retrieves all performance data except costly data.
'Costly' - the costly performance information will be retrieved. \
Be accurate using this value.
Combination if object title's indexes, for example '2 6 8 120'. Use such a value \
to retrieve performance data only for some desired objects.

}
//CE_Desc_End
constructor TPerfData.Create(Key: String);
begin
    inherited Create;
    _BufferSize:=$2000;
    _PerfData := AllocMem(_BufferSize);
    _Key:=Key;
    Refresh;
end;

//CE_Desc_Begin(TPerfData.Destroy)
{\
Disposes an instance of TPerfData object and frees its resources.

}
//CE_Desc_End
destructor TPerfData.Destroy;
begin
    FreeMem(_PerfData);
    inherited;
end;

function TPerfData.GetPrevPerfData: PPerf_Data_Block;
begin
    result:=@_PrevPerfData;
end;

//CE_Desc_Begin(TPerfData.SystemTime)
{\
This property returns a value of SystemTime field of TPerf_Data_Block structure.

}
//CE_Desc_End
function TPerfData.SystemTime : TSystemTime;
begin
    result:=_PerfData.SystemTime;
end;

//CE_Desc_Begin(TPerfData.PerfTime)
{\
This property returns a value of PerfTime field of TPerf_Data_Block structure.

}
//CE_Desc_End
function TPerfData.PerfTime : TInt64;
begin
    result:=_PerfData.PerfTime;
end;

//CE_Desc_Begin(TPerfData.PerfFreq)
{\
This property returns a value of PerfFreq field of TPerf_Data_Block structure.

}
//CE_Desc_End
function TPerfData.PerfFreq : TInt64;
begin
    result:=_PerfData.PerfFreq;
end;

//CE_Desc_Begin(TPerfData.PerfTime100nSec)
{\
This property returns a value of PerfTime100nSec field of TPerf_Data_Block structure.

}
//CE_Desc_End
function TPerfData.PerfTime100nSec : TInt64;
begin
    result:=_PerfData.PerfTime100nSec;
end;

//CE_Desc_Begin(TPerfData.SystemName)
{\
SystemName is a name of system under performance monitoring \
stored in the TPerf_Data_Block stucture.

}
//CE_Desc_End
function TPerfData.SystemName : String;
begin
    result:=WideCharToString(PWideChar(DWORD(_PerfData)+ _PerfData.SystemNameOffset));
end;

//CE_Desc_Begin(TPerfData.PerfData)
{\
This property returns a pointer to the TPerf_Data_Block structure.

Note tha this pointer is volatile and can be \
changed during call of <%LINK Refresh%>. Do not hold it for a long time. \
Instead use this property to obtain a valid pointer.


}
//CE_Desc_End

//CE_Desc_Begin(TPerfData.Key)
{\
This property returns a value of Key parameter \
used in <%LINK Create%> constructor.

}
//CE_Desc_End

initialization
    GlobalRefCount:=0;
    GlobalPerfData:=nil;
end.
