unit PanelButton;

interface

 uses
  Windows,SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls, Menus,
  Forms, Dialogs,StdCtrls, ExtCtrls, DBCtrls, DsgnIntf, GlyphControl;

  (*
 *****************************************************************************

                       TPanelButton component

                       All Rights reserved

    1.    PanelButton Overview.
     TPanelButton is a button which can have multiple controls on it. It
     maintains  simliar functionality to TButton. The main difference is that
     Controls (labels, Images, Panels etc.) can be dropped onto the
     TPanelButton at design time. At design time you can manipulate the
     PanelButton exactly as you do with a Panel or GroupBox. At run time
     the TPanelButton has the same behavior as TButton (with raised and lowered
     visual effects). All controls & captions (TLabels) move along with the face
     of the TPanelButton just as a TButton's Caption does.
      In addition, TPanelButton has a Color property which allows creation
     of different (from standard Windows) interface - for example blue Form
     with blue PanelButtons. Also TPanelButton has Stuck property enabling more
     raised drawing effect at run time - when mouse is moving over the TPanelButton.

   2.    Properties and methods.
     TPanelButton is a descendant of TButton and inherits all corresponding
     properties and methods. In addition it has following properties:
        Color: TColor ;
        Bevel Width: Integer;
        AccelText: TCaption;
        Stuck: Boolean;
     and method:
        UnStick;

     TPanelButton has caption, but it is blank initially. You can choose not
     to use the caption, but drop label(s) instead. In this  case you need to
     explicitly set the accelerator key using the AccelText property. In Object
     Inspector you can select the AccelText string via a drop-down displaying
     the captions of the labels already dropped onto the TPanelButton.
     Corresponding Property Editor is included.

     Property Stuck enables the visual effect of 'sticking' TPanelButton
     at run time -more raised drawing when mouse is moving over the PanelButton.
     TPanelButton does not affect its parent OnMouseMove event. In other words
     TPanelButton can't 'unstick' itself (on component level) when cursor is
     outside of TPanelButton area. PanelButton has the method  'Unstick' to
     handle this case still. This method should be called in procedure  linked
     to parent OnMouseMove event. For example project has Form1 where PanelButton1
     is dropped. PanelButton1.stuck property is set to true to activate the effect.
     Form1MouseMove event handler has the code: PanelButton1.UnStick;


   3.   Component design Notices.
     TPanelButton is a descendant of TButton. TButton has Controls property
     as TWinControl descendant but does not have a canvas and has the default
     method of drawing. I've added Canvas and Paint related methods from
     TCustomControl Class, published the color property and added the
     Bevel Width property. The Protected Paint method draws the button
     (up or down) on the canvas.  Another part of the design (more complex
     to explain but fun to do) was managing Click and Mouse event handlers
     for all child controls. I've designed DoClick, DoMouseUp, DoMouseDown,
     DoMouseMove procedures  which handle visual moving Up and Down  for all
     the controls inside of TPanelButton. Each procedure restores original
     Event handlers for the child controls at the end.

    4. Limitations and requirements
       There is a limitation I'd like to mention. I'm using internal Image(fImage)
     to reduce flickering. It means You can not use Sender parameter directly
     in event handlers of TPanelButton to access the TPanelButton itself.
     You can work around using:  (sender as TControl).parent as TPanelButton
     istead. I also use the fImage to implement proper event handling for
     the child controls.
         Since PanelButton 'replays' mouse and click events of child controls
     child controls are expected to have those events declared expicitely
     (as published). If you drop component who 'hides' mouse and click events


     Developed by Galina Tydykov        (Boston, USA,
                                         tydykov@sonamed.com,
                                         ValeryGalina@Mediaone.net) .

     Published by permission of SonaMed Co.
     This is freeware and should be used as it is.

 This software is provided "AS IS," without a warranty of any kind. ALL
 * EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES, INCLUDING
 * ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
 * OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED. SONAMED SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THE SOFTWARE OR ITS DERIVATIVES. IN NO EVENT WILL SONAMED
 * BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT,INDIRECT,
 * SPECIAL, CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER CAUSED
 * AND REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF
 * OR INABILITY TO USE SOFTWARE, EVEN IF SONAMED HAS BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *


******************************************************************************)


 type
  TDEventsRecorder = class;

   TPanelButton = class(TButton)
   private
      { Private declarations }
      FCanvas: TCanvas;
      FAlignment: TAlignment;
      FBevelWidth: TBevelWidth;
      fAccelText: TCaption;
      FImage : TImage ;

      fUpState,fMousePressed: Boolean;
      fSentByDoMove: boolean;

      fSpacing: integer;
      FOnResize: TNotifyEvent;
      fDEventsRecorder: TDEventsRecorder;
      fBlack:boolean;
      fStuck:boolean;
      fTopColor, fBottomColor: TColor;
      fControlsList:TList;

      fControlSender,
      fClickSender:TControl;
      xxname:string;

      fOutof:boolean;

      procedure WMSetCursor(var Message: TMessage); message WM_SETCURSOR;
      procedure CMColorChanged(var Message: TMessage); message CM_COLORCHANGED;
      procedure WMPaint(var Message: TWMPaint); message WM_PAINT;
      procedure CMDialogChar(var Message: TCMDialogChar); message CM_DIALOGCHAR;
      procedure CMTextChanged(var Message: TMessage); message CM_TEXTCHANGED;
      procedure CMEnabledChanged(var Message: TMessage); message CM_ENABLEDCHANGED;
      procedure SetAlignment(Value: TAlignment);
      procedure SetBevelWidth(Value: TBevelWidth);
      procedure PanelUp;
      procedure PanelDown;
      procedure DoUp(sender:Tobject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
      procedure DoDown(sender:Tobject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
      procedure DoMove(sender:Tobject;Shift: TShiftState; X, Y: Integer);
      procedure DoClick(sender:Tobject);
      function GetPosition(MyControl:TControl):TPoint;
      function FindControl(var ind:integer): boolean;
      function GetIndex(TheControl:TControl): integer;
   protected
      procedure CreateParams(var Params: TCreateParams); override;
      procedure PaintWindow(DC: HDC); override;
      procedure Paint; virtual;
      property Canvas: TCanvas read FCanvas;
      procedure Click; override;
      procedure Loaded; override;
      procedure MouseMove(Shift: TShiftState; X, Y: Integer);  override;
      procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);  override;
      procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
   public
      { Public declarations }
      constructor Create(AOwner: TComponent);  override;
      Destructor Destroy;  override;
   published
      { Published declarations }
      property Color;
      property BevelWidth: TBevelWidth read FBevelWidth write SetBevelWidth default 1;
      property AccelText: TCaption read fAccelText write fAccelText;
      property Alignment: TAlignment read FAlignment write SetAlignment default taCenter;
      property OnResize: TNotifyEvent read FOnResize write FOnResize;
      property Stuck:boolean read FStuck write FStuck ;
      procedure UnStick;

   end;

{  EventList class is designed to store Click and Mouse Event handlers for
   for all the controls parented by TPanelButton, so TPanelButton
   could restore original event handlers after 'control moving' is done}


  TEventsItem = class
  public
     fowner:TDEventsRecorder;
     Constructor Create(owner:TDEventsRecorder); virtual;
  end ;

  TMultiEventsItem = class(TEventsItem)
  public
     OnTimer: TNotifyEvent;
     OnClick: TNotifyEvent;
     OnMouseMove: TMouseMoveEvent;
     OnMouseDown: TMouseEvent;
     OnMouseUp: TMouseEvent ;
   end ;


  TDEventsRecorder = class(TObject)
  private
    { Private declarations }
    fList: TList;
    function GetCount: integer;
    function GetEventItem(Index: Integer): TEventsItem;
    procedure PutEventItem(Index: Integer; Item: TEventsItem);
   protected
   public
    { Public declarations }
     property EventsItem[Index: Integer]: TEventsItem read GetEventItem write PutEventItem;
     property Count: Integer read GetCount;
     procedure delete(Index: Integer);
     Constructor Create;
     destructor Destroy; override;
     procedure Clear;
   end;

{  This class declared for 'Beauty' purpose - to public Click and Mouse events
   and be able extract the corresponding method pointers.
   I definitely could use any other appropriate class for this }

   type TEventControl = class(TControl)
   public
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnClick;
   end;


  { TAccelTextProperty - Property Editor which provide ability to select AccellText from
  captions of labels dropped on TPanelButton}
  type
     TAccelTextProperty = class(TStringProperty)
  public
    function GetAttributes: TPropertyAttributes; override;
    procedure GetValues(Proc: TGetStrProc); override;
  end;


function GetHighlightColor(BaseColor: TColor): TColor;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('Galina', [TPanelButton ]);
  RegisterPropertyEditor(TypeInfo(TCaption), TPanelButton, 'AccelText',TAccelTextProperty);
end;


function Smallest(X, Y: Integer): Integer;
begin
	if X < Y then Result := X else Result := Y;
end;

function Largest(X, Y: Integer): Integer;
begin
	if X > Y then Result := X else Result := Y;
end;

function GetHighlightColor(BaseColor: TColor): TColor;
begin
  Result := RGB(Smallest(GetRValue(ColorToRGB(BaseColor)) + 64, 255),
            Smallest(GetGValue(ColorToRGB(BaseColor)) + 64, 255),
            Smallest(GetBValue(ColorToRGB(BaseColor)) + 64, 255));
end;

function GetShadowColor(BaseColor: TColor): TColor;
begin
  Result := RGB(
  Largest(GetRValue(ColorToRGB(BaseColor)) - 64, 0),
  Largest(GetGValue(ColorToRGB(BaseColor)) - 64, 0),
  Largest(GetBValue(ColorToRGB(BaseColor)) - 64, 0));
end;


{ TAccelTextProperty }

{ This is Property Editor which provide ability to select AccellText from
  captions of labels dropped on TPanelButton}

function TAccelTextProperty.GetAttributes: TPropertyAttributes;
begin
  Result := [paValueList];
end;

procedure TAccelTextProperty.GetValues(Proc: TGetStrProc);
var
  I: Integer;
  Component: TComponent;
begin
  for I := 0 to Designer.Form.ComponentCount - 1 do
  begin
    Component := Designer.Form.Components[I];
    if (Component is TLabel) and (Component.Name <> '') and
        ((Designer.Form.Components[i] as TLabel).parent is TPanelButton)
      and (((Designer.Form.Components[i] as TLabel).parent as TComponent)=GetComponent(0))  then
       Proc((Designer.Form.Components[i] as TLabel).caption);
  end;
end;



{TDEventsRecorder}

{  EventList class is designed to store Click and Mouse Event handlers for
   for all the controls parented by TPanelButton, so TPanelButton
   could restore original event handlers after 'control moving' is done}

{  TEventsItem }
 Constructor TEventsItem.Create(owner:TDEventsRecorder);
 begin
   fowner:=owner;
   fowner.flist.add(self);
 end;


{TDEventsRecorder}

Constructor TDEventsRecorder.Create;
begin
  fList:=TList.create;
end;

destructor TDEventsRecorder.Destroy;
begin
  Clear;
  fList.free;
  inherited Destroy;
end;

function TDEventsRecorder.GetCount: integer;
begin
  result:=fList.Count;
end;


procedure TDEventsRecorder.Clear;
var
 i: integer;
begin
  for i:=0 to fList.count-1 do
    delete(fList.count-1);
end;


function TDEventsRecorder.GetEventItem(Index: Integer): TEventsItem;
begin
  if (Index < 0) or (Index >= fList.Count) then
      raise EListError.Create('DEventsRecorder Error');
  Result :=TEventsItem(fList.items[Index]);
end;


procedure TDEventsRecorder.PutEventItem(Index: Integer; Item: TEventsItem);
begin
  if (Index < 0) or (Index >= fList.Count) then
      raise EListError.Create('DEventsRecorder Error');
  TEventsItem((fList.items[Index])^):= Item;
end;

procedure TDEventsRecorder.delete(Index: Integer);
begin
  EventsItem[Index].free;
  flist.Delete(Index);
end;


{ TPanelButton}

constructor TPanelButton.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  FCanvas := TControlCanvas.Create;
  TControlCanvas(FCanvas).Control := Self;

  ControlStyle := [csAcceptsControls,csCaptureMouse, csClickEvents,
    csSetCaption, csOpaque, csDoubleClicks ];

  Width := 140;
  Height := 80;
  BevelWidth := 1;
  TabStop:=true;

  FAlignment := taCenter;
  fUpState:=true;
  fSentByDoMove:=false;
  fSpacing:=3;

  fDEventsRecorder:=TDEventsRecorder.create;
  fControlsList:=TList.create;

  FImage:=nil;
  fblack:=false;
  fStuck:=false;


end;


Destructor TPanelButton.Destroy;
begin
  fControlsList.clear;
  fControlsList.free;
  fDEventsRecorder.free;
  FCanvas.Free;
  if FImage<>nil then
    FImage.Free;
  inherited Destroy;
end;

procedure TPanelButton.CreateParams(var Params: TCreateParams);
begin
{Clean caption when TPanelButton is created}
  inherited CreateParams(Params);
  if (csDesigning in ComponentState) and  Not(csLoading in ComponentState) then
    Params.caption:='';
end;

procedure TPanelButton.WMPaint(var Message: TWMPaint);
begin
  PaintHandler(Message);
end;

procedure TPanelButton.PaintWindow(DC: HDC);
begin
  fCanvas.Lock;
  try
    fCanvas.Handle := DC;
    try
      Paint;
    finally
      fCanvas.Handle := 0;
    end;
  finally
    fCanvas.Unlock;
  end;
end;

procedure TPanelButton.SetBevelWidth(Value: TBevelWidth);
begin
  fBevelWidth := Value;
  Realign;
  Invalidate;
end;

procedure TPanelButton.SetAlignment(Value: TAlignment);
begin
  fAlignment := Value;
  Invalidate;
end;

procedure TPanelButton.CMTextChanged(var Message: TMessage);
begin
  inherited;
  Invalidate;
end;

procedure TPanelButton.UnStick;
begin
 if fBlack then
 begin
  fBlack:=false  ;
  invalidate;
 end;
end;

procedure TPanelButton.Loaded;
var
  i, j, k: integer;
  OnClickMM: TNotifyEvent;
  MyControl:TControl;
begin
  inherited Loaded;
  if (not (csDesigning in ComponentState)) then
  begin
    if fImage=nil then
    begin
      fImage:=TImage.Create(Self) ;
      fImage.Name:='PseudoImage' ;
      fImage.Visible := True;
      fImage.Enabled := True;
      fImage.Parent  := Self;
      fImage.align:=alClient;
      fImage.OnMouseMove:=OnMouseMove;
    end;
    OnMouseMove:=nil;  { in order to not to call OnMouseMove event handler when
                           mouse is moving over the other control  }
    fControlsList.clear;
    for I:= 0 to ControlCount -1 do
      fControlsList.add(Controls[I]);
    j:=0 ;
    while j<=fControlsList.count-1 do
    begin
      MyControl:=TControl(fControlsList[j]);
      with TMultiEventsItem.create(fDEventsRecorder) do
      begin
        if (MyControl.name<>'') and  not (MyControl is TPanelbutton) then
        begin
          OnClick:=TEventControl(MyControl).OnClick;
          TEventControl(MyControl).OnClick:=DoClick;
          OnMouseDown:=TEventControl(MyControl).OnMouseDown;
          TEventControl(MyControl).OnMouseDown:=DoDown;
          OnMouseUp:=TEventControl(MyControl).OnMouseUp;
          TEventControl(MyControl).OnMouseUp:=DoUp;
          OnMouseMove:=TEventControl(MyControl).OnMouseMove;
          TEventControl(MyControl).OnMouseMove:=DoMove;
        end;
      end;
      if MyControl is TWinControl then
      begin
        for k:=0 to TWinControl(MyControl).ControlCount -1 do
          fControlsList.add(TWinControl(MyControl).Controls[k]);
      end;
      j:=j+1;
    end;
 end;
end;


procedure TPanelButton.CMEnabledChanged(var Message: TMessage);
var
  i: integer;
begin
  with Message do
  begin
    for I:= 0 to fControlsList.Count-1 do
      TControl(fControlsList[I]).enabled:=self.Enabled;
    inherited;
  end;
  invalidate;
end;

procedure TPanelButton.CMDialogChar(var Message: TCMDialogChar);
begin
  with Message do
    if IsAccel(CharCode, AccelText) and CanFocus then
    begin
      click;
      Result := 1;
    end
    else
      inherited;
end;

procedure TPanelButton.CMColorChanged(var Message: TMessage);
begin
  with Message do
    inherited;
  fTopColor := GetHighlightColor(Color);
  fBottomColor := GetShadowColor(Color);
  invalidate;
end;

procedure TPanelButton.WMSetCursor(var Message: TMessage);
var I: integer;
begin
  with Message do
  begin
    if (not (csDesigning in ComponentState)) and Stuck then
    begin
      if not fBlack then
      begin
        fBlack:=true;
        invalidate;
      end;
      for I:= 0 to parent.ControlCount-1 do
        if (parent.Controls[I] is TPanelButton) then
          if (parent.Controls[I] as TPanelButton)<>self then
            if (parent.Controls[I] as TPanelButton).Stuck then
             (parent.Controls[I] as TPanelButton).unStick;
    end;
    inherited;
  end;
end;

procedure TPanelButton.MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  inherited MouseUp(Button, Shift, X, Y);
  PanelUp;
  fMousePressed:=false;
end;

procedure TPanelButton.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  inherited MouseDown(Button, Shift, X, Y);
  PanelDown;
  fMousePressed:=true;
  fOutof:=false;         {  Mouse is over TPanelButton}
end;

procedure TPanelButton.MouseMove(Shift: TShiftState; X, Y: Integer);
var
  p:TPoint;
  xx, yy, kk: integer;
  found:boolean;
begin
  inherited MouseMove(Shift, X, Y);
  if fMousePressed then
  begin
    xx:=X;
    yy:=Y;
    if fSentByDoMove then
    begin
       p:=GetPosition(fControlSender);
       if xx<>0 then
         xx:=xx+p.x;
       if yy<>0 then
         yy:=yy+p.y;
    end;
    if ((xx < 0) or (xx >ClientWidth) or (yy < 0) or (yy > ClientHeight))
        and not fUpState then
    begin
      fOutof:=true;     {  Mouse is off TPanelButton}
      PanelUp;
    end;
    if ((xx > 0) and (xx < ClientWidth) and (yy > 0) and (yy < ClientHeight))
       and  fUpState then
        PanelDown;
    if ( ( (-1 < xx)and(xx < 1) ) or ( (ClientWidth+1>xx)and(xx >ClientWidth-1) ) or
          ( (-1<yy)and(yy<1) )     or ( (ClientHeight+1>YY)and(yy>ClientHeight-1) ) ) then
       invalidate;
   end;
   fSentByDoMove:=false;
end;

procedure TPanelButton.Click;
var
  Form: TCustomForm;
begin
  if not fUpState then
    PanelUp;
  Form := GetParentForm(Self);
  if Form <> nil then
     Form.ModalResult := ModalResult;
  inherited Click;
end;


procedure TPanelButton.PanelUp;
var
  i: integer;
begin
  if  not fUpState then
  begin
    fUpState:=true;
    for I:= 0 to ControlCount -1 do
    begin
     Controls[I].left:=Controls[I].left-BevelWidth;
     Controls[I].top:=Controls[I].top-BevelWidth;
    end;
    repaint;
  end;
end;

procedure TPanelButton.PanelDown;
var
  i: integer;
begin
  if  fUpState then
  begin
    fUpState:=false;
    for I:= 0 to ControlCount -1 do
    begin
      Controls[I].left:=Controls[I].left+BevelWidth;
      Controls[I].top:=Controls[I].top+BevelWidth;
    end;
    repaint;
   end;
end;


function TPanelButton.GetPosition(MyControl:TControl):TPoint;
var
 MyControlMM:Tcontrol;
begin
  result.x:= MyControl.left;
  result.y:= MyControl.top;
  MyControlMM:=MyControl;
  if MyControlMM.parent<>self then
    while (MyControlMM<>self) and MyControlMM.hasparent  do
    begin
      MyControlMM:=MyControlMM.parent;
      if MyControlMM<>self then
      begin
        result.x:=result.x+MyControlMM.Left;
        result.y:=result.y+ MyControlmm.top;
      end;
    end;
end;


function TPanelButton.FindControl(var ind:integer): boolean;
var
  ThePoint:TPoint;

  function ControlFound(MyControl:TControl): boolean;
  var
    ControlPosition: TPoint;
  begin
   result:=false ;
   if MyControl=fimage then
     exit;
   ControlPosition:=GetPosition(MyControl);
   if (ThePoint.x>=ControlPosition.x) and (ThePoint.x<=ControlPosition.x+MyControl.width) and
     (ThePoint.y>=ControlPosition.y) and (ThePoint.y<=ControlPosition.y+MyControl.height) then
   begin
     xxname:=MyControl.name;
      result:=true;
   end
  end;

begin
  GetCursorPos(ThePoint);
  ThePoint:=ScreenToClient(ThePoint);
  ind:=fControlsList.Count-1;
  result:=false;
  while (ind>=0) and not result do
    if ControlFound(TControl(fControlsList[ind])) then
      result:=true
    else
      ind:=ind-1;
end;

function TPanelButton.GetIndex(TheControl:TControl): integer;
var
 j:integer;
begin
  result:=-1;
  for j:=0 to fControlsList.count-1 do
     if TheControl=TControl(fControlsList[j]) then
       result:=j;
end;

procedure TPanelButton.DoUp(sender:Tobject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var
 OnMouseUpMM: TMouseEvent;
 kk:integer;
begin
  MouseUp(Button,Shift,X, Y);
  if FindControl(kk) then
  begin
    xxname:=TControl(fControlsList[kk]).name;
    OnMouseUpMM:=TMultiEventsItem(fDEventsRecorder.EventsItem[kk]).OnMouseUp;
    if assigned(OnMouseUpMM) then
        OnMouseUpMM(TControl(fControlsList[kk]) as Tobject,Button,Shift,X,Y);
  end    ;
  if assigned(OnMouseUp) then
      OnMouseUp(sender,Button,Shift,X,Y);

  xxName:=TControl(sender).Name;

  if not fOutof then
  begin
   {Call PanelButton click event handler}
   OnClick:=self.OnClick;
   if assigned(OnClick) then
     OnClick(sender);
  end;

end;

procedure TPanelButton.DoDown(sender:Tobject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var
  kk:integer;
  OnMouseDownMM: TMouseEvent;
begin
  MouseDown(Button,Shift,X, Y);
  if FindControl(kk) then
  begin
    fClickSender:=TControl(fControlsList[kk]);
    OnMouseDownMM:=TMultiEventsItem(fDEventsRecorder.EventsItem[kk]).OnMouseDown;
    if assigned(OnMouseDownMM) then
        OnMouseDownMM(TControl(fControlsList[kk]) as Tobject,Button,Shift,X,Y);
  end    ;
  if assigned(OnMouseDown) then
      OnMouseDown(sender,Button,Shift,X,Y);
end;


procedure TPanelButton.DoMove(sender:Tobject;Shift: TShiftState; X, Y: Integer);
var
  kk: integer;
  OnMouseMoveMM: TMouseMoveEvent;
  mycontrol, mycontrolTT:TControl;
begin
  fSentByDoMove:=true;
  fControlSender:=(sender as TControl);
  MouseMove(Shift, X, Y);
  if FindControl(kk) then
  begin
    OnMouseMoveMM:=TMultiEventsItem(fDEventsRecorder.EventsItem[kk]).OnMouseMove;
    if assigned(OnMouseMoveMM) then
      OnMouseMoveMM(TControl(fControlsList[kk]) as Tobject,Shift,X,Y);
  end;
end;

procedure TPanelButton.DoClick(Sender: Tobject);
var
  ii, kk : integer;
  OnClickMM: TNotifyEvent;
  DoIt:boolean;
  MyControl:Tcontrol;
begin

 { Sender's event handler is to be  called
   Sender control is owned by TPanelButton since DoClick called  }

  { Get Sender}
  If fClickSender<> nil then
    MyControl:=TControl(fClickSender)
  else
    MyControl:=TControl(Sender);

  fClickSender:=nil;

  If MyControl=fImage then
    DoIt:=true    { Ready to call PanelButton click event handler }
  else
  begin
    { Find and execute event handler of the Sender }
    ii:=GetIndex(MyControl);
    if (ii>-1) and (ii<=fControlsList.count-1)  then
    begin
      OnClickMM:=TMultiEventsItem(fDEventsRecorder.EventsItem[ii]).OnClick;
      if assigned(OnClickMM) then
        OnClickMM(TControl(fControlsList[ii]) as Tobject);
    end;
    { Find out if mouse is over sender - to call PanelButton click event handler }
    DoIt:=FindControl(kk);
    DoIt:=(DoIt and (ii=kk));
  end;
end;

procedure TPanelButton.Paint;
const
  Alignments: array[TAlignment] of Word = (DT_LEFT, DT_RIGHT, DT_CENTER);
var
  Rectan: TRect;
  Text: array[0..255] of Char;
  FontHeight: Integer;
  fSpacing: integer;

begin
  Rectan  := GetClientRect;
  fSpacing:=2;
  if Focused  or (not fUpState) then
  begin
    if fUpState  then
    begin
       if ( (csDesigning in ComponentState)) or( (not Stuck) or ( Enabled)) then
       begin
          Frame3D(Canvas, Rectan, clblack,clblack ,1) ;
          if fblack then
            Frame3D(Canvas, Rectan, fTopColor,fBottomColor ,1)
       end;
       Frame3D(Canvas, Rectan, fTopColor, fBottomColor,BevelWidth);
    end
    else
    begin
      Frame3D(Canvas, Rectan, clblack,fTopColor ,1);
      Frame3D(Canvas, Rectan, fBottomColor,fBottomColor,1);
    end
  end
  else
  begin
    if ( (csDesigning in ComponentState)) or( (not Stuck) or (fBlack and Enabled)) then
         Frame3D(Canvas, Rectan, fTopColor,clblack ,1);
    Frame3D(Canvas, Rectan, fTopColor,fBottomColor,BevelWidth);
  end;
  with Canvas do
  begin
    Brush.Color := Color;
    FillRect(Rectan);
    if Focused  or (not fUpState) then
    begin
       if ( (csDesigning in ComponentState)) or( (not Stuck) or (fBlack and Enabled)) then
          DrawFocusRect(Rect(Rectan.Left + BevelWidth+FSpacing-2,
                       Rectan.Top + BevelWidth+FSpacing-2,
                       Rectan.Right -BevelWidth-FSpacing+2 ,
                       Rectan.Bottom - BevelWidth-FSpacing+2))
         else
          if fUpState  then
            DrawFocusRect(Rect(Rectan.Left + BevelWidth+FSpacing-1,
                       Rectan.Top + BevelWidth+FSpacing-1,
                       Rectan.Right -BevelWidth-FSpacing+1 ,
                       Rectan.Bottom - BevelWidth-FSpacing+1))
         else
            DrawFocusRect(Rect(Rectan.Left + BevelWidth-2,
                       Rectan.Top + BevelWidth-2,
                       Rectan.Right -BevelWidth+2 ,
                       Rectan.Bottom - BevelWidth+2))

    end;
    Font := Self.Font;
    FontHeight := TextHeight('W');
    with Rectan do
    begin
      Left:=Left+BevelWidth+FSpacing+2;
      right:=right-BevelWidth-FSpacing-2;
      Top := Top+BevelWidth+FSpacing+2;
      Bottom := Bottom-BevelWidth-FSpacing-2;
      Top := ((Bottom + Top) - FontHeight) shr 1;
      Bottom := Top + FontHeight;
      if fUpState then
      begin
        Left:=Left-BevelWidth;
        right:=right-BevelWidth;
        Top:=Top-BevelWidth;
      end;
      if ( (csDesigning in ComponentState)) or( (not Stuck) or (fBlack and Enabled)) then
        right:=right+1;
    end;

    StrPCopy(Text, Caption);
    if not Enabled then
    begin
      Brush.Style := bsClear;
      OffsetRect(Rectan, 1, 1);
      Font.Color := clBtnHighlight;
      DrawText(Handle, Text, StrLen(Text), Rectan,(DT_EXPANDTABS or DT_VCENTER) or Alignments[FAlignment]);
      OffsetRect(Rectan, -1, -1);
      Font.Color := clBtnShadow;
      DrawText(Handle, Text, StrLen(Text), Rectan,(DT_EXPANDTABS or DT_VCENTER) or Alignments[FAlignment]);
    end
    else
      DrawText(Canvas.Handle, Text, StrLen(Text), Rectan,
        (DT_EXPANDTABS or DT_VCENTER) or Alignments[FAlignment]);
  end;
end;


end.
