unit OxOFile;
{Delphi Add-in Expert Overrides Ctrl-Enter to quickly open files}
{Copyright 1997 Eric Maddox}

interface
uses
  Forms,
  Windows,
  EditIntf,
  ExptIntf,
  ToolIntf,
  Controls,
  Classes,
  Messages,

  Dialogs {For Debug};


type
  TFileOpener = class(TIExpert)
  private
  {Overriding to avoid abstract error}
    function GetComment: string; Override;
    function GetPage: string; Override;
    function GetGlyph: HICON; Override;
    function GetMenuText: string; Override;
    function GetState: TExpertState; Override;
    procedure Execute; Override;
  protected
    OldOnMessage: TMessageEvent;
    Function DoKeyDown(var Message: TWMKey): Boolean;
    procedure HookEditor;
    procedure UnHookEditor;
    Function GetCursorFileName: String;
  public
    Function Open(FileName: String): Boolean;
    Function InEditControl: Boolean;
    Procedure OnMessage(Var Msg: TMsg; Var Handled: Boolean);
    Procedure KeyDown(var Key: Word; Shift: TShiftState);
    constructor Create; virtual;
    destructor Destroy; override;
    function GetName: string; override;
    function GetAuthor: string; override;
    function GetStyle: TExpertStyle; override;
    function GetIDString: string; override;
  end;

procedure Register;
implementation
Uses
  OXFOpen,
  Sysutils;

procedure Register;
begin
  RegisterLibraryExpert(TFileOpener.Create);
end;


{ TFileOpener code }

  {Overriding to avoid abstract error}
function TFileOpener.GetComment: string;
begin
  Result := 'FileOpener'
end;

function TFileOpener.GetPage: string;
begin
  Result := 'FileOpener'
end;

function TFileOpener.GetState: TExpertState;
begin
  Result := [];
end;
function TFileOpener.GetGlyph: HICON;
begin
  Result := 0;
end;

function TFileOpener.GetMenuText: string;
begin
  Result := 'FileOpener'
end;

procedure TFileOpener.Execute;
begin

end;

function TFileOpener.GetName: String;
begin
  Result := 'FileOpener'
end;

function TFileOpener.GetAuthor: String;
begin
  Result := 'Eric_Maddox'; { author }
end;

function TFileOpener.GetStyle: TExpertStyle;
begin
  Result := esAddIn;
end;

function TFileOpener.GetIDString: String;
begin
  Result := 'OX.QuickFileOpen';
end;

constructor TFileOpener.Create;
begin
  inherited Create;
  HookEditor;
end;

destructor TFileOpener.Destroy;
begin
  UnHookEditor;
  inherited Destroy;
end;{Destroy}

Procedure TFileOpener.HookEditor;
{Hook the Application's MessageProcessor}
begin
  If Application <> nil then
    begin
      OldOnMessage := Application.OnMessage;
      Application.OnMessage := OnMessage;
    end;
end;

(*
Procedure TFileOpener.HookEditor(Owner: TComponent);
{Recursive Proc to find and Hook the editor window}
{Pass Application for Owner}
{Hook the popup menu for Ctrl-Enter}
{This works if user selects popupmenu item, but
 not if user presses Ctrl-Enter}
Var
 Ctr: Integer;
 Item: Integer;
begin
  Ctr := 0;
  While (Ctr < Owner.ComponentCount) and (EditControl = nil) do
    begin
      If Owner.Components[Ctr].Classname = 'TEditControl' then
        begin
          EditControl := Owner.Components[Ctr] as TCustomControl;
          If TForm(EditControl).PopUpMenu <> nil then
            with TForm(EditControl).PopUpMenu do
            For Item := 0 to Items.Count -1 do
             If Items[Item].ShortCut = ShortCut(vk_return,[ssCtrl]) then
               begin
                 MenuItem := Items[Item];
                 OldonClick := Items[Item].OnClick;
                 Items[Item].OnClick := Self.OnClick;
                 Break;
               end;
        end
      else
        HookEditor(Owner.Components[Ctr]);
      Inc(Ctr);
    end;
end;
*)

Procedure TFileOpener.UnHookEditor;
{Restore the old messageProcess hook}
  Function EQUMethods(Const M1, M2): Boolean;
  {M1, and M2 should be event pointers}
  {Returns true if M1=M2}
  Type
    TMethod = packed Record
                Data,
                Proc: Pointer;
              end;
  begin
    Result := (TMethod(M1).Data = TMethod(M2).Data) and
              (TMethod(M1).Proc = TMethod(M2).Proc)
  end;
Var
  FOnMessage: TMessageEvent;
begin
  If Application <> nil then
    begin
      FOnMessage := OnMessage;
    {Only try to unhook the hook we put in}
      {If Application.OnMessage = OnMessage then}
        If EquMethods(Application.OnMessage, FOnMessage) then
          Application.OnMessage := OldOnMessage;
    end;
end;

Function TFileOpener.InEditControl: Boolean;
{Returns true if user is editing code}
var
  EditControl: TWinControl;
begin
  EditControl := FindControl(GetFocus);
  If (EditControl <> nil) and (EditControl.Classname = 'TEditControl') then
    Result := True {The user is editing code}
  else {User is not editing code or we didn't find the EditControl}
    Result := False;
end;

Var
  FakedCtrl :Boolean; {True if we are Faking holding down the Ctrl Key}
Procedure FakeCtrl(Value: Boolean);
{If Value Simulate Pressing the Ctrl Key
 If Value = False release the Ctrl Key}
var
  KeyState: TKeyboardState;
begin
  Win32Check(GetKeyboardState(KeyState));
  If Value then
    KeyState[VK_CONTROL] := $80
  else
    KeyState[VK_CONTROL] := $01;

  Win32Check(SetKeyboardState(KeyState));
  FakedCtrl := Value;
end;

Function FindAFileName(Line: String; Cursor: Integer): String;
{Line is a line of text,
Cursor is the Zero based position of the cursor on the line}
var
  Start: integer;
  Mid: Integer; {One Based}
begin
  {Search from cursor pos -1 back to beginning for invalid char}
  Mid := Cursor; {Cursor is 0 based, Mid is 1 based. Mid is 1 less than cursor}
  While (Mid > 0) and not (Line[Mid] in [#0..#32,',',';',#127..#255]) do
    Dec(Mid);
  Start := Mid + 1;
  {Search from cursor pos Forward to end for invalid char}
  Mid := Cursor + 1;
  While (Mid < Length(Line)) and not (Line[Mid] in [#0..#32,',',';',#127..#255]) do
    Inc(Mid);
  Result := Trim(Copy(Line, Start, Mid - Start));
end;

Function TFileOpener.GetCursorFileName: String;
{Get the filename from text near the cursor}
Var
  ModIntf: TIModuleInterface;
  EditorIntf: TIEditorInterface;
  Reader: TIEditReader;
  View: TIEditView;
  CPos,
  BeginPos: TCharPos; {Beginning of line}
  EditPos: TEditPos;
  FilePos: LongInt; {Position in reader}
  Line: String;
begin
  Result := '';
  Line := '';
  If ToolServices <> nil then
    begin
      With ToolServices do
        ModIntf := GetModuleInterface(GetCurrentFile);
      If ModIntf <> nil then
        try
          EditorIntf := ModIntf.GetEditorInterface;
          If EditorIntf <> nil then
            Try
              Reader := EditorIntf.CreateReader;
              If Reader <> nil then
                try
                  If EditorIntf.GetViewCount > 0 then
                    begin
                      View := EditorIntf.GetView(0);
                      If View <> nil then
                        begin
                          Try
                            With View do
                              begin
                                {to avoid VAR param problem}
                                EditPos := CursorPos;
                                ConvertPos(True, EditPos, CPos);
                                BeginPos := CPos;
                                BeginPos.CharIndex := 0;
                                FilePos := CharPostoPos(BeginPos);
                              end;
                          Finally
                            View.Free;
                          end;
                          {Get from the beginning of the line to
                           80 chars past the cursor pos}
                          SetLength(Line, Cpos.CharIndex + 80);
                          SetLength(Line, Reader.GetText(FilePos, PChar(Line),
                            Length(Line)));
                        end; {if}
                    end; {IF}
                Finally
                  Reader.Free;
                end;
            Finally
              EditorIntf.Free;
            end;
        Finally
          ModIntf.Free;
        end;

      Result := FindAFileName(Line, CPos.CharIndex);
    end; {if}
end;


Procedure TFileOpener.KeyDown(var Key: Word; Shift: TShiftState);
begin
  If (Key = vk_Return) and (Shift = [ssCtrl]) then
    If InEditControl then
      begin
        QuickOpenFileDialog := TQuickOpenFileDialog.Create(nil);
        With QuickOpenFileDialog do
          try
            Configure(Toolservices);
            FileName := GetCursorFileName;
            Case ShowModal of
              mrOK: {Open}
                begin
                  Open(FullPath);
                  Key := 0; {Message is handled}
                end;
              mrIgnore: {Browse} {Let the default handler take care of it.}
                Begin
                  FakeCtrl(True);
                  {The message key is Enter,
                  Fake a control key to get the origional Ctrl-Enter behavior}
                end;
              mrCancel:
                Key := 0; {Message is handled}
              else
                Raise Exception.Create('Unexpected Modal Result');
              end;{Case}
            Finally
              Free;
              QuickOpenFileDialog := nil;
            end;
      end;
end; {KeyDown}

function TFileOpener.DoKeyDown(var Message: TWMKey): Boolean;
{Result = true if handled}
var
  ShiftState: TShiftState;
begin
  Result := True;
  with Message do
    begin
      ShiftState := KeyDataToShiftState(KeyData);
      KeyDown(CharCode, ShiftState);
      if CharCode = 0 then
        Exit; {Result = True, Message has been handled}
    end;
  Result := False;
end;

Procedure TFileOpener.OnMessage(var Msg: TMsg; var Handled: Boolean);
Var
  WMKeyDown: TWMKeyDown;
begin
  try
    If FakedCtrl Then
      FakeCtrl(False); {Let the Ctrl Key up}
    Case Msg.Message of
      WM_KeyDown:
        begin
          WMKeyDown.Msg := Msg.Message;
          WMKeyDown.CharCode := Msg.WParam;
          WMKeyDown.KeyData := Msg.LParam;
          Handled := DoKeyDown(WMKeyDown);
        end;
    end;
    {Chain to previous Message Handler}
    try
      If Not Handled and Assigned(OldOnMessage) then
        OldOnMessage(Msg, Handled);
    except
      On EAccessViolation do
        OldOnMessage := nil; {It gets 1 chance, if it crashes it's outta here}
    end;
  except {If except isn't here Delphi Crashes}
    On E: Exception do
      ShowException(E,ExceptAddr);
  end;
end;

Function TFileOpener.Open(FileName: String): Boolean;
Var
  ModInterface: TIModuleInterface;
  CurrentProject: String;
begin
  Result := False;
  If ToolServices <> nil then
    begin
      If UpperCase(ExtractFileExt(FileName)) = '.DPR' then
        begin
          CurrentProject := Toolservices.GetProjectName;
          ShowMessage('CurrProj: ' + CurrentProject);
          If Toolservices.OpenProjectInfo(FileName) then
            begin
              ModInterface := Toolservices.GetModuleInterface(FileName);
              if (ModInterface <> nil) then
                try
                  If CurrentProject <> '' then
                    begin
                      ShowMessage('Toolservices.GetProjectName: ' + Toolservices.GetProjectName);
                      IF ToolServices.CloseFile(CurrentProject) then
                        ShowMessage('Closed: ' + CurrentProject);
                      If ToolServices.OpenProjectInfo(CurrentProject) then
                        ShowMessage('Reopened Toolservices.GetProjectName: ' + Toolservices.GetProjectName);
                    end;
                  If ModInterface.ShowSource then
                    ShowMessage('ShowSource');
                finally
                  ModInterface.Release;
                  ShowMessage('Released');
                end;
            end;
        end
      else
        Result := ToolServices.OpenFile(FileName)
    end;
end;

end.
