unit OXFOpen;

interface


uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms,
  StdCtrls, ToolIntf, Registry, OXUPath;

type
  TQuickOpenFileDialog = class(TForm)
    Button1: TButton;
    Button2: TButton;
    Button3: TButton;
    FileCB: TComboBox;
    Label2: TLabel;
    GroupBox1: TGroupBox;
    PathCB: TComboBox;
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure Button1Click(Sender: TObject);
  private
    Reg: TRegistry;
    {These procedures are private because they use REG}
    Procedure FillHistory(KeyName: String; HistoryList: TStrings);
    Procedure SaveHistory(Const KeyName, LastValue :String; HistoryList: TStrings);
    Procedure SaveDisabled;
    Function CheckDisabled(Caption: String): Boolean;
    Function CreateCheckBox(Num: Integer; Caption: String): TCheckBox;
  protected
    FSearchPath: TPathList;
    FToolServices: TIToolServices;
    Function GetFileName: String;
    Function GetSearchPath: TPathList;
    Procedure SetFileName(Const FName: String);
    Function SearchProjectUnits(Var Filename: String): Boolean;
  public
    Paths: TStrings;
    CheckBoxes: TStrings; {Objects of this list are checkboxes}
    FullPath: String; {The return value of this dialog} 
    Property SearchPathList: TPathList Read GetSearchPath;
    Property FileName: String Read GetFileName Write SetFileName;
    Procedure Configure(Toolservices: TIToolServices);
  end;

var
  QuickOpenFileDialog: TQuickOpenFileDialog;

implementation
{$R *.DFM}
Uses OXReg, OxMacro;

Function TQuickOpenFileDialog.GetFileName: String;
begin
  Result := Trim(FileCB.Text);
end;

Procedure TQuickOpenFileDialog.SetFileName(Const FName: String);
  Function NoQuote(S: String): String;
  begin
    If (Length(S) > 1) and (S[1] in ['''','"']) and (S[Length(S)] = S[1]) then
      Result := Copy(S, 2, Length(S) -2)
    else
      Result := S;
  end;
begin
  FileCB.Text := NoQuote(Trim(FName));
end;


Function NoCaseIndexOf(S: String; List: TStrings): Integer;
var
  Ctr: Integer;
begin
  S := UpperCase(S); {only change the local copy}
  Result := -1; {Not Found}
  For Ctr := 0 to List.Count -1 do
    begin
      If S = UpperCase(List[Ctr]) then
        Result := Ctr;
    end;
end;

Procedure TQuickOpenFileDialog.SaveHistory(Const KeyName, LastValue :String; HistoryList: TStrings);
{Save a history list in the registry}
var
  Ctr: Integer;
  LastValuePos: Integer;
begin
  IF Reg.OpenKey(RegKey + '\' + KeyName, True) then
    begin
      LastValuePos := NoCaseIndexOf(LastValue, HistoryList);
      If LastValuePos > -1 then {If it was found delete the old one}
        HistoryList.Delete(LastValuePos);
      HistoryList.insert(0,LastValue);
      If HistoryList.Count > MaxHistory then
        HistoryList.Delete(HistoryList.Count-1);
      Reg.WriteString('Count',IntToStr(HistoryList.Count)); {Just because Borland does it}
      For Ctr := 0 to HistoryList.Count -1 do
        Reg.WriteString('H' + IntToStr(Ctr), HistoryList[Ctr]);
    end;
end;

Procedure TQuickOpenFileDialog.FillHistory(KeyName: String; HistoryList: TStrings);
{Read a history list}
Var
  Ctr : Integer;
  HistoryValues: TStringList;
begin
  HistoryList.Clear;
  If Reg.OpenKey(RegKey + '\' + KeyName, False) then
    begin
      HistoryValues := TStringList.Create;
      Try
        Reg.GetValueNames(HistoryValues);
        For Ctr := 0 to HistoryValues.Count -1 do
          begin
            If (HistoryValues[Ctr] <> 'Count') then
              HistoryList.Add(Reg.ReadString(HistoryValues[Ctr]));
          end;
      Finally
        HistoryValues.Free;
      end;
    end;
end;

Procedure TQuickOpenFileDialog.SaveDisabled;
{Save the state of each checkbox }
Var
  Ctr: Integer;
begin
  If Reg.OpenKey(RegKey + '\Disabled', True) then
    begin
      With GroupBox1 do
        For Ctr := 0 to ControlCount -1 do
          If Controls[Ctr] is TCheckBox then
            If TCheckBox(Controls[Ctr]).Checked then
              Reg.DeleteValue(TCheckBox(Controls[Ctr]).Caption)
            else
              Reg.WriteString(TCheckBox(Controls[Ctr]).Caption, 'Unchecked');
    end;
end;

Function TQuickOpenFileDialog.CheckDisabled(Caption: String): Boolean;
{Returns True if the CheckBox is NOT checked}
begin
  Result := False;
  If Reg.OpenKey(RegKey + '\Disabled', False) then
    begin
      If Reg.ReadString(Caption) <> '' then
        Result := True;
    end;
end;

Function TQuickOpenFileDialog.CreateCheckBox(Num: Integer; Caption: String): TCheckBox;
{Num is 0 to maxPaths -1}
begin
  Result := TCheckBox.Create(Self);
  {The form is the owner and will free the checkboxes automatically}
  If Num < MaxPaths div 2 then {0,1,2 on left}
    Result.Left := 4
  else {Put checkbox in right column}
    Result.Left := GroupBox1.Width Div 2 + 2;
  Result.Width := GroupBox1.Width Div 2 - 3;
  Result.Top := 16 + Num Mod (MaxPaths div 2) * (Result.Height + 2);
  Result.Caption := Caption;
  Result.Parent := GroupBox1;
  Result.Checked := Not CheckDisabled(Caption);
  Result.ShowHint := True;
end;


Procedure TQuickOpenFileDialog.Configure(Toolservices: TIToolServices);
Var
  Ctr: Integer;
begin
  FToolServices := ToolServices;
  InitMacros(ToolServices);
  FillPathItems(CheckBoxes, Paths);
  Reg := TRegistry.Create;
  try
    For Ctr := 0 to CheckBoxes.Count -1 do
      begin
        CheckBoxes.Objects[Ctr] := CreateCheckBox(Ctr, CheckBoxes[Ctr]);
        Paths[Ctr] := ExpandPath(Paths[Ctr]);
        TCheckBox(CheckBoxes.Objects[Ctr]).Hint := Paths[Ctr];
      end;
    FillHistory('hiPath', PathCB.Items);
    FillHistory('hiFile', FileCB.Items);
    If PathCB.Items.Count > 0 then
      PathCB.Text := PathCB.Items[0]
    else
      PathCB.Text := '';
    FileName := '';
  finally
    Reg.Free;
  end;
end;

procedure TQuickOpenFileDialog.FormClose(Sender: TObject; var Action: TCloseAction);
begin
  If ModalResult = mrOK then
    begin
      Reg := TRegistry.Create;
      try
        SaveHistory('hiPath', PathCB.Text, PathCB.Items);
        SaveHistory('hiFile', FileCB.Text, FileCB.Items);
        SaveDisabled;
      finally
        Reg.Free;
      end;
    end;
end;

Function TQuickOpenFileDialog.SearchProjectUnits(Var Filename: String): Boolean;
{The project file may contain files like
  uses unit1 in 'C:\units\unit1.pas',
Filename would be changed from unit1 to C:\units\unit1.pas}
{Returns true if filename is found among the project files} 
var
  Ctr: Integer;
  UnitCount: integer;
  LocalFileName: String;
begin
  Result := False;
  If FToolServices <> nil then
    begin
      unitCount := FtoolServices.GetUnitCount;
      LocalFileName :=UpperCase(FileName);
      If ExtractFileExt(LocalFileName) = '' then
        LocalFileName := ChangeFileExt(LocalFileName,'.PAS');
      For Ctr := 0 to UnitCount -1 do
        begin
          If LocalFileName = UpperCase(ExtractFileName(FToolServices.GetUnitName(Ctr))) then
            begin
              Result := True;
              FileName := FToolServices.GetUnitName(Ctr);
              Break;
            end;
        end;
    end;
end;

Function TQuickOpenFileDialog.GetSearchPath: TPathList;
{This Form will free the result in the form destroy}
var
  Ctr: Integer;
begin
  If FSearchPath = nil then
    FSearchPath := TPathList.Create
  else
    FSearchPath.Clear;

  FSearchPath.AddPaths(PathCB.Text); {First search user entry}

  if FToolServices <> nil then {Search the project directory}
    FSearchPath.Add(ExtractFilePath(FToolServices.GetProjectName));
    
  For Ctr := 0 to CheckBoxes.Count -1 do
    If TCheckBox(CheckBoxes.Objects[Ctr]).Checked then
      FSearchPath.AddPaths(Paths[Ctr]);

  Result := FSearchPath;
end;

Function GetDLLFileName: String;
{Returns the Path and File name of the current module}
Var
  NewLen: integer;
begin
  NewLen := 255;
  SetLength(Result, NewLen);
  NewLen := GetModuleFileName(HInstance, PChar(Result), NewLen);
  If NewLen = 0 then
    RaiseLastWin32Error;
  SetLength(Result, NewLen);
end;

procedure TQuickOpenFileDialog.FormCreate(Sender: TObject);
begin
{sets the Helpfile to the filename of the EXE in the same path}
  HelpFile := ChangeFileExt(GetDLLFileName,'.hlp');
  CheckBoxes:= TStringList.Create;
  Paths := TStringList.Create;
end;

procedure TQuickOpenFileDialog.FormDestroy(Sender: TObject);
begin
  Paths.Free;
  CheckBoxes.Free;
end;

Function FSearch(Const Path, FileName:String): String;
{Search for FileName on Path}
var
  Len: Integer;
  FileNoPath: PChar;
  Return: Integer;
begin
  Len := 255;
  SetLength(Result, Len);
  Return := SearchPath(PChar(Path),
              PChar(FileName),'.PAS', len,
              PChar(Result),FileNoPath);
  If Return > Len then {Call with a bigger buffer}
    begin
      Len := Return;
      SetLength(Result, Len);
      Return := SearchPath(PChar(Path),
               PChar(FileName),'.PAS', Len,
               PChar(Result),FileNoPath);
    end;
  If Return = 0 then
    RaiselastWin32Error;
  SetLength(Result, Return);
end;

procedure TQuickOpenFileDialog.Button1Click(Sender: TObject);
var
  FFileName: String;
begin
  Try
    Screen.Cursor := crHourGlass;
    try
      If ((Pos('\', FileName) > 0) And  { If Path specified and fileexists }
         FileExists(FileName)) then    { then skip Search}
        FullPath := FileName
      else
        begin
          FFileName := FileName;
          If SearchProjectUnits(FFileName) then
            FullPath := FFileName
          else
            FullPath := FSearch(SearchPathList.AsString, FileName);
        end;

      {If a file was not found we will get an exception and not set modalresult}
      ModalResult := mrOK; {Close the dialog}

    finally
      SCreen.Cursor := crDefault;
    end;
  except
    On E:EWin32Error do
      begin
        If E.ErrorCode = 2 then
          MessageBox(Handle,'File not found','Open File', MB_ICONEXCLAMATION)
        else
          Raise;
      end;
  end;
end;

end.
