{
  TfrmImageManager
  (c) Ilya Vdovin, 1999
}

unit ImageManager;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls, ComCtrls, StdCtrls, Buttons, Spin, RXSplit, Menus, RXCtrls,
  ExtDlgs, OTMisc;

type
  TCompareMode = (cmName, cmSize, cmDate);

  TfrmImageManager = class(TForm)
    VertSplitter: TRxSplitter;
    PRight: TPanel;
    PUp: TPanel;
    Images: TImageList;
    FImages: TImageList;
    FImagesBig: TImageList;
    PListBtn: TPanel;
    BIcons: TRxSpeedButton;
    BSmallIcons: TRxSpeedButton;
    BList: TRxSpeedButton;
    BReport: TRxSpeedButton;
    BThumb: TRxSpeedButton;
    BView: TRxSpeedButton;
    BDelImg: TRxSpeedButton;
    BCopyMode: TRxSpeedButton;
    BMoveMode: TRxSpeedButton;
    BOptions: TRxSpeedButton;
    mOptions: TPopupMenu;
    mShowButtons: TMenuItem;
    mTransparent: TMenuItem;
    N1: TMenuItem;
    N2: TMenuItem;
    mAutoSave: TMenuItem;
    mActions: TPopupMenu;
    mScan: TMenuItem;
    N3: TMenuItem;
    mAbout: TMenuItem;
    mCropToSquare: TMenuItem;
    mShowStatus: TMenuItem;
    PBottom: TPanel;
    PButtons: TPanel;
    BOk: TBitBtn;
    BCancel: TBitBtn;
    PProgress: TPanel;
    ProgressBar: TProgressBar;
    PLeft: TPanel;
    PTree: TPanel;
    Tree: TTreeView;
    PTreeBtns: TPanel;
    BAdd: TRxSpeedButton;
    BDel: TRxSpeedButton;
    BBrowse: TRxSpeedButton;
    BSave: TRxSpeedButton;
    BClear: TRxSpeedButton;
    BAction: TRxSpeedButton;
    PDown: TPanel;
    PBtnEx: TPanel;
    BEx1: TRxSpeedButton;
    BEx2: TRxSpeedButton;
    BEx3: TRxSpeedButton;
    PImage: TPanel;
    Image: TImage;
    HorizSplitter: TRxSplitter;
    PClipbrd: TPanel;
    BCopy: TRxSpeedButton;
    BPaste: TRxSpeedButton;
    StatusBar: TStatusBar;
    BSaveImg: TRxSpeedButton;
    SaveDlg: TSavePictureDialog;
    mRestorePath: TMenuItem;
    mDecreaseColors: TMenuItem;
    BOpenBar: TRxSpeedButton;
    BvToolBarSep: TBevel;
    BCopyAgain: TRxSpeedButton;
    PFileList: TPanel;
    FileList: TListView;
    mHotTrack: TMenuItem;
    N4: TMenuItem;
    mGridLines: TMenuItem;
    mShowRoots: TMenuItem;
    mShowLines: TMenuItem;
    mGrayImages: TMenuItem;
    LProgressInfo: TLabel;
    mFromIcon: TMenuItem;
    mSaveLast: TMenuItem;
    procedure FormShow(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure TreeEdited(Sender: TObject; Node: TTreeNode; var S: String);
    procedure TreeEditing(Sender: TObject; Node: TTreeNode;
      var AllowEdit: Boolean);
    procedure TreeKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure TreeChange(Sender: TObject; Node: TTreeNode);
    procedure BAddClick(Sender: TObject);
    procedure BDelClick(Sender: TObject);
    procedure BSaveClick(Sender: TObject);
    procedure BIconsClick(Sender: TObject);
    procedure FileListChange(Sender: TObject; Item: TListItem;
      Change: TItemChange);
    procedure FileListDblClick(Sender: TObject);
    procedure TreeDragOver(Sender, Source: TObject; X, Y: Integer;
      State: TDragState; var Accept: Boolean);
    procedure FileListCompare(Sender: TObject; Item1, Item2: TListItem;
      Data: Integer; var Compare: Integer);
    procedure FileListColumnClick(Sender: TObject; Column: TListColumn);
    procedure FileListEditing(Sender: TObject; Item: TListItem;
      var AllowEdit: Boolean);
    procedure FileListEdited(Sender: TObject; Item: TListItem;
      var S: String);
    procedure BBrowseClick(Sender: TObject);
    procedure FileListKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure BClearClick(Sender: TObject);
    procedure TreeDeletion(Sender: TObject; Node: TTreeNode);
    procedure TreeDragDrop(Sender, Source: TObject; X, Y: Integer);
    procedure BThumbClick(Sender: TObject);
    procedure BViewClick(Sender: TObject);
    procedure BDelImgClick(Sender: TObject);
    procedure mShowButtonsClick(Sender: TObject);
    procedure mTransparentClick(Sender: TObject);
    procedure mAutoSaveClick(Sender: TObject);
    procedure PImageResize(Sender: TObject);
    procedure PUpResize(Sender: TObject);
    procedure mAboutClick(Sender: TObject);
    procedure mCropToSquareClick(Sender: TObject);
    procedure mShowStatusClick(Sender: TObject);
    procedure mScanClick(Sender: TObject);
    procedure TreeDblClick(Sender: TObject);
    procedure BCopyClick(Sender: TObject);
    procedure BPasteClick(Sender: TObject);
    procedure TreeExit(Sender: TObject);
    procedure FileListExit(Sender: TObject);
    procedure BSaveImgClick(Sender: TObject);
    procedure mRestorePathClick(Sender: TObject);
    procedure mDecreaseColorsClick(Sender: TObject);
    procedure BOpenBarClick(Sender: TObject);
    procedure BCopyAgainClick(Sender: TObject);
    procedure mHotTrackClick(Sender: TObject);
    procedure mGridLinesClick(Sender: TObject);
    procedure mShowRootsClick(Sender: TObject);
    procedure mShowLinesClick(Sender: TObject);
    procedure mGrayImagesClick(Sender: TObject);
    procedure mFromIconClick(Sender: TObject);
    procedure mSaveLastClick(Sender: TObject);
  private
    FDirRoot, FCatRoot, FLstRoot: TTreeNode;
    FChanges: Boolean;
    FComp: TCompareMode;
    FAscSort: Boolean;
    FStarting: Boolean;
    FReading: Boolean;
    FSelectedImageName: String;
    FStartDir: String;
    FLastTarget: TTreeNode;
    FWasAutoAdd: Boolean;
    FTranspColor: TColor;
    procedure UpdateBtnEx;
    procedure SaveToRegistry(SaveTreeData: Boolean);
    procedure LoadFromRegistry;
    procedure SetIndexs(Node: TTreeNode; Idx: Integer);
    procedure HasChange;
    procedure CheckIcon(FileName: String);
    procedure AddFileToList(FileName: String);
    procedure SelectNewByName(FileName: String);
    function BrowseDir(S: String): String;
    procedure FillListForDir(S: String);
    procedure FillFromNodeList(Node: TTreeNode);
    procedure CommonFillEnd;
    function AddNameToNode(Node: TTreeNode; FileName: String): Boolean;
    function DeleteNameFromNode(Node: TTreeNode; FileName: String): Boolean;
    procedure ClearThumbs;
    procedure CreateNewThumbs;
    procedure CropCurrentImage;
    function AddDirNode(Folder: String): TTreeNode;
    procedure UpdateImageMeasure;
    procedure DecreaseColors;
    procedure AdjustToolBarHeight;
    procedure SetLastTarget(Node: TTreeNode);
    procedure CopyOrMoveToNode(TargetNode: TTreeNode);
    procedure GrayImage(Bitmap: TBitmap);
    procedure GrayCurrentImage;
    procedure UpdateCurrentThumbnails;
    procedure FixLinks(OldPath, NewPath: String);
    procedure StartVisual(pText: String; pMax: Integer);
    procedure NextVisual(pValue: Integer);
    procedure DoneVisual;
  public
    Picture: TBitmap;
    SubCaption: String;
  end;

var
  frmImageManager: TfrmImageManager;

implementation

uses ShellApi, Registry, CommCtrl, FileUtil, StrUtils, OTAbout, Info,
     ImgManScan, Clipbrd, ImgManLoadFromIcon;

{$R *.DFM}

const CRegKey = '\Software\Viv\Delphi\OpenTools\ImageManager\';
      CPathKey = '\Software\Borland\Delphi\3.0';

      IDX_DIR_ROOT = 0;
      IDX_CAT_ROOT = 1;
      IDX_LST_ROOT = 2;
      IDX_DIR_ITEM = 3;
      IDX_CAT_ITEM = 4;

      IDX_SEL_INC  = 2;

      IDX_DIR_OPEN = IDX_DIR_ITEM + IDX_SEL_INC;
      IDX_CAT_OPEN = IDX_CAT_ITEM + IDX_SEL_INC;

procedure Error(S: String);
begin
  Application.MessageBox(PChar(S), 'Error', MB_OK or MB_ICONERROR);
end;

procedure Warning(S: String);
begin
  Application.MessageBox(PChar(S), 'Warning', MB_OK or MB_ICONWARNING);
end;

function Confirm(S: String): Boolean;
begin
  Result := MessageDlg(S, mtConfirmation, [mbYes, mbNo], 0) = mrYes;
end;

procedure TfrmImageManager.FormCreate(Sender: TObject);

  function AddRoot(S: String; I: Integer): TTreeNode;
  begin
    Result := Tree.Items.AddChild(nil, S);
    SetIndexs(Result, I);
  end;

begin
  GetDir(0, FStartDir);
  FDirRoot := AddRoot('Directories', IDX_DIR_ROOT);
  FCatRoot := AddRoot('Categories', IDX_CAT_ROOT);
  FLstRoot := AddRoot('Last used', IDX_LST_ROOT);
  FChanges := False;
  FStarting := True;
  FComp := cmName;
  FAscSort := True;
  FWasAutoAdd := False;
  LoadFromRegistry;
  FStarting := False;
  if FWasAutoAdd then HasChange;
  FReading := False;
  Tree.Selected := FLstRoot;
  FSelectedImageName := '';
  FLastTarget := nil;
end;

procedure TfrmImageManager.UpdateImageMeasure;
var S: String;
begin
  if (Image.Picture.Bitmap = nil) or (Image.Picture.Bitmap.Empty) then
    StatusBar.Panels[1].Text := '' else
    begin
      case Image.Picture.Bitmap.PixelFormat of
        pf1bit: S:= ', 2';
        pf4bit: S:= ', 16';
        pf8bit: S:= ', 256';
        pf15bit: S:= ', 32K';
        pf16bit: S:= ', 64K';
        pf24bit: S:= ', 16M';
        pf32bit: S:= ', 16M';
        else S := '';
      end;
      StatusBar.Panels[1].Text := IntToStr(Image.Picture.Bitmap.Width) + 'x' +
                                  IntToStr(Image.Picture.Bitmap.Height) + S;
    end;
end;

procedure TfrmImageManager.FormShow(Sender: TObject);
begin
  Image.Picture.Bitmap := Picture;
  StatusBar.Panels[2].Text := SubCaption;
  if SubCaption <> '' then Caption := Caption + ' - ' + SubCaption; 
  UpdateBtnEx;
end;

procedure TfrmImageManager.FormClose(Sender: TObject;
  var Action: TCloseAction);
begin
  SaveToRegistry((ModalResult = mrOk) or
                 (FChanges and (mAutoSave.Checked or
                                Confirm('Save changes?'))));
  if mRestorePath.Checked then ChDir(FStartDir);
end;

procedure TfrmImageManager.UpdateBtnEx;
begin
  BEx1.Glyph := Image.Picture.Bitmap;
  BEx2.Glyph := BEx1.Glyph;
  BEx3.Glyph := BEx1.Glyph;
  UpdateImageMeasure;
end;

procedure TfrmImageManager.SaveToRegistry(SaveTreeData: Boolean);
var Reg: TRegistry;
    I: Integer;
    SL: TStringList;

  procedure ClearValues;
  var I: Integer;
  begin
    SL := TStringList.Create;
    Reg.GetValueNames(SL);
    for I := 0 to SL.Count - 1 do Reg.DeleteValue(SL[I]);
    SL.Free;
  end;

  procedure SaveNodeList(Node: TTreeNode);
  var I: Integer;
      SL: TStringList;
  begin
    if Node.Data = nil then Exit;
    SL := TStringList(Node.Data);
    Reg.WriteInteger('Count', SL.Count);
    for I := 0 to SL.Count - 1 do
      Reg.WriteString(IntToStr(I), SL[I]);
  end;

begin
  Reg := TRegistry.Create;
  try
    if Reg.OpenKey(CRegKey + 'Directories', True) then
    begin
      if SaveTreeData then
      begin
        ClearValues;
        Reg.WriteString('', FDirRoot.Text);
        Reg.WriteBool('Expanded', FDirRoot.Expanded);
        Reg.WriteInteger('Count', FDirRoot.Count);
        for I := 0 to FDirRoot.Count - 1 do
          Reg.WriteString(IntToStr(I), FDirRoot.Item[I].Text);
      end else
      begin
        Reg.WriteBool('Expanded', FDirRoot.Expanded);
      end;
      Reg.CloseKey;
    end;
    if Reg.OpenKey(CRegKey + 'Categories', True) then
    begin
      if SaveTreeData then
      begin
        Reg.WriteString('', FCatRoot.Text);
        Reg.WriteBool('Expanded', FCatRoot.Expanded);
        SL := TStringList.Create;
        Reg.GetKeyNames(SL);
        for I := 0 to SL.Count - 1 do Reg.DeleteKey(SL[I]);
        SL.Free;
        Reg.CloseKey;
        for I := 0 to FCatRoot.Count - 1 do
          if Reg.OpenKey(CRegKey + 'Categories\' + FCatRoot.Item[I].Text, True) then
          begin
            SaveNodeList(FCatRoot.Item[I]);
            Reg.CloseKey;
          end;
      end else begin
        Reg.WriteBool('Expanded', FCatRoot.Expanded);
        Reg.CloseKey;
      end;
    end;
    if Reg.OpenKey(CRegKey + 'LastUsed', True) then
    begin
      if SaveTreeData then
      begin
        ClearValues;
        Reg.WriteString('', FLstRoot.Text);
        SaveNodeList(FLstRoot);
        Reg.CloseKey;
      end;
    end;
    if Reg.OpenKey(CRegKey + 'Options', True) then
    begin
      Reg.WriteBool('Button example', mShowButtons.Checked);
      Reg.WriteBool('Transparent', mTransparent.Checked);
      Reg.WriteBool('AutoSave', mAutoSave.Checked);
      Reg.WriteInteger('ViewStyle', Integer(FileList.ViewStyle));
      Reg.WriteInteger('PLeft.Width', PLeft.Width);
      Reg.WriteInteger('PDown.Height', PDown.Height);
      Reg.WriteBool('Thumbnails', BThumb.Down);
      Reg.WriteBool('CopyMode', BCopyMode.Down);
      Reg.WriteBool('CropToSquare', mCropToSquare.Checked);
      Reg.WriteBool('ShowStatus', mShowStatus.Checked);
      Reg.WriteString('InitialDir', SaveDlg.InitialDir);
      Reg.WriteBool('RestorePath', mRestorePath.Checked);
      Reg.WriteBool('DecreaseColors', mDecreaseColors.Checked);
      Reg.WriteBool('OpenBar', BOpenBar.Down);
      for I := 0 to FileList.Columns.Count - 1 do
        Reg.WriteInteger(Format('Columns[%d].Width', [I]), FileList.Columns[I].Width);
      Reg.WriteInteger('CompareMode', Integer(FComp));
      Reg.WriteBool('AscSort', FAscSort);
      Reg.WriteBool('HotTrack', mHotTrack.Checked);
      Reg.WriteBool('GridLines', mGridLines.Checked);
      Reg.WriteBool('ShowRoots', mShowRoots.Checked);
      Reg.WriteBool('ShowLines', mShowLines.Checked);
      Reg.WriteBool('GrayImages', mGrayImages.Checked);
      Reg.WriteBool('SaveLast', mSaveLast.Checked);
      Reg.WriteInteger('TranspColor', Integer(FTranspColor));
      Reg.CloseKey;
    end;
  finally
    Reg.Free;
  end;
end;

procedure TfrmImageManager.LoadFromRegistry;
var Reg: TRegistry;
    S: String;
    SL: TStringList;
    I, Num: Integer;
    Node: TTreeNode;
    Exp: Boolean;

  function ReadBool(Name: String; Def: Boolean): Boolean;
  begin
    try
      Result := Reg.ReadBool(Name);
    except
      Result := Def;
    end;
  end;

  function ReadStr(Name, Def: String): String;
  begin
    try
      Result := Reg.ReadString(Name);
    except
      Result := Def;
    end;
  end;

  function ReadInt(Name: String; Def: Integer): Integer;
  begin
    try
      Result := Reg.ReadInteger(Name);
    except
      Result := Def;
    end;
  end;

  procedure LoadNodeList(Node: TTreeNode);
  var I, Num: Integer;
      SL: TStringList;
      S: String;
  begin
    SL := TStringList.Create;
    Num := ReadInt('Count', 0);
    for I := 0 to Num - 1 do
    begin
      S := ReadStr(IntToStr(I), '');
      if S <> '' then SL.Add(S);
    end;
    Node.Data := SL;
  end;

begin
  Reg := TRegistry.Create;
  try
    if Reg.OpenKey(CRegKey + 'Directories', True) then
    begin
      S := ReadStr('', 'Directories');
      if S <> '' then FDirRoot.Text := S;
      Exp := ReadBool('Expanded', False);
      Num := ReadInt('Count', 0);
      for I := 0 to Num - 1 do
      begin
        S := ReadStr(IntToStr(I), '');
        if S = '' then Continue;
        Node := Tree.Items.AddChild(FDirRoot, AnsiLowerCase(S));
        SetIndexs(Node, IDX_DIR_ITEM);
      end;
      Reg.CloseKey;
      if FDirRoot.Count = 0 then
      begin
        Reg.RootKey := HKEY_LOCAL_MACHINE;
        if Reg.OpenKey(CPathKey, False) then
        begin
          S := ReadStr('RootDir', '');
          S := NormalizePath(S + '\Images\Buttons');
          if AddDirNode(S) <> nil then Exp := True;
          Reg.CloseKey;
          FWasAutoAdd := True;
        end;
        Reg.RootKey := HKEY_CURRENT_USER;
      end;
      if Exp then FDirRoot.Expand(False);
    end;
    if Reg.OpenKey(CRegKey + 'Categories', True) then
    begin
      S := ReadStr('', 'Categories');
      if S <> '' then FCatRoot.Text := S;
      Exp := ReadBool('Expanded', False);
      SL := TStringList.Create;
      Reg.GetKeyNames(SL);
      Reg.CloseKey;
      for I := 0 to SL.Count - 1 do
        if Reg.OpenKey(CRegKey + 'Categories\' + SL[I], False) then
        begin
          Node := Tree.Items.AddChild(FCatRoot, SL[I]);
          SetIndexs(Node, IDX_CAT_ITEM);
          LoadNodeList(Node);
          Reg.CloseKey;
        end;
      SL.Free;
      if Exp then FCatRoot.Expand(False);
    end{ else FCatRoot.Data := TStringList.Create};
    if Reg.OpenKey(CRegKey + 'LastUsed', True) then
    begin
      S := ReadStr('', 'Last used');
      if S <> '' then FLstRoot.Text := S;
      LoadNodeList(FLstRoot);
      Reg.CloseKey;
    end{ else FLstRoot.Data := TStringList.Create};
    if Reg.OpenKey(CRegKey + 'Options', True) then
    begin
      mShowButtons.Checked := ReadBool('Button example', True);
      mTransparent.Checked := ReadBool('Transparent', True);
      mAutoSave.Checked := ReadBool('AutoSave', True);
      //BSave.Visible := not mAutoSave.Checked;
      FileList.ViewStyle := TViewStyle(ReadInt('ViewStyle', 0));
      case FileList.ViewStyle of
        vsIcon: BIcons.Down := True;
        vsSmallIcon: BSmallIcons.Down := True;
        vsList: BList.Down := True;
        vsReport: BReport.Down := True;
      end;
      PLeft.Width := ReadInt('PLeft.Width', 260);
      PDown.Height := ReadInt('PDown.Height', 86);
      BThumb.Down := ReadBool('Thumbnails', False);
      if ReadBool('CopyMode', True) then
        BCopyMode.Down := True else BMoveMode.Down := True;
      mCropToSquare.Checked := ReadBool('CropToSquare', False);
      mShowStatus.Checked := ReadBool('ShowStatus', True);
      SaveDlg.InitialDir := ReadStr('InitialDir', '');
      StatusBar.Visible := mShowStatus.Checked;
      mRestorePath.Checked := ReadBool('RestorePath', True);
      mDecreaseColors.Checked := ReadBool('DecreaseColors', False);
      BOpenBar.Down := ReadBool('OpenBar', False);
      AdjustToolBarHeight;
      FileList.Columns[0].Width := ReadInt('Columns[0].Width', 150);
      FileList.Columns[1].Width := ReadInt('Columns[1].Width', 80);
      FileList.Columns[2].Width := ReadInt('Columns[2].Width', 100);
      FComp := TCompareMode(ReadInt('CompareMode', 0));
      FAscSort := ReadBool('AscSort', True);
      mHotTrack.Checked := ReadBool('HotTrack', False);
      FileList.HotTrack := mHotTrack.Checked;
      mGridLines.Checked := ReadBool('GridLines', False);
      FileList.GridLines := mGridLines.Checked;
      mShowRoots.Checked := ReadBool('ShowRoots', True);
      Tree.ShowRoot := mShowRoots.Checked;
      mShowLines.Checked := ReadBool('ShowLines', True);
      Tree.ShowLines := mShowLines.Checked;
      mGrayImages.Checked := ReadBool('GrayImages', False);
      mSaveLast.Checked := ReadBool('SaveLast', True);
      FTranspColor := TColor(ReadInt('TranspColor', Integer(clPurple)));
      Reg.CloseKey;
    end;
  finally
    Reg.Free;
  end;
end;

procedure TfrmImageManager.SetIndexs(Node: TTreeNode; Idx: Integer);
begin
  Node.ImageIndex := Idx;
  if Idx in [IDX_DIR_ITEM, IDX_CAT_ITEM] then Inc(Idx, IDX_SEL_INC);
  Node.SelectedIndex := Idx;
end;

procedure TfrmImageManager.CheckIcon(FileName: String);
var Inf: TSHFileInfo;
begin
  if FImagesBig.Count = 0 then
  begin
    SHGetFileInfo(PChar(FileName), 0, Inf, SizeOf(Inf), SHGFI_ICON);
    if Inf.hIcon <> 0 then
    begin
      FImagesBig.Width := GetSystemMetrics(SM_CXICON);
      FImagesBig.Height := FImagesBig.Width;
      ImageList_AddIcon(FImagesBig.Handle, Inf.hIcon);
    end;
  end;
  if FImages.Count = 0 then
  begin
    SHGetFileInfo(PChar(FileName), 0, Inf, SizeOf(Inf), SHGFI_ICON or SHGFI_SMALLICON);
    if Inf.hIcon <> 0 then
    begin
      FImages.Width := GetSystemMetrics(SM_CXSMICON);
      FImages.Height := FImages.Width;
      ImageList_AddIcon(FImages.Handle, Inf.hIcon);
    end;
  end;
end;

procedure TfrmImageManager.AddFileToList(FileName: String);
begin
  if not FileExists(FileName) then Exit;
  FileName := AnsiLowerCase(FileName);
  CheckIcon(FileName);
  with FileList.Items.Add do
  begin
    Caption := ExtractFileName(FileName);
    SubItems.Add(IntToStr(GetFileSize(FileName)));
    SubItems.Add(DateTimeToStr(FileDateTime(FileName)));
    SubItems.Add(AnsiLowerCase(FileName));
  end;
end;

procedure TfrmImageManager.TreeEdited(Sender: TObject; Node: TTreeNode;
  var S: String);
begin
  if S = '' then S := Node.Text else
  if (Node.Parent = FDirRoot) and (not DirExists(S)) then
  begin
    Error('Invalid directory name');
    S := Node.Text
  end else
  begin
    HasChange;
    Node.Text := S;
    Node.EndEdit(False);
    Tree.OnChange(Self, Node);
  end;
  BCancel.Cancel := True;
end;

procedure TfrmImageManager.TreeEditing(Sender: TObject; Node: TTreeNode;
  var AllowEdit: Boolean);
begin
  BCancel.Cancel := False;
end;

procedure TfrmImageManager.TreeKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  if (Shift <> []) or (Tree.Selected = nil) then Exit;
  case Key of
    VK_F2: Tree.Selected.EditText;
    VK_INSERT: if BAdd.Enabled then BAdd.OnClick(Self) else Exit;
    VK_DELETE: if (not Tree.IsEditing) and BDel.Enabled then BDel.OnClick(Self) else Exit;
    VK_F3: if BBrowse.Enabled then BBrowse.OnClick(Self) else Exit;
    VK_ESCAPE: if (not Tree.IsEditing) and (not BCancel.Cancel) then
               begin
                 BCancel.Cancel := True;
                 BCancel.Click;
                 Exit;
               end;
    else Exit;
  end;
  Key := 0;
end;

procedure TfrmImageManager.TreeChange(Sender: TObject; Node: TTreeNode);
begin
  FileList.Visible := False;
  FReading := True;
  FileList.Items.Clear;
  Image.Picture.Bitmap := nil;
  SelectNewByName('');
  ClearThumbs;
  FileList.ReadOnly := True; 
  if (Tree.Selected = nil) or (Tree.Selected = FLstRoot) then
  begin
    BAdd.Enabled := False;
    BDel.Enabled := False;
    BBrowse.Enabled := False;
    BDelImg.Enabled := (Tree.Selected = FLstRoot);
    if Tree.Selected = FLstRoot then FillFromNodeList(FLstRoot);
  end else
  if (Tree.Selected = FDirRoot) or (Tree.Selected.Parent = FDirRoot) then
  begin
    BAdd.Enabled := True;
    BDel.Enabled := (Tree.Selected <> FDirRoot);
    BBrowse.Enabled := (Tree.Selected <> FDirRoot);
    BDelImg.Enabled := (Tree.Selected <> FDirRoot);
    FileList.ReadOnly := False; 
    if (Tree.Selected <> FDirRoot) then
       FillListForDir(Tree.Selected.Text);
  end else
  if (Tree.Selected = FCatRoot) or (Tree.Selected.Parent = FCatRoot) then
  begin
    BAdd.Enabled := True;
    BDel.Enabled := (Tree.Selected <> FCatRoot);
    BBrowse.Enabled := False;
    BDelImg.Enabled := (Tree.Selected <> FCatRoot);
    if (Tree.Selected <> FCatRoot) then
      FillFromNodeList(Tree.Selected);
  end;
  if BThumb.Down then CreateNewThumbs;
  FReading := False;
  BCopyAgain.Enabled := (FLastTarget <> nil) and (FLastTarget <> Tree.Selected);
end;

function TfrmImageManager.BrowseDir(S: String): String;
var S2: String;
begin
  Result := '';
  if S = '' then S2 := 'Select new directory...' else
                 S2 := 'Current: ' + AnsiLowerCase(S) + #13 + 'Select other directory...';
  if BrowseDirectory(S, S2, 0) then
    Result := S;
end;

function TfrmImageManager.AddDirNode(Folder: String): TTreeNode;
var I: Integer;
begin
  Folder := NormalizePath(Folder);
  Result := nil;
  if not DirExists(Folder) then Exit;
  for I := 0 to FDirRoot.Count - 1 do
    if NormalizePath(FDirRoot.Item[I].Text) = Folder then Exit;
  Result := Tree.Items.AddChild(FDirRoot, AnsiLowerCase(Folder));
  SetIndexs(Result, IDX_DIR_ITEM);
end;

procedure TfrmImageManager.BAddClick(Sender: TObject);
var Node: TTreeNode;
    S: String;
begin
  if Tree.Selected = nil then Exit else
  if (Tree.Selected = FDirRoot) or (Tree.Selected.Parent = FDirRoot) then
  begin
    S := BrowseDir('');
    if S <> '' then
    begin
      Node := AddDirNode(S);
      if Node <> nil then Tree.Selected := Node;
      HasChange;
    end;
  end else
  if (Tree.Selected = FCatRoot) or (Tree.Selected.Parent = FCatRoot) then
  begin
    Node := Tree.Items.AddChild(FCatRoot, 'New Category');
    SetIndexs(Node, IDX_CAT_ITEM);
    Node.Data := TStringList.Create;
    Tree.Selected := Node;
    Node.EditText;
    HasChange;
  end;
end;

procedure TfrmImageManager.BDelClick(Sender: TObject);
begin
  if Tree.Selected = nil then Exit else
  if (Tree.Selected.Parent = FDirRoot) then
  begin
    if Confirm('Delete link to directory?') then
    begin
      Tree.Items.Delete(Tree.Selected);
      HasChange;
    end;
  end else
  if (Tree.Selected.Parent = FCatRoot) then
  begin
    if Confirm('Delete category?') then
    begin
      Tree.Items.Delete(Tree.Selected);
      HasChange;
    end;
  end;
end;

procedure TfrmImageManager.HasChange;
begin
  if FStarting then Exit; 
  BSave.Enabled := True;
  FChanges := True;
end;

procedure TfrmImageManager.BSaveClick(Sender: TObject);
begin
  SaveToRegistry(True);
  BSave.Enabled := False;
  FChanges := False;
end;

procedure TfrmImageManager.BIconsClick(Sender: TObject);
begin
  FileList.ViewStyle := TViewStyle(TSpeedButton(Sender).Tag - 1);
end;

procedure TfrmImageManager.DecreaseColors;
begin
  if Image.Picture.Bitmap.PixelFormat <> pf4bit then
    Image.Picture.Bitmap.PixelFormat := pf4bit;
end;

procedure TfrmImageManager.SelectNewByName(FileName: String);
begin
  FSelectedImageName := AnsiLowerCase(FileName);
  if FileName = '' then
  begin
    Image.Picture.Bitmap := nil;
    StatusBar.Panels[0].Text := '';
    StatusBar.Panels[2].Text := '';
  end else
  begin
    try
      Image.Picture.Bitmap.LoadFromFile(FileName);
      if mCropToSquare.Checked then CropCurrentImage;
      if mDecreaseColors.Checked then DecreaseColors;
      if mGrayImages.Checked then GrayCurrentImage;
      StatusBar.Panels[0].Text := IntToStr(GetFileSize(FileName)) + ' bytes(s)';
      StatusBar.Panels[2].Text := FileName;
    except
      Image.Picture.Bitmap := nil;
      StatusBar.Panels[0].Text := 'Invalid';
      StatusBar.Panels[2].Text := FileName;
    end;
  end;
  UpdateBtnEx;
end;

procedure TfrmImageManager.FileListChange(Sender: TObject; Item: TListItem;
  Change: TItemChange);
begin
  if FReading then Exit;
  if (Item = nil) or (Item.SubItems.Count < 3) then
    SelectNewByName('') else
      SelectNewByName(Item.SubItems[2]);
end;

procedure TfrmImageManager.FileListDblClick(Sender: TObject);
begin
  if FileList.Selected <> nil then ModalResult := mrOk;
end;

procedure TfrmImageManager.FileListCompare(Sender: TObject; Item1,
  Item2: TListItem; Data: Integer; var Compare: Integer);
var Sz1, Sz2: Integer;
    Dt1, Dt2: TDateTime;
    S1, S2: String;

 function SortByName: Integer;
 begin
   S1 := AnsiUpperCase(Item1.Caption);
   S2 := AnsiUpperCase(Item2.Caption);
   if S1 > S2 then Result := 1 else
   if S1 < S2 then Result := -1 else Result := 0;
 end;

begin
  case FComp of
    cmName: Compare := SortByName;
    cmSize:
      begin
        if (Item1.SubItems.Count >= 1) or
           (Item2.SubItems.Count >= 1) then
        begin
          Sz1 := StrToInt(Item1.SubItems[0]);
          Sz2 := StrToInt(Item2.SubItems[0]);
          if Sz1 > Sz2 then Compare := 1 else
          if Sz1 < Sz2 then Compare := -1 else Compare := SortByName;
        end;
      end;
    cmDate:
      begin
        if (Item1.SubItems.Count >= 2) or
           (Item2.SubItems.Count >= 2) then
        begin
          Dt1 := StrToDateTime(Item1.SubItems[1]);
          Dt2 := StrToDateTime(Item2.SubItems[1]);
          if Dt1 > Dt2 then Compare := 1 else
          if Dt1 < Dt2 then Compare := -1 else Compare := SortByName;
        end;
      end;
    else Exit;
  end;
  if not FAscSort then Compare := -Compare;
end;

procedure TfrmImageManager.FileListColumnClick(Sender: TObject;
  Column: TListColumn);
var NewSort: TCompareMode;
begin
  case Column.Index of
    0: NewSort := cmName;
    1: NewSort := cmSize;
    2: NewSort := cmDate;
    else Exit;
  end;
  if NewSort = FComp then FAscSort := not FAscSort else
  begin
    FComp := NewSort;
    FAscSort := True;
  end;
  FileList.AlphaSort;
end;

procedure TfrmImageManager.FileListEditing(Sender: TObject;
  Item: TListItem; var AllowEdit: Boolean);
begin
  BCancel.Cancel := False;
end;

procedure TfrmImageManager.FixLinks(OldPath, NewPath: String);

  procedure FixLinkForNode(Node: TTreeNode);
  var SL: TStringList;
      I: Integer;
  begin
    if Node.Data = nil then Exit;
    SL := Node.Data;
    for I := 0 to SL.Count - 1 do
      if NormalizePath(SL[I]) = OldPath then
        SL[I] := NewPath;
  end;

var I: Integer;
begin
  OldPath := NormalizePath(OldPath);
  NewPath := NormalizePath(NewPath);
  FixLinkForNode(FLstRoot);
  for I := 0 to FCatRoot.Count - 1 do
    FixLinkForNode(FCatRoot.Item[I]);
end;

procedure TfrmImageManager.FileListEdited(Sender: TObject; Item: TListItem;
  var S: String);
var OldName, NewName: String;
    Idx: Integer;
begin
  if S = '' then S := Item.Caption else
  if Pos('.BMP', AnsiUpperCase(S)) = 0 then
  begin
    Warning('File name should have a "bmp" extension');
    S := Item.Caption;
  end else
  begin
    OldName := Item.SubItems[2];
    NewName := NormalizePath(ExtractFilePath(OldName) + '\' + S);
    if not RenameFile(OldName, NewName) then
    begin
      Error('Can''t rename file');
      S := Item.Caption;
    end else
    begin
      Item.SubItems[2] := NewName;
      if (Tree.Selected <> nil) and (Tree.Selected.Data <> nil) then
      begin
        Idx := TStringList(Tree.Selected.Data).IndexOf(OldName);
        if Idx <> -1 then TStringList(Tree.Selected.Data)[Idx] := NewName;
      end;
      HasChange;
      FixLinks(OldName, NewName);
    end;
  end;
  BCancel.Cancel := True;
end;

procedure TfrmImageManager.BBrowseClick(Sender: TObject);
var S: String;
begin
  if Tree.Selected = nil then Exit else
  if (Tree.Selected.Parent = FDirRoot) then
  begin
    S := BrowseDir(Tree.Selected.Text);
    if DirExists(S) then
    begin
      Tree.Selected.Text := AnsiLowerCase(S);
      HasChange;
      TreeChange(Self, Tree.Selected);
    end;
  end;
end;

procedure TfrmImageManager.StartVisual(pText: String; pMax: Integer);
begin
  LProgressInfo.Caption := pText;
  ProgressBar.Max := pMax;
  ProgressBar.Position := 0;
  if pMax < 10 then Exit;
  LProgressInfo.Visible := True;
  ProgressBar.Visible := True;
  LProgressInfo.Refresh;
  ProgressBar.Refresh;
end;

procedure TfrmImageManager.NextVisual(pValue: Integer);
begin
  ProgressBar.Position := pValue;
  ProgressBar.Refresh;
end;

procedure TfrmImageManager.DoneVisual;
begin
  LProgressInfo.Visible := False;
  ProgressBar.Visible := False;
end;

procedure TfrmImageManager.CommonFillEnd;
begin
  FileList.AlphaSort;
  FileList.Visible := True;
  StatusBar.Panels[0].Text := IntToStr(FileList.Items.Count) + ' bitmap(s)';
  if FileList.Items.Count > 0 then FileList.Selected := FileList.Items[0];
end;

procedure TfrmImageManager.FillListForDir(S: String);
var SR: TSearchRec;
    Found: Integer;
begin
  StatusBar.Panels[0].Text := 'Reading...';
  Found := FindFirst(NormalizePath(S + '\*.bmp'), faAnyFile, SR);
  while Found = 0 do
  begin
    AddFileToList(NormalizePath(S + '\' + SR.Name));
    Found := FindNext(SR);
  end;
  FindClose(SR);
  CommonFillEnd;
  StatusBar.Panels[2].Text := NormalizePath(S);
end;

procedure TfrmImageManager.FillFromNodeList(Node: TTreeNode);
var I: Integer;
    SL: TStringList;
begin
  if Node.Data = nil then Exit;
  SL := TStringList(Node.Data);
  for I := 0 to SL.Count - 1 do AddFileToList(SL[I]);
  CommonFillEnd;
end;

procedure TfrmImageManager.FileListKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
var I: Integer;
begin
  if (Shift = []) then
  begin
    case Key of
      VK_F2: if FileList.Selected <> nil then FileList.Selected.EditCaption else Exit;
      VK_DELETE: if (not FileList.IsEditing) and (FileList.Selected <> nil) then BDelImg.OnClick(Self) else Exit;
      VK_ESCAPE: if (not FileList.IsEditing) and (not BCancel.Cancel) then
                 begin
                   BCancel.Cancel := True;
                   BCancel.Click;
                   Exit;
                 end;
      else Exit;
    end;
    Key := 0;
  end else
  if (Key = Byte('A')) and (Shift = [ssCtrl]) then
  begin
    for I := 0 to FileList.Items.Count - 1 do
      FileList.Items[I].Selected := True;
    Key := 0;
  end;
end;

function TfrmImageManager.AddNameToNode(Node: TTreeNode; FileName: String): Boolean;
var I: Integer;
    SL: TStringList;
begin
  Result := False;
  if Node.Data = nil then Exit;
  FileName := AnsiLowerCase(FileName);
  SL := TStringList(Node.Data);
  for I := 0 to SL.Count - 1 do
    if AnsiLowerCase(SL[I]) = FileName then Exit; 
  SL.Add(FileName);
  Result := True;
end;

function TfrmImageManager.DeleteNameFromNode(Node: TTreeNode; FileName: String): Boolean;
var I: Integer;
    SL: TStringList;
begin
  Result := False;
  if Node.Data = nil then Exit;
  FileName := AnsiLowerCase(FileName);
  SL := TStringList(Node.Data);
  for I := 0 to SL.Count - 1 do
    if AnsiLowerCase(SL[I]) = FileName then
    begin
      SL.Delete(I);
      Result := True;
      Break;
    end;
end;

procedure TfrmImageManager.FormCloseQuery(Sender: TObject;
  var CanClose: Boolean);
begin
  CanClose := True;
  if ModalResult <> mrOk then Exit;
  Picture := Image.Picture.Bitmap;
  if (Image.Picture.Bitmap <> nil) and (not Image.Picture.Bitmap.Empty) and
     (mSaveLast.Checked) and (FSelectedImageName <> '') then
     AddNameToNode(FLstRoot, FSelectedImageName);
  HasChange;
  CanClose := True;
end;

procedure TfrmImageManager.BClearClick(Sender: TObject);
begin
  Image.Picture.Bitmap := nil;
  ModalResult := mrOk;
end;

procedure TfrmImageManager.TreeDeletion(Sender: TObject; Node: TTreeNode);
begin
  if Node.Data <> nil then
  begin
    TStringList(Node.Data).Free;
    Node.Data := nil;
    if Node = FLastTarget then SetLastTarget(nil);
  end;
end;

procedure TfrmImageManager.TreeDragOver(Sender, Source: TObject; X,
  Y: Integer; State: TDragState; var Accept: Boolean);
var Node: TTreeNode;
begin
  Node := Tree.GetNodeAt(X, Y);
  Accept := False;
  if (Node = nil) or (Node = Tree.Selected) or (Node.Level = 0) then Exit;
  Accept := (Tree.Selected.Parent = Node.Parent) or
            (Node.Parent = FCatRoot);
end;

procedure TfrmImageManager.CopyOrMoveToNode(TargetNode: TTreeNode);
var Num, Idx: Integer;
    Item: TListItem;
    Succ: Boolean;
    NewName: String;
begin
  Num := FileList.SelCount;
  Idx := FileList.Selected.Index;
  InitInfo;
  while (Idx < FileList.Items.Count) and (Num > 0) do
  begin
    Item := FileList.Items[Idx];
    Succ := False;
    if Item.Selected then
    begin
      if (Tree.Selected.Parent = FDirRoot) and (TargetNode.Parent = FDirRoot) then
      begin
        NewName := NormalizePath(TargetNode.Text + '\' + Item.Caption);
        if not Windows.CopyFile(PChar(Item.SubItems[2]), PChar(NewName), True) then
          Info_AddError('Can''t copy file "' + Item.Caption + '". Another file with same name exists.') else
          if BMoveMode.Down then
          begin
            Succ := DeleteFile(Item.SubItems[2]);
            if not Succ then Info_AddError('Can''t delete file "' + Item.Caption + '". Moving operation incomplete.');
          end;
      end else
      if (TargetNode.Parent = FCatRoot) then
      begin
        if not AddNameToNode(TargetNode, Item.SubItems[2]) then
          Info_AddError('Can''t copy link for "' + Item.Caption + '". Such link already exists.') else
          if BMoveMode.Down and (Tree.Selected.Parent = FCatRoot) then
          begin
            Succ := DeleteNameFromNode(Tree.Selected, Item.SubItems[2]);
            if not Succ then Info_AddError('Can''t delete link for "' + Item.Caption + '". Moving operation incomplete.');
          end;
      end;
      Dec(Num);
    end;
    if Succ then FileList.Items.Delete(Item.Index) else Inc(Idx);
  end;
  if BMoveMode.Down then FileList.Arrange(arDefault);
  ShowInfo;
  StatusBar.Panels[2].Text := 'Copy/Move operation complete';
  HasChange;
end;

procedure TfrmImageManager.TreeDragDrop(Sender, Source: TObject; X,
  Y: Integer);
begin
  if (FileList.Selected = nil) or (Tree.DropTarget = Nil) or
     (Tree.DropTarget.Level <> 1) then Exit;
  CopyOrMoveToNode(Tree.DropTarget);
  SetLastTarget(Tree.DropTarget);
end;

procedure TfrmImageManager.ClearThumbs;
var I: Integer;
begin
  for I := 0 to FileList.Items.Count - 1 do
    FileList.Items[I].ImageIndex := 0;
  for I := FImages.Count - 1 downto 1 do FImages.Delete(I);
  for I := FImagesBig.Count - 1 downto 1 do FImagesBig.Delete(I);
end;

procedure TfrmImageManager.CreateNewThumbs;
var I, Sz, N: Integer;
    Bm, Bm2: TBitmap;
    Col: TColor;
const Kb = 1024;

procedure DrawEx(Src, Tgt: TBitmap; var Col: TColor);
var X, Y, W, H: Integer;
begin
  with Tgt.Canvas do
  begin
    Col := Src.Canvas.Pixels[0, Src.Height - 1];
    Brush.Color := Col;
    FillRect(Rect(0, 0, Tgt.Width, Tgt.Height));
    if (Tgt.Width >= Src.Width) and (Tgt.Height >= Src.Height) then
    begin
      X := (Tgt.Width - Src.Width) div 2;
      Y := (Tgt.Height - Src.Height) div 2;
      Draw(X, Y, Src);
    end else
    begin
      if Tgt.Width > Src.Width then W := Src.Width else W := Tgt.Width;
      if Tgt.Height > Src.Height then H := Src.Height else H := Tgt.Height;
      X := (Tgt.Width - W) div 2;
      Y := (Tgt.Height - H) div 2;
      StretchDraw(Rect(X, Y, X + W, Y + H), Src);
    end;
  end;
end;

begin
  StartVisual('Create thumbnails...', FileList.Items.Count);
  Bm := TBitmap.Create;
  Bm2 := TBitmap.Create;
  N := 1;
  for I := 0 to FileList.Items.Count - 1 do
  begin
    try
      Sz := GetFileSize(FileList.Items[I].SubItems[2]);
      if Sz > 4*Kb then Abort;
      Bm.LoadFromFile(FileList.Items[I].SubItems[2]);
      if (Bm.Width mod Bm.Height) = 0 then Bm.Width := Bm.Height;
      if mGrayImages.Checked then GrayImage(Bm);
      Bm2.Assign(Bm);
      Bm2.Width := FImagesBig.Width;
      Bm2.Height := FImagesBig.Height;
      DrawEx(Bm, Bm2, Col);
      FImagesBig.AddMasked(Bm2, Col);
      FileList.Items[I].ImageIndex := N;
      Bm2.Width := FImages.Width;
      Bm2.Height := FImages.Height;
      DrawEx(Bm, Bm2, Col);
      FImages.AddMasked(Bm2, Col);
      Inc(N);
    except
    end;
    if I mod 10 = 0 then NextVisual(I);
  end;
  Bm2.Free;
  Bm.Free;
  DoneVisual;
end;

procedure TfrmImageManager.BThumbClick(Sender: TObject);
begin
  FileList.Selected := nil;
  Tree.Refresh;
  if BThumb.Down then CreateNewThumbs else ClearThumbs;
end;

procedure TfrmImageManager.BViewClick(Sender: TObject);
begin
  if FileList.Selected <> nil then
    ShellExecute(Handle, nil, PChar(FileList.Selected.SubItems[2]), nil, nil, SW_SHOW);
end;

procedure TfrmImageManager.BDelImgClick(Sender: TObject);
var Num, Idx: Integer;
    Item: TListItem;
begin
  if (Tree.Selected = nil) or (FileList.Selected = nil) then Exit else
  if (Tree.Selected.Parent = FDirRoot) then
  begin
    if Confirm('Delete files(s)?') then
    begin
      Num := FileList.SelCount;
      Idx := FileList.Selected.Index;
      InitInfo;
      while (Idx < FileList.Items.Count) and (Num > 0) do
      begin
        Item := FileList.Items[Idx];
        if Item.Selected then
        begin
          if DeleteFile(Item.SubItems[2]) then
             FileList.Items.Delete(Idx) else
             begin
               Info_AddError('Can''t delete file "' + Item.Caption + '"');
               Inc(Idx);
             end;
          Dec(Num);
        end else
          Inc(Idx);
      end;
      ShowInfo;
      HasChange;
      FileList.Arrange(arDefault);
    end;
  end else
  if (Tree.Selected.Parent = FCatRoot) or (Tree.Selected = FLstRoot) then
  begin
    if Confirm('Delete link(s)?') then
    begin
      Num := FileList.SelCount;
      Idx := FileList.Selected.Index;
      InitInfo;
      while (Idx < FileList.Items.Count) and (Num > 0) do
      begin
        Item := FileList.Items[Idx];
        if Item.Selected then
        begin
          if DeleteNameFromNode(Tree.Selected, Item.SubItems[2]) then
            FileList.Items.Delete(Idx) else
            begin
              Info_AddError('Can''t delete link for "' + Item.Caption + '"');
              Inc(Idx);
            end;
          Dec(Num);
        end else
          Inc(Idx);
      end;
      ShowInfo;
      HasChange;
      FileList.Arrange(arDefault);
    end;
  end;
end;

procedure TfrmImageManager.mShowButtonsClick(Sender: TObject);
begin
  mShowButtons.Checked := not mShowButtons.Checked;
  PBtnEx.Visible := mShowButtons.Checked;
  PBtnEx.Left := PButtons.Left - PBtnEx.Width - 4;
end;

procedure TfrmImageManager.mTransparentClick(Sender: TObject);
begin
  mTransparent.Checked := not mTransparent.Checked;
  Image.Transparent := mTransparent.Checked;
end;

procedure TfrmImageManager.mAutoSaveClick(Sender: TObject);
begin
  mAutoSave.Checked := not mAutoSave.Checked;
end;

procedure TfrmImageManager.PImageResize(Sender: TObject);
begin
  if mTransparent.Checked then
    Image.Picture.Bitmap := Image.Picture.Bitmap;
end;

procedure TfrmImageManager.PUpResize(Sender: TObject);
begin
  FileList.Arrange(arDefault);
end;

procedure TfrmImageManager.mAboutClick(Sender: TObject);
begin
  ShowOtAbout('Image Manager');
end;

procedure TfrmImageManager.CropCurrentImage;
begin
  with Image.Picture.Bitmap do
    if (not Empty) and (Width <> Height) then
    begin
      if Width > Height then Width := Height else
        Height := Width;
      UpdateBtnEx;
    end;
end;

procedure TfrmImageManager.mCropToSquareClick(Sender: TObject);
begin
  mCropToSquare.Checked := not mCropToSquare.Checked;
  if mCropToSquare.Checked then CropCurrentImage else
    if FileList.Selected <> nil then FileList.OnChange(Self, FileList.Selected, ctState);
  UpdateBtnEx;
end;

procedure TfrmImageManager.mShowStatusClick(Sender: TObject);
begin
  mShowStatus.Checked := not mShowStatus.Checked;
  StatusBar.Visible := mShowStatus.Checked;
end;

procedure TfrmImageManager.mScanClick(Sender: TObject);
var SL: TStringList;
    I: Integer;
begin
  SL := ScanForDirs;
  if SL <> nil then
  begin
    for I := 0 to SL.Count - 1 do AddDirNode(SL[I]);
    SL.Free;
    HasChange;
  end;
end;

procedure TfrmImageManager.TreeDblClick(Sender: TObject);
begin
  if (Tree.Selected <> nil) and (Tree.Selected.Parent = FDirRoot) then
    ShellExecute(Handle, nil, PChar(Tree.Selected.Text), nil, nil, SW_SHOW);
end;

procedure TfrmImageManager.BCopyClick(Sender: TObject);
var
  MyFormat: Word;
  AData,APalette: THandle;
begin
  if not Image.Picture.Bitmap.Empty then
  begin
    Image.Picture.Bitmap.SaveToClipboardFormat(MyFormat, AData, APalette);
    ClipBoard.SetAsHandle(MyFormat, AData);
  end;
end;

procedure TfrmImageManager.BPasteClick(Sender: TObject);
begin
  Image.Picture.Bitmap.LoadFromClipboardFormat(cf_BitMap, ClipBoard.GetAsHandle(cf_Bitmap), 0);
  FSelectedImageName := '';
  UpdateBtnEx;
  StatusBar.Panels[0].Text := '';
  StatusBar.Panels[2].Text := 'From clipboard';
end;

procedure TfrmImageManager.TreeExit(Sender: TObject);
begin
  if not BCancel.Cancel then BCancel.Cancel := True;
end;

procedure TfrmImageManager.FileListExit(Sender: TObject);
begin
  if not BCancel.Cancel then BCancel.Cancel := True;
end;

procedure TfrmImageManager.BSaveImgClick(Sender: TObject);
begin
  if Image.Picture.Bitmap.Empty then Exit;
  if SaveDlg.Execute then
  begin
    Image.Picture.Bitmap.SaveToFile(SaveDlg.FileName);
    SaveDlg.InitialDir := ExtractFilePath(SaveDlg.FileName);
  end;
end;

procedure TfrmImageManager.mRestorePathClick(Sender: TObject);
begin
  mRestorePath.Checked := not mRestorePath.Checked;
end;

procedure TfrmImageManager.mDecreaseColorsClick(Sender: TObject);
begin
  mDecreaseColors.Checked := not mDecreaseColors.Checked;
  if mDecreaseColors.Checked then DecreaseColors else
    if FileList.Selected <> nil then FileList.OnChange(Self, FileList.Selected, ctState);
  UpdateBtnEx;
end;

procedure TfrmImageManager.AdjustToolBarHeight;
const CHeights: array[false..true] of Integer = (32, 64);
begin
  PListBtn.Height := CHeights[BOpenBar.Down];
  BvToolBarSep.Visible := BOpenBar.Down;
end;

procedure TfrmImageManager.BOpenBarClick(Sender: TObject);
begin
  AdjustToolBarHeight;
end;

procedure TfrmImageManager.SetLastTarget(Node: TTreeNode);
begin
  FLastTarget := Node;
  if Node = nil then
  begin
    BCopyAgain.Enabled := False;
    BCopyAgain.Hint := 'Copy/Move to last target';
  end else begin
    BCopyAgain.Enabled := True;
    BCopyAgain.Hint := 'Copy/Move to "' + Node.Text + '"';
  end;
end;

procedure TfrmImageManager.BCopyAgainClick(Sender: TObject);
begin
  if FLastTarget <> nil then CopyOrMoveToNode(FLastTarget);
end;

procedure TfrmImageManager.mHotTrackClick(Sender: TObject);
begin
  mHotTrack.Checked := not mHotTrack.Checked;
  FileList.HotTrack := mHotTrack.Checked;
end;

procedure TfrmImageManager.mGridLinesClick(Sender: TObject);
begin
  mGridLines.Checked := not mGridLines.Checked;
  FileList.GridLines := mGridLines.Checked;
end;

procedure TfrmImageManager.mShowRootsClick(Sender: TObject);
begin
  mShowRoots.Checked := not mShowRoots.Checked;
  Tree.ShowRoot := mShowRoots.Checked;
end;

procedure TfrmImageManager.mShowLinesClick(Sender: TObject);
begin
  mShowLines.Checked := not mShowLines.Checked;
  Tree.ShowLines := mShowLines.Checked;
end;

procedure TfrmImageManager.mGrayImagesClick(Sender: TObject);
begin
  mGrayImages.Checked := not mGrayImages.Checked;
  if mGrayImages.Checked then GrayCurrentImage else
    if FileList.Selected <> nil then FileList.OnChange(Self, FileList.Selected, ctState);
  UpdateCurrentThumbnails;
  UpdateBtnEx;
end;

procedure TfrmImageManager.GrayCurrentImage;
begin
  GrayImage(Image.Picture.Bitmap);
end;

procedure TfrmImageManager.GrayImage(Bitmap: TBitmap);
type
  TRGB = record
    R, G, B, F: Byte;
  end;
var I, J, Sum: Integer;
    C, MaskC: TColor;
begin
  if Bitmap.Empty then Exit;
  with Bitmap.Canvas do
  begin
    MaskC := Pixels[0, Bitmap.Height - 1];
    for I := 0 to Bitmap.Height - 1 do
      for J := 0 to Bitmap.Width - 1 do
      begin
        C := Pixels[J, I];
        if C = MaskC then Pixels[J, I] := clPurple else
        begin
          with TRGB(C) do Sum := (R + G + B) div 3;
          Pixels[J, I] := RGB(Sum, Sum, Sum);
        end;
      end;
  end;
end;

procedure TfrmImageManager.UpdateCurrentThumbnails;
begin
  if (Tree.Selected <> nil) and BThumb.Down then
  begin
    ClearThumbs;
    CreateNewThumbs;
  end;
end;

procedure TfrmImageManager.mFromIconClick(Sender: TObject);
begin
  frmFromIcon := TfrmFromIcon.Create(nil);
  try
    frmFromIcon.BackCol.ColorValue := FTranspColor;
    if frmFromIcon.ShowModal = mrOk then
    begin
      Image.Picture.Bitmap.Assign(frmFromIcon.PrevImage.Picture.Bitmap);
      StatusBar.Panels[0].Text := '';
      StatusBar.Panels[2].Text := 'From Icon';
      FSelectedImageName := '';
      UpdateBtnEx;
    end;
    FTranspColor := frmFromIcon.BackCol.ColorValue;
  finally
    frmFromIcon.Free;
  end;
end;

procedure TfrmImageManager.mSaveLastClick(Sender: TObject);
begin
  mSaveLast.Checked := not mSaveLast.Checked;
end;

end.
