{===============================================================}
{    Oxygen Plan Library, v0.91  1999-2000 Oxygen Software     }
{---------------------------------------------------------------}
{        Written by Oleg Fyodorov, ofyodorov@mtu-net.ru         }
{                   oxygensoft@mtu-net.ru                       }
{                  http://oxygen.webzone.ru                     }
{===============================================================}

unit O2Plan;

{$Include O2Plan.inc}
{$OPTIMIZATION OFF}

interface

  uses
    Windows, Classes, Controls, Graphics, StrF, ExtCtrls, SysUtils, MGraph,
{$IFDEF USE_RX}
       TimerLst, RxGif, VCLUtils,
{$ENDIF}
       jpeg;

  resourcestring
    // Plan commands
    cmScale             = 'Scale ';
    cmSetPenColor       = 'SetPenColor ';
    cmSetBrushColor     = 'SetBrushColor ';
    cmSetLineWidth      = 'SetLineWidth ';
    cmSetFont           = 'SetFont ';
    cmStandardPicture   = 'StandardPicture ';
    cmStandardAniGif    = 'StandardAniGif ';
    cmBackgroundPicture = 'BackgroundPicture ';
    cmLayer             = 'NewLayer ';

    // Plan objects
    ocLine              = 'Line';
    ocBox               = 'Box';
    ocCircle            = 'Circle';
    ocText              = 'Text';
    ocPicture           = 'Picture';
    ocPolygon           = 'Polygon';
    ocCompositeObject   = 'Composite';
    ocAniGif            = 'AniGif';
    ocAniPicture        = 'AniPicture';

    sStartObject = 'Object ';
    sEndObject   = 'End';

    sDefaultLayer = 'Default';

    // Plan colors
    pcBlack   = 'Black';
    pcMaroon  = 'Maroon';
    pcGreen   = 'Green';
    pcOlive   = 'Olive';
    pcNavy    = 'Navy';
    pcPurple  = 'Purple';
    pcTeal    = 'Teal';
    pcGray    = 'Gray';
    pcSilver  = 'Silver';
    pcRed     = 'Red';
    pcLime    = 'Lime';
    pcYellow  = 'Yellow';
    pcBlue    = 'Blue';
    pcFuchsia = 'Fuchsia';
    pcAqua    = 'Aqua';
    pcWhite   = 'White';

    sFileNotFound = 'File %s not found';

  type

  TOxygenPlan = class;
  TPlanObject = class;
  TPlanObjectClass = class of TPlanObject;

  EOxygenPlanException = class(Exception);

  TMarginType = (mtNone, mtLeft, mtTopLeft, mtTop, mtTopRight, mtRight, mtBottomRight, mtBottom, mtBottomLeft, mtMovePoint);

  TCallHandler = (chMouseDown, chMouseMove, chMouseUp);

{$IFDEF USE_RX}
  TBlinkType = (btNone, btPen, btBrush, btPicture, btFont, btText);
{$ENDIF}

  PPointArray = ^TPointArray;
  TPointArray = array[0..10000] of TPoint;

  TPlanLayer = class
    private
      FOwner      : TOxygenPlan;
      FObjectList : TList;
      FVisible    : Boolean;
      HRegion     : HRGN;
      FReadOnly   : Boolean;
      function  GetObject(Index:Integer):TPlanObject;
      function  GetObjectsCount : Integer;
      procedure SetVisible(Value:Boolean);
      procedure SetReadOnly(const Value : Boolean);
    public
      Name    : String;
      Index   : Integer;
      property Objects[Index : Integer] : TPlanObject read GetObject; default;
      property Count:Integer read GetObjectsCount;
      property ReadOnly : Boolean read FReadOnly write SetReadOnly;
      property Visible : Boolean read FVisible write SetVisible;
      constructor Create(AOwner : TOxygenPlan);
      procedure Free;
      function  Add(po:TPlanObject):Integer;
      function  IndexOf(po:TPlanObject):Integer;
      procedure Delete(Index:Integer);
      procedure Clear;
      procedure Draw(ACanvas:TCanvas);
      function  ObjectFromCoord(X, Y : Integer; var MarginType : TMarginType):TPlanObject;
  end;

  TPlanLayers = class
    private
      FOwner : TOxygenPlan;
      FLayersList:TList;
      function NewLayer(const LName:String):TPlanLayer;
      function Get(Index:Integer):TPlanLayer;
      procedure Put(Index:Integer; Layer:TPlanLayer);
      function GetCount:Integer;
    public
      property Layers[Index:Integer]:TPlanLayer read Get write Put; default;
      property Count:Integer read GetCount;
      constructor Create(AOwner : TOxygenPlan);
      procedure Free;
      function  Add(const LayerName:String):Integer;
      procedure Delete(Index:Integer);
      function  IndexOf(const LayerName:String):Integer;
      procedure Clear;
      function  LayerByName(const LayerName:String):TPlanLayer;
      procedure AddLayersDescr(sl:TStrings);
  end;

  TPlanObject = class(TObject)
    private
      FOwner       : TOxygenPlan;
      FParent      : TPlanObject;
      FChildren    : TList;
      FLayer       : TPlanLayer;
      FID          : Integer;
      FName        : String;
      FSelected    : Boolean;
      FPen         : TPen;
      FBrush       : TBrush;
      FSelectColor : TColor;
      FVisible     : Boolean;
      FCanResize   : Boolean;
      FClippingRgn : HRGN;
      FDestroying  : Boolean;
      FReadOnly    : Boolean;
      FProperties  : TStrings;
      FCursor      : TCursor;

{$IFDEF USE_RX}
      // blinking
      FBlinkTimerHandle       : Integer;
      FBlinkType              : TBlinkType;
      FBlinkPen, FSavePen     : TPen;
      FBlinkBrush, FSaveBrush : TBrush;
      FBlinkIndex             : Integer; // 0 - Save, 1 - Blink
{$ENDIF}

      function  GetChildrenCount : Integer;
      procedure SetPen(Value : TPen); virtual;
      procedure SetBrush(Value : TBrush); virtual;
      procedure SetLayer(Value : TPlanLayer); virtual;
      procedure SetName(Value : String);
      function  GetObjectsCount : Integer;
      function  GetObject(Index : Integer) : TPlanObject;
      procedure SetParent(Value : TPlanObject);
      procedure SetSelected(Value : Boolean); virtual;
      function  GetSelectedByColor : Boolean;
      procedure SetVisible(const Value : Boolean);
{$IFDEF USE_RX}
      function  GetBlinking : Boolean;
{$ENDIF}
   protected
      function  GetObjectTypeDescr : String; virtual;
      procedure FillObjectProperties; virtual;
      procedure SaveCoords; virtual;
      function  GetDisplayRect:TRect; virtual; abstract;
      procedure DrawChildren(ACanvas:TCanvas); virtual;
      function  AddPoint(const CoordX,CoordY : Integer; CallHandler : TCallHandler {$IFNDEF VER100} = chMouseMove {$ENDIF}) : Boolean; virtual; abstract; // returns False when drawing is finished
{$IFDEF USE_RX}
      procedure BlinkTimer(Sender: TObject); virtual;  // blinking handler
{$ENDIF}
      function  GetClippingRgn : HRGN; virtual;
      procedure SetReadOnly(const Value : Boolean);
      procedure SetProperties(Value : TStrings);
    public
      property Properties      : TStrings read FProperties write SetProperties;
{$IFDEF USE_RX}
      property Blinking        : Boolean read GetBlinking;
      property BlinkType       : TBlinkType read FBlinkType write FBlinkType;
      property BlinkPen        : TPen read FBlinkPen;
      property BlinkBrush      : TBrush read FBlinkBrush;
{$ENDIF}
      property Brush           : TBrush read FBrush write SetBrush;
      property CanResize       : Boolean read FCanResize write FCanResize;
      property ChildrenCount   : Integer read GetChildrenCount;
      property Cursor          : TCursor read FCursor write FCursor;
      property DisplayRect     : TRect read GetDisplayRect;
      property ID              : Integer read FID write FID;
      property Name            : String read FName write SetName;
      property Layer           : TPlanLayer read FLayer write SetLayer;
      property Objects[Index : Integer] : TPlanObject read GetObject;
      property ObjectsCount    : Integer read GetObjectsCount;
      property ObjectTypeDescr : String read GetObjectTypeDescr;
      property Owner           : TOxygenPlan read FOwner;
      property Parent          : TPlanObject read FParent write SetParent;
      property Pen             : TPen read FPen write SetPen;
      property ReadOnly        : Boolean read FReadOnly write SetReadOnly;
      property Selected        : Boolean read FSelected write SetSelected;
      property SelectedByColor : Boolean read GetSelectedByColor;
      property Visible : Boolean read FVisible write SetVisible;

      procedure   AddObjectDescr(const SpaceNum : Integer; sl : TStrings); virtual;
      function    CanBeParent : Boolean; virtual; abstract;
      function    Clone(Parent : TPlanObject) : TPlanObject; virtual; abstract;
      constructor Create(Owner:TOxygenPlan); virtual;
      procedure   Draw(ACanvas:TCanvas); virtual; abstract;
      procedure   Free; virtual;
      function    HasInternalPoint(X,Y:Integer):Boolean; virtual; abstract;
      function    HasInternalProperty(const PropertyName : String) : Boolean; virtual;
      function    HasMarginPoint(X,Y:Integer; var MarginType:TMarginType):Boolean; virtual; abstract;
      procedure   Move(APoint:TPoint); virtual;
      procedure   NeedRepaint; virtual; abstract;
      function    ObjectFromCoord(X, Y:Integer; var MarginType : TMarginType):TPlanObject; virtual;
      procedure   ParentResized; virtual;
      procedure   Resize(APoint:TPoint); virtual; abstract;
      procedure   Select; virtual;
      procedure   SelectByColor(const AColor:TColor); virtual;
{$IFDEF USE_RX}
      procedure   StartBlink(const Interval:Integer); virtual;
      procedure   StopBlink; virtual;
{$ENDIF}
      procedure   UnSelect; virtual;
      procedure   UnSelectColor; virtual;
      procedure   UpdateObjectProperties; virtual;
  end;



  TPlanLine = class(TPlanObject)
    private
      Fx1,Fy1,Fx2,Fy2   :Integer;
      ScrSave1,ScrSave2 :TPoint;
      procedure SetX1(Value:Integer);
      procedure SetX2(Value:Integer);
      procedure SetY1(Value:Integer);
      procedure SetY2(Value:Integer);
      function  GetScrPoint1:TPoint;
      function  GetScrPoint2:TPoint;
      procedure SetScrPoint1(p:TPoint);
      procedure SetScrPoint2(p:TPoint);
    protected
      function  GetObjectTypeDescr : String; override;
      procedure FillObjectProperties; override;
      procedure SaveCoords; override;
      function  AddPoint(const CoordX,CoordY:Integer; CallHandler : TCallHandler {$IFNDEF VER100}= chMouseMove{$ENDIF}):Boolean; override;
      function  GetClippingRgn : HRGN; override;
    public
      property X1 : Integer read Fx1 write SetX1;
      property Y1 : Integer read Fy1 write SetY1;
      property X2 : Integer read Fx2 write SetX2;
      property Y2 : Integer read Fy2 write SetY2;
      property ScrPoint1 : TPoint read GetScrPoint1 write SetScrPoint1;
      property ScrPoint2 : TPoint read GetScrPoint2 write SetScrPoint2;

      function    CanBeParent : Boolean; override;
      function    Clone(Parent : TPlanObject) : TPlanObject; override;
      constructor Create(Owner:TOxygenPlan); override;
      procedure   Draw(ACanvas:TCanvas); override;
      function    GetDisplayRect:TRect; override;
      function    HasInternalPoint(X,Y:Integer):Boolean; override;
      function    HasInternalProperty(const PropertyName : String) : Boolean; override;
      function    HasMarginPoint(X,Y:Integer; var MarginType:TMarginType):Boolean; override;
      procedure   Move(APoint:TPoint); override;
      procedure   NeedRepaint; override;
      procedure   Resize(APoint:TPoint); override;
      procedure   UpdateObjectProperties; override;
  end;

  TPlanBox = class(TPlanObject)
    private
      Fx1,Fy1,Fx2,Fy2   : Integer;
      ScrSave1,ScrSave2 : TPoint;
      procedure SetX1(Value:Integer);
      procedure SetX2(Value:Integer);
      procedure SetY1(Value:Integer);
      procedure SetY2(Value:Integer);
      function  GetScrPoint1:TPoint;
      function  GetScrPoint2:TPoint;
      procedure SetScrPoint1(p:TPoint);
      procedure SetScrPoint2(p:TPoint);
    protected
      function  GetObjectTypeDescr : String; override;
      procedure FillObjectProperties; override;
      procedure SaveCoords; override;
      function  AddPoint(const CoordX,CoordY:Integer; CallHandler : TCallHandler {$IFNDEF VER100}= chMouseMove{$ENDIF}):Boolean; override;
    public
      property X1 : Integer read Fx1 write SetX1;
      property Y1 : Integer read Fy1 write SetY1;
      property X2 : Integer read Fx2 write SetX2;
      property Y2 : Integer read Fy2 write SetY2;
      property ScrPoint1 : TPoint read GetScrPoint1 write SetScrPoint1;
      property ScrPoint2 : TPoint read GetScrPoint2 write SetScrPoint2;

      function    CanBeParent : Boolean; override;
      function    Clone(Parent : TPlanObject) : TPlanObject; override;
      constructor Create(Owner:TOxygenPlan); override;
      procedure   Draw(ACanvas:TCanvas); override;
      function    GetDisplayRect:TRect; override;
      function    HasInternalPoint(X,Y:Integer):Boolean; override;
      function    HasInternalProperty(const PropertyName : String) : Boolean; override;
      function    HasMarginPoint(X,Y:Integer; var MarginType:TMarginType):Boolean; override;
      procedure   Move(APoint:TPoint); override;
      procedure   NeedRepaint; override;
      procedure   Resize(APoint:TPoint); override;
      procedure   UpdateObjectProperties; override;
  end;

  TPlanCircle = class(TPlanObject)
    private
      Fx,Fy,FRadius : Integer;
      ScrSaveCenter : TPoint;
      ScrSaveRadius : Integer;
      procedure SetX(Value:Integer);
      procedure SetY(Value:Integer);
      procedure SetRadius(Value:Integer);
      function  GetScrCenter:TPoint;
      function  GetScrRadius:Integer;
      procedure SetScrCenter(p:TPoint);
      procedure SetScrRadius(r:Integer);
    protected
      function  GetObjectTypeDescr : String; override;
      procedure FillObjectProperties; override;
      procedure SaveCoords; override;
      function  AddPoint(const CoordX,CoordY:Integer; CallHandler : TCallHandler {$IFNDEF VER100}= chMouseMove{$ENDIF}):Boolean; override;
      function  GetClippingRgn : HRGN; override;
    public
      property X : Integer read Fx write SetX;
      property Y : Integer read Fy write SetY;
      property Radius : Integer read FRadius write SetRadius;
      property ScrCenter : TPoint read GetScrCenter write SetScrCenter;
      property ScrRadius : Integer read GetScrRadius write SetScrRadius;

      function    CanBeParent : Boolean; override;
      function    Clone(Parent : TPlanObject) : TPlanObject; override;
      constructor Create(Owner:TOxygenPlan); override;
      procedure   Draw(ACanvas:TCanvas); override;
      function    GetDisplayRect:TRect; override;
      function    HasInternalPoint(X,Y:Integer):Boolean; override;
      function    HasInternalProperty(const PropertyName : String) : Boolean; override;
      function    HasMarginPoint(X,Y:Integer; var MarginType:TMarginType):Boolean; override;
      procedure   Move(APoint:TPoint); override;
      procedure   NeedRepaint; override;
      procedure   Resize(APoint:TPoint); override;
      procedure   UpdateObjectProperties; override;
  end;

  TPlanText = class(TPlanObject)
    private
      Fx1,Fy1,Fx2,Fy2 : Integer;
      FText, FSaveText, FBlinkText : String;
      FFont, FSaveFont, FBlinkFont : TFont;
      FAlignment : TAlignment;
      ScrSave1,ScrSave2 : TPoint;
      procedure SetX1(Value:Integer);
      procedure SetX2(Value:Integer);
      procedure SetY1(Value:Integer);
      procedure SetY2(Value:Integer);
      procedure SetText(Value:String);
      procedure SetFont(Value:TFont);
      procedure SetAlignment(Value:TAlignment);
      function  GetScrPoint1:TPoint;
      function  GetScrPoint2:TPoint;
      procedure SetScrPoint1(p:TPoint);
      procedure SetScrPoint2(p:TPoint);
    protected
      function  GetObjectTypeDescr : String; override;
      procedure FillObjectProperties; override;
      procedure SaveCoords; override;
      function  AddPoint(const CoordX,CoordY:Integer; CallHandler : TCallHandler {$IFNDEF VER100}= chMouseMove{$ENDIF}):Boolean; override;
{$IFDEF USE_RX}
      procedure BlinkTimer(Sender: TObject); override;
{$ENDIF}
    public
{$IFDEF USE_RX}
      property BlinkFont : TFont read FBlinkFont;
      property BlinkText : String read FBlinkText write FBlinkText;
{$ENDIF}
      property X1 : Integer read Fx1 write SetX1;
      property Y1 : Integer read Fy1 write SetY1;
      property X2 : Integer read Fx2 write SetX2;
      property Y2 : Integer read Fy2 write SetY2;
      property Text : String read FText write SetText;
      property Font : TFont read FFont write SetFont;
      property Alignment : TAlignment read FAlignment write SetAlignment;
      property ScrPoint1 : TPoint read GetScrPoint1 write SetScrPoint1;
      property ScrPoint2 : TPoint read GetScrPoint2 write SetScrPoint2;

      function    CanBeParent : Boolean; override;
      function    Clone(Parent : TPlanObject) : TPlanObject; override;
      constructor Create(Owner:TOxygenPlan); override;
      procedure   Free; override;
      procedure   Draw(ACanvas:TCanvas); override;
      function    GetDisplayRect:TRect; override;
      function    HasInternalPoint(X,Y:Integer):Boolean; override;
      function    HasInternalProperty(const PropertyName : String) : Boolean; override;
      function    HasMarginPoint(X,Y:Integer; var MarginType:TMarginType):Boolean; override;
      procedure   Move(APoint:TPoint); override;
      procedure   NeedRepaint; override;
      procedure   Resize(APoint:TPoint); override;
{$IFDEF USE_RX}
      procedure   StartBlink(const Interval:Integer); override;
      procedure   StopBlink; override;
{$ENDIF}
      procedure   UpdateObjectProperties; override;
  end;

  TPlanPicture = class(TPlanBox)
    private
      FPicture {$IFDEF USE_RX}, FSavePicture, FBlinkPicture {$ENDIF} : TPicture;
      FStdPictureName : String;
      procedure SetStdPictureName(const Value : String);
      function  GetTransparent : Boolean;
      procedure SetTransparent(const Value : Boolean);
    protected
      function  GetObjectTypeDescr : String; override;
      procedure FillObjectProperties; override;
{$IFDEF USE_RX}
      procedure BlinkTimer(Sender: TObject); override;
{$ENDIF}
    public
{$IFDEF USE_RX}
      property BlinkPicture : TPicture read FBlinkPicture;
{$ENDIF}
      property X1;
      property Y1;
      property X2;
      property Y2;
      property ScrPoint1;
      property ScrPoint2;
      property StdPictureName : String read FStdPictureName write SetStdPictureName;
      property Picture : TPicture read FPicture; {write FPicture;}
      property Transparent : Boolean read GetTransparent write SetTransparent;

      function    Clone(Parent : TPlanObject) : TPlanObject; override;
      constructor Create(Owner:TOxygenPlan); override;
      procedure   Draw(ACanvas:TCanvas); override;
      procedure   Free; override;
      function    HasInternalProperty(const PropertyName : String) : Boolean; override;
{$IFDEF USE_RX}
      procedure   StartBlink(const Interval:Integer); override;
      procedure   StopBlink; override;
{$ENDIF}
      procedure   UpdateObjectProperties; override;
  end;

{$IFDEF USE_RX}
  TPlanAniPicture = class(TPlanPicture)
    private
      FImageIndex : Integer;
      FImageCount : Integer;
      FCurBitmap  : TBitmap;
      FInterval   : Integer;
      procedure SetImageIndex(const Value : Integer);
      procedure SetImageCount(const Value : Integer);
    protected
      procedure FillObjectProperties; override;
      procedure BlinkTimer(Sender: TObject); override;
      function  GetObjectTypeDescr : String; override;
    public
      property ImageCount : Integer read FImageCount write SetImageCount;
      property ImageIndex : Integer read FImageIndex write SetImageIndex;
      property Interval : Integer read FInterval write FInterval;

      function    CanBeParent : Boolean; override;
      function    Clone(Parent : TPlanObject) : TPlanObject; override;
      constructor Create(Owner:TOxygenPlan); override;
      procedure Animate(const Interval:Integer);
      procedure Draw(ACanvas:TCanvas); override;
      procedure Free; override;
      function  HasInternalProperty(const PropertyName : String) : Boolean; override;
      procedure StartBlink(const Interval:Integer); override;
      procedure Stop;
      procedure StopBlink; override;
      procedure UpdateObjectProperties; override;
  end;
{$ENDIF}


  TPlanPolygon = class(TPlanObject)
    private
      FPointCount:Integer;
      FPointArray:PPointArray;
      FScrPointArray:PPointArray;
      FScrSaveArray:PPointArray;
      FCurMovingPointIndex : Integer;
      procedure SetPointCount(Value:Integer);
      function  GetPoint(Index:Integer):TPoint;
      procedure SetPoint(Index:Integer; Value:TPoint);
    protected
      function  GetObjectTypeDescr : String; override;
      procedure FillObjectProperties; override;
      procedure SaveCoords; override;
      function  AddPoint(const CoordX,CoordY:Integer; CallHandler : TCallHandler {$IFNDEF VER100}= chMouseMove{$ENDIF}):Boolean; override;
      function  GetClippingRgn : HRGN; override;
    public
      property PointCount:Integer read FPointCount write SetPointCount;
      property Points[Index:Integer]:TPoint read GetPoint write SetPoint;

      function    CanBeParent : Boolean; override;
      function    Clone(Parent : TPlanObject) : TPlanObject; override;
      constructor Create(Owner:TOxygenPlan); override;
      procedure   ParentResized; override;
      procedure   Draw(ACanvas:TCanvas); override;
      procedure   DeletePoint(const Index : Integer);
      procedure   Free; override;
      function    GetDisplayRect:TRect; override;
      function    HasInternalPoint(X,Y:Integer):Boolean; override;
      function    HasInternalProperty(const PropertyName : String) : Boolean; override;
      function    HasMarginPoint(X,Y:Integer; var MarginType:TMarginType):Boolean; override;
      procedure   Move(APoint:TPoint); override;
      procedure   NeedRepaint; override;
      procedure   Resize(APoint:TPoint); override;
      procedure   UpdateObjectProperties; override;
  end;

  TPlanCompositeObject = class(TPlanBox)
    private
      procedure SetPen(Value:TPen); override;
      procedure SetBrush(Value:TBrush); override;
      procedure SetSelected(Value:Boolean); override;
   protected
      function  GetObjectTypeDescr : String; override;
    public
      function  CanBeParent : Boolean; override;
      procedure Draw(ACanvas:TCanvas); override;
      procedure Select; override;
      procedure UnSelect; override;
      procedure SelectByColor(const AColor:TColor); override;
      procedure UnSelectColor; override;
  end;

{$IFDEF USE_RX}
  TPlanAniGif = class(TPlanBox)
    private
      FImage : TGIFImage;
      FStdAniGifName : String;
      function GetDelayTime(Index: Integer): Cardinal;
      procedure SetStdAniGifName(const Value : String);
      procedure SetGIFImage(Value:TGIFImage);
    protected
      procedure BlinkTimer(Sender: TObject); override;
      function  GetFrameBitmap(Index: Integer; var TransColor: TColor): TBitmap;
      function  GetObjectTypeDescr : String; override;
      procedure FillObjectProperties; override;
    public
      property StdAniGifName : String read FStdAniGifName write SetStdAniGifName;
      property GIFImage : TGIFImage read FImage write SetGIFImage;

      function    CanBeParent : Boolean; override;
      function    Clone(Parent : TPlanObject) : TPlanObject; override;
      constructor Create(Owner:TOxygenPlan); override;
      procedure Animate;
      procedure Draw(ACanvas:TCanvas); override;
      procedure Free; override;
      function  HasInternalProperty(const PropertyName : String) : Boolean; override;
      procedure StartBlink(const Interval:Integer); override;
      procedure StopBlink; override;
      procedure Stop;
      procedure UpdateObjectProperties; override;
  end;
{$ENDIF}

  TPlanStandardPicture = class(TPicture)
    public
      Name : String;
      function GetDescrString:String;
  end;

{$IFDEF USE_RX}
  TPlanStandardAniGif = class(TGifImage)
    public
      Name : String;
      function GetDescrString:String;
  end;
{$ENDIF}

  TPlanOption = (poAutoAnimate, poSaveStandardPictures, poSaveBackgroundPicture, poChildrenClipping);
  TPlanOptions = set of TPlanOption;

  TObjectClickEvent    = procedure (Plan : TOxygenPlan; PlanObject : TPlanObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer) of object;
  TObjectDblClickEvent = procedure (Plan : TOxygenPlan; PlanObject : TPlanObject) of object;
  TParseProgressEvent  = procedure (Plan : TOxygenPlan; const PercentsCompleted : Integer) of object;
  TNeedPictureEvent    = function  (Plan : TOxygenPlan; const PictureName : String) : Boolean of object;
{$IFDEF USE_RX}
  TNeedAniGifEvent     = function  (Plan : TOxygenPlan; const AniGifName : String) : Boolean of object;
{$ENDIF}

  TOxygenPlan = class(TCustomPanel)
    private
      FScaleX, FScaleY      : Integer;
      FCurX,FCurY           : Integer;
      FStartCoord           : TPoint;
      FActiveObject         : TPlanObject;
      FActiveMarginType     : TMarginType;
      FBackgroundPicture    : TPicture;
      FReadOnly             : Boolean;
      FPlanStrings          : TStrings;
      FNewClickNum          : Integer;    // click number when drawing new object
      FSelectColor          : TColor;     // small selection boxes color
      FCurLayer             : TPlanLayer; // current layer
      FCurDrawingObject     : TPlanObject; // current drawing object
      FCurDrawingPointNum   : Integer;     // current drawing object's point number
      FPen                  : TPen;
      FBrush                : TBrush;
      NewObjUM, OldObjUM    : TPlanObject;
      FDisableRepaintCount  : Integer;
      FReadyToMoving        : Boolean;

{$IFDEF USE_RX}
      FTimerList            : TRxTimerList;
      FStandardAniGifsList  : TList;
{$ENDIF}
      FStandardPicturesList : TList;
      FOptions              : TPlanOptions;

      FObjectClick          : TObjectClickEvent;
      FObjectDblClick       : TObjectDblClickEvent;
      FObjectMouseMove      : TObjectDblClickEvent;
      FObjectMouseEnter     : TObjectDblClickEvent;
      FObjectMouseLeave     : TObjectDblClickEvent;
      FAfterDrawNewObject   : TObjectDblClickEvent;
      FParseProgress        : TParseProgressEvent;
      FNeedPicture          : TNeedPictureEvent;
{$IFDEF USE_RX}
      FNeedAniGif           : TNeedAniGifEvent;
{$ENDIF}

      function  GetPlanStrings:TStrings;
      procedure SetPlanStrings(Value:TStrings);
      procedure SetBackgroundPicture(Value:TPicture);
      procedure SetScaleX(Value:Integer);
      procedure SetScaleY(Value:Integer);
      procedure SetReadOnly(Value:Boolean);
      function  GetBackgroundPictureType:String;
      procedure SetCurLayer(Value:TPlanLayer);
      function  GetPicture(Index:Integer):TPlanStandardPicture;
      procedure ClearPictures;
{$IFDEF USE_RX}
      function  GetAniGif(Index:Integer):TPlanStandardAniGif;
      procedure ClearAniGifs;
      function  GetAniGifCount : Integer;
      procedure StdAniGifChange(Sender : TObject);
{$ENDIF}
      function  GetPictureCount : Integer;
      procedure SetActiveObject(PlanObject:TPlanObject);
      procedure ReadStringData(Reader: TReader);
      procedure StdPictureChange(Sender : TObject);
      procedure SetOptions(Value : TPlanOptions);
      function  GetObjectCount : Integer;
      procedure SetBrush(Value : TBrush);
      procedure SetPen(Value : TPen);

    protected
      procedure DoObjectClick(PlanObject : TPlanObject; Button : TMouseButton; Shift : TShiftState; X, Y : Integer); virtual;
      procedure DoObjectDblClick(PlanObject : TPlanObject); virtual;
      procedure DoObjectMouseMove(PlanObject : TPlanObject); virtual;
      procedure DoObjectMouseEnter(PlanObject : TPlanObject); virtual;
      procedure DoObjectMouseLeave(PlanObject : TPlanObject); virtual;
      procedure DoAfterDrawNewObject(PlanObject : TPlanObject); virtual;
      procedure DoParseProgress(const PercentsCompleted : Integer); virtual;
      function  DoNeedPicture(Plan : TOxygenPlan; const PictureName : String) : Boolean; virtual;
{$IFDEF USE_RX}
      function  DoNeedAniGif(Plan : TOxygenPlan; const AniGifName : String) : Boolean; virtual;
{$ENDIF}
      procedure DefineProperties(Filer: TFiler); override;
      procedure Loaded; override;

      procedure Resize; override;
      procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
      procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
      procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
      procedure DblClick; override;
      procedure Paint; override;

    public
      Layers : TPlanLayers;

      property ActiveObject:TPlanObject read FActiveObject write SetActiveObject;
{$IFDEF USE_RX}
      property AniGifs[Index : Integer] : TPlanStandardAniGif read GetAniGif;
      property AniGifCount : Integer read GetAniGifCount;
{$ENDIF}
      property BackgroundPictureType : String read GetBackgroundPictureType;
      property Canvas;
      property CurrentLayer : TPlanLayer read FCurLayer write SetCurLayer stored False;
      property ObjectCount : Integer read GetObjectCount;
      property Pictures[Index : Integer] : TPlanStandardPicture read GetPicture;
      property PictureCount : Integer read GetPictureCount;

{$IFDEF USE_RX}
      function    AddAniGif(const AniGifName : String; AniGif : TGIFImage):Integer;
      function    AniGifIndex(const AniGifName:String):Integer;
      procedure   DeleteAniGif(Index:Integer);
      procedure   LoadAniGif(const AniGifName : String; PlanAniGif : TPlanAniGif);
{$ENDIF}
      function    AddPicture(const PictureName : String; Picture : TPicture):Integer;
      procedure   DeletePicture(Index : Integer);
      procedure   LoadPicture(const PictureName : String; PlanPicture : TPlanPicture);
      function    PictureIndex(const PictureName:String):Integer;

      procedure   Assign(Source : TPersistent); override;
      constructor Create(AOwner : TComponent); override;
      procedure   CancelDrawNewObject;
      procedure   Clear;
      procedure   EnableRepaint;
      procedure   DisableRepaint;
      procedure   ExportToMetafile(const FileName : String); // 0.91
      procedure   ExportToBitmap(const FileName : String);  // 0.91
      procedure   ExportToJpeg(const FileName : String);  // 0.91
{$IFDEF USE_RX}
      procedure   ExportToGif(const FileName : String);  // 0.91
{$ENDIF}
      procedure   DrawNewObject(PlanObjectClass : TPlanObjectClass);
      function    FindObject(ObjectClass : TPlanObjectClass; const ObjectName : String; const ObjectID : Integer):TPlanObject;
      procedure   GetBitmap(Bitmap : TBitmap); // 0.91
      procedure   LoadFromFile(FileName : TFileName);
      procedure   LoadFromStream(Stream : TStream);
      function    ObjectByID(ObjectClass:TPlanObjectClass; const ObjectID:Integer):TPlanObject;
      function    ObjectByName(ObjectClass : TPlanObjectClass; const ObjectName : String):TPlanObject;
      function    ObjectUnderMouse(var MarginType : TMarginType) : TPlanObject;
      procedure   PaintToCanvas(ACanvas : TCanvas); // 0.91
      procedure   Parse(PlanData : TStrings);
      procedure   SaveTo(PlanData : TStrings);
      procedure   SaveToFile(FileName:String);
      procedure   SaveToStream(Stream:TStream);
      destructor  Destroy; override;

    published
      property Align;
      //property Alignment;
      // Start animating TPlanAniGif and TPlanAniPicture automatically
      property BackgroundPicture     : TPicture read FBackgroundPicture write SetBackgroundPicture;
      property BevelInner;
      property BevelOuter;
      property BevelWidth;
      property BorderWidth;
      property BorderStyle;
      property Brush                 : TBrush read FBrush write SetBrush;
      {$IFNDEF VER100}
      property Constraints;
      {$ENDIF}
      property DragCursor;
      property DragMode;
      {$IFNDEF VER100}
      property DragKind;
      property DockSite;
      property DoubleBuffered;
      {$ENDIF}
      property Enabled;
      property Color;
      property Ctl3D;
      property Font;
      property Locked;
      property Options               : TPlanOptions read FOptions write SetOptions default [poAutoAnimate];
      property ParentColor;
      property ParentCtl3D;
      property ParentFont;
      property ParentShowHint;
      property Pen                   : TPen read FPen write SetPen;
      property PlanStrings           : TStrings read GetPlanStrings write SetPlanStrings;
      property PopupMenu;
      property ReadOnly              : Boolean read FReadOnly write SetReadOnly;
      property SelectColor           : TColor read FSelectColor write FSelectColor;
      property ScaleX                : Integer read FScaleX write SetScaleX;
      property ScaleY                : Integer read FScaleY write SetScaleY;
      property ShowHint;
      property TabOrder;
      property TabStop;
      property Visible;
      {$IFNDEF VER100}
      property UseDockManager default True;
      {$ENDIF}

      // events
      property OnClick;
      property OnDblClick;
      {$IFNDEF VER100}
      property OnCanResize;
      property OnConstrainedResize;
      property OnDockDrop;
      property OnDockOver;
      property OnStartDock;
      property OnEndDock;
      property OnUnDock;
      property OnGetSiteInfo;
      {$ENDIF}
      property OnDragDrop;
      property OnDragOver;
      property OnEndDrag;
      property OnEnter;
      property OnExit;
      property OnKeyDown;
      property OnKeyPress;
      property OnKeyUp;
      property OnMouseDown;
      property OnMouseMove;
      property OnMouseUp;
      {$IFNDEF VER100}
      property OnMouseWheel;
      property OnMouseWheelDown;
      property OnMouseWheelUp;
      {$ENDIF}
      property OnResize;
      property OnStartDrag;

      property OnAfterDrawNewObject : TObjectDblClickEvent read FAfterDrawNewObject write FAfterDrawNewObject;
      property OnNeedPicture        : TNeedPictureEvent read FNeedPicture write FNeedPicture;
{$IFDEF USE_RX}
      property OnNeedAniGif         : TNeedAniGifEvent read FNeedAniGif write FNeedAniGif;
{$ENDIF}
      property OnObjectClick        : TObjectClickEvent read FObjectClick write FObjectClick;
      property OnObjectDblClick     : TObjectDblClickEvent read FObjectDblClick write FObjectDblClick;
      property OnObjectMouseMove    : TObjectDblClickEvent read FObjectMouseMove write FObjectMouseMove;
      property OnObjectMouseEnter   : TObjectDblClickEvent read FObjectMouseEnter write FObjectMouseEnter;
      property OnObjectMouseLeave   : TObjectDblClickEvent read FObjectMouseLeave write FObjectMouseLeave;
      property OnParseProgress      : TParseProgressEvent read FParseProgress write FParseProgress;
  end;

  function  GetPictureType(Picture : TPicture) : String;
  function  LoadPictureFromStream(const PictureType : String; Picture : TPicture; Stream : TStream) : Boolean;
  function  GetPictureStreamString(Picture : TPicture) : String;
  procedure RenameStdPicture(Plan : TOxygenPlan; const OldName,NewName : String);
  function  StandardPictureReferenceCount(Plan : TOxygenPlan; const StdPictureName : String) : Integer;
{$IFDEF USE_RX}
  procedure RenameStdAniGif(Plan : TOxygenPlan; const OldName,NewName : String);
  function  StandardAniGifReferenceCount(Plan : TOxygenPlan; const StdAniGifName : String) : Integer;
{$ENDIF}

procedure Register;

implementation

  const
    MaxDelayTime = 10000;
    MinDelayTime = 50;

function ReadColor(s : String):TColor;
begin
  Result := clWhite;
  if (UpperCase(s)=UpperCase(pcBlack)) then   Result:=clBlack
  else if (UpperCase(s)=UpperCase(pcMaroon)) then  Result:=clMaroon
  else if (UpperCase(s)=UpperCase(pcGreen)) then   Result:=clGreen
  else if (UpperCase(s)=UpperCase(pcOlive)) then   Result:=clOlive
  else if (UpperCase(s)=UpperCase(pcNavy)) then    Result:=clNavy
  else if (UpperCase(s)=UpperCase(pcPurple)) then  Result:=clPurple
  else if (UpperCase(s)=UpperCase(pcTeal)) then    Result:=clTeal
  else if (UpperCase(s)=UpperCase(pcGray)) then    Result:=clGray
  else if (UpperCase(s)=UpperCase(pcSilver)) then  Result:=clSilver
  else if (UpperCase(s)=UpperCase(pcRed)) then     Result:=clRed
  else if (UpperCase(s)=UpperCase(pcLime)) then    Result:=clLime
  else if (UpperCase(s)=UpperCase(pcYellow)) then  Result:=clYellow
  else if (UpperCase(s)=UpperCase(pcBlue)) then    Result:=clBlue
  else if (UpperCase(s)=UpperCase(pcFuchsia)) then Result:=clFuchsia
  else if (UpperCase(s)=UpperCase(pcAqua)) then    Result:=clAqua
  else if (UpperCase(s)=UpperCase(pcWhite)) then   Result:=clWhite
  else try Result:=StrToInt(s) except end;
end;

function EncodeColor(c:TColor):String;
begin
  Result:=IntToStr(c);
  if (c=clBlack) then     Result:=pcBlack;
  if (c=clMaroon) then  Result:=pcMaroon;
  if (c=clGreen) then   Result:=pcGreen;
  if (c=clOlive) then   Result:=pcOlive;
  if (c=clNavy) then    Result:=pcNavy;
  if (c=clPurple) then  Result:=pcPurple;
  if (c=clTeal) then    Result:=pcTeal;
  if (c=clGray) then    Result:=pcGray;
  if (c=clSilver) then  Result:=pcSilver;
  if (c=clRed) then     Result:=pcRed;
  if (c=clLime) then    Result:=pcLime;
  if (c=clYellow) then  Result:=pcYellow;
  if (c=clBlue) then    Result:=pcBlue;
  if (c=clFuchsia) then Result:=pcFuchsia;
  if (c=clAqua) then    Result:=pcAqua;
  if (c=clWhite) then   Result:=pcWhite;
end;

function GetStyleString(AFont:TFont):String;
begin
  Result:='';
  if fsItalic in AFont.Style then Result:=Result+'I';
  if fsBold in AFont.Style then Result:=Result+'B';
  if fsUnderline in AFont.Style then Result:=Result+'U';
  if fsStrikeout in AFont.Style then Result:=Result+'S';
end;

function Stream2String(ms:TMemoryStream):String; 
  var b:Byte;
begin
  Result:='';
  ms.Position:=0;
  while (ms.Position<ms.Size) do try
    ms.Read(b,1);
    Result:=Result+IntToHex(b,2);
  except end;
end;

procedure String2Stream(const s:String; var ms:TMemoryStream);
  var b:Byte;
      st:String;
begin
  st:=s;
  ms.Clear;
  while (Length(st)>0) do try
    b:=Byte(Hex2Dec(System.Copy(st,1,2)));
    ms.Write(b,1);
    System.Delete(st,1,2);
  except end;
  ms.Position:=0;
end;

function GetPictureType(Picture : TPicture) : String;
  var g : TGraphic;
begin
  Result:='';
  if ((Picture = nil) or (Picture.Graphic = nil)) then Exit;
  g:=Picture.Graphic;
  if (g is TIcon) then Result:='Icon'
  else if (g is TMetafile) then Result:='Metafile'
{$IFDEF USE_RX}
  else if (g is TGIFImage) then Result:='GIF'
{$ENDIF}
  else if (g is TJPEGImage) then Result:='JPEG'
  else Result:='Bitmap';
end;

function LoadPictureFromStream(const PictureType : String; Picture : TPicture; Stream : TStream) : Boolean;
  var g  : TGraphic;
      pt : String;

  function TryFormat(gc : TGraphicClass) : Boolean;
  begin
    Result:=False;
    Stream.Position:=0;
    g:=gc.Create;
    try
      g.LoadFromStream(Stream);
      Picture.Assign(g);
      Result:=True;
      g.Free;
    except
      g.Free;
    end;
  end;

begin
  Result:=False;
  if ((Picture = nil) or (Stream.Size = 0)) then Exit;
  Stream.Position:=0;
  pt:=UpperCase(Trim(PictureType));
  if (pt = '') then Result:=TryFormat(TBitmap) or TryFormat(TMetafile) or TryFormat(TIcon) or TryFormat(TJPEGImage) {$IFDEF USE_RX} or TryFormat(TGIFImage) {$ENDIF}
  else if (pt = 'BITMAP') then Result:=TryFormat(TBitmap)
  else if (pt = 'METAFILE') then Result:=TryFormat(TMetafile)
  else if (pt = 'ICON') then Result:=TryFormat(TIcon)
{$IFDEF USE_RX}
  else if (pt = 'GIF') then Result:=TryFormat(TGIFImage)
{$ENDIF}
  else if (pt = 'JPEG') then Result:=TryFormat(TJPEGImage);
end;

function GetPictureStreamString(Picture : TPicture) : String;
  var ms  : TMemoryStream;
      s   : String;
      pic : TPicture;
begin
  Result := '';
  if ((Picture.Graphic=nil) or Picture.Graphic.Empty) then Exit;
  pic:=TPicture.Create;
  pic.Assign(Picture);
  ms:=TMemoryStream.Create;
  Picture.Graphic.SaveToStream(ms);
  s:=GetPictureType(pic);
  Result:=s+','+IntToStr(Integer(Picture.Graphic.Transparent))+','+Stream2String(ms);
  ms.Free;
  Picture.Assign(pic);
  pic.Free;
end;

procedure RenameStdPicture(Plan : TOxygenPlan; const OldName,NewName : String);
  var l,i : Integer;
      pp : TPlanPicture;
begin
  for l := 1 to Plan.Layers.Count do
    for i := 1 to Plan.Layers[l-1].Count do
      if (Plan.Layers[l-1].Objects[i-1] is TPlanPicture) then begin
        pp := (Plan.Layers[l-1].Objects[i-1] as TPlanPicture);
        if (AnsiUpperCase(pp.StdPictureName) = AnsiUpperCase(OldName)) then pp.StdPictureName := NewName;
      end;
end;

function StandardPictureReferenceCount(Plan : TOxygenPlan; const StdPictureName : String) : Integer;
  var l,i : Integer;
      pp : TPlanPicture;
begin
  Result := 0;
  for l := 1 to Plan.Layers.Count do
    for i := 1 to Plan.Layers[l-1].Count do
      if (Plan.Layers[l-1].Objects[i-1] is TPlanPicture) then begin
        pp := (Plan.Layers[l-1].Objects[i-1] as TPlanPicture);
        if (AnsiUpperCase(pp.StdPictureName) = AnsiUpperCase(StdPictureName)) then Inc(Result);
      end;

end;

{$IFDEF USE_RX}
procedure RenameStdAniGif(Plan : TOxygenPlan; const OldName,NewName : String);
  var l,i : Integer;
      pp : TPlanAniGif;
begin
  for l := 1 to Plan.Layers.Count do
    for i := 1 to Plan.Layers[l-1].Count do
      if (Plan.Layers[l-1].Objects[i-1] is TPlanAniGif) then begin
        pp := (Plan.Layers[l-1].Objects[i-1] as TPlanAniGif);
        if (AnsiUpperCase(pp.StdAniGifName) = AnsiUpperCase(OldName)) then pp.StdAniGifName := NewName;
      end;
end;

function StandardAniGifReferenceCount(Plan : TOxygenPlan; const StdAniGifName : String) : Integer;
  var l,i : Integer;
      pp : TPlanAniGif;
begin
  Result := 0;
  for l := 1 to Plan.Layers.Count do
    for i := 1 to Plan.Layers[l-1].Count do
      if (Plan.Layers[l-1].Objects[i-1] is TPlanAniGif) then begin
        pp := (Plan.Layers[l-1].Objects[i-1] as TPlanAniGif);
        if (AnsiUpperCase(pp.StdAniGifName) = AnsiUpperCase(StdAniGifName)) then Inc(Result);
      end;

end;
{$ENDIF}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//   TPlanStandardPicture
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
function TPlanStandardPicture.GetDescrString:String;
  var ms  : TMemoryStream;
      s   : String;
      pic : TPicture;
begin
  Result:='';
  if ((Graphic=nil) or Graphic.Empty) then Exit;
  pic:=TPicture.Create;
  pic.Assign(Self);
  Result:=cmStandardPicture+Name+',';
  ms:=TMemoryStream.Create;
  Graphic.SaveToStream(ms);
  s:=GetPictureType(pic);
  Result:=Result+s+','+IntToStr(Integer(Graphic.Transparent))+','+Stream2String(ms);
  ms.Free;
  Self.Assign(pic);
  pic.Free;
end;


{$IFDEF USE_RX}
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//   TPlanStandardAniGif
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
function TPlanStandardAniGif.GetDescrString:String;
  var ms:TMemoryStream;
      s:String;
      gif:TGifImage;
begin
  Result:='';
  if Empty then Exit;
  gif:=TGifImage.Create;
  gif.Assign(Self);
  Result:=cmStandardAniGif+Name+',';
  ms:=TMemoryStream.Create;
  gif.SaveToStream(ms);
  Result:=Result+s+Stream2String(ms);
  ms.Free;
  gif.Free;
end;
{$ENDIF}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//    TPlanLayer
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
constructor TPlanLayer.Create(AOwner : TOxygenPlan);
begin
  inherited Create;
  FObjectList:=TList.Create;
  FOwner:=AOwner;
  HRegion:=0;
  FReadOnly := False;
end;

procedure TPlanLayer.Free;
begin
  Clear;
  FObjectList.Free;
  inherited Free;
end;

function TPlanLayer.GetObject(Index:Integer):TPlanObject;
begin
  Result:=TPlanObject(FObjectList[Index]);
end;

function TPlanLayer.GetObjectsCount : Integer;
begin
  Result:=FObjectList.Count;
end;

procedure TPlanLayer.SetVisible(Value:Boolean);
begin
  if (FVisible=Value) then Exit;
  FVisible:=Value;
  FOwner.Repaint;
end;

procedure TPlanLayer.SetReadOnly(const Value : Boolean);
begin
  FReadOnly:=Value;
  if Value then if Assigned(FOwner.ActiveObject) then if (FOwner.ActiveObject.Layer=Self) then FOwner.ActiveObject.UnSelect;
end;

function TPlanLayer.Add(po:TPlanObject):Integer;
begin
  Result:=FObjectList.Add(po);
end;

function TPlanLayer.IndexOf(po:TPlanObject):Integer;
begin
  Result:=FObjectList.IndexOf(po);
end;

procedure TPlanLayer.Delete(Index:Integer);
begin
  TPlanObject(FObjectList[Index]).Free;
end;

procedure TPlanLayer.Clear;
begin
  while (Count>0) do Delete(0);
  FObjectList.Clear;
end;

procedure TPlanLayer.Draw(ACanvas:TCanvas);
  var i:Integer;
      po:TPlanObject;
begin
  if (FOwner.FDisableRepaintCount > 0) then Exit;
  for i:=1 to FObjectList.Count do begin
    po:=TPlanObject(FObjectList[i-1]);
    if ((po.FParent=nil) and po.Visible) then po.Draw(ACanvas);
  end;
end;

function TPlanLayer.ObjectFromCoord(X, Y:Integer; var MarginType : TMarginType):TPlanObject;
  var i:Integer;
      po:TPlanObject;
      mt:TMarginType;
begin
  Result:=nil;
  MarginType := mtNone;
  if not Visible then Exit;
  for i:=Count downto 1 do begin
    if (Objects[i-1].HasMarginPoint(X,Y,mt) and Objects[i-1].Visible) then begin
      Result:=Objects[i-1];
      MarginType:=mt;
      Exit;
    end
    else if (Objects[i-1].HasInternalPoint(X,Y) and Objects[i-1].Visible) then begin
      Result:=Objects[i-1];
      MarginType := mtNone;
      po:=Objects[i-1].ObjectFromCoord(X,Y,mt);
      if (po<>nil) then begin
        Result:=po;
        MarginType := mt;
      end;
      Exit;
    end;
  end;
end;

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//    TPlanLayers
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
constructor TPlanLayers.Create(AOwner : TOxygenPlan);
begin
  inherited Create;
  FLayersList:=TList.Create;
  FOwner:=AOwner;
end;

procedure TPlanLayers.Free;
begin
  Clear;
  FLayersList.Free;
  inherited Free;
end;

function TPlanLayers.NewLayer(const LName:String):TPlanLayer;
begin
  Result:=TPlanLayer.Create(FOwner);
  with Result do begin
    Name:=LName;
    Index:=Self.Count;
    FVisible:=True;
  end;
end;

function TPlanLayers.Get(Index:Integer):TPlanLayer;
begin
  Result:=TPlanLayer(FLayersList[Index]);
end;

procedure TPlanLayers.Put(Index:Integer; Layer:TPlanLayer);
begin
  FLayersList[Index]:=Layer;
end;

function TPlanLayers.GetCount:Integer;
begin
  Result:=FLayersList.Count;
end;

function TPlanLayers.Add(const LayerName:String):Integer;
begin
  Result:=FLayersList.Add(NewLayer(LayerName));
end;

procedure TPlanLayers.Delete(Index:Integer);
begin
  TPlanLayer(FLayersList[Index]).Free;
  FLayersList.Delete(Index);
end;

function TPlanLayers.IndexOf(const LayerName:String):Integer;
  var i:Integer;
begin
  Result:=-1;
  for i:=1 to FLayersList.Count do if (UpperCase(TPlanLayer(FLayersList[i-1]).Name)=UpperCase(LayerName)) then begin
    Result:=i-1;
    Exit;
  end;
end;

procedure TPlanLayers.Clear;
begin
  while (Count>0) do Delete(0);
  FLayersList.Clear;
end;

function TPlanLayers.LayerByName(const LayerName:String):TPlanLayer;
  var i:Integer;
begin
  Result:=nil;
  for i:=1 to Count do if (AnsiUpperCase(Layers[i-1].Name)=AnsiUpperCase(LayerName)) then begin
    Result:=Layers[i-1];
    Exit;
  end;
end;

procedure TPlanLayers.AddLayersDescr(sl:TStrings);
  var i:Integer;
      pl:TPlanLayer;
      s:String;
begin
  for i:=1 to Count do begin
    pl:=TPlanLayer(FLayersList[i-1]);
    if pl.Visible then s:='1' else s:='0';
    if pl.ReadOnly then s:=s+',1' else s:=s+',0';
    sl.Add(cmLayer+pl.Name+','{+IntToStr(pl.Index)+','}+s);
  end;
end;

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//   TPlanObject
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
constructor TPlanObject.Create(Owner:TOxygenPlan);
begin
  inherited Create;
  FDestroying:=False;
  FChildren:=TList.Create;
  FProperties := TStringList.Create;
  FPen:=TPen.Create;
  FBrush:=TBrush.Create;
  FPen.Assign(Owner.Pen);
  FBrush.Assign(Owner.Brush);
  FID:=0;
  FName:='';
  FOwner:=Owner;
  FLayer:=FOwner.CurrentLayer;
  FParent:=nil;
  FSelected:=False;
  FSelectColor:=-1;
  FVisible:=True;
  FCanResize:=True;
{$IFDEF USE_RX}
  FBlinkTimerHandle:=0;
  FBlinkType:=btNone;
  FBlinkPen:=TPen.Create;
  FSavePen:=TPen.Create;
  FBlinkBrush:=TBrush.Create;
  FSaveBrush:=TBrush.Create;
  FBlinkIndex:=0;
{$ENDIF}
  FReadOnly := False;
  FCursor := crDefault;
end;

function TPlanObject.GetObjectTypeDescr : String;
begin
  Result := FProperties.Values['ObjectType'];
end;

procedure TPlanObject.DrawChildren(ACanvas:TCanvas);
  var i   : Integer;
      rgn : HRGN;
      FWasClippingRgn : Boolean;
begin
  if (FOwner.FDisableRepaintCount>0) then Exit;
  if ((not CanBeParent) or (FChildren.Count=0)) then Exit;
  if (poChildrenClipping in FOwner.Options) then begin
    rgn := CreateRectRgn(0,0,0,0);
    i := GetClipRgn(ACanvas.Handle,rgn); // Old clipping region
    FWasClippingRgn := (i=1);
    FClippingRgn := GetClippingRgn;
    if FWasClippingRgn then CombineRgn(FClippingRgn,rgn,FClippingRgn,RGN_AND);
    SelectClipRgn(ACanvas.Handle,FClippingRgn);
    DeleteObject(FClippingRgn);
    for i:=1 to FChildren.Count do TPlanObject(FChildren[i-1]).Draw(ACanvas);
    if FWasClippingRgn then SelectClipRgn(ACanvas.Handle,rgn) else SelectClipRgn(ACanvas.Handle,0);
    DeleteObject(rgn);
  end
  else for i:=1 to FChildren.Count do TPlanObject(FChildren[i-1]).Draw(ACanvas);
end;

function TPlanObject.GetClippingRgn : HRGN;
  var r:TRect;
begin
  r := GetDisplayRect;
  {Inc(r.Left,MarginDelta+FPen.Width);
  Inc(r.Top,MarginDelta+FPen.Width);
  Dec(r.Right,MarginDelta+FPen.Width);
  Dec(r.Bottom,MarginDelta+FPen.Width);}
  Result:=CreateRectRgnIndirect(r);
end;

procedure TPlanObject.SetReadOnly(const Value : Boolean);
begin
  FReadOnly:=Value;
  if Value then if Assigned(FOwner.ActiveObject) then if (FOwner.ActiveObject = Self) then UnSelect;
end;

procedure TPlanObject.SetProperties(Value : TStrings);
  var i : Integer;
begin
  FProperties.Assign(Value);
  for i := 1 to FProperties.Count do FProperties[i-1] := LTrim(FProperties[i-1]);
end;

{$IFDEF USE_RX}
procedure TPlanObject.BlinkTimer(Sender: TObject);
begin
  FBlinkIndex:=(FBlinkIndex xor 1);
  case FBlinkType of
    btNone  : Exit;
    btPen   : if (FBlinkIndex = 1) then FPen.Assign(FBlinkPen) else FPen.Assign(FSavePen);
    btBrush : if (FBlinkIndex = 1) then FBrush.Assign(FBlinkBrush) else FBrush.Assign(FSaveBrush);
  end;
  if ((FLayer <> FOwner.CurrentLayer) or ((FBlinkType = btBrush) and (FBrush.Style = bsClear)) or
      (FBlinkType = btPen)) then NeedRepaint else Draw(FOwner.Canvas);
end;
{$ENDIF}

procedure TPlanObject.SetPen(Value:TPen);
begin
  FPen.Assign(Value);
  FSelectColor:=-1;
  NeedRepaint;
end;

procedure TPlanObject.SetBrush(Value:TBrush);
begin
  FBrush.Assign(Value);
  NeedRepaint;
end;

procedure TPlanObject.SetLayer(Value : TPlanLayer);
  var i:Integer;
begin
  if ((FLayer=Value) or (Value=nil)) then Exit;
  if (FOwner.ActiveObject=Self) then FOwner.ActiveObject:=nil;
  Unselect;
  FOwner.DisableRepaint;
  try
    if (FLayer<>nil) then begin
      i:=FLayer.IndexOf(Self);
      if (i>=0) then FLayer.FObjectList.Delete(i);
    end;
    FLayer:=Value;
    FLayer.Add(Self);
  finally
    FOwner.EnableRepaint;
  end;
end;

procedure TPlanObject.SetName(Value : String);
  var i:Integer;
begin
  if (Value <> '') then for i:=1 to FLayer.Count do if ((UpperCase(FLayer.Objects[i-1].Name)=UpperCase(Value)) and (FLayer.Objects[i-1]<>Self)) then begin
    EOxygenPlanException.Create('Object named "'+Value+'" already exists');
    Exit;
  end;
  FName:=Value;
end;

function TPlanObject.GetObject(Index:Integer):TPlanObject;
begin
  Result := TPlanObject(FChildren[Index]);
end;

function TPlanObject.GetObjectsCount:Integer;
  var i:Integer;
begin
  Result := FChildren.Count;
  for i:=1 to FChildren.Count do Inc(Result,TPlanObject(FChildren[i-1]).ObjectsCount);
end;

procedure TPlanObject.SetParent(Value:TPlanObject);
begin
  if (FParent<>nil) then try FParent.FChildren.Delete(FParent.FChildren.IndexOf(Self)) except end;
  FParent:=Value;
  if (FParent<>nil) then FParent.FChildren.Add(Self);
end;

function TPlanObject.GetChildrenCount:Integer;
begin
  Result:=FChildren.Count;
end;

procedure TPlanObject.SetSelected(Value:Boolean);
begin
  if Value then Select else UnSelect;
end;

function TPlanObject.GetSelectedByColor:Boolean;
begin
  Result:=(FSelectColor<>-1);
end;

procedure TPlanObject.SetVisible(const Value:Boolean);
begin
  if (FVisible=Value) then Exit;
  FVisible:=Value;
  NeedRepaint;
end;

{$IFDEF USE_RX}
function TPlanObject.GetBlinking : Boolean;
begin
  Result:=(FBlinkTimerHandle<>0);
end;
{$ENDIF}

procedure TPlanObject.SaveCoords;
  var i:Integer;
begin
  for i:=1 to ChildrenCount do Objects[i-1].SaveCoords;
end;

procedure TPlanObject.Move(APoint:TPoint);
  var i:Integer;
begin
  for i:=1 to ChildrenCount do Objects[i-1].Move(APoint);
end;

procedure TPlanObject.Select;
begin
  FSelected:=True;
  SaveCoords;
  FOwner.SetActiveObject(Self);
  NeedRepaint;
end;

procedure TPlanObject.UnSelect;
begin
  FSelected:=False;
  NeedRepaint;
end;

procedure TPlanObject.ParentResized;
begin
end;

procedure TPlanObject.SelectByColor(const AColor:TColor);
begin
  FSelectColor:=AColor;
  NeedRepaint;
end;

procedure TPlanObject.UnSelectColor;
begin
  FSelectColor:=-1;
  NeedRepaint;
end;

{$IFDEF USE_RX}
procedure TPlanObject.StartBlink(const Interval:Integer);
begin
  StopBlink;
  FSavePen.Assign(FPen);
  FSaveBrush.Assign(FBrush);
  FBlinkTimerHandle:=FOwner.FTimerList.Add(BlinkTimer,Interval,True);
  FOwner.FTimerList.Activate;
end;

procedure TPlanObject.StopBlink;
begin
  if (FBlinkTimerHandle=0) then Exit;
  FOwner.FTimerList.Delete(FBlinkTimerHandle);
  FBlinkType:=btNone;
  FPen.Assign(FSavePen);
  FBrush.Assign(FSaveBrush);
  FBlinkTimerHandle:=0;
  NeedRepaint;
end;
{$ENDIF}

procedure TPlanObject.FillObjectProperties;
  var ms:TMemoryStream;
begin
  FProperties.Values['Name']:=FName;
  FProperties.Values['ID']:=IntToStr(FID);
  FProperties.Values['Layer']:=IntToStr(FLayer.Index);
  FProperties.Values['PenColor']:=EncodeColor(FPen.Color);
  FProperties.Values['PenStyle']:=IntToStr(Integer(FPen.Style));
  FProperties.Values['PenMode']:=IntToStr(Integer(FPen.Mode));
  FProperties.Values['PenWidth']:=IntToStr(FPen.Width);
  FProperties.Values['BrushColor']:=EncodeColor(FBrush.Color);
  FProperties.Values['BrushStyle']:=IntToStr(Integer(FBrush.Style));
  FProperties.Values['Cursor']:=IntToStr(Integer(FCursor));
  if (FBrush.Bitmap<>nil) then begin
    ms:=TMemoryStream.Create;
    try
      FBrush.Bitmap.SaveToStream(ms);
      Properties.Values['BrushBitmap']:=Stream2String(ms);
    except end;
    ms.Free;
  end;
  FProperties.Values['ReadOnly'] := IntToStr(Integer(FReadOnly));
end;

procedure TPlanObject.AddObjectDescr(const SpaceNum:Integer; sl:TStrings);
  var s:String;
      i:Integer;
begin
  s:=MakeStr(' ',SpaceNum);
  sl.Add(s + sStartObject + GetObjectTypeDescr);
  for i:=1 to FProperties.Count do FProperties[i-1]:=LTrim(FProperties[i-1]);

  FillObjectProperties;
  for i:=1 to FProperties.Count do sl.Add(s+'  '+LTrim(FProperties[i-1]));
  //sl.AddStrings(FProperties);
  for i:=1 to FChildren.Count do TPlanObject(FChildren[i-1]).AddObjectDescr(SpaceNum+2,sl);
  sl.Add(s + sEndObject);
end;

procedure TPlanObject.UpdateObjectProperties;
  var i  : Integer;
      s  : String;
      ms : TMemoryStream;
      b  : TBitmap;
begin
  for i:=1 to FProperties.Count do FProperties[i-1] := LTrim(StrF.ReplaceText(FProperties[i-1],' = ','='));
  FName := FProperties.Values['Name'];
  s:=FProperties.Values['ID'];
  if (s<>'') then try FID:=StrToInt(s) except end;
  if (FLayer<>nil) then begin
    i:=FLayer.IndexOf(Self);
    if (i>=0) then FLayer.FObjectList.Delete(i);
  end;
  s := FProperties.Values['Layer'];
  if (s <> '') then try FLayer := FOwner.Layers[StrToInt(s)] except FLayer := FOwner.Layers[0] end;
  FLayer.Add(Self);
  s := FProperties.Values['PenColor'];
  if (s <> '') then try FPen.Color := ReadColor(s) except end;
  s := FProperties.Values['PenStyle'];
  if (s <> '') then try FPen.Style := TPenStyle(StrToInt(s)) except end;
  s := FProperties.Values['PenMode'];
  if (s <> '') then try FPen.Mode := TPenMode(StrToInt(s)) except end;
  s:=FProperties.Values['PenWidth'];
  if (s <> '') then try FPen.Width := StrToInt(s) except end;
  s := FProperties.Values['BrushColor'];
  if (s <> '') then try FBrush.Color := ReadColor(s) except end;
  s := FProperties.Values['BrushStyle'];
  if (s <> '') then try FBrush.Style := TBrushStyle(StrToInt(s)) except end else FBrush.Style := bsClear;
  s := FProperties.Values['Cursor'];
  if (s <> '') then try FCursor := TCursor(StrToInt(s)) except end else FCursor := crDefault;
  s := FProperties.Values['BrushBitmap'];
  if (s <> '') then begin
    ms := TMemoryStream.Create;
    b := TBitmap.Create;
    try
      String2Stream(s,ms);
      b.LoadFromStream(ms);
      FBrush.Bitmap := b;
    except end;
    ms.Free;
  end;
  s := FProperties.Values['ReadOnly'];
  if (s <> '') then try ReadOnly := (StrToInt(s) <> 0) except end;
end;

function TPlanObject.ObjectFromCoord(X,Y:Integer; var MarginType : TMarginType):TPlanObject;
  var i:Integer;
      po:TPlanObject;
      mt : TMarginType;
begin
  Result:=nil;
  MarginType := mtNone;
  if not Visible then Exit;
  for i:=FChildren.Count downto 1 do begin
    if Objects[i-1].HasMarginPoint(X,Y,mt) then begin
      Result:=Objects[i-1];
      MarginType:=mt;
      Exit;
    end
    else if Objects[i-1].HasInternalPoint(X,Y) then begin
      Result:=Objects[i-1];
      MarginType := mtNone;
      po:=Objects[i-1].ObjectFromCoord(X,Y,mt);
      if (po<>nil) then begin
        Result:=po;
        MarginType := mt;
      end;
      Exit;
    end;
  end;
end;

function TPlanObject.HasInternalProperty(const PropertyName : String) : Boolean;
  var s : String;
begin
  s := UpperCase(PropertyName);
  Result := (s = 'NAME') or (s = 'LAYER') or (s = 'ID') or (s = 'PENCOLOR') or (s = 'PENWIDTH') or (s = 'PENSTYLE') or
            (s = 'BRUSHCOLOR') or (s = 'BRUSHSTYLE') or (s = 'BRUSHBITMAP') or (s = 'READONLY') or (s = 'CURSOR') or
            (s = 'PENMODE');
end;

procedure TPlanObject.Free;
  var i:Integer;
begin
  FVisible:=False;
  if not (csDestroying in FOwner.ComponentState) then try NeedRepaint except end;
  FDestroying:=True;
{$IFDEF USE_RX}
  StopBlink;
{$ENDIF}
  if (FOwner.ActiveObject=Self) then FOwner.ActiveObject:=nil;
  for i:=FChildren.Count downto 1 do TPlanObject(FChildren[i-1]).Free;
  FChildren.Free;
  i:=FLayer.IndexOf(Self);
  if (i>=0) then FLayer.FObjectList.Delete(i);
  FPen.Free;
  //if (FBrush.Bitmap<>nil) then FBrush.Bitmap.Free;
  FBrush.Free;
  FProperties.Free;
{$IFDEF USE_RX}
  FBlinkPen.Free;
  FSavePen.Free;
  FBlinkBrush.Free;
  FSaveBrush.Free;
{$ENDIF}
  SetParent(nil);
  inherited Free;
end;

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//   TPlanLine
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
constructor TPlanLine.Create(Owner:TOxygenPlan);
begin
  inherited Create(Owner);
  Fx1:=0;
  Fy1:=0;
  Fx2:=0;
  Fy2:=0;
end;

function TPlanLine.Clone(Parent : TPlanObject) : TPlanObject;
begin
  Result := TPlanLine.Create(Owner);
  FillObjectProperties;
  Result.Parent := Parent;
  Result.Properties.AddStrings(FProperties);
  Result.Properties.Values['Name'] := '';
  Result.Properties.Values['ID'] := '0';
  Result.UpdateObjectProperties;
  with (Result as TPlanLine) do begin
    X1 := X1 + 4;
    Y1 := Y1 + 4;
    X2 := X2 + 4;
    Y2 := Y2 + 4;
  end;
end;

procedure TPlanLine.SetX1(Value:Integer);
begin
  if (Fx1=Value) then Exit;
  NeedRepaint;
  Fx1:=Value;
  NeedRepaint;
end;

procedure TPlanLine.SetX2(Value:Integer);
begin
  if (Fx2=Value) then Exit;
  NeedRepaint;
  Fx2:=Value;
  NeedRepaint;
end;

procedure TPlanLine.SetY1(Value:Integer);
begin
  if (Fy1=Value) then Exit;
  NeedRepaint;
  Fy1:=Value;
  NeedRepaint;
end;

procedure TPlanLine.SetY2(Value:Integer);
begin
  if (Fy2=Value) then Exit;
  NeedRepaint;
  Fy2:=Value;
  NeedRepaint;
end;

function TPlanLine.GetScrPoint1:TPoint;
begin
  Result:=Point(Trunc(Fx1*(FOwner.Width/FOwner.ScaleX)),Trunc(Fy1*(FOwner.Height/FOwner.ScaleY)));
end;

function TPlanLine.GetScrPoint2:TPoint;
begin
  Result:=Point(Trunc(Fx2*(FOwner.Width/FOwner.ScaleX)),Trunc(Fy2*(FOwner.Height/FOwner.ScaleY)));
end;

procedure TPlanLine.SetScrPoint1(p:TPoint);
  var sp:TPoint;
begin
  sp:=ScrPoint1;
  if (sp.X<>p.X) then Fx1:=Trunc(p.X*(FOwner.ScaleX/FOwner.Width));
  if (sp.Y<>p.Y) then Fy1:=Trunc(p.Y*(FOwner.ScaleY/FOwner.Height));
end;

procedure TPlanLine.SetScrPoint2(p:TPoint);
  var sp:TPoint;
begin
  sp:=ScrPoint2;
  if (sp.X<>p.X) then Fx2:=Trunc(p.X*(FOwner.ScaleX/FOwner.Width));
  if (sp.Y<>p.Y) then Fy2:=Trunc(p.Y*(FOwner.ScaleY/FOwner.Height));
end;

procedure TPlanLine.Draw(ACanvas:TCanvas);
  var p1,p2:TPoint;
begin
  if (FOwner.FDisableRepaintCount>0) then Exit;
  if not (FLayer.Visible and FVisible) then Exit;
  p1:=ScrPoint1;
  p2:=ScrPoint2;
  with ACanvas do begin
    Pen.Assign(Self.FPen);
    Brush.Assign(Self.FBrush);
    if (FSelectColor<>-1) then ACanvas.Pen.Color:=FSelectColor;
    PolyLine([p1,p2]);
    if (FSelected and (not (FOwner.ReadOnly or FLayer.ReadOnly or FReadOnly))) then begin
      ACanvas.Pen.Color:=FOwner.SelectColor;
      ACanvas.Pen.Width:=1;
      ACanvas.Brush.Color:=FOwner.SelectColor;
      ACanvas.Brush.Style:=bsSolid;
      Rectangle(p1.X-MarginDelta,p1.Y-MarginDelta,p1.X+MarginDelta,p1.Y+MarginDelta);
      Rectangle(p2.X-MarginDelta,p2.Y-MarginDelta,p2.X+MarginDelta,p2.Y+MarginDelta);
    end;
  end;
end;

function TPlanLine.HasInternalPoint(X,Y:Integer):Boolean;
begin
  Result:=PointOnLine(Point(X,Y), ScrPoint1, ScrPoint2, Pen.Width);
end;

function TPlanLine.HasInternalProperty(const PropertyName : String) : Boolean;
  var s : String;
begin
  Result := inherited HasInternalProperty(PropertyName);
  if Result then Exit;
  s := UpperCase(PropertyName);
  Result := (s = 'X1') or (s = 'X2') or (s = 'Y1') or (s = 'Y2');
end;

function TPlanLine.HasMarginPoint(X,Y:Integer; var MarginType:TMarginType):Boolean;
  var p1,p2:TPoint;
begin
  MarginType:=mtNone;
  p1:=ScrPoint1;
  p2:=ScrPoint2;
  Result:=PointInRect(Point(X,Y),Rect(p1.X-MarginDelta,p1.Y-MarginDelta,p1.X+MarginDelta,p1.Y+MarginDelta));
  if Result then begin
    if (p1.X<p2.X) then MarginType:=mtLeft else MarginType:=mtTop;
    Exit;
  end;
  Result:=PointInRect(Point(X,Y),Rect(p2.X-MarginDelta,p2.Y-MarginDelta,p2.X+MarginDelta,p2.Y+MarginDelta));
  if Result then if (p1.X<p2.X) then MarginType:=mtRight else MarginType:=mtBottom;
end;

procedure TPlanLine.Move(APoint:TPoint);
  var rgn1,rgn2:HRGN;
      r:TRect;
      clr:HRGN;
      p1,p2:TPoint;
begin
  r := GetDisplayRect;
  Dec(r.Left,MarginDelta + Pen.Width);
  Dec(r.Top,MarginDelta + Pen.Width);
  Inc(r.Right,MarginDelta + Pen.Width);
  Inc(r.Bottom,MarginDelta + Pen.Width);
  rgn1:=CreateRectRgnIndirect(r);

  p1:=Point(ScrSave1.X+(APoint.X-FOwner.FStartCoord.X),ScrSave1.Y+(APoint.Y-FOwner.FStartCoord.Y));
  p2:=Point(ScrSave2.X+(APoint.X-FOwner.FStartCoord.X),ScrSave2.Y+(APoint.Y-FOwner.FStartCoord.Y));
  if (FParent <> nil) and (poChildrenClipping in FOwner.Options) then begin
    clr := FParent.GetClippingRgn;
    if not (PtInRegion(clr,p1.X,p1.Y) or PtInRegion(clr,p2.X,p2.Y)) then begin
      DeleteObject(rgn1);
      DeleteObject(clr);
      Exit;
    end;
    DeleteObject(clr);
  end;
  ScrPoint1:=p1;
  ScrPoint2:=p2;

  r:=GetDisplayRect;
  Inc(r.Left,MarginDelta + Pen.Width);
  Inc(r.Top,MarginDelta + Pen.Width);
  Dec(r.Right,MarginDelta + Pen.Width);
  Dec(r.Bottom,MarginDelta + Pen.Width);
  rgn2:=CreateRectRgnIndirect(r);
  if (CombineRgn(rgn1,rgn1,rgn2,RGN_DIFF)<>NULLREGION) then InvalidateRgn(FOwner.Handle,rgn1,False);
  DeleteObject(rgn1);
  DeleteObject(rgn2);
  NeedRepaint;

  inherited Move(APoint);
end;

procedure TPlanLine.Resize(APoint:TPoint);
  var rgn1,rgn2:HRGN;
      r:TRect;
begin
  if not FCanResize then Exit;
  r:=GetDisplayRect;
  Dec(r.Left,MarginDelta + Pen.Width);
  Dec(r.Top,MarginDelta + Pen.Width);
  Inc(r.Right,MarginDelta + Pen.Width);
  Inc(r.Bottom,MarginDelta + Pen.Width);
  rgn1:=CreateRectRgnIndirect(r);

  if (FOwner.FActiveMarginType in [mtLeft,mtTop]) then ScrPoint1:=APoint else ScrPoint2:=APoint;

  r:=GetDisplayRect;
  Inc(r.Left,MarginDelta + Pen.Width);
  Inc(r.Top,MarginDelta + Pen.Width);
  Dec(r.Right,MarginDelta + Pen.Width);
  Dec(r.Bottom,MarginDelta + Pen.Width);
  rgn2:=CreateRectRgnIndirect(r);
  if (CombineRgn(rgn1,rgn1,rgn2,RGN_DIFF)<>NULLREGION) then InvalidateRgn(FOwner.Handle,rgn1,False);
  DeleteObject(rgn1);
  DeleteObject(rgn2);
  NeedRepaint;
end;

function TPlanLine.CanBeParent : Boolean;
begin
  Result:=False;
end;

function TPlanLine.GetObjectTypeDescr : String;
begin
  Result:=ocLine;
end;

procedure TPlanLine.FillObjectProperties;
begin
  inherited FillObjectProperties;
  FProperties.Values['X1']:=IntToStr(Fx1);
  FProperties.Values['Y1']:=IntToStr(Fy1);
  FProperties.Values['X2']:=IntToStr(Fx2);
  FProperties.Values['Y2']:=IntToStr(Fy2);
end;

procedure TPlanLine.SaveCoords;
begin
  ScrSave1:=ScrPoint1;
  ScrSave2:=ScrPoint2;
  inherited SaveCoords;
end;

procedure TPlanLine.NeedRepaint;
  var p1,p2:TPoint;
      r:TRect;
begin
  if FDestroying then Exit;
  p1:=ScrPoint1;
  p2:=ScrPoint2;
  r:=NormalizeRect(p1.X,p1.Y,p2.X,p2.Y);
  r:=Rect(r.Left - MarginDelta - Pen.Width, r.Top - MarginDelta - Pen.Width,
          r.Right + MarginDelta + Pen.Width, r.Bottom + MarginDelta + Pen.Width);
  Windows.InvalidateRect(FOwner.Handle,@r,False);
end;

function TPlanLine.GetDisplayRect:TRect;
begin
  Result.TopLeft := ScrPoint1;
  Result.BottomRight := ScrPoint2;
  Result := NormalizeRect(Result.Left,Result.Top,Result.Right,Result.Bottom);
  Dec(Result.Top, Pen.Width div 2);
  Dec(Result.Left, Pen.Width div 2);
  Inc(Result.Bottom, Pen.Width div 2);
  Inc(Result.Right, Pen.Width div 2);
end;

function TPlanLine.AddPoint(const CoordX,CoordY:Integer; CallHandler:TCallHandler):Boolean;
begin
  Result:=(FOwner.FCurDrawingPointNum<2);
  if not Result then Exit;
  NeedRepaint;
  case FOwner.FCurDrawingPointNum of
    0: begin ScrPoint1:=Point(CoordX,CoordY); ScrPoint2:=Point(CoordX,CoordY); end;
    1: ScrPoint2:=Point(CoordX,CoordY);
  end;
  NeedRepaint;
end;

function TPlanLine.GetClippingRgn : HRGN;
begin
  Result:=CreateRectRgn(0,0,0,0);
end;

procedure TPlanLine.UpdateObjectProperties;
  var s:String;
begin
  inherited UpdateObjectProperties;
  s:=FProperties.Values['X1'];
  if (s<>'') then try Fx1:=StrToInt(s) except end;
  s:=FProperties.Values['Y1'];
  if (s<>'') then try Fy1:=StrToInt(s) except end;
  s:=FProperties.Values['X2'];
  if (s<>'') then try Fx2:=StrToInt(s) except end;
  s:=FProperties.Values['Y2'];
  if (s<>'') then try Fy2:=StrToInt(s) except end;
end;

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//   TPlanBox
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
constructor TPlanBox.Create(Owner:TOxygenPlan);
begin
  inherited Create(Owner);
  Fx1:=0;
  Fy1:=0;
  Fx2:=0;
  Fy2:=0;
end;

function TPlanBox.Clone(Parent : TPlanObject) : TPlanObject;
begin
  Result := TPlanBox.Create(Owner);
  FillObjectProperties;
  Result.Parent := Parent;
  Result.Properties.AddStrings(FProperties);
  Result.Properties.Values['Name'] := '';
  Result.Properties.Values['ID'] := '0';
  Result.UpdateObjectProperties;
  with (Result as TPlanBox) do begin
    X1 := X1 + 4;
    Y1 := Y1 + 4;
    X2 := X2 + 4;
    Y2 := Y2 + 4;
  end;
end;

procedure TPlanBox.SetX1(Value:Integer);
begin
  if (Fx1=Value) then Exit;
  NeedRepaint;
  Fx1:=Value;
  NeedRepaint;
end;

procedure TPlanBox.SetX2(Value:Integer);
begin
  if (Fx2=Value) then Exit;
  NeedRepaint;
  Fx2:=Value;
  NeedRepaint;
end;

procedure TPlanBox.SetY1(Value:Integer);
begin
  if (Fy1=Value) then Exit;
  NeedRepaint;
  Fy1:=Value;
  NeedRepaint;
end;

procedure TPlanBox.SetY2(Value:Integer);
begin
  if (Fy2=Value) then Exit;
  NeedRepaint;
  Fy2:=Value;
  NeedRepaint;
end;

function TPlanBox.GetScrPoint1:TPoint;
begin
  Result:=Point(Trunc(Fx1*(FOwner.Width/FOwner.ScaleX)),Trunc(Fy1*(FOwner.Height/FOwner.ScaleY)));
end;

function TPlanBox.GetScrPoint2:TPoint;
begin
  Result:=Point(Trunc(Fx2*(FOwner.Width/FOwner.ScaleX)),Trunc(Fy2*(FOwner.Height/FOwner.ScaleY)));
end;

procedure TPlanBox.SetScrPoint1(p:TPoint);
  var sp:TPoint;
begin
  sp:=ScrPoint1;
  if (sp.X<>p.X) then Fx1:=Trunc(p.X*(FOwner.ScaleX/FOwner.Width));
  if (sp.Y<>p.Y) then Fy1:=Trunc(p.Y*(FOwner.ScaleY/FOwner.Height));
end;

procedure TPlanBox.SetScrPoint2(p:TPoint);
  var sp:TPoint;
begin
  sp:=ScrPoint2;
  if (sp.X<>p.X) then Fx2:=Trunc(p.X*(FOwner.ScaleX/FOwner.Width));
  if (sp.Y<>p.Y) then Fy2:=Trunc(p.Y*(FOwner.ScaleY/FOwner.Height));
end;

procedure TPlanBox.Draw(ACanvas:TCanvas);
  var p1,p2:TPoint;
begin
  if (FOwner.FDisableRepaintCount>0) then Exit;
  if not (FLayer.Visible and FVisible) then Exit;
  p1:=ScrPoint1;
  p2:=ScrPoint2;
  with ACanvas do begin
    Pen.Assign(Self.FPen);
    Brush.Assign(Self.FBrush);
    if (FSelectColor<>-1) then ACanvas.Pen.Color:=FSelectColor;
    Rectangle(p1.X,p1.Y,p2.X,p2.Y);
    if (FSelected and (not (FOwner.ReadOnly or FLayer.ReadOnly or FReadOnly))) then begin
      ACanvas.Pen.Color:=FOwner.SelectColor;
      ACanvas.Pen.Width:=1;
      ACanvas.Brush.Color:=FOwner.SelectColor;
      ACanvas.Brush.Style:=bsSolid;
      Rectangle(p1.X-MarginDelta,p1.Y-MarginDelta,p1.X+MarginDelta,p1.Y+MarginDelta);
      Rectangle(p2.X-MarginDelta,p1.Y-MarginDelta,p2.X+MarginDelta,p1.Y+MarginDelta);
      Rectangle(p2.X-MarginDelta,p2.Y-MarginDelta,p2.X+MarginDelta,p2.Y+MarginDelta);
      Rectangle(p1.X-MarginDelta,p2.Y-MarginDelta,p1.X+MarginDelta,p2.Y+MarginDelta);
    end;
  end;
  DrawChildren(ACanvas);
end;

function TPlanBox.HasInternalPoint(X,Y:Integer):Boolean;
  var p1,p2:TPoint;
begin
  p1:=ScrPoint1;
  p2:=ScrPoint2;
  Result:=PointInRect(Point(X,Y),NormalizeRect(p1.X,p1.Y,p2.X,p2.Y));
end;

function TPlanBox.HasInternalProperty(const PropertyName : String) : Boolean;
  var s : String;
begin
  Result := inherited HasInternalProperty(PropertyName);
  if Result then Exit;
  s := UpperCase(PropertyName);
  Result := (s = 'X1') or (s = 'X2') or (s = 'Y1') or (s = 'Y2');
end;

function TPlanBox.HasMarginPoint(X,Y:Integer; var MarginType:TMarginType):Boolean;
  var p1,p2:TPoint;
begin
  MarginType:=mtNone;
  p1:=ScrPoint1;
  p2:=ScrPoint2;
  Result:=PointInRect(Point(X,Y),Rect(p1.X-MarginDelta,p1.Y-MarginDelta,p1.X+MarginDelta,p1.Y+MarginDelta));
  if Result then begin
    MarginType:=mtTopLeft;
    Exit;
  end;
  Result:=PointInRect(Point(X,Y),Rect(p2.X-MarginDelta,p1.Y-MarginDelta,p2.X+MarginDelta,p1.Y+MarginDelta));
  if Result then begin
    MarginType:=mtTopRight;
    Exit;
  end;
  Result:=PointInRect(Point(X,Y),Rect(p2.X-MarginDelta,p2.Y-MarginDelta,p2.X+MarginDelta,p2.Y+MarginDelta));
  if Result then begin
    MarginType:=mtBottomRight;
    Exit;
  end;
  Result:=PointInRect(Point(X,Y),Rect(p1.X-MarginDelta,p2.Y-MarginDelta,p1.X+MarginDelta,p2.Y+MarginDelta));
  if Result then begin
    MarginType:=mtBottomLeft;
    Exit;
  end;
  Result:=PointInRect(Point(X,Y),NormalizeRect(p1.X,p1.Y-MarginDelta,p2.X,p1.Y+MarginDelta));
  if Result then begin
    MarginType:=mtTop;
    Exit;
  end;
  Result:=PointInRect(Point(X,Y),Rect(p2.X-MarginDelta,p1.Y,p2.X+MarginDelta,p2.Y));
  if Result then begin
    MarginType:=mtRight;
    Exit;
  end;
  Result:=PointInRect(Point(X,Y),NormalizeRect(p1.X,p2.Y-MarginDelta,p2.X,p2.Y+MarginDelta));
  if Result then begin
    MarginType:=mtBottom;
    Exit;
  end;
  Result:=PointInRect(Point(X,Y),Rect(p1.X-MarginDelta,p1.Y,p1.X+MarginDelta,p2.Y));
  if Result then begin
    MarginType:=mtLeft;
    Exit;
  end;
end;

procedure TPlanBox.Move(APoint:TPoint);
  var rgn1,rgn2:HRGN;
      r:TRect;
      clr:HRGN;
      p1,p2:TPoint;
begin
  r:=GetDisplayRect;
  Dec(r.Left,MarginDelta + Pen.Width);
  Dec(r.Top,MarginDelta + Pen.Width);
  Inc(r.Right,MarginDelta + Pen.Width);
  Inc(r.Bottom,MarginDelta + Pen.Width);
  rgn1:=CreateRectRgnIndirect(r);

  p1:=Point(ScrSave1.X+(APoint.X-FOwner.FStartCoord.X),ScrSave1.Y+(APoint.Y-FOwner.FStartCoord.Y));
  p2:=Point(ScrSave2.X+(APoint.X-FOwner.FStartCoord.X),ScrSave2.Y+(APoint.Y-FOwner.FStartCoord.Y));
  if (FParent <> nil) and (poChildrenClipping in FOwner.Options) then begin
    clr:=FParent.GetClippingRgn;
    if not (PtInRegion(clr,p1.X,p1.Y) or PtInRegion(clr,p2.X,p2.Y)) then begin
      DeleteObject(rgn1);
      DeleteObject(clr);
      Exit;
    end;
    DeleteObject(clr);
  end;
  ScrPoint1:=p1;
  ScrPoint2:=p2;

  r:=GetDisplayRect;
  Inc(r.Left,MarginDelta + Pen.Width);
  Inc(r.Top,MarginDelta + Pen.Width);
  Dec(r.Right,MarginDelta + Pen.Width);
  Dec(r.Bottom,MarginDelta + Pen.Width);
  rgn2:=CreateRectRgnIndirect(r);
  if (CombineRgn(rgn1,rgn1,rgn2,RGN_DIFF)<>NULLREGION) then InvalidateRgn(FOwner.Handle,rgn1,False);
  DeleteObject(rgn1);
  DeleteObject(rgn2);
  NeedRepaint;

  inherited Move(APoint);
end;

procedure TPlanBox.Resize(APoint:TPoint);
  var rgn1,rgn2:HRGN;
      r:TRect;
begin
  if not FCanResize then Exit;
  r:=GetDisplayRect;
  Dec(r.Left,MarginDelta + Pen.Width);
  Dec(r.Top,MarginDelta + Pen.Width);
  Inc(r.Right,MarginDelta + Pen.Width);
  Inc(r.Bottom,MarginDelta + Pen.Width);
  rgn1:=CreateRectRgnIndirect(r);
  case FOwner.FActiveMarginType of
    mtLeft:        ScrPoint1:=Point(APoint.X,ScrPoint1.Y);
    mtTopLeft:     ScrPoint1:=APoint;
    mtTop:         ScrPoint1:=Point(ScrPoint1.X,APoint.Y);
    mtTopRight:    begin ScrPoint1:=Point(ScrPoint1.X,APoint.Y); ScrPoint2:=Point(APoint.X,ScrPoint2.Y); end;
    mtRight:       ScrPoint2:=Point(APoint.X,ScrPoint2.Y);
    mtBottomRight: ScrPoint2:=APoint;
    mtBottom:      ScrPoint2:=Point(ScrPoint2.X,APoint.Y);
    mtBottomLeft:  begin ScrPoint1:=Point(APoint.X,ScrPoint1.Y); ScrPoint2:=Point(ScrPoint2.X,APoint.Y); end;
  end;
  r:=GetDisplayRect;
  Inc(r.Left,MarginDelta + Pen.Width);
  Inc(r.Top,MarginDelta + Pen.Width);
  Dec(r.Right,MarginDelta + Pen.Width);
  Dec(r.Bottom,MarginDelta + Pen.Width);
  rgn2:=CreateRectRgnIndirect(r);
  if (CombineRgn(rgn1,rgn1,rgn2,RGN_DIFF)<>NULLREGION) then InvalidateRgn(FOwner.Handle,rgn1,False);
  DeleteObject(rgn1);
  DeleteObject(rgn2);
  NeedRepaint;
end;

function TPlanBox.CanBeParent : Boolean;
begin
  Result:=True;
end;

function TPlanBox.GetObjectTypeDescr : String;
begin
  Result:=ocBox;
end;

procedure TPlanBox.FillObjectProperties;
begin
  inherited FillObjectProperties;
  FProperties.Values['X1']:=IntToStr(Fx1);
  FProperties.Values['Y1']:=IntToStr(Fy1);
  FProperties.Values['X2']:=IntToStr(Fx2);
  FProperties.Values['Y2']:=IntToStr(Fy2);
end;

procedure TPlanBox.SaveCoords;
begin
  ScrSave1:=ScrPoint1;
  ScrSave2:=ScrPoint2;
  inherited SaveCoords;
end;

procedure TPlanBox.NeedRepaint;
  var p1,p2:TPoint;
      r:TRect;
begin
  if FDestroying then Exit;
  p1:=ScrPoint1;
  p2:=ScrPoint2;
  r:=NormalizeRect(p1.X,p1.Y,p2.X,p2.Y);
  r:=Rect(r.Left - MarginDelta - Pen.Width, r.Top - MarginDelta - Pen.Width,
          r.Right + MarginDelta + Pen.Width, r.Bottom + MarginDelta + Pen.Width);
  Windows.InvalidateRect(FOwner.Handle,@r,False);
end;

function TPlanBox.GetDisplayRect:TRect;
begin
  Result.TopLeft:=ScrPoint1;
  Result.BottomRight:=ScrPoint2;
  Result := NormalizeRect(Result.Left,Result.Top,Result.Right,Result.Bottom);
  Dec(Result.Top, Pen.Width div 2);
  Dec(Result.Left, Pen.Width div 2);
  Inc(Result.Bottom, Pen.Width div 2);
  Inc(Result.Right, Pen.Width div 2);
end;

function TPlanBox.AddPoint(const CoordX,CoordY:Integer; CallHandler:TCallHandler):Boolean;
begin
  Result:=(FOwner.FCurDrawingPointNum<2);
  if not Result then Exit;
  NeedRepaint;
  case FOwner.FCurDrawingPointNum of
    0: begin ScrPoint1:=Point(CoordX,CoordY); ScrPoint2:=Point(CoordX,CoordY); end;
    1: ScrPoint2:=Point(CoordX,CoordY);
  end;
  NeedRepaint;
end;

procedure TPlanBox.UpdateObjectProperties;
  var s:String;
begin
  inherited UpdateObjectProperties;
  s:=FProperties.Values['X1'];
  if (s<>'') then try Fx1:=StrToInt(s) except end;
  s:=FProperties.Values['Y1'];
  if (s<>'') then try Fy1:=StrToInt(s) except end;
  s:=FProperties.Values['X2'];
  if (s<>'') then try Fx2:=StrToInt(s) except end;
  s:=FProperties.Values['Y2'];
  if (s<>'') then try Fy2:=StrToInt(s) except end;
end;

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//   TPlanCircle
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
constructor TPlanCircle.Create(Owner:TOxygenPlan);
begin
  inherited Create(Owner);
  Fx:=0;
  Fy:=0;
  FRadius:=0;
end;

function TPlanCircle.Clone(Parent : TPlanObject) : TPlanObject;
begin
  Result := TPlanCircle.Create(Owner);
  FillObjectProperties;
  Result.Parent := Parent;
  Result.Properties.AddStrings(FProperties);
  Result.Properties.Values['Name'] := '';
  Result.Properties.Values['ID'] := '0';
  Result.UpdateObjectProperties;
  with (Result as TPlanCircle) do begin
    X := X + 4;
    Y := Y + 4;
  end;
end;

procedure TPlanCircle.SetX(Value:Integer);
begin
  if (Fx=Value) then Exit;
  NeedRepaint;
  Fx:=Value;
  NeedRepaint;
end;

procedure TPlanCircle.SetY(Value:Integer);
begin
  if (Fy=Value) then Exit;
  NeedRepaint;
  Fy:=Value;
  NeedRepaint;
end;

procedure TPlanCircle.SetRadius(Value:Integer);
begin
  if (FRadius=Value) then Exit;
  NeedRepaint;
  FRadius:=Value;
  NeedRepaint;
end;

function TPlanCircle.GetScrCenter:TPoint;
begin
  Result:=Point(Trunc(Fx*(FOwner.Width/FOwner.ScaleX)),Trunc(Fy*(FOwner.Height/FOwner.ScaleY)));
end;

function TPlanCircle.GetScrRadius:Integer;
begin
  Result:=Trunc(FRadius*(FOwner.Width/FOwner.ScaleX))
end;

procedure TPlanCircle.SetScrCenter(p:TPoint);
  var sc:TPoint;
begin
  sc:=ScrCenter;
  if (sc.X<>p.X) then Fx:=Trunc(p.X*(FOwner.ScaleX/FOwner.Width));
  if (sc.Y<>p.Y) then Fy:=Trunc(p.Y*(FOwner.ScaleY/FOwner.Height));
end;

procedure TPlanCircle.SetScrRadius(r:Integer);
begin
  FRadius:=Trunc(r*(FOwner.ScaleX/FOwner.Width));
end;

procedure TPlanCircle.Draw(ACanvas:TCanvas);
  var p:TPoint;
      r:Integer;
begin
  if (FOwner.FDisableRepaintCount>0) then Exit;
  if not (FLayer.Visible and FVisible) then Exit;
  p:=ScrCenter;
  r:=ScrRadius;
  with ACanvas do begin
    Pen.Assign(Self.FPen);
    Brush.Assign(Self.FBrush);
    if (FSelectColor<>-1) then ACanvas.Pen.Color:=FSelectColor;
    Ellipse(p.X-r,p.Y-r,p.X+r,p.Y+r);
    if (FSelected and (not (FOwner.ReadOnly or FLayer.ReadOnly or FReadOnly))) then begin
      ACanvas.Pen.Color:=FOwner.SelectColor;
      ACanvas.Brush.Color:=FOwner.SelectColor;
      ACanvas.Brush.Style:=bsSolid;
      ACanvas.Pen.Width:=1;
      Rectangle(p.X-MarginDelta,p.Y-r-MarginDelta,p.X+MarginDelta,p.Y-r+MarginDelta);
      Rectangle(p.X+r-MarginDelta,p.Y-MarginDelta,p.X+r+MarginDelta,p.Y+MarginDelta);
      Rectangle(p.X-MarginDelta,p.Y+r-MarginDelta,p.X+MarginDelta,p.Y+r+MarginDelta);
      Rectangle(p.X-r-MarginDelta,p.Y-MarginDelta,p.X-r+MarginDelta,p.Y+MarginDelta);
    end;
  end;
  DrawChildren(ACanvas);
end;

function TPlanCircle.HasInternalPoint(X,Y:Integer):Boolean;
  var d:Integer;
begin
  d:=Distance(ScrCenter,Point(X,Y));
  Result:=(d<=ScrRadius);
end;

function TPlanCircle.HasInternalProperty(const PropertyName : String) : Boolean;
  var s : String;
begin
  Result := inherited HasInternalProperty(PropertyName);
  if Result then Exit;
  s := UpperCase(PropertyName);
  Result := (s = 'X') or (s = 'Y') or (s = 'RADIUS');
end;

function TPlanCircle.HasMarginPoint(X,Y:Integer; var MarginType:TMarginType):Boolean;
  var p:TPoint;
      r,d:Integer;
begin
  MarginType:=mtNone;
  p:=ScrCenter;
  r:=ScrRadius;
  d:=Distance(p,Point(X,Y));
  Result:=((d>r-MarginDelta) and (d<r+MarginDelta));
  if not Result then Exit;
  if PointInRect(Point(X,Y),Rect(p.X-MarginDelta,p.Y-r-MarginDelta,p.X+MarginDelta,p.Y-r+MarginDelta)) then begin
    MarginType:=mtTop;
    Exit;
  end;
  if PointInRect(Point(X,Y),Rect(p.X+r-MarginDelta,p.Y-MarginDelta,p.X+r+MarginDelta,p.Y+MarginDelta)) then begin
    MarginType:=mtRight;
    Exit;
  end;
  if PointInRect(Point(X,Y),Rect(p.X-MarginDelta,p.Y+r-MarginDelta,p.X+MarginDelta,p.Y+r+MarginDelta)) then begin
    MarginType:=mtBottom;
    Exit;
  end;
  if PointInRect(Point(X,Y),Rect(p.X-r-MarginDelta,p.Y-MarginDelta,p.X+r+MarginDelta,p.Y+MarginDelta)) then begin
    MarginType:=mtLeft;
    Exit;
  end;
  if (X<p.X) and (Y<p.Y) then begin
    MarginType:=mtTopLeft;
    Exit;
  end;
  if (X>p.X) and (Y<p.Y) then begin
    MarginType:=mtTopRight;
    Exit;
  end;
  if (X<p.X) and (Y>p.Y) then begin
    MarginType:=mtBottomLeft;
    Exit;
  end;
  if (X>p.X) and (Y>p.Y) then begin
    MarginType:=mtBottomRight;
    Exit;
  end;
end;

procedure TPlanCircle.Move(APoint:TPoint);
  var rgn1,rgn2:HRGN;
      r:TRect;
      clr:HRGN;
      p:TPoint;
begin
  r:=GetDisplayRect;
  Dec(r.Left,MarginDelta + Pen.Width);
  Dec(r.Top,MarginDelta + Pen.Width);
  Inc(r.Right,MarginDelta + Pen.Width);
  Inc(r.Bottom,MarginDelta + Pen.Width);
  rgn1:=CreateEllipticRgnIndirect(r);

  p:=Point(ScrSaveCenter.X+(APoint.X-FOwner.FStartCoord.X),ScrSaveCenter.Y+(APoint.Y-FOwner.FStartCoord.Y));
  if (FParent <> nil) and (poChildrenClipping in FOwner.Options) then begin
    clr:=FParent.GetClippingRgn;
    if not PtInRegion(clr,p.X,p.Y) then begin
      DeleteObject(rgn1);
      DeleteObject(clr);
      Exit;
    end;
    DeleteObject(clr);
  end;
  ScrCenter:=p;

  r:=GetDisplayRect;
  Inc(r.Left,MarginDelta + Pen.Width);
  Inc(r.Top,MarginDelta + Pen.Width);
  Dec(r.Right,MarginDelta + Pen.Width);
  Dec(r.Bottom,MarginDelta + Pen.Width);
  rgn2:=CreateEllipticRgnIndirect(r);
  if (CombineRgn(rgn1,rgn1,rgn2,RGN_DIFF)<>NULLREGION) then InvalidateRgn(FOwner.Handle,rgn1,False);
  DeleteObject(rgn1);
  DeleteObject(rgn2);
  NeedRepaint;

  inherited Move(APoint);
end;

procedure TPlanCircle.Resize(APoint:TPoint);
  var rgn1,rgn2:HRGN;
      r:TRect;
begin
  if not FCanResize then Exit;
  r:=GetDisplayRect;
  Dec(r.Left,MarginDelta + Pen.Width);
  Dec(r.Top,MarginDelta + Pen.Width);
  Inc(r.Right,MarginDelta + Pen.Width);
  Inc(r.Bottom,MarginDelta + Pen.Width);
  rgn1:=CreateEllipticRgnIndirect(r);

  if (FOwner.FActiveMarginType<>mtNone) then ScrRadius:=Distance(ScrCenter,APoint);

  r:=GetDisplayRect;
  Inc(r.Left,MarginDelta + Pen.Width);
  Inc(r.Top,MarginDelta + Pen.Width);
  Dec(r.Right,MarginDelta + Pen.Width);
  Dec(r.Bottom,MarginDelta + Pen.Width);
  rgn2:=CreateEllipticRgnIndirect(r);
  if (CombineRgn(rgn1,rgn1,rgn2,RGN_DIFF)<>NULLREGION) then InvalidateRgn(FOwner.Handle,rgn1,False);
  DeleteObject(rgn1);
  DeleteObject(rgn2);
  NeedRepaint;
end;

function TPlanCircle.CanBeParent : Boolean;
begin
  Result:=True;
end;

function TPlanCircle.GetObjectTypeDescr : String;
begin
  Result:=ocCircle;
end;

procedure TPlanCircle.FillObjectProperties;
begin
  inherited FillObjectProperties;
  Properties.Values['X']:=IntToStr(Fx);
  Properties.Values['Y']:=IntToStr(Fy);
  Properties.Values['Radius']:=IntToStr(FRadius);
end;

procedure TPlanCircle.SaveCoords;
begin
  ScrSaveCenter:=ScrCenter;
  ScrSaveRadius:=ScrRadius;
  inherited SaveCoords;
end;

procedure TPlanCircle.NeedRepaint;
  var p:TPoint;
      rad:Integer;
      rgn:HRgn;
begin
  if FDestroying then Exit;
  p:=ScrCenter;
  rad:=Abs(ScrRadius)+FPen.WIdth;
  rgn:=CreateEllipticRgn(p.X - rad - MarginDelta - Pen.Width, p.Y - rad - MarginDelta - Pen.Width,
                         p.X + rad + MarginDelta + Pen.Width, p.Y + rad + MarginDelta + Pen.Width);
  Windows.InvalidateRgn(FOwner.Handle,rgn,False);
end;

function TPlanCircle.GetDisplayRect:TRect;
  var p:TPoint;
      rad:Integer;
begin
  p:=ScrCenter;
  rad:=Abs(ScrRadius);
  Result:=Rect(p.X-rad,p.Y-rad,p.X+rad,p.Y+rad);
  Dec(Result.Top, Pen.Width div 2);
  Dec(Result.Left, Pen.Width div 2);
  Inc(Result.Bottom, Pen.Width div 2);
  Inc(Result.Right, Pen.Width div 2);
end;

function TPlanCircle.AddPoint(const CoordX,CoordY:Integer; CallHandler:TCallHandler):Boolean;
begin
  Result := (FOwner.FCurDrawingPointNum < 2);
  if not Result then Exit;
  NeedRepaint;
  case FOwner.FCurDrawingPointNum of
    0: begin ScrCenter := Point(CoordX,CoordY); ScrRadius := 1; end;
    1: ScrRadius := Distance(ScrCenter,Point(CoordX,CoordY));
  end;
  NeedRepaint;
end;

function TPlanCircle.GetClippingRgn : HRGN;
  var r:TRect;
begin
  r:=GetDisplayRect;
  Inc(r.Left,MarginDelta+FPen.Width);
  Inc(r.Top,MarginDelta+FPen.Width);
  Dec(r.Right,MarginDelta+FPen.Width);
  Dec(r.Bottom,MarginDelta+FPen.Width);
  Result:=CreateEllipticRgnIndirect(r);
end;

procedure TPlanCircle.UpdateObjectProperties;
  var s:String;
begin
  inherited UpdateObjectProperties;
  s:=FProperties.Values['X'];
  if (s<>'') then try Fx:=StrToInt(s) except end;
  s:=FProperties.Values['Y'];
  if (s<>'') then try Fy:=StrToInt(s) except end;
  s:=FProperties.Values['Radius'];
  if (s<>'') then try FRadius:=StrToInt(s) except end;
end;


/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//  TPlanText
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
constructor TPlanText.Create(Owner:TOxygenPlan);
begin
  inherited Create(Owner);
  Fx1:=0;
  Fy1:=0;
  Fx2:=0;
  Fy2:=0;
  FText := '';
  FSaveText := '';
  FBlinkText := '';
  FFont:=TFont.Create;
  FSaveFont := TFont.Create;
  FBlinkFont := TFont.Create;
  FFont.Assign(FOwner.Font);
  FAlignment := taCenter;
  FPen.Mode := pmNop;
end;

function TPlanText.Clone(Parent : TPlanObject) : TPlanObject;
begin
  Result := TPlanText.Create(Owner);
  FillObjectProperties;
  Result.Parent := Parent;
  Result.Properties.AddStrings(FProperties);
  Result.Properties.Values['Name'] := '';
  Result.Properties.Values['ID'] := '0';
  Result.UpdateObjectProperties;
  with (Result as TPlanText) do begin
    X1 := X1 + 4;
    Y1 := Y1 + 4;
    X2 := X2 + 4;
    Y2 := Y2 + 4;
  end;
end;

procedure TPlanText.SetX1(Value:Integer);
begin
  if (Fx1=Value) then Exit;
  NeedRepaint;
  Fx1:=Value;
  NeedRepaint;
end;

procedure TPlanText.SetX2(Value:Integer);
begin
  if (Fx2=Value) then Exit;
  NeedRepaint;
  Fx2:=Value;
  NeedRepaint;
end;

procedure TPlanText.SetY1(Value:Integer);
begin
  if (Fy1=Value) then Exit;
  NeedRepaint;
  Fy1:=Value;
  NeedRepaint;
end;

procedure TPlanText.SetY2(Value:Integer);
begin
  if (Fy2=Value) then Exit;
  NeedRepaint;
  Fy2:=Value;
  NeedRepaint;
end;

procedure TPlanText.SetText(Value:String);
begin
  if (FText=Value) then Exit;
  NeedRepaint;
  FText:=Value;
  NeedRepaint;
end;

procedure TPlanText.SetFont(Value:TFont);
begin
  FFont.Assign(Value);
  NeedRepaint;
end;

procedure TPlanText.SetAlignment(Value:TAlignment);
begin
  if (FAlignment=Value) then Exit;
  FAlignment:=Value;
  NeedRepaint;
end;

procedure TPlanText.Free;
begin
  FVisible:=False;
  if not (csDestroying in FOwner.ComponentState) then try NeedRepaint except end;
  FDestroying:=True;
{$IFDEF USE_RX}
  StopBlink;
  FSaveFont.Free;
  FBlinkFont.Free;
{$ENDIF}
  FFont.Free;
  inherited Free;
end;

function TPlanText.GetScrPoint1:TPoint;
begin
  Result:=Point(Trunc(Fx1*(FOwner.Width/FOwner.ScaleX)),Trunc(Fy1*(FOwner.Height/FOwner.ScaleY)));
end;

function TPlanText.GetScrPoint2:TPoint;
begin
  Result:=Point(Trunc(Fx2*(FOwner.Width/FOwner.ScaleX)),Trunc(Fy2*(FOwner.Height/FOwner.ScaleY)));
end;

procedure TPlanText.SetScrPoint1(p:TPoint);
  var sp:TPoint;
begin
  sp:=ScrPoint1;
  if (sp.X<>p.X) then Fx1:=Trunc(p.X*(FOwner.ScaleX/FOwner.Width));
  if (sp.Y<>p.Y) then Fy1:=Trunc(p.Y*(FOwner.ScaleY/FOwner.Height));
end;

procedure TPlanText.SetScrPoint2(p:TPoint);
  var sp:TPoint;
begin
  sp:=ScrPoint2;
  if (sp.X<>p.X) then Fx2:=Trunc(p.X*(FOwner.ScaleX/FOwner.Width));
  if (sp.Y<>p.Y) then Fy2:=Trunc(p.Y*(FOwner.ScaleY/FOwner.Height));
end;

procedure TPlanText.Draw(ACanvas:TCanvas);
  var p1,p2 : TPoint;
      fmt   : Word;
      r     : TRect;
begin
  if (FOwner.FDisableRepaintCount>0) then Exit;
  if not (FLayer.Visible and FVisible) then Exit;
  p1:=ScrPoint1;
  p2:=ScrPoint2;
  r := NormalizeRect(p1.X,p1.Y,p2.X,p2.Y);
  Inc(r.Top,MarginDelta*2);
  Inc(r.Left,MarginDelta*2);
  Dec(r.Bottom,MarginDelta*2);
  Dec(r.Right,MarginDelta*2);
  fmt:=DT_LEFT+DT_WORDBREAK+DT_NOPREFIX;
  case FAlignment of
    taLeftJustify:  fmt:=DT_LEFT+DT_WORDBREAK+DT_NOPREFIX;
    taRightJustify: fmt:=DT_RIGHT+DT_WORDBREAK+DT_NOPREFIX;
    taCenter:       fmt:=DT_CENTER+DT_WORDBREAK+DT_NOPREFIX;
  end;
  with ACanvas do begin
    Pen.Assign(Self.FPen);
    Brush.Assign(Self.FBrush);
    Font.Assign(Self.FFont);
    with Font do Size := Trunc(Size*(FOwner.Width/FOwner.ScaleX));
    if (FSelectColor <> -1) then Font.Color := FSelectColor;
    if (FText = '') then begin
      Pen.Color := FOwner.Pen.Color;
      Pen.Width := 1;
      Pen.Style := psDash;
      Pen.Mode := pmCopy;
    end;
    Rectangle(p1.X,p1.Y,p2.X,p2.Y);
    Windows.DrawText(ACanvas.Handle,PChar(FText),-1,r,fmt);
    if (FSelected and (not (FOwner.ReadOnly or FLayer.ReadOnly or FReadOnly))) then begin
      Pen.Mode:=pmCopy;
      ACanvas.Pen.Color:=FOwner.SelectColor;
      ACanvas.Brush.Color:=FOwner.SelectColor;
      ACanvas.Brush.Style:=bsSolid;
      ACanvas.Pen.Width:=1;
      Rectangle(p1.X-MarginDelta,p1.Y-MarginDelta,p1.X+MarginDelta,p1.Y+MarginDelta);
      Rectangle(p2.X-MarginDelta,p1.Y-MarginDelta,p2.X+MarginDelta,p1.Y+MarginDelta);
      Rectangle(p2.X-MarginDelta,p2.Y-MarginDelta,p2.X+MarginDelta,p2.Y+MarginDelta);
      Rectangle(p1.X-MarginDelta,p2.Y-MarginDelta,p1.X+MarginDelta,p2.Y+MarginDelta);
    end;
  end;
end;

function TPlanText.HasInternalPoint(X,Y:Integer):Boolean;
  var p1,p2:TPoint;
begin
  p1:=ScrPoint1;
  p2:=ScrPoint2;
  Result:=PointInRect(Point(X,Y),NormalizeRect(p1.X,p1.Y,p2.X,p2.Y));
end;

function TPlanText.HasInternalProperty(const PropertyName : String) : Boolean;
  var s : String;
begin
  Result := inherited HasInternalProperty(PropertyName);
  if Result then Exit;
  s := UpperCase(PropertyName);
  Result := (s = 'X1') or (s = 'X2') or (s = 'Y1') or (s = 'Y2') or (s = 'CAPTION') or (s = 'FONTNAME') or (s = 'FONTSIZE') or
            (s = 'STYLE') or (s = 'ALIGNMENT') or (s = 'FONTCOLOR');
end;

function TPlanText.HasMarginPoint(X,Y:Integer; var MarginType:TMarginType):Boolean;
  var p1,p2:TPoint;
begin
  MarginType:=mtNone;
  p1:=ScrPoint1;
  p2:=ScrPoint2;
  Result:=PointInRect(Point(X,Y),Rect(p1.X-MarginDelta,p1.Y-MarginDelta,p1.X+MarginDelta,p1.Y+MarginDelta));
  if Result then begin
    MarginType:=mtTopLeft;
    Exit;
  end;
  Result:=PointInRect(Point(X,Y),Rect(p2.X-MarginDelta,p1.Y-MarginDelta,p2.X+MarginDelta,p1.Y+MarginDelta));
  if Result then begin
    MarginType:=mtTopRight;
    Exit;
  end;
  Result:=PointInRect(Point(X,Y),Rect(p2.X-MarginDelta,p2.Y-MarginDelta,p2.X+MarginDelta,p2.Y+MarginDelta));
  if Result then begin
    MarginType:=mtBottomRight;
    Exit;
  end;
  Result:=PointInRect(Point(X,Y),Rect(p1.X-MarginDelta,p2.Y-MarginDelta,p1.X+MarginDelta,p2.Y+MarginDelta));
  if Result then begin
    MarginType:=mtBottomLeft;
    Exit;
  end;
  Result:=PointInRect(Point(X,Y),Rect(p1.X,p1.Y-MarginDelta,p2.X,p1.Y+MarginDelta));
  if Result then begin
    MarginType:=mtTop;
    Exit;
  end;
  Result:=PointInRect(Point(X,Y),Rect(p2.X-MarginDelta,p1.Y,p2.X+MarginDelta,p2.Y));
  if Result then begin
    MarginType:=mtRight;
    Exit;
  end;
  Result:=PointInRect(Point(X,Y),Rect(p1.X,p2.Y-MarginDelta,p2.X,p2.Y+MarginDelta));
  if Result then begin
    MarginType:=mtBottom;
    Exit;
  end;
  Result:=PointInRect(Point(X,Y),Rect(p1.X-MarginDelta,p1.Y,p1.X+MarginDelta,p2.Y));
  if Result then begin
    MarginType:=mtLeft;
    Exit;
  end;
end;

procedure TPlanText.Move(APoint:TPoint);
  var rgn1,rgn2:HRGN;
      r:TRect;
      clr:HRGN;
      p1,p2:TPoint;
begin
  r:=GetDisplayRect;
  Dec(r.Left,MarginDelta + Pen.Width);
  Dec(r.Top,MarginDelta + Pen.Width);
  Inc(r.Right,MarginDelta + Pen.Width);
  Inc(r.Bottom,MarginDelta + Pen.Width);
  rgn1:=CreateRectRgnIndirect(r);

  p1:=Point(ScrSave1.X+(APoint.X-FOwner.FStartCoord.X),ScrSave1.Y+(APoint.Y-FOwner.FStartCoord.Y));
  p2:=Point(ScrSave2.X+(APoint.X-FOwner.FStartCoord.X),ScrSave2.Y+(APoint.Y-FOwner.FStartCoord.Y));
  if (FParent <> nil) and (poChildrenClipping in FOwner.Options) then begin
    clr:=FParent.GetClippingRgn;
    if not (PtInRegion(clr,p1.X,p1.Y) or PtInRegion(clr,p2.X,p2.Y)) then begin
      DeleteObject(rgn1);
      DeleteObject(clr);
      Exit;
    end;
    DeleteObject(clr);
  end;
  ScrPoint1:=p1;
  ScrPoint2:=p2;

  r:=GetDisplayRect;
  Inc(r.Left,MarginDelta + Pen.Width);
  Inc(r.Top,MarginDelta + Pen.Width);
  Dec(r.Right,MarginDelta + Pen.Width);
  Dec(r.Bottom,MarginDelta + Pen.Width);
  rgn2:=CreateRectRgnIndirect(r);
  if (CombineRgn(rgn1,rgn1,rgn2,RGN_DIFF)<>NULLREGION) then InvalidateRgn(FOwner.Handle,rgn1,False);
  DeleteObject(rgn1);
  DeleteObject(rgn2);
  NeedRepaint;

  inherited Move(APoint);
end;

procedure TPlanText.Resize(APoint:TPoint);
  var rgn1,rgn2:HRGN;
      r:TRect;
begin
  if not FCanResize then Exit;
  r:=GetDisplayRect;
  Dec(r.Left,MarginDelta + Pen.Width);
  Dec(r.Top,MarginDelta + Pen.Width);
  Inc(r.Right,MarginDelta + Pen.Width);
  Inc(r.Bottom,MarginDelta + Pen.Width);
  rgn1:=CreateRectRgnIndirect(r);

  case FOwner.FActiveMarginType of
    mtLeft:        ScrPoint1:=Point(APoint.X,ScrPoint1.Y);
    mtTopLeft:     ScrPoint1:=APoint;
    mtTop:         ScrPoint1:=Point(ScrPoint1.X,APoint.Y);
    mtTopRight:    begin ScrPoint1:=Point(ScrPoint1.X,APoint.Y); ScrPoint2:=Point(APoint.X,ScrPoint2.Y); end;
    mtRight:       ScrPoint2:=Point(APoint.X,ScrPoint2.Y);
    mtBottomRight: ScrPoint2:=APoint;
    mtBottom:      ScrPoint2:=Point(ScrPoint2.X,APoint.Y);
    mtBottomLeft:  begin ScrPoint1:=Point(APoint.X,ScrPoint1.Y); ScrPoint2:=Point(ScrPoint2.X,APoint.Y); end;
  end;

  r:=GetDisplayRect;
  Inc(r.Left,MarginDelta + Pen.Width);
  Inc(r.Top,MarginDelta + Pen.Width);
  Dec(r.Right,MarginDelta + Pen.Width);
  Dec(r.Bottom,MarginDelta + Pen.Width);
  rgn2:=CreateRectRgnIndirect(r);
  if (CombineRgn(rgn1,rgn1,rgn2,RGN_DIFF)<>NULLREGION) then InvalidateRgn(FOwner.Handle,rgn1,False);
  DeleteObject(rgn1);
  DeleteObject(rgn2);
  NeedRepaint;
end;

{$IFDEF USE_RX}
procedure TPlanText.StartBlink(const Interval:Integer);
begin
  StopBlink;
  FSaveFont.Assign(FFont);
  FSaveText:=FText;
  inherited StartBlink(Interval);
end;

procedure TPlanText.StopBlink;
begin
  if (FBlinkTimerHandle=0) then Exit;
  inherited StopBlink;
  FFont.Assign(FSaveFont);
  FText:=FSaveText;
  NeedRepaint;
end;
{$ENDIF}

function TPlanText.CanBeParent : Boolean;
begin
  Result:=False;
end;

function TPlanText.GetObjectTypeDescr : String;
begin
  Result:=ocText;
end;

procedure TPlanText.FillObjectProperties;
  var s:String;
begin
  inherited FillObjectProperties;
  case FAlignment of
    taLeftJustify:  s:='Left';
    taRightJustify: s:='Right';
    taCenter:       s:='Center';
  end;
  FProperties.Values['X1']:=IntToStr(Fx1);
  FProperties.Values['Y1']:=IntToStr(Fy1);
  FProperties.Values['X2']:=IntToStr(Fx2);
  FProperties.Values['Y2']:=IntToStr(Fy2);
  FProperties.Values['Caption']:=FText;
  FProperties.Values['FontName']:=FFont.Name;
  FProperties.Values['FontSize']:=IntToStr(FFont.Size);
  FProperties.Values['FontColor']:=EncodeColor(FFont.Color);
  FProperties.Values['Style']:=GetStyleString(FFont);
  FProperties.Values['Alignment']:=s;
end;

procedure TPlanText.SaveCoords;
begin
  ScrSave1:=ScrPoint1;
  ScrSave2:=ScrPoint2;
  inherited SaveCoords;
end;

procedure TPlanText.NeedRepaint;
  var p1,p2:TPoint;
      r:TRect;
begin
  if FDestroying then Exit;
  p1:=ScrPoint1;
  p2:=ScrPoint2;
  r:=NormalizeRect(p1.X,p1.Y,p2.X,p2.Y);
  r:=Rect(r.Left - MarginDelta - Pen.Width, r.Top - MarginDelta - Pen.Width,
          r.Right + MarginDelta + Pen.Width, r.Bottom + MarginDelta + Pen.Width);
  Windows.InvalidateRect(FOwner.Handle,@r,False);
end;

function TPlanText.GetDisplayRect:TRect;
begin
  Result.TopLeft:=ScrPoint1;
  Result.BottomRight:=ScrPoint2;
  Result := NormalizeRect(Result.Left,Result.Top,Result.Right,Result.Bottom);
  Dec(Result.Top, Pen.Width div 2);
  Dec(Result.Left, Pen.Width div 2);
  Inc(Result.Bottom, Pen.Width div 2);
  Inc(Result.Right, Pen.Width div 2);
end;

function TPlanText.AddPoint(const CoordX,CoordY:Integer; CallHandler:TCallHandler):Boolean;
begin
  Result:=(FOwner.FCurDrawingPointNum<2);
  if not Result then Exit;
  NeedRepaint;
  case FOwner.FCurDrawingPointNum of
    0: begin ScrPoint1:=Point(CoordX,CoordY); ScrPoint2:=Point(CoordX,CoordY); end;
    1: ScrPoint2:=Point(CoordX,CoordY);
  end;
  NeedRepaint;
end;

{$IFDEF USE_RX}
procedure TPlanText.BlinkTimer(Sender: TObject);
begin
  FBlinkIndex:=(FBlinkIndex xor 1);
  case FBlinkType of
    btNone  : Exit;
    btPen   : if (FBlinkIndex=1) then FPen.Assign(FBlinkPen) else FPen.Assign(FSavePen);
    btBrush : if (FBlinkIndex=1) then FBrush.Assign(FBlinkBrush) else FBrush.Assign(FSaveBrush);
    btFont  : if (FBlinkIndex=1) then FFont.Assign(FBlinkFont) else FFont.Assign(FSaveFont);
    btText  : if (FBlinkIndex=1) then FText:=FBlinkText else FText:=FSaveText;
  end;
  if ((FLayer <> FOwner.CurrentLayer) or (FBrush.Style = bsClear) or
      (FBlinkType = btPen)) then NeedRepaint else Draw(FOwner.Canvas);
end;
{$ENDIF}

procedure TPlanText.UpdateObjectProperties;
  var s:String;
begin
  inherited UpdateObjectProperties;
  s:=FProperties.Values['X1'];
  if (s<>'') then try Fx1:=StrToInt(s) except end;
  s:=FProperties.Values['Y1'];
  if (s<>'') then try Fy1:=StrToInt(s) except end;
  s:=FProperties.Values['X2'];
  if (s<>'') then try Fx2:=StrToInt(s) except end;
  s:=FProperties.Values['Y2'];
  if (s<>'') then try Fy2:=StrToInt(s) except end;
  FText:=FProperties.Values['Caption'];
  FFont.Name:=FProperties.Values['FontName'];
  FFont.Style:=[];
  s:=FProperties.Values['Style'];
  if (Pos('I',s)>0) then with FFont do Style:=Style+[fsItalic];
  if (Pos('B',s)>0) then with FFont do Style:=Style+[fsBold];
  if (Pos('U',s)>0) then with FFont do Style:=Style+[fsUnderline];
  if (Pos('S',s)>0) then with FFont do Style:=Style+[fsStrikeOut];
  try FFont.Size:=StrToInt(FProperties.Values['FontSize']) except end;
  s:=FProperties.Values['FontColor'];
  if (s<>'') then try FFont.Color:=ReadColor(s) except end;
  s:=FProperties.Values['Alignment'];
  if (s='Left') then FAlignment:=taLeftJustify
  else if (s='Right') then FAlignment:=taRightJustify
  else if (s='Center') then FAlignment:=taCenter;
end;


/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//   TPlanPicture
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
constructor TPlanPicture.Create(Owner:TOxygenPlan);
begin
  inherited Create(Owner);
  FPicture:=TPicture.Create;
{$IFDEF USE_RX}
  FSavePicture:=TPicture.Create;
  FBlinkPicture:=TPicture.Create;
{$ENDIF}
  FBrush.Color:=clBlack;
  FStdPictureName:='';
end;

function TPlanPicture.Clone(Parent : TPlanObject) : TPlanObject;
begin
  Result := TPlanPicture.Create(Owner);
  FillObjectProperties;
  Result.Parent := Parent;
  Result.Properties.AddStrings(FProperties);
  Result.Properties.Values['Name'] := '';
  Result.Properties.Values['ID'] := '0';
  Result.UpdateObjectProperties;
  with (Result as TPlanPicture) do begin
    X1 := X1 + 4;
    Y1 := Y1 + 4;
    X2 := X2 + 4;
    Y2 := Y2 + 4;
  end;
end;

procedure TPlanPicture.Free;
begin
  FVisible:=False;
  if not (csDestroying in FOwner.ComponentState) then try NeedRepaint except end;
  FDestroying:=True;
{$IFDEF USE_RX}
  StopBlink;
  FSavePicture.Free;
  FBlinkPicture.Free;
{$ENDIF}
  FPicture.Free;
  inherited Free;
end;

procedure TPlanPicture.Draw(ACanvas:TCanvas);
  var p1,p2:TPoint;
begin
  if (FOwner.FDisableRepaintCount>0) then Exit;
  if not (FLayer.Visible and FVisible) then Exit;
  p1:=ScrPoint1;
  p2:=ScrPoint2;
  with ACanvas do begin
    Pen.Assign(Self.FPen);
    Brush.Assign(Self.FBrush);
    if (FPicture = nil) or (FPicture.Graphic = nil) or FPicture.Graphic.Empty then begin
      Pen.Color := FOwner.Pen.Color;
      Pen.Width := 1;
      Pen.Style := psDash;
      Pen.Mode := pmCopy;
      Rectangle(p1.X,p1.Y,p2.X,p2.Y);
    end
    else begin
      //if ((FPicture.Graphic is TBitmap) and (FPicture.Bitmap <> nil) and Transparent) then FPicture.Bitmap.TransparentColor := FPicture.Bitmap.Canvas.Pixels[0,0];
      if Assigned(FPicture) then if (FPicture.Graphic <> nil) then StretchDraw(Rect(p1.X,p1.Y,p2.X,p2.Y),FPicture.Graphic);
    end;
    if (FSelected and (not (FOwner.ReadOnly or FLayer.ReadOnly or FReadOnly))) then begin
      ACanvas.Pen.Color:=FOwner.SelectColor;
      ACanvas.Brush.Color:=FOwner.SelectColor;
      ACanvas.Brush.Style:=bsSolid;
      ACanvas.Pen.Width:=1;
      Rectangle(p1.X-MarginDelta,p1.Y-MarginDelta,p1.X+MarginDelta,p1.Y+MarginDelta);
      Rectangle(p2.X-MarginDelta,p1.Y-MarginDelta,p2.X+MarginDelta,p1.Y+MarginDelta);
      Rectangle(p2.X-MarginDelta,p2.Y-MarginDelta,p2.X+MarginDelta,p2.Y+MarginDelta);
      Rectangle(p1.X-MarginDelta,p2.Y-MarginDelta,p1.X+MarginDelta,p2.Y+MarginDelta);
    end;
    if (FSelectColor <> -1) then begin
      ACanvas.Pen.Color := FSelectColor;
      ACanvas.Pen.Width := 2;
      ACanvas.Brush.Color := FOwner.Color;
      ACanvas.Brush.Style := bsClear;
      Rectangle(p1.X,p1.Y,p2.X,p2.Y);
    end;
  end;
  DrawChildren(ACanvas);
end;

{$IFDEF USE_RX}
procedure TPlanPicture.StartBlink(const Interval:Integer);
begin
  StopBlink;
  FSavePicture.Assign(FPicture);
  inherited StartBlink(Interval);
end;

procedure TPlanPicture.StopBlink;
begin
  if (FBlinkTimerHandle=0) then Exit;
  inherited StopBlink;
  FPicture.Assign(FSavePicture);
  NeedRepaint;
end;
{$ENDIF}

function TPlanPicture.GetObjectTypeDescr : String;
begin
  Result:=ocPicture;
end;

procedure TPlanPicture.SetStdPictureName(const Value : String);
  var i:Integer;
begin
  if (FStdPictureName = Value) then Exit;
  FStdPictureName:='';
  i:=FOwner.PictureIndex(AnsiUpperCase(Value));
  if (i>=0) then begin
    FPicture.Assign(FOwner.Pictures[i]);
    FStdPictureName := AnsiUpperCase(Value);
  end
  else FPicture.Assign(nil);
end;

function TPlanPicture.GetTransparent : Boolean;
begin
  Result := False;
  if (FPicture <> nil) and (FPicture.Graphic <> nil) then Result := FPicture.Graphic.Transparent;
end;

procedure TPlanPicture.SetTransparent(const Value : Boolean);
begin
  if (FPicture <> nil) and (FPicture.Graphic <> nil) then FPicture.Graphic.Transparent := Value;
  FProperties.Values['Transparent'] := IntToStr(Integer(Value));
end;

procedure TPlanPicture.FillObjectProperties;
  var s:String;
      ms:TMemoryStream;
      pic:TPicture;
begin
  inherited FillObjectProperties;
  if (FStdPictureName <> '') then begin
    FProperties.Values['Graphic']:='%'+FStdPictureName+'%';
    FProperties.Values['Transparent']:=IntToStr(Integer(FPicture.Graphic.Transparent));
  end
  else begin
    pic:=TPicture.Create;
    pic.Assign(FPicture);
    if ((pic.Graphic=nil) or pic.Graphic.Empty) then s:=''
    else begin
      ms:=TMemoryStream.Create;
      pic.Graphic.SaveToStream(ms);
      FProperties.Values['Graphic']:=Stream2String(ms);
      ms.Free;
      FProperties.Values['Transparent']:=IntToStr(Integer(pic.Graphic.Transparent));
    end;
    FProperties.Values['Type']:=GetPictureType(pic);
    pic.Free;
  end;
end;

{$IFDEF USE_RX}
procedure TPlanPicture.BlinkTimer(Sender: TObject);
begin
  FBlinkIndex:=(FBlinkIndex xor 1);
  case FBlinkType of
    btNone    : Exit;
    btPen     : if (FBlinkIndex=1) then FPen.Assign(FBlinkPen) else FPen.Assign(FSavePen);
    btBrush   : if (FBlinkIndex=1) then FBrush.Assign(FBlinkBrush) else FBrush.Assign(FSaveBrush);
    btPicture : if (FBlinkIndex=1) then FPicture.Assign(FBlinkPicture) else FPicture.Assign(FSavePicture);
  end;
  if (FLayer = FOwner.CurrentLayer) then begin
    if (FPicture.Graphic<>nil) then if not FPicture.Graphic.Transparent then Draw(FOwner.Canvas) else NeedRepaint;
  end
  else NeedRepaint;
end;
{$ENDIF}

function TPlanPicture.HasInternalProperty(const PropertyName : String) : Boolean;
  var s : String;
begin
  Result := inherited HasInternalProperty(PropertyName);
  if Result then Exit;
  s := UpperCase(PropertyName);
  Result := (s = 'GRAPHIC') or (s = 'TYPE') or (s = 'TRANSPARENT');
end;

procedure TPlanPicture.UpdateObjectProperties;
  var s,gr:String;
      ms:TMemoryStream;
      i:Integer;
begin
  inherited UpdateObjectProperties;

  FPicture.Assign(nil);
  FStdPictureName:='';
  gr:=FProperties.Values['Graphic'];
  if (Length(gr)>2) then if ((gr[1]='%') and (gr[Length(gr)]='%')) then begin
    gr:=AnsiUpperCase(System.Copy(gr,2,Length(gr)-2));
    i:=FOwner.PictureIndex(gr);
    if (i>=0) then begin
      FPicture.Assign(FOwner.Pictures[i]);
      FStdPictureName := gr;
      try FPicture.Graphic.Transparent := (StrToInt(Properties.Values['Transparent'])<>0) except end;
    end
    else FOwner.LoadPicture(gr,Self);
  end
  else begin
    s := FProperties.Values['Type'];
    ms:=TMemoryStream.Create;
    String2Stream(gr,ms);
    LoadPictureFromStream(s, FPicture, ms);
    ms.Free;
    if not ((FPicture.Graphic = nil) or FPicture.Graphic.Empty) then try
      FPicture.Graphic.Transparent:=(StrToInt(Properties.Values['Transparent'])<>0);
    except end;
    if ((FPicture.Graphic is TBitmap) and (FPicture.Bitmap <> nil) and Transparent) then FPicture.Bitmap.TransparentColor := FPicture.Bitmap.Canvas.Pixels[0,0];
  end;
end;

{$IFDEF USE_RX}
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// TPlanAniPicture
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
constructor TPlanAniPicture.Create(Owner:TOxygenPlan);
begin
  inherited Create(Owner);
  FImageIndex:=0;
  FImageCount:=1;
  FInterval:=1000;
  FCurBitmap:=TBitmap.Create;
end;

function TPlanAniPicture.Clone(Parent : TPlanObject) : TPlanObject;
begin
  Result := TPlanAniPicture.Create(Owner);
  FillObjectProperties;
  Result.Parent := Parent;
  Result.Properties.AddStrings(FProperties);
  Result.Properties.Values['Name'] := '';
  Result.Properties.Values['ID'] := '0';
  Result.UpdateObjectProperties;
  with (Result as TPlanAniPicture) do begin
    X1 := X1 + 4;
    Y1 := Y1 + 4;
    X2 := X2 + 4;
    Y2 := Y2 + 4;
  end;
end;

procedure TPlanAniPicture.Free;
begin
  FVisible:=False;
  if not (csDestroying in FOwner.ComponentState) then try NeedRepaint except end;
  FDestroying:=True;
  StopBlink;
  FCurBitmap.Free;
  inherited Free;
end;

procedure TPlanAniPicture.SetImageIndex(const Value : Integer);
begin
  if ((FImageIndex = Value) or (Value > (FImageCount-1)) or (Value < 0)) then Exit;
  FImageIndex:=Value;
  NeedRepaint;
end;

procedure TPlanAniPicture.SetImageCount(const Value : Integer);
begin
  if ((FImageCount = Value) or (not Assigned(FPicture)) or (FPicture.Graphic = nil) or (Value < 1)) then Exit;
  FImageCount:=Value;
  FCurBitmap.Width:=FPicture.Width div FImageCount;
  FCurBitmap.Height:=FPicture.Height;
  FCurBitmap.Transparent:=FPicture.Graphic.Transparent;
end;

function TPlanAniPicture.CanBeParent : Boolean;
begin
  Result:=False;
end;

procedure TPlanAniPicture.FillObjectProperties;
begin
  inherited FillObjectProperties;
  Properties.Values['ImageCount']:=IntToStr(FImageCount);
  Properties.Values['Interval']:=IntToStr(FInterval);
end;

function TPlanAniPicture.HasInternalProperty(const PropertyName : String) : Boolean;
  var s : String;
begin
  Result := inherited HasInternalProperty(PropertyName);
  if Result then Exit;
  s := UpperCase(PropertyName);
  Result := (s = 'IMAGECOUNT') or (s = 'INTERVAL');
end;

procedure TPlanAniPicture.BlinkTimer(Sender: TObject);
begin
  if Assigned(FPicture) then if (FPicture.Graphic<>nil) then if (FPicture.Graphic is TBitmap) then begin
    if (FImageIndex < FImageCount-1) then Inc(FImageIndex) else FImageIndex:=0;
    if (FLayer=FOwner.CurrentLayer) then Draw(FOwner.Canvas) else NeedRepaint;
  end;
end;

function TPlanAniPicture.GetObjectTypeDescr : String;
begin
  Result:=ocAniPicture;
end;

procedure TPlanAniPicture.Draw(ACanvas:TCanvas);
  var p1,p2:TPoint;
begin
  if (FOwner.FDisableRepaintCount>0) then Exit;
  if not (FLayer.Visible and FVisible) then Exit;
  p1:=ScrPoint1;
  p2:=ScrPoint2;
  with ACanvas do begin
    Pen.Assign(Self.FPen);
    Brush.Assign(Self.FBrush);
    if (FPicture = nil) or (FPicture.Graphic = nil) or FPicture.Graphic.Empty or (not (FPicture.Graphic is TBitmap)) then begin
      Pen.Color := FOwner.Pen.Color;
      Pen.Width := 1;
      Pen.Style := psDash;
      Pen.Mode := pmCopy;
      Rectangle(p1.X,p1.Y,p2.X,p2.Y);
    end
    else if Assigned(FPicture) then if (FPicture.Graphic<>nil) then if (FPicture.Graphic is TBitmap) then begin
      FCurBitmap.Canvas.CopyRect(Rect(0,0,FCurBitmap.Width,FCurBitmap.Height),FPicture.Bitmap.Canvas,
                                 Rect(FImageIndex*FCurBitmap.Width,0,(FImageIndex+1)*FCurBitmap.Width,FCurBitmap.Height));
      StretchDraw(Rect(p1.X,p1.Y,p2.X,p2.Y),FCurBitmap);
    end;
    if (FSelected and (not (FOwner.ReadOnly or FLayer.ReadOnly or FReadOnly))) then begin
      ACanvas.Pen.Color:=FOwner.SelectColor;
      ACanvas.Brush.Color:=FOwner.SelectColor;
      ACanvas.Brush.Style:=bsSolid;
      ACanvas.Pen.Width:=1;
      Rectangle(p1.X-MarginDelta,p1.Y-MarginDelta,p1.X+MarginDelta,p1.Y+MarginDelta);
      Rectangle(p2.X-MarginDelta,p1.Y-MarginDelta,p2.X+MarginDelta,p1.Y+MarginDelta);
      Rectangle(p2.X-MarginDelta,p2.Y-MarginDelta,p2.X+MarginDelta,p2.Y+MarginDelta);
      Rectangle(p1.X-MarginDelta,p2.Y-MarginDelta,p1.X+MarginDelta,p2.Y+MarginDelta);
    end;
    if (FSelectColor<>-1) then begin
      ACanvas.Pen.Color:=FSelectColor;
      ACanvas.Pen.Width:=2;
      ACanvas.Brush.Color:=FOwner.Color;
      ACanvas.Brush.Style:=bsClear;
      Rectangle(p1.X,p1.Y,p2.X,p2.Y);
    end;
  end;
end;

procedure TPlanAniPicture.Animate(const Interval:Integer);
begin
  StartBlink(Interval);
end;

procedure TPlanAniPicture.StartBlink(const Interval:Integer);
begin
  StopBlink;
  if (FImageCount=0) then Exit;
  FInterval:=Interval;
  FBlinkTimerHandle:=FOwner.FTimerList.Add(BlinkTimer,Interval,True);
  FOwner.FTimerList.Activate;
end;

procedure TPlanAniPicture.StopBlink;
begin
  if (FBlinkTimerHandle=0) then Exit;
  FImageIndex:=0;
  FOwner.FTimerList.Delete(FBlinkTimerHandle);
  FBlinkType:=btNone;
  FBlinkTimerHandle:=0;
  if FVisible then NeedRepaint;
end;

procedure TPlanAniPicture.Stop;
begin
  StopBlink;
end;

procedure TPlanAniPicture.UpdateObjectProperties;
  var s:String;
begin
  inherited UpdateObjectProperties;
  s:=Properties.Values['Interval'];
  if (s='') then FInterval:=100 else try FInterval:=StrToInt(s) except end;

  s:=Properties.Values['ImageCount'];
  if (s='') then SetImageCount(1) else try SetImageCount(StrToInt(s)) except end;
  if ((poAutoAnimate in FOwner.Options) and (FImageCount>0)) then StartBlink(FInterval);
end;
{$ENDIF}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//   TPlanPolygon
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
constructor TPlanPolygon.Create(Owner:TOxygenPlan);
begin
  inherited Create(Owner);
  FPointCount:=0;
  FPointArray:=nil;
  FScrPointArray:=nil;
  FScrSaveArray:=nil;
  FCurMovingPointIndex:=-1;
end;

function TPlanPolygon.Clone(Parent : TPlanObject) : TPlanObject;
  var i : Integer;
      p : TPoint;
begin
  Result := TPlanPolygon.Create(Owner);
  FillObjectProperties;
  Result.Parent := Parent;
  Result.Properties.AddStrings(FProperties);
  Result.Properties.Values['Name'] := '';
  Result.Properties.Values['ID'] := '0';
  Result.UpdateObjectProperties;
  with (Result as TPlanPolygon) do for i := 1 to PointCount do begin
    p := Points[i-1];
    Points[i-1] := Point(p.X + 4, p.Y + 4);
  end;
end;

procedure TPlanPolygon.SetPointCount(Value:Integer);
  var pa:PPointArray;
      i:Integer;
begin
  if (Value = FPointCount) then Exit;
  GetMem(pa,FPointCount*SizeOf(TPoint));
  System.Move(FPointArray^,pa^,FPointCount*SizeOf(TPoint));
  FreeMem(FPointArray,FPointCount*SizeOf(TPoint));
  if (FScrPointArray<>nil) then FreeMem(FScrPointArray,FPointCount*SizeOf(TPoint));
  if (FScrSaveArray<>nil) then FreeMem(FScrSaveArray,FPointCount*SizeOf(TPoint));
  GetMem(FPointArray,Value*SizeOf(TPoint));
  for i:=1 to Value do if (FPointCount>=i) then FPointArray^[i-1]:=pa^[i-1] else FPointArray^[i-1]:=Point(0,0);
  FreeMem(pa,FPointCount*SizeOf(TPoint));
  FPointCount:=Value;
  GetMem(FScrPointArray,FPointCount*SizeOf(TPoint));
  GetMem(FScrSaveArray,FPointCount*SizeOf(TPoint));
  for i:=1 to FPointCount do
      FScrPointArray^[i-1]:=Point(Trunc(FPointArray^[i-1].X*(FOwner.Width/FOwner.ScaleX)),
                                  Trunc(FPointArray^[i-1].Y*(FOwner.Height/FOwner.ScaleY)))
end;

function  TPlanPolygon.GetPoint(Index:Integer):TPoint;
begin
  Result:=Point(-1,-1);
  if (Index>(FPointCount-1)) then Exit;
  Result:=FPointArray^[Index];
end;

procedure TPlanPolygon.SetPoint(Index:Integer; Value:TPoint);
begin
  if (Index>(FPointCount-1)) then Exit;
  NeedRepaint;
  FPointArray^[Index]:=Value;
  FScrPointArray^[Index]:=Point(Trunc(FPointArray^[Index].X*(FOwner.Width/FOwner.ScaleX)),Trunc(FPointArray^[Index].Y*(FOwner.Height/FOwner.ScaleY)));
  NeedRepaint;
end;

procedure TPlanPolygon.ParentResized;
  var i:Integer;
begin
  if (FPointCount=0) then Exit;
  if (FScrPointArray=nil) then GetMem(FScrPointArray,FPointCount*SizeOf(TPoint));
  if (FScrSaveArray=nil) then GetMem(FScrSaveArray,FPointCount*SizeOf(TPoint));
  for i:=1 to FPointCount do
      FScrPointArray^[i-1]:=Point(Trunc(FPointArray^[i-1].X*(FOwner.Width/FOwner.ScaleX)),
                                  Trunc(FPointArray^[i-1].Y*(FOwner.Height/FOwner.ScaleY)))
end;

procedure TPlanPolygon.DeletePoint(const Index : Integer);
  var pa:PPointArray;
begin
  if ((Index >= FPointCount) or (FPointCount <= 3)) then Exit;
  if (Index<FPointCount-1) then begin
    pa:=Addr(FPointArray^[Index+1]);
    System.Move(pa^,FPointArray^[Index],(FPointCount-1-Index)*SizeOf(TPoint));
  end;
  SetPointCount(FPointCount-1);
  NeedRepaint;
end;

procedure TPlanPolygon.Draw(ACanvas:TCanvas);
  var i:Integer;
begin
  if (FOwner.FDisableRepaintCount>0) then Exit;
  if not (FLayer.Visible and FVisible) then Exit;
  with ACanvas do begin
    Pen.Assign(Self.FPen);
    Brush.Assign(Self.FBrush);
    if (FSelectColor<>-1) then ACanvas.Pen.Color:=FSelectColor;
    Windows.Polygon(ACanvas.Handle,FScrPointArray^,FPointCount);
    if (FSelected and (not (FOwner.ReadOnly or FLayer.ReadOnly or FReadOnly))) then begin
      ACanvas.Pen.Color:=FOwner.SelectColor;
      ACanvas.Brush.Color:=FOwner.SelectColor;
      ACanvas.Brush.Style:=bsSolid;
      ACanvas.Pen.Width:=1;
      for i:=1 to FPointCount do Rectangle(FScrPointArray^[i-1].X-MarginDelta,FScrPointArray^[i-1].Y-MarginDelta,FScrPointArray^[i-1].X+MarginDelta,FScrPointArray^[i-1].Y+MarginDelta);
    end;
  end;
  DrawChildren(ACanvas);
end;

procedure TPlanPolygon.Free;
begin
  FVisible:=False;
  if not (csDestroying in FOwner.ComponentState) then try NeedRepaint except end;
  FDestroying:=True;
{$IFDEF USE_RX}
  StopBlink;
{$ENDIF}
  if (FPointArray<>nil) then FreeMem(FPointArray,FPointCount*SizeOf(TPoint));
  if (FScrPointArray<>nil) then FreeMem(FScrPointArray,FPointCount*SizeOf(TPoint));
  if (FScrSaveArray<>nil) then FreeMem(FScrSaveArray,FPointCount*SizeOf(TPoint));
  FPointCount:=0;
  FPointArray:=nil;
  FScrPointArray:=nil;
  FScrSaveArray:=nil;
  inherited Free;
end;

function TPlanPolygon.HasInternalPoint(X,Y:Integer):Boolean;
  var rgn:HRgn;
begin
  rgn:=CreatePolygonRgn(FScrPointArray^,FPointCount,WINDING);
  Result:=PtInRegion(rgn,X,Y);
  Windows.DeleteObject(rgn);
end;

function TPlanPolygon.HasInternalProperty(const PropertyName : String) : Boolean;
  var s : String;
begin
  Result := inherited HasInternalProperty(PropertyName);
  if Result then Exit;
  s := UpperCase(PropertyName);
  Result := (s = 'POINTCOUNT');
  if Result then Exit;
  Result := (s[1] = 'X') or (s[1] = 'Y');
  if Result then try
    StrToInt(System.Copy(s,2,255));
  except
    Result := False;
  end;
end;

function TPlanPolygon.HasMarginPoint(X,Y:Integer; var MarginType:TMarginType):Boolean;
  var i:Integer;
      p:TPoint;
begin
  Result:=False;
  for i:=1 to FPointCount do begin
    p:=FScrPointArray^[i-1];
    if PointInRect(Point(X,Y),Rect(p.X-MarginDelta,p.Y-MarginDelta,p.X+MarginDelta,p.Y+MarginDelta)) then begin
      FCurMovingPointIndex:=i-1;
      Result:=True;
      MarginType:=mtMovePoint;
      Exit;
    end;
  end;
end;

procedure TPlanPolygon.Move(APoint:TPoint);
  var i,dx,dy:Integer;
      tmp:PPointArray;
      clr:HRGN;
      f:Boolean;
begin
  NeedRepaint;
  dx:=APoint.X-FOwner.FStartCoord.X;
  dy:=APoint.Y-FOwner.FStartCoord.Y;

  GetMem(tmp,FPointCount*SizeOf(TPoint));
  for i:=1 to FPointCount do tmp^[i-1]:=Point(FScrSaveArray^[i-1].X+dx,FScrSaveArray^[i-1].Y+dy);
  if (FParent <> nil) and (poChildrenClipping in FOwner.Options) then begin
    clr:=FParent.GetClippingRgn;
    f:=False;
    for i:=1 to FPointCount do if PtInRegion(clr,tmp^[i-1].X,tmp^[i-1].Y) then begin
      f:=True;
      Break;
    end;
    if not f then begin
      DeleteObject(clr);
      FreeMem(tmp,FPointCount*SizeOf(TPoint));
      Exit;
    end;
    DeleteObject(clr);
  end;
  System.Move(tmp^,FScrPointArray^,FPointCount*SizeOf(TPoint));
  FreeMem(tmp,FPointCount*SizeOf(TPoint));
  for i:=1 to FPointCount do FPointArray^[i-1]:=Point(Trunc(FScrPointArray^[i-1].X*(FOwner.ScaleX/FOwner.Width)),
                                                      Trunc(FScrPointArray^[i-1].Y*(FOwner.ScaleY/FOwner.Height)));
  NeedRepaint;

  inherited Move(APoint);
end;

procedure TPlanPolygon.Resize(APoint:TPoint);
begin
  if not FCanResize then Exit;
  if (FCurMovingPointIndex=-1) then Exit;
  NeedRepaint;
  FScrPointArray[FCurMovingPointIndex]:=APoint;
  FPointArray^[FCurMovingPointIndex]:=Point(Trunc(APoint.X*(FOwner.ScaleX/FOwner.Width)),Trunc(APoint.Y*(FOwner.ScaleY/FOwner.Height)));
  NeedRepaint;
end;

function TPlanPolygon.CanBeParent : Boolean;
begin
  Result:=True;
end;

function TPlanPolygon.GetObjectTypeDescr : String;
begin
  Result:=ocPolygon;
end;

procedure TPlanPolygon.FillObjectProperties;
  var i:Integer;
begin
  inherited FillObjectProperties;
  FProperties.Values['PointCount']:=IntToStr(FPointCount);
  for i:=1 to FPointCount do begin
    FProperties.Values['X'+IntToStr(i)]:=IntToStr(FPointArray^[i-1].X);
    FProperties.Values['Y'+IntToStr(i)]:=IntToStr(FPointArray^[i-1].Y);
  end;
end;

procedure TPlanPolygon.SaveCoords;
begin
  if (FPointCount=0) then Exit;
  System.Move(FScrPointArray^,FScrSaveArray^,FPointCount*SizeOf(TPoint));
  inherited SaveCoords;
end;

procedure TPlanPolygon.NeedRepaint;
  var r:TRect;
begin
  if FDestroying then Exit;
  if (FPointCount=0) then Exit;
  r:=DisplayRect;
  Dec(r.Top,MarginDelta + Pen.Width);
  Dec(r.Left,MarginDelta + Pen.Width);
  Inc(r.Bottom,MarginDelta + Pen.Width);
  Inc(r.Right,MarginDelta + Pen.Width);
  Windows.InvalidateRect(FOwner.Handle,@r,False);
end;

function TPlanPolygon.GetDisplayRect:TRect;
  var i:Integer;
begin
  Result:=Rect(FScrPointArray^[0].X,FScrPointArray^[0].Y,FScrPointArray^[0].X,FScrPointArray^[0].Y);
  for i:=1 to FPointCount do begin
    if (FScrPointArray^[i-1].X < Result.Left) then Result.Left:=FScrPointArray^[i-1].X;
    if (FScrPointArray^[i-1].X > Result.Right) then Result.Right:=FScrPointArray^[i-1].X;
    if (FScrPointArray^[i-1].Y < Result.Top) then Result.Top:=FScrPointArray^[i-1].Y;
    if (FScrPointArray^[i-1].Y > Result.Bottom) then Result.Bottom:=FScrPointArray^[i-1].Y;
  end;
  Dec(Result.Top, Pen.Width div 2);
  Dec(Result.Left, Pen.Width div 2);
  Inc(Result.Bottom, Pen.Width div 2);
  Inc(Result.Right, Pen.Width div 2);
end;

function TPlanPolygon.AddPoint(const CoordX,CoordY:Integer; CallHandler:TCallHandler):Boolean;
  var i:Integer;
begin
  if ((FPointCount > 3) and (CallHandler = chMouseDown)) then Result:=(Distance(FScrPointArray^[0],Point(CoordX,CoordY)) > Cardinal(MarginDelta*2)) else Result:=True;
  if not Result then Exit;
  i:=FOwner.FCurDrawingPointNum;
  NeedRepaint;
  PointCount := i + 1;
  FScrPointArray^[i]:=Point(CoordX,CoordY);
  Points[i]:=Point(Trunc(FScrPointArray^[i].X*(FOwner.ScaleX/FOwner.Width)),Trunc(FScrPointArray^[i].Y*(FOwner.ScaleY/FOwner.Height)));
end;

function TPlanPolygon.GetClippingRgn : HRGN;
begin
  Result:=CreatePolygonRgn(FScrPointArray^,FPointCount,WINDING);
end;

procedure TPlanPolygon.UpdateObjectProperties;
  var i:Integer;
      s:String;
begin
  inherited UpdateObjectProperties;
  PointCount:=0;
  s:=FProperties.Values['PointCount'];
  if (s<>'') then try PointCount:=StrToInt(s) except end;
  for i:=1 to FPointCount do try
    FPointArray^[i-1]:=Point(StrToInt(FProperties.Values['X'+IntToStr(i)]),StrToInt(Properties.Values['Y'+IntToStr(i)]));
    FScrPointArray^[i-1]:=Point(Trunc(FPointArray^[i-1].X*(FOwner.Width/FOwner.ScaleX)),Trunc(FPointArray^[i-1].Y*(FOwner.Height/FOwner.ScaleY)));
  except end;
end;


/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//   TPlanCompositeObject
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
procedure TPlanCompositeObject.SetPen(Value:TPen);
  var i:Integer;
begin
  for i:=1 to ChildrenCount do Objects[i-1].SetPen(Value);
  inherited SetPen(Value);
end;

procedure TPlanCompositeObject.SetBrush(Value:TBrush);
  var i:Integer;
begin
  for i:=1 to ChildrenCount do Objects[i-1].SetBrush(Value);
  inherited SetBrush(Value);
end;

procedure TPlanCompositeObject.SetSelected(Value:Boolean);
  var i:Integer;
begin
  for i:=1 to ChildrenCount do Objects[i-1].SetSelected(Value);
  inherited SetSelected(Value);
end;

procedure TPlanCompositeObject.Draw(ACanvas:TCanvas);
  var p1,p2:TPoint;
begin
  if (FOwner.FDisableRepaintCount>0) then Exit;
  if not (FLayer.Visible and FVisible) then Exit;
  p1:=ScrPoint1;
  p2:=ScrPoint2;
  ACanvas.Pen.Assign(FPen);
  ACanvas.Pen.Style:=psDash;
  ACanvas.Brush.Style:=bsClear;
  ACanvas.Rectangle(p1.X,p1.Y,p2.X,p2.Y);
  DrawChildren(ACanvas);
end;

procedure TPlanCompositeObject.Select;
  var i:Integer;
begin
  for i:=1 to ChildrenCount do Objects[i-1].Select;
  FSelected:=True;
  NeedRepaint;
end;

procedure TPlanCompositeObject.UnSelect;
  var i:Integer;
begin
  for i:=1 to ChildrenCount do Objects[i-1].UnSelect;
  inherited Unselect;
end;

procedure TPlanCompositeObject.SelectByColor(const AColor:TColor);
  var i:Integer;
begin
  for i:=1 to ChildrenCount do Objects[i-1].SelectByColor(AColor);
end;

procedure TPlanCompositeObject.UnSelectColor;
  var i:Integer;
begin
  for i:=1 to ChildrenCount do Objects[i-1].UnSelectColor;
end;

function TPlanCompositeObject.CanBeParent : Boolean;
begin
  Result:=True;
end;

function TPlanCompositeObject.GetObjectTypeDescr : String;
begin
  Result:=ocCompositeObject;
end;


{$IFDEF USE_RX}
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// TPlanAniGif
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
constructor TPlanAniGif.Create(Owner:TOxygenPlan);
begin
  inherited Create(Owner);
  FImage:=TGIFImage.Create;
  FStdAniGifName:='';
end;

function TPlanAniGif.Clone(Parent : TPlanObject) : TPlanObject;
begin
  Result := TPlanAniGif.Create(Owner);
  FillObjectProperties;
  Result.Parent := Parent;
  Result.Properties.AddStrings(FProperties);
  Result.Properties.Values['Name'] := '';
  Result.Properties.Values['ID'] := '0';
  Result.UpdateObjectProperties;
  with (Result as TPlanAniGif) do begin
    X1 := X1 + 4;
    Y1 := Y1 + 4;
    X2 := X2 + 4;
    Y2 := Y2 + 4;
  end;
end;

procedure TPlanAniGif.Free;
begin
  FVisible:=False;
  if not (csDestroying in FOwner.ComponentState) then try NeedRepaint except end;
  FDestroying:=True;
  StopBlink;
  FImage.Free;
  inherited Free;
end;

function TPlanAniGif.GetDelayTime(Index: Integer): Cardinal;
  var FFrameIndex:Integer;
begin
  FFrameIndex:=FImage.FrameIndex;
  if ((FFrameIndex >= 0) and (FFrameIndex < FImage.Count) and (FImage.Count > 1)) then begin
    Result := FImage.Frames[FFrameIndex].AnimateInterval;
    if (Result < MinDelayTime) then Result := MinDelayTime else if (Result > MaxDelayTime) then Result := MaxDelayTime;
  end
  else Result := 0;
end;

procedure TPlanAniGif.SetStdAniGifName(const Value : String);
  var b:Boolean;
      i:Integer;
begin
  if (FStdAniGifName=Value) then Exit;
  b:=Blinking;
  StopBlink;
  FStdAniGifName:='';
  i:=FOwner.AniGifIndex(AnsiUpperCase(Value));
  if (i>=0) then begin
    FImage.Assign(FOwner.AniGifs[i]);
    FStdAniGifName:=Value;
  end
  else FImage.Assign(nil);
  if b then StartBlink(0);
end;

procedure TPlanAniGif.SetGIFImage(Value : TGIFImage);
  var b:Boolean;
begin
  b:=Blinking;
  StopBlink;
  FStdAniGifName:='';
  FImage.Assign(Value);
  if b then StartBlink(0);
end;

procedure TPlanAniGif.BlinkTimer(Sender: TObject);
  var FFrameIndex:Integer;
      FTimer : TRxTimerEvent;
begin
  if FImage.Empty then begin
    StopBlink;
    Exit;
  end;
  FFrameIndex := FImage.FrameIndex;
  if (FFrameIndex < FImage.Count-1) then Inc(FFrameIndex) else FFrameIndex:=0;
  if (FImage.FrameIndex<>FFrameIndex) then begin
    FImage.FrameIndex:=FFrameIndex;
    if ((FLayer = FOwner.CurrentLayer) and (FImage.Frames[0].TransparentColor = clNone)) then Draw(FOwner.Canvas)
    else NeedRepaint;
  end;
  if (FFrameIndex >= 0) and (FImage.Count > 0) then begin
    FTimer:=FOwner.FTimerList.ItemByHandle(FBlinkTimerHandle);
    FTimer.Interval := GetDelayTime(FFrameIndex);
    FTimer.Enabled:=True;
  end;
end;

function TPlanAniGif.CanBeParent : Boolean;
begin
  Result:=False;
end;

function TPlanAniGif.GetFrameBitmap(Index: Integer; var TransColor: TColor): TBitmap;
  var I, Last, First: Integer;
      SavePal: HPalette;
begin
  Index := Min(Index, FImage.Count - 1);
  Result := TBitmap.Create;
  Result.Canvas.Lock;
  try
    with Result do begin
      Width := FImage.ScreenWidth;
      Height := FImage.ScreenHeight;
      Last := Index;
      First := Max(0, Last);
      SavePal := 0;
      if FImage.Palette <> 0 then begin
        SavePal := SelectPalette(Canvas.Handle, FImage.Palette, False);
        RealizePalette(Canvas.Handle);
      end;
      if (FImage.Frames[FImage.FrameIndex].TransparentColor <> clNone) then begin
        TransColor := GetNearestColor(Canvas.Handle,ColorToRGB(FImage.Frames[FImage.FrameIndex].TransparentColor));
        Canvas.Brush.Color := PaletteColor(TransColor);
      end
      else if ((FImage.BackgroundColor<>clNone) and FImage.Transparent) then Canvas.Brush.Color:=PaletteColor(FImage.BackgroundColor) else Canvas.Brush.Color:=PaletteColor(clWindow);
      Canvas.FillRect(Bounds(0, 0, Width, Height));
      while (First>0) do begin
        if (FImage.ScreenWidth=FImage.Frames[First].Width) and (FImage.ScreenHeight=FImage.Frames[First].Height) then begin
          if (FImage.Frames[First].TransparentColor = clNone) or ((FImage.Frames[First].DisposalMethod = dmRestoreBackground) and (First < Last)) then Break;
        end;
        Dec(First);
      end;
      for I := First to Last - 1 do begin
        with FImage.Frames[I] do case DisposalMethod of
          dmUndefined, dmLeave: Draw(Canvas, Bounds(Origin.X, Origin.Y, Width, Height), True);
          dmRestoreBackground: if (I>First) then Canvas.FillRect(Bounds(Origin.X, Origin.Y, Width, Height));
          dmRestorePrevious: begin { do nothing } end;
        end;
      end;
      with FImage.Frames[Last] do Draw(Canvas,Bounds(Origin.X,Origin.Y,Width,Height), True);
      if (TransColor <> clNone) and FImage.Transparent then begin
        TransparentColor:=PaletteColor(TransColor);
        Transparent:=True;
      end;
      if (FImage.Palette<>0) then SelectPalette(Canvas.Handle, SavePal, False);
    end;
    Result.Canvas.Unlock;
  except
    Result.Canvas.Unlock;
    Result.Free;
    raise;
  end;
end;

function TPlanAniGif.GetObjectTypeDescr : String;
begin
  Result:=ocAniGif;
end;

procedure TPlanAniGif.FillObjectProperties;
  var ms:TMemoryStream;
      gif:TGifImage;
      s:String;
begin
  inherited FillObjectProperties;
  if (FStdAniGifName<>'') then Properties.Values['Graphic']:='%'+FStdAniGifName+'%'
  else begin
    gif:=TGifImage.Create;
    gif.Assign(FImage);
    if ((gif.Count=0) or gif.Empty) then s:=''
    else begin
      ms:=TMemoryStream.Create;
      gif.SaveToStream(ms);
      Properties.Values['Graphic']:=Stream2String(ms);
      ms.Free;
    end;
    gif.Free;
  end;
end;

function TPlanAniGif.HasInternalProperty(const PropertyName : String) : Boolean;
  var s : String;
begin
  Result := inherited HasInternalProperty(PropertyName);
  if Result then Exit;
  s := UpperCase(PropertyName);
  Result := (s = 'GRAPHIC');
end;


procedure TPlanAniGif.UpdateObjectProperties;
  var s:String;
      ms:TMemoryStream;
      i:Integer;
begin
  inherited UpdateObjectProperties;

  FImage.Clear;
  FStdAniGifName:='';
  s:=Properties.Values['Graphic'];
  if (Length(s)>2) then if ((s[1]='%') and (s[Length(s)]='%')) then begin
    s:=AnsiUpperCase(System.Copy(s,2,Length(s)-2));
    i:=FOwner.AniGifIndex(s);
    if (i>=0) then begin
      FImage.Assign(FOwner.AniGifs[i]);
      FStdAniGifName:=s;
    end
    else FOwner.LoadAniGif(s,Self);
  end
  else begin
    ms:=TMemoryStream.Create;
    String2Stream(s,ms);
    try FImage.LoadFromStream(ms) except end;
    ms.Free;
  end;
  if ((poAutoAnimate in FOwner.Options) and (FImage.Count > 0)) then StartBlink(0);
end;

procedure TPlanAniGif.Draw(ACanvas:TCanvas);
  var p1,p2:TPoint;
      Frame: TBitmap;
      TransColor: TColor;
begin
  if (FOwner.FDisableRepaintCount>0) then Exit;
  if not (FLayer.Visible and FVisible) then Exit;
  p1:=ScrPoint1;
  p2:=ScrPoint2;
  if FImage.Empty or (FImage.Count = 0) then with ACanvas do begin
    Pen.Color := FOwner.Pen.Color;
    Pen.Width := 1;
    Pen.Style := psDash;
    Pen.Mode := pmCopy;
    Brush.Assign(Self.FBrush);
    Rectangle(p1.X,p1.Y,p2.X,p2.Y);
  end
  else if ((FImage.Count>0) and (not FImage.Empty) and (FImage.ScreenWidth > 0) and (FImage.ScreenHeight > 0)) then begin
    TransColor := clNone;
    Frame:=GetFrameBitmap(FImage.FrameIndex,TransColor);
    Frame.Canvas.Lock;
    try
      if (TransColor=clNone) or (not FImage.Transparent) then ACanvas.StretchDraw(Rect(p1.X,p1.Y,p2.X,p2.Y), Frame)
      else StretchBitmapRectTransparent(ACanvas,p1.X,p1.Y,p2.X-p1.X,p2.Y-p1.Y,Bounds(0,0,Frame.Width,Frame.Height),Frame,TransColor);
    finally
      Frame.Canvas.Unlock;
    end;
    Frame.Free;
  end;

  with ACanvas do begin
    Pen.Assign(Self.FPen);
    Brush.Assign(Self.FBrush);
    if (FSelectColor<>-1) then begin
      Pen.Color:=FSelectColor;
      Pen.Width:=2;
      Brush.Color:=FOwner.Color;
      Brush.Style:=bsClear;
      Rectangle(p1.X{-2},p1.Y{-2},p2.X{+2},p2.Y{+2});
    end;
    if (FSelected and (not (FOwner.ReadOnly or FLayer.ReadOnly or FReadOnly))) then begin
      Pen.Color:=FOwner.SelectColor;
      Pen.Width:=1;
      Brush.Color:=FOwner.SelectColor;
      Brush.Style:=bsSolid;
      Rectangle(p1.X-MarginDelta,p1.Y-MarginDelta,p1.X+MarginDelta,p1.Y+MarginDelta);
      Rectangle(p2.X-MarginDelta,p1.Y-MarginDelta,p2.X+MarginDelta,p1.Y+MarginDelta);
      Rectangle(p2.X-MarginDelta,p2.Y-MarginDelta,p2.X+MarginDelta,p2.Y+MarginDelta);
      Rectangle(p1.X-MarginDelta,p2.Y-MarginDelta,p1.X+MarginDelta,p2.Y+MarginDelta);
    end;
  end;
end;

procedure TPlanAniGif.Animate;
begin
  StartBlink(0);
end;

procedure TPlanAniGif.StartBlink(const Interval:Integer);
  var intv:Integer;
begin
  StopBlink;
  if ((FImage.Count=0) or FImage.Empty) then Exit;
  intv:=GetDelayTime(0);
  FBlinkTimerHandle:=FOwner.FTimerList.Add(BlinkTimer,intv,True{False});
  FOwner.FTimerList.Activate;
end;

procedure TPlanAniGif.StopBlink;
begin
  if (FBlinkTimerHandle=0) then Exit;
  FImage.FrameIndex:=0;
  FOwner.FTimerList.Delete(FBlinkTimerHandle);
  FBlinkType:=btNone;
  FBlinkTimerHandle:=0;
  if FVisible then NeedRepaint;
end;

procedure TPlanAniGif.Stop;
begin
  StopBlink;
end;

{$ENDIF}
//{$ENDIF}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//    TOxygenPlan
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
constructor TOxygenPlan.Create(AOwner: TComponent);
begin
  Layers := TPlanLayers.Create(Self);
  FCurLayer := Layers[Layers.Add(sDefaultLayer)];
  FPen:=TPen.Create;
  FBrush:=TBrush.Create;
  Pen.Color := clWhite;
  Pen.Style := psSolid;
  Pen.Width := 1;
  Brush.Color := clBlack;
  Brush.Style := bsSolid;
  inherited Create(AOwner);
  inherited Caption := '';
  inherited Color := clBlack;
  Font.Color := clWhite;
  {$IFNDEF VER100}
  DoubleBuffered := True;
  {$ENDIF}
  FCurDrawingObject := nil;
  FCurDrawingPointNum := 0;
  FBackgroundPicture:=TPicture.Create;
  FCurX:=0;
  FCurY:=0;
  FActiveObject:=nil;
  FActiveMarginType:=mtNone;
  FObjectClick:=nil;
  FObjectDblClick:=nil;
  FObjectMouseMove:=nil;
  FObjectMouseEnter:=nil;
  FObjectMouseLeave:=nil;
  FAfterDrawNewObject:=nil;
  FParseProgress := nil;
  FNeedPicture:=nil;
{$IFDEF USE_RX}
  FNeedAniGif:=nil;
{$ENDIF}
  FReadOnly:=False;
  FNewClickNum:=-1;
  FPlanStrings:=TStringList.Create;
  FSelectColor:=clSilver;
  NewObjUM:=nil;
  OldObjUM:=nil;
  FDisableRepaintCount:=0;
{$IFDEF USE_RX}
  FTimerList:=TRxTimerList.Create(Self);
  FStandardAniGifsList:=TList.Create;
{$ENDIF}
  FStandardPicturesList:=TList.Create;
  FOptions := [poAutoAnimate];
  FReadyToMoving := False;
end;

destructor TOxygenPlan.Destroy;
begin
  Destroying;
  Layers.Clear;
  //Layers.Free;
  FPen.Free;
  FBrush.Free;
  FPlanStrings.Free;
  FBackgroundPicture.Free;
{$IFDEF USE_RX}
  FTimerList.Free;
  ClearAniGifs;
  FStandardAniGifsList.Free;
{$ENDIF}
  ClearPictures;
  FStandardPicturesList.Free;
  inherited Destroy;
end;

function TOxygenPlan.AddPicture(const PictureName:String; Picture:TPicture):Integer;
  var pp:TPlanStandardPicture;
begin
  pp:=TPlanStandardPicture.Create;
  pp.Assign(Picture);
  pp.Name:=PictureName;
  pp.OnChange := StdPictureChange;
  Result:=FStandardPicturesList.Add(pp);
end;

{$IFDEF USE_RX}
function TOxygenPlan.AddAniGif(const AniGifName:String; AniGif:TGIFImage):Integer;
  var pag:TPlanStandardAniGif;
begin
  pag:=TPlanStandardAniGif.Create;
  pag.Assign(AniGif);
  pag.Name:=AniGifName;
  pag.OnChange := StdAniGifChange;
  Result:=FStandardAniGifsList.Add(pag);
end;

function TOxygenPlan.AniGifIndex(const AniGifName:String):Integer;
  var i:Integer;
begin
  Result:=-1;
  for i:=1 to FStandardAniGifsList.Count do
    if (AnsiUpperCase(TPlanStandardAniGif(FStandardAniGifsList[i-1]).Name)=AnsiUpperCase(AniGifName)) then begin
      Result:=i-1;
      Exit;
    end;
end;
{$ENDIF}

procedure TOxygenPlan.Assign(Source : TPersistent);
  var src:TOxygenPlan;
      i:Integer;
      sp : TPlanStandardPicture;
{$IFDEF USE_RX}
      sa : TPlanStandardAniGif;
{$ENDIF}
      sl : TStringList;
      po : TPlanOptions;
begin
  DisableRepaint;
  sl := TStringList.Create;
  if (Source is TOxygenPlan) then try
    src := (Source as TOxygenPlan);
    Clear;
    FBackgroundPicture.Assign(src.BackgroundPicture);
    for i:=1 to src.PictureCount do begin
      sp := src.Pictures[i-1];
      AddPicture(sp.Name,sp);
    end;
{$IFDEF USE_RX}
    for i:=1 to src.AniGifCount do begin
      sa := src.AniGifs[i-1];
      AddAniGif(sa.Name,sa);
    end;
{$ENDIF}
{$IFNDEF VER100}
    DoubleBuffered := src.DoubleBuffered;
{$ENDIF}
    ScaleX := src.ScaleX;
    ScaleY := src.ScaleY;
    Options := src.Options;
    Brush.Assign(src.Brush);
    Pen.Assign(src.Pen);
    Font.Assign(src.Font);
    SelectColor := src.SelectColor;
    po := src.Options;
    src.Options := [];
    src.SaveTo(sl);
    src.Options := po;
    Parse(sl);
    try CurrentLayer := Layers.LayerByName(src.CurrentLayer.Name) except end;
  except end;
  EnableRepaint;
  sl.Free;
end;

procedure TOxygenPlan.ReadStringData(Reader: TReader);
begin
  Reader.ReadString;
end;

procedure TOxygenPlan.StdPictureChange(Sender : TObject);
  var l,i : Integer;
      pp : TPlanPicture;
      sp : TPlanStandardPicture;
begin
  sp := (Sender as TPlanStandardPicture);
  DisableRepaint;
  try
    for l := 1 to Layers.Count do
      for i := 1 to Layers[l-1].Count do
        if (Layers[l-1].Objects[i-1] is TPlanPicture) then begin
          pp := (Layers[l-1].Objects[i-1] as TPlanPicture);
          if (AnsiUpperCase(pp.StdPictureName) = AnsiUpperCase(sp.Name)) then pp.Picture.Assign(sp);
        end;
  finally
    EnableRepaint;
  end;
end;

{$IFDEF USE_RX}
procedure TOxygenPlan.StdAniGifChange(Sender : TObject);
  var l,i : Integer;
      pp : TPlanAniGif;
      sp : TPlanStandardAniGif;
begin
  sp := (Sender as TPlanStandardAniGif);
  DisableRepaint;
  try
    for l := 1 to Layers.Count do
      for i := 1 to Layers[l-1].Count do
        if (Layers[l-1].Objects[i-1] is TPlanAniGif) then begin
          pp := (Layers[l-1].Objects[i-1] as TPlanAniGif);
          if (AnsiUpperCase(pp.StdAniGifName) = AnsiUpperCase(sp.Name)) then pp.GIFImage.Assign(sp);
        end;
  finally
    EnableRepaint;
  end;
end;
{$ENDIF}

procedure TOxygenPlan.SetOptions(Value : TPlanOptions);
begin
  if (Value <> FOptions) then FOptions := Value;
  Repaint;
end;

function TOxygenPlan.GetObjectCount : Integer;
  var i : Integer;
begin
  Result := 0;
  for i := 1 to Layers.Count do Inc(Result, Layers[i-1].Count);
end;

procedure TOxygenPlan.SetBrush(Value : TBrush);
begin
  FBrush.Assign(Value);
end;

procedure TOxygenPlan.SetPen(Value : TPen);
begin
  FPen.Assign(Value);
end;

procedure TOxygenPlan.DefineProperties(Filer: TFiler);
begin
  if Filer is TReader then inherited DefineProperties(Filer);
  Filer.DefineProperty('PlanStrings', ReadStringData, nil, False);
end;

procedure TOxygenPlan.Loaded;
begin
  inherited Loaded;
  Parse(FPlanStrings);
end;

procedure TOxygenPlan.SetActiveObject(PlanObject:TPlanObject);
  var i:Integer;
      po:TPlanObject;
begin
  //if (FActiveObject <> nil) then FActiveObject.Unselect;
  FActiveObject := PlanObject;
  i:=CurrentLayer.IndexOf(PlanObject);
  if (i=-1) then Exit;
  po:=CurrentLayer.Objects[i];
  CurrentLayer.FObjectList.Delete(i);
  CurrentLayer.FObjectList.Add(po);
end;

function TOxygenPlan.GetPlanStrings:TStrings;
begin
  SaveTo(FPlanStrings);
  Result:=FPlanStrings;
end;

procedure TOxygenPlan.SetPlanStrings(Value:TStrings);
begin
  FPlanStrings.Clear;
  FPlanStrings.AddStrings(Value);
  Parse(FPlanStrings);
  Repaint;
end;

procedure TOxygenPlan.SetBackgroundPicture(Value:TPicture);
begin
  FBackgroundPicture.Assign(Value);
  Repaint;
end;

procedure TOxygenPlan.SetScaleX(Value:Integer);
begin
  if (Value>0) then FScaleX:=Value;
  Resize;
end;

procedure TOxygenPlan.SetScaleY(Value:Integer);
begin
  if (Value>0) then FScaleY:=Value;
  Resize;
end;

procedure TOxygenPlan.SetReadOnly(Value:Boolean);
begin
  FReadOnly:=Value;
  if Value then if Assigned(FActiveObject) then FActiveObject.UnSelect;
end;

function TOxygenPlan.GetBackgroundPictureType:String;
begin
  Result:=GetPictureType(FBackgroundPicture);
end;

procedure TOxygenPlan.SetCurLayer(Value:TPlanLayer);
begin
  if Assigned(FActiveObject) then FActiveObject.UnSelect;
  FCurLayer:=Value;
  Repaint;
end;

function TOxygenPlan.GetPicture(Index:Integer):TPlanStandardPicture;
begin
  Result:=nil;
  if (Index>=FStandardPicturesList.Count) then Exit;
  Result:=TPlanStandardPicture(FStandardPicturesList[Index]);
end;

procedure TOxygenPlan.ClearPictures;
begin
  DisableRepaint;
  while (FStandardPicturesList.Count>0) do DeletePicture(0);
  EnableRepaint;
end;

{$IFDEF USE_RX}
function TOxygenPlan.GetAniGif(Index:Integer):TPlanStandardAniGif;
begin
  Result:=nil;
  if (Index>=FStandardAniGifsList.Count) then Exit;
  Result:=TPlanStandardAniGif(FStandardAniGifsList[Index]);
end;

procedure TOxygenPlan.ClearAniGifs;
begin
  DisableRepaint;
  while (FStandardAniGifsList.Count>0) do DeleteAniGif(0);
  EnableRepaint;
end;

function TOxygenPlan.GetAniGifCount : Integer;
begin
  Result:=Self.FStandardAniGifsList.Count;
end;
{$ENDIF}

function TOxygenPlan.GetPictureCount : Integer;
begin
  Result:=FStandardPicturesList.Count;
end;

procedure TOxygenPlan.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
  var mt:TMarginType;
      p:TPlanObject;
begin
  inherited MouseDown(Button,Shift,X,Y);
  if Assigned(FActiveObject) then FActiveObject.UnSelect;
  if FReadOnly or CurrentLayer.ReadOnly then begin
    FActiveObject := ObjectUnderMouse(mt);
    FActiveMarginType:=mtNone;
    Exit;
  end;
  if (FCurDrawingObject<>nil) then begin
    if (FCurDrawingPointNum=0) then begin
      p:=ObjectUnderMouse(mt);
      if (p<>nil) then begin
        FCurDrawingObject.FParent:=p;
        FCurDrawingObject.FParent.FChildren.Add(FCurDrawingObject);
      end;
      FCurDrawingObject.FVisible:=True;
      FCurDrawingObject.FLayer.Add(FCurDrawingObject);
      FCurDrawingObject.Properties.Add('Layer='+IntToStr(FCurDrawingObject.FLayer.Index));
    end;
    FActiveObject:=FCurDrawingObject;
    if FCurDrawingObject.AddPoint(X,Y,chMouseDown) then Inc(FCurDrawingPointNum)
    else begin
      FCurDrawingObject.FillObjectProperties;
      FCurDrawingObject.Select;
      DoAfterDrawNewObject(FCurDrawingObject);
      FCurDrawingObject:=nil;
      FCurDrawingPointNum:=0;
    end;
  end
  else begin
    FActiveObject:=ObjectUnderMouse(mt);
    FActiveMarginType:=mt;
    if (FActiveObject<>nil) then begin
      FStartCoord:=Point(X,Y);
      FActiveObject.Select;
      FReadyToMoving := (ssLeft in Shift);
    end;
  end;
end;

procedure TOxygenPlan.MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
  var po:TPlanObject;
      mt : TMarginType;
begin
  if (FCurDrawingObject<>nil) then begin
    if not FCurDrawingObject.AddPoint(X,Y,chMouseUp) then begin
      FCurDrawingObject.FillObjectProperties;
      FCurDrawingObject.Select;
      DoAfterDrawNewObject(FCurDrawingObject);
      FCurDrawingObject:=nil;
      FCurDrawingPointNum:=0;
      Repaint;
    end;
  end;
  po:=ObjectUnderMouse(mt);
  if (po<>nil) then DoObjectClick(po,Button,Shift,X,Y);
  FReadyToMoving := False;
  inherited MouseUp(Button,Shift,X,Y);
end;

procedure TOxygenPlan.MouseMove(Shift: TShiftState; X, Y: Integer);
  var mt : TMarginType;
begin
  inherited MouseMove(Shift,X,Y);
  FCurX := X;
  FCurY := Y;
  NewObjUM := ObjectUnderMouse(mt);
  if (NewObjUM <> OldObjUM) then begin
    if (OldObjUM <> nil) then DoObjectMouseLeave(OldObjUM);
    if (NewObjUM <> nil) then DoObjectMouseEnter(NewObjUM);
  end;
  if (NewObjUM <> nil) then DoObjectMouseMove(NewObjUM);
  OldObjUM := NewObjUM;
  if (FReadOnly or CurrentLayer.ReadOnly) then begin
    if (NewObjUM <> nil) then Cursor := NewObjUM.Cursor else Cursor := crDefault;
    Exit;
  end;
  if (FActiveObject <> nil) then if FActiveObject.ReadOnly then begin
    if (NewObjUM <> nil) then Cursor := NewObjUM.Cursor else Cursor := crDefault;
    Exit;
  end;
  if (FCurDrawingObject <> nil) then begin
    Cursor:=crDefault;
    if not FCurDrawingObject.AddPoint(X,Y,chMouseMove) then begin
      FCurDrawingObject.FillObjectProperties;
      FCurDrawingObject.Select;
      DoAfterDrawNewObject(FCurDrawingObject);
      FCurDrawingObject := nil;
      FCurDrawingPointNum := 0;
    end;
    Exit;
  end;
  if ((ssLeft in Shift) and (FActiveObject<>nil)) then begin
    if (FActiveMarginType <> mtNone) then FActiveObject.Resize(Point(X,Y))
    else if FReadyToMoving then FActiveObject.Move(Point(X,Y));
  end
  else begin
    if ((NewObjUM <> nil) and (FCurDrawingObject = nil)) then begin
      if not NewObjUM.ReadOnly then
        case mt of
          mtLeft, mtRight:          Cursor:=crSizeWE;
          mtTop, mtBottom:          Cursor:=crSizeNS;
          mtTopLeft, mtBottomRight: Cursor:=crSizeNWSE;
          mtTopRight, mtBottomLeft: Cursor:=crSizeNESW;
          mtMovePoint             : Cursor:=crDrag;
          else Cursor:=crDefault;
        end
      else Cursor := NewObjUM.Cursor; //crDefault;
    end
    else Cursor:=crDefault;
  end;
  FReadyToMoving := (ssLeft in Shift);
end;

procedure TOxygenPlan.DblClick;
begin
  if (FActiveObject<>nil) then DoObjectDblClick(FActiveObject);
  inherited DblClick;
end;

procedure TOxygenPlan.DoObjectClick(PlanObject:TPlanObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  if Assigned(FObjectClick) then FObjectClick(Self,PlanObject,Button,Shift,X,Y);
end;

procedure TOxygenPlan.DoObjectDblClick(PlanObject:TPlanObject);
begin
  if Assigned(FObjectDblClick) then FObjectDblClick(Self,PlanObject);
end;

procedure TOxygenPlan.DoObjectMouseMove(PlanObject:TPlanObject);
begin
  if Assigned(FObjectMouseMove) then FObjectMouseMove(Self,PlanObject);
end;

procedure TOxygenPlan.DoObjectMouseEnter(PlanObject:TPlanObject);
begin
  if Assigned(FObjectMouseEnter) then FObjectMouseEnter(Self,PlanObject);
end;

procedure TOxygenPlan.DoObjectMouseLeave(PlanObject:TPlanObject);
begin
  if Assigned(FObjectMouseLeave) then FObjectMouseLeave(Self,PlanObject);
end;

procedure TOxygenPlan.DoAfterDrawNewObject(PlanObject:TPlanObject);
begin
  if Assigned(FAfterDrawNewObject) then FAfterDrawNewObject(Self,PlanObject);
end;

procedure TOxygenPlan.DoParseProgress(const PercentsCompleted : Integer);
begin
  if Assigned(FParseProgress) then FParseProgress(Self,PercentsCompleted);
end;

{$IFDEF USE_RX}
function TOxygenPlan.DoNeedAniGif(Plan : TOxygenPlan; const AniGifName : String) : Boolean;
begin
  if Assigned(FNeedAniGif) then Result:=FNeedAniGif(Self, AniGifName) else Result:=False;
end;
{$ENDIF}

function TOxygenPlan.DoNeedPicture(Plan : TOxygenPlan; const PictureName : String) : Boolean;
begin
  if Assigned(FNeedPicture) then Result:=FNeedPicture(Self, PictureName) else Result:=False;
end;

function TOxygenPlan.ObjectUnderMouse(var MarginType:TMarginType):TPlanObject;
begin
  Result:=nil;
  if not CurrentLayer.Visible then Exit;
  Result:=CurrentLayer.ObjectFromCoord(FCurX,FCurY,MarginType);
end;

function TOxygenPlan.ObjectByID(ObjectClass : TPlanObjectClass; const ObjectID : Integer):TPlanObject;
  var i,j:Integer;
      po:TPlanObject;
begin
  Result:=nil;
  for i:=1 to Layers.Count do
    for j:=1 to Layers[i-1].Count do begin
      po:=Layers[i-1].Objects[j-1];
      if ((po is ObjectClass) and (po.ID=ObjectID)) then begin
        Result:=po;
        Exit;
      end;
    end;
end;

function TOxygenPlan.ObjectByName(ObjectClass : TPlanObjectClass; const ObjectName : String):TPlanObject;
  var i,j:Integer;
      po:TPlanObject;
begin
  Result := nil;
  for i := 1 to Layers.Count do
    for j := 1 to Layers[i-1].Count do begin
      po := Layers[i-1].Objects[j-1];
      if ((po is ObjectClass) and (UpperCase(po.Name) = UpperCase(ObjectName))) then begin
        Result := po;
        Exit;
      end;
    end;
end;

function TOxygenPlan.FindObject(ObjectClass : TPlanObjectClass; const ObjectName : String; const ObjectID : Integer) : TPlanObject;
  var i,j:Integer;
      po:TPlanObject;
begin
  Result := nil;
  for i := 1 to Layers.Count do
    for j := 1 to Layers[i-1].Count do begin
      po := Layers[i-1].Objects[j-1];
      if ((po is ObjectClass) and (UpperCase(po.Name) = UpperCase(ObjectName)) and (po.ID = ObjectID)) then begin
        Result := po;
        Exit;
      end;
    end;
end;


procedure TOxygenPlan.CancelDrawNewObject;
begin
  if (FCurDrawingObject<>nil) then try FCurDrawingObject.Free except end;
  FCurDrawingObject:=nil;
  FCurDrawingPointNum:=0;
end;

procedure TOxygenPlan.Clear;
begin
  Layers.Clear;
  ClearPictures;
{$IFDEF USE_RX}
  ClearAniGifs;
{$ENDIF}  
  FCurLayer:=Layers[Layers.Add(sDefaultLayer)];
  FPlanStrings.Clear;
  FBackgroundPicture.Assign(nil);
  Repaint;
end;

{$IFDEF USE_RX}
procedure TOxygenPlan.DeleteAniGif(Index:Integer);
begin
  if (Index>=FStandardAniGifsList.Count) then Exit;
  with TPlanStandardAniGif(FStandardAniGifsList[Index]) do try
    OnChange := nil;
    Free;
  except end;
  FStandardAniGifsList.Delete(Index);
end;
{$ENDIF}

procedure TOxygenPlan.DeletePicture(Index:Integer);
begin
  if (Index>=FStandardPicturesList.Count) then Exit;
  with TPlanStandardPicture(FStandardPicturesList[Index]) do try
    OnChange := nil;
    Free;
  except end;
  FStandardPicturesList.Delete(Index);
end;

procedure TOxygenPlan.EnableRepaint;
begin
  if (FDisableRepaintCount>0) then Dec(FDisableRepaintCount);
  if (FDisableRepaintCount=0) then Repaint;
end;

procedure TOxygenPlan.DisableRepaint;
begin
  Inc(FDisableRepaintCount);
end;

procedure TOxygenPlan.ExportToMetafile(const FileName : String); // 0.91
  var mf : TMetafile;
      mfc : TMetafileCanvas;
begin
  mf := TMetafile.Create;
  try
    mf.Width := Width;
    mf.Height := Height;
    mfc := TMetafileCanvas.Create(mf, 0);
    try
      PaintToCanvas(mfc);
    finally
      mfc.Free;
    end;
    mf.SaveToFile(FileName);
  finally
    mf.Free;
  end;
end;

procedure TOxygenPlan.ExportToBitmap(const FileName : String);  // 0.91
  var b : TBitmap;
begin
  b := TBitmap.Create;
  try
    b.Width := Width;
    b.Height := Height;
    PaintToCanvas(b.Canvas);
    b.SaveToFile(FileName);
  finally
    b.Free;
  end;
end;

procedure TOxygenPlan.ExportToJpeg(const FileName : String);  // 0.91
  var b : TBitmap;
      jpg : TJpegImage;
begin
  b := TBitmap.Create;
  jpg := TJpegImage.Create;
  try
    b.Width := Width;
    b.Height := Height;
    PaintToCanvas(b.Canvas);
    jpg.Assign(b);
    jpg.SaveToFile(FileName);
  finally
    b.Free;
    jpg.Free;
  end;
end;

{$IFDEF USE_RX}
procedure TOxygenPlan.ExportToGif(const FileName : String);  // 0.91
  var b : TBitmap;
      gif : TGifImage;
begin
  b := TBitmap.Create;
  gif := TGifImage.Create;
  try
    b.Width := Width;
    b.Height := Height;
    PaintToCanvas(b.Canvas);
    gif.Assign(b);
    gif.SaveToFile(FileName);
  finally
    b.Free;
    gif.Free;
  end;
end;
{$ENDIF}

procedure TOxygenPlan.GetBitmap(Bitmap : TBitmap); // 0.91
begin
  with Bitmap do begin
    Width := Self.Width;
    Height := Self.Height;
    PaintToCanvas(Canvas);
  end;
end;

procedure TOxygenPlan.DrawNewObject(PlanObjectClass:TPlanObjectClass);
begin
  CancelDrawNewObject;
  FCurDrawingObject:=TPlanObject(PlanObjectClass.NewInstance);
  FCurDrawingObject.Create(Self);
  FCurDrawingObject.FVisible:=False;
end;

{$IFDEF USE_RX}
procedure TOxygenPlan.LoadAniGif(const AniGifName : String; PlanAniGif : TPlanAniGif);
begin
  if DoNeedAniGif(Self, AniGifName) then PlanAniGif.StdAniGifName:=AniGifName;
end;
{$ENDIF}

procedure TOxygenPlan.LoadPicture(const PictureName : String; PlanPicture : TPlanPicture);
begin
  if DoNeedPicture(Self, PictureName) then PlanPicture.StdPictureName:=PictureName;
end;

procedure TOxygenPlan.Parse(PlanData:TStrings);
  label NextLine;
  var i,ps:Integer;
      ParentObjects:TList;
      s,st:String;
      l:TPlanLayer;
      CurObject:TPlanObject;
      p:TPlanStandardPicture;
      ms:TMemoryStream;
      b:Boolean;
{$IFDEF USE_RX}
      ag:TPlanStandardAniGif;
{$ENDIF}
begin
  if (PlanData.Count=0) then Exit;
  Layers.Clear;
  {ClearPictures;
  ClearAniGifs;}
  //FCurLayer := Layers[Layers.Add(sDefaultLayer)];
  //FBackgroundPicture.Assign(nil);
  CurObject:=nil;
  ParentObjects:=TList.Create;
  try
    for i:=1 to PlanData.Count do begin
      DoParseProgress(Trunc((i*100)/PlanData.Count));
      s := Trim(PlanData[i-1]);
      ps := Pos(';',s);
      if (ps > 0) then s := System.Copy(s,1,ps-1);
      if (s='') then goto NextLine;
      // -------------------------------------- Scale
      if (Pos(UpperCase(cmScale),AnsiUpperCase(s))=1) then begin
        s:=Copy(s,Length(cmScale)+1,255);
        if (Pos(',',s)>0) then try
          FScaleX:=StrToInt(Copy(s,1,Pos(',',s)-1));
          FScaleY:=StrToInt(Copy(s,Pos(',',s)+1,255));
          if (FScaleX<=0) then FScaleX:=500;
          if (FScaleY<=0) then FScaleY:=500;
        except end;
      end
      // -------------------------------------- SetPenColor
      else if (Pos(UpperCase(cmSetPenColor),AnsiUpperCase(s))=1) then begin
        s := Copy(s,Length(cmSetPenColor)+1,1000);
        Pen.Color := ReadColor(s);
        Canvas.Pen.Color := Pen.Color;
      end
      // -------------------------------------- SetBrushColor
      else if (Pos(UpperCase(cmSetBrushColor),AnsiUpperCase(s))=1) then begin
        s := Copy(s,Length(cmSetBrushColor)+1,1000);
        Brush.Color := ReadColor(s);
        Canvas.Brush.Color := Brush.Color;
      end
      // -------------------------------------- SetLineWidth
      else if (Pos(UpperCase(cmSetLineWidth),AnsiUpperCase(s))=1) then begin
        s := Copy(s,Length(cmSetLineWidth)+1,1000);
        try Pen.Width := StrToInt(s) except end;
        if (Pen.Width = 0) then Pen.Width := 1;
      end
      // -------------------------------------- SetFont
      else if (Pos(UpperCase(cmSetFont),AnsiUpperCase(s))=1) then begin
        s:=Copy(s,Length(cmSetFont)+1,1000);
        ps:=Pos(',',s);
        if (ps>0) then try
          Font.Name:=Copy(s,1,ps-1);
          s:=Copy(s,ps+1,255);
          ps:=Pos(',',s);
          if (ps=0) then ps:=Length(s)+1;
          Font.Size:=StrToInt(Copy(s,1,ps-1));
          Font.Style:=[];
          s:=Copy(s,ps+1,255);
          if (s<>'') then begin
            if (Pos('I',s)>0) then with Font do Style:=Style+[fsItalic];
            if (Pos('B',s)>0) then with Font do Style:=Style+[fsBold];
            if (Pos('U',s)>0) then with Font do Style:=Style+[fsUnderline];
            if (Pos('S',s)>0) then with Font do Style:=Style+[fsStrikeOut];
          end;
        except end;
      end
      // -------------------------------------- BackgroundPicture
      else if (Pos(UpperCase(cmBackgroundPicture),AnsiUpperCase(s))=1) then begin
        System.Delete(s,1,Length(cmBackgroundPicture));
        if (Pos(',',s)>0) then try
          st := Copy(s,1,Pos(',',s)-1); // format
          System.Delete(s,1,Pos(',',s));
          if (Pos(',',s)>0) then begin
            b:=(Copy(s,1,Pos(',',s)-1)<>'0');
            System.Delete(s,1,Pos(',',s));
          end
          else b:=False;
          ms:=TMemoryStream.Create;
          try
            String2Stream(s,ms);
            LoadPictureFromStream(st, FBackgroundPicture, ms);
          except end;
          ms.Free;
          if not ((FBackgroundPicture.Graphic=nil) or FBackgroundPicture.Graphic.Empty) then FBackgroundPicture.Graphic.Transparent:=b;
        except end;
      end
      // -------------------------------------- StandardPicture
      else if (Pos(UpperCase(cmStandardPicture),AnsiUpperCase(s))=1) then begin
        System.Delete(s,1,Length(cmStandardPicture));
        p:=TPlanStandardPicture.Create;
        p.Assign(nil);
        if (Pos(',',s)>0) then try
          p.Name:=Copy(s,1,Pos(',',s)-1);
          System.Delete(s,1,Pos(',',s));
          if (Pos(',',s)>0) then try
            st:=Copy(s,1,Pos(',',s)-1);
            System.Delete(s,1,Pos(',',s));
            if (Pos(',',s)>0) then begin
              b:=(Copy(s,1,Pos(',',s)-1)<>'0');
              System.Delete(s,1,Pos(',',s));
            end
            else b:=False;
            ms:=TMemoryStream.Create;
            try
              String2Stream(s,ms);
              LoadPictureFromStream(st, p, ms);
            except end;
            ms.Free;
            if not ((p.Graphic=nil) or p.Graphic.Empty) then p.Graphic.Transparent:=b;
            AddPicture(p.Name,p);
          except end;
        except end;
        p.Free;
      end
{$IFDEF USE_RX}
      // -------------------------------------- StandardAniGif
      else if (Pos(UpperCase(cmStandardAniGif),AnsiUpperCase(s))=1) then begin
        System.Delete(s,1,Length(cmStandardAniGif));
        ag:=TPlanStandardAniGif.Create;
        ag.Assign(nil);
        if (Pos(',',s)>0) then try
          ag.Name:=Copy(s,1,Pos(',',s)-1);
          System.Delete(s,1,Pos(',',s));
          ms:=TMemoryStream.Create;
          try
            String2Stream(s,ms);
            ag.LoadFromStream(ms)
          except end;
          ms.Free;
          AddAniGif(ag.Name,ag);
        except end;
        ag.Free;
      end
{$ENDIF}
      // -------------------------------------- Layer
      else if (Pos(UpperCase(cmLayer),AnsiUpperCase(s))=1) then begin
        s:=Copy(s,Length(cmLayer)+1,255);
        ps:=Pos(',',s);
        if (ps>0) then try
          st:=Copy(s,1,ps-1);
          l:=Layers.LayerByName(st);
          if (l=nil) then l:=Layers[Layers.Add(st)];
          if (Layers.Count = 1) then FCurLayer := l; 
          s:=Copy(s,ps+1,255);
          ps:=Pos(',',s);
          if (ps=0) then ps:=Length(s)+1;
          l.Visible:=(Copy(s,1,ps-1)<>'0');
          s:=Copy(s,ps+1,255);
          if (s<>'') then begin // ReadOnly
            ps:=Pos(',',s);
            if (ps=0) then ps:=Length(s)+1;
            l.ReadOnly:=(Copy(s,1,ps-1)<>'0');
          end;
        except end;
      end
      // -------------------------------------- Start Object
      else if (Pos(UpperCase(sStartObject),AnsiUpperCase(s))=1) then begin
        if (Layers.Count = 0) then FCurLayer := Layers[Layers.Add(sDefaultLayer)];
        if (CurObject<>nil) then CurObject.UpdateObjectProperties;
        s:=Copy(s,Length(sStartObject)+1,255);
        if (AnsiUpperCase(s)=UpperCase(ocLine)) then CurObject:=TPlanLine.Create(Self)
        else if (AnsiUpperCase(s)=UpperCase(ocBox)) then CurObject:=TPlanBox.Create(Self)
        else if (AnsiUpperCase(s)=UpperCase(ocCircle)) then CurObject:=TPlanCircle.Create(Self)
        else if (AnsiUpperCase(s)=UpperCase(ocText)) then CurObject:=TPlanText.Create(Self)
        else if (AnsiUpperCase(s)=UpperCase(ocPicture)) then CurObject:=TPlanPicture.Create(Self)
        else if (AnsiUpperCase(s)=UpperCase(ocPolygon)) then CurObject:=TPlanPolygon.Create(Self)
        else if (AnsiUpperCase(s)=UpperCase(ocCompositeObject)) then CurObject:=TPlanCompositeObject.Create(Self)
{$IFDEF USE_RX}
        else if (AnsiUpperCase(s)=UpperCase(ocAniGif)) then CurObject:=TPlanAniGif.Create(Self)
        else if (AnsiUpperCase(s)=UpperCase(ocAniPicture)) then CurObject:=TPlanAniPicture.Create(Self)
{$ENDIF}
        else raise EOxygenPlanException.Create('Invalid object: '+s);
        if (CurObject <> nil) then begin
          if (ParentObjects.Count=0) then CurObject.FParent:=nil else begin
            CurObject.FParent:=TPlanObject(ParentObjects[ParentObjects.Count-1]);
            CurObject.FParent.FChildren.Add(CurObject);
          end;
          ParentObjects.Add(CurObject);
        end;
      end
      // -------------------------------------- End Object
      else if (Pos(UpperCase(sEndObject),AnsiUpperCase(s))=1) then begin
        if (ParentObjects.Count > 0) then ParentObjects.Delete(ParentObjects.Count-1);
      end
      else if (CurObject<>nil) then CurObject.Properties.Add(s);
NextLine:
    end;
  finally
    DoParseProgress(100);
    if (Layers.Count = 0) then FCurLayer := Layers[Layers.Add(sDefaultLayer)];
    if (CurObject<>nil) then CurObject.UpdateObjectProperties;
    ParentObjects.Free;
  end;
end;

function TOxygenPlan.PictureIndex(const PictureName:String):Integer;
  var i:Integer;
begin
  Result:=-1;
  for i:=1 to FStandardPicturesList.Count do
    if (AnsiUpperCase(TPlanStandardPicture(FStandardPicturesList[i-1]).Name)=AnsiUpperCase(PictureName)) then begin
      Result:=i-1;
      Exit;
    end;
  //if (Result = -1) then
end;

procedure TOxygenPlan.LoadFromFile(FileName:TFileName);
begin
  if not FileExists(FileName) then begin
    raise EOxygenPlanException.Create(Format(sFileNotFound,[FileName]));
    Exit;
  end;
  DisableRepaint;
  try
    Clear;
    (FPlanStrings as TStringList).LoadFromFile(FileName);
    Parse(FPlanStrings);
  finally
    EnableRepaint;
  end;
end;

procedure TOxygenPlan.SaveTo(PlanData:TStrings);
  var i,j:Integer;
      po:TPlanObject;
      s:String;
begin
  PlanData.Clear;
  PlanData.Add(cmScale+IntToStr(ScaleX)+','+IntToStr(ScaleY));
  PlanData.Add(cmSetPenColor+EncodeColor(Pen.Color));
  PlanData.Add(cmSetBrushColor+EncodeColor(Brush.Color));
  PlanData.Add(cmSetLineWidth+IntToStr(Pen.Width));
  if (poSaveBackgroundPicture in Options) then if (FBackgroundPicture <> nil) then
    if (FBackgroundPicture.Graphic <> nil) then if not FBackgroundPicture.Graphic.Empty then begin
      s := Trim(GetPictureStreamString(FBackgroundPicture));
      if (s <> '') then PlanData.Add(cmBackgroundPicture + s);
    end;
  Layers.AddLayersDescr(PlanData);
  if (poSaveStandardPictures in Options) then begin
    for i:=1 to FStandardPicturesList.Count do begin
      s:=TPlanStandardPicture(FStandardPicturesList[i-1]).GetDescrString;
      if (s<>'') then PlanData.Add(s);
    end;
{$IFDEF USE_RX}
    for i:=1 to FStandardAniGifsList.Count do begin
      s:=TPlanStandardAniGif(FStandardAniGifsList[i-1]).GetDescrString;
      if (s<>'') then PlanData.Add(s);
    end;
{$ENDIF}
  end;
  for i:=1 to Layers.Count do
    for j:=1 to Layers[i-1].Count do begin
      po := Layers[i-1].Objects[j-1];
      if (po.FParent=nil) then po.AddObjectDescr(0,PlanData);
    end;
end;

procedure TOxygenPlan.SaveToFile(FileName:String);
begin
  SaveTo(FPlanStrings);
  (FPlanStrings as TStringList).SaveToFile(FileName);
end;

procedure TOxygenPlan.LoadFromStream(Stream:TStream);
begin
  DisableRepaint;
  try
    Clear;
    (FPlanStrings as TStringList).LoadFromStream(Stream);
    Parse(FPlanStrings);
  finally
    EnableRepaint;
  end;
end;

procedure TOxygenPlan.SaveToStream(Stream:TStream);
begin
  SaveTo(FPlanStrings);
  (FPlanStrings as TStringList).SaveToStream(Stream);
end;

procedure TOxygenPlan.Resize;
  var i,j:Integer;
begin
  if (csDestroying in ComponentState) then Exit;
  inherited Resize;
  for i:=1 to Layers.Count do
    for j:=1 to Layers[i-1].Count do Layers[i-1].Objects[j-1].ParentResized;
  Repaint;
end;

procedure TOxygenPlan.PaintToCanvas(ACanvas : TCanvas);
  var i:Integer;
      l:TPlanLayer;
begin
  if ((csDestroying in ComponentState) or (FDisableRepaintCount>0)) then Exit;
  ACanvas.Lock;
  ACanvas.Font := Font;
  ACanvas.Pen := Pen;
  ACanvas.Brush := Brush;
  ACanvas.Brush.Color := Color;
  ACanvas.Brush.Style := bsSolid;
  try
    ACanvas.FillRect(ACanvas.ClipRect);
    if (FBackgroundPicture.Graphic<>nil) then if not FBackgroundPicture.Graphic.Empty then ACanvas.StretchDraw(Rect(0,0,Width,Height),FBackgroundPicture.Graphic);
    if (Layers.Count>0) then begin
      for i:=1 to Layers.Count do begin
        l:=Layers[i-1];
        if ((l<>CurrentLayer) and (l.Visible)) then l.Draw(ACanvas);
      end;
      if CurrentLayer.Visible then CurrentLayer.Draw(ACanvas);
    end;
  finally
    ACanvas.Unlock;
  end;
end;

procedure TOxygenPlan.Paint;
begin
  PaintToCanvas(Canvas);
end;

procedure Register;
begin
  RegisterComponents('Oxygen', [TOxygenPlan]);
end;

end.




