//------------------------------------------------------------------------------
//  MSTreeView - 1.00
//  by Jeff Chang
//  All Rights Reserved (c) 1997
//
//  Description:
//    A TreeView that permits multi-selection like
//    the ListView
//------------------------------------------------------------------------------

unit MSTreeView;

{$R-}

interface

uses Messages, Windows, SysUtils, CommCtrl, Classes, Controls, Forms, StdCtrls,
  Dialogs, Graphics;

type
{ TMSTreeNode }

  TMSCustomTreeView = class;
  TMSTreeNodes = class;

  TNodeState = (nsCut, nsDropHilited, nsFocused, nsSelected, nsExpanded);
  TNodeAttachMode = (naAdd, naAddFirst, naAddChild, naAddChildFirst, naInsert);
  TAddMode = (taAddFirst, taAdd, taInsert);

  PNodeInfo = ^TNodeInfo;
  TNodeInfo = packed record
    // multiselect
    Order: Extended;

    ImageIndex: Integer;
    SelectedIndex: Integer;
    StateIndex: Integer;
    OverlayIndex: Integer;
    Data: Pointer;
    Count: Integer;
    Text: string[255];
  end;

  TMSTreeNode = class(TPersistent)
  private
    FOwner: TMSTreeNodes;
    FText: string;
    FData: Pointer;
    FItemId: HTreeItem;
    FImageIndex: Integer;
    FSelectedIndex: Integer;
    FOverlayIndex: Integer;
    FStateIndex: Integer;
    FDeleting: Boolean;
    FInTree: Boolean;
    // multiselect
    FOrder: Extended;

    function CompareCount(CompareMe: Integer): Boolean;
    function DoCanExpand(Expand: Boolean): Boolean;
    procedure DoExpand(Expand: Boolean);
    procedure ExpandItem(Expand: Boolean; Recurse: Boolean);
    function GetAbsoluteIndex: Integer;
    function GetExpanded: Boolean;
    function GetLevel: Integer;
    function GetParent: TMSTreeNode;
    function GetChildren: Boolean;
    function GetCut: Boolean;
    function GetDropTarget: Boolean;
    function GetFocused: Boolean;
    function GetIndex: Integer;
    function GetItem(Index: Integer): TMSTreeNode;
    function GetSelected: Boolean;
    function GetState(NodeState: TNodeState): Boolean;
    function GetCount: Integer;
    function GeTMSTreeView: TMSCustomTreeView;
    procedure InternalMove(ParentNode, Node: TMSTreeNode; HItem: HTreeItem;
      AddMode: TAddMode);
    function IsEqual(Node: TMSTreeNode): Boolean;
    function IsNodeVisible: Boolean;
    procedure ReadData(Stream: TStream; Info: PNodeInfo);
    procedure SetChildren(Value: Boolean);
    procedure SetCut(Value: Boolean);
    procedure SetData(Value: Pointer);
    procedure SetDropTarget(Value: Boolean);
    procedure SetItem(Index: Integer; Value: TMSTreeNode);
    procedure SetExpanded(Value: Boolean);
    procedure SetFocused(Value: Boolean);
    procedure SetImageIndex(Value: Integer);
    procedure SetOverlayIndex(Value: Integer);
    procedure SetSelectedIndex(Value: Integer);
    procedure SetSelected(Value: Boolean);
    procedure SetStateIndex(Value: Integer);
    procedure SetText(const S: string);
    procedure WriteData(Stream: TStream; Info: PNodeInfo);
    // multiselect
    procedure SetOrder(Value: Extended);
    function GetTotalCount: Integer;
    procedure OrderChildren(Node,NextNode: TMSTreeNode; PrevOrder,Incr: Extended);
  protected
    // multiselect
    property Order: Extended read FOrder write SetOrder;
  public
    constructor Create(AOwner: TMSTreeNodes);
    destructor Destroy; override;
    function AlphaSort: Boolean;
    procedure Assign(Source: TPersistent); override;
    procedure Collapse(Recurse: Boolean);
    function CustomSort(SortProc: TTVCompare; Data: Longint): Boolean;
    procedure Delete;
    procedure DeleteChildren;
    function DisplayRect(TextOnly: Boolean): TRect;
    function EditText: Boolean;
    procedure EndEdit(Cancel: Boolean);
    procedure Expand(Recurse: Boolean);
    function GetFirstChild: TMSTreeNode;
    function GetHandle: HWND;
    function GetLastChild: TMSTreeNode;
    function GetNext: TMSTreeNode;
    function GetNextChild(Value: TMSTreeNode): TMSTreeNode;
    function GetNextSibling: TMSTreeNode;
    function GetNextVisible: TMSTreeNode;
    function GetPrev: TMSTreeNode;
    function GetPrevChild(Value: TMSTreeNode): TMSTreeNode;
    function GetPrevSibling: TMSTreeNode;
    function GetPrevVisible: TMSTreeNode;
    function HasAsParent(Value: TMSTreeNode): Boolean;
    function IndexOf(Value: TMSTreeNode): Integer;
    procedure MakeVisible;
    procedure MoveTo(Destination: TMSTreeNode; Mode: TNodeAttachMode); virtual;
    // multiselect
    function IsBold: Boolean;
    function IsSelected: Boolean;
    procedure MakeBold(Value: Boolean);
    procedure MakeFocused(Value: Boolean);
    procedure MakeSelected(Value: Boolean);

    property AbsoluteIndex: Integer read GetAbsoluteIndex;
    property Count: Integer read GetCount;
    property Cut: Boolean read GetCut write SetCut;
    property Data: Pointer read FData write SetData;
    property Deleting: Boolean read FDeleting;
    property Focused: Boolean read GetFocused write SetFocused;
    property DropTarget: Boolean read GetDropTarget write SetDropTarget;
    property Selected: Boolean read GetSelected write SetSelected;
    property Expanded: Boolean read GetExpanded write SetExpanded;
    property Handle: HWND read GetHandle;
    property HasChildren: Boolean read GetChildren write SetChildren;
    property ImageIndex: Integer read FImageIndex write SetImageIndex;
    property Index: Integer read GetIndex;
    property IsVisible: Boolean read IsNodeVisible;
    property Item[Index: Integer]: TMSTreeNode read GetItem write SetItem; default;
    property ItemId: HTreeItem read FItemId;
    property Level: Integer read GetLevel;
    property OverlayIndex: Integer read FOverlayIndex write SetOverlayIndex;
    property Owner: TMSTreeNodes read FOwner;
    property Parent: TMSTreeNode read GetParent;
    property SelectedIndex: Integer read FSelectedIndex write SetSelectedIndex;
    property StateIndex: Integer read FStateIndex write SetStateIndex;
    property Text: string read FText write SetText;
    property TreeView: TMSCustomTreeView read GeTMSTreeView;
  end;

{ TMSTreeNodes }

  TMSTreeNodes = class(TPersistent)
  private
    FOwner: TMSCustomTreeView;
    FUpdateCount: Integer;
    procedure AddedNode(Value: TMSTreeNode);
    function GetHandle: HWND;
    function GetNodeFromIndex(Index: Integer): TMSTreeNode;
    procedure ReadData(Stream: TStream);
    procedure Repaint(Node: TMSTreeNode);
    procedure WriteData(Stream: TStream);
    // multiselect
    procedure ReOrder(Node: TMSTreeNode);
  protected
    function AddItem(Parent, Target: HTreeItem; const Item: TTVItem;
      AddMode: TAddMode): HTreeItem;
    function InternalAddObject(Node: TMSTreeNode; const S: string;
      Ptr: Pointer; AddMode: TAddMode): TMSTreeNode;
    procedure DefineProperties(Filer: TFiler); override;
    function CreateItem(Node: TMSTreeNode): TTVItem;
    function GetCount: Integer;
    procedure SetItem(Index: Integer; Value: TMSTreeNode);
    procedure SetUpdateState(Updating: Boolean);
  public
    constructor Create(AOwner: TMSCustomTreeView);
    destructor Destroy; override;
    function AddChildFirst(Node: TMSTreeNode; const S: string): TMSTreeNode;
    function AddChild(Node: TMSTreeNode; const S: string): TMSTreeNode;
    function AddChildObjectFirst(Node: TMSTreeNode; const S: string;
      Ptr: Pointer): TMSTreeNode;
    function AddChildObject(Node: TMSTreeNode; const S: string;
      Ptr: Pointer): TMSTreeNode;
    function AddFirst(Node: TMSTreeNode; const S: string): TMSTreeNode;
    function Add(Node: TMSTreeNode; const S: string): TMSTreeNode;
    function AddObjectFirst(Node: TMSTreeNode; const S: string;
      Ptr: Pointer): TMSTreeNode;
    function AddObject(Node: TMSTreeNode; const S: string;
      Ptr: Pointer): TMSTreeNode;
    procedure Assign(Source: TPersistent); override;
    procedure BeginUpdate;
    procedure Clear;
    procedure Delete(Node: TMSTreeNode);
    procedure EndUpdate;
    function GetFirstNode: TMSTreeNode;
    function GetNode(ItemId: HTreeItem): TMSTreeNode;
    function Insert(Node: TMSTreeNode; const S: string): TMSTreeNode;
    function InsertObject(Node: TMSTreeNode; const S: string;
      Ptr: Pointer): TMSTreeNode;
    property Count: Integer read GetCount;
    property Handle: HWND read GetHandle;
    property Item[Index: Integer]: TMSTreeNode read GetNodeFromIndex; default;
    property Owner: TMSCustomTreeView read FOwner;
  end;

{ TMSCustomTreeView }

  THitTest = (htAbove, htBelow, htNowhere, htOnItem, htOnButton,
    htOnIcon, htOnIndent, htOnLabel, htOnRight,
    htOnStateIcon, htToLeft, htToRight);
  THitTests = set of THitTest;
  ETreeViewError = class(Exception);

  TTVChangingEvent = procedure(Sender: TObject; Node: TMSTreeNode;
    var AllowChange: Boolean) of object;
  TTVChangedEvent = procedure(Sender: TObject; Node: TMSTreeNode) of object;
  TTVEditingEvent = procedure(Sender: TObject; Node: TMSTreeNode;
    var AllowEdit: Boolean) of object;
  TTVEditedEvent = procedure(Sender: TObject; Node: TMSTreeNode; var S: string) of object;
  TTVExpandingEvent = procedure(Sender: TObject; Node: TMSTreeNode;
    var AllowExpansion: Boolean) of object;
  TTVCollapsingEvent = procedure(Sender: TObject; Node: TMSTreeNode;
    var AllowCollapse: Boolean) of object;
  TTVExpandedEvent = procedure(Sender: TObject; Node: TMSTreeNode) of object;
  TTVCompareEvent = procedure(Sender: TObject; Node1, Node2: TMSTreeNode;
    Data: Integer; var Compare: Integer) of object;

  TSortType = (stNone, stData, stText, stBoth);

  TMSCustomTreeView = class(TWinControl)
  private
    // multiselect
    FSelNodes: TList;
    FCtrlDown: Boolean;
    FDeleting: Boolean;
    FShiftDown: Boolean;
    FShiftKeyDown: Boolean;
    FMultiSelect: Boolean;

    FShowLines: Boolean;
    FShowRoot: Boolean;
    FShowButtons: Boolean;
    FBorderStyle: TBorderStyle;
    FReadOnly: Boolean;
    FImages: TImageList;
    FStateImages: TImageList;
    FImageChangeLink: TChangeLink;
    FStateChangeLink: TChangeLink;
    FDragImage: TImageList;
    FTreeNodes: TMSTreeNodes;
    FSortType: TSortType;
    FSaveItems: TStringList;
    FSaveTopIndex: Integer;
    FSaveIndex: Integer;
    FSaveIndent: Integer;
    FHideSelection: Boolean;
    FMemStream: TMemoryStream;
    FEditInstance: Pointer;
    FDefEditProc: Pointer;
    FEditHandle: HWND;
    FDragged: Boolean;
    FRClickNode: TMSTreeNode;
    FLastDropTarget: TMSTreeNode;
    FDragNode: TMSTreeNode;
    FManualNotify: Boolean;
    FRightClickSelect: Boolean;
    FSavedSort: TSortType;
    FStateChanging: Boolean;
    FWideText: WideString;
    FOnEditing: TTVEditingEvent;
    FOnEdited: TTVEditedEvent;
    FOnExpanded: TTVExpandedEvent;
    FOnExpanding: TTVExpandingEvent;
    FOnCollapsed: TTVExpandedEvent;
    FOnCollapsing: TTVCollapsingEvent;
    FOnChanging: TTVChangingEvent;
    FOnChange: TTVChangedEvent;
    FOnCompare: TTVCompareEvent;
    FOnDeletion: TTVExpandedEvent;
    FOnGetImageIndex: TTVExpandedEvent;
    FOnGetSelectedIndex: TTVExpandedEvent;
    procedure CMColorChanged(var Message: TMessage); message CM_COLORCHANGED;
    procedure CMCtl3DChanged(var Message: TMessage); message CM_CTL3DCHANGED;
    procedure CMDrag(var Message: TCMDrag); message CM_DRAG;
    procedure CNNotify(var Message: TWMNotify); message CN_NOTIFY;
    procedure EditWndProc(var Message: TMessage);
    procedure DoDragOver(Source: TDragObject; X, Y: Integer; CanDrop: Boolean);
    procedure GetImageIndex(Node: TMSTreeNode);
    procedure GetSelectedIndex(Node: TMSTreeNode);
    function GetDropTarget: TMSTreeNode;
    function GetIndent: Integer;
    function GetNodeFromItem(const Item: TTVItem): TMSTreeNode;
    function GetSelection: TMSTreeNode;
    function GetTopItem: TMSTreeNode;
    procedure ImageListChange(Sender: TObject);
    procedure SetBorderStyle(Value: TBorderStyle);
    procedure SetButtonStyle(Value: Boolean);
    procedure SetDropTarget(Value: TMSTreeNode);
    procedure SetHideSelection(Value: Boolean);
    procedure SetImageList(Value: HImageList; Flags: Integer);
    procedure SetIndent(Value: Integer);
    procedure SetImages(Value: TImageList);
    procedure SetLineStyle(Value: Boolean);
    procedure SetReadOnly(Value: Boolean);
    procedure SetRootStyle(Value: Boolean);
    procedure SetSelection(Value: TMSTreeNode);
    procedure SetSortType(Value: TSortType);
    procedure SetStateImages(Value: TImageList);
    procedure SetStyle(Value: Integer; UseStyle: Boolean);
    procedure SeTMSTreeNodes(Value: TMSTreeNodes);
    procedure SetTopItem(Value: TMSTreeNode);
    procedure WMLButtonDown(var Message: TWMLButtonDown); message WM_LBUTTONDOWN;
    procedure WMRButtonDown(var Message: TWMRButtonDown); message WM_RBUTTONDOWN;
    procedure WMRButtonUp(var Message: TWMRButtonUp); message WM_RBUTTONUP;
    procedure WMNotify(var Message: TWMNotify); message WM_NOTIFY;
    procedure CMSysColorChange(var Message: TMessage); message CM_SYSCOLORCHANGE;
    // multiselect
    function GetSelNode(Index: Integer): TMSTreeNode;
    function GetSelCount: Integer;
    procedure SetMultiSelect(Value: Boolean);
    procedure WMKeyDown(var Message: TWMKeyDown); message WM_KEYDOWN;
  protected
    function CanEdit(Node: TMSTreeNode): Boolean; dynamic;
    function CanChange(Node: TMSTreeNode): Boolean; dynamic;
    function CanCollapse(Node: TMSTreeNode): Boolean; dynamic;
    function CanExpand(Node: TMSTreeNode): Boolean; dynamic;
    procedure Change(Node: TMSTreeNode); dynamic;
    procedure Collapse(Node: TMSTreeNode); dynamic;
    function CreateNode: TMSTreeNode; virtual;
    procedure CreateParams(var Params: TCreateParams); override;
    procedure CreateWnd; override;
    procedure DestroyWnd; override;
    procedure DoEndDrag(Target: TObject; X, Y: Integer); override;
    procedure DoStartDrag(var DragObject: TDragObject); override;
    procedure Edit(const Item: TTVItem); dynamic;
    procedure Expand(Node: TMSTreeNode); dynamic;
    function GetDragImages: TCustomImageList; override;
    procedure Loaded; override;
    procedure Notification(AComponent: TComponent;
      Operation: TOperation); override;
    procedure SetDragMode(Value: TDragMode); override;
    procedure WndProc(var Message: TMessage); override;
    // multiselect
    procedure DoEnter; override;

    property OnEditing: TTVEditingEvent read FOnEditing write FOnEditing;
    property OnEdited: TTVEditedEvent read FOnEdited write FOnEdited;
    property OnExpanding: TTVExpandingEvent read FOnExpanding write FOnExpanding;
    property OnExpanded: TTVExpandedEvent read FOnExpanded write FOnExpanded;
    property OnCollapsing: TTVCollapsingEvent read FOnCollapsing write FOnCollapsing;
    property OnCollapsed: TTVExpandedEvent read FOnCollapsed write FOnCollapsed;
    property OnChanging: TTVChangingEvent read FOnChanging write FOnChanging;
    property OnChange: TTVChangedEvent read FOnChange write FOnChange;
    property OnCompare: TTVCompareEvent read FOnCompare write FOnCompare;
    property OnDeletion: TTVExpandedEvent read FOnDeletion write FOnDeletion;
    property OnGetImageIndex: TTVExpandedEvent read FOnGetImageIndex write FOnGetImageIndex;
    property OnGetSelectedIndex: TTVExpandedEvent read FOnGetSelectedIndex write FOnGetSelectedIndex;
    property ShowButtons: Boolean read FShowButtons write SetButtonStyle default True;
    property BorderStyle: TBorderStyle read FBorderStyle write SetBorderStyle default bsSingle;
    property ShowLines: Boolean read FShowLines write SetLineStyle default True;
    property ShowRoot: Boolean read FShowRoot write SetRootStyle default True;
    property ReadOnly: Boolean read FReadOnly write SetReadOnly default False;
    property RightClickSelect: Boolean read FRightClickSelect write FRightClickSelect default False;
    property Indent: Integer read GetIndent write SetIndent;
    property Items: TMSTreeNodes read FTreeNodes write SeTMSTreeNodes;
    property SortType: TSortType read FSortType write SetSortType default stNone;
    property HideSelection: Boolean read FHideSelection write SetHideSelection default True;
    property Images: TImageList read FImages write SetImages;
    property StateImages: TImageList read FStateImages write SetStateImages;
    property MultiSelect: Boolean read FMultiSelect write SetMultiSelect;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function AlphaSort: Boolean;
    function CustomSort(SortProc: TTVCompare; Data: Longint): Boolean;
    procedure FullCollapse;
    procedure FullExpand;
    function GetHitTestInfoAt(X, Y: Integer): THitTests;
    function GetNodeAt(X, Y: Integer): TMSTreeNode;
    function IsEditing: Boolean;
    procedure LoadFromFile(const FileName: string);
    procedure LoadFromStream(Stream: TStream);
    procedure SaveToFile(const FileName: string);
    procedure SaveToStream(Stream: TStream);
    // multiselect
    procedure AddSelNode(Node: TMSTreeNode);
    procedure ClearSelNodes;
    procedure DelSelNode(Node: TMSTreeNode);
    function FindSelNode(Node: TMSTreeNode): Integer;
    procedure SortSelNodes;

    // multiselect
    property SelNodes[Index: Integer]: TMSTreeNode read GetSelNode;
    property SelCount: Integer read GetSelCount;

    property DropTarget: TMSTreeNode read GetDropTarget write SetDropTarget;
    property Selected: TMSTreeNode read GetSelection write SetSelection;
    property TopItem: TMSTreeNode read GetTopItem write SetTopItem;
  end;

  TMSTreeView = class(TMSCustomTreeView)
  published
    property ShowButtons;
    property BorderStyle;
    property DragCursor;
    property ShowLines;
    property ShowRoot;
    property ReadOnly;
    property RightClickSelect;
    property DragMode;
    property HideSelection;
    property Indent;
    property Items;
    property OnEditing;
    property OnEdited;
    property OnExpanding;
    property OnExpanded;
    property OnCollapsing;
    property OnCompare;
    property OnCollapsed;
    property OnChanging;
    property OnChange;
    property OnDeletion;
    property OnGetImageIndex;
    property OnGetSelectedIndex;
    property Align;
    property Enabled;
    property Font;
    property Color;
    // multiselect
    property MultiSelect;

    property ParentColor default False;
    property ParentCtl3D;
    property Ctl3D;
    property SortType;
    property TabOrder;
    property TabStop default True;
    property Visible;
    property OnClick;
    property OnEnter;
    property OnExit;
    property OnDragDrop;
    property OnDragOver;
    property OnStartDrag;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnDblClick;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property PopupMenu;
    property ParentFont;
    property ParentShowHint;
    property ShowHint;
    property Images;
    property StateImages;
  end;

function InitCommonControl(CC: Integer): Boolean;
procedure CheckCommonControl(CC: Integer);
{procedure Register;}

implementation

uses Consts, ComStrs;

function InitCommonControl(CC: Integer): Boolean;
var
  ICC: TInitCommonControlsEx;
begin
  ICC.dwSize := SizeOf(TInitCommonControlsEx);
  ICC.dwICC := CC;
  Result := InitCommonControlsEx(ICC);
  if not Result then InitCommonControls;
end;

procedure CheckCommonControl(CC: Integer);
begin
  if not InitCommonControl(CC) then
    raise EComponentError.Create(SInvalidComCtl32);
end;

{ TMSTreeNode }

function DefaulTMSTreeViewSort(Node1, Node2: TMSTreeNode; lParam: Integer): Integer; stdcall;
begin
  with Node1 do
    if Assigned(TreeView.OnCompare) then
      TreeView.OnCompare(TreeView, Node1, Node2, lParam, Result)
    else Result := lstrcmp(PChar(Node1.Text), PChar(Node2.Text));
end;

procedure TreeViewError(const Msg: string);
begin
  raise ETreeViewError.Create(Msg);
end;

procedure TreeViewErrorFmt(const Msg: string; Format: array of const);
begin
  raise ETreeViewError.CreateFmt(Msg, Format);
end;

constructor TMSTreeNode.Create(AOwner: TMSTreeNodes);
begin
  inherited Create;
  // multiselect
  FOrder := -1;

  FOverlayIndex := -1;
  FStateIndex := -1;
  FOwner := AOwner;
end;

destructor TMSTreeNode.Destroy;
var
  Node: TMSTreeNode;
  CheckValue: Integer;
begin
  // multiselect
  TreeView.FDeleting := True;
  TreeView.FCtrlDown := False;
  TreeView.FShiftDown := False;

  FDeleting := True;
  if Owner.Owner.FLastDropTarget = Self then
    Owner.Owner.FLastDropTarget := nil;
  Node := Parent;
  if (Node <> nil) and (not Node.Deleting) then
  begin
    if Node.IndexOf(Self) <> -1 then CheckValue := 1
    else CheckValue := 0;
    if Node.CompareCount(CheckValue) then
    begin
      Expanded := False;
      Node.HasChildren := False;
    end;
  end;
  if ItemId <> nil then TreeView_DeleteItem(Handle, ItemId);
  Data := nil;
  inherited Destroy;
  // multiselect
  TreeView.FDeleting := False;
end;

function TMSTreeNode.GetHandle: HWND;
begin
  Result := TreeView.Handle;
end;

function TMSTreeNode.GeTMSTreeView: TMSCustomTreeView;
begin
  Result := Owner.Owner;
end;

function TMSTreeNode.HasAsParent(Value: TMSTreeNode): Boolean;
begin
  if Value <> Nil then
  begin
    if Parent = Nil then Result := False
    else if Parent = Value then Result := True
    else Result := Parent.HasAsParent(Value);
  end
  else Result := True;
end;

procedure TMSTreeNode.SetText(const S: string);
var
  Item: TTVItem;
begin
  FText := S;
  with Item do
  begin
    mask := TVIF_TEXT;
    hItem := ItemId;
    pszText := LPSTR_TEXTCALLBACK;
  end;
  TreeView_SetItem(Handle, Item);
  if (TreeView.SortType in [stText, stBoth]) and FInTree then
  begin
    if (Parent <> nil) then Parent.AlphaSort
    else TreeView.AlphaSort;
  end;
end;

procedure TMSTreeNode.SetData(Value: Pointer);
begin
  FData := Value;
  if (TreeView.SortType in [stData, stBoth]) and Assigned(TreeView.OnCompare)
    and (not Deleting) and FInTree then
  begin
    if Parent <> nil then Parent.AlphaSort
    else TreeView.AlphaSort;
  end;
end;

function TMSTreeNode.GetState(NodeState: TNodeState): Boolean;
var
  Item: TTVItem;
begin
  Result := False;
  with Item do
  begin
    mask := TVIF_STATE;
    hItem := ItemId;
    if TreeView_GetItem(Handle, Item) then
      case NodeState of
        nsCut: Result := (state and TVIS_CUT) <> 0;
        nsFocused: Result := (state and TVIS_FOCUSED) <> 0;
        nsSelected: Result := (state and TVIS_SELECTED) <> 0;
        nsExpanded: Result := (state and TVIS_EXPANDED) <> 0;
        nsDropHilited: Result := (state and TVIS_DROPHILITED) <> 0;
      end;
  end;
end;

procedure TMSTreeNode.SetImageIndex(Value: Integer);
var
  Item: TTVItem;
begin
  FImageIndex := Value;
  with Item do
  begin
    mask := TVIF_IMAGE or TVIF_HANDLE;
    hItem := ItemId;
    iImage := I_IMAGECALLBACK;
  end;
  TreeView_SetItem(Handle, Item);
end;

procedure TMSTreeNode.SetSelectedIndex(Value: Integer);
var
  Item: TTVItem;
begin
  FSelectedIndex := Value;
  with Item do
  begin
    mask := TVIF_SELECTEDIMAGE or TVIF_HANDLE;
    hItem := ItemId;
    iSelectedImage := I_IMAGECALLBACK;
  end;
  TreeView_SetItem(Handle, Item);
end;

procedure TMSTreeNode.SetOverlayIndex(Value: Integer);
var
  Item: TTVItem;
begin
  FOverlayIndex := Value;
  with Item do
  begin
    mask := TVIF_STATE or TVIF_HANDLE;
    stateMask := TVIS_OVERLAYMASK;
    hItem := ItemId;
    state := IndexToOverlayMask(OverlayIndex + 1);
  end;
  TreeView_SetItem(Handle, Item);
end;

procedure TMSTreeNode.SetStateIndex(Value: Integer);
var
  Item: TTVItem;
begin
  FStateIndex := Value;
  if Value >= 0 then Dec(Value);
  with Item do
  begin
    mask := TVIF_STATE or TVIF_HANDLE;
    stateMask := TVIS_STATEIMAGEMASK;
    hItem := ItemId;
    state := IndexToStateImageMask(Value + 1);
  end;
  TreeView_SetItem(Handle, Item);
end;

function TMSTreeNode.CompareCount(CompareMe: Integer): Boolean;
var
  Count: integer;
  Node: TMSTreeNode;
Begin
  Count := 0;
  Result := False;
  Node := GetFirstChild;
  while Node <> nil do
  begin
    Inc(Count);
    Node := Node.GetNextChild(Node);
    if Count > CompareMe then Exit;
  end;
  if Count = CompareMe then Result := True;
end;

function TMSTreeNode.DoCanExpand(Expand: Boolean): Boolean;
begin
  Result := False;
  if HasChildren then
  begin
    if Expand then Result := TreeView.CanExpand(Self)
    else Result := TreeView.CanCollapse(Self);
  end;
end;

procedure TMSTreeNode.DoExpand(Expand: Boolean);
begin
  if HasChildren then
  begin
    if Expand then TreeView.Expand(Self)
    else TreeView.Collapse(Self);
  end;
end;

procedure TMSTreeNode.ExpandItem(Expand: Boolean; Recurse: Boolean);
var
  Flag: Integer;
  Node: TMSTreeNode;
begin
  if Recurse then
  begin
    Node := Self;
    repeat
      Node.ExpandItem(Expand, False);
      Node := Node.GetNext;
    until (Node = nil) or (not Node.HasAsParent(Self));
  end
  else begin
    TreeView.FManualNotify := True;
    try
      Flag := 0;
      if Expand then
      begin
        if DoCanExpand(True) then
        begin
          Flag := TVE_EXPAND;
          DoExpand(True);
        end;
      end
      else begin
        if DoCanExpand(False) then
        begin
          Flag := TVE_COLLAPSE;
          DoExpand(False);
        end;
      end;
      if Flag <> 0 then TreeView_Expand(Handle, ItemId, Flag);
    finally
      TreeView.FManualNotify := False;
    end;
  end;
end;

procedure TMSTreeNode.Expand(Recurse: Boolean);
begin
  ExpandItem(True, Recurse);
end;

procedure TMSTreeNode.Collapse(Recurse: Boolean);
begin
  ExpandItem(False, Recurse);
end;

function TMSTreeNode.GetExpanded: Boolean;
begin
  Result := GetState(nsExpanded);
end;

procedure TMSTreeNode.SetExpanded(Value: Boolean);
begin
  if Value then Expand(False)
  else Collapse(False);
end;

function TMSTreeNode.GetSelected: Boolean;
begin
  Result := GetState(nsSelected);
end;

procedure TMSTreeNode.SetSelected(Value: Boolean);
begin
  if Value then TreeView_SelectItem(Handle, ItemId)
  else if Selected then TreeView_SelectItem(Handle, nil);
end;

function TMSTreeNode.GetCut: Boolean;
begin
  Result := GetState(nsCut);
end;

procedure TMSTreeNode.SetCut(Value: Boolean);
var
  Item: TTVItem;
  Template: Integer;
begin
  if Value then Template := -1
  else Template := 0;
  with Item do
  begin
    mask := TVIF_STATE;
    hItem := ItemId;
    stateMask := TVIS_CUT;
    state := stateMask and Template;
  end;
  TreeView_SetItem(Handle, Item);
end;

function TMSTreeNode.GetDropTarget: Boolean;
begin
  Result := GetState(nsDropHilited);
end;

procedure TMSTreeNode.SetDropTarget(Value: Boolean);
begin
  if Value then TreeView_SelectDropTarget(Handle, ItemId)
  else if DropTarget then TreeView_SelectDropTarget(Handle, nil);
end;

function TMSTreeNode.GetChildren: Boolean;
var
  Item: TTVItem;
begin
  Item.mask := TVIF_CHILDREN;
  Item.hItem := ItemId;
  if TreeView_GetItem(Handle, Item) then Result := Item.cChildren > 0
  else Result := False;
end;

procedure TMSTreeNode.SetFocused(Value: Boolean);
var
  Item: TTVItem;
  Template: Integer;
begin
  if Value then Template := -1
  else Template := 0;
  with Item do
  begin
    mask := TVIF_STATE;
    hItem := ItemId;
    stateMask := TVIS_FOCUSED;
    state := stateMask and Template;
  end;
  TreeView_SetItem(Handle, Item);
end;

function TMSTreeNode.GetFocused: Boolean;
begin
  Result := GetState(nsFocused);
end;

procedure TMSTreeNode.SetChildren(Value: Boolean);
var
  Item: TTVItem;
begin
  with Item do
  begin
    mask := TVIF_CHILDREN;
    hItem := ItemId;
    cChildren := Ord(Value);
  end;
  TreeView_SetItem(Handle, Item);
end;

function TMSTreeNode.GetParent: TMSTreeNode;
begin
  with FOwner do
    Result := GetNode(TreeView_GetParent(Handle, ItemId));
end;

function TMSTreeNode.GetNextSibling: TMSTreeNode;
begin
  with FOwner do
    Result := GetNode(TreeView_GetNextSibling(Handle, ItemId));
end;

function TMSTreeNode.GetPrevSibling: TMSTreeNode;
begin
  with FOwner do
    Result := GetNode(TreeView_GetPrevSibling(Handle, ItemId));
end;

function TMSTreeNode.GetNextVisible: TMSTreeNode;
begin
  if IsVisible then
    with FOwner do
      Result := GetNode(TreeView_GetNextVisible(Handle, ItemId))
  else Result := nil;
end;

function TMSTreeNode.GetPrevVisible: TMSTreeNode;
begin
  with FOwner do
    Result := GetNode(TreeView_GetPrevVisible(Handle, ItemId));
end;

function TMSTreeNode.GetNextChild(Value: TMSTreeNode): TMSTreeNode;
begin
  if Value <> nil then Result := Value.GetNextSibling
  else Result := nil;
end;

function TMSTreeNode.GetPrevChild(Value: TMSTreeNode): TMSTreeNode;
begin
  if Value <> nil then Result := Value.GetPrevSibling
  else Result := nil;
end;

function TMSTreeNode.GetFirstChild: TMSTreeNode;
begin
  with FOwner do
    Result := GetNode(TreeView_GetChild(Handle, ItemId));
end;

function TMSTreeNode.GetLastChild: TMSTreeNode;
var
  Node: TMSTreeNode;
begin
  Result := GetFirstChild;
  if Result <> nil then
  begin
    Node := Result;
    repeat
      Result := Node;
      Node := Result.GetNextSibling;
    until Node = nil;
  end;
end;

function TMSTreeNode.GetNext: TMSTreeNode;
var
  NodeID, ParentID: HTreeItem;
  Handle: HWND;
begin
  Handle := FOwner.Handle;
  NodeID := TreeView_GetChild(Handle, ItemId);
  if NodeID = nil then NodeID := TreeView_GetNextSibling(Handle, ItemId);
  ParentID := ItemId;
  while (NodeID = nil) and (ParentID <> nil) do
  begin
    ParentID := TreeView_GetParent(Handle, ParentID);
    NodeID := TreeView_GetNextSibling(Handle, ParentID);
  end;
  Result := FOwner.GetNode(NodeID);
end;

function TMSTreeNode.GetPrev: TMSTreeNode;
var
  Node: TMSTreeNode;
begin
  Result := GetPrevSibling;
  if Result <> nil then
  begin
    Node := Result;
    repeat
      Result := Node;
      Node := Result.GetLastChild;
    until Node = nil;
  end else
    Result := Parent;
end;

function TMSTreeNode.GetAbsoluteIndex: Integer;
var
  Node: TMSTreeNode;
begin
  Result := -1;
  Node := Self;
  while Node <> nil do
  begin
    Inc(Result);
    Node := Node.GetPrev;
  end;
end;

function TMSTreeNode.GetIndex: Integer;
var
  Node: TMSTreeNode;
begin
  Result := -1;
  Node := Self;
  while Node <> nil do
  begin
    Inc(Result);
    Node := Node.GetPrevSibling;
  end;
end;

function TMSTreeNode.GetItem(Index: Integer): TMSTreeNode;
begin
  Result := GetFirstChild;
  while (Result <> nil) and (Index > 0) do
  begin
    Result := GetNextChild(Result);
    Dec(Index);
  end;
  if Result = nil then TreeViewError(SListIndexError);
end;

procedure TMSTreeNode.SetItem(Index: Integer; Value: TMSTreeNode);
begin
  item[Index].Assign(Value);
end;

function TMSTreeNode.IndexOf(Value: TMSTreeNode): Integer;
var
  Node: TMSTreeNode;
begin
  Result := -1;
  Node := GetFirstChild;
  while (Node <> nil) do
  begin
    Inc(Result);
    if Node = Value then Break;
    Node := GetNextChild(Node);
  end;
  if Node = nil then Result := -1;
end;

function TMSTreeNode.GetCount: Integer;
var
  Node: TMSTreeNode;
begin
  Result := 0;
  Node := GetFirstChild;
  while Node <> nil do
  begin
    Inc(Result);
    Node := Node.GetNextChild(Node);
  end;
end;

procedure TMSTreeNode.EndEdit(Cancel: Boolean);
begin
  TreeView_EndEditLabelNow(Handle, Cancel);
end;

procedure TMSTreeNode.InternalMove(ParentNode, Node: TMSTreeNode;
  HItem: HTreeItem; AddMode: TAddMode);
var
  I: Integer;
  NodeId: HTreeItem;
  TreeViewItem: TTVItem;
  Children: Boolean;
  IsSelected: Boolean;
begin
  if (AddMode = taInsert) and (Node <> nil) then
    NodeId := Node.ItemId else
    NodeId := nil;
  Children := HasChildren;
  IsSelected := Selected;
  if (Parent <> nil) and (Parent.CompareCount(1)) then
  begin
    Parent.Expanded := False;
    Parent.HasChildren := False;
  end;
  with TreeViewItem do
  begin
    mask := TVIF_PARAM;
    hItem := ItemId;
    lParam := 0;
  end;
  TreeView_SetItem(Handle, TreeViewItem);
  with Owner do
    HItem := AddItem(HItem, NodeId, CreateItem(Self), AddMode);
  if HItem = nil then
    raise EOutOfResources.Create(sInsertError);
  for I := Count - 1 downto 0 do
    Item[I].InternalMove(Self, nil, HItem, taAddFirst);
  TreeView_DeleteItem(Handle, ItemId);
  FItemId := HItem;
  Assign(Self);
  HasChildren := Children;
  Selected := IsSelected;
end;

procedure TMSTreeNode.MoveTo(Destination: TMSTreeNode; Mode: TNodeAttachMode);
var
  AddMode: TAddMode;

  // multiselect
  ChildCount: Integer;
  PrevNode,NextNode,

  Node: TMSTreeNode;
  HItem: HTreeItem;
  OldOnChanging: TTVChangingEvent;
  OldOnChange: TTVChangedEvent;
begin
  OldOnChanging := TreeView.OnChanging;
  OldOnChange := TreeView.OnChange;
  TreeView.OnChanging := nil;
  TreeView.OnChange := nil;
  try
    if (Destination = nil) or not Destination.HasAsParent(Self) then
    begin
      AddMode := taAdd;
      if (Destination <> nil) and not (Mode in [naAddChild, naAddChildFirst]) then
        Node := Destination.Parent else
        Node := Destination;
      case Mode of
        naAdd,
        naAddChild:
          begin

            // multiselect
            if Destination<>nil then
              NextNode := Destination.GetNextSibling
            else
              NextNode := nil;

            AddMode := taAdd;
          end;
        naAddFirst,
        naAddChildFirst:
          begin

            // multiselect
            if Destination<>nil then
              NextNode := Destination.GetFirstChild
            else
              NextNode := nil;

            AddMode := taAddFirst;
          end;
        naInsert:
          begin

            // multiselect
            if Destination<>nil then
              NextNode := Destination
            else
              NextNode := nil;

            Destination := Destination.GetPrevSibling;
            if Destination = nil then AddMode := taAddFirst
            else AddMode := taInsert;
          end;
      end;
      if Node <> nil then
        HItem := Node.ItemId else
        HItem := nil;
      InternalMove(Node, Destination, HItem, AddMode);
      Node := Parent;
      if Node <> nil then
      begin
        Node.HasChildren := True;
        Node.Expanded := True;
      end;

      // multiselect
      if Self<>nil then begin
        PrevNode := Self.GetPrev;
        if PrevNode=nil then
          if NextNode=nil then
            OrderChildren(Self,NextNode,0,0)
            //Self.OrderIndex := 1
          else
            if (NextNode.Order=0) or
              (Self.Order=NextNode.Order) or
              (NextNode.Order/(Self.TreeView.SelCount+1)=0) then
              Self.TreeView.Items.ReOrder(PrevNode)
            else
              OrderChildren(Self,NextNode,0,NextNode.Order/(Self.TreeView.SelCount+1))
              //Self.OrderIndex := NextNode.OrderIndex/(Self.TreeView.SelCount+1)
        else
          if NextNode=nil then
            OrderChildren(Self,NextNode,PrevNode.Order,0)
            //Self.OrderIndex := PrevNode.OrderIndex+1
          else
            if (NextNode.Order=0) or
              (PrevNode.Order=NextNode.Order) then
              Self.TreeView.Items.ReOrder(PrevNode)
            else
              OrderChildren(Self,NextNode,PrevNode.Order,
                (NextNode.Order-PrevNode.Order)/(Self.TreeView.SelCount+Self.GetTotalCount+1));
              //Self.OrderIndex := PrevNode.OrderIndex+((NextNode.OrderIndex-PrevNode.OrderIndex)/(Self.TreeView.SelCount+1));
      end;

    end;
  finally
    TreeView.OnChanging := OldOnChanging;
    TreeView.OnChange := OldOnChange;
  end;
end;

procedure TMSTreeNode.MakeVisible;
begin
  TreeView_EnsureVisible(Handle, ItemId);
end;

function TMSTreeNode.GetLevel: Integer;
var
  Node: TMSTreeNode;
begin
  Result := 0;
  Node := Parent;
  while Node <> nil do
  begin
    Inc(Result);
    Node := Node.Parent;
  end;
end;

function TMSTreeNode.IsNodeVisible: Boolean;
var
  Rect: TRect;
begin
  Result := TreeView_GetItemRect(Handle, ItemId, Rect, True);
end;

function TMSTreeNode.EditText: Boolean;
begin
  Result := TreeView_EditLabel(Handle, ItemId) <> 0;
end;

function TMSTreeNode.DisplayRect(TextOnly: Boolean): TRect;
begin
  FillChar(Result, SizeOf(Result), 0);
  TreeView_GetItemRect(Handle, ItemId, Result, TextOnly);
end;

function TMSTreeNode.AlphaSort: Boolean;
begin
  Result := CustomSort(nil, 0);
end;

function TMSTreeNode.CustomSort(SortProc: TTVCompare; Data: Longint): Boolean;
var
  SortCB: TTVSortCB;
begin
  with SortCB do
  begin
    if not Assigned(SortProc) then lpfnCompare := @DefaulTMSTreeViewSort
    else lpfnCompare := SortProc;
    hParent := ItemId;
    lParam := Data;
  end;
  Result := TreeView_SortChildrenCB(Handle, SortCB, 0);
end;

procedure TMSTreeNode.Delete;
begin
  if not Deleting then Free;
end;

procedure TMSTreeNode.DeleteChildren;
begin
  TreeView_Expand(TreeView.Handle, ItemID, TVE_COLLAPSE or TVE_COLLAPSERESET);
  HasChildren := False;
end;

procedure TMSTreeNode.Assign(Source: TPersistent);
var
  Node: TMSTreeNode;
begin
  if Source is TMSTreeNode then
  begin
    Node := TMSTreeNode(Source);
    Text := Node.Text;
    Data := Node.Data;
    // multiselect
    Order := Node.Order;

    ImageIndex := Node.ImageIndex;
    SelectedIndex := Node.SelectedIndex;
    StateIndex := Node.StateIndex;
    OverlayIndex := Node.OverlayIndex;
    Focused := Node.Focused;
    DropTarget := Node.DropTarget;
    Cut := Node.Cut;
    HasChildren := Node.HasChildren;
  end
  else inherited Assign(Source);
end;

function TMSTreeNode.IsEqual(Node: TMSTreeNode): Boolean;
begin
  Result := (Text = Node.Text) and (Data = Node.Data);
end;

procedure TMSTreeNode.ReadData(Stream: TStream; Info: PNodeInfo);
var
  I, Size, ItemCount: Integer;
begin
  Stream.ReadBuffer(Size, SizeOf(Size));
  Stream.ReadBuffer(Info^, Size);
  Text := Info^.Text;
  // multiselect
  Order := Info^.Order;

  ImageIndex := Info^.ImageIndex;
  SelectedIndex := Info^.SelectedIndex;
  StateIndex := Info^.StateIndex;
  OverlayIndex := Info^.OverlayIndex;
  Data := Info^.Data;
  ItemCount := Info^.Count;
  for I := 0 to ItemCount - 1 do
    with Owner.AddChild(Self, '') do ReadData(Stream, Info);
end;

procedure TMSTreeNode.WriteData(Stream: TStream; Info: PNodeInfo);
var
  I, Size, L, ItemCount: Integer;
begin
  L := Length(Text);
  if L > 255 then L := 255;
  Size := SizeOf(TNodeInfo) + L - 255;
  Info^.Text := Text;
  // multiselect
  Info^.Order := Order;

  Info^.ImageIndex := ImageIndex;
  Info^.SelectedIndex := SelectedIndex;
  Info^.OverlayIndex := OverlayIndex;
  Info^.StateIndex := StateIndex;
  Info^.Data := Data;
  ItemCount := Count;
  Info^.Count := ItemCount;
  Stream.WriteBuffer(Size, SizeOf(Size));
  Stream.WriteBuffer(Info^, Size);
  for I := 0 to ItemCount - 1 do Item[I].WriteData(Stream, Info);
end;

// multiselect
procedure TMSTreeNode.SetOrder(Value: Extended);
begin
  FOrder := Value;
end;

// multiselect
function TMSTreeNode.IsBold: Boolean;
var
  Item: TTVItem;
begin
  Result := False;
  if Self=nil then Exit;

  with Item do
  begin
    mask := TVIF_STATE;
    hItem := ItemId;
    if TreeView_GetItem(Handle, Item) then
      Result := (state and TVIS_BOLD) <> 0;
  end;
end;

// multiselect
function TMSTreeNode.IsSelected: Boolean;
var
  Item: TTVItem;
begin
  Result := False;
  if Self=nil then Exit;

  with Item do
  begin
    mask := TVIF_STATE;
    hItem := ItemId;
    if TreeView_GetItem(Handle, Item) then
      Result := (state and TVIS_SELECTED) <> 0;
  end;
end;

// multiselect
procedure TMSTreeNode.MakeBold(Value: Boolean);
var
  Item: TTVItem;
  Template: Integer;
begin
  if Self=nil then Exit;

  if Value then Template := -1
  else Template := 0;

  with Item do
  begin
    mask := TVIF_STATE;
    hItem := ItemId;
    stateMask := TVIS_BOLD;
    state := stateMask and Template;
  end;
  TreeView_SetItem(Handle, Item);
end;

// multiselect
procedure TMSTreeNode.MakeFocused(Value: Boolean);
var
  Item: TTVItem;
  Template: Integer;
begin
  if Self=nil then Exit;

  if Value then Template := -1
  else Template := 0;

  with Item do
  begin
    mask := TVIF_STATE;
    hItem := ItemId;
    stateMask := TVIS_FOCUSED;
    state := stateMask and Template;
  end;
  TreeView_SetItem(Handle, Item);
end;

// multiselect
procedure TMSTreeNode.MakeSelected(Value: Boolean);
var
  Item: TTVItem;
  Template: Integer;
begin
  if Self=nil then Exit;

  if Value then Template := -1
  else Template := 0;

  with Item do
  begin
    mask := TVIF_STATE;
    hItem := ItemId;
    stateMask := TVIS_SELECTED;
    state := stateMask and Template;
  end;
  TreeView_SetItem(Handle, Item);
end;

// multiselect
function TMSTreeNode.GetTotalCount: Integer;
var
  Node: TMSTreeNode;
begin
  Result := 0;
  Node := GetFirstChild;
  while Node <> nil do
  begin
    Inc(Result);
    if Node.HasChildren then
      Result := Result+Node.GetTotalCount;
    Node := Node.GetNextChild(Node);
  end;
end;

// multiselect
procedure TMSTreeNode.OrderChildren(Node,NextNode: TMSTreeNode; PrevOrder,Incr: Extended);
var
  SelNode: TMSTreeNode;
  Cnt: Integer;
begin
  if Node<>nil then
    if Incr>0 then
      begin
        SelNode := Node;
        Cnt := 1;
        while SelNode<>NextNode do begin
          SelNode.Order := PrevOrder+(Cnt*Incr);
          SelNode := SelNode.GetNext;
        end;
      end
    else
      TreeView.Items.ReOrder(Node.GetPrev);
end;

{ TMSTreeNodes }

constructor TMSTreeNodes.Create(AOwner: TMSCustomTreeView);
begin
  inherited Create;
  FOwner := AOwner;
end;

destructor TMSTreeNodes.Destroy;
begin
  Clear;
  inherited Destroy;
end;

function TMSTreeNodes.GetCount: Integer;
begin
  if Owner.HandleAllocated then Result := TreeView_GetCount(Handle)
  else Result := 0;
end;

function TMSTreeNodes.GetHandle: HWND;
begin
  Result := Owner.Handle;
end;

procedure TMSTreeNodes.Delete(Node: TMSTreeNode);
begin
  if (Node.ItemId = nil) and Assigned(Owner.FOnDeletion) then
    Owner.FOnDeletion(Self, Node);
  Node.Delete;
end;

procedure TMSTreeNodes.Clear;
begin
  if Owner.HandleAllocated then
    TreeView_DeleteAllItems(Handle);
end;

function TMSTreeNodes.AddChildFirst(Node: TMSTreeNode; const S: string): TMSTreeNode;
begin
  Result := AddChildObjectFirst(Node, S, nil);
end;

function TMSTreeNodes.AddChildObjectFirst(Node: TMSTreeNode; const S: string;
  Ptr: Pointer): TMSTreeNode;
begin
  Result := InternalAddObject(Node, S, Ptr, taAddFirst);
end;

function TMSTreeNodes.AddChild(Node: TMSTreeNode; const S: string): TMSTreeNode;
begin
  Result := AddChildObject(Node, S, nil);
end;

function TMSTreeNodes.AddChildObject(Node: TMSTreeNode; const S: string;
  Ptr: Pointer): TMSTreeNode;
begin
  Result := InternalAddObject(Node, S, Ptr, taAdd);
end;

function TMSTreeNodes.AddFirst(Node: TMSTreeNode; const S: string): TMSTreeNode;
begin
  Result := AddObjectFirst(Node, S, nil);
end;

function TMSTreeNodes.AddObjectFirst(Node: TMSTreeNode; const S: string;
  Ptr: Pointer): TMSTreeNode;
begin
  if Node <> nil then Node := Node.Parent;
  Result := InternalAddObject(Node, S, Ptr, taAddFirst);
end;

function TMSTreeNodes.Add(Node: TMSTreeNode; const S: string): TMSTreeNode;
begin
  Result := AddObject(Node, S, nil);
end;

procedure TMSTreeNodes.Repaint(Node: TMSTreeNode);
var
  R: TRect;
begin
  if FUpdateCount < 1 then
  begin
    while (Node <> nil) and not Node.IsVisible do Node := Node.Parent;
    if Node <> nil then
    begin
      R := Node.DisplayRect(False);
      InvalidateRect(Owner.Handle, @R, True);
    end;
  end;
end;

function TMSTreeNodes.AddObject(Node: TMSTreeNode; const S: string;
  Ptr: Pointer): TMSTreeNode;
begin
  if Node <> nil then Node := Node.Parent;
  Result := InternalAddObject(Node, S, Ptr, taAdd);
end;

function TMSTreeNodes.Insert(Node: TMSTreeNode; const S: string): TMSTreeNode;
begin
  Result := InsertObject(Node, S, nil);
end;

procedure TMSTreeNodes.AddedNode(Value: TMSTreeNode);
begin
  if Value <> nil then
  begin
    Value.HasChildren := True;
    Repaint(Value);
  end;
end;

function TMSTreeNodes.InsertObject(Node: TMSTreeNode; const S: string;
  Ptr: Pointer): TMSTreeNode;
var
  Item, ItemId: HTreeItem;

  // multiselect
  NextNode,PrevNode,

  Parent: TMSTreeNode;
  AddMode: TAddMode;
begin
  // multiselect
  Owner.FCtrlDown := False;
  Owner.FShiftDown := False;

  Result := Owner.CreateNode;
  try
    Item := nil;
    ItemId := nil;
    Parent := nil;
    AddMode := taInsert;
    if Node <> nil then
    begin
      Parent := Node.Parent;
      if Parent <> nil then Item := Parent.ItemId;
      Node := Node.GetPrevSibling;
      if Node <> nil then ItemId := Node.ItemId
      else AddMode := taAddFirst;
    end;
    Result.Data := Ptr;
    Result.Text := S;
    Item := AddItem(Item, ItemId, CreateItem(Result), AddMode);
    if Item = nil then
      raise EOutOfResources.Create(sInsertError);
    Result.FItemId := Item;
    AddedNode(Parent);

    // multiselect
    PrevNode := Result.GetPrev;
    NextNode := Result.GetNext;
    while NextNode=Result do
      NextNode := NextNode.GetNext;
    if PrevNode=nil then
      if NextNode=nil then
        Result.Order := 1
      else
        if (NextNode.Order=0) or (Result.Order=NextNode.Order) or
          ((NextNode.Order/2)=0) then
          ReOrder(PrevNode)
        else
          Result.Order := NextNode.Order/2
    else
      if NextNode=nil then
        Result.Order := PrevNode.Order+1
      else
        if (NextNode.Order=0) or (PrevNode.Order=NextNode.Order) or
          (((NextNode.Order-PrevNode.Order)/2)=0) then
          ReOrder(PrevNode)
        else
          Result.Order := PrevNode.Order+((NextNode.Order-PrevNode.Order)/2);

  except
    Result.Free;
    raise;
  end;
end;

function TMSTreeNodes.InternalAddObject(Node: TMSTreeNode; const S: string;
  Ptr: Pointer; AddMode: TAddMode): TMSTreeNode;
var
  Item: HTreeItem;

  // multiselect
  NextNode,PrevNode: TMSTreeNode;
begin
  // multiselect
  Owner.FCtrlDown := False;
  Owner.FShiftDown := False;

  Result := Owner.CreateNode;
  try
    if Node <> nil then Item := Node.ItemId
    else Item := nil;
    Result.Data := Ptr;
    Result.Text := S;
    Item := AddItem(Item, nil, CreateItem(Result), AddMode);
    if Item = nil then
      raise EOutOfResources.Create(sInsertError);
    Result.FItemId := Item;
    AddedNode(Node);

    // multiselect implementation
    PrevNode := Result.GetPrev;
    NextNode := Result.GetNext;
    while NextNode=Result do
      NextNode := NextNode.GetNext;
    if PrevNode=nil then
      if NextNode=nil then
        Result.Order := 1
      else
        if (NextNode.Order=0) or (Result.Order=NextNode.Order) or
          ((NextNode.Order/2)=0) then
          ReOrder(PrevNode)
        else
          Result.Order := NextNode.Order/2
    else
      if NextNode=nil then
        Result.Order := PrevNode.Order+1
      else
        if (NextNode.Order=0) or (PrevNode.Order=NextNode.Order) or
          (((NextNode.Order-PrevNode.Order)/2)=0) then
          ReOrder(PrevNode)
        else
          Result.Order := PrevNode.Order+((NextNode.Order-PrevNode.Order)/2);

  except
    Result.Free;
    raise;
  end;
end;

function TMSTreeNodes.CreateItem(Node: TMSTreeNode): TTVItem;
begin
  Node.FInTree := True;
  with Result do
  begin
    mask := TVIF_TEXT or TVIF_PARAM or TVIF_IMAGE or TVIF_SELECTEDIMAGE;
    lParam := Longint(Node);
    pszText := LPSTR_TEXTCALLBACK;
    iImage := I_IMAGECALLBACK;
    iSelectedImage := I_IMAGECALLBACK;
  end;
end;

function TMSTreeNodes.AddItem(Parent, Target: HTreeItem;
  const Item: TTVItem; AddMode: TAddMode): HTreeItem;
var
  InsertStruct: TTVInsertStruct;
begin
  with InsertStruct do
  begin
    hParent := Parent;
    case AddMode of
      taAddFirst:
        hInsertAfter := TVI_FIRST;
      taAdd:
        hInsertAfter := TVI_LAST;
      taInsert:
        hInsertAfter := Target;
    end;
  end;
  InsertStruct.item := Item;
  Result := TreeView_InsertItem(Handle, InsertStruct);
end;

function TMSTreeNodes.GetFirstNode: TMSTreeNode;
begin
  Result := GetNode(TreeView_GetRoot(Handle));
end;

function TMSTreeNodes.GetNodeFromIndex(Index: Integer): TMSTreeNode;
begin
  Result := GetFirstNode;
  while (Index <> 0) and (Result <> nil) do
  begin
    Result := Result.GetNext;
    Dec(Index);
  end;
  if Result = nil then TreeViewError(sInvalidIndex);
end;

function TMSTreeNodes.GetNode(ItemId: HTreeItem): TMSTreeNode;
var
  Item: TTVItem;
begin
  with Item do
  begin
    hItem := ItemId;
    mask := TVIF_PARAM;
  end;
  if TreeView_GetItem(Handle, Item) then Result := TMSTreeNode(Item.lParam)
  else Result := nil;
end;

procedure TMSTreeNodes.SetItem(Index: Integer; Value: TMSTreeNode);
begin
  GetNodeFromIndex(Index).Assign(Value);
end;

procedure TMSTreeNodes.BeginUpdate;
begin
  if FUpdateCount = 0 then SetUpdateState(True);
  Inc(FUpdateCount);
end;

procedure TMSTreeNodes.SetUpdateState(Updating: Boolean);
begin
  SendMessage(Handle, WM_SETREDRAW, Ord(not Updating), 0);
  if Updating then Owner.Refresh;
end;

procedure TMSTreeNodes.EndUpdate;
begin
  Dec(FUpdateCount);
  if FUpdateCount = 0 then SetUpdateState(False);
end;

procedure TMSTreeNodes.Assign(Source: TPersistent);
var
  TreeNodes: TMSTreeNodes;
  MemStream: TMemoryStream;
begin
  if Source is TMSTreeNodes then
  begin
    TreeNodes := TMSTreeNodes(Source);
    Clear;
    MemStream := TMemoryStream.Create;
    try
      TreeNodes.WriteData(MemStream);
      MemStream.Position := 0;
      ReadData(MemStream);
    finally
      MemStream.Free;
    end;
  end
  else inherited Assign(Source);
end;

procedure TMSTreeNodes.DefineProperties(Filer: TFiler);

  function WriteNodes: Boolean;
  var
    I: Integer;
    Nodes: TMSTreeNodes;
  begin
    Nodes := TMSTreeNodes(Filer.Ancestor);
    if Nodes = nil then
      Result := Count > 0
    else if Nodes.Count <> Count then
      Result := True
    else
    begin
      Result := False;
      for I := 0 to Count - 1 do
      begin
        Result := not Item[I].IsEqual(Nodes[I]);
        if Result then Break;
      end
    end;
  end;

begin
  inherited DefineProperties(Filer);
  Filer.DefineBinaryProperty('Data', ReadData, WriteData, WriteNodes);
end;

procedure TMSTreeNodes.ReadData(Stream: TStream);
var
  I, Count: Integer;
  NodeInfo: TNodeInfo;
begin
  Clear;
  Stream.ReadBuffer(Count, SizeOf(Count));
  for I := 0 to Count - 1 do
    Add(nil, '').ReadData(Stream, @NodeInfo);
end;

procedure TMSTreeNodes.WriteData(Stream: TStream);
var
  I: Integer;
  Node: TMSTreeNode;
  NodeInfo: TNodeInfo;
begin
  I := 0;
  Node := GetFirstNode;
  while Node <> nil do
  begin
    Inc(I);
    Node := Node.GetNextSibling;
  end;
  Stream.WriteBuffer(I, SizeOf(I));
  Node := GetFirstNode;
  while Node <> nil do
  begin
    Node.WriteData(Stream, @NodeInfo);
    Node := Node.GetNextSibling;
  end;
end;

// multiselect
procedure TMSTreeNodes.ReOrder(Node: TMSTreeNode);
var
  SelNode: TMSTreeNode;
  Cnt: Integer;
begin
  if Node=nil then
    SelNode := GetFirstNode
  else
    SelNode := Node;
  if SelNode<>nil then begin
    Cnt := Round(SelNode.Order)+2;
    while SelNode<>nil do begin
      Inc(Cnt);
      SelNode.Order := Cnt;
      SelNode := SelNode.GetNext;
    end;
  end;
end;

type
  TMSTreeStrings = class(TStrings)
  private
    FOwner: TMSTreeNodes;
  protected
    function Get(Index: Integer): string; override;
    function GetBufStart(Buffer: PChar; var Level: Integer): PChar;
    function GetCount: Integer; override;
    function GetObject(Index: Integer): TObject; override;
    procedure PutObject(Index: Integer; AObject: TObject); override;
    procedure SetUpdateState(Updating: Boolean); override;
  public
    constructor Create(AOwner: TMSTreeNodes);
    function Add(const S: string): Integer; override;
    procedure Clear; override;
    procedure Delete(Index: Integer); override;
    procedure Insert(Index: Integer; const S: string); override;
    procedure LoadTreeFromStream(Stream: TStream);
    procedure SaveTreeToStream(Stream: TStream);
    property Owner: TMSTreeNodes read FOwner;
  end;

constructor TMSTreeStrings.Create(AOwner: TMSTreeNodes);
begin
  inherited Create;
  FOwner := AOwner;
end;

function TMSTreeStrings.Get(Index: Integer): string;
const
  TabChar = #9;
var
  Level, I: Integer;
  Node: TMSTreeNode;
begin
  Result := '';
  Node := Owner.GetNodeFromIndex(Index);
  Level := Node.Level;
  for I := 0 to Level - 1 do Result := Result + TabChar;
  Result := Result + Node.Text;
end;

function TMSTreeStrings.GetBufStart(Buffer: PChar; var Level: Integer): PChar;
begin
  Level := 0;
  while Buffer^ in [' ', #9] do
  begin
    Inc(Buffer);
    Inc(Level);
  end;
  Result := Buffer;
end;

function TMSTreeStrings.GetObject(Index: Integer): TObject;
begin
  Result := Owner.GetNodeFromIndex(Index).Data;
end;

procedure TMSTreeStrings.PutObject(Index: Integer; AObject: TObject);
begin
  Owner.GetNodeFromIndex(Index).Data := AObject;
end;

function TMSTreeStrings.GetCount: Integer;
begin
  Result := Owner.Count;
end;

procedure TMSTreeStrings.Clear;
begin
  Owner.Clear;
end;

procedure TMSTreeStrings.Delete(Index: Integer);
begin
  Owner.GetNodeFromIndex(Index).Delete;
end;

procedure TMSTreeStrings.SetUpdateState(Updating: Boolean);
begin
  SendMessage(Owner.Handle, WM_SETREDRAW, Ord(not Updating), 0);
  if not Updating then Owner.Owner.Refresh;
end;

function TMSTreeStrings.Add(const S: string): Integer;
var
  Level, OldLevel, I: Integer;
  NewStr: string;
  Node: TMSTreeNode;
begin
  Result := GetCount;
  if (Length(S) = 1) and (S[1] = Chr($1A)) then Exit;
  Node := nil;
  OldLevel := 0;
  NewStr := GetBufStart(PChar(S), Level);
  if Result > 0 then
  begin
    Node := Owner.GetNodeFromIndex(Result - 1);
    OldLevel := Node.Level;
  end;
  if (Level > OldLevel) or (Node = nil) then
  begin
    if Level - OldLevel > 1 then TreeViewError(sInvalidLevel);
  end
  else begin
    for I := OldLevel downto Level do
    begin
      Node := Node.Parent;
      if (Node = nil) and (I - Level > 0) then
        TreeViewError(sInvalidLevel);
    end;
  end;
  Owner.AddChild(Node, NewStr);
end;

procedure TMSTreeStrings.Insert(Index: Integer; const S: string);
begin
  with Owner do
    Insert(GetNodeFromIndex(Index), S);
end;

procedure TMSTreeStrings.LoadTreeFromStream(Stream: TStream);
var
  List: TStringList;
  ANode, NextNode: TMSTreeNode;
  ALevel, i: Integer;
  CurrStr: string;
begin
  List := TStringList.Create;
  Owner.BeginUpdate;
  try
    try
      Clear;
      List.LoadFromStream(Stream);
      ANode := nil;
      for i := 0 to List.Count - 1 do
      begin
        CurrStr := GetBufStart(PChar(List[i]), ALevel);
        if ANode = nil then
          ANode := Owner.AddChild(nil, CurrStr)
        else if ANode.Level = ALevel then
          ANode := Owner.AddChild(ANode.Parent, CurrStr)
        else if ANode.Level = (ALevel - 1) then
          ANode := Owner.AddChild(ANode, CurrStr)
        else if ANode.Level > ALevel then
        begin
          NextNode := ANode.Parent;
          while NextNode.Level > ALevel do
            NextNode := NextNode.Parent;
          ANode := Owner.AddChild(NextNode.Parent, CurrStr);
        end
        else TreeViewErrorFmt(sInvalidLevelEx, [ALevel, CurrStr]);
      end;
    finally
      Owner.EndUpdate;
      List.Free;
    end;
  except
    Owner.Owner.Invalidate;  // force repaint on exception
    raise;
  end;
end;

procedure TMSTreeStrings.SaveTreeToStream(Stream: TStream);
const
  TabChar = #9;
  EndOfLine = #13#10;
var
  i: Integer;
  ANode: TMSTreeNode;
  NodeStr: string;
begin
  if Count > 0 then
  begin
    ANode := Owner[0];
    while ANode <> nil do
    begin
      NodeStr := '';
      for i := 0 to ANode.Level - 1 do NodeStr := NodeStr + TabChar;
      NodeStr := NodeStr + ANode.Text + EndOfLine;
      Stream.Write(Pointer(NodeStr)^, Length(NodeStr));
      ANode := ANode.GetNext;
    end;
  end;
end;

{ TMSCustomTreeView }

constructor TMSCustomTreeView.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ControlStyle := ControlStyle - [csCaptureMouse] + [csDisplayDragImage, csReflector];
  Width := 121;
  Height := 97;
  TabStop := True;
  ParentColor := False;
  FTreeNodes := TMSTreeNodes.Create(Self);
  FBorderStyle := bsSingle;
  // multiselect
  FMultiSelect := False;
  FSelNodes := TList.Create;

  FShowButtons := True;
  FShowRoot := True;
  FShowLines := True;
  FHideSelection := True;
  FDragImage := TImageList.CreateSize(32, 32);
  FSaveIndent := -1;
  FEditInstance := MakeObjectInstance(EditWndProc);
  FImageChangeLink := TChangeLink.Create;
  FImageChangeLink.OnChange := ImageListChange;
  FStateChangeLink := TChangeLink.Create;
  FStateChangeLink.OnChange := ImageListChange;
end;

destructor TMSCustomTreeView.Destroy;
begin
  Items.Free;
  FSaveItems.Free;
  FDragImage.Free;
  FMemStream.Free;
  FreeObjectInstance(FEditInstance);
  FImageChangeLink.Free;
  FStateChangeLink.Free;
  inherited Destroy;
end;

procedure TMSCustomTreeView.CreateParams(var Params: TCreateParams);
const
  BorderStyles: array[TBorderStyle] of Integer = (0, WS_BORDER);
  LineStyles: array[Boolean] of Integer = (0, TVS_HASLINES);
  RootStyles: array[Boolean] of Integer = (0, TVS_LINESATROOT);
  ButtonStyles: array[Boolean] of Integer = (0, TVS_HASBUTTONS);
  EditStyles: array[Boolean] of Integer = (TVS_EDITLABELS, 0);
  HideSelections: array[Boolean] of Integer = (TVS_SHOWSELALWAYS, 0);
  DragStyles: array[TDragMode] of Integer = (TVS_DISABLEDRAGDROP, 0);
begin
  InitCommonControl(ICC_TREEVIEW_CLASSES);
  inherited CreateParams(Params);
  CreateSubClass(Params, WC_TREEVIEW);
  with Params do
  begin
    Style := Style or LineStyles[FShowLines] or BorderStyles[FBorderStyle] or
      RootStyles[FShowRoot] or ButtonStyles[FShowButtons] or
      EditStyles[FReadOnly] or HideSelections[FHideSelection] or
      DragStyles[DragMode];
    if Ctl3D and NewStyleControls and (FBorderStyle = bsSingle) then
    begin
      Style := Style and not WS_BORDER;
      ExStyle := Params.ExStyle or WS_EX_CLIENTEDGE;
    end;
    WindowClass.style := WindowClass.style and not (CS_HREDRAW or CS_VREDRAW);
  end;
end;

procedure TMSCustomTreeView.CreateWnd;
begin
  FStateChanging := False;
  inherited CreateWnd;
  if FMemStream <> nil then
  begin
    Items.ReadData(FMemStream);
    FMemStream.Destroy;
    FMemStream := nil;
    SetTopItem(Items.GetNodeFromIndex(FSaveTopIndex));
    FSaveTopIndex := 0;
    SetSelection(Items.GetNodeFromIndex(FSaveIndex));
    FSaveIndex := 0;
  end;
  if FSaveIndent <> -1 then Indent := FSaveIndent;
  if (Images <> nil) and Images.HandleAllocated then
    SetImageList(Images.Handle, TVSIL_NORMAL);
  if (StateImages <> nil) and StateImages.HandleAllocated then
    SetImageList(StateImages.Handle, TVSIL_STATE);
end;

procedure TMSCustomTreeView.DestroyWnd;
var
  Node: TMSTreeNode;
begin
  FStateChanging := True;
  if Items.Count > 0 then
  begin
    FMemStream := TMemoryStream.Create;
    Items.WriteData(FMemStream);
    FMemStream.Position := 0;
    Node := GetTopItem;
    if Node <> nil then FSaveTopIndex := Node.AbsoluteIndex;
    Node := Selected;
    if Node <> nil then FSaveIndex := Node.AbsoluteIndex;
  end;
  FSaveIndent := Indent;
  inherited DestroyWnd;
end;

procedure TMSCustomTreeView.EditWndProc(var Message: TMessage);
begin
  try
    with Message do
    begin
      case Msg of
        WM_KEYDOWN,
        WM_SYSKEYDOWN: if DoKeyDown(TWMKey(Message)) then Exit;
        WM_CHAR: if DoKeyPress(TWMKey(Message)) then Exit;
        WM_KEYUP,
        WM_SYSKEYUP: if DoKeyUp(TWMKey(Message)) then Exit;
        CN_KEYDOWN,
        CN_CHAR, CN_SYSKEYDOWN,
        CN_SYSCHAR:
          begin
            WndProc(Message);
            Exit;
          end;
      end;
      Result := CallWindowProc(FDefEditProc, FEditHandle, Msg, WParam, LParam);
    end;
  except
    Application.HandleException(Self);
  end;
end;

procedure TMSCustomTreeView.CMColorChanged(var Message: TMessage);
begin
  inherited;
  RecreateWnd;
end;

procedure TMSCustomTreeView.CMCtl3DChanged(var Message: TMessage);
begin
  inherited;
  if FBorderStyle = bsSingle then RecreateWnd;
end;

procedure TMSCustomTreeView.CMSysColorChange(var Message: TMessage);
begin
  inherited;
  if not (csLoading in ComponentState) then
  begin
    Message.Msg := WM_SYSCOLORCHANGE;
    DefaultHandler(Message);
  end;
end;

function TMSCustomTreeView.AlphaSort: Boolean;
var
  I: Integer;
begin
  if HandleAllocated then
  begin
    Result := CustomSort(nil, 0);
    for I := 0 to Items.Count - 1 do
      with Items[I] do
        if HasChildren then AlphaSort;
  end
  else Result := False;
end;

function TMSCustomTreeView.CustomSort(SortProc: TTVCompare; Data: Longint): Boolean;
var
  SortCB: TTVSortCB;
  I: Integer;
  Node: TMSTreeNode;
begin
  Result := False;
  if HandleAllocated then
  begin
    with SortCB do
    begin
      if not Assigned(SortProc) then lpfnCompare := @DefaulTMSTreeViewSort
      else lpfnCompare := SortProc;
      hParent := TVI_ROOT;
      lParam := Data;
      Result := TreeView_SortChildrenCB(Handle, SortCB, 0);
    end;
    for I := 0 to Items.Count - 1 do
    begin
      Node := Items[I];
      if Node.HasChildren then Node.CustomSort(SortProc, Data);
    end;
  end;
end;

procedure TMSCustomTreeView.SetSortType(Value: TSortType);
begin
  if SortType <> Value then
  begin
    FSortType := Value;
    if ((SortType in [stData, stBoth]) and Assigned(OnCompare)) or
      (SortType in [stText, stBoth]) then
      AlphaSort;
  end;
end;

procedure TMSCustomTreeView.SetStyle(Value: Integer; UseStyle: Boolean);
var
  Style: Integer;
begin
  if HandleAllocated then
  begin
    Style := GetWindowLong(Handle, GWL_STYLE);
    if not UseStyle then Style := Style and not Value
    else Style := Style or Value;
    SetWindowLong(Handle, GWL_STYLE, Style);
  end;
end;

procedure TMSCustomTreeView.SetBorderStyle(Value: TBorderStyle);
begin
  if BorderStyle <> Value then
  begin
    FBorderStyle := Value;
    RecreateWnd;
  end;
end;

procedure TMSCustomTreeView.SetDragMode(Value: TDragMode);
begin
  if Value <> DragMode then
    SetStyle(TVS_DISABLEDRAGDROP, Value = dmManual);
  inherited;
end;

procedure TMSCustomTreeView.SetButtonStyle(Value: Boolean);
begin
  if ShowButtons <> Value then
  begin
    FShowButtons := Value;
    SetStyle(TVS_HASBUTTONS, Value);
  end;
end;

procedure TMSCustomTreeView.SetLineStyle(Value: Boolean);
begin
  if ShowLines <> Value then
  begin
    FShowLines := Value;
    SetStyle(TVS_HASLINES, Value);
  end;
end;

procedure TMSCustomTreeView.SetRootStyle(Value: Boolean);
begin
  if ShowRoot <> Value then
  begin
    FShowRoot := Value;
    SetStyle(TVS_LINESATROOT, Value);
  end;
end;

procedure TMSCustomTreeView.SetReadOnly(Value: Boolean);
begin
  if ReadOnly <> Value then
  begin
    FReadOnly := Value;
    SetStyle(TVS_EDITLABELS, not Value);
  end;
end;

procedure TMSCustomTreeView.SetHideSelection(Value: Boolean);
begin
  if HideSelection <> Value then
  begin
    FHideSelection := Value;
    SetStyle(TVS_SHOWSELALWAYS, not Value);
    Invalidate;
  end;
end;

function TMSCustomTreeView.GetNodeAt(X, Y: Integer): TMSTreeNode;
var
  HitTest: TTVHitTestInfo;
begin
  with HitTest do
  begin
    pt.X := X;
    pt.Y := Y;
    if TreeView_HitTest(Handle, HitTest) <> nil then
      Result := Items.GetNode(HitTest.hItem)
    else Result := nil;
  end;
end;

function TMSCustomTreeView.GetHitTestInfoAt(X, Y: Integer): THitTests;
var
  HitTest: TTVHitTestInfo;
begin
  Result := [];
  with HitTest do
  begin
    pt.X := X;
    pt.Y := Y;
    TreeView_HitTest(Handle, HitTest);
    if (flags and TVHT_ABOVE) <> 0 then Include(Result, htAbove);
    if (flags and TVHT_BELOW) <> 0 then Include(Result, htBelow);
    if (flags and TVHT_NOWHERE) <> 0 then Include(Result, htNowhere);
    if (flags and TVHT_ONITEM) <> 0 then Include(Result, htOnItem);
    if (flags and TVHT_ONITEMBUTTON) <> 0 then Include(Result, htOnButton);
    if (flags and TVHT_ONITEMICON) <> 0 then Include(Result, htOnIcon);
    if (flags and TVHT_ONITEMINDENT) <> 0 then Include(Result, htOnIndent);
    if (flags and TVHT_ONITEMLABEL) <> 0 then Include(Result, htOnLabel);
    if (flags and TVHT_ONITEMRIGHT) <> 0 then Include(Result, htOnRight);
    if (flags and TVHT_ONITEMSTATEICON) <> 0 then Include(Result, htOnStateIcon);
    if (flags and TVHT_TOLEFT) <> 0 then Include(Result, htToLeft);
    if (flags and TVHT_TORIGHT) <> 0 then Include(Result, htToRight);
  end;
end;

procedure TMSCustomTreeView.SeTMSTreeNodes(Value: TMSTreeNodes);
begin
  Items.Assign(Value);
end;

procedure TMSCustomTreeView.SetIndent(Value: Integer);
begin
  if Value <> Indent then TreeView_SetIndent(Handle, Value);
end;

function TMSCustomTreeView.GetIndent: Integer;
begin
  Result := TreeView_GetIndent(Handle)
end;

procedure TMSCustomTreeView.FullExpand;
var
  Node: TMSTreeNode;
begin
  Node := Items.GetFirstNode;
  while Node <> nil do
  begin
    Node.Expand(True);
    Node := Node.GetNextSibling;
  end;
end;

procedure TMSCustomTreeView.FullCollapse;
var
  Node: TMSTreeNode;
begin
  Node := Items.GetFirstNode;
  while Node <> nil do
  begin
    Node.Collapse(True);
    Node := Node.GetNextSibling;
  end;
end;

procedure TMSCustomTreeView.Loaded;
begin
  inherited Loaded;
  if csDesigning in ComponentState then FullExpand;
end;

function TMSCustomTreeView.GetTopItem: TMSTreeNode;
begin
  if HandleAllocated then
    Result := Items.GetNode(TreeView_GetFirstVisible(Handle))
  else Result := nil;
end;

procedure TMSCustomTreeView.SetTopItem(Value: TMSTreeNode);
begin
  if HandleAllocated and (Value <> nil) then
    TreeView_SelectSetFirstVisible(Handle, Value.ItemId);
end;

function TMSCustomTreeView.GetSelection: TMSTreeNode;
begin
  if HandleAllocated then
  begin
    if FRightClickSelect and Assigned(FRClickNode) then
      Result := FRClickNode
    else
      Result := Items.GetNode(TreeView_GetSelection(Handle));
  end
  else Result := nil;
end;

procedure TMSCustomTreeView.SetSelection(Value: TMSTreeNode);
begin
  if Value <> nil then Value.Selected := True
  else TreeView_SelectItem(Handle, nil);
end;

function TMSCustomTreeView.GetDropTarget: TMSTreeNode;
begin
  if HandleAllocated then
  begin
    Result := Items.GetNode(TreeView_GetDropHilite(Handle));
    if Result = nil then Result := FLastDropTarget;
  end
  else Result := nil;
end;

procedure TMSCustomTreeView.SetDropTarget(Value: TMSTreeNode);
begin
  if HandleAllocated then
    if Value <> nil then Value.DropTarget := True
    else TreeView_SelectDropTarget(Handle, nil);
end;

function TMSCustomTreeView.GetNodeFromItem(const Item: TTVItem): TMSTreeNode;
begin
  with Item do
    if (state and TVIF_PARAM) <> 0 then Result := Pointer(lParam)
    else Result := Items.GetNode(hItem);
end;

function TMSCustomTreeView.IsEditing: Boolean;
var
  ControlHand: HWnd;
begin
  ControlHand := TreeView_GetEditControl(Handle);
  Result := (ControlHand <> 0) and IsWindowVisible(ControlHand);
end;

procedure TMSCustomTreeView.CNNotify(var Message: TWMNotify);
var
  // multiselect
  OldNode,SelNode,

  Node: TMSTreeNode;
  MousePos: TPoint;
begin
  with Message.NMHdr^ do
    case code of
{      NM_CUSTOMDRAW:
        begin
          with PNMCustomDrawInfo(Pointer(Message.NMHdr))^ do begin
            if (dwDrawStage and CDDS_PREPAINT)=CDDS_PREPAINT then
              Message.Result := CDRF_NOTIFYITEMDRAW;
            if (dwDrawStage and CDDS_ITEMPREPAINT)=CDDS_ITEMPREPAINT then begin
              if (uItemstate AND CDIS_SELECTED)=0 then begin
                SetTextColor(hdc,Font.Color);
                SetBkColor(hdc,Color);
              end;
              if (Not (csDesigning in ComponentState)) and Assigned (OnAskForColor) then begin
                SetTextColor(hdc,GetColor(OnAskForColor(Self,lItemlParam)));
                SetBkColor(hdc,GetColor(OnAskForColor(Self,lItemlParam)));
              end;
              Message.Result:= CDRF_NOTIFYITEMDRAW;
            end;
          end;
        end;}
      TVN_BEGINDRAG:
        begin
          FDragged := True;
          with PNMTreeView(Pointer(Message.NMHdr))^ do
            FDragNode := GetNodeFromItem(ItemNew);
        end;
      TVN_BEGINLABELEDIT:
        begin
          with PTVDispInfo(Pointer(Message.NMHdr))^ do
            if Dragging or not CanEdit(GetNodeFromItem(item)) then
              Message.Result := 1;
          if Message.Result = 0 then
          begin
            FEditHandle := TreeView_GetEditControl(Handle);
            FDefEditProc := Pointer(GetWindowLong(FEditHandle, GWL_WNDPROC));
            SetWindowLong(FEditHandle, GWL_WNDPROC, LongInt(FEditInstance));
          end;
        end;
      TVN_ENDLABELEDIT:
        with PTVDispInfo(Pointer(Message.NMHdr))^ do
          Edit(item);
      TVN_ITEMEXPANDING:
        if not FManualNotify then
        begin
          with PNMTreeView(Pointer(Message.NMHdr))^ do
          begin
            Node := GetNodeFromItem(ItemNew);
            if (action = TVE_EXPAND) and not CanExpand(Node) then
              Message.Result := 1
            else if (action = TVE_COLLAPSE) and
              not CanCollapse(Node) then Message.Result := 1;
          end;
        end;
      TVN_ITEMEXPANDED:
        if not FManualNotify then
        begin
          with PNMTreeView(Pointer(Message.NMHdr))^ do
          begin
            Node := GetNodeFromItem(itemNew);
            if (action = TVE_EXPAND) then Expand(Node)
            else if (action = TVE_COLLAPSE) then Collapse(Node);
          end;
        end;
      TVN_SELCHANGING:
        with PNMTreeView(Pointer(Message.NMHdr))^ do
          if not CanChange(GetNodeFromItem(itemNew)) then
            Message.Result := 1;
      TVN_SELCHANGED:
        with PNMTreeView(Pointer(Message.NMHdr))^ do begin
          if MultiSelect then begin
            Node := GetNodeFromItem(itemNew);
            if Node=nil then begin
              ClearSelNodes;
              AddSelNode(Node);
            end
            else if FDeleting then begin
              OldNode := GetNodeFromItem(itemOld);
              if Node<>nil then begin
                Node.MakeSelected(False);
                Node.MakeFocused(True);
              end;
              OldNode.MakeSelected(True);
            end
            else if FShiftDown then begin
              OldNode := GetNodeFromItem(itemOld);
              if OldNode=nil then begin
                ClearSelNodes;
                AddSelNode(Node);
              end
              else begin
                SelNode := Node;
                if Node.Order<=OldNode.Order then
                  Repeat
                    AddSelNode(SelNode);
                    SelNode := SelNode.GetNextVisible
                  Until SelNode=OldNode
                else
                  Repeat
                    AddSelNode(SelNode);
                    SelNode := SelNode.GetPrevVisible;
                  Until SelNode=OldNode;
                OldNode.MakeSelected(True);
              end;
            end
            else if FShiftKeyDown then begin
              OldNode := GetNodeFromItem(itemOld);
              if OldNode=nil then begin
                ClearSelNodes;
                AddSelNode(Node);
              end
              else begin
                SelNode := Node;
                if Node.Order<=OldNode.Order then
                  Repeat
                    AddSelNode(SelNode);
                    SelNode := SelNode.GetNextVisible;
                  Until SelNode=OldNode
                else
                  Repeat
                    AddSelNode(SelNode);
                    SelNode := SelNode.GetPrevVisible;
                  Until SelNode=OldNode;
                OldNode.MakeSelected(True);
              end;
            end
            else if FCtrlDown then begin
              OldNode := GetNodeFromItem(itemOld);
              if Node=nil then begin
                ClearSelNodes;
                AddSelNode(Node);
              end
              else begin
                if FindSelNode(OldNode)<>-1 then
                  OldNode.MakeSelected(True);
                if FindSelNode(Node)=-1 then
                  AddSelNode(Node)
                else
                  DelSelNode(Node);
              end;
            end
            else begin
              if DragMode=dmAutomatic then
                if not FDragged then begin
                  ClearSelNodes;
                  AddSelNode(Node);
                end
                else begin
                  if FindSelNode(Node)=-1 then begin
                    ClearSelNodes;
                    AddSelNode(Node);
                  end
                  else begin
                    OldNode := GetNodeFromItem(itemOld);
                    OldNode.MakeSelected(True);
                  end;
                end
              else begin
                ClearSelNodes;
                AddSelNode(Node);
              end;
            end;
          end;
          Change(GetNodeFromItem(itemNew));
        end;
      TVN_DELETEITEM:
        begin
          if not FStateChanging then
          begin
            with PNMTreeView(Pointer(Message.NMHdr))^ do
              Node := GetNodeFromItem(itemOld);
            if Node <> nil then
            begin
              // multiselect
              DelSelNode(Node);

              Node.FItemId := nil;
              Items.Delete(Node);
            end;
          end;
        end;
      TVN_SETDISPINFO:
        with PTVDispInfo(Pointer(Message.NMHdr))^ do
        begin
          Node := GetNodeFromItem(item);
          if (Node <> nil) and ((item.mask and TVIF_TEXT) <> 0) then
            Node.Text := item.pszText;
        end;
      TVN_GETDISPINFO:
        with PTVDispInfo(Pointer(Message.NMHdr))^ do
        begin
          Node := GetNodeFromItem(item);
          if Node <> nil then
          begin
            if (item.mask and TVIF_TEXT) <> 0 then
              StrLCopy(item.pszText, PChar(Node.Text), item.cchTextMax);
            if (item.mask and TVIF_IMAGE) <> 0 then
            begin
              GetImageIndex(Node);
              item.iImage := Node.ImageIndex;
            end;
            if (item.mask and TVIF_SELECTEDIMAGE) <> 0 then
            begin
              GetSelectedIndex(Node);
              item.iSelectedImage := Node.SelectedIndex;
            end;
          end;
        end;
      NM_RCLICK:
        begin
          if RightClickSelect then
          begin
            GetCursorPos(MousePos);
            with PointToSmallPoint(ScreenToClient(MousePos)) do
            begin
              FRClickNode := GetNodeAt(X, Y);
              Perform(WM_RBUTTONUP, 0, MakeLong(X, Y));
            end;
          end
          else FRClickNode := Pointer(1);
        end;
    end;
end;

function TMSCustomTreeView.GetDragImages: TCustomImageList;
begin
  if FDragImage.Count > 0 then
    Result := FDragImage else
    Result := nil;
end;

procedure TMSCustomTreeView.WndProc(var Message: TMessage);
begin
  if not (csDesigning in ComponentState) and ((Message.Msg = WM_LBUTTONDOWN) or
    (Message.Msg = WM_LBUTTONDBLCLK)) and not Dragging and (DragMode = dmAutomatic) then
  begin
    if not IsControlMouseMsg(TWMMouse(Message)) then
    begin
      ControlState := ControlState + [csLButtonDown];
      Dispatch(Message);
    end;
  end
  else inherited WndProc(Message);
end;

procedure TMSCustomTreeView.DoStartDrag(var DragObject: TDragObject);
var
  ImageHandle: HImageList;
  DragNode: TMSTreeNode;
  P: TPoint;
begin
  inherited DoStartDrag(DragObject);
  DragNode := FDragNode;
  FLastDropTarget := nil;
  FDragNode := nil;
  if DragNode = nil then
  begin
    GetCursorPos(P);
    with ScreenToClient(P) do DragNode := GetNodeAt(X, Y);
  end;
  if DragNode <> nil then
  begin
    ImageHandle := TreeView_CreateDragImage(Handle, DragNode.ItemId);
    if ImageHandle <> 0 then
      with FDragImage do
      begin
        Handle := ImageHandle;
        SetDragImage(0, 2, 2);
      end;
  end;
end;

procedure TMSCustomTreeView.DoEndDrag(Target: TObject; X, Y: Integer);
begin
  inherited DoEndDrag(Target, X, Y);
  FLastDropTarget := nil;
end;

procedure TMSCustomTreeView.CMDrag(var Message: TCMDrag);
begin
  inherited;
  with Message, DragRec^ do
    case DragMessage of
      dmDragMove: with ScreenToClient(Pos) do DoDragOver(Source, X, Y, Message.Result<>0);
      dmDragLeave:
        begin
          TDragObject(Source).HideDragImage;
          FLastDropTarget := DropTarget;
          DropTarget := nil;
          TDragObject(Source).ShowDragImage;
        end;
      dmDragDrop: FLastDropTarget := nil;
    end;
end;

procedure TMSCustomTreeView.DoDragOver(Source: TDragObject; X, Y: Integer; CanDrop: Boolean);
var
  Node: TMSTreeNode;
begin
  Node := GetNodeAt(X, Y);
  if (Node <> nil) and
    ((Node <> DropTarget) or (Node = FLastDropTarget)) then
  begin
    FLastDropTarget := nil;
    TDragObject(Source).HideDragImage;
    Node.DropTarget := CanDrop;
    TDragObject(Source).ShowDragImage;
  end;
end;

procedure TMSCustomTreeView.GetImageIndex(Node: TMSTreeNode);
begin
  if Assigned(FOnGetImageIndex) then FOnGetImageIndex(Self, Node);
end;

procedure TMSCustomTreeView.GetSelectedIndex(Node: TMSTreeNode);
begin
  if Assigned(FOnGetSelectedIndex) then FOnGetSelectedIndex(Self, Node);
end;

function TMSCustomTreeView.CanChange(Node: TMSTreeNode): Boolean;
begin
  Result := True;
  if Assigned(FOnChanging) then FOnChanging(Self, Node, Result);
end;

procedure TMSCustomTreeView.Change(Node: TMSTreeNode);
begin
  if Assigned(FOnChange) then FOnChange(Self, Node);
end;

procedure TMSCustomTreeView.Expand(Node: TMSTreeNode);
begin
  if Assigned(FOnExpanded) then FOnExpanded(Self, Node);
end;

function TMSCustomTreeView.CanExpand(Node: TMSTreeNode): Boolean;
begin
  Result := True;
  if Assigned(FOnExpanding) then FOnExpanding(Self, Node, Result);
end;

procedure TMSCustomTreeView.Collapse(Node: TMSTreeNode);
begin
  if Assigned(FOnCollapsed) then FOnCollapsed(Self, Node);
end;

function TMSCustomTreeView.CanCollapse(Node: TMSTreeNode): Boolean;
begin
  Result := True;
  if Assigned(FOnCollapsing) then FOnCollapsing(Self, Node, Result);
end;

function TMSCustomTreeView.CanEdit(Node: TMSTreeNode): Boolean;
begin
  Result := True;
  if Assigned(FOnEditing) then FOnEditing(Self, Node, Result);
end;

procedure TMSCustomTreeView.Edit(const Item: TTVItem);
var
  S: string;
  Node: TMSTreeNode;
begin
  with Item do
    if pszText <> nil then
    begin
      S := pszText;
      Node := GetNodeFromItem(Item);
      if Assigned(FOnEdited) then FOnEdited(Self, Node, S);
      if Node <> nil then Node.Text := S;
    end;
end;

function TMSCustomTreeView.CreateNode: TMSTreeNode;
begin
  Result := TMSTreeNode.Create(Items);
end;

procedure TMSCustomTreeView.SetImageList(Value: HImageList; Flags: Integer);
begin
  if HandleAllocated then TreeView_SetImageList(Handle, Value, Flags);
end;

procedure TMSCustomTreeView.ImageListChange(Sender: TObject);
var
  ImageHandle: HImageList;
begin
  if HandleAllocated then
  begin
    ImageHandle := TImageList(Sender).Handle;
    if Sender = Images then
      SetImageList(ImageHandle, TVSIL_NORMAL)
    else if Sender = StateImages then
      SetImageList(ImageHandle, TVSIL_STATE);
  end;
end;

procedure TMSCustomTreeView.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if Operation = opRemove then
  begin
    if AComponent = Images then Images := nil;
    if AComponent = StateImages then StateImages := nil;
  end;
end;

procedure TMSCustomTreeView.SetImages(Value: TImageList);
begin
  if Images <> nil then
    Images.UnRegisterChanges(FImageChangeLink);
  FImages := Value;
  if Images <> nil then
  begin
    Images.RegisterChanges(FImageChangeLink);
    SetImageList(Images.Handle, TVSIL_NORMAL)
  end
  else SetImageList(0, TVSIL_NORMAL);
end;

procedure TMSCustomTreeView.SetStateImages(Value: TImageList);
begin
  if StateImages <> nil then
    StateImages.UnRegisterChanges(FStateChangeLink);
  FStateImages := Value;
  if StateImages <> nil then
  begin
    StateImages.RegisterChanges(FStateChangeLink);
    SetImageList(StateImages.Handle, TVSIL_STATE)
  end
  else SetImageList(0, TVSIL_STATE);
end;

procedure TMSCustomTreeView.LoadFromFile(const FileName: string);
var
  Stream: TStream;
begin
  Stream := TFileStream.Create(FileName, fmOpenRead);
  try
    LoadFromStream(Stream);
  finally
    Stream.Free;
  end;
end;

procedure TMSCustomTreeView.LoadFromStream(Stream: TStream);
begin
  with TMSTreeStrings.Create(Items) do
    try
      LoadTreeFromStream(Stream);
    finally
      Free;
  end;
end;

procedure TMSCustomTreeView.SaveToFile(const FileName: string);
var
  Stream: TStream;
begin
  Stream := TFileStream.Create(FileName, fmCreate);
  try
    SaveToStream(Stream);
  finally
    Stream.Free;
  end;
end;

procedure TMSCustomTreeView.SaveToStream(Stream: TStream);
begin
  with TMSTreeStrings.Create(Items) do
    try
      SaveTreeToStream(Stream);
    finally
      Free;
  end;
end;

procedure TMSCustomTreeView.WMRButtonDown(var Message: TWMRButtonDown);
var
  MousePos: TPoint;
begin
  FRClickNode := nil;
  try
    if not RightClickSelect then
    begin
      inherited;
      if FRClickNode <> nil then
      begin
        GetCursorPos(MousePos);
        with PointToSmallPoint(ScreenToClient(MousePos)) do
          Perform(WM_RBUTTONUP, 0, MakeLong(X, Y));
      end;
    end
    else DefaultHandler(Message);
  finally
    FRClickNode := nil;

  end;
end;

procedure TMSCustomTreeView.WMRButtonUp(var Message: TWMRButtonUp);

  procedure DoMouseDown(var Message: TWMMouse; Button: TMouseButton;
    Shift: TShiftState);
  begin
    if not (csNoStdEvents in ControlStyle) then
      with Message do
        MouseDown(Button, KeysToShiftState(Keys) + Shift, XPos, YPos);
  end;

begin
  if RightClickSelect then DoMouseDown(Message, mbRight, []);
  inherited;
end;

procedure TMSCustomTreeView.WMLButtonDown(var Message: TWMLButtonDown);
var
  Node: TMSTreeNode;
  MousePos: TPoint;

  // multiselect
  ShiftData: TShiftState;
begin
  FDragged := False;
  FDragNode := nil;
  try
    // multiselect
    ShiftData := KeysToShiftState(Message.Keys);
    if ShiftData=[ssShift,ssLeft] then begin
      FCtrlDown := False;
      FShiftDown := True;
    end
    else if ShiftData=[ssCtrl,ssLeft] then begin
      FCtrlDown := True;
      FShiftDown := False;
    end
    else begin
      FCtrlDown := False;
      FShiftDown := False;
    end;
    FShiftKeyDown := False;

    inherited;
    if DragMode = dmAutomatic then
    begin
      SetFocus;
      if not FDragged then
      begin
        GetCursorPos(MousePos);
        with PointToSmallPoint(ScreenToClient(MousePos)) do
          Perform(WM_LBUTTONUP, 0, MakeLong(X, Y));
      end
      else begin
        Node := GetNodeAt(Message.XPos, Message.YPos);
        if Node <> nil then
        begin
          Node.Focused := True;
          Node.Selected := True;
          BeginDrag(False);
        end;
      end;
    end;
  finally
    FDragNode := nil;
  end;
end;

procedure TMSCustomTreeView.WMNotify(var Message: TWMNotify);
var
  Node: TMSTreeNode;
  MaxTextLen: Integer;
  Pt: TPoint;
begin
  with Message do
    if NMHdr^.code = TTN_NEEDTEXTW then
    begin
      // Work around NT COMCTL32 problem with tool tips >= 80 characters
      GetCursorPos(Pt);
      Pt := ScreenToClient(Pt);
      Node := GetNodeAt(Pt.X, Pt.Y);
      if (Node = nil) or (Node.Text = '') then Exit;
      FWideText := Node.Text;
      MaxTextLen := SizeOf(PToolTipTextW(NMHdr)^.szText) div SizeOf(WideChar);
      if Length(FWideText) >= MaxTextLen then
        SetLength(FWideText, MaxTextLen - 1);
      PToolTipTextW(NMHdr)^.lpszText := PWideChar(FWideText);
      FillChar(PToolTipTextW(NMHdr)^.szText, MaxTextLen, 0);
      Move(Pointer(FWideText)^, PToolTipTextW(NMHdr)^.szText, Length(FWideText));
      PToolTipTextW(NMHdr)^.hInst := 0;
      SetWindowPos(NMHdr^.hwndFrom, HWND_TOP, 0, 0, 0, 0, SWP_NOACTIVATE or
        SWP_NOSIZE or SWP_NOMOVE);
      Result := 1;
    end
    else inherited;
end;

// multiselect
function TMSCustomTreeView.GetSelNode(Index: Integer): TMSTreeNode;
begin
  if Index<FSelNodes.Count then
    Result := TMSTreeNode(FSelNodes[Index])
  else
    Result := nil;
end;

// multiselect
function TMSCustomTreeView.GetSelCount: Integer;
begin
  if MultiSelect then
    Result := FSelNodes.Count
  else
    Result := 0;
end;

// multiselect
procedure TMSCustomTreeView.SetMultiSelect(Value: Boolean);
begin
  FMultiSelect := Value;
end;

// multiselect
procedure TMSCustomTreeView.AddSelNode(Node: TMSTreeNode);
begin
  if (Node<>nil) and MultiSelect then begin
    if FindSelNode(Node)=-1 then begin
      FSelNodes.Add(Node);
      Node.MakeSelected(True);
    end;
  end;
end;

// multiselect
procedure TMSCustomTreeView.ClearSelNodes;
begin
  if MultiSelect then
    while FSelNodes.Count>0 do begin
      TMSTreeNode(FSelNodes[0]).MakeSelected(False);
      FSelNodes.Remove(TMSTreeNode(FSelNodes[0]));
    end;
end;

// multiselect
procedure TMSCustomTreeView.DelSelNode(Node: TMSTreeNode);
begin
  if (Node<>nil) and MultiSelect then begin
    FSelNodes.Remove(Node);
    Node.MakeSelected(False);
  end;
end;

// multiselect
function TMSCustomTreeView.FindSelNode(Node: TMSTreeNode): Integer;
var
  Cnt: Integer;
begin
  Cnt := 0;
  while (Cnt<SelCount) and (SelNodes[Cnt]<>Node) do
    Inc(Cnt);
  if Cnt>=SelCount then
    Result := -1
  else
    Result := 0;
end;

// multiselect
function CustomSortProc(Item1, Item2: Pointer): Integer;
begin
  if TMSTreeNode(Item1).Order<TMSTreeNode(Item2).Order then
    Result := -1
  else if TMSTreeNode(Item1).Order>TMSTreeNode(Item2).Order then
    Result := 1
  else
    Result := 0;
end;

// multiselect
procedure TMSCustomTreeView.SortSelNodes;
begin
  FSelNodes.Sort(CustomSortProc);
end;

// multiselect
procedure TMSCustomTreeView.WMKeyDown(var Message: TWMKeyDown);
var
  ShiftData: TShiftState;
begin
  ShiftData := KeyDataToShiftState(Message.KeyData);
  if not FShiftKeyDown then begin
    if ShiftData=[ssShift] then
      FShiftKeyDown := True;
  end
  else if not (ShiftData=[ssShift]) then
    FShiftKeyDown := False;

  FShiftDown := False;
  FCtrlDown := False;
  inherited;
end;

// multiselect
procedure TMSCustomTreeView.DoEnter;
begin
  inherited DoEnter;
  FShiftDown := False;
  FCtrlDown := False;
end;

{ TMSTreeNodesProperty }

{type
  TMSTreeNodesProperty = class(TClassProperty)
  public
    procedure Edit; override;
    function GetAttributes: TPropertyAttributes; override;
  end;

procedure TMSTreeNodesProperty.Edit;
begin
  if EditTreeViewItems(TMSTreeNodes(GetOrdValue)) then Modified;
end;

function TMSTreeNodesProperty.GetAttributes: TPropertyAttributes;
begin
  Result := [paDialog, paReadOnly];
end;}

{ Registration }

{procedure Register;
begin
  RegisterComponents('New', [TMSTreeView]);
end;}

end.
