unit MSSQL;
(* GERMAN ***************************************************************)
(* DATEI : MSSQL.PAS                   OS : WIN NT 3.51/4.0             *)
(*                                                                      *)
(* Version  Datum      Autor    Kommentar                               *)
(* -------------------------------------------------------------------- *)
(*  1.0.0   19.09.1997 Seban    erstellt                                *)
(*  1.1.0   30.10.1997 Seban    Pro Session kann es nur ein einziges    *)
(*                              Objekt TDBPROCESS existieren, daher     *)
(*                              wird es vom Objekt TSQLSession bereit-  *)
(*                              gestellt.                               *)
(*  2.0.0   09.11.1997 Seban    Da kein Cursor geschlossen werden kann, *)
(*                              wenn dieser innerhalb des IIS mit der   *)
(*                              Funktion dbopencursor geoeffnet wurden. *)
(*                              Als Workaround werden die Daten mit der *)
(*                              Funktion dbopen ausgelesen und im Spei- *)
(*                              cher bereitgestellt.                    *)
(*  2.0.01  22.02.1998 Seban    neue Funktionen zum auslesen der vor-   *)
(*                              Datenbanken und Tabellen erstellt.      *)
(*                                                                      *)
(* Bildet die ESQL-Schnittstelle zum MSSQL-Server nach und erlaubt so   *)
(* die Ausfuehrung von SQL-Anweisungen. Dazu muss der Client vom SQL-   *)
(* Server installiert sein, da die DLL NTWDBLIB.DLL benoetigt wird.     *)
(*                                                                      *)
(* ACHTUNG!!!                                                           *)
(* Die Verwendung eines Fehlercallback schlug fehl. Daher gibt es keine *)
(* Fehlermeldungen.                                                     *)
(************************************************************************)

(* ENGLISH **************************************************************)
(* FILE : MSSQL.PAS                    OS : WIN NT 3.51/4.0             *)
(*                                                                      *)
(* Version  Date       Author   Comment                                 *)
(* -------------------------------------------------------------------- *)
(*  1.0.0   19.09.1997 Seban    created                                 *)
(*  2.0.0   09.11.1997 Seban    If this unit called by IIS from an ISAPI*)
(*                              DLL, open cursors are not closed, if    *)
(*                              used scrollable cursors. In a normal ap-*)
(*                              plication it works fine. As workaround  *)
(*                              i catch all lines with a forward cursor *)
(*                              only into memory and close cursor       *)
(*                              immedently.                             *)
(*  2.0.01  22.02.1998 Seban    added some new function for session     *)
(*                              object                                  *)
(*                                                                      *)
(* Accessing SQL-Server without ODBC and BDE. You need only the DLL     *)
(* NTWDBLIB.DLL from client package of SQL-Server.                      *)
(*                                                                      *)
(************************************************************************)

interface

{$DEFINE USECALLBACKS}         (* Enable Errorreporting from SQL-Server *)

uses LogFiles,
     Windows,
     SysUtils,
     Classes,
     MSSQLLib;

type
  TSQLData         = array[0..31] of Char;

  TSQLState        = (
                       dsInactive,
                       dsActive,
                       dsBrowse,
                       dsEdit,
                       dsInsert,
                       dsDelete
                     );

  TSQLTypes        = (
                       stString,
                       stInteger,
                       stNumeric
                     );

  TErrorObject     = class(TObject)
  private
    fDBProcess       : PDBPRocess;
    fErrorNumber     : Integer;
    fLineNr          : Integer;
    fErrorMsg        : String;
  public
    constructor Create(aDBProc : PDBProcess;
                       aErrStr : PChar;
                       aErrNr  : Integer;
                       aLineNr : Integer);
    property DBProcess : PDBProcess read fDBProcess;
    property ErrorNumber : Integer read fErrorNumber;
    property ErrorMsg : String read fErrorMsg;
    property LineNr : Integer read fLineNr;
  end;

  TSQLList         = class(TStringList)
  protected
    fSize            : Integer;
    fSQL             : PChar;
    function GetSQL : PChar; virtual;
  public
    constructor Create;
    destructor Destroy; override;
    property SQL : PChar read GetSQL;
  end;

  TSQLSession      = class(TObject)
  private
    fActive          : Boolean;
    fLogin           : PLOGINREC;
    fHandle          : PDBPROCESS;
    fHost            : TSQLData;
    fDatabase        : TSQLData;
    fUsername        : TSQLData;
    fPassword        : TSQLData;

    fDevices         : TStringList;
    fTables          : TStringList;
    fIndex           : TStringList;
    fDatabases       : TStringList;
    fUserList        : TStringList;
  protected
    procedure ShowError(Error : Integer; Msg : String); dynamic;
    procedure SetActive(Data : Boolean); virtual;
    function  GetHost : String; virtual;
    procedure SetHost(Data : String); virtual;
    function  GetDatabase : String; virtual;
    procedure SetDatabase(Data : String); virtual;
    function  GetUsername : String; virtual;
    procedure SetUsername(Data : String); virtual;
    function  GetPassword : String; virtual;
    procedure SetPassword(Data : String); virtual;
    function  GetDevices : TStringList;
    function  GetDatabases : TStringList;
    function  GetTables : TStringList;
    function  GetUsers : TStringList;
    function  GetIndex(Tables : String) : TStringList;
  public
    constructor Create;
    destructor Destroy; override;
    function  GetPHost : PChar; virtual;
    function  GetPDatabase : PChar; virtual;
    procedure SetLoginTime(Seconds : Integer); dynamic;
    procedure DoLogin; dynamic;
    procedure DoLogoff; dynamic;

    function  GetDeviceDatafile(const Device : String) : String;

    procedure StartTransaction; dynamic;
    procedure Commit; dynamic;
    procedure Rollback; dynamic;

    property  Active : Boolean read fActive write SetActive;
    property  Database : String read GetDatabase write SetDatabase;
    property  Handle : PDBPROCESS read fHandle;
    property  Host : String read GetHost write SetHost;
    property  Login : PLOGINREC read fLogin;
    property  Password : String read GetPassword write SetPassword;
    property  Username : String read GetUsername write SetUsername;

    property  Devices : TStringList read GetDevices;
    property  Databases : TStringList read GetDatabases;
    property  Tables : TStringList read GetTables;
    property  Index[Tables : String] : TStringList read GetIndex;
    property  Users : TStringList read GetUsers;
  end;

  TSQLQuery        = class;
  TCharBuffer      = array[0..255] of Char;

  (* TField wird vom Objekt TStringList abgeleitet und speichert die    *)
  (* Rueckgabewerte einer Spalter des Abfrageergebnisses. Eine Zeile    *)
  (* wird durch den Zugriff mit dem gleichen Index fuer jede Spalter    *)
  (* ausgelesen.                                                        *)
  TField           = class(TStringList)
  private
    fBuffer           : TCharBuffer;
    fFieldName        : String;
    fFormat           : String;
    fFieldSize        : Integer;
    fQuery            : TSQlQuery;
  protected
    procedure SetString(Data : String); virtual;
    function  GetString : String; virtual;
    function  GetFieldSize : Integer; virtual;
  public
    constructor Create(Query     : TSQlQuery;
                       FieldName : String;
                       Format    : String);
    property AsString : String read  GetString
                               write SetString;
    property Buffer : TCharBuffer read fBuffer;
    property FieldName : String read fFieldname;
    property FieldSize : Integer read GetFieldSize;
  end;

  (* Das Objekt TSQLResult speichert ein Abfrageergebnis in seiner Ge-  *)
  (* samtheit. Dadurch wird der Speicher zwar komplett reserviert, auch *)
  (* wenn nur die ersten Zeilen von Interesse sind, beschleunigt aber   *)
  (* den Zugriff, wenn auf mehrer Seiten nacheinander zugriffen wird.   *)
  (* Der Programmierer kann ueber eine Schnittstelle auf den Prozess    *)
  (* einwirken und die Daten aus der physikalischen Sicht in eine vir-  *)
  (* tuelle Sicht umwandeln.                                            *)
  TOnGetSQLData    = procedure(Cursor : PDBPROCESS);

  TSQLResult       = class(TObject)
  private
    fData             : TObject;
    fOwner            : TSQLQuery;
    fFields           : TList;            (* Liste der Felder im Report *)
  protected
    function GetCell(Col, Row : Cardinal) : String;
    function GetColCount : Cardinal;
    function GetRowCount : Cardinal;
  public
    constructor Create(Query : TSQlQuery);
    destructor Destroy; override;
    procedure Clear;
    procedure AddField(FieldName, Format : String);
    function  FindRow(Start : Cardinal; Ident : array of String) : Cardinal;
    property ColCount : Cardinal read GetColCount;
    property RowCount : Cardinal read GetRowCount;
    property Cell[Col, Row : Cardinal] : String read GetCell;
  end;

  TSQLQuery        = class(TObject)
  private
    fSession         : TSQLSession;              (* Zugehoerige Session *)

    fRowNumber       : Integer;                          (* 0 basierend *)
    fRowCount        : Integer;            (* Anzahl der Ergebnissaetze *)
    fColCount        : Integer;(* Anzahl der Spalten der SQl-Statements *)
    fFields          : TList; (* Liste der Spalten eines SQL-Statements *)

    fState           : TSQLState;

    fErrCode         : Integer;            (* Fehlerflag => SUCCESS = 0 *)
    fSQL             : TSQLList;
  protected
    procedure SetActive(Data : Boolean); virtual;
    function  GetActive : Boolean; virtual;
    function  GetHandle : PDBPROCESS; virtual;
    function  GetFieldByName(Name : String) : TField; virtual;
    function  GetFieldByIndex(Index : Integer) : TField; virtual;
    function  GetRowNumber : Integer; virtual;
    function  GetBOF : Boolean; virtual;
    function  GetEOF : Boolean; virtual;
  public
    constructor Create(aSession : TSQLSession);
    destructor Destroy; override;
    procedure ExecSQL;
    procedure ExecSQLEx(aSQL : String; IgnoreException : Boolean);
    procedure Open(bReadOnly : Boolean; OnGetSQLData : TOnGetSQLData);
    function  First : Boolean;
    function  Previous : Boolean;
    function  Next : Boolean;
    function  Last : Boolean;
    function  MoveTo(Row : Integer) : Boolean;
    function  MoveBy(Row : Integer) : Boolean;
    procedure Close;

    property  Active : Boolean read GetActive write SetActive;
    property  BOF : Boolean read GetBOF;
    property  ColCount : Integer read fColCount;
    property  EOF : Boolean read GetEOF;
    property  FieldByName[Name : String] : TField read GetFieldByName;
    property  Fields[Index : Integer] : TField read GetFieldByIndex;
    property  Handle : PDBPROCESS read GetHandle;

    property  RowCount : Integer read fRowCount;
    property  RowNumber : Integer read GetRowNumber;
    property  Session : TSQLSession read fSession;
    property  SQL : TSQLList read fSQL;
    property  State : TSQLState read fState;
  end;

{$IFDEF USECALLBACKS}
function ErrHandler(DBProc    : Pointer;
                    Severity  : Integer;
                    DBError   : Integer;
                    OSError   : Integer;
                    DBErrStr  : PChar;
                    OSErrStr  : PChar) : Integer; stdcall;

function MsgHandler(DBProc   : Pointer;
                    MsgNo    : Integer;
                    MsgState : Integer;
                    Severity : Integer;
                    MsgText  : PChar;
	            SrvName  : PChar;
                    ProcName : PChar;
                    LineNr   : SmallInt) : Integer; stdcall;
{$ENDIF}

var
  DefaultSession   : TSQLSession;

implementation

const
  TEmptySQL        : PChar = ''#0;

{$IFDEF USECALLBACKS}
var
  LastErrorMsg     : array[0..511] of Char;
  LastErrorCode    : Integer;
  LastLineNr       : Integer;
    
{$ENDIF}

{$IFDEF USECALLBACKS}
function ErrHandler(DBProc    : Pointer;
                    Severity  : Integer;
                    DBError   : Integer;
                    OSError   : Integer;
                    DBErrStr  : PChar;
                    OSErrStr  : PChar) : Integer;
begin
  AddLogItem('MSSQL', 242, '', 'DB-Library : "%s"(%d)', [DBErrStr, DBError]);

  if (DBError <> 10007) then begin
    StrLCopy(LastErrorMsg, DBErrStr, SizeOf(LastErrorMsg) - 1);
    LastErrorCode := DBError;
  end;

  if (Severity = EXCOMM) and (OSErrStr <> nil) then begin
    AddLogItem('MSSQL', 250, '', 'Net-Lib error %d: "%s"', [OSError, OSErrStr]);
  end;

  if (OSError <> DBNOERR) then begin
    AddLogItem('MSSQL', 254, '', 'Operating-system error: "%s"', [OSErrStr]);
  end;

  if (DBProc = nil) or DBDEAD(DBProc) then begin
    Result := INT_EXIT;
  end else begin
    Result := INT_CANCEL;
  end;
end;

function MsgHandler(DBProc   : Pointer;
                    MsgNo    : Integer;
                    MsgState : Integer;
                    Severity : Integer;
                    MsgText  : PChar;
	            SrvName  : PChar;
                    ProcName : PChar;
                    LineNr   : SmallInt) : Integer;
begin
  if (Severity > 0) then begin
    AddLogItem('MSSQL', 274, '', 'SQL-Server : "%s(%d, %d)"', [MsgText, MsgNo, Severity]);
    StrLCopy(LastErrorMsg, MsgText, SizeOf(LastErrorMsg) - 1);
    LastErrorCode := MsgNo;
    LastLineNr    := LineNr;
  end;
  Result := 0;
end;
{$ENDIF}

constructor TErrorObject.Create(aDBProc : PDBProcess;
                                aErrStr : PChar;
                                aErrNr  : Integer;
                                aLineNr : Integer);
begin
  inherited Create;

  fDBProcess   := aDBProc;
  fErrorNumber := aErrNr;
  fErrorMsg    := aErrStr;
  fLineNr      := aLineNr;
end;

(* TSQLList ----------------------------------------------------------- *)
function TSQLList.GetSQL : PChar;
const
  ErrMsg           = 'SQL-String konnte nicht reserviert werden!';
  
var
  Loop             : Integer;
  pEnd             : PChar;
  oSize            : Integer;
  tStr             : String;
begin
  Result := TEmptySQL;
  if (Count > 0) then begin
    (* Bestimme nun die Groesse der SQL-Anweisung                       *)
    oSize := 0;
    for Loop := 0 to (Count - 1) do begin
      oSize := oSize + Length(Strings[Loop]) + 2;
    end;
    Inc(oSize);                       (* Das #0 Zeichen nicht vergessen *)

    if (oSize > fSize) then begin
      if (fSQL <> nil) then begin
        FreeMem(fSQL, fSize);
        fSQL  := nil;
      end;
      fSize := oSize;
      GetMem(fSQL, fSize);
      if (fSQL = nil) then raise EOutOfMemory.Create(ErrMsg);
    end;

    FillChar(fSQL^, fSize, #0);
    pEnd := fSQL;
    for Loop := 0 to (Count - 1) do begin
      tStr  := Strings[Loop] + #13#10;
      while (Length(tStr) > 0) do begin
        pEnd := StrEnd(StrPCopy(pEnd, Copy(tStr, 1, 240)));
        System.Delete(tStr, 1, 240);
      end;
    end;

    Result := fSQL;
  end;
end;

constructor TSQLList.Create;
begin
  inherited Create;

  fSQL  := nil;
  fSize := 0;
end;

destructor TSQLList.Destroy;
begin
  if (fSQL <> nil) then begin
    FreeMem(fSQL, fSize);
    fSQL  := nil;
    fSize := 0;
  end;

  inherited Destroy;
end;

(* TSQLSession -------------------------------------------------------- *)
procedure TSQLSession.ShowError(Error : Integer; Msg : String);
begin
  if (Error <> DBFAIL) then begin
    Exit;
  end;

{$IFDEF USECALLBACKS}
  raise Exception.Create(Msg + ' :' + StrPas(LastErrorMsg));
{$ELSE}
  raise Exception.Create(Msg);
{$ENDIF}
end;

procedure TSQLSession.SetActive(Data : Boolean);
begin
  if (Data = fActive) then Exit;

  if Data then begin
    DoLogin;
  end else begin
    DoLogOff;
  end;
end;

function TSQLSession.GetHost : String;
begin
  Result := StrPas(fHost);
end;

procedure TSQLSession.SetHost(Data : String);
begin
  StrPCopy(fHost, Data);
end;

function TSQLSession.GetDatabase : String;
begin
  Result := StrPas(fDatabase);
end;

procedure TSQLSession.SetDatabase(Data : String);
begin
  StrPCopy(fDatabase, Data);
  if Active then begin
    ShowError(dbuse(fHandle, fDatabase), 'dbuse failed');
  end;
end;

function TSQLSession.GetUsername : String;
begin
  Result := StrPas(fUsername);
end;

procedure TSQLSession.SetUsername(Data : String);
begin
  StrPCopy(fUsername, Data);
end;

function TSQLSession.GetPassword : String;
begin
  Result := StrPas(fPassword);
end;

procedure TSQLSession.SetPassword(Data : String);
begin
  StrPCopy(fPassword, Data);
end;

function TSQLSession.GetDevices : TStringList;
var
  cDevices         : array[0..31] of char;
begin
  Result := fDevices;

  fDevices.Clear;
  if Active then begin
    ShowError(dbcmd(fHandle, 'sp_helpdevice'), 'dbcmd failed');
    ShowError(dbsqlexec(fHandle), 'dbsqlexec failed');
    ShowError(dbresults(fHandle), 'dbresults failed');

    ShowError(dbbind(fHandle, 1, STRINGBIND, 31, @cDevices), '');
    while (dbnextrow(fHandle) <> NO_MORE_ROWS) do begin
      fDevices.Add(Trim(StrPas(@cDevices)));
    end;
  end;
end;

function TSQLSession.GetDatabases : TStringList;
var
  cDBName          : array[0..31] of char;
begin
  Result := fDatabases;

  fDatabases.Clear;
  if Active then begin
    ShowError(dbcmd(fHandle, 'sp_databases'), 'dbcmd failed');
    ShowError(dbsqlexec(fHandle), 'dbsqlexec failed');
    ShowError(dbresults(fHandle), 'dbresults failed');

    ShowError(dbbind(fHandle, 1, STRINGBIND, 31, @cDBName), '');
    while (dbnextrow(fHandle) <> NO_MORE_ROWS) do begin
      fDatabases.Add(Trim(StrPas(@cDBName)));
    end;
  end;
end;

function TSQLSession.GetTables : TStringList;
var
  cDBName          : array[0..31] of char;
  cTable           : array[0..32] of char;
begin
  Result := fTables;

  fTables.Clear;
  if Active then begin
    ShowError(dbcmd(fHandle, 'sp_tables "%"'), 'dbcmd failed');
    ShowError(dbsqlexec(fHandle), 'dbsqlexec failed');
    ShowError(dbresults(fHandle), 'dbresults failed');

    ShowError(dbbind(fHandle, 1, STRINGBIND, 31, @cDBName), '');
    ShowError(dbbind(fHandle, 3, STRINGBIND, 31, @cTable), '');

    while (dbnextrow(fHandle) <> NO_MORE_ROWS) do begin
      fTables.Add(Trim(StrPas(@cTable)));
    end;
  end;
end;

function TSQLSession.GetIndex(Tables : String) : TStringList;
var
  cIndex           : array[0..31] of Char;
  cSQL             : array[0..64] of Char;
  cTable           : array[0..31] of Char;
begin
  Result := fTables;

  StrPCopy(cTable, Tables);
  StrPCopy(cSQL, 'sp_helpindex ');
  StrCat(cSQL, cTable);

  fIndex.Clear;
  if Active then begin
    ShowError(dbcmd(fHandle, cSQL), 'dbcmd failed');
    ShowError(dbsqlexec(fHandle), 'dbsqlexec failed');
    ShowError(dbresults(fHandle), 'dbresults failed');

    ShowError(dbbind(fHandle, 1, STRINGBIND, 31, @cIndex), '');

    while (dbnextrow(fHandle) <> NO_MORE_ROWS) do begin
      fIndex.Add(Trim(StrPas(@cIndex)));
    end;
  end;
end;

function TSQLSession.GetUsers : TStringList;
var
  cUsers           : array[0..32] of char;
begin
  Result := fUserList;

  fUserList.Clear;
  if Active then begin
    ShowError(dbcmd(fHandle, 'sp_helpuser'), 'dbcmd failed');
    ShowError(dbsqlexec(fHandle), 'dbsqlexec failed');
    ShowError(dbresults(fHandle), 'dbresults failed');

    ShowError(dbbind(fHandle, 1, STRINGBIND, 31, @cUsers), '');

    while (dbnextrow(fHandle) <> NO_MORE_ROWS) do begin
      if (CompareText('dbo', Trim(StrPas(@cUsers))) <> 0) then begin
        fUserList.Add(Trim(StrPas(@cUsers)));
      end;
    end;
  end;
end;

constructor TSQLSession.Create;
begin
  inherited Create;

  FillChar(fUsername, SizeOf(fUsername), #0);
  FillChar(fPassword, SizeOf(fPassword), #0);
  FillChar(fDatabase, SizeOf(fDatabase), #0);
  fLogin     := nil;
  fHandle    := nil;
  fActive    := False;

  fDevices   := TStringList.Create;
  fDatabases := TStringList.Create;
  fTables    := TStringList.Create;
  fIndex     := TStringList.Create;
  fUserList  := TStringList.Create;

  AddLogItem('MSSQL', 437, '', 'Sessionobjekt erstellt', [nil]);
end;

destructor TSQLSession.Destroy;
begin
  AddLogItem('MSSQL', 442, '', 'Sessionobjekt geloescht', [nil]);
  Active := False;

  fUserList.Free;
  fTables.Free;
  fIndex.Free;
  fDatabases.Free;
  fDevices.Free;
  
  inherited Destroy;
end;

function TSQLSession.GetPHost : PChar;
begin
  Result := fHost;
end;

function TSQLSession.GetPDatabase : PChar;
begin
  Result := fDatabase;
end;

procedure TSQLSession.SetLoginTime(Seconds : Integer);
begin
  dbsetlogintime(Seconds);
end;

procedure TSQLSession.DoLogin;
begin
  if Active then begin
    ShowError(DBFAIL, 'Already connected to database');
  end;

  fTables.Clear;
  fDatabases.Clear;
  fHandle := nil;
  fLogin := dblogin;
  if (fLogin = nil) then begin
    ShowError(DBFAIL, 'dblogin failed');
  end;

  try
    ShowError(dbsetlname(fLogin, fUsername, DBSETUSER), 'DBSETUSER failed');
    ShowError(dbsetlname(fLogin, fPassword, DBSETPWD), 'DBSETPWD failed');
    ShowError(dbsetlname(fLogin, fDatabase, DBSETAPP), 'DBSETAPP failed');

    fHandle := dbopen(Login, fHost);
    if (fHandle = nil) then begin
      ShowError(DBFAIL, 'Connection to database failed');
    end;
    ShowError(dbuse(fHandle, fDatabase), 'dbuse failed');

    (* Lese jetzt alle Tabellen aus, die in der Datenbank vorhanden     *)
    (* sind.                                                            *)
  except
    if (fHandle <> nil) then begin
      dbclose(fHandle);
      fHandle := nil;
    end;
    dbfreelogin(fLogin);
    raise;
  end;
  fActive := True;
end;

procedure TSQLSession.DoLogoff;
begin
  if Active then begin
    if (fHandle <> nil) then begin
      dbclose(fHandle);
      fHandle := nil;
    end;
    fTables.Clear;
    dbfreelogin(fLogin);
    fActive := False;
    fLogin  := nil;
  end;
end;

function TSQLSession.GetDeviceDatafile(const Device : String) : String;
var
  cDevices         : array[0.. 31] of Char;
  cDataFile        : array[0..127] of Char;
  cDataInfo        : array[0..127] of Char;
begin
  if Active then begin
    StrPCopy(cDataFile, 'sp_helpdevice "' + Device + '"');
    ShowError(dbcmd(fHandle, cDataFile), 'dbcmd failed');
    ShowError(dbsqlexec(fHandle), 'dbsqlexec failed');
    ShowError(dbresults(fHandle), 'dbresults failed');

    ShowError(dbbind(fHandle, 1, STRINGBIND,  31, @cDevices),  '');
    ShowError(dbbind(fHandle, 2, STRINGBIND, 127, @cDatafile), '');
    ShowError(dbbind(fHandle, 3, STRINGBIND, 127, @cDataInfo), '');
    if (dbnextrow(fHandle) <> NO_MORE_ROWS) then begin
      Result := Trim(StrPas(cDatafile)) + ' (' + Trim(StrPas(cDataInfo)) + ')';
      dbCancel(fHandle);
    end else begin
    end;
  end else begin
    Result := '';
  end;
end;

procedure TSQLSession.StartTransaction;
begin
  ShowError(dbcmd(Handle, 'Begin transaction'), 'dbcmd failed');
  ShowError(dbsqlexec(Handle), 'dbsqlexec failed');
  ShowError(dbresults(Handle), 'dbresults failed');
end;

procedure TSQLSession.Commit;
begin
  ShowError(dbcmd(Handle, 'commit'), 'dbcmd failed');
  ShowError(dbsqlexec(Handle), 'dbsqlexec failed');
  ShowError(dbresults(Handle), 'dbresults failed');
end;

procedure TSQLSession.Rollback;
begin
  ShowError(dbcmd(Handle, 'rollback'), 'dbcmd failed');
  ShowError(dbsqlexec(Handle), 'dbsqlexec failed');
  ShowError(dbresults(Handle), 'dbresults failed');
end;

(* TField ------------------------------------------------------------- *)
procedure TField.SetString(Data : String);
begin
  Strings[fQuery.fRowNumber] := Data;

  if (Length(Data) > fFieldSize) then begin
    fFieldSize := Length(Data);
  end;
end;

function TField.GetString : String;
begin
  Result := Strings[fQuery.fRowNumber];
end;

function TField.GetFieldSize : Integer;
begin
  Result := fFieldSize;
end;

constructor TField.Create(Query     : TSQlQuery;
                          FieldName : String;
                          Format    : String);
begin
  inherited Create;
  fFieldName := FieldName;
  fFormat    := Format;
  fFieldSize := 0;
  fQuery     := Query;

  Sorted     := False;
  Duplicates := dupAccept;
end;

(* TSQLResult --------------------------------------------------------- *)
function TSQLResult.GetCell(Col, Row : Cardinal) : String;
var
  Field            : TField;
begin
  Result := '';
  if (Col < ColCount) and (Row < RowCount) then begin
    Field  := fFields.Items[Col + 1];
    Result := Field.Strings[Row];
  end else begin
    raise Exception.Create('Index overflow in column or row!'); 
  end;
end;

function TSQLResult.GetColCount  : Cardinal;
begin
  Result := fFields.Count - 1;
end;

function TSQLResult.GetRowCount  : Cardinal;
var
  Field            : TField;
begin
  Result := 0;
  if (ColCount > 0) then begin
    Field  := fFields.Items[0];
    Result := Field.Count;
  end;
end;

constructor TSQLResult.Create(Query : TSQlQuery);
begin
  inherited Create;

  fOwner  := Query;
  fFields := TList.Create;
end;

destructor TSQLResult.Destroy;
begin
  Clear;
  fFields.Free;

  inherited Destroy;
end;

procedure TSQLResult.Clear;
var
  Field            : TField;
  Loop             : Cardinal;
begin
  for Loop := 0 to (fFields.Count - 1) do begin
    Field := fFields.Items[Loop];
    if Assigned(Field) then begin
      Field.Free;
      fFields.Items[Loop] := nil;
    end;
  end;
  fFields.Clear;
  fFields.Pack;
end;

procedure TSQLResult.AddField(FieldName, Format : String);
var
  Field            : TField;
begin
  Field := TField.Create(fOwner, Fieldname, Format);
  if Assigned(Field) then begin
    fFields.Add(Field);
  end;
end;

function TSQLResult.FindRow(Start : Cardinal;
                            Ident : array of String) : Cardinal;
var
  LoopCol, LoopRow : Cardinal;
  Found            : Boolean; 
begin
  Result := 0;
  (* Works only fine with sorted strings in searched field list         *)
  for LoopRow := Start to (RowCount - 1) do begin
    Found := True;
    for LoopCol := 0 to (High(Ident) - 1) do begin
      if (CompareText(Cell[LoopCol, LoopRow], Ident[LoopCol]) <> 0) then
      begin
        Found := False;
        Break;
      end;
    end;

    if Found then begin
      Result := LoopRow;
      Exit;
    end;
  end; 
end;

(* TSQLQuery ---------------------------------------------------------- *)
procedure TSQLQuery.SetActive(Data : Boolean);
begin
  if (Data = Active) then Exit;

  if not Active then begin
    if fSession.Active then begin
      if (Handle = nil) then begin
        fSession.ShowError(DBFAIL, 'cursor failed');
      end;

      fState     := dsActive;
      fRowNumber := -1;
    end else begin
      fSession.ShowError(DBFAIL, 'No connection to database!');
    end;
  end else begin
    fState     := dsInactive;
    fRowNumber := -1;
  end;
end;

function TSQLQuery.GetActive : Boolean;
begin
  Result := (fState > dsInactive);
end;

function TSQLQuery.GetHandle : PDBPROCESS;
begin
  Result := Session.Handle;
end;

function TSQLQuery.GetFieldByName(Name : String) : TField;
var
  Loop             : Integer;
begin
  for Loop := 0 to (fFields.Count - 1) do begin
    if (CompareText(Name, Fields[Loop].FieldName) = 0) then begin
      Result := fFields.Items[Loop];
      Exit;
    end;
  end;
  Result := nil;
  fSession.ShowError(DBFAIL, 'Unknown field found');
end;

function TSQLQuery.GetFieldByIndex(Index : Integer) : TField;
begin
  Result := fFields.Items[Index];
end;

function TSQLQuery.GetRowNumber : Integer;
begin
  if (fRowCount > 0) then begin
    Result := fRowNumber + 1;
  end else begin
    Result := -1;
  end;
end;

function TSQLQuery.GetBOF : Boolean;
begin
  if (fRowCount = 0) then begin
    Result := True;
  end else begin
    Result := (fRowNumber < 0);
  end;
end;

function TSQLQuery.GetEOF : Boolean;
begin
  if (fRowCount = 0) then begin
    Result := True;
  end else begin
    Result := (fRowNumber >= fRowCount);
  end;
end;

constructor TSQLQuery.Create(aSession : TSQLSession);
begin
  inherited Create;

  fSession     := aSession;
  fState       := dsInactive;
  fSQL         := TSQlList.Create;
  fFields      := TList.Create;
end;

destructor TSQLQuery.Destroy;
begin
  if Active then begin
    Active := False;
  end;

  fFields.Free;
  fSQl.Free;

  inherited Destroy;
end;

procedure TSQLQuery.ExecSQL;
begin
  if Active then begin
    fSession.ShowError(dbuse(Handle, fSession.GetPDatabase), 'dbuse failed');
    fSession.ShowError(dbcmd(Handle, fSQL.SQL), 'dbcmd failed');
    try
      fSession.ShowError(dbsqlexec(Handle), 'dbsqlexec failed');
      fSession.ShowError(dbresults(Handle), 'dbresults failed');
      (* Da SQL-Abfragen, die mit ExecSQL ausgefuehrt werden, keine Er- *)
      (* gebnisse bringen sollten, kann das warten auf ein Ergebnis mit *)
      (* dbcancel angebrochen werden. Wird diese Funtkion nicht aufge-  *)
      (* rufen, so erwartet das System ein Abholen des Ergebnisses.     *)
      fSession.ShowError(dbcancel(Handle), 'dbcancel failed');
    except
      AddLogPChar(fSQL.SQL);
      raise;
    end;
  end else begin
    fSession.ShowError(DBFAIL, 'No connection to database!');
  end;
end;

procedure TSQLQuery.ExecSQLEx(aSQL : String; IgnoreException : Boolean);
begin
  try
    SQL.Clear;
    SQL.Add(aSQL);
    ExecSQL;
  except
    if not IgnoreException then raise;
  end;
end;

procedure TSQLQuery.Open(bReadOnly : Boolean; OnGetSQLData : TOnGetSQLData);
var
  fCursor          : PDBPROCESS;
  iTemp            : Integer;
  NumCnt           : Integer;
  Field            : TField;
  Fieldname        : String;
begin
  if not Active or (State > dsActive) then begin
    if not Active then begin
      fSession.ShowError(DBFAIL, 'Query not Active!');
    end else begin
      fSession.ShowError(DBFAIL, 'Query always in browse mode!');
    end;
  end;

  fRowCount  :=  0;
  fColCount  :=  0;
  fRowNumber := -1;
  fCursor    := dbopen(Session.Login, Session.GetPHost);
  if (fCursor = nil) then begin
    fSession.ShowError(DBFAIL, 'Coudn''t open cursor!');
  end;
  fState  := dsActive;

  try
    fSession.ShowError(dbcmd(fCursor, fSQL.SQL), 'dbcmd failed : ');
    fSession.ShowError(dbsqlexec(fCursor), 'dbsqlexec failed : ');
    fSession.ShowError(dbresults(fCursor), 'Keine Daten selektiert!');

    NumCnt := dbnumcols(fCursor);
    for iTemp := 1 to NumCnt do begin
      if (dbcollen(fCursor, iTemp) < 0) then begin
        Break;
      end;

      Fieldname := StrPas(dbcolname(fCursor, iTemp));
      if (Length(Fieldname) = 0) then begin
        Fieldname := IntToStr(iTemp);
      end;

      Field := TField.Create(Self, Fieldname, '');
      if (Field = nil) then begin
        fSession.ShowError(DBFAIL, 'Feld nicht erstellt');
      end else begin
        try
          fFields.Add(Field);
        except
          Field.Free;
          raise;
        end;
      end;

      fSession.ShowError(dbbind(fCursor, iTemp, STRINGBIND, 0, @Field.Buffer), '');
    end;
    fColCount := NumCnt;

    while (dbnextrow(fCursor) <> NO_MORE_ROWS) do begin
      for iTemp := 1 to fFields.Count do begin
        Field := fFields.Items[iTemp - 1];
        Field.Add(StrPas(Field.Buffer));
      end;
      Inc(fRowCount);
    end;

    dbclose(fCursor);
    fState := dsBrowse;

  except
    dbclose(fCursor);
    Close;
    raise;
  end;
end;

function TSQLQuery.First : Boolean;
begin
  Result := False;
  if (fRowCount > 0) then begin
    fRowNumber := 0;
    Result     := True;
  end;
end;

function TSQLQuery.Previous : Boolean;
begin
  Result := False;
  if (fRowCount > 0) and (fRowNumber >= 0) then begin
    Dec(fRowNumber);
    Result := True;
  end;
end;

function TSQLQuery.Next : Boolean;
begin
  Result := False;
  if (fRowNumber < fRowCount) then begin
    Inc(fRowNumber);
    Result := True;
  end;
end;

function TSQLQuery.Last : Boolean;
begin
  Result := False;
  if (fRowCount > 0) then begin
    fRowNumber := fRowCount;
    Result     := True;
  end;
end;

function TSQLQuery.MoveTo(Row : Integer) : Boolean;
begin
  Result := False;
  if (fRowCount >= Row) and (Row > 0) then begin
    fRowNumber := Row - 1;
    Result     := True;
  end;
end;

function TSQLQuery.MoveBy(Row : Integer) : Boolean;
begin
  Row    := fRowNumber + Row;
  Result := MoveTo(Row);
end;

procedure TSQLQuery.Close;
var
  Loop             : Integer;
begin
  for Loop := 0 to (fFields.Count - 1) do begin
    TField(fFields.Items[0]).Free;
    fFields.Delete(0);
  end;

  fState     := dsActive;
  fRowCount  := 0;
  fColCount  := 0;
  fRowNumber := -1;
end;

{$IFDEF USECALLBACKS}
var
  OldErrHandler    : TDBErrHandler;
  OldMsgHandler    : TDBMsgHandler;
{$ENDIF}

procedure StartDLL;
begin
  AddLogItem('MSSQL', 837, '', 'Initialization Unit MSSQL aufgerufen', [nil]);
  dbInit;

{$IFDEF USECALLBACKS}
  OldErrHandler := dberrhandle(ErrHandler);
  OldMsgHandler := dbmsghandle(MsgHandler);
{$ENDIF}

  DefaultSession := TSQLSession.Create;
end;

procedure EndDLL;
begin
  AddLogItem('MSSQL', 850, '', 'Finalization Unit MSSQL aufgerufen', [nil]);
  DefaultSession.Free;

  dbExit;

{$IFDEF USECALLBACKS}
  dbmsghandle(OldMsgHandler);
  dberrhandle(OldErrHandler);
{$ENDIF}
  AddLogItem('MSSQL', 859, '', 'Finalization Unit MSSQL beendet', [nil]);
end;

initialization

  StartDLL;

finalization

  EndDll;

end.
