unit mDataBas;

{$I mODBC.INC}

interface

uses
  Windows, SysUtils, Classes, Forms, Db,
  mSession, ODBCsql;

type
  TmDataBase = class;
  TmDriverCompletion = (sdPrompt,sdComplete,sdCompleteReq,sdNoPrompt);
  TmOdbcCursors = (ocUSE_IF_NEEDED, ocUSE_ODBC, ocUSE_DRIVER);
  TmOdbcCreateDSN = (cdADD_DSN, cdCONFIG_DSN, cdREMOVE_DSN, cdADD_SYSDSN,
                     cdCONFIG_SYSDSN, cdREMOVE_SYSDSN);
  TmLoginEvent = procedure(Database: TmDatabase) of object;
  TmIsolationLevels = (TxnDefault, TxnDirtyRead, TxnReadCommitted, TxnRepeatableRead, TxnSerializable);

  TmDataBase = class(TComponent)
  private
    { Private declarations }
//    FDataBaseName: String; stored in Fparams
    FDsnParams: String;
    FDataSetList:  TList;
    FDriverCompletion: TmDriverCompletion;
    fhdbc: SQLHDBC;
    FOdbcCursors: TmOdbcCursors;
    FOnLogin: TmLoginEvent;
    FParams: TStrings;
    FSession: TmSession;
    FTransIsolation: TmIsolationLevels;
    function GetConnected:Boolean;
    procedure SetParams(Value: TStrings);
    procedure CheckSQLResult( sqlres:SQLRETURN);
    procedure SetSession( s:TmSession);
    function GetDatabaseName:String;
    procedure SetDatabaseName(Name:String);
  protected
    { Protected declarations }
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
  public
    { Public declarations }
    property Connected: Boolean read GetConnected{ write SetConnected};
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function BDE2SqlType( aFieldType: TFieldType): SQLSMALLINT;
    procedure Connect;
    procedure DisConnect;
    function GetHENV:SQLHANDLE;
    procedure ODBCDriverInfo(
        InfoType:SQLUSMALLINT;
        InfoValuePtr:SQLPOINTER; BufferLength:SQLSMALLINT;
        StringLengthPtr:PSQLSMALLINT);
    procedure ConfigureDSN(
        Action: TmOdbcCreateDSN;
        DsnName: string;
        Driver: string;
        Parameters: TStringList);
    function DriverOfDataSource( DSN:String):String;
    property hdbc: SQLHDBC read fhdbc;
    procedure GetColumnNames(const ATable: string; AList:TStrings);
    procedure GetDatSourceNames( tlist: TStrings);
    procedure GetDriverNames( tlist: TStrings);
    procedure GetTableNames( AList: TStrings);
    procedure GetPrimaryKeys(const ATable: string; AList:TStrings);
    procedure GetProcNames( AList: TStrings);
    procedure GetDsnParams( AList: TStrings);
    procedure StartTransaction;
    procedure Commit;
    procedure Rollback;
    procedure IncludeDataSet( Adataset: TDataSet);
    procedure ExcludeDataSet( Adataset: TDataSet);
  published
    { Published declarations }
    property DataBaseName: String read GetDataBaseName write SetDataBaseName;
    property DriverCompletion: TmDriverCompletion read    FDriverCompletion
                                                  write   FDriverCompletion
                                                  default sdCompleteReq;
    property OdbcCursors: TmOdbcCursors
               read FOdbcCursors write FOdbcCursors default ocUSE_DRIVER;
    property OnConnect: TmLoginEvent read FOnLogin write FOnLogin;
    property Params: TStrings read FParams write SetParams;
    property Session: TmSession read FSession write SetSession;
    property TransIsolation: TmIsolationLevels read FTransIsolation
                                               write FTransIsolation
                                               default TxnDefault;
  end;

implementation

uses mQuery, mconst, mExcept;

const SQL_NAME_LEN = 128;

constructor TmDatabase.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FDataSetList      := TList.Create;
  FParams           := TStringList.Create;
  FDriverCompletion := sdCompleteReq;
  FOdbcCursors      := ocUSE_DRIVER;
  FSession          := nil;
  FTransIsolation   := TxnDefault;
  fhdbc := 0;
end;

destructor TmDataBase.Destroy;
begin
  if Connected then
    DisConnect;
  FDataSetList.free;
  FParams.Free;
  inherited Destroy;
end;

procedure TmDataBase.CheckSQLResult( sqlres:SQLRETURN);
begin
  if sqlres <> SQL_SUCCESS then
    raise ESQLerror.CreateDiag( SQL_HANDLE_DBC, fhdbc, sqlres);
end;

procedure TmDataBase.StartTransaction;
var
   us: SQLUSMALLINT;
begin
  ODBCDriverInfo( SQL_TXN_CAPABLE, SQLPOINTER(@us), Sizeof( SQLUSMALLINT), nil);

  if us = SQL_TC_NONE then
    raise Exception.Create('Driver not support transactions');

  CheckSQLResult( SQLSetConnectAttr( hdbc, SQL_ATTR_AUTOCOMMIT,
                                     SQLPOINTER( SQL_AUTOCOMMIT_OFF),
                                     SQL_IS_UINTEGER));
end;

procedure TmDataBase.Commit;
begin
  CheckSQLResult( SQLEndTran( SQL_HANDLE_DBC, hdbc, SQL_COMMIT));
end;

procedure TmDataBase.Rollback;
begin
  CheckSQLResult( SQLEndTran( SQL_HANDLE_DBC, hdbc, SQL_ROLLBACK));
end;

procedure TmDataBase.Connect;
var
  i:       SQLRETURN;
  SServer: array [0..1025] of Char;
  cbout:   SQLSMALLINT;
  ConnectionString: String;
  dc:      SQLUSMALLINT;
begin
  if Connected then
    exit;

  if SQLAllocHandle(SQL_HANDLE_DBC, GetHENV, fhdbc)<>SQL_SUCCESS then
    raise Exception.Create( SmAllocateHDBCError);

  try
    case FOdbcCursors of
      ocUSE_IF_NEEDED: dc:=SQL_CUR_USE_IF_NEEDED;
      ocUSE_ODBC:      dc:=SQL_CUR_USE_ODBC;
      ocUSE_DRIVER:    dc:=SQL_CUR_USE_DRIVER;
      else             dc:=SQL_CUR_USE_DRIVER;
    end;

    CheckSQLResult( SQLSetConnectAttr( hdbc, SQL_ATTR_ODBC_CURSORS,
                                       SQLPOINTER( dc),
                                       SQL_IS_UINTEGER));
    if Assigned(FOnLogin) then
      FOnLogin( Self);
{
    if DataBaseName <> ''
      then ConnectionString := 'DSN=' + DataBaseName
      else }ConnectionString := '';

    for i := 0 to FParams.Count-1 do
    begin
      if Length(ConnectionString) > 0 then
        ConnectionString := ConnectionString + ';';
      ConnectionString := ConnectionString+FParams[i];
    end;

    case DriverCompletion of
      sdPrompt:      dc := SQL_DRIVER_PROMPT;
      sdComplete:    dc := SQL_DRIVER_COMPLETE;
      sdCompleteReq: dc := SQL_DRIVER_COMPLETE_REQUIRED;
      sdNoPrompt:    dc := SQL_DRIVER_NOPROMPT;
      else           dc := SQL_DRIVER_COMPLETE_REQUIRED;
    end;

    FDsnParams:='';
    try
      CheckSQLResult( SQLDriverConnect( fhdbc, Application.handle,
                                        PChar(ConnectionString), SQL_NTS,
                                        SServer, 1024, cbout, dc));
    except on E: ESQLerror do
      if E.NativeError = SQL_NO_DATA then
      begin
         E.Message:=SmDatabaseNotOpened;
         raise;
      end else
      if (E.SqlState <> '01000') then
        raise;
    end;
    FDsnParams := StrPas(SServer);

    case FTransIsolation of
      TxnDirtyRead:     dc := SQL_TXN_READ_UNCOMMITTED;
      TxnReadCommitted: dc := SQL_TXN_READ_COMMITTED;
      TxnRepeatableRead:dc := SQL_TXN_REPEATABLE_READ;
      TxnSerializable:  dc := SQL_TXN_SERIALIZABLE;
      else              dc := 0;
    end;

    if dc <> 0 then
      CheckSQLResult( SQLSetConnectAttr( hdbc, SQL_ATTR_TXN_ISOLATION,
                                         SQLPOINTER( dc),
                                         SQL_IS_INTEGER));
  except
    SQLFreeHandle( SQL_HANDLE_DBC, fhdbc);
    fhdbc := 0;
    raise;
  end;
end;

procedure TmDataBase.DisConnect;
var
  i: integer;
begin
  if not Connected then
    exit;

  for i := 0 to FDataSetList.Count-1 do
    with TmCustomQuery(FDataSetList.Items[i]) do
       FreeStmt;

  SQLDisConnect(fhdbc);
  SQLFreeHandle(SQL_HANDLE_DBC, fhdbc);
  fhdbc := 0;
end;

function TmDataBase.GetConnected:Boolean;
begin
  Result:=(fhdbc<>0);
end;

procedure TmDataBase.Notification(AComponent: TComponent; Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);

  if (Operation = opInsert) and (AComponent is TmCustomQuery) then
  begin
    with TmCustomQuery(AComponent) do
      if DataBase = nil then
        DataBase := Self;
  end;
end;

procedure TmDatabase.SetParams(Value: TStrings);
begin
  FParams.Assign(Value);
end;

procedure TmDatabase.ODBCDriverInfo( InfoType:SQLUSMALLINT;
                                     InfoValuePtr:SQLPOINTER;
                                     BufferLength:SQLSMALLINT;
                                     StringLengthPtr:PSQLSMALLINT);
var
  sqlres: SQLRETURN;
begin
  sqlres := SQLGetInfo( hdbc, InfoType, InfoValuePtr, BufferLength, StringLengthPtr);
  case sqlres of
    SQL_SUCCESS:;
    SQL_SUCCESS_WITH_INFO: raise ESQLerror.CreateDiag( SQL_HANDLE_DBC, hdbc, sqlres);
    SQL_STILL_EXECUTING:   raise ESQLerror.Create('SQL_STILL_EXECUTING');
    SQL_ERROR:             raise ESQLerror.CreateDiag( SQL_HANDLE_DBC, hdbc, sqlres);
    SQL_INVALID_HANDLE:    raise ESQLerror.Create( 'SQL_INVALID_HANDLE');
    else                   raise ESQLerror.Create( 'unknown SQL result');
  end;
end;

function TmDatabase.DriverOfDataSource( DSN:String):String;
var
  ServerName: array [0..SQL_MAX_DSN_LENGTH] of Char;
  DriverName: array [0..255] of Char;
begin
  Result := '';
  if SQLDataSources( GetHENV, SQL_FETCH_FIRST,
                     ServerName, SQL_MAX_DSN_LENGTH, nil,
                     DriverName, 255, nil) = SQL_SUCCESS then
  repeat
    if StrPas(ServerName) = DSN then
    begin
      Result := StrPas( DriverName);
      exit;
    end;
  until SQLDataSources( GetHENV, SQL_FETCH_NEXT,
                        ServerName, SQL_MAX_DSN_LENGTH, nil,
                        DriverName, 255, nil) <> SQL_SUCCESS;
end;


procedure TmDatabase.ConfigureDSN( Action: TmOdbcCreateDSN;
                                   DsnName: string;
                                   Driver: string;
                                   Parameters: TStringList);
var
  sParameters: string;
  iAction: integer;
  i: integer;
  fErrorCode: SQLINTEGER;
  lpszErrorMsg: array [0..80] of Char;
  cbErrorMsg: SQLUSMALLINT;
begin
  case Action of
    cdADD_DSN        : iAction := ODBC_ADD_DSN;
    cdCONFIG_DSN     : iAction := ODBC_CONFIG_DSN;
    cdREMOVE_DSN     : iAction := ODBC_REMOVE_DSN;
    cdADD_SYSDSN     : iAction := ODBC_ADD_SYS_DSN;
    cdCONFIG_SYSDSN  : iAction := ODBC_CONFIG_SYS_DSN;
    cdREMOVE_SYSDSN  : iAction := ODBC_REMOVE_SYS_DSN;
    else               iAction := 0;
  end;

  sParameters := '';
  sParameters := sParameters + 'DSN=' + DsnName + ';';

  if Assigned(Parameters) then
    for i:=0 to Parameters.Count-1 do
      sParameters := sParameters + Parameters[i]+';';

  if sParameters <> '' then
    Delete(sParameters,Length(sParameters),1);

  if (Action in [cdCONFIG_DSN,cdCONFIG_SYSDSN,
                 cdREMOVE_DSN,cdCONFIG_SYSDSN]) and (Driver='') then
    Driver := DriverOfDataSource(DsnName);

  if iAction <> 0 then
    if SQLConfigDataSource(Application.Handle{SQL_NULL_HANDLE}, iAction, Driver, sParameters)<>1 then
    begin
      SQLInstallerError(1, @fErrorCode, lpszErrorMsg, 80, @cbErrorMsg);
      raise ESQLerror.Create( 'SQLConfigDataSource:' + #13 + StrPas( lpszErrorMsg));
    end;
end;

procedure TmDatabase.GetDatSourceNames( tlist: TStrings);
var
  ServerName: array [0..SQL_MAX_DSN_LENGTH] of Char;
  sl:         SQLSMALLINT;
  sqlResult: integer;
begin
  tlist.Clear;
  if SQLDataSources( GetHENV, SQL_FETCH_FIRST,
                     ServerName, SQL_MAX_DSN_LENGTH, @sl,
                     nil, 0, nil) = SQL_SUCCESS then
  repeat
    tlist.add( StrPas(ServerName));
    sqlresult := SQLDataSources( GetHENV, SQL_FETCH_NEXT,
                                 ServerName, SQL_MAX_DSN_LENGTH, @sl,
                                 nil, 0, nil)
  until (sqlResult <> SQL_SUCCESS) and (sqlResult <> SQL_SUCCESS_WITH_INFO)
end;

procedure TmDatabase.GetDriverNames( tlist: TStrings);
var
  DriverName: array [0..SQL_MAX_OPTION_STRING_LENGTH] of Char;
  sl:         SQLSMALLINT;
  sqlResult: integer;
begin
  tlist.Clear;

  sqlResult := SQLDrivers( GetHENV, SQL_FETCH_FIRST, DriverName,
                           SQL_MAX_OPTION_STRING_LENGTH, @sl, nil, 0, nil);

  if (sqlResult = SQL_SUCCESS) or (sqlResult = SQL_SUCCESS_WITH_INFO) then
  repeat
    tlist.add( StrPas(DriverName));
    sqlResult := SQLDrivers( GetHENV, SQL_FETCH_NEXT, DriverName,
                             SQL_MAX_OPTION_STRING_LENGTH, @sl, nil, 0, nil);

  until (sqlResult <> SQL_SUCCESS) and
        (sqlResult <> SQL_SUCCESS_WITH_INFO)
end;

procedure TmDatabase.GetTableNames( AList: TStrings);
var
  h : SQLHANDLE;
  ATableName : array[0..SQL_NAME_LEN + 1] of char;
  ATableType : array[0..SQL_NAME_LEN + 1] of char;
  l : DWORD;
  Res : SQLRETURN;
begin
  AList.Clear;
  Connect;

  if SQLAllocHandle( SQL_HANDLE_STMT, hdbc,h ) = SQL_SUCCESS then
  try
    Res := SQLTables( h, nil, 0, nil, 0, nil, 0, nil, 0);
    if Res = SQL_SUCCESS then
    begin
      SQLBindCol( h, 3, SQL_CHAR, @ATableName[0], SQL_NAME_LEN, @l);
      SQLBindCol( h, 4, SQL_CHAR, @ATableType[0], SQL_NAME_LEN, @l);

      Res := SQLFetch( h );
      while ( Res = SQL_SUCCESS ) do
      begin
        if StrPas( ATableType)<>'SYSTEM TABLE' then
          AList.Add( StrPas( ATableName ));
        Res := SQLFetch( h);
      end;
    end;
  finally
    SQLFreeHandle( SQL_HANDLE_STMT,h );
  end;
end;

procedure TmDatabase.GetProcNames( AList: TStrings);
var
  h: SQLHANDLE;
  AProcName: array[0..SQL_NAME_LEN + 1] of char;
  l: DWORD;
  Res: SQLRETURN;
begin
  AList.Clear;
  Connect;

  if SQLAllocHandle( SQL_HANDLE_STMT, hdbc,h ) = SQL_SUCCESS then
  try
    Res := SQLProcedures( h, nil, 0, nil, 0, nil, 0);
    if Res = SQL_SUCCESS then
    begin
      SQLBindCol( h, 3, SQL_CHAR, @AProcName[0], SQL_NAME_LEN, @l);

      Res := SQLFetch( h );
      while ( Res = SQL_SUCCESS ) do
      begin
        AList.Add( StrPas( AProcName ) );
        Res := SQLFetch( h );
      end;
    end;
  finally
    SQLFreeHandle( SQL_HANDLE_STMT,h );
  end;
end;

procedure TmDatabase.IncludeDataSet( Adataset: Tdataset);
begin
  if FdataSetList.IndexOf(Adataset)<0 then
    FdataSetList.add(adataset);
end;

procedure TmDatabase.ExcludeDataSet( Adataset: Tdataset);
var
  i: integer;
begin
  i := FdataSetList.IndexOf(Adataset);
  if i >= 0 then
    FdataSetList.delete(i);
end;

procedure TmDataBase.GetColumnNames(const ATable: string; AList: TStrings);
var
  h: SQLHANDLE;
  ATableName: array[0..SQL_NAME_LEN + 1] of char;
  l: DWORD;
  Res: SQLRETURN;
begin
  AList.BeginUpdate;
  AList.Clear;
  Connect;
  if SQLAllocHandle( SQL_HANDLE_STMT, hdbc,h ) = SQL_SUCCESS then
  begin
    try
      Res := SQLColumns( h, nil, 0, nil, 0, pchar(ATable), length(Atable), nil, 0);
      if Res = SQL_SUCCESS then
      begin
        SQLBindCol( h, 4, SQL_CHAR, @ATableName[0], SQL_NAME_LEN, @l);
        Res := SQLFetch( h );
        while ( Res = SQL_SUCCESS ) do
        begin
          AList.Add( StrPas( ATableName ) );
          Res := SQLFetch( h );
        end;
      end;
    finally
      SQLFreeHandle( SQL_HANDLE_STMT,h );
      AList.EndUpdate;
    end;
  end;
end;

procedure TmDataBase.GetPrimaryKeys(const ATable: string; AList: TStrings);
var
  h: SQLHANDLE;
  ATableName: array[0..SQL_NAME_LEN + 1] of char;
  l: DWORD;
  Res: SQLRETURN;
begin
  AList.BeginUpdate;
  AList.Clear;
  Connect;
  if SQLAllocHandle( SQL_HANDLE_STMT, hdbc, h ) = SQL_SUCCESS then
  begin
    try
       Res := SQLPrimaryKeys( h, nil, 0, nil, 0, pchar(ATable), length(Atable));
       if Res = SQL_SUCCESS then
       begin
         SQLBindCol( h, 4, SQL_CHAR, @ATableName[0], SQL_NAME_LEN, @l);
         Res := SQLFetch( h );
         while ( Res = SQL_SUCCESS ) do
         begin
           AList.Add( StrPas( ATableName ) );
           Res := SQLFetch( h );
         end;
       end;
     finally
       SQLFreeHandle( SQL_HANDLE_STMT,h );
       AList.EndUpdate;
    end;
  end;
end;

procedure TmDataBase.SetSession( s:TmSession);
begin
  DisConnect;
  FSession := s;
end;

function TmDatabase.GetHENV:SQLHANDLE;
begin
  if FSession <> nil
    then Result := FSession.HENV
    else Result := GlobalSession.HENV;
end;

procedure TmDatabase.GetDsnParams( AList: TStrings);
begin
    Alist.Text := FDsnParams;
end;

function TmDataBase.GetDatabaseName: String;
begin
  Result := FParams.Values['DSN'];
end;

procedure TmDataBase.SetDatabaseName(Name: String);
begin
  FParams.Values['DSN'] := Name;
end;

function TmDataBase.BDE2SqlType( aFieldType: TFieldType): SQLSMALLINT;
begin // datatypes depend from driver
  case aFieldType of
    ftString:   result := SQL_CHAR;
    ftWord:     result := SQL_SMALLINT;
    ftSmallint: result := SQL_SMALLINT;
    ftInteger,
    ftAutoInc:  result := SQL_INTEGER;
//    ftTime:
    ftDate:     result := SQL_DATE; //SQL_TYPE_DATE;
    ftDateTime: result := SQL_TIMESTAMP; //SQL_TYPE_TIMESTAMP;
    ftBCD,
    ftCurrency,
    ftFloat:    result := SQL_DOUBLE;
    ftBoolean:  result := SQL_SMALLINT;
    ftMemo:     result := SQL_LONGVARCHAR;
    ftBlob:     result := SQL_LONGVARBINARY;
    else        result := 0;
  end;
end;

end.
