{* LsDBUtil *}
{****************************************************************************
 Program     :  LsDBUtil is a database utility, which consists of
									LsDBUtil.dpr	- Project File
									DBUtilMain.* 	- Main Unit
									About.*				- About and Help

 Description :  If you work with Paradox or dBase tables, you may need
								to regenerate table's indexes in order to bring any
								nonmaintained indexes up to date, or to pack tables for
								better performance.  But, for weird reasons, Re-index and
								Pack functions are absent from Delphi's Database Desktop.

								LsDBUtil demonstrates how to pack and reindex Paradox/dBase
								tables by calling BDE functions directly.  BDE functions
								utilized include:

									DbiGetCursorProps() -	get commonly used cursor property
																				(CURProps) structure, eg. using this
																				function to retrieve szTableType in
																				this program.
									DbiDoRestructure()	- Changes the property of a table.
																				If option bPack is True, this function
																				can be used to pack Paradox Tables.
									DbiPackTable()   		- optimizes table space by rebuilding
																				the table (For dBase only).
									DbiRegenIndexes() 	- regenerates indexes to ensure they are
																				up to date, for both Paradox and dBase
																				tables.
									DbiGetErrorString()	- retures the error message (in plain
																				language) associated with a given
																				error code.
									DbiGetDirectory() 	-	retrieves the current directory of
																				the database.
									DbiCopyTable()  		- duplicates the source table with its
																				index file, to a destination table.

								LsDBUtil can also duplicate source tables together with
								index-files of a selected database, to a destination
								directory.

 Version     :  1.00
 Compiler    :  Delphi versions 3 or 4.
								Distributed copy is compiled by Delphi 4.
 Author      :  Leo D. Shih  e-mail: <ldshih@ecn.ab.ca>
 Copyright   :  (C)1998 Leo D. Shih, all rights reserved.

 DISCLAIMER  :  LsDBUtil is distributed as freeware, without warranties
								of any kind, expressed or implied.  Use of this software
								is at your own risk.  However, comments, suggestions or
								bug-reports are welcome.
****************************************************************************}


unit DBUtilMain;

interface

uses
	Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms,
	Dialogs, StdCtrls, Buttons, ComCtrls, ExtCtrls, BDE, Db, DBTables,
	DbiTypes, DbiProcs, DbiErrs, ShlObj, ShellAPI, ActiveX;

type

	TDirDialog = class  // BrowseForFolder Dialog
	public
		hHandle: THandle;
		sTitle: string;
		sDirectory: string;
		function Execute: Boolean;
	end;

	TDBUtilForm = class(TForm)
		Database1: TDatabase;
		Table1: TTable;
		Panel1: TPanel;
		BtnIndex: TSpeedButton;
		BtnPack: TSpeedButton;
		BtnStart: TSpeedButton;
		BtnClose: TSpeedButton;
		BtnAbout: TSpeedButton;
		StatusBar1: TStatusBar;
		Panel2: TPanel;
		Label1: TLabel;
		Label2: TLabel;
		ListBox1: TListBox;
		ListBox2: TListBox;
		Edit1: TEdit;
		Label3: TLabel;
		Label4: TLabel;
		EdtDesDir: TEdit;
		BtnBackup: TSpeedButton;
		BtnDesDir: TSpeedButton;
		Label5: TLabel;
		Panel3: TPanel;
    ProgressBar1: TProgressBar;
    Label6: TLabel;
		procedure DisplayHint(Sender: TObject);
		procedure BtnCloseClick(Sender: TObject);
		procedure BtnStartClick(Sender: TObject);
		procedure BtnPackClick(Sender: TObject);
		procedure BtnIndexClick(Sender: TObject);
		procedure ListBox1DblClick(Sender: TObject);
		procedure FormActivate(Sender: TObject);
		procedure PackTables(IsPackTbl: Boolean);
		procedure BtnAboutClick(Sender: TObject);
		procedure BtnBackupClick(Sender: TObject);
		procedure BtnDesDirClick(Sender: TObject);

	private
 { Private declarations }
		List1: TStrings;
		IndexTbl,
		PackTbl,
		BackUpTbl: Boolean;
		procedure BackUpFiles;
		procedure BtnStatus;
	public
 { Public declarations }
		DirDialog: TDirDialog;
	end;


var
	DBUtilForm: TDBUtilForm;


implementation

{$R *.DFM}

uses About;



// ==============================================================
// Using:
//  - DbiPackTable() to pack dBASE Tables;
//  - DbiDoRestructure() with bPack option to pack Paradox Tables;
//  - DbiRegenIndexes() to regenerate Indexes of both dBASE and
//    Paradox Tables.
// =============================================================

procedure TDBUtilForm.PackTables(IsPackTbl: Boolean);
var
	i						: Integer;
	TType				: array[0..40] of Char;
	TblDesc			: CRTblDesc;
	db					: TDatabase;
//	dbiStatus		: DbiMsg;
	rslt				: DbiResult;
	errorMsg		: array[0..255] of Char;
	errorMsgPtr	: pChar;
	isConnected,
	isActive,
	isExclusive,
	goAhead,
	isBatch			: Boolean;
	msgStr			: string;

	// Get Table Type (szTableType) from Cursor Properties (CURPprops)
	function GetTableType(Handle: hDbiCur): pChar;
	var
		FCurProp: CurProps;
	begin
		Check(DbiGetCursorProps(Handle, FCurProp));
		Result := FCurProp.szTableType;
	end;

	// Since DbiPackTable() is not valid for Paradox Table, therefor
	// Using DbiDoRestructure() with the bPack option instead.
	procedure PackParadoxTable(tbl: TTable);
	begin
		with tbl do
		begin
			db := Database;
			if Active then Active := False;
			FillChar(TblDesc, sizeof(CRTblDesc), #0);
			// Copy TableName & TableType from CRTblDesc
			StrPCopy(TblDesc.szTblName, TableName);
			StrCopy(TblDesc.szTblType, szParadox);
			// Set bPack to True, specifies packing for restructure.
			TblDesc.bPack := True;
			rslt := DbiDoRestructure(db.Handle, 1, @TblDesc, nil, nil, nil, False);
		end;
	end;

begin
	isBatch := False;
	goAhead := False;
	msgStr := '';
	if PackTbl then
		msgStr := 'Do you want to pack table'
	else if IndexTbl then
		msgStr := 'Do you want to regenerate indexes for table';
	isConnected := Database1.Connected;
	if not Database1.Connected then
		Database1.Connected := True;
	if List1.Count > 1 then
	begin
		case MessageDlg('Tables in Database "' +
									UpperCase(Listbox1.Items[ListBox1.ItemIndex]) +
									' " can be processed' + #13 +
									'in a batch or individually' + #13 + #13 +
									'Do you want to process all tables in a batch?',
									mtConfirmation, [mbYes, mbNo], 0) of
			mrYes:	begin
								isBatch := True;
								goAhead := True;
							end;
			mrNo:		begin
								isBatch := False;
								goAhead := False;
							end;
		end;
	end;

	for i := 0 to List1.Count - 1 do
	begin
		Edit1.Text := List1[i];
		ProgressBar1.Position := Trunc(100 * i / (List1.Count));
		Application.ProcessMessages;
		Table1.TableName := List1[i];
		isActive := Table1.Active;
		isExclusive := Table1.Exclusive;
		if isBatch = False then
		begin
			if MessageDlg(msgStr + #13 + #13 + '"' + UpperCase(List1[i]) + '" ?',
										mtConfirmation, [mbYes, mbNo], 0) = mrYes then
				goAhead := True
			else goAhead := False;
		end;

		if goAhead then
		begin
			with Table1 do begin
				DisableControls;
				try
					// ensure that Table's Exclusive property is set to True
					// before the Table is opened
					Active := False;
					Exclusive := True;
					Active := True;
					// Packing (restructuring) table
					if (IsPackTbl = True) then
					begin
						strcopy(TType, GetTableType(Table1.Handle));
						// if it's a Paradox Table
						if strcomp(TType, szParadox) = 0 then
							PackParadoxTable(Table1)
						// else if it's a dBase table then using DbiPackTable()
						// to pack tables
						else if strcomp(TType, szDBase) = 0 then
							rslt := DbiPackTable(DBHandle, Handle, nil, nil, True)
						else
							MessageDlg('Only dBASE and Paradox tables ' + #13 +
												'can be packed or re-indexed',
												mtInformation, [mbOK], 0);
					end
					// Re-generating indexes for both Paradox and dBase tables
					else if (IsPackTbl = False) then
						rslt := DbiRegenIndexes(Handle);
					// Error handling procedure
					if (rslt <> DBIERR_NONE) then
					begin
						errorMsgPtr := @errorMsg;
						DbiGetErrorString(rslt, errorMsgPtr);
						MessageDlg(errorMsg, mtError, [mbOK], 0);
					end;
				finally
					Active := False;
					Exclusive := isExclusive;
					Active := isActive;
				end;  // try ..
				EnableControls;
			end;  // with table1 ..
		end;  // if goAhead
	end;  // For ..
	Database1.Connected := isConnected;
	List1.Free;
	ProgressBar1.Position := 0;
	if IsPackTbl = True then
		Edit1.Text := 'All Tables packed'
	else if IsPackTbl = False then
		Edit1.Text := 'Re-generating indexes completed';
	PackTbl := False;
	IndexTbl := False;
	BtnStatus;
end;

// =============================================================
// Backup (copy, w/o compression) Tables with Index Files of the
// selected databse to a user specified destination directory by
// using DbiCopyTable() function.   Alternatively. ShellAPI
// function SHFileOperation() can be utilized to perform this
// procedure.  See commented code fragment below.
// =============================================================

procedure TDBUtilForm.BackUpFiles;
var
	i						: integer;
	isActive,
	isExclusive,
	isConnected,
	bOverWrite  : Boolean;
	sSrc, sDes  :	AnsiString;
	pSrc, pDes,
	pDriver,
	errorMsg		: array[0..255] of Char;
	errorMsgPtr	: pChar;
	rslt				: DbiResult;
	tblName     : string;

	function GetDriverType(Handle: hDbiCur): pChar;
	var
		FCurProp: CurProps;
	begin
		Check(DbiGetCursorProps(Handle, FCurProp));
		Result := FCurProp.szTableType;
	end;

begin
	bOverWrite := False;
	isConnected := Database1.Connected;
	if not Database1.Connected then
		Database1.Connected := True;
	DbiGetDirectory(Database1.Handle, True, pSrc);
	EdtDesDir.Enabled := True;
	EdtDesDir.SetFocus;
	sDes := EdtDesDir.Text;
	if sDes[length(sDes)] <> '\' then
		sDes := sDes + '\';

	for i := 0 to List1.Count - 1 do
	begin
		isActive := Table1.Active;
		isExclusive := Table1.Exclusive;
		ProgressBar1.Position := Trunc(100 * i / (List1.Count));
		Application.ProcessMessages;
		Edit1.Text := List1[i];
		Table1.TableName := List1[i];
		// Strip the file extension of TableNames.
		tblName := Copy(List1[i], 1, Pos('.', List1[i]) - 1);
		if FileExists(sDes + List1[i]) then
			if MessageDlg('File "'+ UpperCase(List1[i]) +
										'" exists in Destination Directory'
										+ #13 + #13 + 'OverWrite it?', mtConfirmation,
										[mbYes, mbNo], 0) = mrYes then
				bOverWrite := True
			else  bOverWrite := False;

		with Table1 do begin
			DisableControls;
			try
				TableType := ttDefault;
				Active 		:= False;
				Exclusive := True;
				Active 		:= True;

				StrPCopy(pDriver, GetDriverType(Table1.Handle));
				StrPCopy(pSrc, (sSrc + tblName));
				StrPCopy(pDes, (sDes + tblName));

				// when pSrc specifies a table name without a file extension,
				// DbiCopyTable() will copy both table and its index files.

				rslt := DbiCopyTable(Database1.Handle, bOverWrite, pSrc, pDriver, pDes);

				if (rslt <> DBIERR_NONE) then
				begin
					errorMsgPtr := @errorMsg;
					DbiGetErrorString(rslt, errorMsgPtr);
					// if errorMsg <> 'File already exists.'
					if rslt <> 13057 then
						MessageDlg(errorMsg, mtError, [mbOK], 0);
				end;
			finally
				Active := False;
				Exclusive := isExclusive;
				Active := isActive;
			end; 		// try ..
			EnableControls;
		end; 		// with table ..
	end;     // for ..
	Database1.Connected := isConnected;
	List1.Free;
	Edit1.Text := 'All Tables copied';
	ProgressBar1.Position := 0;
	BackupTbl := False;
	BtnStatus;
end;


// ==============================================================
// The following code fragment shows how to use ShellAPI function
// SHFileOperation() to duplicate source tables of the selected
// Database to a destination directory.
// ==============================================================

{*
procedure TDBUtilForm.BackUpFiles;
var
	i						: integer;
	isConnected	: Boolean;
	fo					: TShFileOpStruct;
	sFrom,
	sTo,
	sDes				: AnsiString;
	sSrc				: array[0..254] of char;
begin
	FillChar(sSrc, SizeOf(sSrc), #0);
	Database1.DatabaseName := ListBox1.Items[ListBox1.ItemIndex];
	isConnected := Database1.Connected;
	if not Database1.connected then
		Database1.Connected := True;
	DbiGetDirectory(Database1.Handle, True, sSrc);
	EdtDesDir.Enabled := True;
	EdtDesDir.SetFocus;
	sDes := EdtDesDir.Text;
	if sDes[Length(sDes)] <> '\' then
		sDes := sDes + '\';
	fo.Wnd := GetActiveWindow;
	fo.wFunc := FO_COPY;
	try
		for i := 0 to List1.Count - 1 do
		begin
			sFrom := sSrc + List1[i] + #0;
			sTo := sDes + List1[i] + #0;
			Edit1.Text := List1[i];
			ProgressBar1.Position := Trunc(100 * i / List1.Count);
			Application.ProcessMessages;
			fo.pFrom := pChar(sFrom);
			fo.pTo := pChar(sTo);
			fo.fFlags := FOF_ALLOWUNDO;   //	FOF_NOCONFIRMATION;

			if ShFileOperation(fo) <> 0 then
				ShowMessage('Copy File failed');
			sFrom := '';
			sTo := '';
		end;
	finally
		Database1.Connected := isConnected;
		List1.Free;
		Edit1.Text := 'All Tables copied';
		ProgressBar1.Position := 0;
		BackupTbl := False;
		BtnStatus;
	end;
end;
*}

// =============================================================
//  TDirDialog encapsulates the 'Browse for Folder' dialogbox.
//  that is provided by Windows ShellAPI
// =============================================================

function TDirDialog.Execute: Boolean;
var
	iList,
	Root			: PItemIDList;
	bi				: TBrowseInfo;
	DispName	: string;
	malloc		: IMalloc;

	// Callback function used with the SHBrowseForFolder
	function FNBFFCallBack(Wnd: HWND; uMsg: UINT; lParam,
										lpData: LPARAM): Integer stdcall;
	var
		dir: string;
	begin
		with TDirDialog(lpData) do
		case uMsg of
			BFFM_INITIALIZED:
				try
					hHandle := Wnd;
					if sDirectory = '' then
						sDirectory := GetCurrentDir;
					if (sDirectory <> '') and
							(sDirectory[Length(sDirectory)] = '\') then
						sDirectory := Copy(sDirectory, 1, Length(sDirectory) - 1);
					if (sDirectory <> '') and (hHandle <> 0) then
						SendMessage(hHandle, BFFM_SETSELECTION, Integer(LongBool(True)),
												Integer(pChar(sDirectory)));
				except
					on e: Exception do
						ShowMessage(e.Message);
				end;
			BFFM_SELCHANGED:
				try
					SetString(dir, nil, MAX_PATH);
					if SHGetPathFromIDList(PItemIDList(lParam), PChar(Dir)) then
						sDirectory := pChar(Dir);
				except
					on e: Exception do
						ShowMessage(e.Message);
				end;
		end; // case
		result := 0;
	end;   //  Callback

// =============================================================
// Main function of SHBrowseForFolder, containing BROWSEINFO
// information used to display the 'BrowseForFolder' dialog box.
// =============================================================
begin
	result := false;
	if SHGetSpecialFolderLocation(hHandle, CSIDL_DRIVES, Root) = NOERROR then
	try
		SHGetMalloc(malloc);
		SetString(DispName, nil, MAX_PATH);

		bi.hwndOwner 			:= GetActiveWindow;
		bi.pidlRoot 			:= root;
		bi.pszDisplayName := PChar(DispName);
		bi.lpszTitle 			:= PChar(sTitle);
		bi.ulFlags 				:= BIF_RETURNONLYFSDIRS;
		bi.lpfn 					:= @FNBFFCallBack;
		bi.lParam 				:= Integer(self);

		iList := SHBrowseForFolder(bi);
		hHandle := 0;

		if iList <> nil then
		try
			if SHGetPathFromIDList(iList, PChar(DispName)) then
			begin
				sDirectory := PChar(DispName);
				result := true;
			end;
		finally
			malloc.Free(iList);
		end;
	finally
		malloc.Free(root);
	end;
end;

// =============================================================
// =================     Common procedures    ==================
// =============================================================

procedure TDBUtilForm.BtnStatus;
begin
	if BackupTbl then
	begin
		PackTbl := False;
		IndexTbl := False;
		Label3.Caption := 'Table being copied';
		EdtDesDir.Enabled := (ListBox2.Items.Count <> 0) ;
		if EdtDesDir.Enabled then
			EdtDesDir.Color := clWhite;
		EdtDesDir.Text := '';
	end
	else begin
		EdtDesDir.Enabled := False;
		EdtDesDir.Color := clSilver;
		EdtDesDir.Text := '';
	end;
	if PackTbl then
	begin
		IndexTbl := False;
		BackupTbl := False;
		Label3.Caption := 'Table being packed';
		EdtDesDir.Enabled := False;
		EdtDesDir.Color := clSilver;
		EdtDesDir.Text := '';
	end;
	if IndexTbl then
	begin
		PackTbl := False;
		BackUpTbl := False;
		Label3.Caption := 'Table being re-indexed';
		EdtDesDir.Enabled := False;
		EdtDesDir.Color := clSilver;
		EdtDesDir.Text := '';
	end;
	BtnDesDir.Enabled := BackUpTbl and (ListBox2.Items.Count <> 0);
	BtnBackUp.Down := BackUpTbl;
	BtnPack.Down := PackTbl;
	BtnIndex.Down := IndexTbl;
	BtnStart.Enabled := (Listbox2.Items.Count <> 0) and
		(BackupTbl or PackTbl or IndexTbl);
	ListBox1.Enabled := BackupTbl or PackTbl or IndexTbl;
end;

procedure TDBUtilForm.DisplayHint(Sender: TObject);
begin
	StatusBar1.SimpleText := GetLongHint(Application.Hint);
end;

procedure TDBUtilForm.FormActivate(Sender: TObject);
begin
	BtnStart.Enabled := False;
	// BtnIndex.Down:=True;
	BackUpTbl := False;
	IndexTbl := False;
	PackTbl := False;
	Edit1.Text := '';
	Label3.Caption := 'Table being re-indexed';
	ListBox1.Clear;
	// Retrieve names of BDE Alias fron IDAPI32.cfg and add to ListBox1
	Session.GetAliasNames(ListBox1.Items);
	if (ListBox1.Items.Count > 0) then
		ListBox1.ItemIndex := 0;
	Application.OnHint := DisplayHint;
	// Initialize DirDialog
	DirDialog := TDirDialog.Create;
	MessageDlg('It is highly recommended that you should make' + #13 +
						 'backup copies of your tables, before proceed' + #13 +
						 'with re-generating indexes or packing tables', mtWarning,
						 [mbOK], 0);
end;


procedure TDBUtilForm.BtnCloseClick(Sender: TObject);
begin
	Close;
end;

procedure TDBUtilForm.BtnBackupClick(Sender: TObject);
begin
	BackUpTbl := True;
	PackTbl := False;
	IndexTbl := False;
	BtnStatus;
end;

procedure TDBUtilForm.BtnPackClick(Sender: TObject);
begin
	PackTbl := True;
	BackUpTbl := False;
	IndexTbl := False;
	BtnStatus;
end;

procedure TDBUtilForm.BtnIndexClick(Sender: TObject);
begin
	IndexTbl := True;
	BackUpTbl := False;
	PackTbl := False;
	BtnStatus;
end;

procedure TDBUtilForm.BtnStartClick(Sender: TObject);
begin
	if BackupTbl and (EdtDesDir.Text = '') then
	begin
		MessageDlg('No Destination Directory specified', mtWarning,
			[mbOK], 0);
		exit;
	end;
	Edit1.Text := '';
	ListBox1.Enabled := False;
	BtnStart.Enabled := False;
	Table1.Close;
	Database1.DatabaseName := Listbox1.Items[ListBox1.ItemIndex];
	Table1.DatabaseName := ListBox1.Items[ListBox1.ItemIndex];
	if (IndexTbl = True) then
		PackTables(False)
	else if (PackTbl = True) then
		PackTables(True)
	else if (BackupTbl = True) then
		BackUpFiles;
	ListBox2.Clear;
//	ListBox1.Enabled := True;
//	ListBox1.SetFocus;
end;

procedure TDBUtilForm.ListBox1DblClick(Sender: TObject);
var
	i: integer;
begin
	Database1.Close;
	ListBox2.Items.Clear;
	Edit1.Text := '';
	Database1.AliasName := Listbox1.Items[ListBox1.ItemIndex];
	List1 := TStringList.Create;
	List1.Clear;
	// Retrieve table names of the selected database
	Session.GetTableNames(Database1.AliasName, '', True, False, List1);
	for i := 0 to List1.Count - 1 do
		Listbox2.Items.add(List1[i]);
 // BtnStart.Enabled:=True;
	BtnStatus;
end;

procedure TDBUtilForm.BtnAboutClick(Sender: TObject);
begin
	FrmAbout.ShowModal;
end;

procedure TDBUtilForm.BtnDesDirClick(Sender: TObject);
begin
	with DirDialog do
	begin
		sTitle := 'Select Destination Directory for copying Tables';
		if Execute then
			EdtDesDir.Text := sDirectory;
	end;
end;


end.

