{==============================================================================|
| Project : Notes/Delphi class library                           | 3.8         |
|==============================================================================|
| Content:                                                                     |
|==============================================================================|
| The contents of this file are subject to the Mozilla Public License Ver. 1.0 |
| (the "License"); you may not use this file except in compliance with the     |
| License. You may obtain a copy of the License at http://www.mozilla.org/MPL/ |
|                                                                              |
| Software distributed under the License is distributed on an "AS IS" basis,   |
| WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for |
| the specific language governing rights and limitations under the License.    |
|==============================================================================|
| Initial Developers of the Original Code are:                                 |
|   Sergey Kolchin (Russia) skolchin@usa.net ICQ#2292387                       |
|   Sergey Kucherov (Russia)                                                   |
|   Sergey Okorochkov (Russia)                                                 |
| All Rights Reserved.                                                         |
|   Last Modified:                                                             |
|     27.02.00, Sergey Kolchin                                                 |
|==============================================================================|
| Contributors and Bug Corrections:                                            |
|   Fujio Kurose                                                               |
|   Noah Silva                                                                 |
|   Tibor Egressi                                                              |
|   Andreas Pape                                                               |
|   Anatoly Ivkov                                                              |
|   Winalot                                                                    |
|     and others...                                                            |
|==============================================================================|
| History: see README.TXT                                                      |
|==============================================================================|
| Address selection form and routines                                          |
|==============================================================================|}
unit Form_LnAddress;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, Buttons, ComCtrls, Class_LotusNotes, Util_LnApi, ExtCtrls;

// Delphi version
{$IFDEF VER130}
  {$DEFINE D5}
  {$DEFINE D4}
{$ELSE}
  {$IFDEF VER120}
    {$DEFINE D4}
  {$ELSE}
    {$DEFINE D3}
  {$ENDIF}
{$ENDIF}

type
  TLnAddressOption = (laTo, laCc, laBcc, laSingleAddress, laPersonalABOnly);
  TLnAddressOptions = set of TLnAddressOption;

  TLnAddressDlg = class(TForm)
    Label1: TLabel;
    Label2: TLabel;
    cbAB: TComboBox;
    sbAddTo: TSpeedButton;
    sbAddCc: TSpeedButton;
    sbAddBcc: TSpeedButton;
    eAddress: TEdit;
    lvList: TListView;
    BtOK: TButton;
    BtCancel: TButton;
    sbDel: TSpeedButton;
    sbClear: TSpeedButton;
    DelayTimer: TTimer;
    tvSelected: TTreeView;
    sbCopyToPab: TSpeedButton;
    procedure FormDestroy(Sender: TObject);
    procedure cbABChange(Sender: TObject);
    procedure eAddressChange(Sender: TObject);
    procedure DelayTimerTimer(Sender: TObject);
    procedure lvListDataHint(Sender: TObject; StartIndex,
      EndIndex: Integer);
    procedure lvListData(Sender: TObject; Item: TListItem);
    procedure lvListKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure BtCancelClick(Sender: TObject);
    procedure sbAddToClick(Sender: TObject);
    procedure sbAddCcClick(Sender: TObject);
    procedure sbAddBccClick(Sender: TObject);
    procedure lvListClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure lvListDblClick(Sender: TObject);
    procedure sbClearClick(Sender: TObject);
    procedure sbDelClick(Sender: TObject);
    procedure BtOKClick(Sender: TObject);
    procedure sbCopyToPabClick(Sender: TObject);
  private
    Options: TLnAddressOptions;
    Database: TNotesDatabase;
    AddressBooks: TStringList;
    AB: TNotesDatabase;
    Docs: TNotesView;
    LastIndex: integer;
    Loading, InternalSet, UpdatePAB: boolean;
    ItemTo, ItemCc, ItemBcc: TTreeNode;
    procedure AddAddress (nAddTo: word; Address: string; fCheckMail: boolean);
		procedure SelectAB;
    procedure LoadItem (Item: TListItem; Index: integer);
    procedure AddStdItems;
  public
    SendTo, CcTo, BccTo: string;
    AddressBook: string;
    procedure SetInfo (aOptions: TLnAddressOptions; aTo, aCc, aBcc: string;
      aDatabase: TNotesDatabase);
  end;

var
  LnAddressDlg: TLnAddressDlg;

const
  // These vars hold view columns contain Name and address information
  NameColumn: string = '$39';
  AddressColumn: array [1..4] of string = ('$42', '$41', '$34', '$45');

  // View used to list people and groups
  PeopleViewName: string = '($PeopleGroupsFlat)';

  // Separatator of addresses
  AddressSeparator = ',';

function LnAddress (Options: TLnAddressOptions; var SendTo, Cc, Bcc: string;
      Database: TNotesDatabase): boolean;

implementation

{$R *.DFM}

const
  ADR_TO = 0;
  ADR_CC = 1;
  ADR_BCC = 2;

//****************************************************
function LnAddress;
var
  f: TLnAddressDlg;
begin
  Result := False;
  f := TLnAddressDlg.create(Application);
  try
    f.setInfo (Options, SendTo, Cc, Bcc, Database);
    f.showModal;
    if f.modalResult = mrOk then begin
      SendTo := f.SendTo;
      Cc := f.CcTo;
      Bcc := f.BccTo;
      Result := True;
    end;
  finally
    f.Free;
  end;
end;

//****************************************************
procedure TLnAddressDlg.AddAddress;
var
  ParentItem, Item: TTreeNode;
  Name1, Name2: TNotesName;
  i: integer;
begin
  case nAddTo of
    ADR_TO: ParentItem := ItemTo;
    ADR_CC: ParentItem := ItemCc;
    else    ParentItem := ItemBcc;
  end;
  Name1 := TNotesName.create(Address);
  Name2 := nil;
  try
    Address := Name1.Abbreviated;
    if fCheckMail and (lvList.Selected <> nil) then begin
      // Check the typed address against selected in lvList
      Name2 := TNotesName.create(lvList.Selected.Subitems[0]);
      if compareText(Name1.Common, Name2.Common) = 0 then Address := Name2.Abbreviated;
    end;
    with tvSelected do begin
      for i := 0 to Items.Count-1 do if CompareText(Items[i].Text, Address) = 0 then begin
        MessageBeep(0);
        Abort;
      end;
      Item := Items.AddChild (ParentItem, Address);
      Item.ImageIndex := 3;
      FullExpand;
    end;
  finally
    Name2.free;
    Name1.free;
  end;
end;

//****************************************************
procedure TLnAddressDlg.SelectAB;
var
  ab_name: string;
  i: integer;
begin
  lvList.Items.Clear;
  if cbAB.ItemIndex < 0 then exit;

  if Docs <> nil then begin
    Docs.free;
    Docs := nil;
  end;
  if AB <> nil then begin
    AB.free;
    AB := nil;
  end;

  Screen.Cursor := crHourglass;
  try
    AB := TNotesDatabase.create;
    Docs := nil;
    lvList.Items.BeginUpdate;
    try
      ab_name := AddressBooks[cbAB.itemIndex];
      i := Pos('=', ab_name);
      if i > 0 then delete(ab_name, i, length(ab_name)-i+1);
      if AddressBooks.Objects[cbAB.itemIndex] = pointer(1)
        then Ab.Open ('', ab_name)
        else Ab.Open (Database.Server, ab_name);
      Docs  := TNotesView.Create (Ab, PeopleViewName);
      if UpdatePAB and (cbAB.itemIndex = 0) then begin
        Docs.Update;
        UpdatePAB := False;
      end;
      {$IFDEF D4}
      lvList.Items.Count := Docs.count; //virtual view
      {$ELSE}
      // in D3, there's no "virtual" views
      for i := 0 to Docs.count-1 do begin
        LoadItem(lvList.Items[i], i);
      end;
      LastIndex := Docs.count;
      {$ENDIF}
      lvList.Repaint;
      SbCopyToPAB.enabled := AddressBooks.Objects[cbAB.itemIndex] <> pointer(1);
    except
      Docs.free;
      AB.free;
      raise;
    end;
  finally
    lvList.Items.EndUpdate;
    Screen.Cursor := crDefault;
  end;
end;

//****************************************************
procedure TLnAddressDlg.SetInfo;

procedure ParseAdr (Adr: string; nAdr: integer);
var
  n: integer;
begin
  n := Pos(AddressSeparator, Adr);
  while n <> 0 do begin
    AddAddress (nAdr, trim(copy(Adr,1,n-1)), False);
    delete (Adr, 1, n);
    n := Pos(AddressSeparator, Adr);
  end;
  if Adr <> '' then AddAddress (nAdr, trim(Adr), False);
end;

var
  i,n: integer;
  srv, s: string;
begin
  Options := aOptions;
  Database := aDatabase;

  ParseAdr(aTo, ADR_TO);
  ParseAdr(aCc, ADR_CC);
  ParseAdr(aBcc, ADR_BCC);
  if (Options * [laTo, laCc, laBcc]) = [] then include(Options, laTo);
  sbAddTo.enabled := laTo in Options;
  sbAddCc.enabled := laCc in Options;
  sbAddBcc.enabled := laBcc in Options;

  // List address books
  srv := TNotesName.TranslateName (Database.Server, False, '');
  cbAB.clear;
  AddressBooks := TStringList.create;
  Database.ListAddressBooks ('', AddressBooks, True);
  for i := 0 to AddressBooks.count-1 do AddressBooks.Objects[i] := pointer(1);  //local db

  if not (laPersonalAbOnly in Options) then
    Database.ListAddressBooks (Database.Server, AddressBooks, True);
  for i := 0 to AddressBooks.count-1 do begin
    n := pos('=', AddressBooks[i]);
    s := copy(AddressBooks[i], n+1, length(AddressBooks[i])-n+1);
    if AddressBooks.Objects[i] = nil then appendStr (s, ' on ' + srv);
    cbAB.items.add(s);
  end;
  if cbAB.items.count > 0 then cbAB.itemIndex := 0;

  SelectAB;
end;

procedure TLnAddressDlg.FormDestroy(Sender: TObject);
begin
  AddressBooks.free;
  AB.free;
  Docs.free;
end;

procedure TLnAddressDlg.cbABChange(Sender: TObject);
begin
  SelectAB;
end;

procedure TLnAddressDlg.eAddressChange(Sender: TObject);
begin
  if not InternalSet then begin
    DelayTimer.Enabled := False;
    DelayTimer.Enabled := True;
  end;
end;

procedure TLnAddressDlg.DelayTimerTimer(Sender: TObject);
var
  i: integer;
  Name2: TNotesName;
  s: string;
begin
  if Loading then begin
    // Already in search - stop it
    Loading := False;
    exit;
  end;
  DelayTimer.Enabled := False;
  if eAddress.text = '' then begin
    if lvList.Items.count > 0 then lvList.Selected := lvList.Items[0];
  end
  else begin
    s := UpperCase(eAddress.Text);
    Name2 := TNotesName.create ('');
    Loading := True;
    Screen.Cursor := crHourglass;
    try
      for i := 0 to Docs.count-1 do begin
        if i < LastIndex then begin
          LoadItem(lvList.Items[i], i);
          LastIndex := i;
        end;
        Name2.Abbreviated := lvList.Items[i].SubItems[0];
        if (Pos (s, UpperCase(Name2.Common)) = 1) or
        (Pos (s, UpperCase(lvList.Items[i].Caption)) = 1) then begin
          lvList.Selected := lvList.Items[i];
          break;
        end;
        Application.processMessages;
        if not Loading then break;
      end;
    finally
      //Name1.free;
      Name2.free;
      Loading := False;
      Screen.Cursor := crDefault;
    end;
  end;
  if lvList.Selected = nil then MessageBeep(0) else lvList.Selected.MakeVisible (False);
end;

procedure TLnAddressDlg.LoadItem;
var
  Summary: TStringList;
  s: string;
  i: integer;
begin
  if item.Caption = '' then begin
    Summary := TStringList.create;
    try
      Summary.Text := Docs.SummaryValues[Index];
      if Summary.Values['Type'] = 'Person' then item.ImageIndex := 0
      else if Summary.Values['Type'] = 'Group' then item.ImageIndex := 1
      else item.ImageIndex := 2;
      item.Caption := Summary.Values[NameColumn];
      for i := System.Low(AddressColumn) to System.High(AddressColumn) do begin
        s := Summary.Values[AddressColumn[i]];
        if length(s) = 1 then s := '';
        if s <> '' then break;
      end;
      if s = '' then s := item.Caption;
      item.SubItems.Add (TNotesName.TranslateName (s, False, ''));
    finally
      Summary.free;
    end;
  end;
end;

procedure TLnAddressDlg.AddStdItems;
begin
  ItemTo := tvSelected.Items.Add (nil, 'To:');
  ItemTo.ImageIndex := 0;
  ItemCc := tvSelected.Items.Add (nil, 'Cc:');
  ItemCc.ImageIndex := 1;
  ItemBcc := tvSelected.Items.Add (nil, 'Bcc:');
  ItemBCc.ImageIndex := 2;
end;

procedure TLnAddressDlg.lvListDataHint(Sender: TObject; StartIndex,
  EndIndex: Integer);
var
  i: integer;
begin
  for i := StartIndex to EndIndex do LoadItem(lvList.Items[i], i);
end;

procedure TLnAddressDlg.lvListData(Sender: TObject; Item: TListItem);
begin
  LoadItem (Item, Item.Index);
end;

procedure TLnAddressDlg.lvListKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  Loading := False;
end;

procedure TLnAddressDlg.BtCancelClick(Sender: TObject);
begin
  if Loading then begin
    // Loading in progress
    Loading := False;
    ModalResult := mrNone;
    exit;
  end;
end;

procedure TLnAddressDlg.sbAddToClick(Sender: TObject);
begin
  AddAddress (ADR_TO, eAddress.Text, True);
end;

procedure TLnAddressDlg.sbAddCcClick(Sender: TObject);
begin
  AddAddress (ADR_CC, eAddress.Text, True);
end;

procedure TLnAddressDlg.sbAddBccClick(Sender: TObject);
begin
  AddAddress (ADR_BCC, eAddress.Text, True);
end;

procedure TLnAddressDlg.lvListClick(Sender: TObject);
begin
  if lvList.Selected <> nil then begin
    InternalSet := True;
    eAddress.Text := lvList.Selected.SubItems[0]; //add e-mail address
    InternalSet := False;
  end;
end;

procedure TLnAddressDlg.FormCreate(Sender: TObject);
begin
  AddStdItems;
end;

procedure TLnAddressDlg.lvListDblClick(Sender: TObject);
begin
  if lvList.Selected <> nil then
    AddAddress (ADR_TO, lvList.Selected.SubItems[0], False);
end;

procedure TLnAddressDlg.sbClearClick(Sender: TObject);
begin
  tvSelected.Items.Clear;
  AddStdItems;
end;

procedure TLnAddressDlg.sbDelClick(Sender: TObject);
begin
  with tvSelected do
    if (Selected <> nil) and (Selected <> ItemTo) and (Selected <> ItemCc) and
    (Selected <> ItemBcc) then Items.Delete (Selected);
end;

procedure TLnAddressDlg.BtOKClick(Sender: TObject);

function ListItems (Parent: TTreeNode): string;
var
  Item: TTreeNode;
begin
  Result := '';
  Item := Parent.getFirstChild;
  while Item <> nil do begin
    AppendStr(Result, Item.Text + AddressSeparator);
    Item := Parent.GetNextChild(Item);
  end;
  if (Result <> '') then delete(Result, length(Result), 1);
end;

begin
  try
    SendTo := ListItems(ItemTo);
    CcTo := ListItems(ItemCc);
    BccTo := ListItems(ItemBcc);
    if AB <> nil then AddressBook := Ab.FullName else AddressBook := ''; 
  except
    ModalResult := mrNone;
    raise;
  end;
end;

procedure TLnAddressDlg.sbCopyToPabClick(Sender: TObject);
var
  Doc: TNotesDocument;
  FirstAB: TNotesDatabase;
begin
  if lvList.Selected <> nil then begin
    FirstAB := TNotesDatabase.create;
    Doc := Docs.Document[lvList.Selected.Index];
    try
      FirstAB.OpenPrivateAddressBook;
      Doc.CopyToDatabase(FirstAB);
      FirstAB.close;
      UpdatePAB := True;
    finally
      Doc.free;
      FirstAB.free;
    end;
  end;
end;

end.
