unit KDaoDataBase;
{$DEFINE USEDB}      //DISABLE IF YOU WANT TO USE A PURE DAO WITHOUT KDaoTable;
{$DEFINE DYNADAO}     //This is the preffered way to use DAO - disable if
                     //one of the bottom options are enabled
{DEFINE DAO35}      //Disable DYNADAO and DAO36 to USE
{DEFINE DAO36}       //Disable DYNADAO and DAO35 to USE

//******************************************************************************
//    WARNING!!! - BOTH Database and Table MUST BE SET TO THE SAME DAO METOD !!! 
//******************************************************************************
// Protection for absent-minded people
//******************************************************************************
{$IFDEF DYNADAO}
  {$IFDEF DAO36}
   YOU CANNOT DEFINE BOTH DYNADAO AND DAO36
  {$ENDIF}
  {$IFDEF DAO35}
   YOU CANNOT DEFINE BOTH DYNADAO AND DAO35
  {$ENDIF}
{$ELSE}
  {$IFDEF DAO36}
     {$IFDEF DAO35}
        YOU CANNOT DEFINE BOTH DAO35 AND DAO36
     {$ENDIF}
  {$ENDIF}
{$ENDIF}
//******************************************************************************
//                    Delphi Dao Project Version 1.9
//                 Copyright (c) 2000 by Kiril Antonov
//******************************************************************************

//****************************************** CHANGES ***************************
// 30.05.2000 - Added a checking of database for supporting transactions
//              If database does NOT support transactions
//              NO DAO action is performed                         
// 08.06.2000 - Adding support for Dynamycally setting DAO Version
//                                                               
// 12.06.2000 - Fixed a login bug for ISAM databases
//
// 14.06.2000 - Added support for creating autoincrement fields
//              How to use:
//                 Use constant dbAutoIncInteger for FieldType
//              Restrictions: (This is a DAO restrictions not the component!)
//                 No retrictions when creating new table (BUT ONLY ONE AutoInc
//                    Field per table)
//                 Only ONE AutoInc Field per table
//
// 14.06.2000 - Renamed F_RefreshDefinitions to RefreshDefinitions
//
// 18.06.2000 - Fixed a bug with setting Dao Version when TKADaoDatabase is
//              created.
//              WARNING!!! INITIAL VERSION OF KADAO IS SET TO 3.5 NOW!
// 19.06.2000 - Fixed a minor bug when a database control is deleted
//              Now all tables linked to KADaoDatabase control work properly
//              when control is deleted
//
// 26.06.2000 - Added Idle method to acces DBEngine Idle
//
// 26.06.2000 - Rewrited DaoVersion and SystemDatabase properties
//
// 27.06.2000 - Rewrited EngineType property
//
// 28.06.2000 - Minor fix: now CoreWorkspace is closed each time a new workspace
//              is created
//
// 28.06.2000 - Added read only property DatabaseLanguage for information
//              purpouses. If you want a LocaleCode DatabaseLanguageInt contains
//              them
// 28.06.2000 - Added CompactAccessDatabaseEx - No comment see code
//              Seee also new Language constants in DaoApi.pas
//
// 28.06.2000 - Added CreateAccessDatabaseEx2 - Seee new Language constants
//              in DaoApi.pas
//
// 29.06.2000 - Added CheckEngines method for avoiding exceptions when checking
//              available versions of DAO
//
// 29.06.2000 - Added F_FindWorkspace method for avoiding exceptions when
//              creating a new workspace
//
// 03.07.2000  CreateTable and AddFieldsToTable rewrited
//             Still problems with creating Paradox primary index - HELP NEEDED!
// 05.07 2000 - Fixed a very rediculous bug with Version property
//              It seems that a 4 July is a day of shame for me!
//******************************************************************************
interface
Uses
DAOApi,
{$IFDEF DYNADAO}
ComObj,
{$ENDIF}
{$IFDEF DAO35}
DAO35Api,
{$ENDIF}
{$IFDEF DAO36}
DAO36Api,
{$ENDIF}
Windows,SysUtils,Classes,FileCtrl,DbLogdlg,Registry,DsgnIntf;

Type
 TDaoErrRec=Record
            ErrNo       : Integer;
            Source      : String;
            Description : String;
            HelpFile    : String;
            HelpContext : Integer;
          End;
 PDaoErrRec=^TDaoErrRec;

TDatabaseTypeEditor = class(TStringProperty)
    Public
      Procedure GetValues( Proc: TGetStrProc); override;
      Procedure SetValue(const Value: string); override;
      Function  GetAttributes: TPropertyAttributes; override;
    End;

TDatabaseNameEditor = class(TStringProperty)
    Public
      Procedure Edit;override;
      Procedure SetValue(const Value: string); override;
      Function  GetAttributes: TPropertyAttributes; override;
    End;

TSystemDatabaseNameEditor = class(TStringProperty)
    Public
      Procedure Edit;override;
      Procedure SetValue(const Value: string); override;
      Function  GetAttributes: TPropertyAttributes; override;
    End;

TEngineTypeEditor = class(TIntegerProperty)
    Public
     function  GetValue: string; override;
     Procedure GetValues( Proc: TGetStrProc);override;
     procedure SetValue(const Value: string); override;
     Function  GetAttributes: TPropertyAttributes; override;
    End;

TWorkspaceEditor = class(TStringProperty)
    Public
      Procedure GetValues( Proc: TGetStrProc); override;
      Function  GetAttributes: TPropertyAttributes; override;
    End;

{$IFDEF DYNADAO}
TDaoVersionEditor = class(TStringProperty)
    Public
      Procedure GetValues( Proc: TGetStrProc); override;
      Procedure SetValue(const Value: string); override;
      Function  GetAttributes: TPropertyAttributes; override;
    End;
{$ENDIF}


TKADaoDatabase = Class(TComponent)
       Private
         F_Database        : String;
         F_EngineType      : Integer;
         F_DatabaseType    : String;
         F_Workspace       : String;
         F_CollatingOrder  : String;
         F_DaoVersion      : String;
         F_ActualDaoVersion: String;
         F_SystemDB        : String;
         F_Active          : Boolean;
         F_ReadOnly        : Boolean;
         F_Exclusive       : Boolean;
         F_LoginPrompt     : Boolean;
         F_Username        : String;
         F_Password        : String;
         F_MachineName     : String;
         F_QueryTimeout    : Integer;
         F_LoginDialog     : TLoginDialog;
         F_TableNames      : TStringList;
         F_ActiveTableNames: TStringList;
         F_QueryDefNames   : TStringList;
         F_DBTypesList     : TStringList;
         F_DriverList      : TStringList;
         F_DaoVersionList  : TStringList;
         F_OLE_ON          : Boolean;
         F_Destroying      : Boolean;
         procedure F_Get_DBTypesList(List: TStrings);
         Function  F_Get_DBTypeFileExtension(DBType:String):String;
         Function  F_Get_DBTypeTableType(DBType:String):String;
         procedure F_Get_OdbcDriverList(List: TStrings);
         Procedure F_Set_DaoVersion(Value : String);
         Procedure F_Set_ActualDaoVersion(Value : String);
         Procedure F_Set_Database(Value : String);
         Function  F_Get_SystemDatabaseFromRegistry:String;
         Procedure F_Set_SystemDatabase(Value : String);
         Function  F_FindWorkspace(WS:String):Boolean;
         Procedure F_Set_Workspace(Value : String);
         Function  F_Get_DatabaseType:String;
         Procedure F_Set_DatabaseType(Value : String);
         Function  F_Get_CollatingOrder:String;
         Procedure F_Set_EngineType(Value : Integer);
         Procedure F_Set_UserName(Value : String);
         Procedure F_Set_Password(Value : String);
         Procedure F_Set_Exclusive(Value : Boolean);
         Procedure F_Set_LoginPrompt(Value : Boolean);
         Procedure F_Set_ReadOnly(Value : Boolean);
         Procedure F_Set_Active(Value : Boolean);
       Protected
       Published
         Property    Exclusive       : Boolean Read F_Exclusive Write F_Set_Exclusive;
         Property    EngineType      : Integer Read F_EngineType Write F_Set_EngineType;
         Property    DatabaseLanguage: String Read F_Get_CollatingOrder Write F_CollatingOrder;
         Property    DatabaseType    : String  Read F_Get_DatabaseType Write F_Set_DatabaseType;
         Property    Database        : String  Read F_Database Write F_Set_Database;
         Property    ReadOnly        : Boolean Read F_ReadOnly Write F_Set_ReadOnly;
         Property    LoginPrompt     : Boolean Read F_LoginPrompt Write F_Set_LoginPrompt;
         Property    UserName        : String Read F_Username  Write F_Set_UserName;
         Property    Password        : String Read F_Password  Write F_Set_Password;
         Property    Version         : String Read F_DaoVersion Write F_Set_DaoVersion;
         Property    VersionDetails  : String Read F_ActualDaoVersion Write F_Set_ActualDaoVersion;
         Property    SystemDatabase  : String Read F_SystemDB Write F_Set_SystemDatabase;
         Property    Workspace       : String Read F_Workspace Write F_Set_Workspace;
         Property    QueryTimeout    : Integer Read F_QueryTimeout Write F_QueryTimeout;
         Property    Connected       : Boolean Read F_Active Write F_Set_Active Default False;
       Public
         DatabaseLanguageInt         : Integer;
         {$IFDEF DYNADAO} //****************************************************
         CoreDBEngine                : OleVariant;
         CoreDatabase                : OleVariant;
         CoreWorkspace               : OleVariant;
         {$ENDIF}
         {$IFDEF DAO35}
         CoreDBEngine                : DAO35Api.DBEngine;
         CoreDatabase                : DAO35Api.Database;
         CoreWorkspace               : DAO35Api.Workspace;
         {$ENDIF}
         {$IFDEF DAO36}
         CoreDBEngine                : DAO36Api.DBEngine;
         CoreDatabase                : DAO36Api.Database;
         CoreWorkspace               : DAO36Api.Workspace;
         {$ENDIF}
         Property    QueryDefNames   : TStringList Read F_QueryDefNames;
         Property    TableNames      : TStringList Read F_TableNames;
         Property    ActiveTableNames: TStringList Read F_ActiveTableNames;
         Property    DatabaseTypes   : TStringList Read F_DBTypesList;
         {$IFDEF DYNADAO}
         Procedure                     CheckEngines(DaoVer:String);
         {$ENDIF}
         Function                      GetLastDaoError:TDaoErrRec;
         Procedure                     CreateDBEngine(DaoVer:String);
         Constructor                   Create(AOwner : TComponent); override;
         Destructor                    Destroy; override;
         //****************************************************** Transactions
         Procedure                   StartTransaction;
         Procedure                   Commit;
         Procedure                   Rollback;
         //****************************************************** Utils
         Procedure                   RepairAccessDatabase(DatabaseName:String);
         Procedure                   CompactAccessDatabase(DatabaseName:String);
         Procedure                   CompactAccessDatabaseEx(DatabaseName: String;
                                                             NewLocale   : String;
                                                             Encrypt     : Boolean;
                                                             Decrypt     : Boolean;
                                                             NewVersion  : Integer;
                                                             Password    : String);

         Procedure                   CreateAccessDatabase(DatabaseName:String);
         Procedure                   CreateAccessDatabaseEx(DatabaseName,LANGID,CP,COUNTRY,Password,Version:String;Encrypt:Boolean);
         Procedure                   CreateAccessDatabaseEx2(DatabaseName,Language,Password,Version:String;Encrypt:Boolean);
         //****************************************************** Utils II
         Function                    CreateTable(TableName:String; FieldNames : Variant; FieldTypes : Variant; FieldSizes : Variant; FieldIndexes:Variant):Boolean;
         Procedure                   LinkExternalTable(Database,TableName:String);
         Procedure                   RenameTable(OldTableName,NewTableName:String);
         Procedure                   DeleteTable(TableName:String);


         Function                    CreateIndex(TableName,FieldName:String;IndexType:Integer):Boolean;
         Procedure                   RenameIndex(TableName,OldIndexName,NewIndexName:String);
         Procedure                   DeleteIndexByName(TableName,IndexName:String);
         Procedure                   DeleteIndexByFieldName(TableName,FieldName:String);

         Procedure                   RenameField(TableName,OldFieldName,NewFieldName:String);
         Procedure                   DeleteField(TableName,FieldName:String);
         Function                    AddFieldsToTable(TableName:String; FieldNames : Variant; FieldTypes : Variant; FieldSizes : Variant; FieldIndexes:Variant):Boolean;


         Function                    CreateQueryDef(Name:String;SQL:String):Boolean;
         Function                    GetQueryDefSQLText(Name:String):String;
         Procedure                   RenameQueryDef(OldQueryName,NewQueryName:String);
         Procedure                   DeleteQueryDef(QueryName:String);

         Procedure                   RefreshDefinitions;
         Procedure                   Idle;
      End;

      procedure Register;
{$IFNDEF VER130}
var
  //   ***************************************************
  //   Defined only for Delphi3 and Delphi4
  //   Delphi5 has buildin support for EmptyParam
  //   ***************************************************
  EmptyParam: OleVariant;
{$ENDIF}


//*************************************************************************************************
implementation
Uses Dialogs,Forms,ActiveX{$IFDEF USEDB},DB,KDaoTable{$ENDIF};

Const
  dbLangGeneral = ';LANGID=%s;CP=%s;COUNTRY=%s';
Var
  DlgChooseDatabase  : TOpenDialog;

{$IFNDEF USEDB}
Procedure DatabaseError(Msg:String);
Begin
  Exception.Create(Msg);
End;
{$ENDIF}

Function  TKADaoDatabase.GetLastDaoError:TDaoErrRec;
Begin
  Result.ErrNo         := 0;
  Result.Source        := '';
  Result.Description   := '';
  Result.HelpFile      := '';
  Result.HelpContext   := 0;
  if Not F_Active then Exit;
  if CoreDBEngine.Errors.Count=0 Then Exit;
  Result.ErrNo       := CoreDBEngine.Errors.Item[0].Number;
  Result.Source      := CoreDBEngine.Errors.Item[0].Source;
  Result.Description := CoreDBEngine.Errors.Item[0].Description;
  Result.HelpFile    := CoreDBEngine.Errors.Item[0].HelpFile;
  Result.HelpContext := CoreDBEngine.Errors.Item[0].HelpContext;
End;

{$IFDEF DYNADAO}
Procedure TKADaoDatabase.CheckEngines(DaoVer:String);
Var
  V35 : String;
  V36 : String;
 oReg : TRegistry;
Begin
  V35 := 'DAO.DBEngine.35';
  V36 := 'DAO.DBEngine.36';
  oReg := TRegistry.Create;
  oReg.RootKey := HKEY_CLASSES_ROOT;
  if oReg.OpenKey(V36,False) then
     Begin
       If Not VarIsNull(CoreDBEngine) Then CoreDBEngine := NULL;
       Try
        CoreDBEngine               := CreateOleObject(V36);
        CoreDBEngine               := NULL;
        F_DaoVersionList.Add('3.6');
        oReg.CloseKey;
        Except
        End;
     End;
  oReg.Free;

  oReg := TRegistry.Create;
  oReg.RootKey := HKEY_CLASSES_ROOT;
  if oReg.OpenKey(V35,False) then
     Begin
       If Not VarIsNull(CoreDBEngine) Then CoreDBEngine := NULL;
       Try
        CoreDBEngine               := CreateOleObject(V35);
        CoreDBEngine               := NULL;
        F_DaoVersionList.Add('3.5');
        oReg.CloseKey;
       Except
       End;
     End;
  oReg.Free;
  If Not VarIsNull(CoreDBEngine) Then CoreDBEngine := NULL;
End;
{$ENDIF}


//*************************************************************************************************
Procedure TKADaoDatabase.CreateDBEngine(DaoVer:String);
Var
  V35 : String;
  V36 : String;
Begin
 V35 := 'DAO.DBEngine.35';
 V36 := 'DAO.DBEngine.36';

 F_DaoVersionList.Clear;
 {$IFDEF DYNADAO}
  CheckEngines(DaoVer);
  if DaoVer='3.5' Then
     Begin
       Try
        CoreDBEngine               := CreateOleObject(V35);
        F_DaoVersion               := '3.5';
       Except
         Try
          CoreDBEngine             := CreateOleObject(V36);
          F_DaoVersion             := '3.6';
         Except
          DatabaseError('TKADaoDatabase.Create: You does not have Microsoft DAO installed ot it is an OLD Version!');
         End;
       End;
   End;
  if DaoVer='3.6' Then
     Begin
       Try
        CoreDBEngine             := CreateOleObject(V36);
        F_DaoVersion             := '3.6';
       Except
        DatabaseError('TKADaoDatabase.Create: You does not have Microsoft DAO installed ot it is an OLD Version!');
       End;
   End;


  {$ELSE}
  CoreDBEngine               := Nil;
  CoreDBEngine               := CoDBEngine.Create;
  {$IFDEF DAO35}
  F_DaoVersion               := '3.5';
  {$ENDIF}
  {$IFDEF DAO36}
  F_DaoVersion               := '3.6';
  {$ENDIF}
  F_DaoVersionList.Add(F_DaoVersion);
  {$ENDIF}
  F_ActualDaoVersion       := CoreDBEngine.Version;
End;


Constructor TKADaoDatabase.Create(AOwner : TComponent);
Var
  OLE_INIT:Integer;
Begin
  Inherited Create(AOwner);

  F_OLE_ON:=False;
  OLE_INIT:= CoInitialize(NIL);
  if (OLE_INIT = S_OK) or (OLE_INIT = S_FALSE) then F_OLE_ON:= True
  else DatabaseError('Unable to init OLE objects!');

  F_DaoVersionList             := TStringList.Create;
  F_DaoVersionList.Clear;

  CreateDBEngine('3.5');
  F_EngineType             := dbUseJet;
  F_SystemDB               := F_Get_SystemDatabaseFromRegistry;
  if F_SystemDB <> '' Then
  CoreDBEngine.SystemDB    := F_SystemDB;
  CoreDBEngine.DefaultType := F_EngineType;
  CoreWorkspace            := CoreDBEngine.Workspaces.Item[0];


  F_Workspace              := CoreWorkspace.Name;

  F_MachineName            := '';
  F_DatabaseType           :='Access';
  F_Active                 := False;
  F_Database               := '';
  F_ReadOnly               := False;
  F_Exclusive              := False;
  F_LoginPrompt            := False;
  F_Username               :='Admin';
  F_Password               :='';

  F_LoginDialog            := TLoginDialog.CreateParented(Application.Handle);
  F_TableNames             := TStringList.Create;
  F_ActiveTableNames       := TStringList.Create;
  F_QueryDefNames          := TStringList.Create;
  F_DBTypesList            := TStringList.Create;
  F_DriverList             := TStringList.Create;

  F_Destroying             := False;

  F_Get_DBTypesList(F_DBTypesList);
  F_Get_OdbcDriverList(F_DriverList);
End;

Destructor  TKADaoDatabase.Destroy;
Begin
 F_Destroying:=True;
 If F_Active Then Connected := False;
 F_TableNames.Free;
 F_ActiveTableNames.Free;
 F_QueryDefNames.Free;
 F_LoginDialog.Free;
 F_DBTypesList.Free;
 F_DriverList.Free;
 F_DaoVersionList.Free;

 {$IFDEF DYNADAO}
 CoreDatabase     := NULL;
 CoreWorkspace    := NULL;
 CoreDBEngine     := NULL;
 {$ELSE}
 CoreDatabase  := Nil;
 CoreWorkspace := Nil;
 CoreDBEngine  := Nil;
 {$ENDIF}
 if F_OLE_ON then CoUninitialize;
 Inherited Destroy;
End;

procedure TKADaoDatabase.F_Get_OdbcDriverList(List: TStrings);
var
   oReg : TRegistry;
Begin
     oReg := TRegistry.Create;
     try
     Begin
          oReg.RootKey := HKEY_LOCAL_MACHINE;
          if oReg.OpenKey('SOFTWARE\ODBC\ODBCINST.INI\ODBC Drivers',False) then
          Begin
               List.Clear;
               oReg.GetValueNames(List);
          End;
     End;
     finally
          oReg.Free;
     End;
End;

procedure TKADaoDatabase.F_Get_DBTypesList(List: TStrings);
var
   oReg : TRegistry;
Begin
     oReg := TRegistry.Create;
     try
     Begin
          oReg.RootKey := HKEY_LOCAL_MACHINE;
          {$IFDEF DAO35}
          if oReg.OpenKey('SOFTWARE\Microsoft\JET\3.5\ISAM Formats',False) then
             Begin
               List.Clear;
               oReg.GetKeyNames(List);
             End;
          {$ENDIF}
          {$IFDEF DAO36}
          if oReg.OpenKey('SOFTWARE\Microsoft\JET\4.0\ISAM Formats',False) then
             Begin
               List.Clear;
               oReg.GetKeyNames(List);
             End;
          {$ENDIF}
          {$IFDEF DYNADAO}//****************************************************
          if F_DaoVersion='3.5' then
          if oReg.OpenKey('SOFTWARE\Microsoft\JET\3.5\ISAM Formats',False) then
             Begin
               List.Clear;
               oReg.GetKeyNames(List);
             End;
          if F_DaoVersion='3.6' then
          if oReg.OpenKey('SOFTWARE\Microsoft\JET\4.0\ISAM Formats',False) then
             Begin
               List.Clear;
               oReg.GetKeyNames(List);
             End;
          {$ENDIF}
     End;
     finally
          oReg.Free;
     End;
    List.Insert(0,'Access');
End;


Function TKADaoDatabase.F_Get_DBTypeFileExtension(DBType:String):String;
var
   oReg : TRegistry;
Begin
     oReg := TRegistry.Create;
     try
     Begin
          oReg.RootKey := HKEY_LOCAL_MACHINE;
          {$IFDEF DAO35}
          if oReg.OpenKey('SOFTWARE\Microsoft\JET\3.5\ISAM Formats\'+DBType,False) then
             Begin
               Result:=oReg.ReadString('ExportFilter');
               if Result='' Then Result:=oReg.ReadString('ImportFilter');
             End;
          {$ENDIF}
          {$IFDEF DAO36}
          if oReg.OpenKey('SOFTWARE\Microsoft\JET\4.0\ISAM Formats\'+DBType,False) then
             Begin
               Result:=oReg.ReadString('ExportFilter');
               if Result='' Then Result:=oReg.ReadString('ImportFilter');
             End;
          {$ENDIF}
          {$IFDEF DYNADAO}
          if F_DaoVersion='3.5' then
          if oReg.OpenKey('SOFTWARE\Microsoft\JET\3.5\ISAM Formats\'+DBType,False) then
             Begin
               Result:=oReg.ReadString('ExportFilter');
               if Result='' Then Result:=oReg.ReadString('ImportFilter');
             End;
          if F_DaoVersion='3.6' then
          if oReg.OpenKey('SOFTWARE\Microsoft\JET\4.0\ISAM Formats\'+DBType,False) then
             Begin
               Result:=oReg.ReadString('ExportFilter');
               if Result='' Then Result:=oReg.ReadString('ImportFilter');
             End;
          {$ENDIF}
     End;
     finally
          oReg.Free;
     End;
End;

Function TKADaoDatabase.F_Get_DBTypeTableType(DBType:String):String;
var
   oReg : TRegistry;
   BUF  : Array[1..1000] of Byte;
Begin
     oReg := TRegistry.Create;
     try
     Begin
          oReg.RootKey := HKEY_LOCAL_MACHINE;
          {$IFDEF DAO35}
          if oReg.OpenKey('SOFTWARE\Microsoft\JET\3.5\ISAM Formats\'+DBType,False) then
             Begin
               oReg.ReadBinaryData('OneTablePerFile',BUF,1000);
               Result:=IntToStr(BUF[1]);
             End;
          {$ENDIF}
          {$IFDEF DAO36}
          if oReg.OpenKey('SOFTWARE\Microsoft\JET\4.0\ISAM Formats\'+DBType,False) then
             Begin
               oReg.ReadBinaryData('OneTablePerFile',BUF,1000);
               Result:=IntToStr(BUF[1]);
             End;
          {$ENDIF}
          {$IFDEF DYNADAO}
          if F_DaoVersion='3.5' then
          if oReg.OpenKey('SOFTWARE\Microsoft\JET\3.5\ISAM Formats\'+DBType,False) then
             Begin
               oReg.ReadBinaryData('OneTablePerFile',BUF,1000);
               Result:=IntToStr(BUF[1]);
             End;
          if F_DaoVersion='3.6' then
          if oReg.OpenKey('SOFTWARE\Microsoft\JET\4.0\ISAM Formats\'+DBType,False) then
             Begin
               oReg.ReadBinaryData('OneTablePerFile',BUF,1000);
               Result:=IntToStr(BUF[1]);
             End;
          {$ENDIF}
     End;
     finally
          oReg.Free;
     End;
End;


Procedure TKADaoDatabase.F_Set_Active(Value : Boolean);
Var
  X            : Integer;
  {$IFDEF USEDB}
  ATable       : TKADaoTable;
  {$ENDIF}
  Pwd          : String;
Begin
  if (F_Active) And (Value) Then Exit;
  if (F_Database='') Then
      Begin
       DatabaseError('TKADaoDatabase.F_Set_Active: Database is not defined!');
       Exit;
     End;
  if (F_DatabaseType='') Then
      Begin
       DatabaseError('TKADaoDatabase.F_Set_Active: Database type is not defined!');
       Exit;
     End;
  if (F_Active) And (NOT Value) Then
     Begin
       F_TableNames.Clear;
       F_QueryDefNames.Clear;
       {$IFDEF USEDB}
       For X:=0 to F_ActiveTableNames.Count-1 do
           Begin
             ATable:=TKADaoTable(F_ActiveTableNames.Objects[X]);
             ATable.MainDatabaseShutdown:=True;
             ATable.Active:=False;
             if F_Destroying Then ATable.Database:=Nil;
           End;
       {$ENDIF}
       F_ActiveTableNames.Clear;
       CoreDatabase.Close;
       F_Active:=False;
     End;
  if (NOT F_Active) And (Value) Then
     Begin
        if F_LoginPrompt Then
           Begin
             F_LoginDialog.UserName.Text:=F_Username;
             F_LoginDialog.Password.Text:=F_Password;
             if F_LoginDialog.ShowModal=ID_OK Then
                Begin
                  F_Username:=F_LoginDialog.UserName.Text;
                  F_Password:=F_LoginDialog.Password.Text;
                End
             Else
                Begin
                  ShowMessage('If You not enter Username and Password You may not gain access to your data!');
                End;
           End;
        Pwd:=F_Password;
        if (AnsiCompareText(F_DatabaseType,'Access')=0) Then
           Begin
             CoreDatabase := CoreWorkspace.OpenDatabase(F_Database,F_Exclusive,F_ReadOnly,Format(';UID=%s;PWD=%s',[F_Username,Pwd]));
           End
        Else
           Begin
             if (Pwd='') or (F_Username='')  Then
                CoreDatabase := CoreWorkspace.OpenDatabase(F_Database,F_Exclusive,F_ReadOnly,Format('%s',[F_DatabaseType]))
             Else
                CoreDatabase := CoreWorkspace.OpenDatabase(F_Database,F_Exclusive,F_ReadOnly,Format('%s;UID=%s;PWD=%s',[F_DatabaseType,F_Username,Pwd]));
           End;
        if F_QueryTimeout > 0 Then CoreDatabase.QueryTimeout:=F_QueryTimeout;
        RefreshDefinitions;
        F_CollatingOrder:=F_Get_CollatingOrder;
        F_Active:=True;
    End;
End;

Procedure TKADaoDatabase.RefreshDefinitions;
Var
  X: Integer;
Begin
 CoreDatabase.TableDefs.Refresh;
 CoreDatabase.QueryDefs.Refresh;
 F_TableNames.Clear;
 For X:=0 To CoreDatabase.TableDefs.Count-1 do
     Begin
      if CoreDatabase.TableDefs.Item[X].Attributes And dbSystemObject = 0 Then
      F_TableNames.Add(CoreDatabase.TableDefs.Item[X].Name);
     End;
 F_QueryDefNames.Clear;
 For X:=0 To CoreDatabase.QueryDefs.Count-1 do
     Begin
      F_QueryDefNames.Add(CoreDatabase.QueryDefs.Item[X].Name);
     End;
End;

Procedure TKADaoDatabase.Idle;
Begin
 CoreDBEngine.Idle(dbRefreshCache);
End;

Procedure TKADaoDatabase.F_Set_Database(Value : String);
Begin
  if (F_Active) Then
     Begin
       DatabaseError('TKADaoDatabase.F_Set_Database: Cannot set Database while Database is connected!');
       Exit;
     End;
  F_Database:=Value;
End;

Procedure TKADaoDatabase.F_Set_SystemDatabase(Value : String);
Var
 WSName : String;
Begin
  if (F_Active) Then
     Begin
       DatabaseError('TKADaoDatabase.F_Set_SystemDatabase: Cannot set SystemDatabase while Database is connected!');
       Exit;
     End;
  F_SystemDB:=Value;
  if F_SystemDB = '' Then F_SystemDB := F_Get_SystemDatabaseFromRegistry;
  CreateDBEngine(F_DaoVersion);
  CoreDBEngine.SystemDB     := F_SystemDB;
  CoreDBEngine.DefaultType  := F_EngineType;
  CoreWorkspace             := CoreDBEngine.Workspaces.Item[0];
  WSName                    := CoreWorkspace.Name;
  if F_Workspace='' Then
     F_Workspace            := WSName
  Else
     if F_Workspace <> WSName Then F_Set_Workspace(F_Workspace);
End;

Procedure TKADaoDatabase.F_Set_DaoVersion(Value : String);
{$IFDEF DYNADAO}
Var
 WSName : String;
{$ENDIF}
Begin                               
{$IFDEF DYNADAO}
  if (F_Active) Then
     Begin
       DatabaseError('TKADaoDatabase.F_Set_DaoVersion: Cannot set Dao Version while Database is connected!');
       Exit;
     End;
 Try
  F_DaoVersion:=Copy(Value,1,3);
  CreateDBEngine(Value);
  if F_SystemDB = '' Then F_SystemDB := F_Get_SystemDatabaseFromRegistry;
  CoreDBEngine.SystemDB              := F_SystemDB;
  CoreWorkspace:=NULL;
  CoreWorkspace                      := CoreDBEngine.Workspaces.Item[0];
  WSName                             := CoreWorkspace.Name;
  if F_Workspace='' Then
     F_Workspace                     := WSName
  Else
     if F_Workspace <> WSName Then F_Set_Workspace(F_Workspace);
 Except
 End;
{$ELSE}                                                                         
  //This property is read only for fixed DAO
{$ENDIF}
 F_ActualDaoVersion := CoreDBEngine.Version;
End;

Procedure TKADaoDatabase.F_Set_ActualDaoVersion(Value : String);
Begin
  //This property is read only
End;



Function TKADaoDatabase.F_Get_SystemDatabaseFromRegistry:String;
Var
  RS   : String;
  oReg : TRegistry;
Begin
  Result:='';
  RS:='3.5';
  if F_DaoVersion='3.5' Then RS:='3.5';
  if F_DaoVersion='3.6' Then RS:='4.0';
  oReg := TRegistry.Create;
  Try
    oReg.RootKey := HKEY_LOCAL_MACHINE;
    if oReg.OpenKey(Format('SOFTWARE\Microsoft\JET\%s\Engines',[RS]),False) then
       Begin
         Result:=oReg.ReadString('SystemDB');
       End;
  Finally
    oReg.Free;
  End;
End;

Function TKADaoDatabase.F_FindWorkspace(WS:String):Boolean;
Var
  X : Integer;
Begin
  Result := False;
  For X :=0 to CoreDBEngine.Workspaces.Count-1 do
      Begin
       if CoreDBEngine.Workspaces.Item[X].Name=WS Then
          Begin
            Result := True;
            Exit;
          End;
      End;
End;

Procedure TKADaoDatabase.F_Set_Workspace(Value : String);
Begin
  if (F_Active) Then
     Begin
       DatabaseError('TKADaoDatabase.F_Set_Workspace: Cannot set Database while Database is connected!');
       Exit;
     End;
  if F_FindWorkspace(Value) Then
     Begin
       CoreWorkspace := CoreDBEngine.Workspaces.Item[Value];
     End
   Else
     Begin
      Try
       CoreWorkspace:=CoreDBEngine.CreateWorkspace(Value,F_Username,F_Password,F_EngineType);
       CoreDBEngine.Workspaces.Append(CoreWorkspace);
      Except
       DatabaseError('TKADaoDatabase.F_Set_Workspace: Cannot create workspace! You may have no rights to create workspaces!');
      End;
     End;
 CoreWorkspace := CoreDBEngine.Workspaces.Item[Value];
 F_Workspace:=Value;
End;


Procedure TKADaoDatabase.F_Set_DatabaseType(Value : String);
Begin
  if (F_Active) Then
     Begin
       DatabaseError('TKADaoDatabase.F_Set_DatabaseType: Cannot set Database Type while Database is connected!');
       Exit;
     End;
  F_Database:='';   
  F_DatabaseType:=Value;
End;

Function TKADaoDatabase.F_Get_CollatingOrder:String;
Var
  CO : Integer;
Begin
  Result := '';
  DatabaseLanguageInt:=0;
  if Not F_Active Then Exit;
  CO := CoreDatabase.CollatingOrder;
  DatabaseLanguageInt:=CO;
  Case CO of
     dbSortGeneral	        : Result := 'General (English, French, German, Portuguese, Italian, and Modern Spanish)';
     dbSortArabic	        : Result := 'Arabic';
     dbSortChineseSimplified	: Result := 'Simplified Chinese';
     dbSortChineseTraditional	: Result := 'Traditional Chinese';
     dbSortCyrillic	        : Result := 'Bulgarian or Russian';
     dbSortCzech	        : Result := 'Czech';
     dbSortDutch	        : Result := 'Dutch';
     dbSortGreek	        : Result := 'Greek';
     dbSortHebrew	        : Result := 'Hebrew';
     dbSortHungarian	        : Result := 'Hungarian';
     dbSortIcelandic	        : Result := 'Icelandic';
     dbSortJapanese	        : Result := 'Japanese';
     dbSortKorean	        : Result := 'Korean';
     dbSortNeutral	        : Result := 'Neutral';
     dbSortNorwDan	        : Result := 'Norwegian or Danish';
     dbSortPolish	        : Result := 'Polish';
     dbSortSlovenian	        : Result := 'Slovenian';
     dbSortSpanish	        : Result := 'Spanish';
     dbSortSwedFin	        : Result := 'Swedish or Finnish';
     dbSortThai	                : Result := 'Thai';
     dbSortTurkish	        : Result := 'Turkish';
     dbSortUndefined	        : Result := 'Undefined or unknown';
  End;
  F_CollatingOrder:=Result;
End;


Procedure TKADaoDatabase.F_Set_EngineType(Value : Integer);
Var
  WSName : String;
Begin
  if (F_Active) Then
     Begin
       DatabaseError('TKADaoDatabase.F_Set_EngineType: Cannot set EngineType Type while Database is connected!');
       Exit;
     End;
  F_EngineType:=Value;
  CoreDBEngine.DefaultType := F_EngineType;
  CoreWorkspace             := CoreDBEngine.Workspaces.Item[0];
  WSName                    := CoreWorkspace.Name;
  if F_Workspace='' Then
     F_Workspace            := WSName
  Else
     if F_Workspace <> WSName Then F_Set_Workspace(F_Workspace);
End;

Function  TKADaoDatabase.F_Get_DatabaseType:String;
Begin
  Result:=F_DatabaseType;
End;

Procedure TKADaoDatabase.F_Set_ReadOnly(Value : Boolean);
Begin
 if (F_Active) Then
     Begin
       DatabaseError('TKADaoDatabase.F_Set_ReadOnly: Cannot change mode while Database is connected!');
       Exit;
     End;
 F_ReadOnly:=Value;
End;

Procedure TKADaoDatabase.F_Set_Exclusive(Value : Boolean);
Begin
 if (F_Active) Then
     Begin
       DatabaseError('TKADaoDatabase.F_Set_Exclusive: Cannot change exclusive mode while Database is connected!');
       Exit;
     End;
 F_Exclusive:=Value;
End;

Procedure TKADaoDatabase.F_Set_LoginPrompt(Value : Boolean);
Begin
 if (F_Active) Then
     Begin
       DatabaseError('TKADaoDatabase.F_Set_LoginPrompt: Cannot change LoginPrompt while Database is connected!');
       Exit;
     End;
 F_LoginPrompt:=Value;
End;

Procedure TKADaoDatabase.F_Set_UserName(Value : String);
Begin
 if (F_Active) Then
     Begin
       DatabaseError('TKADaoDatabase.F_Set_UserName: Cannot change User Name while Database is connected!');
       Exit;
     End;
 F_UserName:=Value;
End;

Procedure TKADaoDatabase.F_Set_Password(Value : String);
Begin
 if (F_Active) Then
     Begin
       DatabaseError(' TKADaoDatabase.F_Set_Password: Cannot change Password while Database is connected!');
       Exit;
     End;
 F_Password:=Value;
End;

Procedure TKADaoDatabase.StartTransaction;
Begin
  if (NOT F_Active) Then
     Begin
       DatabaseError('TKADaoDatabase.BeginTrans: Cannot start transactions while Database is NOT connected!');
       Exit;
     End;
  if CoreDatabase.Transactions Then CoreDatabase.BeginTrans;
End;

Procedure TKADaoDatabase.Commit;
Begin
 if (NOT F_Active) Then
     Begin
       DatabaseError('TKADaoDatabase.CommitTrans: Cannot commit transactions while Database is NOT connected!');
       Exit;
     End;
 if CoreDatabase.Transactions Then CoreDatabase.CommitTrans(dbForceOSFlush);
End;

Procedure TKADaoDatabase.Rollback;
Begin
 if (NOT F_Active) Then
     Begin
       DatabaseError('TKADaoDatabase.Rollback: Cannot rollback transactions while Database is NOT connected!');
       Exit;
     End;
 if CoreDatabase.Transactions Then CoreDatabase.Rollback;
End;

//********************************************** WORKS ONLY ON DAO 3.5X
//                                              ON DAO 3.6 USE COMPACT DATABASE
//                                              WICH ALSO DOES REPAIR
//******************************************************************************
Procedure TKADaoDatabase.RepairAccessDatabase(DatabaseName:String);
Begin
  if F_DaoVersion='3.5' Then
     CoreDBEngine.RepairDatabase(DatabaseName)
  Else
     CompactAccessDatabase(DatabaseName);
End;

Procedure  TKADaoDatabase.CompactAccessDatabase(DatabaseName:String);
Var
  TempName : Array[0..1000] of Char;
  TempPath : String;
  Name     : String;
Begin
  TempPath:=ExtractFilePath(DatabaseName);
  GetTempFileName(PChar(TempPath),'mdb',0,TempName);
  Name:=StrPas(TempName);
  DeleteFile(Name);
  OleVariant(CoreDBEngine).CompactDatabase(DatabaseName,Name);
  DeleteFile(DatabaseName);
  RenameFile(Name,DatabaseName);
End;

Procedure  TKADaoDatabase.CompactAccessDatabaseEx(DatabaseName: String;
                                                  NewLocale   : String;
                                                  Encrypt     : Boolean;
                                                  Decrypt     : Boolean;
                                                  NewVersion  : Integer;
                                                  Password    : String);
Var
  TempName : Array[0..1000] of Char;
  TempPath : String;
  Name     : String;
  Options  : Integer;
Begin
  TempPath:=ExtractFilePath(DatabaseName);
  GetTempFileName(PChar(TempPath),'mdb',0,TempName);
  Name:=StrPas(TempName);
  DeleteFile(Name);
  Options:=0;
  if Encrypt Then Options := dbEncrypt;
  if Decrypt Then Options := dbDecrypt;
  if NewVersion <> 0 Then Options:=Options+NewVersion;
  if Password <> '' Then Password:=';pwd='+Password;
  CoreDBEngine.CompactDatabase(DatabaseName,Name,NewLocale,Options,Password);
  DeleteFile(DatabaseName);
  RenameFile(Name,DatabaseName);
End;

Procedure TKADaoDatabase.CreateAccessDatabase(DatabaseName:String);
Var
 CreateOptions : String;
Begin
 CreateOptions:=Format(dbLangGeneral,['0x0402','1251','0']);
 {$IFDEF DAO35}
 CoreWorkspace.CreateDatabase(DatabaseName,CreateOptions, dbVersion30);
 {$ENDIF}
 {$IFDEF DAO36}
 CoreWorkspace.CreateDatabase(DatabaseName,CreateOptions, dbVersion40);
 {$ENDIF}
 {$IFDEF DYNADAO}
 if F_DaoVersion='3.5' then CoreWorkspace.CreateDatabase(DatabaseName,CreateOptions, dbVersion30);
 if F_DaoVersion='3.6' then CoreWorkspace.CreateDatabase(DatabaseName,CreateOptions, dbVersion40);
 {$ENDIF}
End;

Procedure TKADaoDatabase.CreateAccessDatabaseEx(DatabaseName,LANGID,CP,COUNTRY,Password,Version:String;Encrypt:Boolean);
Var
 CreateOptions:String;
Begin
 CreateOptions:=Format(dbLangGeneral,[LANGID,CP,COUNTRY]);
 if Password <> '' Then CreateOptions:=CreateOptions+';PWD='+Password;
 {$IFDEF DAO35}
 if Encrypt Then
    CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, dbVersion30 AND dbEncrypt)
 Else
    CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, dbVersion30);
 {$ENDIF}
 {$IFDEF DAO36}
  if Version='30' Then
     if Encrypt Then
        CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, dbVersion30 AND dbEncrypt)
     Else
        CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, dbVersion30)
  Else
     if Encrypt Then
        CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, dbVersion40 AND dbEncrypt)
     Else
        CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, dbVersion40);
 {$ENDIF}
 {$IFDEF DYNADAO}
 if F_DaoVersion='3.5'  Then
 if Encrypt Then
    CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, dbVersion30 AND dbEncrypt)
 Else
    CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, dbVersion30);
 //*****************************************************************************
  if F_DaoVersion='3.6'  Then
  if Version='30' Then
     if Encrypt Then
        CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, dbVersion30 AND dbEncrypt)
     Else
        CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, dbVersion30)
  Else
     if Encrypt Then
        CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, dbVersion40 AND dbEncrypt)
     Else
        CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, dbVersion40);
 {$ENDIF}
End;

Procedure TKADaoDatabase.CreateAccessDatabaseEx2(DatabaseName,Language,Password,Version:String;Encrypt:Boolean);
Var
 CreateOptions:String;
Begin
 CreateOptions:=Language;
 if Password <> '' Then CreateOptions:=CreateOptions+';PWD='+Password;
 {$IFDEF DAO35}
 if Encrypt Then
    CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, dbVersion30 AND dbEncrypt)
 Else
    CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, dbVersion30);
 {$ENDIF}
 {$IFDEF DAO36}
  if Version='30' Then
     if Encrypt Then
        CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, dbVersion30 AND dbEncrypt)
     Else
        CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, dbVersion30)
  Else
     if Encrypt Then
        CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, dbVersion40 AND dbEncrypt)
     Else
        CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, dbVersion40);
 {$ENDIF}
 {$IFDEF DYNADAO}
 if F_DaoVersion='3.5'  Then
 if Encrypt Then
    CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, dbVersion30 AND dbEncrypt)
 Else
    CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, dbVersion30);
 //*****************************************************************************
  if F_DaoVersion='3.6'  Then
  if Version='30' Then
     if Encrypt Then
        CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, dbVersion30 AND dbEncrypt)
     Else
        CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, dbVersion30)
  Else
     if Encrypt Then
        CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, dbVersion40 AND dbEncrypt)
     Else
        CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, dbVersion40);
 {$ENDIF}
End;

Procedure TKADaoDatabase.RenameTable(OldTableName,NewTableName:String);
Begin
 RefreshDefinitions;
 CoreDatabase.TableDefs.Item[OldTableName].Name:=NewTableName;
 RefreshDefinitions;
End;

Procedure TKADaoDatabase.DeleteTable(TableName:String);
Begin
 RefreshDefinitions;
 CoreDatabase.TableDefs.Delete(TableName);
 RefreshDefinitions;
End;

//******************************************************************************
//  1 = Primary index
//  2 = Unique
//  4 = NormalIndex
//******************************************************************************


Function TKADaoDatabase.CreateIndex(TableName,FieldName:String;IndexType:Integer):Boolean;
Var
  NewTable : OleVariant;
  NewField : OleVariant;
  NewIndex : OleVariant;
  PrimIndex: OleVariant;
Begin
  Result:=False;
  RefreshDefinitions;
  Try
   NewTable  := CoreDatabase.TableDefs.Item[TableName];
   NewIndex  := NewTable.CreateIndex(FieldName);
   if ((IndexType And 1) = 0) Then
      NewIndex.Primary := False
   Else
      Begin
        PrimIndex          := NewTable.CreateIndex('PrimaryKey');
        PrimIndex.Primary  := True;
        PrimIndex.Unique   := True;
        NewField           := NewTable.CreateField(FieldName);
        PrimIndex.Fields.AppEnd(NewField);
        NewIndex.Primary  := True;
        NewTable.Indexes.AppEnd(PrimIndex);
        Exit;
      End;
   if ((IndexType And 2) = 0) Then NewIndex.Unique  := False  Else  NewIndex.Unique  := True;
   NewField := NewTable.CreateField(FieldName);
   NewIndex.Fields.AppEnd(NewField);
   NewTable.Indexes.AppEnd(NewIndex);
  Except
   Exit;
  End;
  RefreshDefinitions;
  Result:=True;
End;

Procedure TKADaoDatabase.RenameIndex(TableName,OldIndexName,NewIndexName:String);
Begin
  RefreshDefinitions;
  CoreDatabase.TableDefs.Item[TableName].Indexes.Item[OldIndexName].Name:=NewIndexName;
  RefreshDefinitions;
End;

Procedure TKADaoDatabase.DeleteIndexByName(TableName,IndexName:String);
Begin
 RefreshDefinitions;
 CoreDatabase.TableDefs.Item[TableName].Indexes.Delete(IndexName);
 RefreshDefinitions;
End;

Procedure TKADaoDatabase.DeleteIndexByFieldName(TableName,FieldName:String);
Var
 X         : Integer;
 TmpName   : String;
 IndexName : String;
 NotFound  : Boolean;
Begin
 RefreshDefinitions;
 Try
  Repeat
   NotFound:=True;
   CoreDatabase.TableDefs.Refresh;
   For X:=0 To CoreDatabase.TableDefs.Item[TableName].Indexes.Count-1 do
       Begin
         TmpName:=CoreDatabase.TableDefs.Item[TableName].Indexes.Item[X].Fields.Item[0].Name;
         if TmpName=FieldName Then
            Begin
              IndexName:=CoreDatabase.TableDefs.Item[TableName].Indexes.Item[X].Name;
              DeleteIndexByName(TableName,IndexName);
              NotFound:=False;
              Break;
            End;
       End;
  Until NotFound;
 Except
 End;
 RefreshDefinitions;
End;



Procedure TKADaoDatabase.DeleteField(TableName,FieldName:String);
Var
 X         : Integer;
 TmpName   : String;
 IndexName : String;
 NotFound  : Boolean;
Begin
 RefreshDefinitions;
 Try
  Repeat
   NotFound:=True;
   CoreDatabase.TableDefs.Refresh;
   For X:=0 To CoreDatabase.TableDefs.Item[TableName].Indexes.Count-1 do
       Begin
         TmpName:=CoreDatabase.TableDefs.Item[TableName].Indexes.Item[X].Fields.Item[0].Name;
         if TmpName=FieldName Then
            Begin
              IndexName:=CoreDatabase.TableDefs.Item[TableName].Indexes.Item[X].Name;
              DeleteIndexByName(TableName,IndexName);
              NotFound:=False;
              Break;
            End;
       End;
  Until NotFound;
 Except
 End;
 CoreDatabase.TableDefs.Item[TableName].Fields.Delete(FieldName);
 RefreshDefinitions;
End;

Procedure TKADaoDatabase.RenameField(TableName,OldFieldName,NewFieldName:String);
Begin
  RefreshDefinitions;
  CoreDatabase.TableDefs.Item[TableName].Fields.Item[OldFieldName].Name:=NewFieldName;
  RefreshDefinitions;
End;


//******************************************************************************
//  1 = Primary index
//  2 = Unique
//  4 = NormalIndex
//******************************************************************************


Function TKADaoDatabase.CreateTable(TableName:String; FieldNames : Variant; FieldTypes : Variant; FieldSizes : Variant; FieldIndexes:Variant):Boolean;
Var
  NewTable : OleVariant;
  NewField : OleVariant;
  NewIndex : OleVariant;
  PrimIndex: OleVariant;
  Primary  : Boolean;
  X        : Integer;
  Count    : Integer;
  AutoInc  : Boolean;
  IndexName: String;
Begin
if (NOT F_Active) Then
     Begin
       DatabaseError('TKADaoDatabase.CreateTable: Cannot create table while Database is NOT connected!');
       CreateTable:=False;
       Exit;
     End;
 Primary := False;
 NewTable:=OleVariant(CoreDatabase).CreateTableDef(TableName);
 Count:=VarArrayHighBound(FieldTypes,VarArrayDimCount(FieldTypes));
 For X:=0 to Count do
     Begin
      AutoInc:=False;
      if FieldTypes[X]=dbAutoIncInteger Then
         Begin
           FieldTypes[X]:=dbLong;
           AutoInc:=True;
         End;
      NewField  := NewTable.CreateField(FieldNames[X],FieldTypes[X],FieldSizes[X]);
      NewTable.Fields.AppEnd(NewField);
      if AutoInc Then NewTable.Fields[FieldNames[X]].Attributes:=dbAutoIncrField;
      if FieldIndexes[X] > 0 Then
         Begin
           if ((FieldIndexes[X] And 1) = 1) Then
               Begin
                 if Not Primary Then
                    Begin
                      PrimIndex          := NewTable.CreateIndex('PrimaryKey');
                      PrimIndex.Primary  := True;
                      PrimIndex.Unique   := True;
                      Primary:=True;
                    End;
                 NewField         := NewTable.CreateField(FieldNames[X],FieldTypes[X],FieldSizes[X]);
                 PrimIndex.Fields.AppEnd(NewField);
                 NewIndex         := NewTable.CreateIndex(FieldNames[X]);
                 NewIndex.Unique  := True;
                 NewField         := NewTable.CreateField(FieldNames[X],FieldTypes[X],FieldSizes[X]);
                 NewIndex.Fields.AppEnd(NewField);
                 NewTable.Indexes.AppEnd(NewIndex);
               End
           Else
               Begin
                 IndexName:=FieldNames[X];
                 NewIndex:=NewTable.CreateIndex(IndexName);
                 if ((FieldIndexes[X] And 2) > 0) Then NewIndex.Unique  := True;
                 NewField  := NewTable.CreateField(FieldNames[X],FieldTypes[X],FieldSizes[X]);
                 NewIndex.Fields.AppEnd(NewField);
                 NewTable.Indexes.AppEnd(NewIndex);
               End;
         End;
     End;
 if Primary Then NewTable.Indexes.AppEnd(PrimIndex);
 CoreDatabase.TableDefs.AppEnd(IDispatch(TVarData(NewTable).vDispatch));
 RefreshDefinitions;
 CreateTable:=True;
End;

Procedure TKADaoDatabase.LinkExternalTable(Database,TableName:String);
Begin
//******************************************************************************
// IN THE NEXT VERSION
//******************************************************************************
End;

//******************************************************************************
//  1 = Primary index
//  2 = Unique
//  4 = NormalIndex
//******************************************************************************


Function TKADaoDatabase.AddFieldsToTable(TableName:String; FieldNames : Variant; FieldTypes : Variant; FieldSizes : Variant; FieldIndexes:Variant):Boolean;
Var
  NewTable : OleVariant;
  NewField : OleVariant;
  PrimIndex: OleVariant;
  NewIndex : OleVariant;
  X        : Integer;
  Count    : Integer;
  Primary  : Boolean;
Begin
if (NOT F_Active) Then
     Begin
       DatabaseError('TKADaoDatabase.CreateTable: Cannot create table while Database is NOT connected!');
       AddFieldsToTable:=False;
       Exit;
     End;
 NewTable:=CoreDatabase.TableDefs.Item[TableName];
 Primary := False;
 For X :=0 to NewTable.Indexes.Count-1 do
     Begin
       if AnsiCompareText(NewTable.Indexes.Item[X].Name,'PrimaryKey')=0 Then Primary:=True;
     End;
 if Primary Then NewTable.Indexes.Delete('PrimaryKey');
 Primary := False;
 Count:=VarArrayHighBound(FieldTypes,VarArrayDimCount(FieldTypes));
 For X:=0 to Count do
     Begin
      NewField  := NewTable.CreateField(FieldNames[X],FieldTypes[X],FieldSizes[X]);
      NewTable.Fields.AppEnd(NewField);
      if FieldIndexes[X] > 0 Then
         Begin
           if ((FieldIndexes[X] And 1) = 1) Then
               Begin
                 if Not Primary Then
                    Begin
                      PrimIndex          := NewTable.CreateIndex('PrimaryKey');
                      PrimIndex.Primary  := True;
                      PrimIndex.Unique   := True;
                      NewField  := NewTable.CreateField(FieldNames[X],FieldTypes[X],FieldSizes[X]);
                      PrimIndex.Fields.AppEnd(NewField);
                      Primary:=True;
                    End
                 Else
                    Begin
                      NewTable.Indexes.Refresh;
                      NewField  := NewTable.CreateField(FieldNames[X],FieldTypes[X],FieldSizes[X]);
                      PrimIndex.Fields.AppEnd(NewField);
                    End;
                 NewIndex:=NewTable.CreateIndex(FieldNames[X]);
                 NewIndex.Unique  := True;
                 NewField         := NewTable.CreateField(FieldNames[X],FieldTypes[X],FieldSizes[X]);
                 NewIndex.Fields.AppEnd(NewField);
                 NewTable.Indexes.AppEnd(NewIndex);
               End
           Else
               Begin
                 NewIndex:=NewTable.CreateIndex(FieldNames[X]);
                 if ((FieldIndexes[X] And 2) > 0) Then NewIndex.Unique  := True;
                 NewField  := NewTable.CreateField(FieldNames[X],FieldTypes[X],FieldSizes[X]);
                 NewIndex.Fields.AppEnd(NewField);
                 NewTable.Indexes.AppEnd(NewIndex);
               End;
         End;
     End;
 if Primary Then NewTable.Indexes.AppEnd(PrimIndex);
 RefreshDefinitions;
 AddFieldsToTable:=True;
End;

Function TKADaoDatabase.CreateQueryDef(Name:String;SQL:String):Boolean;
Var
 {$IFDEF DYNADAO}
 Query : OleVariant;
 {$ELSE}
 Query : QueryDef;
 {$ENDIF}
Begin
 Query:=CoreDatabase.CreateQueryDef(Name,SQL);
 RefreshDefinitions;
 CreateQueryDef:=True;
End;

Function TKADaoDatabase.GetQueryDefSQLText(Name:String):String;
Begin
 Try
   Result:=CoreDatabase.QueryDefs.Item[Name].SQL;
 Except
   Result:='';
 End;
End;

Procedure TKADaoDatabase.RenameQueryDef(OldQueryName,NewQueryName:String);
Begin
 RefreshDefinitions;
 CoreDatabase.QueryDefs.Item[OldQueryName].Name:=NewQueryName;
 RefreshDefinitions;
End;

Procedure TKADaoDatabase.DeleteQueryDef(QueryName:String);
Begin
 RefreshDefinitions;
 CoreDatabase.QueryDefs.Delete(QueryName);
 RefreshDefinitions;
End;

//************************************************************************************************* EDITORS
function GetExeDir: String;
begin
     Result := ExtractFilePath(ParamStr(0));
end;

function GetWorkDir: String;
var
   Tmp : String;
begin
     GetDir(0, Tmp);
end;


//*********************************************************************************** Database TYPE
Procedure TDatabaseTypeEditor.GetValues( Proc: TGetStrProc);
Var
  DBase : TKADaoDatabase;
  X     : Integer;
Begin
  if GetComponent(0) is TKADaoDatabase then
  Begin
    DBase := TKADaoDatabase(GetComponent(0));
    Try
      For X := 0 to DBase.F_DBTypesList.Count-1 do Proc(DBase.F_DBTypesList[X]);
    Finally
    End;
  End;
End;

Procedure TDatabaseTypeEditor.SetValue(const Value: string);
Begin
 if GetComponent(0) is TKADaoDatabase then
  Begin
    inherited SetValue(Value);
    Modified;
  End;
End;

Function TDatabaseTypeEditor.GetAttributes: TPropertyAttributes;
Begin
  Result:= Inherited GetAttributes + [paValueList, paSortList];
End;


//*********************************************************************************** Database
Function TDatabaseNameEditor.GetAttributes: TPropertyAttributes;
Begin
  Result:= [paDialog];
End;

Procedure TDatabaseNameEditor.SetValue(const Value: string);
Begin
if GetComponent(0) is TKADaoDatabase then
  Begin
    inherited SetValue(Value);
    Modified;
  End;
End;

procedure TDatabaseNameEditor.Edit;
var
   FileName : String;
   DBase    : TKADaoDatabase;
   Filter   : String;
   Temp     : String;
   P        : Integer;
   TableType: String;
Begin
     DBase:=TKADaoDatabase(GetComponent(0));
     If DBase.F_DatabaseType='' Then DatabaseError('Please select database type!');
     DlgChooseDatabase := TOpenDialog.Create(Nil);
     FileName := DBase.Database;
     if FileName = '' then
        Begin
               DlgChooseDatabase.FileName   := '';
               if csDesigning in DBase.ComponentState Then
                  DlgChooseDatabase.InitialDir := GetWorkDir
               Else
                  DlgChooseDatabase.InitialDir := GetExeDir;
        End
     Else
        Begin
               DlgChooseDatabase.FileName   := ExtractFileName(FileName);
               DlgChooseDatabase.InitialDir := ExtractFileDir(FileName);
        End;
     if DBase.F_DatabaseType='Access' Then
        Begin
          Filter:='Microsoft Access (*.mdb)|*.mdb';
          Filter:=Filter+'|All files (*.*)|*.*';
          DlgChooseDatabase.Title:='Choose '+DBase.F_DatabaseType+' Database:';
          DlgChooseDatabase.Options:=[ofPathMustExist,ofHideReadOnly];
          DlgChooseDatabase.Filter :=Filter;
          if DlgChooseDatabase.Execute then SetStrValue(DlgChooseDatabase.FileName);
        End
     Else
        Begin
          Filter:=DBase.F_Get_DBTypeFileExtension(DBase.F_DatabaseType);
          TableType:=DBase.F_Get_DBTypeTableType(DBase.F_DatabaseType);
          if TableType='1' Then
             Begin
               if SelectDirectory(FileName,[],0) Then SetStrValue(FileName);
             End
          Else
             Begin
              Temp:=Filter;
              P:=Pos('(',Temp);
              if P > 0 Then
                 Begin
                  Delete(Temp,1,P);
                  P:=Pos(')',Temp);                          
                  if P > 0 Then Temp:=Copy(Temp,1,P-1);
                  Filter:=Filter+'|'+Temp;
                 End;
              Filter:=Filter+'|All files (*.*)|*.*';
              DlgChooseDatabase.Title:='Choose '+DBase.F_DatabaseType+' Database:';
              DlgChooseDatabase.Options:=[ofFileMustExist,ofPathMustExist,ofHideReadOnly];
              DlgChooseDatabase.Filter :=Filter;
              if DlgChooseDatabase.Execute then SetStrValue(DlgChooseDatabase.FileName);
             End;
        End;
     DlgChooseDatabase.Free;
     Modified;
End;

//*********************************************************************************** System Database
Function TSystemDatabaseNameEditor.GetAttributes: TPropertyAttributes;
Begin
  Result:= [paDialog];
End;

Procedure TSystemDatabaseNameEditor.SetValue(const Value: string);
Begin
if GetComponent(0) is TKADaoDatabase then
  Begin
    inherited SetValue(Value);
    Modified;
  End;
End;

procedure TSystemDatabaseNameEditor.Edit;
var
   DBase    : TKADaoDatabase;
   Filter   : String;
Begin
   DBase:=TKADaoDatabase(GetComponent(0));
   DlgChooseDatabase := TOpenDialog.Create(Nil);
   DlgChooseDatabase.InitialDir:=ExtractFilePath(DBase.F_SystemDB);
   Filter:='Microsoft Access security files (*.mda *.mdw)|*.mda;*.mdw';
   Filter:=Filter+'|Microsoft Access (*.mdb)|*.mdb';
   Filter:=Filter+'|All files (*.*)|*.*';
   DlgChooseDatabase.Title:='Choose System Database:';
   DlgChooseDatabase.Options:=[ofFileMustExist,ofPathMustExist,ofHideReadOnly];
   DlgChooseDatabase.Filter :=Filter;
   if DlgChooseDatabase.Execute then SetStrValue(DlgChooseDatabase.FileName);
   DlgChooseDatabase.Free;
   Modified;
End;


//*********************************************************************************** Engine Type
Function TEngineTypeEditor.GetAttributes: TPropertyAttributes;
Begin
  Result := Inherited GetAttributes + [paValueList, paSortList];
End;

Function  TEngineTypeEditor.GetValue: string;
Begin
 if GetComponent(0) is TKADaoDatabase then
    Begin
      if TKADaoDatabase(GetComponent(0)).F_EngineType=dbUseODBC then
         Result:='dbUseODBC'
      Else
         Result:='dbUseJet';
    End;
End;

Procedure TEngineTypeEditor.GetValues( Proc: TGetStrProc);
Begin
  if GetComponent(0) is TKADaoDatabase then
     Begin
       Proc('dbUseODBC');
       Proc('dbUseJet');
     End;
End;


procedure TEngineTypeEditor.SetValue(const Value: string);
Var
 Dat : Integer;
Begin
  if GetComponent(0) is TKADaoDatabase then
       Begin
       if Value='dbUseODBC' Then
          Dat:=dbUseODBC
       Else
          Dat:=dbUseJet;
       Inherited SetValue(IntToStr(Dat));
       Modified;
     End;
End;

//*********************************************************************************** Workspase
Procedure TWorkspaceEditor.GetValues( Proc: TGetStrProc);
Var
  DBase : TKADaoDatabase;
  X     : Integer;
Begin
  if GetComponent(0) is TKADaoDatabase then
  Begin
    DBase := TKADaoDatabase(GetComponent(0));
    Try
      For X := 0 to DBase.CoreDBEngine.Workspaces.Count-1 do
          Begin
            Proc(DBase.CoreDBEngine.Workspaces.Item[X].Name);
          End;
    Finally
    End;
  End;
End;

Function TWorkspaceEditor.GetAttributes: TPropertyAttributes;
Begin
  Result:= Inherited GetAttributes + [paValueList, paSortList];
End;

//************************************************************************************************* EDITORS

//*********************************************************************************** DAO Version
{$IFDEF DYNADAO}
Procedure TDaoVersionEditor.GetValues( Proc: TGetStrProc);
Var
  DBase : TKADaoDatabase;
  X     : Integer;
Begin
  if GetComponent(0) is TKADaoDatabase then
  Begin
    DBase := TKADaoDatabase(GetComponent(0));
    Try
      For X := 0 to DBase.F_DaoVersionList.Count-1 do Proc(DBase.F_DaoVersionList[X]);
    Finally
    End;
  End;
End;

Procedure TDaoVersionEditor.SetValue(const Value: string);
Begin
 if GetComponent(0) is TKADaoDatabase then
  Begin
    inherited SetValue(Value);   
    Modified;
  End;
End;

Function TDaoVersionEditor.GetAttributes: TPropertyAttributes;
Begin
  Result:= Inherited GetAttributes + [paValueList, paSortList];
End;
{$ENDIF}

procedure Register;
Begin
  RegisterComponents('KA Dao', [TKADaoDatabase]);
  RegisterPropertyEditor(TypeInfo(String),TKADaoDatabase,'DatabaseType',TDatabaseTypeEditor);
  RegisterPropertyEditor(TypeInfo(String),TKADaoDatabase,'Database',TDatabaseNameEditor);
  RegisterPropertyEditor(TypeInfo(String),TKADaoDatabase,'SystemDatabase',TSystemDatabaseNameEditor);
  RegisterPropertyEditor(TypeInfo(Integer),TKADaoDatabase, 'EngineType', TEngineTypeEditor);
  RegisterPropertyEditor(TypeInfo(String),TKADaoDatabase, 'Workspace', TWorkspaceEditor);
  {$IFDEF DYNADAO}
  RegisterPropertyEditor(TypeInfo(String),TKADaoDatabase, 'Version', TDaoVersionEditor);
  {$ENDIF}
End;

Initialization
 {$IFNDEF VER130}
  TVarData(EmptyParam).VType := varError;
  TVarData(EmptyParam).VError := $80020004;
 {$ENDIF}
End.


