unit KDaoTable;
{$DEFINE DAO36}

//******************************************************************************
//                    Delphi Dao Project Version 1.5
//                 Copyright (c) 2000 by Kiril Antonov
//******************************************************************************

//******************************* CHANGES **************************************
// 28.05.2000 - Fixed a minor bug which raises exception when
//              getting GetQueryDefSQLText
// 28.05.2000 - Added FieldChanged TList - each item corresponds to a field
//              in the record
//              If  Boolean(FieldChanged[X]) is true then when posting data this
//              field is updated
//              This prevents from writing bak entire record to the database -
//              only changed fields are posted.
// 28.05.2000 - Added new property editor for SortedBy property which allows an
//              easy method to define sort order of a Table/Query
//              A new property SortedByText gives low level access to the
//              SortedBy property
// 28.05.2000 - Added new property editor for QueryDefParameters property which
//              allows an easy method to enter parameters to QUERYDEF dao object
//              A new property QueryDefParametersText gives low level access to
//              the QueryDefParameters property
// 30.05.2000 - Fixed a bug in GetRecNo which gives some troubles with DBGrids
//
// 31.05.2000 - Changed InternalSetToRecord method to speedup positioning inside
//              a table
//
// 01.06.2000 - Created Master/Detail Relationship support with property editor
//              similar to Delphi
//              A few new properties are anounced:
//                - MasterSource : TDataSource; - DataSource of the Master Table
//                - MasterFields : TStrings;    - A StringList with
//                  relationships in the form: "DetailField -> MasterField"
//******************************************************************************


interface
uses
DAOApi,
{$IFDEF DAO35}
DAO35Api,
{$ENDIF}
{$IFDEF DAO36}
DAO36Api,
{$ENDIF}
{$IFDEF VER100}
DBTables,
{$ENDIF}
Windows,SysUtils,Classes,Db,DBCommon,KDaoDataBase,DsgnIntf,ActiveX,Forms;

const
        MYBOOKMARKSIZE=8;

Type

TDaoInfo=record
        RecordNo        : Integer;
        RecordData      : TStringList;
        FieldChanged    : TList;
        BookmarkFlag    : TBookmarkFlag;
        BookmarkData    : Integer;
        DaoBookmark     : TSafeArray;
        BookmarkString  : array[0..MYBOOKMARKSIZE] of Char;
end;
PDaoInfo=^TDaoInfo;

TOO    = (
          dbDenyWrite,
          dbDenyRead,
          dbReadOnly,
          dbAppendOnly,
          dbInconsistent,
          dbConsistent,
          dbSQLPassThrough,
          dbFailOnError,
          dbForwardOnly,
          dbSeeChanges,
          dbRunAsync,
          dbExecDirect
          );
TOOSet = Set of TOO;


TTableNameEditor = class(TStringProperty)
    Public
      Procedure GetValues( Proc: TGetStrProc); override;
      Function  GetAttributes: TPropertyAttributes; override;
    End;

TQueryNameEditor = class(TStringProperty)
    Public
      Procedure GetValues( Proc: TGetStrProc); override;
      Function  GetAttributes: TPropertyAttributes; override;
    End;

TIndexNameEditor = class(TStringProperty)
    Public
      Procedure GetValues( Proc: TGetStrProc); override;
      Function  GetAttributes: TPropertyAttributes; override;
    End;

TTableTypeEditor = class(TIntegerProperty)
    Public
     Function  GetValue: string; override;
     Procedure GetValues( Proc: TGetStrProc);override;
     Procedure SetValue(const Value: string); override;
     Function  GetAttributes: TPropertyAttributes; override;
    End;

TLockTypeEditor = class(TIntegerProperty)
    Public
     Function  GetValue: string; override;
     Procedure GetValues( Proc: TGetStrProc);override;
     Procedure SetValue(const Value: string); override;
     Function  GetAttributes: TPropertyAttributes; override;
    End;

TSortByEditor = class(TStringProperty)
    Public
     Procedure Edit;override;
     Procedure SetValue(const Value: string); override;
     Function  GetAttributes: TPropertyAttributes; override;
    End;

TQueryDefParamsEditor = class(TStringProperty)
    Public
     Procedure Edit;override;
     Procedure SetValue(const Value: string); override;
     Function  GetAttributes: TPropertyAttributes; override;
    End;

TMasterFieldsEditor = class(TStringProperty)
    Public
     Procedure Edit;override;
     Procedure SetValue(const Value: string); override;
     Function  GetAttributes: TPropertyAttributes; override;
    End;

TMasterSourcePropertyEditor = class(TPropertyEditor)
    public
      function GetAttributes: TPropertyAttributes; override;
      function  GetValue: String; override;
      procedure GetValues(Proc: TGetStrProc); override;
      procedure SetValue(const Value: String); override;
  end;

TKADaoTable = class;
TKADAODataLink = class(TMasterDataLink)
    private
      F_KADaoTable        : TKADaoTable;
      F_MasterIsChanged   : Boolean;
    protected
      procedure ActiveChanged; override;
      procedure RecordChanged(Field: TField); override;
      {$IFNDEF VER100}
      function  GetDetailDataSet: TDataSet; override;
      {$ENDIF}
      procedure CheckBrowseMode; override;
    public
      constructor Create(DataSet: TKADaoTable);
  end;



TKADaoTable = class(TDataSet)
private
        F_Active        : Boolean;
        F_RecNo         : Integer;
        F_FilterBuffer  : PChar;
        F_BufferSize    : Integer;
        F_StartMyInfo   : Integer;
        F_StartCalc     : Integer;
        F_MDisabled : Boolean;
        Function        GetActiveRecordBuffer:  PChar;
        Function        FilterRecord(Buffer: PChar): Boolean;
protected
        F_Database               : TKADaoDatabase;
        F_ReadOnly               : Boolean;
        F_Filtered               : Boolean;
        F_DaoTable               : Recordset;
        F_SQL                    : TStrings;
        F_SortedBy               : TStrings;
        F_FieldNames             : TStrings;
        F_FieldTypeNames         : TStrings;
        F_MDFieldNames           : TStrings;

        F_QD_ParamNames          : TStrings;
        F_QD_ParamDaoTypes       : TStrings;
        F_QD_ParamBDETypes       : TStrings;

        F_MasterLink             : TKADAODataLink;
        F_MasterFields           : TStrings;

        F_Detail                 : TStrings;
        F_Master                 : TStrings;

        F_TableName              : String;
        F_QueryDefName           : String;
        F_QueryDefParameters     : TStrings;
        F_QueryDefSQLText        : TStrings;
        F_IndexName              : String;
        F_TableType              : Integer;
        F_LockType               : Integer;
        F_OpenOptions            : TOOSet;
        F_RecordSize             : Integer;
        F_OLE_ON                 : Boolean;
        Function        F_Get_Database:TKADaoDatabase;
        Procedure       F_Set_Database(Value:TKADaoDatabase);
        Function        F_Get_TableName:String;
        Procedure       F_Set_TableName(Value:String);
        Function        F_Get_IndexName:String;
        Procedure       F_Set_IndexName(Value:String);
        Procedure       F_Set_TableType(Value:Integer);
        Procedure       F_Set_LockType(Value:Integer);
        Procedure       F_Set_OpenOptions(Value:TOOSet);
        Procedure       F_Set_ReadOnly(Value:Boolean);
        Procedure       F_Set_Filtered(Value:Boolean);
        Procedure       F_Set_Sort(Value:TStrings);

        Procedure       F_Set_SQL(Value:TStrings);
        Procedure       F_Set_QueryDefName(Value:String);
        Procedure       F_Set_QueryDefParameters(Value:TStrings);
        Procedure       F_Set_QueryDefSQLText(Value:TStrings);

        Function        F_Get_MasterSource: TDataSource;
        Procedure       F_Set_MasterSource(Value: TDataSource);
        Procedure       F_ProcessMasterFields(Value:TStrings);
        Procedure       F_Set_MasterFields(Value:TStrings);

        Procedure       F_Set_Master(Value:TStrings);
        Procedure       F_Set_Detail(Value:TStrings);

        Function        F_ComposeSQL(SQL:TStrings):String;


        Function        InternalCalcRecordSize:Integer;
        Function        IntegerToBuffer(Buffer: Pointer; S: String): Boolean;
        Function        FloatToBuffer(Buffer: Pointer; S: String): Boolean;
        Function        DateToBuffer(Buffer: Pointer; S: String): Boolean;
        Function        TimeToBuffer(Buffer: Pointer; S: String): Boolean;
        Function        DateTimeToBuffer(Buffer: Pointer; S: String): Boolean;
        Function        BooleanToBuffer(Buffer: Pointer; S: String): Boolean;

        Function        BufferToDate(Buffer: Pointer): String;
        Function        BufferToTime(Buffer: Pointer): String;
        Function        BufferToDateTime(Buffer: Pointer): String;

        Procedure       OpenDaoRecordset;
        Procedure       CloseDaoRecordset;

        Procedure       InternalOpen; override;
        Procedure       InternalClose; override;
        Function        IsCursorOpen: Boolean; override;
        Function        GetCanModify: Boolean; override;
        Function        GetRecordSize: Word;override;
        Function        AllocRecordBuffer: PChar; override;
        Procedure       FreeRecordBuffer(var Buffer: PChar); override;
        Procedure       InternalFirst;override;
        Procedure       InternalLast;override;
        Procedure       InternalInitFieldDefs; override;
        Procedure       InternalInitRecord(Buffer: PChar); override;
        Procedure       SetFieldData(Field: TField; Buffer: Pointer);override;
        Procedure       ClearDBCalcFields(Buffer: PChar);

        Procedure       InternalEdit; override;
        Procedure       InternalAddRecord(Buffer: Pointer; Append: Boolean); override;
        Procedure       InternalPost; override;
        Procedure       InternalDelete; override;
        Function        GetRecord(Buffer: PChar; GetMode: TGetMode; DoCheck: Boolean): TGetResult; override;
        Procedure       InternalSetToRecord(Buffer: PChar); override;
        Procedure       InternalRefresh; override;

        Procedure       InternalGotoBookmark(Bookmark: Pointer); override;
        Function        GetBookmarkFlag(Buffer: PChar): TBookmarkFlag; override;
        Procedure       SetBookmarkFlag(Buffer: PChar; Value: TBookmarkFlag); override;
        Function        GetBookmarkStr: TBookmarkStr; override;
        Procedure       SetBookmarkStr(const Value: TBookmarkStr); override;
        Procedure       GetBookmarkData(Buffer: PChar; Data: Pointer); override;
        Procedure       SetBookmarkData(Buffer: PChar; Data: Pointer); override;

        Procedure       InternalHandleException; override;
        Procedure       DataEvent(Event: TDataEvent; Info: Longint); override;

        Function        GetRecordCount  : Integer;override;
        Function        GetRecNo        : Integer; override;


        Function        InsertSQLString(MDString: String): String;
        Function        BuildDetailSQL  : String;
        Procedure       MasterDSChanged;

  public
        MainDatabaseShutdown             : Boolean;

        Function                           GetFieldData(Field: TField; Buffer: Pointer): Boolean; override;
        Function                           CreateBlobStream(Field: TField; Mode: TBlobStreamMode): TStream; override;

        Property  MasterLink             : TKADAODataLink read F_MasterLink;
        Property  LikableFields          : TStrings Read F_MDFieldNames;
        Procedure                          ExecSQL(SQL:TStringList);
        Constructor                        Create(AOwner: TComponent); override;
        Destructor                         Destroy; override;
  published
        Property Database                : TKADaoDatabase Read F_Get_Database Write F_Set_Database;
        Property TableName               : String Read F_Get_TableName Write F_Set_TableName;
        Property SortedBy                : TStrings Read F_SortedBy Write F_Set_Sort;
        Property SortedByText            : TStrings Read F_SortedBy Write F_Set_Sort;
        Property QueryDefName            : String Read F_QueryDefName Write F_Set_QueryDefName;
        Property QueryDefParameters      : TStrings Read F_QueryDefParameters Write F_Set_QueryDefParameters;
        Property QueryDefParametersText  : TStrings Read F_QueryDefParameters Write F_Set_QueryDefParameters;
        Property QueryDefSQLText         : TStrings Read F_QueryDefSQLText Write F_Set_QueryDefSQLText;
        Property SQL                     : TStrings Read F_SQL Write F_Set_SQL;
        Property TableType               : Integer Read F_TableType Write F_Set_TableType;
        Property LockType                : Integer Read F_LockType Write F_Set_LockType;
        Property OpenOptions             : TOOSet Read F_OpenOptions Write F_Set_OpenOptions;
        Property IndexName               : String Read F_Get_IndexName Write F_Set_IndexName;
        Property ReadOnly                : Boolean Read F_ReadOnly Write F_Set_ReadOnly;
        Property Filtered                : Boolean Read F_Filtered Write F_Set_Filtered;

        Property MasterSource            : TDataSource Read F_Get_MasterSource Write F_Set_MasterSource;
        Property MasterFields            : TStrings Read F_MasterFields Write F_Set_MasterFields;
        
        Property Filter;
        Property BeforeOpen;
        Property AfterOpen;
        Property BeforeClose;
        Property AfterClose;
        Property BeforeInsert;
        Property AfterInsert;
        Property BeforeEdit;
        Property AfterEdit;
        Property BeforePost;
        Property AfterPost;
        Property BeforeCancel;
        Property AfterCancel;
        Property BeforeDelete;
        Property AfterDelete;
        Property BeforeScroll;
        Property AfterScroll;
        Property OnCalcFields;
        Property OnDeleteError;
        Property OnEditError;
        Property OnFilterRecord;
        Property OnNewRecord;
        Property OnPostError;
        Property Active;
end;

// Handle Memo fields
  TKBlobStream = class(TStream)
  private
    F_Field      : TBlobField;
    F_DataSet    : TKADaoTable;
    F_Buffer     : PChar;
    F_Mode       : TBlobStreamMode;
    F_Opened     : Boolean;
    F_Modified   : Boolean;
    F_Position   : Longint;
    F_BlobData   : TBlobData;
    F_BlobSize   : Integer;
  public
    constructor Create(Field: TBlobField; Mode: TBlobStreamMode);
    destructor Destroy; override;
    function Read(var Buffer; Count: Longint): Longint; override;
    function Write(const Buffer; Count: Longint): Longint; override;
    function Seek(Offset: Longint; Origin: Word): Longint; override;
    procedure Truncate;
  end;



Procedure Register;

implementation
Uses DaoUtils,Dialogs,SortByDialog,QueryDefDialogUnit,MasterDetailFormUnit;

Const
  CRLF=#13+#10;

  //****************************************************************************
constructor TKADaoTable.Create(AOwner: TComponent);
Var
  OLE_INIT:Integer;
begin
  inherited Create(AOwner);
  F_TableName            := '';
  F_TableType            := dbOpenDynaset;
  F_LockType             := dbPessimistic;
  F_OpenOptions          := [dbInconsistent];
  F_ReadOnly             := False;
  F_Filtered             := False;
  MainDatabaseShutdown   := False;
  F_QueryDefName         := '';
  F_QueryDefSQLText      := TStringList.Create;
  F_QueryDefSQLText.Clear;
  F_SQL                  := TStringList.Create;
  F_SQL.Clear;
  F_SortedBy             := TStringList.Create;
  F_SortedBy.Clear;
  F_FieldNames           := TStringList.Create;
  F_FieldNames.Clear;
  F_FieldTypeNames       := TStringList.Create;
  F_FieldTypeNames.Clear;
  F_MDFieldNames         := TStringList.Create;
  F_MDFieldNames.Clear;
  //****************************************************************************
  F_QD_ParamNames        := TStringList.Create;
  F_QD_ParamNames.Clear;
  F_QD_ParamDaoTypes     := TStringList.Create;
  F_QD_ParamDaoTypes.Clear;
  F_QD_ParamBDETypes     := TStringList.Create;
  F_QD_ParamBDETypes.Clear;

  F_QueryDefParameters   := TStringList.Create;
  F_QueryDefParameters.Clear;
  //************************************************************
  F_MDisabled        := False;
  F_MasterFields         := TStringList.Create;
  F_MasterFields.Clear;
  F_MasterLink           := TKADAODataLink.Create(Self);
  F_Detail               := TStringList.Create;
  F_Detail.Clear;
  F_Master               := TStringList.Create;
  F_Master.Clear;
  //************************************************************
  F_Database             := Nil;

  F_OLE_ON:=False;
  OLE_INIT:= CoInitialize(NIL);
  if (OLE_INIT = S_OK) or (OLE_INIT = S_FALSE) then F_OLE_ON:= True
  else DatabaseError('Unable to init OLE objects!');
end;

destructor TKADaoTable.Destroy;
begin
  if F_Active Then
      Begin
        InternalClose;
        F_Active:=False;
      End;
  F_SQL.Free;
  F_SortedBy.Free;
  F_FieldNames.Free;
  F_FieldTypeNames.Free;
  F_MDFieldNames.Free;
  F_QueryDefParameters.Free;
  F_QueryDefSQLText.Free;
  F_QD_ParamNames.Free;
  F_QD_ParamDaoTypes.Free;
  F_QD_ParamBDETypes.Free;
  F_MasterLink.Free;
  F_MasterFields.Free;
  F_Detail.Free;
  F_Master.Free;
  if F_OLE_ON then CoUninitialize;
  inherited Destroy;
end;

Procedure TKADaoTable.ExecSQL(SQL:TStringList);
Begin
 if Assigned(F_Database) Then
    Begin
      OleVariant(F_Database.CoreDatabase).Execute(F_ComposeSQL(SQL));
    End
 Else
    DatabaseError('TKADaoTable.ExecSQL: Cannot execute SQL query while Database is not assigned!');
End;

Function TKADaoTable.F_Get_Database:TKADaoDatabase;
Begin
 if NOT Assigned(F_Database) Then F_Database:=Nil;
 Result:=F_Database;
End;

Procedure TKADaoTable.F_Set_Database(Value:TKADaoDatabase);
Begin
  if Active Then DatabaseError('TKADaoTable.F_Set_Database: Cannot set Database while table is Active!');
  {
  if Value <> Nil Then
     Begin
      F_IndexName:='';
      F_TableName:='';
      F_SQL.Clear;
      F_QueryDefName:='';
      F_QueryDefParameters.Clear;
      F_SortedBy.Clear;
      F_Master.Clear;
      F_Detail.Clear;
      F_MasterFields.Clear;
      F_MasterLink.DataSource:=Nil;
     End;
   }
 F_Database:=Value;
End;

Function TKADaoTable.F_Get_TableName:String;
Begin
 Result:= F_TableName;
End;

Procedure TKADaoTable.F_Set_TableName(Value:String);
Begin
  if Active Then DatabaseError('Cannot set TableName while Table is active!');
  F_TableName:=Value;
  if Value <> '' Then
     Begin
      F_IndexName:='';
      F_SQL.Clear;
      F_QueryDefName:='';
      F_QueryDefParameters.Clear;
      F_SortedBy.Clear;
      F_Master.Clear;
      F_Detail.Clear;
      F_MasterFields.Clear;
      F_MasterLink.DataSource:=Nil;
     End;
End;

Procedure TKADaoTable.F_Set_SQL(Value:TStrings);
Begin
 F_SQL.SetText(Value.GetText);
 if Length(Value.GetText) > 0 Then
    Begin
     F_QueryDefParameters.Clear;
     F_QueryDefName:='';
     F_IndexName:='';;
     F_TableName:='';
    End;
End;

Procedure TKADaoTable.F_Set_QueryDefName(Value:String);
Begin
  Try
    if Assigned(F_Database) And (F_Database.Connected) Then
       Begin
         F_QueryDefSQLText.Clear;
         if Value <> '' Then F_QueryDefSQLText.SetText(PChar(F_Database.GetQueryDefSQLText(Value)));
       End;
  Except
  End;
  F_QueryDefName:=Value;
  if Value <> '' Then
     Begin
      F_IndexName:='';
      F_TableName:='';
      F_SQL.Clear;
      F_QueryDefParameters.Clear;
      F_Master.Clear;
      F_Detail.Clear;
      F_MasterFields.Clear;
      MasterLink.DataSource:=Nil;
     End;
End;



Function TKADaoTable.F_Get_IndexName:String;
Begin
 Result:= F_IndexName;
End;

Procedure TKADaoTable.F_Set_IndexName(Value:String);
Begin
  if Active Then
     Begin
       if TableType=dbOpenTable Then
          Begin
             F_DaoTable.Index:=Value;
             ClearBuffers;
             CheckBrowseMode;
             ActivateBuffers;
             First;
          End
       Else DatabaseError('Cannot set IndexName while Table is active!');
     End;
  F_IndexName:=Value;
End;

Procedure TKADaoTable.F_Set_TableType(Value:Integer);
Begin
  if Active Then DatabaseError('Cannot set TableType while Table is active!');
  F_TableType:=Value;
End;

Procedure TKADaoTable.F_Set_LockType(Value:Integer);
Begin
  if Active Then DatabaseError('Cannot set LockType while Table is active!');
  F_LockType:=Value;
End;

Procedure TKADaoTable.F_Set_OpenOptions(Value:TOOSet);
Begin
  if Active Then DatabaseError('Cannot set OpenOptions while Table is active!');
  F_OpenOptions:=Value;
End;

Procedure TKADaoTable.F_Set_ReadOnly(Value:Boolean);
Begin
  if Active Then DatabaseError('TKADaoTable.F_Set_ReadOnly: Cannot set ReadOnly while table is Active!');
  F_ReadOnly:=Value;
End;

Procedure TKADaoTable.F_Set_Filtered(Value:Boolean);
Begin
 If (F_Active) And (Filter <> '') And (Value) Then
    Begin
      F_DaoTable.Filter:=Filter;
    End
 Else
 If (F_Active) Then
    Begin
       F_DaoTable.Filter:='';
    End;
 F_Filtered:=Value;
End;

Procedure TKADaoTable.F_Set_Sort(Value:TStrings);
Begin
 F_SortedBy.SetText(Value.GetText);
End;

Procedure TKADaoTable.F_Set_QueryDefParameters(Value:TStrings);
Begin
 F_QueryDefParameters.SetText(Value.GetText);
End;

Procedure TKADaoTable.F_Set_QueryDefSQLText(Value:TStrings);
Begin
//*************************** READ ONLY
End;

Function  TKADaoTable.F_Get_MasterSource : TDataSource;
Begin
 Result:= F_MasterLink.DataSource;
End;

Procedure TKADaoTable.F_Set_MasterSource(Value: TDataSource);
Var
 S: String;
Begin
 if Value=Nil Then
    Begin
      F_MasterFields.Clear;
      F_Master.Clear;
      F_Detail.Clear;
      F_MasterLink.DataSource:=Nil;
      Exit;
    End;

 if (StrLen(SQL.GetText)=0) Then
    Begin
      F_MasterLink.DataSource:=Nil;
      F_MasterFields.Clear;
      F_Master.Clear;
      F_Detail.Clear;
      S:='Detail Table must be Query!'+CRLF+
         '(Has SQL property assigned!)';
      if (F_QueryDefName='') And (F_TableName <>'') Then
      S:=S+CRLF+'For your current selection SQL property'+CRLF+
                'must contain the following text:'+CRLF+
                'Select * From ['+F_TableName+'];';
       if (F_QueryDefName='') And (F_TableName <>'') Then
          Begin
            F_SQL.SetText(PChar('Select * From ['+F_TableName+'];'));
            SQL:=F_SQL;
          End
       Else
          Begin
           DatabaseError(S);
           Exit;
          End;  
    End;
 F_MasterLink.DataSource:= Value;
End;

Procedure TKADaoTable.F_ProcessMasterFields(Value:TStrings);
Var
  X,I:Integer;
  S,MasterField,DetailField:String;
Begin
  F_Detail.Clear;
  F_Master.Clear;
  if (Value.Count=1) And (Pos(';',Value.Strings[0]) > 0) Then
     Begin
       S := Value.Strings[0]; 
       Repeat
        I := Pos(';',S);
        if I > 0 Then
           Begin
            DetailField:=Copy(S,1,I-1);
            System.Delete(S,1,I);
           End
        Else
           Begin
            DetailField:=S;
           End;
        if Length(DetailField) > 0 Then
           Begin
            MasterField:=DetailField;
            F_Detail.Add(DetailField);
            F_Master.Add(MasterField);
           End;
       Until I = 0;
     End
  Else
  For X:=0 to Value.Count-1 do
      Begin
        S := Value.Strings[X];
        I := Pos(' -> ',S);
        if I > 0 Then
        Begin
         DetailField:=Copy(S,1,I-1);
         System.Delete(S,1,I+Length(' -> ')-1);
         MasterField:=S;
         F_Detail.Add(DetailField);
         F_Master.Add(MasterField);
        End;
      End;
End;

Procedure TKADaoTable.F_Set_MasterFields(Value:TStrings);

Begin
 F_ProcessMasterFields(Value);
 F_MasterFields.SetText(Value.GetText);
End;

Procedure TKADaoTable.F_Set_Master(Value:TStrings);
Begin
 F_Master.SetText(Value.GetText);
End;

Procedure TKADaoTable.F_Set_Detail(Value:TStrings);
Begin
 F_Detail.SetText(Value.GetText);
End;

Function TKADaoTable.F_ComposeSQL(SQL:TStrings):String;
Var
 X  : Integer;
Begin
 Result:='';
 For X:=0 To SQL.Count-1 do
     Begin
       Result := Result+SQL.Strings[X];
       if X <  SQL.Count-1 Then Result := Result+' ';
     End;
End;

//******************************************************************************

Function TKADaoTable.CreateBlobStream(Field: TField; Mode: TBlobStreamMode): TStream;
Begin
  Result:=TKBlobStream.Create(Field As TBlobField,Mode);
End;

Function TKADaoTable.InternalCalcRecordSize:Integer;
Begin
 F_RecordSize:=0;
 Result:=F_RecordSize;
End;


Procedure TKADaoTable.OpenDaoRecordset;
Var
 X       : Integer;
 TabType : Integer;
 LoType  : Integer;
 Options : Integer;
 TempRS  : Recordset;
 TabN    : String;
 TempSort: String;
begin
        if Not Assigned(F_Database) Then DatabaseError('TKADaoTable.OpenDaoRecordset: Missing Database');
        if (TableName='') And
           (SQL.Count=0)  And
           (QueryDefName='')
        Then DatabaseError('TKADaoTable.OpenDaoRecordset: Missing TableName or QueryDefName or SQL is empty!');
        if F_Database.Connected= False Then DatabaseError('TKADaoTable.OpenDaoRecordset: Database is NOT connected');
        TabType:=F_TableType;
        LoType:=F_LockType;
        Options:=0;

        if dbDenyWrite      in F_OpenOptions Then Options:=Options And DAOApi.dbDenyWrite;
        if dbDenyRead       in F_OpenOptions Then Options:=Options And DAOApi.dbDenyRead;
        if dbReadOnly       in F_OpenOptions Then Options:=Options And DAOApi.dbReadOnly;
        if dbAppendOnly     in F_OpenOptions Then Options:=Options And DAOApi.dbAppendOnly;
        if dbInconsistent   in F_OpenOptions Then Options:=Options And DAOApi.dbInconsistent;
        if dbConsistent     in F_OpenOptions Then Options:=Options And DAOApi.dbConsistent;
        if dbSQLPassThrough in F_OpenOptions Then Options:=Options And DAOApi.dbSQLPassThrough;
        if dbFailOnError    in F_OpenOptions Then Options:=Options And DAOApi.dbFailOnError;
        if dbForwardOnly    in F_OpenOptions Then Options:=Options And DAOApi.dbForwardOnly;
        if dbSeeChanges     in F_OpenOptions Then Options:=Options And DAOApi.dbSeeChanges;
        if dbRunAsync       in F_OpenOptions Then Options:=Options And DAOApi.dbRunAsync;
        if dbExecDirect     in F_OpenOptions Then Options:=Options And DAOApi.dbExecDirect;

        TabN:=TableName;
        if F_SQL.Count > 0 Then
           Begin
             if F_MasterLink.F_MasterIsChanged Then
                Begin
                  F_MasterLink.F_MasterIsChanged:=False;
                  F_ProcessMasterFields(F_MasterFields);
                  TabN:=BuildDetailSQL;
                  TabN:=InsertSQLString(TabN);
                End
             Else
             TabN:=F_ComposeSQL(F_SQL);
           End;
        if F_QueryDefName <> '' Then
           Begin
             TabN:=F_QueryDefName;
             if Database.CoreDatabase.QueryDefs.Item[TabN].Parameters.Count <> F_QueryDefParameters.Count Then
                DatabaseError('TKADaoTable.OpenDaoRecordset: Different number of QueryDef Parameters!'+CRLF+
                'Expected: '+IntToStr(Database.CoreDatabase.QueryDefs.Item[TabN].Parameters.Count));
             For X:=0 To F_QueryDefParameters.Count-1 do
                 Begin
                   Database.CoreDatabase.QueryDefs.Item[TabN].Parameters.Item[X].Value:=F_QueryDefParameters.Strings[X];
                 End;
           End;
        if (F_QueryDefName <> '') And (Database.CoreDatabase.QueryDefs.Item[TabN].Parameters.Count > 0) Then
            Begin
                F_QueryDefSQLText.Clear;
                if Assigned(F_Database) And (F_Database.Connected) Then
                   F_QueryDefSQLText.SetText(PChar(F_Database.GetQueryDefSQLText(TabN)));
                F_DaoTable:=Database.CoreDatabase.QueryDefs.Item[TabN].OpenRecordset(TabType,Options,LoType);
            End
        Else
            Begin
               F_QueryDefSQLText.Clear;
               if (F_QueryDefName <> '') And Assigned(F_Database) And (F_Database.Connected) Then
                   F_QueryDefSQLText.SetText(PChar(F_Database.GetQueryDefSQLText(TabN)));
               F_DaoTable:=Database.CoreDatabase.OpenRecordset(TabN,TabType,Options,LoType);
            End;
        if F_IndexName <> '' Then F_DaoTable.Index:=F_IndexName
        Else
        if F_SortedBy.Count > 0 Then
           Begin
            TempSort:='';
            For X:=0 To F_SortedBy.Count-1 Do
                Begin
                  TempSort:=TempSort+F_SortedBy.Strings[X];
                  if X < F_SortedBy.Count-1 Then TempSort:=TempSort+',';
                  TempSort:=TempSort+' ';
                End;
            F_DaoTable.Sort:=TempSort;
            TempRS:=F_DaoTable;
            F_DaoTable:=TempRS.OpenRecordset(TabType,Options);
            TempRS.Close;
           End;
End;


Procedure TKADaoTable.InternalOpen;
Var
   X       : Integer;
   TempMD  : Boolean;
Begin
        if CSDesigning in ComponentState Then F_MasterLink.F_MasterIsChanged:=True;
        OpenDaoRecordset;
        if Self.Name='' Then Self.Name:='KADaoTable_'+IntToStr(GetTickCount);
        F_Database.ActiveTableNames.AddObject(Self.Name,Self);
        InternalInitFieldDefs;
        if DefaultFields then CreateFields;
        BindFields(True);
        F_RecNo:=-1;
        BookmarkSize:=MYBOOKMARKSIZE;
        F_StartMyInfo:=InternalCalcRecordSize;
        F_StartCalc:=F_StartMyInfo+SizeOf(TDaoInfo);
        F_BufferSize:=F_RecordSize+Sizeof(TDaoInfo)+CalcFieldsSize;
        //****************************************************************
        TempMD:=F_MDisabled;
        F_MDisabled:= True;
        F_FieldNames.Clear;
        F_MDFieldNames.Clear;
        For X:=0 to FieldDefs.Count-1 do
            Begin
             if NOT (Fields[X].IsBlob) Then
                Begin
                  F_FieldNames.Add(FieldDefs.Items[X].Name);
                  F_FieldTypeNames.Add(GetBDEFieldTypeNames(FieldDefs.Items[X].DataType));
                End;
             if NOT (Fields[X].DataType=ftBlob) Then
                Begin
                  F_MDFieldNames.Add(FieldDefs.Items[X].Name);
                End;
            End;
        F_MDisabled:=TempMD;
        F_Active:=True;
        InternalFirst;
        //****************************************************************
end;

Procedure TKADaoTable.CloseDaoRecordset;
Begin
 F_DaoTable.Close;
End;

Procedure TKADaoTable.InternalClose;
begin
        if Not F_Active Then Exit;
        BindFields(False);
        if DefaultFields then DestroyFields;
        CloseDaoRecordset;
        F_Active:=False;
        if NOT MainDatabaseShutdown Then
           F_Database.ActiveTableNames.Delete(F_Database.ActiveTableNames.IndexOf(Self.Name))
        Else
           MainDatabaseShutdown:=False;
end;

Procedure TKADaoTable.InternalFirst;
begin
    Try
      F_DaoTable.MoveFirst;
      F_DaoTable.MovePrevious;
    Except
    End;
    F_RecNo:=-1;
end;

Procedure TKADaoTable.InternalLast;
begin
     Try
        OleVariant(F_DaoTable).MoveLast;
        F_DaoTable.MoveNext;
        F_RecNo:=F_DaoTable.RecordCount;
     Except
        F_RecNo:=-1;
     End;
end;
                                         
Procedure TKADaoTable.InternalSetToRecord(Buffer: PChar);
Var
  RN     : Integer;
  Delta  : Integer;                               
begin
        RN:=F_RecNo;
        F_RecNo:=PDaoInfo(Buffer+F_StartMyInfo)^.RecordNo;
        Delta:=F_RecNo-RN;
        Try
           //***************************** A slowest method - disabled
           //***************************** F_DaoTable.MoveFirst;
           //***************************** F_DaoTable.Move(F_RecNo,NULL);
           if Delta <> 0 Then F_DaoTable.Move(Delta,NULL);
        Except
          //***************************** HANDLE EXCEPTION WHEN A NEW RECORD IS ADDED OR RECORDSET IS EMPTY
        End;
end;

Procedure TKADaoTable.InternalRefresh;
begin
     Resync([rmExact, rmCenter]);
end;


//*********************************************************************************** BOOKMARK FUNCTIONS
procedure TKADaoTable.InternalGotoBookmark(Bookmark: Pointer);
begin
     if (F_Active) And F_DaoTable.Bookmarkable Then F_DaoTable.Bookmark.pvData := PInteger(Bookmark);
end;


Function TKADaoTable.GetBookmarkFlag(Buffer: PChar): TBookmarkFlag;
begin
        Result:=PDaoInfo(Buffer+F_StartMyInfo).BookmarkFlag;
end;

procedure TKADaoTable.SetBookmarkFlag(Buffer: PChar; Value: TBookmarkFlag);
begin
     PDaoInfo(Buffer + F_StartMyInfo)^.BookmarkFlag := Value;
end;


Function TKADaoTable.GetBookmarkStr: TBookmarkStr;
Begin
  Try
    if F_DaoTable.Bookmarkable Then
       Begin
         Result := TBookmarkStr(IntToStr(PInteger(F_DaoTable.Bookmark.pvData)^));
       End
    Else
       Result := '';
  Except
    Result := '';
  End;
End;

Procedure TKADaoTable.SetBookmarkStr(const Value: TBookmarkStr);
var
   iBook : Integer;
Begin
   if (Value = '') And (F_DaoTable.Bookmarkable) Then iBook := PInteger(F_DaoTable.Bookmark.pvData)^ else iBook := StrToInt(Value);
   if iBook <> PInteger(F_DaoTable.Bookmark.pvData)^ then
      begin
        PInteger(F_DaoTable.Bookmark.pvData)^ := iBook;
        InternalRefresh;
      end;
End;

procedure TKADaoTable.GetBookmarkData(Buffer: PChar; Data: Pointer);
begin
     PInteger(Data)^ := PDaoInfo(Buffer + F_StartMyInfo)^.BookmarkData;
end;

procedure TKADaoTable.SetBookmarkData(Buffer: PChar; Data: Pointer);
begin
     if Buffer <> nil then PDaoInfo(Buffer + F_StartMyInfo)^.BookmarkData := PInteger(Data)^;
end;

//*********************************************************************************** BOOKMARK FUNCTIONS


Procedure TKADaoTable.InternalInitFieldDefs;
Var
  X, Sz :Integer;
begin
        FieldDefs.Clear;
        with FieldDefs do
        begin
          For X:=0 To F_DaoTable.Fields.Count-1 do
              Begin
                Sz:=DaoSizeToBDESize(F_DaoTable.Fields.Item[X].Type_,F_DaoTable.Fields.Item[X].Size);
                Add(F_DaoTable.Fields.Item[X].Name,DaoToBDE(F_DaoTable.Fields.Item[X].Type_),Sz,False);
              End;
        end;
end;

Function TKADaoTable.GetActiveRecordBuffer:  PChar;
begin
        case State of
             dsBrowse      : if IsEmpty then Result := nil else Result := ActiveBuffer;
             dsCalcFields  : Result := CalcBuffer;
             dsFilter      : Result := F_FilterBuffer;
             dsEdit,
             dsInsert      : Result := ActiveBuffer;
        else Result:=nil;
        end;
end;


Procedure TKADaoTable.InternalHandleException;
begin
     Application.HandleException(Self);
end;


procedure TKADaoTable.DataEvent(Event: TDataEvent; Info: Longint);
begin
     inherited DataEvent(Event, Info);
     if (Event in [deDataSetScroll,
                   deUpdateState,
                   deDataSetChange]) and Active then
         Begin
                   InternalSetToRecord(ActiveBuffer);
         End;
end;

Procedure TKADaoTable.ClearDBCalcFields(Buffer: PChar);
begin
        FillChar(Buffer[F_StartCalc],CalcFieldsSize,0);
end;

Function TKADaoTable.GetFieldData(Field: TField; Buffer: Pointer): Boolean;
var
  SourceBuffer : PChar;
  Value        : TStringList;
  FieldNumber  : Integer;
  TempString   : String;
begin
        Result:=False;
        SourceBuffer:=GetActiveRecordBuffer;
        if (not F_Active) or (SourceBuffer=nil) then
           Begin
              Exit;
           End;
        if (Field.FieldKind=fkCalculated) or (Field.FieldKind=fkLookup) then
          begin
                Inc(SourceBuffer,F_StartCalc+Field.Offset);
                if {(SourceBuffer[0]=#0) or} (Buffer=nil) then
                  Begin
                    Exit;
                  End
                Else
                  Begin
                    Move(SourceBuffer[1], Buffer^, Field.DataSize);
                    Result:=True;
                  End;
          end
        else
          Begin
           Value :=PDaoInfo(PChar(SourceBuffer)+F_StartMyInfo)^.RecordData;
           FieldNumber:=Field.FieldNo-1;
           if (Value=Nil) Then Exit;
           if (Buffer = Nil)  Then
            Begin
              Result:=(Value.Strings[FieldNumber] <> '');
            End
               Else
            Begin
              Case Field.DataType of
                ftBytes     : Result:=IntegerToBuffer(Buffer,Value.Strings[FieldNumber]);
                ftInteger   : Result:=IntegerToBuffer(Buffer,Value.Strings[FieldNumber]);
                ftSmallint  : Result:=IntegerToBuffer(Buffer,Value.Strings[FieldNumber]);
                ftCurrency  : Result:=FloatToBuffer(Buffer,Value.Strings[FieldNumber]);
                ftFloat     : Result:=FloatToBuffer(Buffer,Value.Strings[FieldNumber]);
                ftDate      : Result:=DateToBuffer(Buffer,Value.Strings[FieldNumber]);
                ftString    : Begin
                                TempString:=Value.Strings[FieldNumber];
                                TempString:=TempString+#0;
                                CopyMemory(PChar(Buffer),PChar(TempString),Length(TempString));
                                Result := True;
                              End;
                ftAutoInc   : Result:=IntegerToBuffer(Buffer,Value.Strings[FieldNumber]);
                ftTime      : Result:=TimeToBuffer(Buffer,Value.Strings[FieldNumber]);
                ftDateTime  : Result:=DateTimeToBuffer(Buffer,Value.Strings[FieldNumber]);
                ftBoolean   : Result:=BooleanToBuffer(Buffer,Value.Strings[FieldNumber]);
              End;
            End;
          End;
end;

Procedure TKADaoTable.SetFieldData(Field: TField; Buffer: Pointer);
var
        DestinationBuffer: PChar;
        Tmp              : String;
        BTmp             : WordBool;
        BBTmp            : Boolean;
begin
        DestinationBuffer:=GetActiveRecordBuffer;
        if (Field.FieldKind=fkCalculated) or (Field.FieldKind=fkLookup) then //this is a calculated field
        begin
                Inc(DestinationBuffer,F_StartCalc+Field.Offset);
                Boolean(DestinationBuffer[0]):=(Buffer<>nil);
                if Boolean(DestinationBuffer[0]) then CopyMemory(@DestinationBuffer[1],Buffer,Field.DataSize);
        end
        Else
        Begin
          Tmp:='';
          Case Field.DataType of
            ftString     : Tmp := PChar(Buffer);
            ftSmallint   : Tmp := IntToStr(Integer(Buffer^));
            ftWord       : Tmp := IntToStr(Integer(Buffer^));
            ftAutoInc    : Tmp := IntToStr(Integer(Buffer^));
            ftInteger    : Tmp := IntToStr(Integer(Buffer^));

            ftBCD        : Tmp := FloatToStr(Double(Buffer^));
            ftCurrency   : Tmp := FloatToStr(Double(Buffer^));
            ftFloat      : Tmp := FloatToStr(Double(Buffer^));

            ftDate       : Tmp := BufferToDate(Buffer);
            ftTime       : Tmp := BufferToTime(Buffer);
            ftDateTime   : Tmp := BufferToDateTime(Buffer);
            ftBoolean    : Begin
                            BTmp := WordBool(Buffer^);
                            BBTmp:=Boolean(BTmp);                               
                             Case BBTmp Of
                                  True   : Tmp:='True';
                                  False  : Tmp:='False';
                             End;
                           End;
          End;
           PDaoInfo(DestinationBuffer+F_StartMyInfo)^.RecordData[Field.FieldNo-1]:=Tmp;
           PDaoInfo(DestinationBuffer+F_StartMyInfo)^.FieldChanged[Field.FieldNo-1]:=Pointer(True);
           DataEvent(deFieldChange, Longint(Field));
        End;
end;

procedure TKADaoTable.InternalEdit;
begin
     inherited InternalEdit;
end;

procedure TKADaoTable.InternalAddRecord(Buffer: Pointer; Append: Boolean);
Var
 X      : Integer;
 RData  : TStringList;
 S      : String;
begin
    if Append Then InternalLast;
    RData:=PDaoInfo(PChar(Buffer)+F_StartMyInfo)^.RecordData;
    F_DaoTable.AddNew;
    For X:=0 to RData.Count-1 do
        Begin
          S:=RData.Strings[X];
          if (Fields[X].DataType=ftDate) or
             (Fields[X].DataType=ftTime) or
             (Fields[X].DataType=ftDateTime) Then
          Begin
           S:=RemoveNonDigitChars_II(S);
          End;
          if (F_DaoTable.Fields.Item[X].Attributes And dbUpdatableField > 0) Then
              Begin
                if (F_DaoTable.Fields.Item[X].Attributes And dbAutoIncrField) = 0 Then
                    Begin
                      if (F_DaoTable.Fields.Item[X].Attributes And dbSystemField) = 0 Then
                          Begin
                            if (Fields[X].IsBlob) And (S = '(Blob)')  Then
                            Else
                              Begin
                                if (Fields[X].IsBlob) And (S = '') Then S:=' ';
                                F_DaoTable.Fields.Item[X].Value:=S;
                              End;
                          End;
                    End;
              End;
        End;
    OleVariant(F_DaoTable).Update;
end;

procedure TKADaoTable.InternalPost;
Var
 Buffer : PChar;
 X      : Integer;
 RData  : TStringList;
 Changed: Tlist;
 S      : String;
Begin
 CheckActive;
 if State = dsEdit then //***************************************************** EDITING A RECORD
  Begin
    Buffer:=GetActiveRecordBuffer;
    RData:=PDaoInfo(Buffer+F_StartMyInfo)^.RecordData;
    Changed:=PDaoInfo(PChar(Buffer)+F_StartMyInfo)^.FieldChanged;
    F_DaoTable.Edit;
    For X:=0 to RData.Count-1 do
        Begin
        if Boolean(Changed[X]) Then
         Begin
          S:=RData.Strings[X];
          if (Fields[X].DataType=ftDate) or
             (Fields[X].DataType=ftTime) or
             (Fields[X].DataType=ftDateTime) Then            
          Begin
           S:=RemoveNonDigitChars_II(S);
          End;
          if (F_DaoTable.Fields.Item[X].Attributes And dbUpdatableField) > 0 Then
              Begin
                if (F_DaoTable.Fields.Item[X].Attributes And dbAutoIncrField) = 0 Then
                    Begin
                      if (F_DaoTable.Fields.Item[X].Attributes And dbSystemField) = 0 Then
                          Begin
                            if (Fields[X].IsBlob) And (S = '(Blob)')  Then
                            Else
                              Begin
                                if (Fields[X].IsBlob) And (S = '') Then S:=' ';
                                F_DaoTable.Fields.Item[X].Value:=S;
                              End;
                          End;
                    End;
              End;
         End;
        End;
      OleVariant(F_DaoTable).Update;
  End
 Else
  Begin //****************************************************************** ADDING A NEW RECORD
    Buffer:=GetActiveRecordBuffer;
    RData:=PDaoInfo(Buffer+F_StartMyInfo)^.RecordData;
    F_DaoTable.AddNew;
    For X:=0 to RData.Count-1 do
        Begin
         Begin
          S:=RData.Strings[X];
          if (Fields[X].DataType=ftDate) or
             (Fields[X].DataType=ftTime) or
             (Fields[X].DataType=ftDateTime) Then
          Begin
           S:=RemoveNonDigitChars_II(S);
          End;
          if (F_DaoTable.Fields.Item[X].Attributes And dbUpdatableField > 0) Then
              Begin
                if (F_DaoTable.Fields.Item[X].Attributes And dbAutoIncrField) = 0 Then
                    Begin
                      if (F_DaoTable.Fields.Item[X].Attributes And dbSystemField) = 0 Then
                          Begin
                            if (Fields[X].IsBlob) And (S = '(Blob)')  Then
                            Else
                              Begin
                                if (Fields[X].IsBlob) And (S = '') Then S:=' ';
                                F_DaoTable.Fields.Item[X].Value:=S;
                              End;
                          End;
                    End;
              End;
          End;
        End;
      OleVariant(F_DaoTable).Update;
      Try
       InternalLast;
      Except
       //************************************************************ EXCEPTION NO CURRENT RECORD
      End;
  End;
End;


Procedure TKADaoTable.InternalDelete;
Begin
  F_DaoTable.Delete;
  IF F_DaoTable.EOF then OleVariant(F_DaoTable).MoveLast Else F_DaoTable.MoveNext;
End;


Function TKADaoTable.IsCursorOpen: Boolean;
begin
        Result:=F_Active;
end;

Function TKADaoTable.GetCanModify: Boolean;
begin
        Result:= NOT F_ReadOnly;
end;

Function TKADaoTable.GetRecordSize: Word;
begin
        Result:=F_BufferSize;
end;

procedure TKADaoTable.InternalInitRecord(Buffer: PChar);
Var
  X:Integer;
begin
     FillChar(Buffer^, F_BufferSize, 0);
     PDaoInfo(Buffer+F_StartMyInfo)^.RecordData:=TStringList.Create;
     PDaoInfo(Buffer+F_StartMyInfo)^.FieldChanged:=TList.Create;
     PDaoInfo(Buffer+F_StartMyInfo)^.DaoBookmark.pvData:=Nil;
     For X:=0 To F_DaoTable.Fields.Count-1 do
          Begin
            PDaoInfo(Buffer+F_StartMyInfo)^.RecordData.Add('');
            PDaoInfo(Buffer+F_StartMyInfo)^.FieldChanged.Add(Pointer(False));
          End;
     //************************************************************************* TEST METHOD
     InternalLast;
     PDaoInfo(Buffer+F_StartMyInfo)^.RecordNo:=F_RecNo;
     //************************************************************************* TEST METHOD

     //************************************************************************* OK METHOD
     //PDaoInfo(Buffer+F_StartMyInfo)^.RecordNo:=RecordCount;
     //************************************************************************* OK METHOD
end;


Function TKADaoTable.AllocRecordBuffer: PChar;
Var
  X:Integer;
begin
        GetMem(Result,F_BufferSize);
        FillChar(Result^,F_BufferSize,0);
        PDaoInfo(Result+F_StartMyInfo)^.RecordData:=TStringList.Create;
        PDaoInfo(Result+F_StartMyInfo)^.FieldChanged:=TList.Create;
        For X:=0 To F_DaoTable.Fields.Count-1 do
          Begin
            PDaoInfo(Result+F_StartMyInfo)^.RecordData.Add('');
            PDaoInfo(Result+F_StartMyInfo)^.FieldChanged.Add(Pointer(False));
          End;
        PDaoInfo(Result+F_StartMyInfo)^.DaoBookmark.pvData:=Nil;
end;

Procedure TKADaoTable.FreeRecordBuffer(var Buffer: PChar);
begin
        PDaoInfo(Buffer+F_StartMyInfo)^.RecordData.Free;
        PDaoInfo(Buffer+F_StartMyInfo)^.FieldChanged.Free;
        FreeMem(Buffer,F_BufferSize);
        Buffer:=Nil;
end;


Function TKADaoTable.GetRecord(Buffer: PChar; GetMode: TGetMode; DoCheck: Boolean): TGetResult;
var
 Acceptable : Boolean;
 X          : Integer;
 RD         : Variant;
begin
        Result:=grOK;
        Acceptable:=False;
        repeat
                begin
                        case GetMode of
                        gmCurrent:
                                begin
                                    if F_DaoTable.EOF then Result:=grEOF
                                    Else
                                    if F_DaoTable.BOF then Result:=grBOF;
                                end;
                        gmNext:
                                begin
                                    if NOT F_DaoTable.EOF Then
                                        begin
                                          if F_DaoTable.BOF Then F_DaoTable.MoveFirst Else F_DaoTable.MoveNext;
                                          Inc(F_RecNo);
                                          if F_DaoTable.EOF then
                                             Begin
                                               Result:=grEOF;
                                             End;
                                        end
                                    Else
                                         Begin
                                           Result:=grEOF;
                                         End;
                                end;
                        gmPrior:
                                begin
                                        if F_RecNo > 0 then
                                        begin
                                                F_DaoTable.MovePrevious;
                                                Dec(F_RecNo);
                                                Result:=grOK;
                                        end
                                        else Result:=grBOF;
                                end;
                        end;
                        if Result=grOk then
                        begin
                                with PDaoInfo(Buffer+F_StartMyInfo)^ do
                                begin
                                        DaoBookmark:=F_DaoTable.Bookmark^;
                                        GetMem(DaoBookmark.pvData,SizeOF(Integer));
                                        Move(F_DaoTable.Bookmark^.pvData^,DaoBookmark.pvData^,SizeOF(Integer));
                                        RecordNo     := F_RecNo;
                                        BookmarkFlag := bfCurrent;
                                        For X:=0 To F_DaoTable.Fields.Count-1 do
                                            Begin
                                             if NOT Fields[X].IsBlob Then
                                                RD:=F_DaoTable.Fields.Item[X].Value
                                             Else
                                                RD:='(Blob)';
                                             if VarType(RD) = varNull then RD := '';
                                             RecordData.Strings[X]:=RD;
                                             FieldChanged.Items[X]:=Pointer(False);
                                            End;
                                End;
                                ClearDBCalcFields(Buffer);
                                GetCalcFields(Buffer);
                                Acceptable:=FilterRecord(Buffer);
                                if (GetMode=gmCurrent) and (not Acceptable) then Result:=grError;
                        end
                end;
        until (Result<>grOk) or (Acceptable);
end;

Function TKADaoTable.FilterRecord(Buffer: PChar): Boolean;
var
        SaveState: TDatasetState;
begin
        Result:=True;
        if (not Filtered) or (not Assigned(OnFilterRecord)) then Exit;
        SaveState:=SetTempState(dsFilter);
        F_FilterBuffer:=Buffer;
        if Assigned(OnFilterRecord) then OnFilterRecord(self,Result);
        RestoreState(SaveState);
end;


Function TKADaoTable.GetRecordCount: Integer;
Begin
Result:=-1;
Try
   F_DaoTable.MoveFirst;
   OleVariant(F_DaoTable).MoveLast;
   Result:=F_DaoTable.RecordCount;
   F_DaoTable.MoveFirst;
   if F_RecNo=-1 Then
      Begin
        F_DaoTable.MovePrevious;
      End
   Else
      Begin
        OleVariant(F_DaoTable).Move(F_RecNo);
      End;
 Except
 End;
End;


Function  TKADaoTable.GetRecNo: Integer;
Begin
 UpdateCursorPos;
 Result:=F_RecNo;
End;

function TKADaoTable.InsertSQLString(MDString: String): String;
var
  S           : String;
  Found       : Boolean;
  AddData     : Boolean;
  CloseQuote  : Boolean;
  DoubleQuote : Boolean;
  SingleQuote : Boolean;
  NewSQL      : String;
  TempStr     : String;
  TempUpperStr: String;
begin
  Result:= F_ComposeSQL(F_SQL);
  if MDString <> '' then
    begin
      DoubleQuote:= False;
      SingleQuote:= False;
      Found:= False;
      CloseQuote:= False;
      NewSQL:= '';
      S:= Result;
      while S <> '' do
        begin
          AddData:= True;
          if S[1] = '''' then SingleQuote:= not(SingleQuote);
          if S[1] = '"' then DoubleQuote:= not(DoubleQuote);
          if not(SingleQuote) and not(DoubleQuote) then
            begin
              TempStr:= '';
              TempUpperStr:= UpperCase(S);
              if Pos(' ORDER BY ', TempUpperStr) = 1 then
                begin
                  if CloseQuote then
                     Begin
                       NewSQL:= NewSQL + ') ';
                       CloseQuote:=False;
                     End
                  else
                    begin
                      NewSQL:= NewSQL + ' WHERE (' + MDString + ')';
                      Found:= True;
                    end;
                  NewSQL:= NewSQL + ' ORDER BY ';
                  System.Delete(S, 1, Length(' ORDER BY'));
                  AddData:= False;
                end
              else if Pos(' WHERE ', TempUpperStr) = 1 then
                begin
                  NewSQL:= NewSQL + ' WHERE (' + MDString + ') AND (';
                  System.Delete(S, 1, Length(' WHERE '));
                  CloseQuote:= True;
                  AddData:= False;
                  Found:= True;
                end
              else if Pos(';', TempUpperStr) = 1 then
                begin
                  if not(Found) then
                    begin
                      NewSQL:= NewSQL + ' WHERE (' + MDString + ')';
                      AddData:= False;
                      S:= '';
                      Found:= True;
                    end
                  else if CloseQuote then
                       Begin
                         NewSQL:= NewSQL + ')';
                         CloseQuote:=False;
                       End;
                end;
              if TempStr <> '' then
                begin
                  NewSQL:= NewSQL + TempStr + '(' + MDString + ') ';
                  System.Delete(S, 1, Length(TempStr));
                  AddData:= False;
                end;
            end;
          if AddData then
            begin
              NewSQL:= NewSQL + S[1];
              System.Delete(S, 1, 1);
            end;
        end;
      if CloseQuote then NewSQL:= NewSQL + ')';
      if not(Found) then NewSQL:= NewSQL + ' WHERE ' + MDString;
      Result:= NewSQL;
    end;
end;
                                                                                        
Function  TKADaoTable.BuildDetailSQL:String;
Var
 X  : Integer;
 S  : String;
 FT : TField;
Begin
S:='';
Result:='';
if F_Master.Count <> F_Detail.Count Then
     Begin
       DatabaseError('The number of Master and Detail fields must be equal!');
     End;
if F_Master.Count > 0 Then
     Begin
      For X:=0 To F_Master.Count-1 do
         Begin
          S:=S+'(';
          S:=S+F_Detail.Strings[X]+' ';
          FT :=F_MasterLink.Dataset.FieldByName(F_Master.Strings[X]);
          if FT.IsNull then S:= S + 'IS NULL'
          Else
          Case FT.DataType of
             ftString,
             ftMemo     : S:= S + ' = "' + FT.AsString + '"';
             ftCurrency,
             ftFloat,
             ftInteger : S:= S + ' = ' + FT.AsString;
             ftBoolean : If FT.AsBoolean then S:= S + ' = True' Else S:= S + ' = False';
             ftDate,
             ftTime,
             ftDateTime: S:= S + ' = #' + FormatDateTime('m/d/yy', FT.AsDateTime) + '#';
             Else
             DatabaseError('Unable to build Master/Detail SQL Relationship!')
          End;
          S:=S+')';
          if (X < F_Master.Count-1) Then S:=S+' AND ';
         End;
     End;
 Result := S;
End;

Procedure TKADaoTable.MasterDSChanged;
Var
  S    : String;
  TSQL : TStrings;
Begin
  if csDestroying in ComponentState then EXIT;
  if (MasterSource <> NIL) and not(F_MDisabled) then
  Begin
  F_ProcessMasterFields(F_MasterFields);
  if F_Master.Count > 0 Then
     Begin
      TSQL := TStringList.Create;
      S:=BuildDetailSQL;
      S:=InsertSQLString(S);
      TSQL.SetText(F_SQL.GetText);
      F_SQL.SetText(Pchar(S));
      //*************************************************
      ClearBuffers;
      CheckBrowseMode;
      CloseDaoRecordset;
      OpenDaoRecordset;
      ActivateBuffers;
      First;
      //*************************************************
      F_SQL.SetText(TSQL.GetText);
      TSQL.Free;
     End;
  End;
End;


//******************************************************************************
//*                  Master/Detail Data Link Handling
//******************************************************************************
constructor TKADAODataLink.Create(DataSet: TKADaoTable);
begin
  inherited Create(DataSet);
  F_KADaoTable:= DataSet;
  F_MasterIsChanged:=False;
end;

procedure TKADAODataLink.ActiveChanged;
begin
  F_KADaoTable.F_MDisabled:= True;
  if DataSet <> NIL then F_KADaoTable.F_MDisabled:= not(DataSet.Active);
  if (F_KADaoTable.F_Active) then F_KADaoTable.MasterDSChanged Else F_MasterIsChanged:=True;
end;

{$IFNDEF VER100}
function TKADAODataLink.GetDetailDataSet: TDataSet;
begin
  Result:= F_KADaoTable;
end;
{$ENDIF}

procedure TKADAODataLink.RecordChanged(Field: TField);
begin
  if (Field = NIL) and (F_KADaoTable.Active) then F_KADaoTable.MasterDSChanged;
end;

procedure TKADAODataLink.CheckBrowseMode;
begin
  if F_KADaoTable.Active then F_KADaoTable.CheckBrowseMode;
end;

//******************************************************************************
//*                         Blob Stream Handling
//******************************************************************************
constructor TKBlobStream.Create(Field: TBlobField; Mode: TBlobStreamMode);
Var
   RD     : OleVariant;
begin
     F_BlobData := '';
     F_Mode     := Mode;
     F_Field    := Field;
     F_DataSet  := F_Field.DataSet as TKADaoTable;
     F_Buffer   := F_DataSet.GetActiveRecordBuffer;
     if not (F_Buffer = nil) then F_DataSet.InternalSetToRecord(F_Buffer);
     F_Opened := True;
     if not F_Field.Modified then
     begin
          if (F_DataSet.State = dsInsert)
          then
            Begin                                                  
              F_BlobData := '';
             End
          Else
             Begin
               if Not (F_Buffer = nil) Then
                  Begin
                    Try
                      RD:=F_DataSet.F_DaoTable.Fields.Item[F_Field.FieldNo-1].Value;
                      if VarType(RD) = varNull then RD := '';
                      F_BlobData:=RD;
                     Except
                      F_BlobData:='';
                     End;
                  End
               Else
                  Begin
                    F_BlobData:='';
                  End;
             End;
          F_BlobSize:=Length(F_BlobData);
     end;
     if Mode = bmWrite then Truncate;
end;


destructor TKBlobStream.Destroy;
begin
   if F_Modified then
     try
          F_DataSet.DataEvent(deFieldChange, Longint(F_Field));
          F_BlobData:='';
          F_Buffer:=Nil;
     except
          Application.HandleException(Self);
     end;
end;


function TKBlobStream.Read(var Buffer; Count: Longint): Longint;
begin
     Result := 0;
     if F_Opened then
     begin
          if Count > Size - F_Position then
             Result := Size - F_Position else
             Result := Count;
          if Result > 0 then
          begin
               Move(PChar(F_BlobData)[F_Position], Buffer, Result);
               Inc(F_Position, Result);
          end;
     end;
end;


function TKBlobStream.Write(const Buffer; Count: Longint): Longint;
var
   pTemp  : Pointer;
   RData  : TStringList;
   Changed: TList;
begin
     Result := 0;
     if F_Opened then
     begin
          try
          begin
               SetLength(F_BlobData,Count+1);
               pTemp:=PChar(F_BlobData);
               CopyMemory(pTemp, @Buffer, Count);
               F_BlobData  := Copy(F_BlobData,1,Count);
               F_BlobSize  := Length(F_BlobData);
               RData:=PDaoInfo(F_Buffer+F_DataSet.F_StartMyInfo)^.RecordData;
               Changed:=PDaoInfo(F_Buffer+F_DataSet.F_StartMyInfo)^.FieldChanged;
               RData.Strings[F_Field.FieldNo-1]:=F_BlobData;
               Changed[F_Field.FieldNo-1]:=Pointer(True);
               F_Modified := True;
          end;
          finally
          end;
          Inc(F_Position, Count);
          Result := Count;
          F_Modified := True;
     end;
end;


function TKBlobStream.Seek(Offset: Longint; Origin: Word): Longint;
begin
     case Origin of
          0: F_Position := Offset;
          1: Inc(F_Position, Offset);
          2: F_Position := F_BlobSize + Offset;
     end;
     Result := F_Position;
end;


procedure TKBlobStream.Truncate;
begin
     if F_Opened then F_Modified := True;
end;

//***********************************************************************************
Function TKADaoTable.IntegerToBuffer(Buffer: Pointer; S: String): Boolean;
begin
     Result:=False;
     if Buffer=Nil Then Exit;
     Result := (S <> '');
     if S = '' then S := '0';
     Integer(Buffer^) := StrToInt(S);
end;


Function TKADaoTable.FloatToBuffer(Buffer: Pointer; S: String): Boolean;
begin
     Result:=False;
     if Buffer=Nil Then Exit;
     Result := (S <> '');
     if S = '' then S := '0';
     Double(Buffer^) := StrToFloat(S);
end;


Function TKADaoTable.DateToBuffer(Buffer: Pointer; S: String): Boolean;
var
   Ttmp : TTimeStamp;
   dTmp : ^TDateTimeRec;
begin
     Result:=False;
     if Buffer=Nil Then Exit;
     Result := (S <> '');
     S:=RemoveNonDigitChars_II(S);
     Ttmp.Date := 0;
     Ttmp.Time := 0;
     if S <> '' then Ttmp := DateTimeToTimeStamp(StrToDateTime(S));
     Dtmp := Buffer;
     Dtmp.Date := Ttmp.Date;
end;


Function TKADaoTable.TimeToBuffer(Buffer: Pointer; S: String): Boolean;
var
   Ttmp : TTimeStamp;
   Dtmp : ^TDateTimeRec;
begin
     Result:=False;
     if Buffer=Nil Then Exit;
     Result := (S <> '');
     S:=RemoveNonDigitChars(S);
     Ttmp.Date := 0;
     Ttmp.Time := 0;
     if S <> '' then
     begin
          if Length(S) > 10
          then
              Ttmp := DateTimeToTimeStamp(StrToDateTime(S))
          else
              Ttmp := DateTimeToTimeStamp(StrToDate(S));
     end;
     Dtmp := Buffer;
     Dtmp.Time := Ttmp.Time;
end;


Function TKADaoTable.DateTimeToBuffer(Buffer: Pointer; S: String): Boolean;
var
   Ttmp : TTimeStamp;
   Dtmp : ^TDateTimeRec;
begin
     Result:=False;
     if Buffer=Nil Then Exit;
     Result := (S <> '');
     S:=RemoveNonDigitChars(S);
     Ttmp.Date := 0;
     Ttmp.Time := 0;
     if S <> '' then Ttmp := DateTimeToTimeStamp(StrToDateTime(S));
     Dtmp := Buffer;
     Dtmp^.DateTime := TimeStampToMSecs(Ttmp);
end;


Function TKADaoTable.BooleanToBuffer(Buffer: Pointer; S: String): Boolean;
begin
     Result:=False;
     if Buffer=Nil Then Exit;
     Result := (S <> '');
     if S = ''    Then S := '0';
     if S = '-1'  Then S := '1';
     if AnsiLowerCase(S)='false' Then S := '0';
     if AnsiLowerCase(S)='true'  Then S := '1';
     WordBool(Buffer^) := WordBool(StrToInt(S));
end;


Function TKADaoTable.BufferToDate(Buffer: Pointer): String;
var
   Ttmp : TTimeStamp;
   Dtmp : ^TDateTimeRec;
begin
     Dtmp := Buffer;
     Ttmp.Date := Dtmp^.Date;
     Result := DateToStr(TimeStampToDateTime(Ttmp));
end;


Function TKADaoTable.BufferToTime(Buffer: Pointer): String;
var
   TTmp : TTimeStamp;
   Dtmp : ^TDateTimeRec;
begin
     Dtmp := Buffer;
     Ttmp.Time := Dtmp^.Time;
     Result := FormatDateTime('hh:nn:ss', TimeStampToDateTime(Ttmp));
end;


Function TKADaoTable.BufferToDateTime(Buffer: Pointer): String;
var
   TTmp : TTimeStamp;
   Dtmp : ^TDateTimeRec;
begin
     Dtmp   := Buffer;
     Ttmp   := MsecsToTimeStamp(Dtmp.DateTime);
     Result := DateTimeToStr(TimeStampToDateTime(TTmp));
end;



//*********************************************************************************** TableName
Procedure TTableNameEditor.GetValues( Proc: TGetStrProc);
Var
  DTable : TKADaoTable;
  DBase : TKADaodatabase;
  X     : Integer;
Begin
  if GetComponent(0) is TKADaoTable then
  Begin
    DTable := TKADaoTable(GetComponent(0));
    if Assigned(DTable.F_Database) Then
       Begin
        DBase:=DTable.F_Database;
        Try
          For X := 0 to DBase.TableNames.Count-1 do Proc(DBase.TableNames.Strings[X]);
        Finally
        End;
       End;
  End;
End;

Function TTableNameEditor.GetAttributes: TPropertyAttributes;
Begin
  Result:= Inherited GetAttributes + [paValueList, paSortList];
End;
//*********************************************************************************** QueryName

Procedure TQueryNameEditor.GetValues( Proc: TGetStrProc);
Var
  DTable : TKADaoTable;
  DBase : TKADaodatabase;
  X     : Integer;
Begin
  if GetComponent(0) is TKADaoTable then
  Begin
    DTable := TKADaoTable(GetComponent(0));
    if Assigned(DTable.F_Database) Then
       Begin
        DBase:=DTable.F_Database;
        Try
          For X := 0 to DBase.QueryDefNames.Count-1 do Proc(DBase.QueryDefNames.Strings[X]);
        Finally
        End;
       End;
  End;
End;

Function TQueryNameEditor.GetAttributes: TPropertyAttributes;
Begin
  Result:= Inherited GetAttributes + [paValueList, paSortList];
End;
//*********************************************************************************** IndexName
Procedure TIndexNameEditor.GetValues( Proc: TGetStrProc);
Var
  DTable : TKADaoTable;
  DBase  : TKADaodatabase;
  X      : Integer;
  Count  : Integer;
Begin
  if GetComponent(0) is TKADaoTable then
  Begin
    DTable := TKADaoTable(GetComponent(0));
    if Assigned(DTable.F_Database) Then
    if DTable.TableName <> '' Then
       Begin
        DBase:=DTable.F_Database;
        Try
          Count :=DBase.CoreDatabase.TableDefs.Item[DTable.TableName].Indexes.Count;
          For X := 0 to  Count-1 do
              Begin
                Proc(DBase.CoreDatabase.TableDefs.Item[DTable.TableName].Indexes.Item[X].Name);
              End;
        Finally
        End;
       End;
  End;
End;


Function TIndexNameEditor.GetAttributes: TPropertyAttributes;
Begin
  Result:= Inherited GetAttributes + [paValueList, paSortList];
End;

//***************************************************************************************************************************

Function TTableTypeEditor.GetAttributes: TPropertyAttributes;
Begin
  Result := Inherited GetAttributes + [paValueList, paSortList];
End;

Function  TTableTypeEditor.GetValue: string;
Begin
 if GetComponent(0) is TKADaoTable then
    Begin
      Result:='StandardTable';
      if TKADaoTable(GetComponent(0)).F_TableType=dbOpenTable       then Result:='StandardTable';
      if TKADaoTable(GetComponent(0)).F_TableType=dbOpenDynaset     then Result:='DynasetTable';
      if TKADaoTable(GetComponent(0)).F_TableType=dbOpenDynamic     then Result:='DynamicTable';
      if TKADaoTable(GetComponent(0)).F_TableType=dbOpenSnapshot    then Result:='SnapshotTable';
      if TKADaoTable(GetComponent(0)).F_TableType=dbOpenForwardOnly then Result:='ForwardOnlyTable';
    End;
End;

Procedure TTableTypeEditor.GetValues( Proc: TGetStrProc);
Begin
  if GetComponent(0) is TKADaoTable then
     Begin
       Proc('StandardTable');
       Proc('DynasetTable');
       Proc('DynamicTable');
       Proc('SnapshotTable');
       Proc('ForwardOnlyTable');
     End;
End;


Procedure TTableTypeEditor.SetValue(const Value: string);
Var
 Dat : Integer;
Begin
  if GetComponent(0) is TKADaoTable then
       Begin
       Dat:=dbOpenTable;
       if Value='StandardTable'    Then Dat:=dbOpenTable;
       if Value='DynasetTable'     Then Dat:=dbOpenDynaset;
       if Value='DynamicTable'     Then Dat:=dbOpenDynamic;
       if Value='SnapshotTable'    Then Dat:=dbOpenSnapshot;
       if Value='ForwardOnlyTable' Then Dat:=dbOpenForwardOnly;
       Inherited SetValue(IntToStr(Dat));
       Modified;
     End;
End;

//***************************************************************************************************************************

Function TLockTypeEditor.GetAttributes: TPropertyAttributes;
Begin
  Result := Inherited GetAttributes + [paValueList, paSortList];
End;

Function  TLockTypeEditor.GetValue: string;
Begin
 if GetComponent(0) is TKADaoTable then
    Begin
      Result:='dbPessimistic';
      if TKADaoTable(GetComponent(0)).F_LockType=DAOApi.dbPessimistic       then Result:='dbPessimistic';
      if TKADaoTable(GetComponent(0)).F_LockType=DAOApi.dbOptimistic        then Result:='dbOptimistic';
      if TKADaoTable(GetComponent(0)).F_LockType=DAOApi.dbOptimisticValue   then Result:='dbOptimisticValue';
      if TKADaoTable(GetComponent(0)).F_LockType=DAOApi.dbReadOnly          then Result:='dbReadOnly';
      if TKADaoTable(GetComponent(0)).F_LockType=DAOApi.dbOptimisticBatch   then Result:='dbOptimisticBatch';
    End;
End;

Procedure TLockTypeEditor.GetValues( Proc: TGetStrProc);
Begin
  if GetComponent(0) is TKADaoTable then
     Begin
       Proc('dbPessimistic');
       Proc('dbOptimistic');
       Proc('dbOptimisticValue');
       Proc('dbReadOnly');
       Proc('dbOptimisticBatch');
     End;
End;

                                                                  
Procedure TLockTypeEditor.SetValue(const Value: string);
Var
 Dat : Integer;
Begin
  if GetComponent(0) is TKADaoTable then
       Begin
       Dat:=DAOApi.dbPessimistic;
       if Value='dbPessimistic'     Then Dat:=DAOApi.dbPessimistic;
       if Value='dbOptimistic'      Then Dat:=DAOApi.dbOptimistic;
       if Value='dbOptimisticValue' Then Dat:=DAOApi.dbOptimisticValue;
       if Value='dbReadOnly'        Then Dat:=DAOApi.dbReadOnly;
       if Value='dbOptimisticBatch' Then Dat:=DAOApi.dbOptimisticBatch;
       Inherited SetValue(IntToStr(Dat));
       Modified;
     End;
End;


Function TSortByEditor.GetAttributes: TPropertyAttributes;
Begin
  Result:= [paDialog];
End;

Procedure TSortByEditor.SetValue(const Value: string);
Begin
if GetComponent(0) is TKADaoTable then
  Begin
    inherited SetValue(Value);
    Modified;
  End;
End;

procedure TSortByEditor.Edit;
Var
 DT:TKADaoTable;
Begin
 if GetComponent(0) is TKADaoTable then
  Begin
    DT:=GetComponent(0) AS TKADaoTable;
    if (DT.Active=False) and Not (csLoading in DT.ComponentState) Then
       Begin
         Try
           DT.Active:=True;
           DT.Active:=False;
         Except
           DT.F_SortedBy.Clear;
           DT.F_FieldNames.Clear;
           DatabaseError('Cannot obtain field names');
           Exit;
         End;
       End;
    Application.CreateForm(TSortByDialog,SortDialog);
    if SortDialog.Execute(DT.F_FieldNames,DT.F_SortedBy) Then;
    SortDialog.Free;
  End;
End;

Function TQueryDefParamsEditor.GetAttributes: TPropertyAttributes;
Begin
  Result:= [paDialog];
End;

Procedure TQueryDefParamsEditor.SetValue(const Value: string);
Begin
if GetComponent(0) is TKADaoTable then
  Begin
    inherited SetValue(Value);
    Modified;
  End;
End;

procedure TQueryDefParamsEditor.Edit;
Var
 DT : TKADaoTable;
 X  : Integer;
Begin
 if GetComponent(0) is TKADaoTable then
  Begin
    DT:=GetComponent(0) AS TKADaoTable;
    DT.F_QD_ParamNames.Clear;
    DT.F_QD_ParamDaoTypes.Clear;
    DT.F_QD_ParamBDETypes.Clear;
    Try
     For X := 0 To DT.Database.CoreDatabase.QueryDefs.Item[DT.QueryDefName].Parameters.Count-1 do
        Begin
          DT.F_QD_ParamNames.Add(DT.Database.CoreDatabase.QueryDefs.Item[DT.QueryDefName].Parameters[X].Name);
          DT.F_QD_ParamDaoTypes.Add(GetDaoFieldTypeNames(DT.Database.CoreDatabase.QueryDefs.Item[DT.QueryDefName].Parameters[X].Type_));
          DT.F_QD_ParamBDETypes.Add(GetBDEFieldTypeNames(DaoToBDE(DT.Database.CoreDatabase.QueryDefs.Item[DT.QueryDefName].Parameters[X].Type_)));
        End;
    Except
      DatabaseError('Cannot obtain QueryDef Parameters');
      Exit;
    End;
    Application.CreateForm(TQueryDefDialog,QueryDefDialog);
    if QueryDefDialog.Execute(DT.F_QD_ParamNames,DT.F_QD_ParamDaoTypes,DT.F_QD_ParamBDETypes,DT.F_QueryDefParameters) Then;
    QueryDefDialog.Free;
    Modified;
  End;
End;


//***************************************************************************************************************************
Function TMasterFieldsEditor.GetAttributes: TPropertyAttributes;
Begin
  Result:= [paDialog];
End;

Procedure TMasterFieldsEditor.SetValue(const Value: string);
Begin
if GetComponent(0) is TKADaoTable then
  Begin
    inherited SetValue(Value);
    Modified;
  End;
End;

procedure TMasterFieldsEditor.Edit;
Var
 MT,DT        : TKADaoTable;
 MF,DF,TF     : TStrings;
Begin
 if GetComponent(0) is TKADaoTable then
  Begin
    DT:=GetComponent(0) AS TKADaoTable;
    if Not Assigned(DT.MasterSource) Then
       Begin
        DT.F_Master.Clear;
        DT.F_Detail.Clear;
        DT.F_MasterFields.Clear;
        DatabaseError('MasterSource not defined!');
        Exit;
       End;
    if NOT (DT.MasterSource.DataSet is TKADaoTable) Then
       Begin
        DT.F_Master.Clear;
        DT.F_Detail.Clear;
        DT.F_MasterFields.Clear;
        DatabaseError('Cannot link to other datasets exept KADaoTable''s!');
        Exit;
       End;
    MT:=DT.MasterSource.DataSet As TKADaoTable;
    if (DT.Active=False) and Not (csLoading in DT.ComponentState) Then
       Begin
         Try
           DT.Active:=True;
           DT.Active:=False;
         Except
           DT.F_Master.Clear;
           DT.F_Detail.Clear;
           DT.F_MasterFields.Clear;
           DatabaseError('Cannot obtain detail field names!');
           Exit;
         End;
       End;
    if (MT.Active=False) and Not (csLoading in DT.ComponentState) Then
       Begin
         Try
           MT.Active:=True;
           MT.Active:=False;
         Except
           DT.F_Master.Clear;
           DT.F_Detail.Clear;                              
           DT.F_MasterFields.Clear;
           DatabaseError('Cannot obtain master field names!');
           Exit;
         End;
       End;
    MF     := TStringList.Create;
    DF     := TStringList.Create;
    TF     := TStringList.Create;
    MF.SetText(MT.F_MDFieldNames.GetText);
    DF.SetText(DT.F_MDFieldNames.GetText);
    TF.SetText(DT.F_MasterFields.GetText);
    Application.CreateForm(TMasterDetailForm,MasterDetailForm);
    if MasterDetailForm.Execute(DF,MF,TF) Then;
    MasterDetailForm.Free;
    DT.MasterFields:=TF;
    MF.Free;
    DF.Free;
    TF.Free;
    Modified;
  End;
End;

function TMasterSourcePropertyEditor.GetAttributes: TPropertyAttributes;
begin
  Result:= [paValueList, paSortList];
end;


function TMasterSourcePropertyEditor.GetValue: String;
var
  Tab: TKADaoTable;
begin
  Result:= '';
  Tab:= TKADaoTable(GetComponent(0));
  if Tab.MasterSource <> NIL then Result:= Tab.MasterSource.Name
  else Result:= '';
end;

procedure TMasterSourcePropertyEditor.GetValues(Proc: TGetStrProc);
var
  Tab: TKADaoTable;
  DS : TDataSource;
  I  : Integer;
begin
  Tab:= TKADaoTable(GetComponent(0));
  if Tab.Owner = NIL then EXIT;
  for I:= 0 to Tab.Owner.ComponentCount - 1 do
    begin
      DS:= NIL;
      if Tab.Owner.Components[i] is TDataSource then DS:= TDataSource(Tab.Owner.Components[i]);
      if (DS <> NIL) and (DS.DataSet <> Tab) then Proc(DS.Name);
    end;
end;

procedure TMasterSourcePropertyEditor.SetValue(const Value: String);
var
  Tab  : TKADaoTable;
  Comp : TComponent;
begin
  Tab:= TKADaoTable(GetComponent(0));
  if Value = '' then
    begin
      Tab.MasterSource:= NIL;
      EXIT;
    end;
  if Tab.Owner = NIL then EXIT;
  Comp:= Tab.Owner.FindComponent(Value);
  if (Comp = NIL) or not(Comp is TDataSource) then EXIT;
  if NOT ((Comp AS TDataSource).DataSet is TKADaoTable) Then
       Begin
        DatabaseError('Cannot link to other datasets exept KADaoTable''s!');
        Exit;
       End;
  Tab.MasterSource:= TDataSource(Comp);
  Modified;
end;
//***************************************************************************************************************************

Procedure Register;
begin
    RegisterComponents('KA Dao', [TKADaoTable]);
    RegisterPropertyEditor(TypeInfo(String),TKADaoTable,'TableName',TTableNameEditor);
    RegisterPropertyEditor(TypeInfo(String),TKADaoTable,'QueryDefName',TQueryNameEditor);
    RegisterPropertyEditor(TypeInfo(String),TKADaoTable,'IndexName',TIndexNameEditor);
    RegisterPropertyEditor(TypeInfo(Integer),TKADaoTable,'TableType',TTableTypeEditor);
    RegisterPropertyEditor(TypeInfo(Integer),TKADaoTable,'LockType',TLockTypeEditor);
    RegisterPropertyEditor(TypeInfo(TStrings),TKADaoTable,'SortedBy',TSortByEditor);
    RegisterPropertyEditor(TypeInfo(TStrings),TKADaoTable,'QueryDefParameters',TQueryDefParamsEditor);
    RegisterPropertyEditor(TypeInfo(TDataSource), TKADaoTable, 'MasterSource', TMasterSourcePropertyEditor);
    RegisterPropertyEditor(TypeInfo(TStrings),TKADaoTable,'MasterFields',TMasterFieldsEditor);
end;
end.

