unit KDaoDataBase;
{$DEFINE USEDB}      //DISABLE IF YOU WANT TO USE A PURE DAO WITHOUT KDaoTable;
{$DEFINE DAO36}
//******************************************************************************
//                    Delphi Dao Project Version 1.5
//                 Copyright (c) 2000 by Kiril Antonov
//******************************************************************************

//****************************************** CHANGES ***************************
// 30.05.2000 - Added a checking of database for supporting transactions
//              If database does NOT support transactions
//              NO DAO action is performed

interface
Uses
DAOApi,
{$IFDEF DAO35}
DAO35Api,
{$ENDIF}
{$IFDEF DAO36}
DAO36Api,
{$ENDIF}
Windows,SysUtils,Classes,FileCtrl,DbLogdlg,Registry,DsgnIntf;

Type
  DaoError = Class(Exception);

TDatabaseTypeEditor = class(TStringProperty)
    Public
      Procedure GetValues( Proc: TGetStrProc); override;
      Procedure SetValue(const Value: string); override;
      Function  GetAttributes: TPropertyAttributes; override;
    End;

TDatabaseNameEditor = class(TStringProperty)
    Public
      Procedure Edit;override;
      Procedure SetValue(const Value: string); override;
      Function  GetAttributes: TPropertyAttributes; override;
    End;

TEngineTypeEditor = class(TIntegerProperty)
    Public
     function  GetValue: string; override;
     Procedure GetValues( Proc: TGetStrProc);override;
     procedure SetValue(const Value: string); override;
     Function  GetAttributes: TPropertyAttributes; override;
    End;

TWorkspaceEditor = class(TStringProperty)
    Public
      Procedure GetValues( Proc: TGetStrProc); override;
      Function  GetAttributes: TPropertyAttributes; override;
    End;



TKADaoDatabase = Class(TComponent)
       Private
         F_Database        : String;
         F_EngineType      : Integer;
         F_DatabaseType    : String;
         F_Workspace       : String;
         F_DaoVersion      : ShortString;
         F_Active          : Boolean;
         F_ReadOnly        : Boolean;
         F_Exclusive       : Boolean;
         F_LoginPrompt     : Boolean;
         F_Username        : ShortString;
         F_Password        : ShortString;
         F_MachineName     : ShortString;
         F_QueryTimeout    : Integer;
         F_LoginDialog     : TLoginDialog;
         F_TableNames      : TStringList;
         F_ActiveTableNames: TStringList;
         F_QueryDefNames   : TStringList;
         F_DBTypesList     : TStringList;
         F_DriverList      : TStringList;
         procedure F_Get_DBTypesList(List: TStrings);
         Function  F_Get_DBTypeFileExtension(DBType:String):String;
         Function  F_Get_DBTypeTableType(DBType:String):String;
         procedure F_Get_OdbcDriverList(List: TStrings);
         Procedure F_Set_Database(Value : String);
         Procedure F_Set_Workspace(Value : String);
         Function  F_Get_DatabaseType:String;
         Procedure F_Set_DatabaseType(Value : String);
         Procedure F_Set_EngineType(Value : Integer);
         Procedure F_Set_UserName(Value : ShortString);
         Procedure F_Set_MachineName(Value : ShortString);
         Procedure F_Set_Password(Value : ShortString);
         Procedure F_Set_Exclusive(Value : Boolean);
         Procedure F_Set_LoginPrompt(Value : Boolean);
         Procedure F_Set_ReadOnly(Value : Boolean);
         Procedure F_Set_Active(Value : Boolean);
         Procedure F_RefreshDefinitions;
       Protected
       Published
         Property    MachineName     : ShortString Read F_MachineName Write F_Set_MachineName Stored TRUE;
         Property    Exclusive       : Boolean Read F_Exclusive Write F_Set_Exclusive;
         Property    EngineType      : Integer Read F_EngineType Write F_Set_EngineType;
         Property    DatabaseType    : String  Read F_Get_DatabaseType Write F_Set_DatabaseType;
         Property    Database        : String  Read F_Database Write F_Set_Database;
         Property    ReadOnly        : Boolean Read F_ReadOnly Write F_Set_ReadOnly;
         Property    LoginPrompt     : Boolean Read F_LoginPrompt Write F_Set_LoginPrompt;
         Property    UserName        : ShortString Read F_Username  Write F_Set_UserName;
         Property    Password        : ShortString Read F_Password  Write F_Set_Password;
         Property    Version         : ShortString Read F_DaoVersion Write F_DaoVersion;
         Property    Workspace       : String Read F_Workspace Write F_Set_Workspace;
         Property    QueryTimeout    : Integer Read F_QueryTimeout Write F_QueryTimeout;
         Property    Connected       : Boolean Read F_Active Write F_Set_Active Default False;
       Public
         {$IFDEF DAO35}
         CoreDBEngine                : DAO35Api.DBEngine;
         CoreDatabase                : DAO35Api.Database;
         CoreWorkspace               : DAO35Api.Workspace;
         {$ENDIF}
         {$IFDEF DAO36}
         CoreDBEngine                : DAO36Api.DBEngine;
         CoreDatabase                : DAO36Api.Database;
         CoreWorkspace               : DAO36Api.Workspace;
         {$ENDIF}
         Property    QueryDefNames   : TStringList Read F_QueryDefNames;
         Property    TableNames      : TStringList Read F_TableNames;
         Property    ActiveTableNames: TStringList Read F_ActiveTableNames;
         Property    DatabaseTypes   : TStringList Read F_DBTypesList;
         Constructor                   Create(AOwner : TComponent); override;
         Destructor                    Destroy; override;
         //****************************************************** Transactions
         Procedure                   StartTransaction;
         Procedure                   Commit;
         Procedure                   Rollback;
         //****************************************************** Utils
         Procedure                   RepairAccessDatabase(DatabaseName:String);
         Procedure                   CompactAccessDatabase(DatabaseName:String);
         Procedure                   CreateAccessDatabase(DatabaseName:String);
         Procedure                   CreateAccessDatabaseEx(DatabaseName,LANGID,CP,COUNTRY,Password,Version:String;Encrypt:Boolean);
         //****************************************************** Utils II
         Function                    CreateTable(TableName:String; FieldNames : Variant; FieldTypes : Variant; FieldSizes : Variant; FieldIndexes:Variant):Boolean;

         Procedure                   RenameTable(OldTableName,NewTableName:String);
         Procedure                   DeleteTable(TableName:String);

         Procedure                   RenameIndex(TableName,OldIndexName,NewIndexName:String);
         Procedure                   DeleteIndexByName(TableName,IndexName:String);
         Procedure                   DeleteIndexByFieldName(TableName,FieldName:String);

         Procedure                   RenameField(TableName,OldFieldName,NewFieldName:String);
         Procedure                   DeleteField(TableName,FieldName:String);
         Function                    AddFieldsToTable(TableName:String; FieldNames : Variant; FieldTypes : Variant; FieldSizes : Variant; FieldIndexes:Variant):Boolean;

         Function                    CreateQueryDef(Name:String;SQL:String):Boolean;
         Function                    GetQueryDefSQLText(Name:String):String;
         Procedure                   RenameQueryDef(OldQueryName,NewQueryName:String);
         Procedure                   DeleteQueryDef(QueryName:String);
      End;

      procedure Register;
{$IFNDEF VER130}
var
  //   ***************************************************
  //   Defined only for Delphi3 and Delphi4
  //   Delphi5 has buildin support for EmptyParam
  //   ***************************************************
  EmptyParam: OleVariant;
{$ENDIF}

//*************************************************************************************************
implementation
Uses Dialogs,Forms{$IFDEF USEDB},KDaoTable{$ENDIF};

Const
  //dbLangGeneral = ';LANGID=0x0402;CP=1251;COUNTRY=0';
  dbLangGeneral = ';LANGID=%s;CP=%s;COUNTRY=%s';
Var
  DlgChooseDatabase  : TOpenDialog;

  //*************************************************************************************************
Constructor TKADaoDatabase.Create(AOwner : TComponent);
Begin
  Inherited Create(AOwner);

  CoreDBEngine             := CoDBEngine.Create;
  F_EngineType             := dbUseJet;

  CoreDBEngine.DefaultType := F_EngineType;
  CoreWorkspace            := CoreDBEngine.Workspaces.Item[0];
  F_DaoVersion             := CoreDBEngine.Version;
  F_Workspace              := CoreWorkspace.Name;

  F_MachineName            := '';
  F_DatabaseType           :='Access';
  F_Active                 := False;
  F_Database               := '';
  F_ReadOnly               := False;
  F_Exclusive              := False;
  F_LoginPrompt            := False;
  F_Username               :='Admin';
  F_Password               :='';

  F_LoginDialog            := TLoginDialog.CreateParented(Application.Handle);
  F_TableNames             := TStringList.Create;
  F_ActiveTableNames       := TStringList.Create;
  F_QueryDefNames          := TStringList.Create;
  F_DBTypesList            := TStringList.Create;
  F_DriverList             := TStringList.Create;
  F_Get_DBTypesList(F_DBTypesList);
  F_Get_OdbcDriverList(F_DriverList);
End;

Destructor  TKADaoDatabase.Destroy;
Begin
 If F_Active Then Connected := False;
 F_TableNames.Free;
 F_ActiveTableNames.Free;
 F_QueryDefNames.Free;
 F_LoginDialog.Free;
 F_DBTypesList.Free;
 F_DriverList.Free;
 Inherited Destroy;
End;

procedure TKADaoDatabase.F_Get_OdbcDriverList(List: TStrings);
var
   oReg : TRegistry;
Begin
     oReg := TRegistry.Create;
     try
     Begin
          oReg.RootKey := HKEY_LOCAL_MACHINE;
          if oReg.OpenKey('SOFTWARE\ODBC\ODBCINST.INI\ODBC Drivers',False) then
          Begin
               List.Clear;
               oReg.GetValueNames(List);
          End;
     End;
     finally
          oReg.Free;
     End;
End;

procedure TKADaoDatabase.F_Get_DBTypesList(List: TStrings);
var
   oReg : TRegistry;
Begin
     oReg := TRegistry.Create;
     try
     Begin
          oReg.RootKey := HKEY_LOCAL_MACHINE;
          {$IFDEF DAO35}
          if oReg.OpenKey('SOFTWARE\Microsoft\JET\3.5\ISAM Formats',False) then
          {$ENDIF}
          {$IFDEF DAO36}
          if oReg.OpenKey('SOFTWARE\Microsoft\JET\4.0\ISAM Formats',False) then
          {$ENDIF}
          Begin
               List.Clear;
               oReg.GetKeyNames(List);
          End;
     End;
     finally
          oReg.Free;
     End;
    List.Insert(0,'Access');
End;


Function TKADaoDatabase.F_Get_DBTypeFileExtension(DBType:String):String;
var
   oReg : TRegistry;
Begin
     oReg := TRegistry.Create;
     try
     Begin
          oReg.RootKey := HKEY_LOCAL_MACHINE;
          {$IFDEF DAO35}
          if oReg.OpenKey('SOFTWARE\Microsoft\JET\3.5\ISAM Formats\'+DBType,False) then
          {$ENDIF}
          {$IFDEF DAO36}
          if oReg.OpenKey('SOFTWARE\Microsoft\JET\4.0\ISAM Formats\'+DBType,False) then
          {$ENDIF}
          Begin
               Result:=oReg.ReadString('ExportFilter');
               if Result='' Then Result:=oReg.ReadString('ImportFilter');
          End;
     End;
     finally
          oReg.Free;
     End;
End;

Function TKADaoDatabase.F_Get_DBTypeTableType(DBType:String):String;
var
   oReg : TRegistry;
   BUF  : Array[1..1000] of Byte;
Begin
     oReg := TRegistry.Create;
     try
     Begin
          oReg.RootKey := HKEY_LOCAL_MACHINE;
          {$IFDEF DAO35}
          if oReg.OpenKey('SOFTWARE\Microsoft\JET\3.5\ISAM Formats\'+DBType,False) then
          {$ENDIF}
          {$IFDEF DAO36}
          if oReg.OpenKey('SOFTWARE\Microsoft\JET\4.0\ISAM Formats\'+DBType,False) then
          {$ENDIF}
          Begin
               oReg.ReadBinaryData('OneTablePerFile',BUF,1000);
               Result:=IntToStr(BUF[1]);
          End;
     End;
     finally
          oReg.Free;
     End;
End;

Procedure DatabaseError(Msg:String);
Begin
  Exception.Create(Msg);
End;

Procedure TKADaoDatabase.F_Set_Active(Value : Boolean);
Var
  X            : Integer;
  {$IFDEF USEDB}
  ATable       : TKADaoTable;
  {$ENDIF}
  Pwd          : String;
Begin
  if (F_Active) And (Value) Then Exit;
  if (F_Database='') Then
      Begin
       DatabaseError('TKADaoDatabase.F_Set_Active: Database is not defined!');
       Exit;
     End;
  if (F_DatabaseType='') Then
      Begin
       DatabaseError('TKADaoDatabase.F_Set_Active: Database type is not defined!');
       Exit;
     End;
  if (F_Active) And (NOT Value) Then
     Begin
       F_TableNames.Clear;
       F_QueryDefNames.Clear;
       {$IFDEF USEDB}
       For X:=0 to F_ActiveTableNames.Count-1 do
           Begin
             ATable:=TKADaoTable(F_ActiveTableNames.Objects[X]);
             ATable.MainDatabaseShutdown:=True;
             ATable.Active:=False;
           End;
       {$ENDIF}
       F_ActiveTableNames.Clear;
       CoreDatabase.Close;
       F_Active:=False;
     End;
  if (NOT F_Active) And (Value) Then
     Begin
        if F_LoginPrompt Then
           Begin
             F_LoginDialog.UserName.Text:=F_Username;
             F_LoginDialog.Password.Text:=F_Password;
             if F_LoginDialog.ShowModal=ID_OK Then
                Begin
                  F_Username:=F_LoginDialog.UserName.Text;
                  F_Password:=F_LoginDialog.Password.Text;
                End
             Else
                Begin
                  ShowMessage('If You not enter Username and Password You may not gain access to your data!');
                End;
           End;
        Pwd:=F_Password;
        if (F_Username <> '') And (F_Password='') Then Pwd:='';
        if (AnsiCompareText(F_DatabaseType,'Access')=0) Then
           Begin
             CoreDatabase := CoreWorkspace.OpenDatabase(F_Database,F_Exclusive,F_ReadOnly,Format(';UID=%s;PWD=%s',[F_Username,Pwd]));
           End
        Else
           Begin
             CoreDatabase := CoreWorkspace.OpenDatabase(F_Database,F_Exclusive,F_ReadOnly,Format('%s;UID=%s;PWD=%s',[F_DatabaseType,F_Username,Pwd]));
           End;
        if F_QueryTimeout > 0 Then CoreDatabase.QueryTimeout:=F_QueryTimeout;
        F_RefreshDefinitions;
        F_Active:=True;
    End;
End;

Procedure TKADaoDatabase.F_RefreshDefinitions;
Var
  X: Integer;
Begin
 CoreDatabase.TableDefs.Refresh;
 CoreDatabase.QueryDefs.Refresh;
 F_TableNames.Clear;
 For X:=0 To CoreDatabase.TableDefs.Count-1 do
     Begin
      if CoreDatabase.TableDefs.Item[X].Attributes And dbSystemObject = 0 Then
      F_TableNames.Add(CoreDatabase.TableDefs.Item[X].Name);
     End;
 F_QueryDefNames.Clear;
 For X:=0 To CoreDatabase.QueryDefs.Count-1 do
     Begin
      F_QueryDefNames.Add(CoreDatabase.QueryDefs.Item[X].Name);
     End;
End;

Procedure TKADaoDatabase.F_Set_Database(Value : String);
Begin
  if (F_Active) Then
     Begin
       DatabaseError('TKADaoDatabase.F_Set_Database: Cannot set Database while Database is connected!');
       Exit;
     End;
  F_Database:=Value;
End;

Procedure TKADaoDatabase.F_Set_Workspace(Value : String);
Begin
  if (F_Active) Then
     Begin
       DatabaseError('TKADaoDatabase.F_Set_Workspace: Cannot set Database while Database is connected!');
       Exit;
     End;
 Try
   CoreWorkspace := CoreDBEngine.Workspaces.Item[Value];
 Except
   Try
     CoreWorkspace:=CoreDBEngine.CreateWorkspace(Value,F_Username,F_Password,F_EngineType);
     CoreDBEngine.Workspaces.Append(CoreWorkspace);
   Except
     DatabaseError('TKADaoDatabase.F_Set_Workspace: Cannot create workspace! You may have no rights to create workspaces!');
   End;
 End;
 CoreWorkspace := CoreDBEngine.Workspaces.Item[Value];
 F_Workspace:=Value;
End;

Procedure TKADaoDatabase.F_Set_DatabaseType(Value : String);
Begin
  if (F_Active) Then
     Begin
       DatabaseError('TKADaoDatabase.F_Set_DatabaseType: Cannot set Database Type while Database is connected!');
       Exit;
     End;
  F_DatabaseType:=Value;
End;


Procedure TKADaoDatabase.F_Set_EngineType(Value : Integer);
Begin
  if (F_Active) Then
     Begin
       DatabaseError('TKADaoDatabase.F_Set_EngineType: Cannot set EngineType Type while Database is connected!');
       Exit;
     End;
  F_EngineType:=Value;
  CoreDBEngine.DefaultType := F_EngineType;
End;

Function  TKADaoDatabase.F_Get_DatabaseType:String;
Begin
  Result:=F_DatabaseType;
End;

Procedure TKADaoDatabase.F_Set_ReadOnly(Value : Boolean);
Begin
 if (F_Active) Then
     Begin
       DatabaseError('TKADaoDatabase.F_Set_ReadOnly: Cannot change mode while Database is connected!');
       Exit;
     End;
 F_ReadOnly:=Value;
End;

Procedure TKADaoDatabase.F_Set_Exclusive(Value : Boolean);
Begin
 if (F_Active) Then
     Begin
       DatabaseError('TKADaoDatabase.F_Set_Exclusive: Cannot change exclusive mode while Database is connected!');
       Exit;
     End;
 F_Exclusive:=Value;
End;

Procedure TKADaoDatabase.F_Set_LoginPrompt(Value : Boolean);
Begin
 if (F_Active) Then
     Begin
       DatabaseError('TKADaoDatabase.F_Set_LoginPrompt: Cannot change LoginPrompt while Database is connected!');
       Exit;
     End;
 F_LoginPrompt:=Value;
End;

Procedure TKADaoDatabase.F_Set_UserName(Value : ShortString);
Begin
 if (F_Active) Then
     Begin
       DatabaseError('TKADaoDatabase.F_Set_UserName: Cannot change User Name while Database is connected!');
       Exit;
     End;
 F_UserName:=Value;
End;

Procedure TKADaoDatabase.F_Set_MachineName(Value : ShortString);
Begin
 if (F_Active) Then
     Begin
       DatabaseError('TKADaoDatabase.F_Set_MachineName: Cannot change MachineName while Database is connected!');
       Exit;
     End;
 F_MachineName:=Value;
 CoreDBEngine             := CoDBEngine.CreateRemote(F_MachineName);
 CoreDBEngine.DefaultType := F_EngineType;
 CoreWorkspace            := CoreDBEngine.Workspaces.Item[0];
 F_Workspace              := CoreWorkspace.Name;
 F_DaoVersion             := CoreDBEngine.Version;
End;

Procedure TKADaoDatabase.F_Set_Password(Value : ShortString);
Begin
 if (F_Active) Then
     Begin
       DatabaseError(' TKADaoDatabase.F_Set_Password: Cannot change Password while Database is connected!');
       Exit;
     End;
 F_Password:=Value;
End;

Procedure TKADaoDatabase.StartTransaction;
Begin
  if (NOT F_Active) Then
     Begin
       DatabaseError('TKADaoDatabase.BeginTrans: Cannot start transactions while Database is NOT connected!');
       Exit;
     End;
  if CoreDatabase.Transactions Then CoreDatabase.BeginTrans;
End;

Procedure TKADaoDatabase.Commit;
Begin
 if (NOT F_Active) Then
     Begin
       DatabaseError('TKADaoDatabase.CommitTrans: Cannot commit transactions while Database is NOT connected!');
       Exit;
     End;
 if CoreDatabase.Transactions Then CoreDatabase.CommitTrans(dbForceOSFlush);
End;

Procedure TKADaoDatabase.Rollback;
Begin
 if (NOT F_Active) Then
     Begin
       DatabaseError('TKADaoDatabase.Rollback: Cannot rollback transactions while Database is NOT connected!');
       Exit;
     End;
 if CoreDatabase.Transactions Then CoreDatabase.Rollback;
End;

Procedure TKADaoDatabase.RepairAccessDatabase(DatabaseName:String);
Begin
  CoreDBEngine.RepairDatabase(DatabaseName);
End;

Procedure  TKADaoDatabase.CompactAccessDatabase(DatabaseName:String);
Var
  TempName : Array[0..1000] of Char;
  TempPath : String;
  Name     : String;
Begin
  TempPath:=ExtractFilePath(DatabaseName);
  GetTempFileName(PChar(TempPath),'mdb',0,TempName);
  Name:=StrPas(TempName);
  DeleteFile(Name);
  OleVariant(CoreDBEngine).CompactDatabase(DatabaseName,Name);
  DeleteFile(DatabaseName);
  RenameFile(Name,DatabaseName);
End;

Procedure TKADaoDatabase.CreateAccessDatabase(DatabaseName:String);
Var
 CreateOptions : String;
Begin
 CreateOptions:=Format(dbLangGeneral,['0x0402','1251','0']);
 {$IFDEF DAO35}
 CoreWorkspace.CreateDatabase(DatabaseName,CreateOptions, dbVersion30);
 {$ENDIF}
 {$IFDEF DAO36}
 CoreWorkspace.CreateDatabase(DatabaseName,CreateOptions, dbVersion40);
 {$ENDIF}
End;

Procedure TKADaoDatabase.CreateAccessDatabaseEx(DatabaseName,LANGID,CP,COUNTRY,Password,Version:String;Encrypt:Boolean);
Var
 CreateOptions:String;
Begin
 CreateOptions:=Format(dbLangGeneral,[LANGID,CP,COUNTRY]);
 if Password <> '' Then CreateOptions:=CreateOptions+';PWD='+Password;
 {$IFDEF DAO35}
 if Encrypt Then
    CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, dbVersion30 AND dbEncrypt)
 Else
    CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, dbVersion30);
 {$ENDIF}
 {$IFDEF DAO36}
  if Version='30' Then
     if Encrypt Then
        CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, dbVersion30 AND dbEncrypt)
     Else
        CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, dbVersion30)
  Else
     if Encrypt Then
        CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, dbVersion40 AND dbEncrypt)
     Else
        CoreWorkspace.CreateDatabase(DatabaseName, CreateOptions, dbVersion40);
 {$ENDIF}
End;

Procedure TKADaoDatabase.RenameTable(OldTableName,NewTableName:String);
Begin
 F_RefreshDefinitions;
 CoreDatabase.TableDefs.Item[OldTableName].Name:=NewTableName;
 F_RefreshDefinitions;
End;

Procedure TKADaoDatabase.DeleteTable(TableName:String);
Begin
 F_RefreshDefinitions;
 CoreDatabase.TableDefs.Delete(TableName);
 F_RefreshDefinitions;
End;

Procedure TKADaoDatabase.RenameIndex(TableName,OldIndexName,NewIndexName:String);
Begin
  F_RefreshDefinitions;
  CoreDatabase.TableDefs.Item[TableName].Indexes.Item[OldIndexName].Name:=NewIndexName;
  F_RefreshDefinitions;
End;

Procedure TKADaoDatabase.DeleteIndexByName(TableName,IndexName:String);
Begin
 F_RefreshDefinitions;
 CoreDatabase.TableDefs.Item[TableName].Indexes.Delete(IndexName);
 F_RefreshDefinitions;
End;

Procedure TKADaoDatabase.DeleteIndexByFieldName(TableName,FieldName:String);
Var
 X         : Integer;
 TmpName   : String;
 IndexName : String;
 NotFound  : Boolean;                                             
Begin
 F_RefreshDefinitions;
 Try
  Repeat
   NotFound:=True;
   CoreDatabase.TableDefs.Refresh;
   For X:=0 To CoreDatabase.TableDefs.Item[TableName].Indexes.Count-1 do
       Begin
         TmpName:=CoreDatabase.TableDefs.Item[TableName].Indexes.Item[X].Fields.Item[0].Name;
         if TmpName=FieldName Then
            Begin
              IndexName:=CoreDatabase.TableDefs.Item[TableName].Indexes.Item[X].Name;
              DeleteIndexByName(TableName,IndexName);
              NotFound:=False;
              Break;
            End;
       End;
  Until NotFound;
 Except
 End;
 F_RefreshDefinitions;
End;



Procedure TKADaoDatabase.DeleteField(TableName,FieldName:String);
Var
 X         : Integer;
 TmpName   : String;
 IndexName : String;
 NotFound  : Boolean;
Begin
 F_RefreshDefinitions;
 Try
  Repeat
   NotFound:=True;
   CoreDatabase.TableDefs.Refresh;
   For X:=0 To CoreDatabase.TableDefs.Item[TableName].Indexes.Count-1 do
       Begin
         TmpName:=CoreDatabase.TableDefs.Item[TableName].Indexes.Item[X].Fields.Item[0].Name;
         if TmpName=FieldName Then
            Begin
              IndexName:=CoreDatabase.TableDefs.Item[TableName].Indexes.Item[X].Name;
              DeleteIndexByName(TableName,IndexName);
              NotFound:=False;
              Break;
            End;
       End;
  Until NotFound;
 Except
 End;
 CoreDatabase.TableDefs.Item[TableName].Fields.Delete(FieldName);
 F_RefreshDefinitions;
End;

Procedure TKADaoDatabase.RenameField(TableName,OldFieldName,NewFieldName:String);
Begin
  F_RefreshDefinitions;
  CoreDatabase.TableDefs.Item[TableName].Fields.Item[OldFieldName].Name:=NewFieldName;
  F_RefreshDefinitions;
End;

{
1 = Primary index
2 = Unique
4 = NormalIndex
}

Function TKADaoDatabase.CreateTable(TableName:String; FieldNames : Variant; FieldTypes : Variant; FieldSizes : Variant; FieldIndexes:Variant):Boolean;
Var
  NT       : OleVariant;
  NewTable : OleVariant;
  NewField : OleVariant;
  NewIndex : OleVariant;
  PrimIndex: OleVariant;
  X        : Integer;
  Count    : Integer;
  Primary  : Boolean;
Begin
Primary    := False;
if (NOT F_Active) Then
     Begin
       DatabaseError('TKADaoDatabase.CreateTable: Cannot create table while Database is NOT connected!');
       CreateTable:=False;
       Exit;
     End;
 NT:=OleVariant(CoreDatabase).CreateTableDef(TableName);
 NewTable:=NT;
 Count:=VarArrayHighBound(FieldTypes,VarArrayDimCount(FieldTypes));
 For X:=0 to Count do
     Begin
      NewField  := NewTable.CreateField(FieldNames[X],FieldTypes[X],FieldSizes[X]);
      NewTable.Fields.AppEnd(NewField);
      if FieldIndexes[X] > 0 Then
         Begin
           NewIndex:=NewTable.CreateIndex(FieldNames[X]);
           if ((FieldIndexes[X] And 1) = 0) Then NewIndex.Primary := False
           Else
               Begin
                 if NOT Primary Then
                    Begin
                      PrimIndex          := NewTable.CreateIndex('PrimaryKey');
                      PrimIndex.Primary  := True;
                      PrimIndex.Unique   := True;
                      Primary:=True;
                    End;
                 NewField  := NewTable.CreateField(FieldNames[X],FieldTypes[X],FieldSizes[X]);
                 PrimIndex.Fields.AppEnd(NewField);
                 NewIndex.Primary  := True;
               End;
           if ((FieldIndexes[X] And 2) = 0) Then NewIndex.Unique  := False  Else  NewIndex.Unique  := True;
           NewField  := NewTable.CreateField(FieldNames[X],FieldTypes[X],FieldSizes[X]);
           NewIndex.Fields.AppEnd(NewField);
           NewTable.Indexes.AppEnd(NewIndex);
         End;
     End;
 if Primary Then NewTable.Indexes.AppEnd(PrimIndex);
 CoreDatabase.TableDefs.AppEnd(IDispatch(TVarData(NewTable).vDispatch));
 F_RefreshDefinitions;
 CreateTable:=True;
End;
                                                             
{
1 = Primary index
2 = Unique
4 = NormalIndex
}

Function TKADaoDatabase.AddFieldsToTable(TableName:String; FieldNames : Variant; FieldTypes : Variant; FieldSizes : Variant; FieldIndexes:Variant):Boolean;
Var
  NT       : OleVariant;
  NewTable : OleVariant;
  NewField : OleVariant;
  NewIndex : OleVariant;
  X        : Integer;
  Count    : Integer;
Begin
if (NOT F_Active) Then
     Begin
       DatabaseError('TKADaoDatabase.CreateTable: Cannot create table while Database is NOT connected!');
       AddFieldsToTable:=False;
       Exit;
     End;
 NT:=CoreDatabase.TableDefs.Item[TableName];
 NewTable:=NT;
 Count:=VarArrayHighBound(FieldTypes,VarArrayDimCount(FieldTypes));
 For X:=0 to Count do                             
     Begin
      NewField  := NewTable.CreateField(FieldNames[X],FieldTypes[X],FieldSizes[X]);
      NewTable.Fields.AppEnd(NewField);
      if FieldIndexes[X] > 0 Then
         Begin                               
                 if FieldIndexes[X]=1 Then  FieldIndexes[X]:=3;
                 NewIndex:=NewTable.CreateIndex(FieldNames[X]);
                 NewIndex.Primary := False;
                 if FieldIndexes[X]=3 Then NewIndex.Primary:=True;
                 if ((FieldIndexes[X] And 2) = 0) Then NewIndex.Unique  := False  Else  NewIndex.Unique  := True;
                 NewField  := NewTable.CreateField(FieldNames[X],FieldTypes[X],FieldSizes[X]);
                 NewIndex.Fields.AppEnd(NewField);
                 NewTable.Indexes.AppEnd(NewIndex);
         End;
     End;
 F_RefreshDefinitions;
 AddFieldsToTable:=True;
End;                                         

Function TKADaoDatabase.CreateQueryDef(Name:String;SQL:String):Boolean;
Var
  Query : QueryDef;
Begin
 Query:=CoreDatabase.CreateQueryDef(Name,SQL);
 F_RefreshDefinitions;
 CreateQueryDef:=True;
End;

Function TKADaoDatabase.GetQueryDefSQLText(Name:String):String;
Begin
 Try
   Result:=CoreDatabase.QueryDefs.Item[Name].SQL;
 Except
   Result:='';
 End;
End;

Procedure TKADaoDatabase.RenameQueryDef(OldQueryName,NewQueryName:String);
Begin
 F_RefreshDefinitions;
 CoreDatabase.QueryDefs.Item[OldQueryName].Name:=NewQueryName;
 F_RefreshDefinitions;
End;

Procedure TKADaoDatabase.DeleteQueryDef(QueryName:String);
Begin
 F_RefreshDefinitions;
 CoreDatabase.QueryDefs.Delete(QueryName);
 F_RefreshDefinitions;
End;

//************************************************************************************************* EDITORS
function GetExeDir: String;
begin
     Result := ExtractFilePath(ParamStr(0));
end;

function GetWorkDir: String;
var
   Tmp : String;
begin
     GetDir(0, Tmp);
end;


//*********************************************************************************** Database TYPE
Procedure TDatabaseTypeEditor.GetValues( Proc: TGetStrProc);
Var
  DBase : TKADaoDatabase;
  X     : Integer;
Begin
  if GetComponent(0) is TKADaoDatabase then
  Begin
    DBase := TKADaoDatabase(GetComponent(0));
    Try
      For X := 0 to DBase.F_DBTypesList.Count-1 do Proc(DBase.F_DBTypesList[X]);
    Finally
    End;
  End;
End;

Procedure TDatabaseTypeEditor.SetValue(const Value: string);
Var
  DBase : TKADaoDatabase;
Begin
 if GetComponent(0) is TKADaoDatabase then
  Begin
    DBase := TKADaoDatabase(GetComponent(0));
    if DBase.F_Active Then DBase.F_Set_Active(False); 
    DBase.F_Database:='';
    inherited SetValue(Value);
    Modified;
  End;
End;

Function TDatabaseTypeEditor.GetAttributes: TPropertyAttributes;
Begin
  Result:= Inherited GetAttributes + [paValueList, paSortList];
End;


//*********************************************************************************** Database
Function TDatabaseNameEditor.GetAttributes: TPropertyAttributes;
Begin
  Result:= [paDialog];
End;

Procedure TDatabaseNameEditor.SetValue(const Value: string);
Begin
if GetComponent(0) is TKADaoDatabase then
  Begin
    inherited SetValue(Value);
    Modified;
  End;
End;

procedure TDatabaseNameEditor.Edit;
var
   FileName : String;
   DBase    : TKADaoDatabase;
   Filter   : String;
   Temp     : String;
   P        : Integer;
   TableType: String;
Begin
     DBase:=TKADaoDatabase(GetComponent(0));
     If DBase.F_DatabaseType='' Then DatabaseError('Please select database type!');
     DlgChooseDatabase := TOpenDialog.Create(nil);
     FileName := DBase.Database;
     if FileName = '' then
        Begin
               DlgChooseDatabase.FileName   := '';
               if csDesigning in DBase.ComponentState Then
                  DlgChooseDatabase.InitialDir := GetWorkDir
               Else
                  DlgChooseDatabase.InitialDir := GetExeDir;
        End
     Else
        Begin
               DlgChooseDatabase.FileName   := ExtractFileName(FileName);
               DlgChooseDatabase.InitialDir := ExtractFileDir(FileName);
        End;
     if DBase.F_DatabaseType='Access' Then
        Begin
          Filter:='Microsoft Access (*.mdb)|*.mdb';
          Filter:=Filter+'|All files (*.*)|*.*';
          DlgChooseDatabase.Title:='Choose '+DBase.F_DatabaseType+' Database:';
          DlgChooseDatabase.Options:=[ofPathMustExist,ofHideReadOnly];
          DlgChooseDatabase.Filter :=Filter;
          if DlgChooseDatabase.Execute then SetStrValue(DlgChooseDatabase.FileName);
        End
     Else
        Begin
          Filter:=DBase.F_Get_DBTypeFileExtension(DBase.F_DatabaseType);
          TableType:=DBase.F_Get_DBTypeTableType(DBase.F_DatabaseType);
          if TableType='1' Then
             Begin
               if SelectDirectory(FileName,[],0) Then SetStrValue(FileName);
             End
          Else
             Begin
              Temp:=Filter;
              P:=Pos('(',Temp);
              if P > 0 Then
                 Begin
                  Delete(Temp,1,P);
                  P:=Pos(')',Temp);
                  if P > 0 Then Temp:=Copy(Temp,1,P-1);
                  Filter:=Filter+'|'+Temp;
                 End;
              Filter:=Filter+'|All files (*.*)|*.*';
              DlgChooseDatabase.Title:='Choose '+DBase.F_DatabaseType+' Database:';
              DlgChooseDatabase.Options:=[ofFileMustExist,ofPathMustExist,ofHideReadOnly];
              DlgChooseDatabase.Filter :=Filter;
              if DlgChooseDatabase.Execute then SetStrValue(DlgChooseDatabase.FileName);
             End;
        End;
     DlgChooseDatabase.Free;
     Modified;
End;


//*********************************************************************************** Engine Type
Function TEngineTypeEditor.GetAttributes: TPropertyAttributes;
Begin
  Result := Inherited GetAttributes + [paValueList, paSortList];
End;

Function  TEngineTypeEditor.GetValue: string;
Begin
 if GetComponent(0) is TKADaoDatabase then
    Begin
      if TKADaoDatabase(GetComponent(0)).F_EngineType=dbUseODBC then
         Result:='dbUseODBC'
      Else
         Result:='dbUseJet';
    End;
End;

Procedure TEngineTypeEditor.GetValues( Proc: TGetStrProc);
Begin
  if GetComponent(0) is TKADaoDatabase then
     Begin
       Proc('dbUseODBC');
       Proc('dbUseJet');
     End;
End;


procedure TEngineTypeEditor.SetValue(const Value: string);
Var
 Dat : Integer;
Begin
  if GetComponent(0) is TKADaoDatabase then
       Begin
       if Value='dbUseODBC' Then
          Dat:=dbUseODBC
       Else
          Dat:=dbUseJet;
       Inherited SetValue(IntToStr(Dat));
       Modified;
     End;
End;

//*********************************************************************************** Workspase
Procedure TWorkspaceEditor.GetValues( Proc: TGetStrProc);
Var
  DBase : TKADaoDatabase;
  X     : Integer;
Begin
  if GetComponent(0) is TKADaoDatabase then
  Begin
    DBase := TKADaoDatabase(GetComponent(0));
    Try
      For X := 0 to DBase.CoreDBEngine.Workspaces.Count-1 do
          Begin
            Proc(DBase.CoreDBEngine.Workspaces.Item[X].Name);
          End;
    Finally
    End;
  End;
End;

Function TWorkspaceEditor.GetAttributes: TPropertyAttributes;
Begin
  Result:= Inherited GetAttributes + [paValueList, paSortList];
End;


//************************************************************************************************* EDITORS

procedure Register;
Begin
  RegisterComponents('KA Dao', [TKADaoDatabase]);
  RegisterPropertyEditor(TypeInfo(String),TKADaoDatabase,'DatabaseType',TDatabaseTypeEditor);
  RegisterPropertyEditor(TypeInfo(String),TKADaoDatabase,'Database',TDatabaseNameEditor);
  RegisterPropertyEditor(TypeInfo(Integer),TKADaoDatabase, 'EngineType', TEngineTypeEditor);
  RegisterPropertyEditor(TypeInfo(String),TKADaoDatabase, 'Workspace', TWorkspaceEditor);
End;

Initialization
 {$IFNDEF VER130}
  TVarData(EmptyParam).VType := varError;
  TVarData(EmptyParam).VError := $80020004;
 {$ENDIF}
End.
