{*****************************************************************************
 TjgNTSharesC v1.0 by John Gonzalez 1999

 Description : Console mode object to retrieve shares on a specified machine.

 Events :
   None

 Methods :
   Execute - Retrieves the shares on specified machine.

 Properties :
   ErrorCode - Win32 error code if there was a problem in the Execute method.
   MachineName - Machine to retrieve shares from.
   Items - List of shares.

 ToDo :
   Add code to handle the shares security descriptor.

 Revision History :
   6/10/99 Initial release.

*****************************************************************************}
unit jgNTSharesC;

interface

uses Windows, SysUtils, Classes, jgNTApi, jgNTCommon, jgNTBaseC;

type
  TShareTypes = ( stDisk, stPrintQueue, stCommDevice, stIPC );
  TSharePermissions = ( spRead, spWrite, spCreate, spExec,
                        SpDelete, spAttrib, spPermissions );
  TSharePermissionsSet = set of TSharePermissions;

  { TShareItem }
  TShareItem = class(TCollectionItem)
  private
    FHighWord : Word;
    FPassword : string;
    FCurrentConnections : DWord;
    function GetIsDefaultShare : boolean;
  public
    name : string;
    shareType : TShareTypes;
    comment : string;
    permissions : TSharePermissionsSet;
    maxConnections : DWord;
    path : string;
    sd : PSECURITY_DESCRIPTOR;
    procedure CreateShare;
    procedure DeleteShare;
    procedure UpdateShare;
    property Password : string write FPassword;
    property currentConnections : DWord read FCurrentConnections;
    property IsDefaultShare : boolean read GetIsDefaultShare;
  end;

  TjgNTSharesC = class;

  { TShareItemList }
  TShareItemList = class(TCollection)
  private
    FOwner : TjgNTSharesC;
    function GetItem(index : integer): TShareItem;
    procedure SetItem(index : integer; value : TShareItem);
  public
    constructor Create(owner : TjgNTSharesC);
    function Add : TShareItem;
    property Items[index : integer] : TShareItem read GetItem write SetItem; default;
  end;

  { TjgNTShares }
  TjgNTSharesC = class(TjgNTBaseC)
  private
    FMachineName : string;
    FItems : TShareItemList;
  public
    constructor Create;
    destructor Destroy; override;
    function Execute : boolean;
    property MachineName : string read FMachineName write FMachineName;
    property Items : TShareItemList read FItems;
  end;

const
  ShareTypesText : array[TShareTypes] of string =
    ('Folder','Print','Comm','IPC');

implementation

{ TShareItem }

{*****************************************************************************}
procedure TShareItem.CreateShare;
var
  share : TShareInfo502;
  nameBuffer, commentBuffer, pathBuffer, machineBuffer : array[0..256] of WideChar;
  parmErr : DWord;
  comp : TjgNTSharesC;
  pMachine : PWideChar;
begin
  comp:= (collection as TShareItemList).FOwner;
  comp.FErrorCode:= 0;
  FillChar(share,SizeOf(TShareInfo502),0);

  { Convert our settings to the format that the share API understands }
  share.name:= StringToWideChar(name,nameBuffer,256);
  case shareType of
    stDisk : share.shareType:= STYPE_DISKTREE;
    stPrintQueue : share.shareType:= STYPE_PRINTQ;
    stCommDevice : share.shareType:= STYPE_DEVICE;
    stIPC : share.shareType:= STYPE_IPC;
  end;
  share.shareType:= share.shareType or FHighWord;
  share.remark:= StringToWideChar(comment,commentBuffer,256);

  if spRead in permissions then share.permissions:= ACCESS_READ;
  if spWrite in permissions then share.permissions:= share.permissions or ACCESS_WRITE;
  if spCreate in permissions then share.permissions:= share.permissions or ACCESS_CREATE;
  if spExec in permissions then share.permissions:= share.permissions or ACCESS_EXEC;
  if SpDelete in permissions then share.permissions:= share.permissions or ACCESS_DELETE;
  if spAttrib in permissions then share.permissions:= share.permissions or ACCESS_ATRIB;
  if spPermissions in permissions then share.permissions:= share.permissions or ACCESS_PERM;

  share.maxUses:= maxConnections;
  share.path:= StringToWideChar(path,pathBuffer,256);
  //shares[i].securityDescriptor:= sd;

  pMachine:= StringToWideChar(comp.FMachineName,machineBuffer,256);
  if NetShareAdd(pMachine,502,@share,parmErr) <> 0 then
    comp.FErrorCode:= GetLastError;
end;

{*****************************************************************************}
procedure TShareItem.DeleteShare;
var
  machineBuffer, nameBuffer : array[0..256] of WideChar;
  comp : TjgNTSharesC;
begin
  comp:= (collection as TShareItemList).FOwner;
  comp.FErrorCode:= 0;
  if NetShareDel(StringToWideChar(comp.FMachineName,machineBuffer,256),
                 StringToWideChar(name,nameBuffer,256),
                 0) <> 0 then comp.FErrorCode:= GetLastError;
  Free;
end;

{*****************************************************************************}
procedure TShareItem.UpdateShare;
var
  share : TShareInfo502;
  nameBuffer, commentBuffer, pathBuffer, machineBuffer : array[0..256] of WideChar;
  parmErr : DWord;
  comp : TjgNTSharesC;
begin
  comp:= (collection as TShareItemList).FOwner;
  comp.FErrorCode:= 0;
  FillChar(share,SizeOf(TShareInfo502),0);

  { Convert our settings to the format that the sharing API understands }
  share.name:= StringToWideChar(name,nameBuffer,256);
  case shareType of
    stDisk : share.shareType:= STYPE_DISKTREE;
    stPrintQueue : share.shareType:= STYPE_PRINTQ;
    stCommDevice : share.shareType:= STYPE_DEVICE;
    stIPC : share.shareType:= STYPE_IPC;
  end;
  share.shareType:= share.shareType or FHighWord;
  share.remark:= StringToWideChar(comment,commentBuffer,256);

  if spRead in permissions then share.permissions:= ACCESS_READ;
  if spWrite in permissions then share.permissions:= share.permissions or ACCESS_WRITE;
  if spCreate in permissions then share.permissions:= share.permissions or ACCESS_CREATE;
  if spExec in permissions then share.permissions:= share.permissions or ACCESS_EXEC;
  if spDelete in permissions then share.permissions:= share.permissions or ACCESS_DELETE;
  if spAttrib in permissions then share.permissions:= share.permissions or ACCESS_ATRIB;
  if spPermissions in permissions then share.permissions:= share.permissions or ACCESS_PERM;

  share.maxUses:= maxConnections;
  share.currentUses:= currentConnections;
  share.path:= StringToWideChar(path,pathBuffer,256);
  //shares[i].securityDescriptor:= sd;

  if NetShareSetInfo(StringToWideChar(comp.FMachineName,machineBuffer,256),
                     StringToWideChar(name,nameBuffer,256),502,
                     @share,parmErr) <> 0 then
    comp.FErrorCode:= GetLastError;
end;

{*****************************************************************************}
function TShareItem.GetIsDefaultShare : boolean;
begin
  result:= FHighWord > 0;
end;

{ TShareItemList }

{*****************************************************************************}
constructor TShareItemList.Create(owner : TjgNTSharesC);
begin
  FOwner:= owner;
  inherited Create(TShareItem);
end;

{*****************************************************************************}
function TShareItemList.Add : TShareItem;
begin
  result:= TShareItem(inherited Add);
end;

{*****************************************************************************}
function TShareItemList.GetItem(index : integer) : TShareItem;
begin
  result:= TShareItem(inherited GetItem(index));
end;

{*****************************************************************************}
procedure TShareItemList.SetItem(index : integer; value : TShareItem);
begin
  inherited SetItem(index,TCollectionItem(value));
end;

{ TjgNTSharesC }

{*****************************************************************************}
constructor TjgNTSharesC.Create;
begin
  FItems:= TShareItemList.Create(self);
end;

{*****************************************************************************}
destructor TjgNTSharesC.Destroy;
begin
  FItems.Free;
  inherited;
end;

{*****************************************************************************}
function TjgNTSharesC.Execute : boolean;
type
  TShareInfo502Array = array[0..0] of TShareInfo502;
var
  buffer : array[0..256] of WideChar;
	shares : ^TShareInfo502Array;
	i, rc, entriesRead, totalEntries, resumeHandle : DWord;
	loop : boolean;
begin
  FErrorCode:= 0;
  result:= false;
  FItems.Clear; { Clear current share list }
  loop:= true;    { This will control the loop until all jobs are read }
  resumeHandle:= 0;    { Start at the first job }
	while loop do
    begin
      { This function gets the jobs from the chosen machine }
      rc:= NetShareEnum(StringToWideChar(FMachineName,buffer,256),
                        502,pointer(shares),8192,entriesRead,totalEntries,resumeHandle);
      if (rc <> ERROR_SUCCESS) and (rc <> ERROR_MORE_DATA) then
        begin
          FErrorCode:= GetLastError;
          exit;
        end;

      { Loop thru the shares here and add them to our list, converting them into a
        "friendly" format }
		  for i:= 0 to entriesRead - 1 do
        with FItems.Add do
          begin
            name:= WideCharToString(shares[i].name);
            FHighWord:= shares[i].shareType and $ffff0000;
            case shares[i].shareType and $0000000f of
              STYPE_DISKTREE : shareType:= stDisk;
              STYPE_PRINTQ : shareType:= stPrintQueue;
              STYPE_DEVICE : shareType:= stCommDevice;
              STYPE_IPC : shareType:= stIPC;
            end;
            comment:= WideCharToString(shares[i].remark);
            if (shares[i].permissions and ACCESS_READ) > 0 then permissions:= [spRead];
            if (shares[i].permissions and ACCESS_WRITE) > 0 then permissions:= permissions + [spWrite];
            if (shares[i].permissions and ACCESS_CREATE) > 0 then permissions:= permissions + [spCreate];
            if (shares[i].permissions and ACCESS_EXEC) > 0 then permissions:= permissions + [spExec];
            if (shares[i].permissions and ACCESS_DELETE) > 0 then permissions:= permissions + [spDelete];
            if (shares[i].permissions and ACCESS_ATRIB) > 0 then permissions:= permissions + [spAttrib];
            if (shares[i].permissions and ACCESS_PERM) > 0 then permissions:= permissions + [spPermissions];
            maxConnections:= shares[i].maxUses;
            FCurrentConnections:= shares[i].currentUses;
            path:= WideCharToString(shares[i].path);
            //sd:= shares[i].securityDescriptor;
          end;
		  if shares <> nil then NetApiBufferFree(shares);
		  if rc = ERROR_SUCCESS then loop:= false;
      resumeHandle:= entriesRead + 1;
	  end;
  result:= true;
end;

end.
