{*****************************************************************************
 TjgNTJobsC v1.0 by John Gonzalez 1999

 Description : Console mode NetJob* API functions wrapper.

 Events :
   None

 Methods :

 Properties :

 Revision History :
   6/10/99  Initial release

*****************************************************************************}
unit jgNTJobsC;

interface

uses Windows, SysUtils, Classes, WinSvc, jgNTApi, jgNTCommon, jgNTBaseC;

type
  TDaysOfWeek = (dowMonday,dowTuesday,dowWednesday,dowThursday,dowFriday,dowSaturday,dowSunday);
  TDaysOfWeekSet = set of TDaysOfWeek;

  TJobItem = class(TCollectionItem)
  private
    FJobId : DWord;
    FError : boolean;
    FToday : boolean;
    procedure _DeleteJob;
    function  GetDay(index: integer): boolean;
    procedure SetDay(index: integer; value: boolean);
  public
    daysOfWeek : TDaysOfWeekSet;
    daysOfMonth : DWord;
    every : boolean;
    interactive : boolean;
    jobTime : TDateTime;
    command : string;
    function  AddJob: boolean;
    procedure DeleteJob;
    function  UpdateJob: boolean;
    property  Days[index : integer] : boolean read GetDay write SetDay;
    property  Error : boolean read FError;
    property  jobID : DWord read FJobId;
    property  today : boolean read FToday;
  end;

  TjgNTJobsC = class;

  TJobItemList = class(TCollection)
  private
    FOwner : TjgNTJobsC;
    function  GetItem(index : integer): TJobItem;
    procedure SetItem(index : integer; value : TJobItem);
  public
    constructor Create(owner : TjgNTJobsC);
    function Add : TJobItem;
    property Items[index : integer] : TJobItem read GetItem write SetItem; default;
  end;

  TjgNTJobsC = class(TjgNTBaseC)
  private
    FMachineName : string;
    FItems : TJobItemList;
  public
    constructor Create;
    destructor  Destroy; override;
    function  Execute : boolean;
    function  IsServiceRunning: boolean;
    function  StartJobService: boolean;
    property  Items : TJobItemList read FItems;
    property  MachineName : string read FMachineName write FMachineName;
  end;

implementation

{ TJobItem }

{*****************************************************************************}
function TJobItem.AddJob : boolean;
var
  info : TAtInfo;
  timeStamp : TTimeStamp;
  commandBuffer, machineBuffer : array[0..256] of WideChar;
  comp : TjgNTJobsC;
begin
  { Convert our "friendly" settings back to the format that the scheduling API
    understands }
  timeStamp:= DateTimeToTimeStamp(jobTime);
  info.jobTime:= timeStamp.time;

  info.daysOfWeek:= 0;
  if dowMonday in daysOfWeek then info.daysOfWeek:= 1;
  if dowTuesday in daysOfWeek then info.daysOfWeek:= info.daysOfWeek or 2;
  if dowWednesday in daysOfWeek then info.daysOfWeek:= info.daysOfWeek or 4;
  if dowThursday in daysOfWeek then info.daysOfWeek:= info.daysOfWeek or 8;
  if dowFriday in daysOfWeek then info.daysOfWeek:= info.daysOfWeek or 16;
  if dowSaturday in daysOfWeek then info.daysOfWeek:= info.daysOfWeek or 32;
  if dowSunday in daysOfWeek then info.daysOfWeek:= info.daysOfWeek or 64;

  info.daysOfMonth:= daysOfMonth;
  if every then info.flags:= JOB_RUN_PERIODICALLY;
  if not interactive then info.flags:= info.flags or JOB_NONINTERACTIVE;

  info.command:= StringToWideChar(command,commandBuffer,256);
  comp:= (collection as TJobItemList).FOwner;
  StringToWideChar(comp.FMachineName,machineBuffer,256);
  if NetScheduleJobAdd(PWideChar(@machineBuffer),pointer(@info),FJobId) <> 0 then
    comp.FErrorCode:= GetLastError;
  result:= jobId > 0;
  if result = false then Free;
end;

{*****************************************************************************}
procedure TJobItem.DeleteJob;
begin
  _DeleteJob;
  Free;
end;

{*****************************************************************************}
procedure TJobItem._DeleteJob;
var
  buffer : array[0..256] of WideChar;
  comp : TjgNTJobsC;
begin
  comp:= (collection as TJobItemList).FOwner;
  comp.FErrorCode:= 0;
  if NetScheduleJobDel(StringToWideChar(comp.FMachineName,buffer,256),jobId,jobId) <> 0 then
    comp.FErrorCode:= GetLastError;
end;

{*****************************************************************************}
function TJobItem.UpdateJob : boolean;
begin
  { There isn't an API to change job parameters so we fake it here by deleting
    the job and then resubmitting it }
  _DeleteJob;
  result:= AddJob;
end;

{*****************************************************************************}
function TJobItem.GetDay(index : integer) : boolean;
begin
  { Decrease index because the first day is accessed as 1 and we need to access
    a 0-based array }
  Dec(index);
  result:= (daysOfMonth and (1 shl index)) > 0;
end;

{*****************************************************************************}
procedure TJobItem.SetDay(index : integer; value : boolean);
begin
  { Decrease index because the first day is accessed as 1 and we need to access
    a 0-based array }
  Dec(index);
  if value then daysOfMonth:= daysOfMonth or (1 shl index)
  else daysOfMonth:= daysOfMonth and (not (1 shl index));
end;

{ TJobItemList }

{*****************************************************************************}
constructor TJobItemList.Create(owner : TjgNTJobsC);
begin
  FOwner:= owner;
  inherited Create(TJobItem);
end;

{*****************************************************************************}
function TJobItemList.Add : TJobItem;
begin
  result:= TJobItem(inherited Add);
end;

{*****************************************************************************}
function TJobItemList.GetItem(index : integer) : TJobItem;
begin
  result:= TJobItem(inherited GetItem(index));
end;

{*****************************************************************************}
procedure TJobItemList.SetItem(index : integer; value : TJobItem);
begin
  inherited SetItem(index,TCollectionItem(value));
end;

{ TjgNTJobsC }

{*****************************************************************************}
constructor TjgNTJobsC.Create;
begin
  FItems:= TJobItemList.Create(self);
  inherited Create;
end;

{*****************************************************************************}
destructor TjgNTJobsC.Destroy;
begin
  FItems.Free;
  inherited Destroy;
end;

{*****************************************************************************}
function TjgNTJobsC.Execute : boolean;
type
  TAtEnumArray = array[0..0] of TAtEnum;
var
  buffer : array[0..256] of WideChar;
	Job : ^TAtEnumArray;
	i, rc, nRead, nLeftBeforeCall, hResume : DWord;
	loop : boolean;
  timeStamp : TTimeStamp;
begin
  FItems.Clear;   { Clear current Job list }
  loop:= true;    { This will control the loop until all Job are read }
  hResume:= 0;    { Start at the first Job }
  FErrorCode:= 0;
  result:= false;
	while loop do
    begin
      { This function gets the Job from the chosen machine }
      rc:= NetScheduleJobEnum(StringToWideChar(FMachineName,buffer,256),
                              pointer(Job),8192,nRead,nLeftBeforeCall,hResume);
      if (rc <> ERROR_SUCCESS) and (rc <> ERROR_MORE_DATA) then
        begin
          FErrorCode:= GetLastError;
          exit;
        end;
      if nRead = 0 then exit;

      { Loop thru the Job here and add them to our list, converting them into a
        "friendly" format }
		  for i:= 0 to nRead - 1 do
        with FItems.Add do
          begin
            FJobId:= Job[i].JobId;
            daysOfMonth:= Job[i].daysOfMonth;
            timeStamp:= MSecsToTimeStamp(Job[i].JobTime);
            JobTime:= TimeStampToDateTime(timeStamp);
            if (Job[i].daysOfWeek and 1) > 0 then daysOfWeek:= [dowMonday];
            if (Job[i].daysOfWeek and 2) > 0 then daysOfWeek:= daysOfWeek + [dowTuesday];
            if (Job[i].daysOfWeek and 4) > 0 then daysOfWeek:= daysOfWeek + [dowWednesday];
            if (Job[i].daysOfWeek and 8) > 0 then daysOfWeek:= daysOfWeek + [dowThursday];
            if (Job[i].daysOfWeek and 16) > 0 then daysOfWeek:= daysOfWeek + [dowFriday];
            if (Job[i].daysOfWeek and 32) > 0 then daysOfWeek:= daysOfWeek + [dowSaturday];
            if (Job[i].daysOfWeek and 64) > 0 then daysOfWeek:= daysOfWeek + [dowSunday];

            if (Job[i].flags and Job_RUN_PERIODICALLY) > 0 then every:= true;
            if (Job[i].flags and Job_EXEC_ERROR) > 0 then FError:= true;
            if (Job[i].flags and Job_NONINTERACTIVE) = 0 then interactive:= true;
            if (Job[i].flags and Job_RUNS_TODAY) > 0 then FToday:= true;
            command:= WideCharToString(Job[i].command);
          end;
		  if Job <> nil then NetApiBufferFree(Job);
		  if rc = ERROR_SUCCESS then loop:= false;
	  end;
  result:= true;
end;

{*****************************************************************************}
function TjgNTJobsC.StartJobService : boolean;
var
  serviceDBHandle, serviceHandle : SC_HANDLE;
  serviceStatus : TServiceStatus;
  pTemp : PChar;
  loop : boolean;
begin
  pTemp:= nil;
  FErrorCode:= 0;
  result:= false;

  { Open the service database on the chosen machine }
  serviceDBHandle:= OpenSCManager(PChar(FMachineName),nil,SC_MANAGER_CONNECT);
  if serviceDBHandle = 0 then
    begin
      FErrorCode:= GetLastError;
      exit;
    end;

  { Open the service on the chosen machine }
  serviceHandle:= OpenService(serviceDBHandle,'Schedule',SERVICE_QUERY_STATUS or
                              SERVICE_START);
  if serviceDBHandle = 0 then
    begin
      FErrorCode:= GetLastError;
      CloseServiceHandle(serviceDBHandle);
      exit;
    end;

  if not StartService(serviceHandle,0,pTemp) then
    begin
      FErrorCode:= GetLastError;
      CloseServiceHandle(serviceHandle);
      CloseServiceHandle(serviceDBHandle);
      exit;
    end;

  { We loop here until the service is actually started }
  loop:= true;
  repeat
    if not QueryServiceStatus(serviceHandle,serviceStatus) then
      begin
        FErrorCode:= GetLastError;
        CloseServiceHandle(serviceHandle);
        CloseServiceHandle(serviceDBHandle);
        exit;
      end;
    if serviceStatus.dwCurrentState = SERVICE_RUNNING then loop:= false
    else Sleep(serviceStatus.dwWaitHint);
  until not loop;
  CloseServiceHandle(serviceHandle);
  CloseServiceHandle(serviceDBHandle);
  result:= true;
end;

{*****************************************************************************}
function TjgNTJobsC.IsServiceRunning : boolean;
var
  serviceDBHandle, serviceHandle : SC_HANDLE;
  serviceStatus : TServiceStatus;
begin
  FErrorCode:= 0;
  result:= false;

  { Open the service database on the chosen machine }
  serviceDBHandle:= OpenSCManager(PChar(FMachineName),nil,SC_MANAGER_CONNECT);
  if serviceDBHandle = 0 then
    begin
      FErrorCode:= GetLastError;
      exit;
    end;

  { Open the service on the chosen machine }
  serviceHandle:= OpenService(serviceDBHandle,'Schedule',SERVICE_QUERY_STATUS or
                              SERVICE_START);
  if serviceDBHandle = 0 then
    begin
      FErrorCode:= GetLastError;
      CloseServiceHandle(serviceDBHandle);
      exit;
    end;

  { Get current state of the service first }
  if not QueryServiceStatus(serviceHandle,serviceStatus) then
    begin
      FErrorCode:= GetLastError;
      CloseServiceHandle(serviceHandle);
      CloseServiceHandle(serviceDBHandle);
      exit;
    end;
  result:= serviceStatus.dwCurrentState = SERVICE_RUNNING;
  CloseServiceHandle(serviceHandle);
  CloseServiceHandle(serviceDBHandle);
end;

end.
