{*****************************************************************************
 TjgNTGui v1.0 by John Gonzalez 1999

 Description : Encapsulates the add user/group dialog found in netui2.dll.

 Events :
   None

 Methods :
   Execute - Displays the add user/group dialog box with the chosen settings.

 Properties :
   Domain - Initial domain to display.
   Options - Various options for the dialog box.
   Title - Text to display as the caption of the dialog.
   UserGroupList - List of chosen usernames.

 Notes :
   This code is based on an MFC sample found on  
   http://www.geocities.com/SiliconValley/1741/miscprog/mp_main.html

 Revision History :
   6/10/99 Initial release

*****************************************************************************}
unit jgNTGui;

interface

uses Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
     jgNTBase, jgNTCommon, jgNTApi;

type
  { TOpenUserBrowser }
  POpenUserBrowser = ^TOpenUserBrowser;
  TOpenUserBrowser = record
	  cbSize : DWord;
	  fCancelled : DWord;
	  unknown2 : DWord;
    hWndParent : THandle;
	  szTitle : PWideChar;
	  szDomainName : PWideChar;
    dwFlags : DWord;
	  dwHelpId : DWord;
	  szHelpFile : PWideChar;
  end;

  { TEnumUserBrowser }
  PEnumUserBrowser = ^TEnumUserBrowser;
  TEnumUserBrowser = record
    sidType : SID_NAME_USE;
	  sid1 : PSid;
	  sid2 : PSid;
	  szFullName : PWideChar;
    szUserName : PWideChar;
	  szDisplayName : PWideChar;
	  szDomainName : PWideChar;
    szDescription : PWideChar;
  end;

  TOpenUserBrowserPtr = function(var value : TOpenUserBrowser) : THandle; stdcall;
  TEnumUserBrowserSelectionPtr = function(handle : THandle; p : pointer;
                                          var w : DWord) : BOOL; stdcall;
  TCloseUserBrowserPtr = function(handle : THandle) : BOOL; stdcall;

  TUserBrowserOptions = ( uboIncludeSystem, uboSingleSelection, uboNoLocalDomain,
                          uboIncludeCreatorOwner, uboIncludeEveryone, uboIncludeInteractive,
                          uboIncludeNetwork, uboIncludeUsers, uboIncludeUserButtons,
                          uboIncludeGroups, uboIncludeAliases );
  TUserBrowserOptionsSet = set of TUserBrowserOptions;

  TSidTypes = (stUser,stGroup,stDomain,stAlias,stWellKnownGroup,
               stDeletedAccount,stInvalid,stUnknown);

  TUserGroupItem = class(TCollectionItem)
  public
    UserType : TSidTypes;
    UserName : string;
    FullName : string;
    Domain : string;
    Description : string;
  end;

  TUserGroupItemList = class(TCollection)
  private
    function GetItem(index : integer): TUserGroupItem;
    procedure SetItem(index : integer; value : TUserGroupItem);
  public
    function Add : TUserGroupItem;
    property Items[index : integer] : TUserGRoupItem read GetItem write SetItem; default;
  end;

  TjgNTGui = class(TjgNTBase)
  private
    hBrowser : THandle;
	  hNetu1Dll : THandle;
	  pOpenUserBrowser : TOpenUserBrowserPtr;
	  pEnumUserBrowserSelection : TEnumUserBrowserSelectionPtr;
	  pCloseUserBrowser : TCloseUserBrowserPtr;
    FUserGroupList : TUserGroupItemList;
    FTitle: string;
    FDomain: string;
    FOptions: TUserBrowserOptionsSet;
  public
    constructor Create(aOwner : TComponent); override;
    destructor Destroy; override;
    function Execute : boolean;
    property UserGroupList : TUserGroupItemList read FUserGroupList;
  published
    property Domain : string read FDomain write FDomain;
    property Options : TUserBrowserOptionsSet read FOptions write FOptions;
    property Title : string read FTitle write FTitle;
  end;

const
  OPENUSERBROWSER_INCLUDE_SYSTEM        = 00010000;
  OPENUSERBROWSER_SINGLE_SELECTION      = 00001000;
  OPENUSERBROWSER_NO_LOCAL_DOMAIN       = 00000100;
  OPENUSERBROWSER_INCLUDE_CREATOR_OWNER = 00000080;
  OPENUSERBROWSER_INCLUDE_EVERYONE      = 00000040;
  OPENUSERBROWSER_INCLUDE_INTERACTIVE   = 00000020;
  OPENUSERBROWSER_INCLUDE_NETWORK       = 00000010;
  OPENUSERBROWSER_INCLUDE_USERS         = 00000008;
  OPENUSERBROWSER_INCLUDE_USER_BUTTONS  = 00000004;
  OPENUSERBROWSER_INCLUDE_GROUPS        = 00000002;
  OPENUSERBROWSER_INCLUDE_ALIASES       = 00000001;

  TOptionsArray : array[0..10] of DWord =
    ( OPENUSERBROWSER_INCLUDE_SYSTEM,
      OPENUSERBROWSER_SINGLE_SELECTION,
      OPENUSERBROWSER_NO_LOCAL_DOMAIN,
      OPENUSERBROWSER_INCLUDE_CREATOR_OWNER,
      OPENUSERBROWSER_INCLUDE_EVERYONE,
      OPENUSERBROWSER_INCLUDE_INTERACTIVE,
      OPENUSERBROWSER_INCLUDE_NETWORK,
      OPENUSERBROWSER_INCLUDE_USERS,
      OPENUSERBROWSER_INCLUDE_USER_BUTTONS,
      OPENUSERBROWSER_INCLUDE_GROUPS,
      OPENUSERBROWSER_INCLUDE_ALIASES );

  SidTypeNames : array[stUser..stUnknown] of string =
    ('User','Group','Domain','Alias','WellKnownGroup',
     'Deleted Account','Invalid','Unknown');

implementation

{ TUserGroupItemList }

{*****************************************************************************}
function TUserGroupItemList.Add : TUserGroupItem;
begin
  result:= TUserGroupItem(inherited Add);
end;

{*****************************************************************************}
function TUserGroupItemList.GetItem(index : integer) : TUserGroupItem;
begin
  result:= TUserGroupItem(inherited GetItem(index));
end;

{*****************************************************************************}
procedure TUserGroupItemList.SetItem(index : integer; value : TUserGroupItem);
begin
  inherited SetItem(index,TCollectionItem(value));
end;

{ TjgNTGui }

{****************************************************************}
constructor TjgNTGui.Create(aOwner: TComponent);
begin
  inherited Create(aOwner);

  { Don't load dll at design time }
  if csDesigning in componentState then exit;

  { Attempt to load dll }
  hNetu1Dll:= LoadLibrary('netui2.dll');
  if hNetu1Dll = 0 then raise Exception.Create('Error loading dll');

  { Get pointers to desired procedures }
  pOpenUserBrowser:=
    TOpenUserBrowserPtr(GetProcAddress(hNetu1Dll,'OpenUserBrowser'));
  pEnumUserBrowserSelection:=
    TEnumUserBrowserSelectionPtr(GetProcAddress(hNetu1Dll,'EnumUserBrowserSelection'));
  pCloseUserBrowser:=
    TCloseUserBrowserPtr(GetProcAddress(hNetu1Dll,'CloseUserBrowser'));

  { See if all the procedures were found }
  if not Assigned(pOpenUserBrowser) or
     not Assigned(pEnumUserBrowserSelection) or
     not Assigned(pCloseUserBrowser) then
    begin
      FreeLibrary(hNetu1Dll);
      if hNetu1Dll = 0 then raise Exception.Create('Error loading dll');
    end;

  FUserGroupList:= TUserGroupItemList.Create(TUserGroupItem);
end;

{****************************************************************}
destructor TjgNTGui.Destroy;
begin
  inherited Destroy;
  if csDesigning in componentState then exit;
  if hNetu1Dll > 0 then FreeLibrary(hNetu1Dll);
  FUserGroupList.Free;
end;

{****************************************************************}
function TjgNTGui.Execute: boolean;
var
  oub : TOpenUserBrowser;
  userBuffer : PEnumUserBrowser;
  size, error : DWord;
  titleBuffer, domainBuffer : array[0..255] of Widechar;
  i : TUserBrowserOptions;
begin
  result:= false;
  size:= SizeOf(titleBuffer);

  { Initialize record }
  FillChar(oub,SizeOf(TOpenUserBrowser),0);
  with oub do
    begin
      cbSize:= SizeOf(TOpenUserBrowser);
      hWndParent:= application.handle;
      szTitle:= StringToWideChar(FTitle,titleBuffer,size);
      szDomainName:= StringToWideChar(FDomain,domainBuffer,size);
    end;

  { Convert options to DWord bitflag }
  for i:= Low(TUserBrowserOptions) to High(TUserBrowserOptions) do
    if i in FOptions then oub.dwFlags:= oub.dwFlags or TOptionsArray[Ord(i)];

  { Call OpenUserBrowser. If result is 0 then either user canceled it away
    or there was an error. Use GetLastError to determine which }
  hBrowser:= pOpenUserBrowser(oub);
  if hBrowser = 0 then
    begin
      error:= GetLastError;
      if error = 0 then exit;
      raise Exception.Create('OpenUserBrowser - ' + SysErrorMessage(error));
    end;

  { Retrieve users and/or groups }
  size:= 8192;
  GetMem(userBuffer,size);
  try
    FUserGroupList.Clear;
    while pEnumUserBrowserSelection(hBrowser,userBuffer,size) do
      with FUserGroupList.Add do
        begin
          UserType:= TSidTypes(userBuffer^.sidType - 1);
          UserName:= WideCharToString(userBuffer^.szUserName);
          FullName:= WideCharToString(userBuffer^.szFullName);
          Domain:= WideCharToString(userBuffer^.szDomainName);
          Description:= WideCharToString(userBuffer^.szDescription);
        end;
    { Possible errors here...
      122 - Buffer passed is too small.
      259 - No more Data is available. Normal.
      error:= GetLastError;
      ShowMessage(SysErrorMessage(error) + ' - ' + IntToStr(error));}
  finally
    FreeMem(userBuffer);
  end;
  pCloseUserBrowser(hBrowser);
  result:= true;
end;

end.
