unit
  utilHttp;
(*##*)
(*******************************************************************
*                                                                 *
*   U  T  I  L  H  T  T  P   http loader component                 *
*                                                                 *
*   Copyright (c) 1999, 2000 A.Ivanov. All rights reserved.        *
*   Based on Delphi 4 TPageProducer component                     *
*   Conditional defines:                                           *
*                                                                 *
*   Last Revision: May 19 1999                                     *
*   Last fix     :                                                *
*   Lines        :                                                 *
*   History      : see CHANGES.TXT file                           *
*   Printed      : ---                                             *
*                                                                 *
********************************************************************)
(*##*)

interface
uses
  Classes,  Windows, SysUtils, NMHttp, PSock, Controls, ExtCtrls,
  scktcomp,
  util1, GifImage;

type
  THttpGifLoader = class(TImage)
  private
    FStarted: Boolean;
    FLoaded: Boolean;
    FUrl: String;
    FData: String;
    FClientSocket: TClientSocket;
    FProxy: String;
    FProxyPort: Integer;
    FGifImage: TGifImage;
    FLastLoad: TDateTime;
    FNextTimeOutSec: Integer;
    function GetDataPtr: Pointer;
    function GetDataLen: Integer;
    procedure SetUrl(AValue: String);
    procedure SetProxy(AValue: String);
    procedure SetProxyPort(AValue: Integer);
    procedure SetTimeOutSec(AValue: Integer);
    procedure SetNextTimeOutSec(AValue: Integer);
    procedure SetStarted(AStart: Boolean);
    procedure FClientSocketWrite(Sender: TObject; Socket: TCustomWinSocket);
    procedure FClientSocketDisconnect(Sender: TObject; Socket: TCustomWinSocket);
    procedure FClientSocketError(Sender: TObject; Socket: TCustomWinSocket; ErrorEvent: TErrorEvent; var ErrorCode: Integer);
    procedure FClientSocketRead(Sender: TObject; Socket: TCustomWinSocket);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Clear;
    procedure ReadProxySettings;
    property DataPtr: Pointer read GetDataPtr;
  published
    property Started: Boolean read FStarted write SetStarted;
    property IsLoaded: Boolean read FLoaded;
    property Data: String read FData;
    property DataLen: Integer read GetDataLen;
    property Url: String write SetUrl;
    property Proxy: String write SetProxy;
    property ProxyPort: Integer write SetProxyPort;
    property TimeOutSec: Integer write SetTimeOutSec;
    property NextTimeOutSec: Integer read FNextTimeOutSec write SetNextTimeOutSec;
  end;

implementation

function ExtractHeader(const Header, Fld: String): String;
var
  sl: TStrings;
  i: Integer;
  p: Integer;
  uc, uc1: String;
begin
  Result:= '';
  sl:= TStringList.Create;
  sl.Text:= Header;
  uc:= ANSIUpperCase(Fld);
  for i:= 0 to sl.Count - 1 do begin
    uc1:= ANSIUppercase(sl[i]);
    if Pos(uc, uc1) = 1 then begin
      p:= Pos(':', uc1);
      if p>0 then begin
        Result:= Copy(sl[i], p+2, MaxInt);
        util1.DeleteLeadTerminateSpaceStr(Result);
        Exit;
      end;
    end;
  end;
  sl.Free;
end;

procedure THttpGifLoader.SetStarted(AStart: Boolean);
begin
  if AStart then begin
    if FStarted
    then Exit;

    if Now < FLastLoad + (FNextTimeOutSec/sysutils.SecsPerDay)
    then Exit;

    FData:= '';
    try
      FClientSocket.Active:= True;
      FClientSocket.Socket.SendText('');
      FLastLoad:= Now;
    except
      Exit;
    end;
    FStarted:= True;
    FLoaded:= False;
  end else begin
    if FClientSocket.Active then begin
      try
        FClientSocket.Active:= False;
        // FLastLoad:= 0.0;
      except
      end;
    end;
    FStarted:= False;
    FLoaded:= False;
  end;
end;

procedure THttpGifLoader.SetUrl(AValue: String);
var
  protocol, host, IP, fn: String;
  port: Integer;
begin
  util1.ParseUrl(AValue, protocol, host, IP, fn, port);
  if ansiCompareStr(protocol, 'http') <> 0
  then raise Exception.CreateFmt('invalid protocol: %s', [protocol]);
  if FProxy = '' then begin
    FClientSocket.Host:= host;
    FClientSocket.Port:= port;
    FUrl:= fn;
  end else begin
    FClientSocket.Host:= FProxy;
    FClientSocket.Port:= FProxyPort;
    FUrl:= host+':'+IntToStr(port)+fn;
  end;
end;

procedure THttpGifLoader.SetProxy(AValue: String);
begin
  FProxy:= AValue;
end;

procedure THttpGifLoader.SetProxyPort(AValue: Integer);
begin
  FProxyPort:= AValue;
end;

procedure THttpGifLoader.SetTimeOutSec(AValue: Integer);
begin
  // FClientSocket.Socket.TimeOut:= AValue * 1000; // ms
end;

procedure THttpGifLoader.SetNextTimeOutSec(AValue: Integer);
begin
  FNextTimeOutSec:= AValue;
  FLastLoad:= Now;
end;

procedure THttpGifLoader.FClientSocketDisconnect(Sender: TObject; Socket: TCustomWinSocket);
var
  p: Integer;
  t: TStream;
begin
  p:= Pos(#13#10#13#10, FData);
  if (p > 0) and
    (ANSICompareText(ExtractHeader(Copy(FData, 1, p), 'Content-Type'), 'image/gif') = 0) then begin
    Delete(FData, 1, p+3);
    t:= TStringStream.Create(FData);
    FGifImage.LoadFromStream(t);
    Picture.Graphic:= FGifImage;
    Self.Width:= FGifImage.Width;
    Self.Height:= FGifImage.Height;
    t.Free;
  end;
  FStarted:= False;
  FLoaded:= True;
end;

procedure THttpGifLoader.FClientSocketError(Sender: TObject; Socket: TCustomWinSocket; ErrorEvent: TErrorEvent; var ErrorCode: Integer);
begin
  // ClientSocket1.Active:= False;
  ErrorCode:= 0;
  FStarted:= False;
  FLoaded:= False;
  // SetStarted(True);
end;

procedure THttpGifLoader.Clear;
begin
  FData:= '';
end;

function THttpGifLoader.GetDataPtr: Pointer;
begin
  Result:= @FData;
end;

function THttpGifLoader.GetDataLen: Integer;
begin
  Result:= Length(FData);
end;

constructor THttpGifLoader.Create(AOwner: TComponent);
begin
  inherited;
  try
    FClientSocket:= TClientSocket.Create(Self);
  except
  end;
  FProxy:= '';
  FProxyPort:= 0;

  FClientSocket.OnRead:= FClientSocketRead;
  FClientSocket.OnWrite:= FClientSocketWrite;
  FClientSocket.OnDisconnect:= FClientSocketDisconnect;
  FClientSocket.OnError:= FClientSocketError;
  FGifImage:= TGifImage.Create;
  FStarted:= False;
  FLoaded:= False;
  FNextTimeOutSec:= 0; // sec, no wait to load next image
  FLastLoad:= 0.0;       // TDateTime
  ShowHint:= True;
  Clear;
end;

destructor THttpGifLoader.Destroy;
begin
  SetStarted(False);
  FGifImage.Free;
  FClientSocket.Free;
  inherited;
end;

procedure THttpGifLoader.ReadProxySettings;
var
  vProxy: String;
  vProxyPort: Integer;
begin
  if util1.ReadIEProxySettings('http', FClientSocket.Host, vProxy, vProxyPort) then begin
    Proxy:= vProxy;
    ProxyPort:= vProxyPort;
  end;
end;

procedure THttpGifLoader.FClientSocketRead(Sender: TObject; Socket: TCustomWinSocket);
begin
  try
    FData:= FData + Socket.ReceiveText;
  except
  end;  
end;

procedure THttpGifLoader.FClientSocketWrite(Sender: TObject; Socket: TCustomWinSocket);
begin
  Socket.SendText('GET ' + FURL + ' HTTP/1.0'#13#10+
    'Connection: Keep-Alive'#13#10+
    'User-Agent: Mozilla/4.51 [en] (WinNT; I)'#13#10+
//  'Host: '+ FClientSocket.Host + #13#10+
    'Accept: image/gif, image/jpeg, */*'#13#10#13#10);
end;


end.
