unit htmlprod;
(*##*)
(*******************************************************************
*                                                                 *
*   H  T  M  L  P  R  O  D   TPageProducer replacement             *
*                                                                 *
*   Copyright (c) 1999, 2000 A.Ivanov. All rights reserved.        *
*   Based on Delphi 4 TPageProducer component                     *
*   Conditional defines:                                           *
*                                                                 *
*   Last Revision: May 19 1999                                     *
*   Last fix     :                                                *
*   Lines        :                                                 *
*   History      : see CHANGES.TXT file                           *
*   Printed      : ---                                             *
*                                                                 *
********************************************************************)
(*##*)
interface
uses
  SysUtils, Classes,
{$IFDEF VER140}
    HTTPProd,
{$ENDIF}
  httpapp;

const
  PREFIXLEN = 16;
type
  TECustomPageProducer = class(TCustomContentProducer)
  private
    FStripParamQuotes: Boolean;
    FTag1Char: Char;
    FTag2Chars: String[PREFIXLEN-1];
    FHTMLFile: TFileName;
    FHTMLDoc: String;           // TPageProducer.FHTMLDoc: TStrings;
    FContentRequestCount: Integer;
    FOldValues: TStrings;
    FEnableCollectOldValues: Boolean;
    procedure SetHTMLFile(const Value: TFileName);
    procedure SetHTMLDoc(Value: String);
    procedure SetTagPrefix(Value: String);
    function GetTagPrefix: String;
    procedure SetEnableCollectOldValues(AValue: Boolean);
    function GetOldValue(AInd: String): String;
    procedure SetOldValue(AInd: String; const AValue: String);
  protected
    property StripParamQuotes: Boolean read FStripParamQuotes write FStripParamQuotes default True;
    function HandleTag(const TagString: string; TagParams: TStrings): string; virtual;
    property HTMLDoc: String read FHTMLDoc write SetHTMLDoc;
    property HTMLFile: TFileName read FHTMLFile write SetHTMLFile;
    property ContentRequestCount: Integer read FContentRequestCount;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function Content: string; override;
    function ContentFromStream(Stream: TStream): string; override;
    function ContentFromString(const S: string): string; override;
    property TagPrefix: String read GetTagPrefix write SetTagPrefix;
    property EnableCollectOldValues: Boolean read FEnableCollectOldValues write SetEnableCollectOldValues;
    property OldValue[ind: String]: String read GetOldValue write SetOldValue;
    procedure ClearOldValues;
  end;

  TEPageProducer = class(TECustomPageProducer)
  private
    FOnHTMLTag: THTMLTagEvent;
  protected
    function HandleTag(const TagString: string; TagParams: TStrings): string; override;
    procedure DoTagEvent(Tag: TTag; const TagString: string; TagParams: TStrings;
      var ReplaceText: string); dynamic;
  published
    property HTMLDoc;
    property HTMLFile;
    property OnHTMLTag: THTMLTagEvent read FOnHTMLTag write FOnHTMLTag;
  end;

implementation

uses
  Consts, Windows, WebConst, CopyPrsr;
const
  DEFAULT1CHAR = '#';
  SPACE = #32;
{ TCopyParser }
  ParseBufSize = 4096;
{ TECustomPageProducer }

constructor TECustomPageProducer.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FContentRequestCount:= 0;
  FOldValues:= Nil;
  FEnableCollectOldValues:= False;

  FTag1Char:= DEFAULT1CHAR;
  RPR;
  FHTMLDoc:= ''; // TStringList.Create;
  FHTMLFile:= '';
  FStripParamQuotes:= True;
end;

procedure TECustomPageProducer.SetEnableCollectOldValues(AValue: Boolean);
begin
  if FEnableCollectOldValues then begin
    if not AValue then begin
      FOldValues.Free; // off
      FEnableCollectOldValues:= False;
    end;
  end else begin
    if AValue then begin
      FOldValues:= TStringList.Create;
      FEnableCollectOldValues:= True;
    end;
  end;
end;

function TECustomPageProducer.GetOldValue(AInd: String): String;
begin
  if FEnableCollectOldValues
  then Result:= FOldValues.Values[AInd]
  else Result:= '';
end;

procedure TECustomPageProducer.SetOldValue(AInd: String; const AValue: String);
begin
  if FEnableCollectOldValues
  then FOldValues.Values[AInd]:= AValue;
end;

procedure TECustomPageProducer.ClearOldValues;
begin
  if FEnableCollectOldValues
  then FOldValues.Clear;
end;

procedure TECustomPageProducer.SetTagPrefix(Value: String);
begin
  if Length(Value) = 0 then begin
    FTag1Char:= DEFAULT1CHAR;
    FTag2Chars:= '';
  end else begin
    FTag1Char:= Value[1];
    FTag2Chars:= Copy(Value, 2, PREFIXLEN-1);
  end;
end;

function TECustomPageProducer.GetTagPrefix: String;
begin
  Result:= FTag1Char + FTag2Chars;
end;

destructor TECustomPageProducer.Destroy;
begin
  FHTMLDoc:= ''; // FHTMLDoc.Free;
  if FEnableCollectOldValues
  then FOldValues.Free;
  inherited Destroy;
end;

function TECustomPageProducer.Content: string;
var
  InStream: TStream;
begin
  Result:= '';
  if FHTMLFile <> '' then begin
    InStream:= TFileStream.Create(FHTMLFile, fmOpenRead + fmShareDenyWrite);
    try
      Result:= ContentFromStream(InStream);
    finally
      InStream.Free;
    end;
  end else begin
    Result:= ContentFromString(FHTMLDoc);
  end;
  Inc(FContentRequestCount);
end;

function TECustomPageProducer.ContentFromStream(Stream: TStream): string;
var
  Parser: TCopyParser;
  OutStream: TStringStream;
  ParamStr, ReplaceStr, TokenStr: string;
  ParamList: TStringList;
begin
  OutStream := TStringStream.Create('');
  try
    Parser := TCopyParser.Create(Stream, OutStream);
    with Parser do
    try
      while True do
      begin
        while not (Token in [toEof, '<']) do
        begin
          CopyTokenToOutput;
          SkipToken(True);
        end;
        if Token = toEOF then Break;
        if Token = '<' then
        begin
          if SkipToken(False) = '#' then begin
            SkipToken(False);
            TokenStr := TokenString;
            ParamStr := TrimLeft(TrimRight(SkipToToken('>')));
            ParamList := TStringList.Create;
            try
              ExtractHTTPFields([' '], [' '], PChar(ParamStr), ParamList, FStripParamQuotes);
              ReplaceStr := HandleTag(TokenStr, ParamList);
              OutStream.WriteString(ReplaceStr);
            finally
              ParamList.Free;
            end;
            SkipToken(True);
          end else
          begin
            OutStream.WriteString('<');
            CopyTokenToOutput;
            SkipToken(True);
          end;
        end;
      end;
    finally
      Parser.Free;
    end;
    Result := OutStream.DataString;
  finally
    OutStream.Free;
  end;
end;

function TECustomPageProducer.ContentFromString(const S: string): string;
var
  ParamStr, ReplaceStr, TokenStr: string;
  ParamList: TStringList;
  p, Len, st: Integer;
begin
  p:= 1;
  Len:= Length(S);
  Result:= '';
  while True do begin
    { looking for first tag symbol "<" }
    while (p <= Len) and (s[p] <> '<') do begin
      Result:= Result + s[p];
      Inc(p);
    end;
    Inc(p);
    if p > Len then Break; // all was copied
    // while (p < Len) and (s[p] <= SPACE) do Inc(p);
    if s[p] = FTag1Char then begin
      Inc(p);
      // skip spaces after <#..
      while (p<=Len) and (s[p]<=SPACE) and (s[p]<>'>') do Inc(p);
      st:= p;
      // search the end of the first token
      while (p<=Len) and (s[p]>SPACE) and (s[p]<>'>') do Inc(p);
      TokenStr:= Copy(S, st, p - st);
      // search the end tag: '>'
      st:= p;
      while (p<=Len) and (s[p]<>'>') do Inc(p);
      ParamStr:= TrimLeft(TrimRight(Copy(S, st, p-st)));
      ParamList:= TStringList.Create;
      try
        ExtractHTTPFields([' '], [' '], PChar(ParamStr), ParamList, True);
        ReplaceStr:= HandleTag(TokenStr, ParamList);
        Result:= Result + ReplaceStr;
      finally
        ParamList.Free;
      end;
      Inc(p);
    end else begin
      Result:= Result + '<';
    end;
  end;
end;

function TECustomPageProducer.HandleTag(const TagString: string; TagParams: TStrings): string;
begin
  Result:= Format('<#%s>', [TagString]);
end;

procedure TECustomPageProducer.SetHTMLFile(const Value: TFileName);
begin
  if CompareText(FHTMLFile, Value) <> 0 then begin
    FContentRequestCount:= 0;
    FHTMLDoc:= '';
    FHTMLFile:= Value;
  end;
end;

procedure TECustomPageProducer.SetHTMLDoc(Value: String);
begin
  FContentRequestCount:= 0;
  FHTMLDoc:= Value;
  FHTMLFile:= '';
end;

{ TEPageProducer }

var
  TagSymbols: array[TTag] of string =
    ('', 'LINK', 'IMAGE', 'TABLE', 'IMAGEMAP', 'OBJECT', 'EMBED');

function TEPageProducer.HandleTag(const TagString: string; TagParams: TStrings): string;
var
  Tag: TTag;
begin
  Tag := High(TTag);
  while Tag >= Low(TTag) do begin
    if (Tag = tgCustom) or (CompareText(TagSymbols[Tag], TagString) = 0) then Break;
    Dec(Tag);
  end;
  Result := '';
  DoTagEvent(Tag, TagString, TagParams, Result);
end;

procedure TEPageProducer.DoTagEvent(Tag: TTag; const TagString: string;
  TagParams: TStrings; var ReplaceText: string);
begin
  if Assigned(FOnHTMLTag)
  then FOnHTMLTag(Self, Tag, TagString, TagParams, ReplaceText);
end;

end.
