(**********************************************)
(*  RAS component                             *)
(*  (c) ArGo Software Design, 1996-1999       *)
(*  Based partially on TRasComponent by       *)
(*  Douglas Olender, dolender@wpcusrgrp.org   *)
(**********************************************)
unit msRasCmp;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls, msRasAPI, msRasCls, DsgnIntf;

type

  EmsRASError = class(Exception)
    LastRasError : Integer;
    constructor CreateRes(Ident : Integer);
  end;

  TRASNotifyEvent=procedure(Sender : TObject; RASStatusStr : string) of Object;

  TmsRas = class(TComponent)
  private
    { Private declarations }
    FEntries : TStrings;
    FService : string;
    FRasCon : THandle;
    FRasMessage : UINT;
    RasWndHandle : THandle;
    FOnStatusChange : TRASNotifyEvent;
    FOnDisconnected : TNotifyEvent;
    FMinimizeAtConnect : boolean;
    FOwnConnection : boolean;  //true if dialed in this app
    FPassword : string;
    FRemovePassword : boolean;
    FRasLibHandle : THandle;
    FForceService : boolean;
    FActiveConnections : TStrings;
    procedure SetService(Value : string);
    function GetIsActive : boolean;
    procedure DoDisconnected(Sender : TObject);
    function GetEntries : TStrings;
    function GetInstalled : boolean;
    procedure SetOwnConnection(Value : boolean);
    function GetTimeOut : Integer;
    procedure SetTimeOut(Value : Integer);
    function GetActiveConnections : TStrings;
  protected
    { Protected declarations }
    rs : UINT;
    TC : TmsTimeCounter;
    Canceled : boolean;
    procedure WndProc(var Msg : TMessage);
    function GetRASStatusString(Num : word) : string;
    procedure MinimizeConnectWindow;
    procedure ReInit;
  public
    { Public declarations }
    Connected : boolean;
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
    procedure Cancel;
    procedure Dial;
    procedure HangUp;
    property Entries : TStrings read GetEntries;
    property Installed : boolean read GetInstalled;
    property OwnConnection : boolean read FOwnConnection write SetOwnConnection;
    property ActiveConnections : TStrings read GetActiveConnections;
    property IsActive : boolean read GetIsActive;
    property RasConHandle : THandle read FRasCon;
  published
    { Published declarations }
    property Service : string read FService write SetService;
    property ForceService : boolean read FForceService write FForceService
               default true;
    property MinimizeAtConnect : boolean read FMinimizeAtConnect
                                    write FMinimizeAtConnect default false;
    property Password : string read FPassword write FPassword;
    property RemovePassword : boolean read FRemovePassword write FRemovePassword
                  default true;
    property TimeOut : Integer read GetTimeOut write SetTimeOut default 120;
    property OnStatusChange : TRasNotifyEvent read FOnStatusChange
                   write FOnStatusChange;
    property OnDisconnected : TNotifyEvent read FOnDisconnected
                   write FOnDisconnected;
  end;

  TServiceNameProperty = class(TPropertyEditor)
    function GetAttributes: TPropertyAttributes; override;
    procedure GetValues(Proc: TGetStrProc); override;
    function GetValue: string; override;
    procedure SetValue(const AValue: string); override;
  end;

procedure Register;

implementation

const
  BAD_HANDLE = THandle(-1);

procedure Register;
begin
  RegisterComponents('Internet Mail Suite',[TmsRas]);
  RegisterPropertyEditor(TypeInfo(string),TmsRas,'Service',TServiceNameProperty);
end;

constructor EmsRasError.CreateRes(Ident : Integer);
var
  RasErrorString : TRasErrorString;
  rs : DWORD;
  RasGetErrorString : TRasGetErrorString;
  FRasLibHandle : THandle;
begin
  LastRasError:=Ident;
  FillChar(RasErrorString,SizeOf(RasErrorString),0);
  FRasLibHandle:=LoadLibrary(RasAPI32);
  try
    @RasGetErrorString:=GetProcAddress(FRasLibHandle,'RasGetErrorStringA');
    if @RasGetErrorString=nil then Exit;
    rs:=RasGetErrorString(Ident,RasErrorString,256);
    if rs<>0 then
      StrPCopy(@RasErrorString,'RAS error no '+IntToStr(Ident));
  finally
    FreeLibrary(FRasLibHandle);
    inherited Create(RasErrorString);
  end;
end;

constructor TmsRas.Create(AOwner : TComponent);
begin
  inherited Create(AOwner);
  if not (csDesigning in ComponentState) then
    RasWndHandle:=AllocateHWnd(WndProc);
  FRASMessage:=RegisterWindowMessage('RASDIALEVENT');
//  if FRASMessage=0 then
//    FRASMessage:=WM_RASDIALEVENT;  {?}
  FRasCon:=BAD_HANDLE;
  Connected:=false;
  Canceled:=false;
  TC:=TmsTimeCounter.Create;
  TimeOut:=120;
  FMinimizeAtConnect:=false;
  FRemovePassword:=true;
  FEntries:=TStringList.Create;
  FActiveConnections:=TStringList.Create;
  FForceService:=true;
  FRasLibHandle:=LoadLibrary(RasAPI32);
  GetEntries;
end;

destructor TmsRas.Destroy;
begin
  FreeLibrary(FRasLibHandle);
  FActiveConnections.Free;
  FEntries.Free;
  TC.Free;
  if not (csDesigning in ComponentState) then
    DeallocateHWnd(RasWndHandle);
  inherited Destroy;
end;

procedure TmsRas.DoDisconnected(Sender : TObject);
begin
  if Assigned(FOnDisconnected) then
    FOnDisconnected(Sender);
end;

function TmsRas.GetEntries : TStrings;
var
  RasEntries : array[1..25] of TRasEntryName;
  cb : DWORD;
  Count : DWORD;
  i : DWORD;
  RasEnumEntries : TRasEnumEntries;
begin
  FEntries.Clear;
  Result:=FEntries;
  if FRasLibHandle=0 then Exit;
  @RasEnumEntries:=GetProcAddress(FRasLibHandle,'RasEnumEntriesA');
  if @RasEnumEntries=nil then Exit;
  RasEntries[1].Size:=SizeOf(TRasEntryName);
  cb:=SizeOf(RasEntries);
  rs:=RasEnumEntries(nil,nil,@RasEntries,cb,Count);
  if rs=0 then
  begin
    for i:=1 to Count do
      FEntries.Add(RasEntries[i].EntryName);
  end;
{  else
  if not csDesigning in ComponentState then
    raise ERasError.CreateRes(rs);}
  Result:=FEntries;
end;

function TmsRas.GetActiveConnections : TStrings;
var
  RasEnumConnections : TRasEnumConnections;
  RasGetProjectionInfo : TRasGetProjectionInfo;
  RasConns : array[1..15] of TRasConn;
  cb,i : DWORD;
  Connections : DWORD;
  RasPPPIP : TRasPPPIP;
  s : string;
begin
  FActiveConnections.Clear;
  Result:=FActiveConnections;
  @RasEnumConnections:=GetProcAddress(FRasLibHandle,'RasEnumConnectionsA');
  if @RasEnumConnections=nil then exit;
  FillChar(RasConns,SizeOf(RasConns),0);
  RasConns[1].Size:=SizeOf(TRasConn);
  cb:=SizeOf(RasConns);
  rs:=RasEnumConnections(@RasConns,cb,Connections);
  if rs<>0 then
    raise EmsRasError.CreateRes(rs);
  RasGetProjectionInfo:=GetProcAddress(FRasLibHandle,'RasGetProjectionInfoA');
  if @RasGetProjectionInfo=nil then exit;
  for i:=1 to Connections do
  begin
    cb:=SizeOf(RasPPPIP);
    FillChar(RasPPPIP,cb,0);
    RasPPPIP.Size:=cb;
    rs:=RasGetProjectionInfo(RasConns[i].Handle,RasP_PppIp,RasPPPIP,cb);
    if rs<>0 then
      raise EmsRasError.CreateRes(rs);
    if RasPPPIP.IPAddress<>'' then
    begin
      s:=RasConns[i].EntryName;
      FActiveConnections.Add(s);
    end;
  end;
  Result:=FActiveConnections;
end;

function TmsRas.GetInstalled : boolean;
begin
  Result:=FEntries.Count>0;
end;

procedure TmsRas.SetOwnConnection(Value : boolean);
begin
  if FRasCon<>BAD_HANDLE then
    FOwnConnection:=Value
  else
    MessageDlg('You can set this property only after'^M+
               'calling Dail method.',mtError,[mbOK],0);
end;

function TmsRas.GetTimeOut : Integer;
begin
  Result:=TC.TimeOut;
end;

procedure TmsRas.SetTimeOut(Value : Integer);
begin
  TC.TimeOut:=Value;
end;

procedure TmsRas.ReInit;
begin
  FRasCon:=BAD_HANDLE;
  Connected:=false;
  Canceled:=false;
  TC.TimedOut:=false;
end;

procedure TmsRas.WndProc(var Msg : TMessage);
begin
  with Msg do
  if Msg=FRasMessage then
  begin
    if Assigned(OnStatusChange) then
      FOnStatusChange(Self,GetRASStatusString(wParam));
    if not Connected then
      Connected:=wParam=RASCS_Done;
    if lParam<>0 then
    begin
      rs:=lParam;
      if Connected then
        raise EmsRasError.CreateRes(rs);
    end;
  end
  else
    Result:=DefWindowProc(RasWndHandle,Msg,wParam,lParam);
end;

function TmsRas.GetRASStatusString(Num : word) : string;
var
  ResNo : word;
  Buf : PChar;
begin
  ResNo:=0;
  case Num of
    RASCS_DONE : ResNo:=129;  //Succesfully connected
    RASCS_OpenPort : ResNo:=120; //Initializing...
    RASCS_ConnectDevice : ResNo:=122; // Dialing...
    RASCS_DeviceConnected : ResNo:=122; //Dialing...
    RASCS_AllDevicesConnected : ResNo:=122; //Dialing...
    RASCS_Authenticate: ResNo:=124; //Verifying user name and password
    RASCS_StartAuthentication : ResNo:=124; //Verifying user name and password
    RASCS_Authenticated : ResNo:=131; //User name and password verified
    RASCS_LogonNetwork : ResNo:=128;  //Logging on to Network
//    RASCS_DISCONNECTED : ReNo:='Dosconnected';
  end;
  Buf:=StrAlloc(255);
  LoadString(FRasLibHandle,ResNo,Buf,255);
  Result:=StrPas(Buf);
  StrDispose(Buf);
  if Result='' then
  begin
    case ResNo of
      RASCS_DONE : Result:='Successfully connected';  //Succesfully connected
      RASCS_OpenPort : Result:='Initializing...'; //Initializing...
      RASCS_ConnectDevice,
      RASCS_DeviceConnected,
      RASCS_AllDevicesConnected : Result:='Dialing...'; //Dialing...
      RASCS_Authenticate,
      RASCS_StartAuthentication : Result:='Verifying user name and password'; //Verifying user name and password
      RASCS_Authenticated : Result:='User name and password verified'; //User name and password verified
      RASCS_LogonNetwork : Result:='Logging on to Network';  //Logging on to Network
    end;
  end;
end;

procedure TmsRas.MinimizeConnectWindow;
var
  Handle : THandle;
begin
  Sleep(1000);
  Handle:=FindWindow('#32770',nil);
  if Handle>0 then
    SendMessage(Handle,WM_SYSCOMMAND,SC_MINIMIZE,0);
end;

procedure TmsRas.Cancel;
begin
  HangUp;
  Canceled:=true;
end;

procedure TmsRas.SetService(Value : string);
begin
  if Value<>'' then
  begin
    if not Installed then
    begin
      MessageDlg('TmsRas component was unable to detect'^M+
                 'RAS on your system.',mtError,[mbOK],0);
    end
    else
    if Connected then
    begin
      MessageDlg('This property can be set only before'^M+
                 'establishing the connection',mtError,[mbOK],0);
    end
    else
    if Entries.IndexOf(Value)=-1 then
    begin
      MessageDlg('Invalid service name',mtError,[mbOk],0);
      FService:='';
    end
    else
      FService:=Value;
  end
  else
    FService:='';
end;

function TmsRas.GetIsActive : boolean;
var
  RasEnumConnections : TRasEnumConnections;
  RasGetProjectionInfo : TRasGetProjectionInfo;
  RasConns : array[1..10] of TRasConn;
  cb,i : DWORD;
  Connections : DWORD;
  RasPPPIP : TRasPPPIP;
begin
  Result:=false;
  @RasEnumConnections:=GetProcAddress(FRasLibHandle,'RasEnumConnectionsA');
  if @RasEnumConnections=nil then exit;
  FillChar(RasConns,SizeOf(RasConns),0);
  RasConns[1].Size:=SizeOf(TRasConn);
  cb:=SizeOf(RasConns);
  rs:=RasEnumConnections(@RasConns,cb,Connections);
  if rs<>0 then
    raise EmsRasError.CreateRes(rs);
  i:=1;
  while (i<=Connections) and (not Result) do
  begin
    Result:=(FService=RasConns[i].EntryName)
        and (FRasCon<>RasConns[i].Handle);
    Inc(i);
  end;
  if Result then
    FRasCon:=RasConns[i-1].Handle;
  if (not Result) and (not FForceService) then  {1.3}
  begin  //Check if there is TCP/IP connection active
    RasGetProjectionInfo:=GetProcAddress(FRasLibHandle,'RasGetProjectionInfoA');
    if @RasGetProjectionInfo=nil then exit;
    i:=1;
    while (i<=Connections) and (not Result) do
    begin
      cb:=SizeOf(RasPPPIP);
      FillChar(RasPPPIP,cb,0);
      RasPPPIP.Size:=cb;
      rs:=RasGetProjectionInfo(RasConns[i].Handle,RasP_PppIp,RasPPPIP,cb);
      if rs<>0 then
        raise EmsRasError.CreateRes(rs);
      Result:=RasPPPIP.IPAddress<>'';
      Inc(i);
    end;
    if Result then
      FRasCon:=RasConns[i-1].Handle;
  end;
end;

procedure TmsRas.Dial;
var
  DP : TRasDialParams;
  PasswordOK : LongBool;
  RasGetEntryDialParams : TRasGetEntryDialParams;
  RasSetEntryDialParams : TRasSetEntryDialParams;
  RasDial : TRasDial;
begin
  if not IsActive and (FService<>'') then
  begin
    @RasGetEntryDialParams:=GetProcAddress(FRasLibHandle,'RasGetEntryDialParamsA');
    if @RasGetEntryDialParams=nil then exit;
    @RasSetEntryDialParams:=GetProcAddress(FRasLibHandle,'RasSetEntryDialParamsA');
    if @RasSetEntryDialParams=nil then exit;
    @RasDial:=GetProcAddress(FRasLibHandle,'RasDialA');
    if @RasDial=nil then exit;
    FOwnConnection:=true;
    ReInit;
    FillChar(DP,SizeOf(TRasDialParams),0);
    with DP do
    begin
      StrPCopy(EntryName,FService);
      Size:=SizeOf(TRasDialParams);
    end;
    rs:=RasGetEntryDialParams(nil,DP,PasswordOK);
    if rs<>0 then
      raise EmsRasError.CreateRes(rs);
    if not PasswordOK then
    begin
      StrPCopy(DP.Password,FPassword);
      rs:=RasSetEntryDialParams(nil,DP,FRemovePassword);
      if rs<>0 then
        raise EmsRasError.CreateRes(rs);
    end;
    rs:=RasDial(nil,nil,DP,$FFFFFFFF,Pointer(RasWndHandle),FRasCon);
    if rs<>0 then
    begin
      HangUp;
      raise EmsRasError.CreateRes(rs);
    end;
  //wait until connected or error
    TC.TimerOn;
    repeat
      Application.ProcessMessages;
    until Connected or (rs<>0) or Canceled or TC.TimedOut;
    TC.TimerOff;
    if rs<>0 then
    begin
      HangUp;
      raise EmsRasError.CreateRes(rs);
    end;
    if Canceled then
      raise EmsRasError.Create('Operation has been canceled');
    if TC.TimedOut then
      raise EmsRasError.Create('Operation timed out');
    if FMinimizeAtConnect then
      MinimizeConnectWindow;
  end
  else
  begin
    Connected:=true;
    FOwnConnection:=false;
  end;
end;

procedure TmsRas.HangUp;
var
  RasHangUp : TRasHangUp;
begin
  @RasHangUp:=GetProcAddress(FRasLibHandle,'RasHangUpA');
  if @RasHangUp=nil then Exit;
  if Installed and FOwnConnection then
  begin
    RasHangUp(FRasCon);
    DoDisconnected(Self);
  end;
  Connected:=false;
  FRasCon:=BAD_HANDLE;
end;

{TServiceNameProperty}
function TServiceNameProperty.GetAttributes: TPropertyAttributes;
begin
  Result := [paValueList];
end;

procedure TServiceNameProperty.GetValues(Proc: TGetStrProc);
var
  I: Integer;
  Ras : TmsRas;
begin
  Ras:=GetComponent(0) as TmsRas;
  for i:=0 to Ras.FEntries.Count-1 do
    Proc(Ras.FEntries[i]);
end;

function TServiceNameProperty.GetValue: string;
begin
  Result:=GetStrValue;
end;

procedure TServiceNameProperty.SetValue(const AValue: string);
begin
  SetStrValue(AValue);
end;

end.
