(*
Component : TIF_Gradient
Created   : 07.21.2000
Author    : Tibor Reiter
e-mail    : reiter@interface.hu

Version   : 1.0
Tested    : Delphi 5 / NT


COPYYRIGHT information

Copyright 2000 Tibor Riter, reiter@interface.hu

You are allowed to used this component in any project for free.
Component is supplied without warranty of any kind.
TO USE THIS COMPONENT AT YOUR OWN RISK.

If you are going to use my component or code
plase send an e-mail to me (reiter@interface.hu).
I'd like to know how my code is used.

If you find bugs or alter the component,
please DO NOT public the corrected/new code on the internet,
but instead send it to me, so I can put it into the official version.
You will be accredited if you do so.


*)



unit IF_Gradient;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls;

type
  TIF_CustomGradient = class(TGraphicControl)
  private
    FAlignment: TAlignment;
    FAutoSize: Boolean;
    FWordWrap: Boolean;
    FStartColor: TColor;
    FEndColor: TColor;
    FSelectedColor: TColor;
    FSelectedStartColor: TColor;
    FSelectedEndColor: TColor;
    FGradient: Boolean;
    FDoSelection: boolean;
    FSelected: boolean;
    procedure SetAlignment(Value: TAlignment);
    procedure SetWordWrap(Value: Boolean);
    procedure SetEndColor(const Value: TColor);
    procedure SetStartColor(const Value: TColor);
    procedure SetSelectedColor(const Value: TColor);
    procedure SetSelectedEndColor(const Value: TColor);
    procedure SetSelectedStartColor(const Value: TColor);
    procedure SetGradient(Value: boolean);
    procedure SetSelected(Value: boolean);
    procedure CMMouseEnter(var Message: TMessage); message CM_MOUSEENTER;
    procedure CMMouseLeave(var Message: TMessage); message CM_MOUSELEAVE;
    procedure CMTextChanged(var Message: TMessage); message CM_TEXTCHANGED;
    procedure CMFontChanged(var Message: TMessage); message CM_FONTCHANGED;
  protected
    procedure Loaded; override;
    procedure AdjustBounds;
    procedure DoDrawText(var Rect: TRect; Flags: Integer);
    procedure SetAutoSize(Value: Boolean); virtual;
    procedure Paint; override;
    property Alignment: TAlignment read FAlignment write SetAlignment default taLeftJustify;
    property AutoSize: Boolean read FAutoSize write SetAutoSize default True;
    property WordWrap: Boolean read FWordWrap write SetWordWrap default False;
    property StartColor: TColor read FStartColor write SetStartColor;
    property EndColor: TColor read FEndColor write SetEndColor;
    property SelectedColor: TColor read FSelectedColor write SetSelectedColor;
    property SelectedEndColor: TColor read FSelectedEndColor write SetSelectedEndColor;
    property SelectedStartColor: TColor read FSelectedStartColor write SetSelectedStartColor;
    property Gradient: Boolean read FGradient write SetGradient default True;
    property DoSelection: Boolean read FDoSelection write FDoSelection default false;
    property Selected: Boolean read FSelected write SetSelected default false;
  public
    constructor Create(AOwner: TComponent); override;
    property canvas;
  end;

  TIF_Gradient = class(TIF_CustomGradient)
  published
    property Alignment;
    property Autosize;
    property Caption;
{$IFNDEF ver100}
    property Anchors;
    property Constraints;
{$ENDIF}
    property DoSelection;
    property Enabled;
    property EndColor;
    property Font;
    property Gradient;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property Selected;
    property SelectedColor;
    property SelectedEndColor;
    property SelectedStartColor;
    property StartColor;
    property ShowHint;
    property Visible;
    property WordWrap;

    property OnClick;
    property OnDblClick;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

procedure Register;
procedure DrawGradient(ACanvas: TCanvas; ARect: TRect; StartColor,EndColor:Tcolor);

implementation

{$R *.Res}

procedure Register;
begin
  RegisterComponents('IF_Lib', [TIF_Gradient]);
end;

procedure DrawGradient(ACanvas: TCanvas; ARect: TRect; StartColor,EndColor:Tcolor);
var Red, Green, Blue : double;
    StepRed, StepGreen, StepBlue : double;
    i : integer;
begin
  Red   := GetRValue(StartColor);
  Green := GetGValue(StartColor);
  Blue  := GetBValue(StartColor);
  if StartColor = EndColor
  then begin
    ACanvas.brush.Color:=RGB(Round(Red), Round(Green), Round(Blue));
    ACanvas.fillrect(ARect);
  end
  else begin
    StepRed   := (GetRValue(EndColor) - Red)  / (Arect.Right - Arect.Left +1);
    StepGreen := (GetGValue(EndColor) - Green)/ (Arect.Right - Arect.Left +1);
    StepBlue  := (GetBValue(EndColor) - Blue) / (Arect.Right - Arect.Left +1);
    with Acanvas do
    begin
      for i:=0 to Arect.right-arect.left-1 do
      begin
        brush.Color:=RGB(Round(Red), Round(Green), Round(Blue));
        fillrect(Rect(Arect.left+i,Arect.top,Arect.left+i+1,Arect.bottom));
        Red   := Red   + StepRed;
        Green := Green + StepGreen;
        Blue  := Blue  + StepBlue;
      end;
    end;
  end;
end;


constructor TIF_CustomGradient.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ControlStyle := ControlStyle + [csReplicatable];
  FStartColor:=clTeal;
  FEndColor:=clSilver;
  FSelectedColor:=clGray;
  FSelectedStartColor:=clAqua;
  FSelectedEndColor:=clNavy;
  Caption:=name;
  FGradient:=true;
  Fselected:=False;
  FDoSelection:=false;
  Fautosize:=True;
end;


procedure TIF_CustomGradient.DoDrawText(var Rect: TRect; Flags: Longint);
var Text: string;
begin
  Text := Caption;
  Flags := Flags or DT_NOPREFIX;
  Canvas.Font := Font;
  if not Enabled then
  begin
    OffsetRect(Rect, 1, 1);
    Canvas.Font.Color := clBtnHighlight;
    DrawText(Canvas.Handle, PChar(Text), Length(Text), Rect, Flags);
    OffsetRect(Rect, -1, -1);
    Canvas.Font.Color := clBtnShadow;
    DrawText(Canvas.Handle, PChar(Text), Length(Text), Rect, Flags);
  end
  else begin
    if Fselected
      then Canvas.font.color:=FselectedColor;
    DrawText(Canvas.Handle, PChar(Text), Length(Text), Rect, Flags);
  end;
end;


procedure TIF_CustomGradient.AdjustBounds;
const
  WordWraps: array[Boolean] of Word = (0, DT_WORDBREAK);
var
  DC: HDC;
  X: Integer;
  Rect: TRect;
begin
  if not (csReading in ComponentState) and FAutoSize then
  begin
    Rect := ClientRect;
    DC := GetDC(0);
    Canvas.Handle := DC;
    DoDrawText(Rect, (DT_EXPANDTABS or DT_CALCRECT) or WordWraps[FWordWrap]);
    Canvas.Handle := 0;
    ReleaseDC(0, DC);
    X := Left;
    if FAlignment = taRightJustify then Inc(X, Width - Rect.Right);
    SetBounds(X, Top, Rect.Right, Rect.Bottom);
  end;
end;


procedure TIF_CustomGradient.Paint;
const
  Alignments: array[TAlignment] of Word = (DT_LEFT, DT_RIGHT, DT_CENTER);
  WordWraps: array[Boolean] of Word = (0, DT_WORDBREAK);
var
  Rect, CalcRect: TRect;
  DrawStyle: Longint;
begin
  if FGradient then
    if Fselected then
      Drawgradient(Canvas,Clientrect,FSelectedStartColor,FSelectedEndColor)
    else
      Drawgradient(Canvas,Clientrect,FStartColor,FEndColor);
  with Canvas do begin
    Brush.Style:=bsClear;
    rect:=ClientRect;
    DrawStyle := DT_EXPANDTABS or WordWraps[FWordWrap] or Alignments[FAlignment];
    CalcRect := Rect;
    DoDrawText(CalcRect, DrawStyle or DT_CALCRECT);
    OffsetRect(Rect, 0, (Height - CalcRect.Bottom) div 2);
    DoDrawText(Rect, DrawStyle);
  end;
end;

procedure TIF_CustomGradient.SetEndColor(const Value: TColor);
begin
  FEndColor:=Value;
  Invalidate;
end;

procedure TIF_CustomGradient.SetStartColor(const Value: TColor);
begin
  FStartColor:=Value;
  Invalidate;
end;

procedure TIF_CustomGradient.SetSelectedColor(const Value: TColor);
begin
  FSelectedColor:=Value;
  if Fselected then
    if (csDesigning in ComponentState) then
      invalidate
    else
      paint;
end;

procedure TIF_CustomGradient.SetSelectedEndColor(const Value: TColor);
begin
  FSelectedEndColor:=Value;
  if Fselected then
    if (csDesigning in ComponentState) then
      invalidate
    else
      paint;
end;

procedure TIF_CustomGradient.SetSelectedStartColor(const Value: TColor);
begin
  FSelectedStartColor:=Value;
  if Fselected then
    if (csDesigning in ComponentState) then
      invalidate
    else
      paint;
end;

procedure TIF_CustomGradient.SetGradient(Value: Boolean);
begin
  FGradient:=Value;
  Invalidate;
end;

procedure TIF_CustomGradient.SetSelected(Value: Boolean);
begin
  FSelected:=Value;
  if csDesigning in ComponentState then
    invalidate
  else
    paint;
end;

procedure TIF_CustomGradient.CMMouseLeave(var Message: TMessage);
begin
  If Fdoselection and Enabled
  then begin
    FSelected:=false;
    paint;
  end;
end;

procedure TIF_CustomGradient.CMMouseEnter(var Message: TMessage);
begin
  if FDoSelection and Enabled
  then begin
    FSelected:=true;
    paint;
  end;
end;

procedure TIF_CustomGradient.CMFontChanged(var Message: TMessage);
begin
  inherited;
  AdjustBounds;
end;

procedure TIF_CustomGradient.CMTextChanged(var Message: TMessage);
begin
  invalidate;
  AdjustBounds;
end;

procedure TIF_CustomGradient.SetAlignment(Value: TAlignment);
begin
  if FAlignment <> Value then
  begin
    FAlignment := Value;
    Invalidate;
  end;
end;

procedure TIF_CustomGradient.SetAutoSize(Value: Boolean);
begin
  if FAutoSize <> Value then
  begin
    FAutoSize := Value;
    AdjustBounds;
  end;
end;


procedure TIF_CustomGradient.SetWordWrap(Value: Boolean);
begin
  if FWordWrap <> Value then
  begin
    FWordWrap := Value;
    AdjustBounds;
    Invalidate;
  end;
end;

procedure TIF_CustomGradient.Loaded;
begin
  inherited;
  AdjustBounds;
end;

end.
