/* Written by Marivie A. Galeon and Lawrence Blue B. Cuenca <Azuer88@crosswinds.net>*/
unit ID3Tag;

interface

uses
	Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
	ID3;

type
	TID3Tag = class(TComponent)
	private
		{id3Tag Fields}
		FTitle,
		FArtist,
		FAlbum,
		FYear,
		FComment,
		FGenre: String;

		{other properties}
		FTagPos: LongInt;
		FFileName: TFileName;
    FModified: boolean;

		{property method implementation}
		procedure SetAlbum(const Value: string);
		procedure SetArtist(const Value: string);
		procedure SetComment(const Value: string);
		procedure SetGenre(const Value: string);
		procedure SetTitle(const Value: string);
		procedure SetYear(const Value: string);
		procedure SetFileName(const Value: TFileName);
		procedure SetModified(const Value: boolean);
	protected
		{ Protected declarations }
		procedure ReadTag; virtual;
		procedure WriteTag; virtual;
		property Modified: boolean read FModified write SetModified;
	public
		{ Public declarations }
		procedure Read;
		procedure Write;

		property FileName: TFileName read FFileName write SetFileName;

		property Title: string read FTitle write SetTitle;
		property Artist: string read FArtist write SetArtist;
		property Album: string read FAlbum write SetAlbum;
		property Year: string read FYear write SetYear;
		property Comment: string read FComment write SetComment;
		property Genre: String read FGenre write SetGenre;
	published
		{ Published declarations }
	end;

	function ID3GenreList: TStringList;
	function ID3NameToInteger(AName: String): Integer;

procedure Register;

implementation

procedure Register;
begin
	RegisterComponents('Custom', [TID3Tag]);
end;

function ID3GenreList: TStringList;
var
	aStringList: TStringList;
	i: ID3Genre;
begin
	aStringList := TStringList.Create;
	for i := Low(ID3Genre) to High(ID3Genre) do
	begin
		aStringList.Add(StrPas(ID3GenreName[i]));
	end;
	result := aStringList;
end;

function ID3NameToGenre(AName: String): ID3Genre;
var RetVal: ID3Genre;
begin
	Result := ID3UnknowGenre;
	for RetVal := Low(ID3Genre) to High(ID3Genre) do
		if UpperCase(AName) = UpperCase(StrPas(ID3GenreName[RetVal])) then
			begin
				Result := RetVal;
				Break;
			end;
end;

function ID3NameToInteger(AName: String): Integer;
begin
	Result := Integer(ID3NameToGenre(AName));
end;

{ TID3Tag }
procedure TID3Tag.Read;
begin
	ReadTag;
end;

procedure TID3Tag.ReadTag;
var
	FFileStream: TFileStream;
	i, FPos: LongInt;
	FTagRecord: ID3Struct;
begin
	i := SizeOf(FTagRecord);
	FFileStream := TFileStream.Create(FFileName, fmOpenRead);
	try
		FPos := FFileStream.Seek(0, soFromEnd);
		FTagPos := FPos - i;
		if FFileStream.Seek(FTagPos, soFromBeginning)=FTagPos then
		begin
			FFileStream.Read(FTagRecord, i);
		end;
	finally
		FFileStream.Free;
	end;

	{Check if file has TAG Record}
	if StrLComp(FTagRecord.Magic, ID3Magic, SizeOf(FTagRecord.Magic))<>0 then
	begin
		FillChar(FTagRecord, SizeOf(FTagRecord), ' ');
		StrLCopy(FTagRecord.Magic, ID3Magic, SizeOf(FTagRecord.Magic));
		FTagPos := FPos;
	end;

	{Get Tag record Values}
	with FTagRecord do
	begin
		FTitle := Trim(Copy(Title, 1, SizeOf(Title)));
		FArtist := Trim(Copy(Artist, 1, SizeOf(Artist)));
		FAlbum := Trim(Copy(Album, 1, SizeOf(Album)));
		FYear := Trim(Copy(Year, 1, SizeOf(Album)));
		FComment := Trim(Copy(Comment, 1, SizeOf(Comment)));
		FGenre := ID3GenreName[Genre];
	end;
end;


procedure TID3Tag.SetAlbum(const Value: string);
begin
	if FAlbum<>Value then
	begin
		FAlbum := Value;
		FModified := True;
	end;
end;

procedure TID3Tag.SetArtist(const Value: string);
begin
	if FArtist<>Value then
	begin
		FArtist := Value;
		FModified := True;
	end;
end;

procedure TID3Tag.SetComment(const Value: string);
begin
	if FComment<>Value then
	begin
		FComment := Value;
		FModified := True;
	end;
end;

procedure TID3Tag.SetFileName(const Value: TFileName);
begin
	if FFileName<>Value then
	begin
		FFileName := Value;
		if FileExists(FFileName) then
			ReadTag;
	end;
end;

procedure TID3Tag.SetGenre(const Value: string);
begin
	if FGenre<>Value then
	begin
		if ID3NameToGenre(Value)=ID3UnknowGenre then
			FGenre := 'Unknown'
		else
			FGenre := Value;
		FModified := True;
	end;
end;

procedure TID3Tag.SetModified(const Value: boolean);
begin
	FModified := Value;
end;

procedure TID3Tag.SetTitle(const Value: string);
begin
	if FTitle<>Value then
	begin
		FTitle := Value;
		FModified := True;
	end;
end;

procedure TID3Tag.SetYear(const Value: string);
begin
	if FYear<>Value then
	begin
		FYear := Value;
		FModified := True;
	end;
end;

procedure TID3Tag.Write;
begin
	WriteTag;
end;

function PadSpace(s: string; len: integer): string;
var
	i: integer;
	q: string;
begin
	q := Copy(s, 1, len);
	i := len - Length(s);
	while (i>0) do
	begin
		q := q + ' ';
		dec(i);
	end;
	result := q;
end;

procedure TID3Tag.WriteTag;
var
	FFileStream: TFileStream;
	FTagRecord: ID3Struct;
	s: string;
begin
	if FModified then
	begin
		FillChar(FTagRecord, SizeOf(FTagRecord), ' ');
		with FTagRecord do
		begin
			StrMove(Magic, ID3Magic, SizeOf(Magic));

			s := PadSpace(FTitle, SizeOf(Title));
			StrMove(Title, PChar(s), Length(s));

			s := PadSpace(FArtist, SizeOf(Artist));
			StrMove(Artist, PChar(s), Length(s));

			s := PadSpace(FAlbum, SizeOf(Album));
			StrMove(Album, PChar(s), Length(s));

			s := PadSpace(FYear, SizeOf(Year));
			StrMove(Year, PChar(s), Length(s));

			s := PadSpace(FComment, SizeOf(Comment));
			StrMove(Comment, PChar(s), Length(s));

			Genre := ID3NameToGenre(FGenre);
		end;

		FFileStream := TFileStream.Create(FFileName, fmOpenReadWrite);
		try
			if FFileStream.Seek(FTagPos, soFromBeginning)=FTagPos then
			begin
				FFileStream.Write(FTagRecord, SizeOf(FTagRecord));
			end;
		finally
			FFileStream.Free;
		end;
		FModified := False;
	end;
end;

end.
