unit FiltrDlg;

{

+------------------------------------------------------------------------------+
| TFilerDialog Component Version 1.0                                           |
+------------------------------------------------------------------------------+
| Author: Herv Roz                                                            |
| e-mail: hroz@chez.com                                                        |
+------------------------------------------------------------------------------+
| Description:                                                                 |
|   A component that allows you to perform table filter operation              |
| by selecting columns name, operators and assigning values in a               |
| filter dialog box.                                                           |
+------------------------------------------------------------------------------+
| Properties:                                                                  |
|   Filter: Specify the text of the active data set filter.                    |
|   Filtered: Specify if the filter on the data set is active.                 |
|   FilterOptions:                                                             |
|     -foCaseInsensitive                                                       |
|     -foNoPartialCompare                                                      |
|   Table: Data set of wich you apply the filter.                              |
|   Title: Title of the dialog box.                                            |
|                                                                              |
| (*): Read only property                                                      |
+------------------------------------------------------------------------------+
| Methods:                                                                     |
|   Execute: Show the filter dialog box. Returns False if the user select the  |
|     Cancel Button.                                                           |
+------------------------------------------------------------------------------+
| Events:                                                                      |
+------------------------------------------------------------------------------+

+------------------------------------------------------------------------------+
| Installation                                                                 |
+------------------------------------------------------------------------------+
| 1. Unzip the file to any directory you like, for instance                    |
|   C:\Delphi 3\Source\FiltrDlg, ...                                           |
| 2. Start Delphi and close any projects and/or files                          |
| 3. From the menu, choose 'Components/Install Components...'                  |
| 4. In the dialog, click 'Browse'                                             |
| 5. Navigate to your chosen directory and select 'FiltrDlg.pas'               |
| 6. Click OK the way back                                                     |
| 7. In the dialog you have now set the unit file                              |
| 8. Click OK to perform the install                                           |
+------------------------------------------------------------------------------+

+------------------------------------------------------------------------------+
| ******************************* IMPORTANT ********************************** |
+------------------------------------------------------------------------------+
|     This software is absolutely free. Use it and enjoy Delphi.               |
|     All I want is to know why and how you are using this                     |
|     software.                                                                |
|                                                                              |
|     Suggestions for improvement and fixes are always welcome,                |
|     although no guarantee is made when we'll implemement them.               |
|                                                                              |
|     If you do use this component, please drop me an email                    |
|     telling me how you used it.                                              |
|                                                                              |
|     For bug reports, suggestions, new versions etc. contact me:              |
|                                                                              |
|     e-mail: hroz@chez.com                                                    |
|     URL: http://www.chez.com/hroz                                            |
|                                                                              |
+------------------------------------------------------------------------------+

}

interface

uses Windows, SysUtils, Classes, Graphics, Forms, Controls, StdCtrls,
  Buttons, ExtCtrls, DB, BDE, dbTables, Dsgnintf, Mask, DBCtrls;

type
  TFilterDialog = class(TComponent)
  private
    { Private declarations }
    FFilter: string;
    FFiltered: Boolean;
    FFilterOptions: TFilterOptions;
    FTable: TTable;
    FTitle: string;

    procedure SetTable(ATable: TTable);
    procedure DoApplyFilter;
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function Execute: Boolean;
  published
    property Filter: string read FFilter write FFilter;
    property Filtered: Boolean read FFiltered write FFiltered;
    property FilterOptions: TFilterOptions read FFilterOptions write FFilterOptions;
    property Table: TTable read FTable write SetTable;
    property Title: string read FTitle write FTitle;
  end;

  TFFilterDialog = class(TForm)
    ApplyBtn: TButton;
    CancelBtn: TButton;
    gbCondition: TGroupBox;
    gbListCondition: TGroupBox;
    lbxListCondition: TListBox;
    rbAND: TRadioButton;
    rbOR: TRadioButton;
    lbColName: TLabel;
    cbColNames: TComboBox;
    cbOperators: TComboBox;
    Label1: TLabel;
    lbValue: TLabel;
    BtnAdd: TButton;
    DeleteBtn: TButton;
    SuppressBtn: TButton;
    chkCaseSensitive: TCheckBox;
    chkPartialCompare: TCheckBox;
    edValue: TEdit;
    procedure FormCreate(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure BtnAddClick(Sender: TObject);
    procedure DeleteBtnClick(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure ApplyBtnClick(Sender: TObject);
    procedure SuppressBtnClick(Sender: TObject);
  private
    { Private declarations }
    FActive: Boolean;
    FBookmark: TBookmark;
    ConditionCount: Integer;

    procedure EnableControls(Sender: TObject);
    procedure ClearControls(Sender: TObject);
  public
    { Public declarations }
    FFilter: string;
    FFiltered: Boolean;
    FFilterOptions: TFilterOptions;
    FTable: TTable;
  end;

var
  FFilterDialog: TFFilterDialog;

procedure Register;

implementation

{$R *.DFM}

{ TFilterDialog }

constructor TFilterDialog.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FFilterDialog := TFFilterDialog.Create(Self);
end;

destructor TFilterDialog.Destroy;
begin
  FFilterDialog.Free;
  FFilterDialog := nil;
  inherited Destroy;
end;

procedure TFilterDialog.SetTable(ATable: TTable);
begin
  FTable := ATable;
  FFilter := ATable.Filter;
  FFilterOptions := ATable.FilterOptions;
  FFiltered := ATable.Filtered;
end;

procedure TFilterDialog.DoApplyFilter;
begin
  FTable.Filter := FFilter;
  FTable.FilterOptions := FFilterOptions;
  FTable.Filtered := FFiltered;
end;

function TFilterDialog.Execute: Boolean;
begin
  if FTable = nil then raise Exception.Create('Proprit Table manquante.');

  if FTitle <> '' then FFilterDialog.Caption := FTitle;
  FFilterDialog.FTable := FTable;
  FFilterDialog.FFilter := FFilter;
  FFilterDialog.FFilterOptions := FFilterOptions;
  FFilterDialog.FFiltered := FFiltered;
  Result := (FFilterDialog.ShowModal = mrOk);
  if Result then
  begin
    FFilter := FFilterDialog.FFilter;
    FFilterOptions := FFilterDialog.FFilterOptions;
    FFiltered := FFilterDialog.FFiltered;
    DoApplyFilter;
  end;
end;

{ TFFilterDialog }

procedure TFFilterDialog.FormCreate(Sender: TObject);
begin
  ConditionCount := 0;
end;

procedure TFFilterDialog.FormShow(Sender: TObject);
var
  I: Integer;
begin
  with FTable do
  begin
    FActive := Active;
    if FActive then FBookmark := GetBookmark
    else Open;
    DisableControls;
    cbColNames.Items.Clear;
    for I := 0 to FieldCount-1 do cbColNames.Items.Add(Fields[I].FieldName);
  end;
  chkCaseSensitive.Checked := not (foCaseInsensitive in FFilterOptions);
  chkPartialCompare.Checked := not (foNoPartialCompare in FFilterOptions);
  EnableControls(Self);
  ClearControls(Self);
end;

procedure TFFilterDialog.FormClose(Sender: TObject;
  var Action: TCloseAction);
begin
  with FTable do
  begin
    if FActive then
    begin
      GotoBookmark(FBookmark);
      FreeBookmark(FBookmark);
    end
    else Close;
    EnableControls;
  end;
end;

procedure TFFilterDialog.ClearControls(Sender: TObject);
begin
  cbColNames.ItemIndex := -1;
  cbOperators.ItemIndex := -1;
  edValue.Text := '';
  cbColNames.SetFocus;
end;

procedure TFFilterDialog.EnableControls(Sender: TObject);
begin
  rbAND.Enabled := ConditionCount > 0;
  rbOR.Enabled := ConditionCount > 0;
  if rbAND.Enabled then
  begin
    rbAND.Checked := rbAND.Enabled;
    rbOR.Checked  := not rbAND.Enabled;
  end
  else
  begin
    rbAND.Checked := rbAND.Enabled;
    rbOR.Checked  := rbAND.Enabled;
  end;
  DeleteBtn.Enabled := ConditionCount > 0;
  ApplyBtn.Enabled := ConditionCount > 0;
  SuppressBtn.Enabled := FFiltered and (ConditionCount > 0);
end;

procedure TFFilterDialog.BtnAddClick(Sender: TObject);
var
  S, Separator: string;
begin
  if (cbColNames.ItemIndex = -1) or
     (cbColNames.ItemIndex = -1) or
     (edValue.Text = '') then Exit;

  if ConditionCount > 0 then
    with lbxListCondition do
    begin
      S := Items[Items.Count-1];
      if rbAND.Checked then S := S + Chr(9) + 'and'
      else S := S + Chr(9) + 'or ';
      Items[Items.Count-1] := S;
    end;
  S := cbColNames.Text;
  S := S + Chr(9) + cbOperators.Text;
  case FTable.FieldByName(cbColNames.Text).DataType of
    ftString:       Separator := ''''; (*Champ caractre ou chane.*)
    ftSmallint,                        (*Champ entier sur 16 bits.*)
    ftInteger,                         (*Champ entier sur 32 bits.*)
    ftWord,	                       (*Champ entier non sign sur 16 bits.*)
    ftBoolean,                         (*Champ boolen.*)
    ftFloat,	                       (*Champ numrique  virgule flottante.*)
    ftCurrency,                        (*Champ montaire.*)
    ftBCD:	    Separator := '';   (*Champ dcimal cod binaire.*)
    ftDate,	                       (*Champ date.*)
    ftTime,	                       (*Champ heure.*)
    ftDateTime:     Separator := ''''; (*Champ date et heure.*)
    ftBytes,	                       (*Nombre fixe d'octets (stockage binaire).*)
    ftVarBytes:     Exit;              (*Nombre variable d'octets (stockage binaire).*)
    ftAutoInc:      Separator := '';   (*Champ compteur auto-incrment entier sur 32 bits.*)
    ftBlob,                            (*Champ objet binaire volumineux (BLOB).*)
    ftMemo,	                       (*Champ mmo texte.*)
    ftGraphic,	                       (*Champ bitmap.*)
    ftFmtMemo,	                       (*Champ mmo texte format.*)
    ftParadoxOle,	               (*Champ OLE Paradox.*)
    ftDBaseOle,	                       (*Champ OLE dBASE.*)
    ftTypedBinary,	               (*Champ binaire typ.*)
    ftCursor:	      Exit;            (*Ne s'applique pas aux composants champ.*)
  end;
  S := S + Chr(9) + Separator + edValue.Text + Separator;
  lbxListCondition.Items.Add(S);
  Inc(ConditionCount);
  EnableControls(Self);
  ClearControls(Self);
end;

procedure TFFilterDialog.DeleteBtnClick(Sender: TObject);
var
  S: string;
  LastItem: Boolean;
begin
  with lbxListCondition do
  begin
    if ItemIndex = -1 then Exit;
    LastItem := (ItemIndex = Items.Count-1) and (Items.Count > 1);
    Items.Delete(ItemIndex);
    if LastItem then
    begin
      S := Items[Items.Count-1];
      S := Copy(S,1,Length(S)-3);
      Items[Items.Count-1] := S;
    end;
  end;
  Dec(ConditionCount);
  EnableControls(Self);
end;

procedure TFFilterDialog.ApplyBtnClick(Sender: TObject);
begin
  FFilterOptions := [];
  if not chkCaseSensitive.Checked then Include(FFilterOptions,foCaseInsensitive);
  if not chkPartialCompare.Checked then Include(FFilterOptions,foNoPartialCompare);
  FFilter := lbxListCondition.Items.Text;
  FFiltered := True;
end;

procedure TFFilterDialog.SuppressBtnClick(Sender: TObject);
begin
  FFilterOptions := [];
  FFilter := '';
  FFiltered := False;
end;

procedure Register;
begin
  RegisterComponents('VCL', [TFilterDialog]);
end;

end.
