{ Copyright (C) 1998 Tivo Leedjrv                    }
{ Authors e-mail: toivo@kuusalu.edu.ee                 }
{ All rights reserved. Kik igused reserveeritud.     }
{                                                      }
{ TFilterList - Easier handling of TTable filters      }
{                                                      }
unit FilterList;

interface

uses Classes, DBTables;

type

{ TFilterList class }

  PDB = ^TTable;  { TTable pointer TFilterList sisemiseks kasutamiseks }

  TFilterList = class(TStringList)
  private
    FFilter : PDB;
    FAutoBuild : boolean;
    Fs : string;
    function CheckLeft(s : string) : integer;
    function CheckRight(s : string) : integer;
    procedure TrimLeft(var s : string);
    procedure TrimRight(var s : string);
    procedure TrimBoth(var s : string);
    procedure ReBuild;
  public
    constructor Create(Filter : PDB);
    procedure AssignTable(Filter : PDB);
    function Add(const S: string): Integer; override;
    procedure Clear; override;
    procedure DeleteIndex(Index: Integer);
    procedure DeleteF(S: string);
    procedure Insert(Index: Integer; const S: string); override;
    procedure Build;
    function FilterToStr : string;
    property AutoBuild : boolean read FAutoBuild write FAutoBuild default true;
  end;       // Autobuild = true -> Rebuilds table filter every time TFilterList
             // is modified. Sometimes useful, sometimes annoying and slow.
implementation

{ TFilterList }

function TFilterList.CheckLeft(s : string) : integer;
var s3, s4, s5 : string;
begin
  s3:= Copy(s, 1, 3);
  s4:= Copy(s, 1, 4);
  s5:= Copy(s, 1, 5);
  if (s4 = ' or ') or (s5 = ' and ') then
    Result:= 2
  else  if (s3 = 'or ') or (s4 = 'and ') then
    Result:= 1
  else Result:= 0;
end;

function TFilterList.CheckRight(s : string) : integer;
var s3, s4, s5 : string;
begin
  s3:= Copy(s, Length(s) - 2, 3);
  s4:= Copy(s, Length(s) - 3, 4);
  s5:= Copy(s, Length(s) - 4, 5);
  if (s4 = ' or ') or (s5 = ' and ') then
    Result:= 2
  else  if (s3 = ' or') or (s4 = ' and') then
    Result:= 1
  else Result:= 0;
end;

procedure TFilterList.TrimLeft(var s : string);
begin
  case CheckLeft(s) of
    1 : System.Delete(s, 1, 3);
    2 : System.Delete(s, 1, 4);
  end;
end;

procedure TFilterList.TrimRight(var s : string);
begin
  case CheckRight(s) of
    1 : System.Delete(s, Length(s) - 3, 3);
    2 : System.Delete(s, Length(s) - 4, 4);
  end;
end;

procedure TFilterList.TrimBoth(var s :string);
begin
  TrimLeft(s);
  TrimRight(s);
end;

procedure TFilterList.ReBuild;
var i : integer;
begin
  Fs:= '';
  if Count > 0 then
  begin
    Fs:= Strings[0];
    TrimBoth(Fs);
  end;
  for i:= 1 to Count - 1 do
  begin
    if CheckRight(Fs) <> 0 then TrimRight(Fs);
    if (CheckRight(Fs) = 0) and (CheckLeft(Strings[i]) = 2) then
      Fs:= Fs + Strings[i]
    else if (CheckRight(Fs) = 0) and (CheckLeft(Strings[i]) = 1) then
      Fs:= Fs + ' ' + Strings[i]
    else Fs:= Fs + ' and ' + Strings[i];
  end;
  if FFilter <> nil then FFilter^.Filter:= Fs;
end;

{ Assigns different TTable to FilterList f.e.: F.AssignTable(@Table1) }
procedure TFilterList.AssignTable(Filter : PDB);
begin
  FFilter:= Filter;
  if FAutoBuild then ReBuild;
end;

{ Parameter @TTable. f.e.: F.Create(@Table1); nil if no table         }
{ Parameetriks @TTable, mille filtrit hakkab muutma, nil kui ei       }
constructor TFilterList.Create(Filter : PDB);
begin
  inherited Create;
  FAutoBuild:= true;
  FFilter:= Filter;
end;

{ Adds one or more filter conditions. If condition is already in the  }
{ filter then nothing happens                                         }
{ Lisab filtrisse tingimuse S kujul 'vli tehe tingimus' ntx 'nr <> 6'}
{ operaatoreid and, or etc. ei tohi kasutada tingimuse otstes         }
{ ntx 'nr <> 6 and nr <> 7' => lubatud                                }
{ ntx ' and nr <> 6 and ' => keelatud                                 }
function TFilterList.Add(const S: string): Integer;
begin
  if (inherited IndexOf(S)) = -1 then
  begin
    Result:= (inherited Add(S));
    if FAutoBuild then ReBuild;
  end else
    Result:= -1;
end;

{ Clears the filter                                                   }
{ Teeb filtri thjaks                                                 }
procedure TFilterList.Clear;
begin
  inherited Clear;
  if FAutoBuild then ReBuild;
end;

{ Removes the condition specified by the Index parameter              }
{ Kustutab filtrist index-nda tingimuse. Lppuser ei kasuta eriti     }
{ vt. DeleteF                                                         }
procedure TFilterList.DeleteIndex(Index: Integer);
begin
  inherited Delete(Index);
  if FAutoBuild then ReBuild;
end;

{ Removes the condition specified by S parameter, very useful         }
{ Kustutab filtrist tingimuse S kujul 'vli tehe tingimus'            }
{ ntx 'nr <> 6'                                                       }
procedure TFilterList.DeleteF(S: string);
begin
  DeleteIndex((inherited IndexOf(S)));
end;

{ Inserts the condition to position specified by Index parameter      }
{ Lisab filtrisse soovitud kohale soovitud tingimuse                  }
procedure TFilterList.Insert(Index: Integer; const S: string);
begin
  inherited Insert(Index, S);
  if FAutoBuild then ReBuild;
end;

{ Returns the current filter. Build must be executed before if        }
{ AutoBuild is false                                                  }
{ Tagastab terve filtri stringina                                     }
{ Vib kasutada ntx koos Create(nil)-ga muutes TTable.Filtrit oma     }
{ tahtmise jrgi: TTable.Filter:= TFilterList.FilterToStr             }
function TFilterList.FilterToStr : string;
begin
  Result:= Fs;
end;

{ ReBuilds the filter. No need if AutoBuild is true                   }
{ Buildib terve filtri. Kui AutoBuild = true, siis pole vaja kasutada }
procedure TFilterList.Build;
begin
  ReBuild;
end;

{ TFilterList end }

end.
